/*
 * linux/drivers/net/irda/pxa_ir.c
 *
 * Copyright (C) 2004 Intel Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Infra-red SIR and FIR driver for the PXA 210/250 embedded microprocessors
 *  Based on linux/drivers/net/irda/sa1100_ir.c
 *  06/30/2002  add support to PXA2XX/Lubbock by Hao Wu (hao.h.wu@intel.com)
 *  07/06/2003  add support to Bulverde/Mainstone by Mingzhi Lai (mingzhi.lai@intel.com)
 *  09/10/2004  add support to kernel 2.6 by Fengwei Yin (fengwei.yin@intel.com)
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/slab.h>
#include <linux/rtnetlink.h>
#include <linux/interrupt.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/ioport.h>

#include <net/irda/irda.h>
#include <net/irda/wrapper.h>
#include <net/irda/irda_device.h>

#include <asm/irq.h>
#include <asm/dma.h>
#include <asm/hardware.h>
#include <asm/mach-types.h>
#include <asm/arch/pxa-regs.h>
#include <asm/arch/lubbock.h>

#ifdef CONFIG_MACH_MAINSTONE
#include <asm/arch/mainstone.h>
/* If you use Bulverde, you can unremark following definition to enable LED for debug purpose */
/* #define DEBUG_LED_ON  */
#define DEBUG_LED_ON
#define LUB_MISC_WR_IRDA_MODE (1<<4)
#endif				/*  */

#define IrSR_RXPL_NEG_IS_ZERO (1<<4)
#define IrSR_RXPL_POS_IS_ZERO 0x0
#define IrSR_TXPL_NEG_IS_ZERO (1<<3)
#define IrSR_TXPL_POS_IS_ZERO 0x0
#define IrSR_XMODE_PULSE_1_6  (1<<2)
#define IrSR_XMODE_PULSE_3_16 0x0
#define IrSR_RCVEIR_IR_MODE   (1<<1)
#define IrSR_RCVEIR_UART_MODE 0x0
#define IrSR_XMITIR_IR_MODE   (1<<0)
#define IrSR_XMITIR_UART_MODE 0x0

#define IrSR_IR_RECEIVE_ON (\
	IrSR_RXPL_NEG_IS_ZERO | \
	IrSR_TXPL_POS_IS_ZERO | \
	IrSR_XMODE_PULSE_3_16 | \
	IrSR_RCVEIR_IR_MODE | \
	IrSR_XMITIR_UART_MODE)

#define IrSR_IR_TRANSMIT_ON (\
	IrSR_RXPL_NEG_IS_ZERO | \
	IrSR_TXPL_POS_IS_ZERO | \
	IrSR_XMODE_PULSE_3_16 | \
	IrSR_RCVEIR_UART_MODE | \
	IrSR_XMITIR_IR_MODE)

#define IS_FIR(si)		((si)->speed >= 4000000)
#define IRDA_FRAME_SIZE_LIMIT	2047

#define DEBUG_LED_RX	D21
#define DEBUG_LED_TX	D22

#define BOARD_IRDA_SIR	0
#define BOARD_IRDA_FIR	1

#define	IRDA_DMA_ALIGN	8
#define	MEMRES_LENGTH	0xFFFF

//#define DEBUG
#undef	DEBUG
#ifdef	DEBUG
#define dbg(format, arg...)	printk("<1>" __FILE__ ":" format "\n",##arg)
#else				/*  */
#define dbg(format, arg...)
#endif				/*  */


struct pxa_irda {
	struct device *dev;
	struct net_device *netdev;
	unsigned char open;
	int speed;
	int newspeed;
	int mtt;
	dma_addr_t dma_rx_buff_phy;
	dma_addr_t dma_tx_buff_phy;
	unsigned int dma_tx_buff_len;
	int txdma;
	int rxdma;
	struct sk_buff *rxskb;
	struct sk_buff *txskb;
	struct net_device_stats stats;
	struct irlap_cb *irlap;
	struct qos_info qos;
	iobuff_t tx_buff;
	iobuff_t rx_buff;
};


/*
 * Allocate and map the receive buffer, unless it is already allocated.
 */
static int pxa_irda_rx_alloc(struct pxa_irda *si)
{
	if (si->rxskb)
		return 0;

	si->rxskb = alloc_skb(IRDA_FRAME_SIZE_LIMIT + IRDA_DMA_ALIGN, GFP_ATOMIC);
	if (!si->rxskb) {
		printk(KERN_ERR "sa1100_ir: out of memory for RX SKB\n");
		return -ENOMEM;
	}

	/*
	 * Align any IP headers that may be contained
	 * within the frame.
	 * In Bulverde,It must be align on 8 bytes -- by Fengwei Yin.
	 */
	
	skb_reserve(si->rxskb, (IRDA_DMA_ALIGN - ((ulong) si->rxskb->data & IRDA_DMA_ALIGN)));
	si->dma_rx_buff_phy = dma_map_single(si->dev, si->rxskb->data, IRDA_FRAME_SIZE_LIMIT, DMA_FROM_DEVICE);
	return 0;
}

static inline void board_irda_select(int irda_mode)
{
	if (machine_is_lubbock()) {
		if (irda_mode == BOARD_IRDA_SIR) {

			/* select SIR on LUBBOCK */
			LUB_MISC_WR &= ~LUB_MISC_WR_IRDA_MODE;
		} else {

			/* select FIR on LUBBOCK */
			LUB_MISC_WR |= LUB_MISC_WR_IRDA_MODE;
		}
	}

	if (machine_is_mainstone()) {
#ifdef CONFIG_MACH_MAINSTONE
		if (irda_mode == BOARD_IRDA_SIR) {

			/* select SIR on Mainstone */
			MST_MSCWR1 &= ~(1 << 4);
		} else {

			/* select FIR on Mainstone */
			MST_MSCWR1 |= (1 << 4);
		}
#endif
	}
}

static void empty_rx_FIFO(void)
{
	volatile unsigned data;

	while (ICSR1 & ICSR1_RNE) {
		data = ICDR;
	}
}

static inline void pxa_irda_fir_dma_rx_start(struct pxa_irda *si)
{
	unsigned long flags;

	if (!si->rxskb) {
		printk(KERN_ERR "pxa_ir: rx buffer went missing!\n");
		return;
	}

	/*
	 * First empty receive FIFO
	 */

	empty_rx_FIFO();
	local_irq_save(flags);
	DCSR(si->rxdma) = DCSR_NODESC;
	DSADR(si->rxdma) = __PREG(ICDR);
	DTADR(si->rxdma) = si->dma_rx_buff_phy;
	DCMD(si->rxdma) = DCMD_INCTRGADDR | DCMD_FLOWSRC | DCMD_WIDTH1 | DCMD_BURST32 | IRDA_FRAME_SIZE_LIMIT;
	DCSR(si->rxdma) |= DCSR_RUN;
	local_irq_restore(flags);
}

static inline void pxa_irda_fir_dma_tx_start(struct pxa_irda *si)
{
	unsigned long flags;

	local_irq_save(flags);
	DCSR(si->txdma) = DCSR_NODESC;
	DSADR(si->txdma) = si->dma_tx_buff_phy;
	DTADR(si->txdma) = __PREG(ICDR);
	DCMD(si->txdma) = DCMD_INCSRCADDR | DCMD_FLOWTRG | DCMD_ENDIRQEN | DCMD_WIDTH1 | DCMD_BURST32 | si->txskb->len;
	DCSR(si->txdma) |= DCSR_RUN;
	local_irq_restore(flags);
}

/*
 * Set the IrDA communications speed.
 */
static int pxa_irda_set_speed(struct pxa_irda *si, int speed)
{
	unsigned long flags;
	int divisor, ret = -EINVAL;

#ifdef DEBUG_LED_ON
	if (speed == 9600)
		MST_LEDDAT1 = 0x9600;
	else if (speed == 19200)
		MST_LEDDAT1 = 0x19200;
	else if (speed == 38400)
		MST_LEDDAT1 = 0x38400;
	else if (speed == 57600) 
		MST_LEDDAT1 = 0x57600;
	else if (speed == 115200)
		MST_LEDDAT1 = 0x115200;
	else if (speed == 4000000)
		MST_LEDDAT1 = 0x4000000;
#endif				/*  */

	switch (speed) {
	case 9600:
	case 19200:
	case 38400:
	case 57600:
	case 115200:

		/* refer to Bulverde Developer's Manual 10-7 */
		/*  BaudRate = 14.7456 MHz / (16*Divisor) */
		divisor = 14745600 / (16 * speed);
		local_irq_save(flags);
		if (IS_FIR(si)) {

			/* stop RX DMA */
			DCSR(si->rxdma) &= ~DCSR_RUN;

			/* disable FICP */
			ICCR0 &= ~ICCR0_ITR;
			CKEN &= ~CKEN13_FICP;
			board_irda_select(BOARD_IRDA_SIR);

			/* configure GPIO46/47 */
			pxa_gpio_mode(GPIO46_STRXD_MD);
			pxa_gpio_mode(GPIO47_STTXD_MD);

			/* enable the STUART clock */
			CKEN |= CKEN5_STUART;
		}

		/* disable STUART first */
		STIER &= ~IER_UUE;

		/* access DLL & DLH */
		STLCR |= LCR_DLAB;
		STDLL = divisor;
		STDLH = 0;

		/* restore to access THR, RBR & IER */
		STLCR &= ~LCR_DLAB;
		si->speed = speed;
		STISR = IrSR_IR_RECEIVE_ON | IrSR_XMODE_PULSE_1_6;
		STIER = IER_UUE | IER_RLSE | IER_RAVIE | IER_RTIOE;
		local_irq_restore(flags);
		ret = 0;
		break;
	case 4000000:
		local_irq_save(flags);

		/* disable STUART */
		STIER = 0;
		STISR = 0;
		CKEN &= ~CKEN5_STUART;

		/* disable FICP first */
		ICCR0 &= ~ICCR0_ITR;
		board_irda_select(BOARD_IRDA_FIR);

		/* configure GPIO46/47 */
		pxa_gpio_mode(GPIO46_ICPRXD_MD);
		pxa_gpio_mode(GPIO47_ICPTXD_MD);

		/* enable the FICP clock */
		CKEN |= CKEN13_FICP;
		si->speed = speed;
		pxa_irda_rx_alloc(si);
		pxa_irda_fir_dma_rx_start(si);
		ICCR0 = ICCR0_ITR | ICCR0_RXE;
		local_irq_restore(flags);
		ret = 0;
		break;

	default:
		break;
	}
	return ret;
}


/* SIR interrupt service routine. */
static irqreturn_t pxa_irda_sir_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct pxa_irda *si = dev->priv;
	volatile unsigned long iir, lsr, data;

	iir = STIIR;
	switch (iir & 0x0F) {
	case 0x06:

		/* Receiver Line Status */
		lsr = STLSR;
		while (lsr & LSR_FIFOE) {
			data = STRBR;
			if (lsr & (LSR_OE | LSR_PE | LSR_FE | LSR_BI)) {
				printk(KERN_DEBUG "pxa_ir: sir receiving error\n");
				si->stats.rx_errors++;
				if (lsr & LSR_FE)
					si->stats.rx_frame_errors++;
				if (lsr & LSR_OE)
					si->stats.rx_fifo_errors++;
			} else {
				async_unwrap_char(dev, &si->stats, &si->rx_buff, data);
			}
			lsr = STLSR;
		}
		break;
	case 0x04:

		/* Received Data Available */
		/* forth through */
	case 0x0C:

		/* Character Timeout Indication */
		do {
			async_unwrap_char(dev, &si->stats, &si->rx_buff, STRBR);
		} while (STLSR & LSR_DR);
		dev->last_rx = jiffies;
		break;
	case 0x02:

		/* Transmit FIFO Data Request */
		while ((si->tx_buff.len) && (STLSR & LSR_TDRQ)) {
			STTHR = *si->tx_buff.data++;
			si->tx_buff.len -= 1;
		}
		if (si->tx_buff.len == 0) {
			si->stats.tx_packets++;
			si->stats.tx_bytes += si->tx_buff.data - si->tx_buff.head;

			/*
			 * We need to ensure that the transmitter has
			 * finished.
			 */
			while ((STLSR & LSR_TEMT) == 0) {
				rmb();
			}

			/*
			 * Ok, we've finished transmitting.  Now enable
			 * the receiver.  Sometimes we get a receive IRQ
			 * immediately after a transmit...
			 */
#ifdef DEBUG_LED_ON
			DISCRETE_LED_ON(DEBUG_LED_RX);
			DISCRETE_LED_OFF(DEBUG_LED_TX);

#endif				/*  */
			if (si->newspeed) {
				pxa_irda_set_speed(si, si->newspeed);
				si->newspeed = 0;
			} else {

				/* enable IR Receiver, disable IR Transmitter */
				STISR = IrSR_IR_RECEIVE_ON | IrSR_XMODE_PULSE_1_6;

				/* enable STUART and receive interrupts */
				STIER = IER_UUE | IER_RLSE | IER_RAVIE | IER_RTIOE;
			}

			/* I'm hungry! */
			netif_wake_queue(dev);
		}
		break;
	case 0:

		/* Modem Status */
		break;

	default:
		break;
	}
	return IRQ_HANDLED;
}


/* FIR Receive DMA interrupt handler */
static void pxa_irda_fir_dma_rx_irq(int channel, void *data, struct pt_regs *regs)
{
	int dcsr = DCSR(channel);

	DCSR(channel) &= ~DCSR_RUN;
	DCSR(channel) |= DCSR_STARTINTR | DCSR_ENDINTR | DCSR_BUSERR;
	if (dcsr & DCSR_BUSERR) {
		printk(KERN_DEBUG "pxa_ir: fir rx dma bus error.\n");
	}
}


/* FIR Transmit DMA interrupt handler */
static void pxa_irda_fir_dma_tx_irq(int channel, void *data, struct pt_regs *regs)
{
	struct pxa_irda *si = data;
	struct net_device *netdev = si->netdev;
	struct sk_buff *skb = si->txskb;
	int dcsr;

	dcsr = DCSR(channel);
	DCSR(channel) &= ~DCSR_RUN;
	if (dcsr & DCSR_ENDINTR) {
		DCSR(channel) |= DCSR_ENDINTR;
		si->txskb = NULL;
		while (ICSR1 & ICSR1_TBY) {
			rmb();
		}

#ifdef DEBUG_LED_ON
		DISCRETE_LED_ON(DEBUG_LED_RX);
		DISCRETE_LED_OFF(DEBUG_LED_TX);

#endif				/*  */
		/* minimal turn-around time delay */
		if (si->mtt)
			udelay(si->mtt);
		if (si->newspeed) {
			pxa_irda_set_speed(si, si->newspeed);
			si->newspeed = 0;
		} else {
			pxa_irda_fir_dma_rx_start(si);
			ICCR0 = ICCR0_ITR | ICCR0_RXE;
		}
		if (skb) {
			dma_unmap_single(si->dev, si->dma_tx_buff_phy, skb->len, DMA_TO_DEVICE);
			si->stats.tx_packets++;
			si->stats.tx_bytes += skb->len;
			dev_kfree_skb(skb);
		}
		netif_wake_queue(netdev);
	}
}


/* EIF(Error in FIFO/End in Frame) handler for FIR */
static void pxa_irda_fir_irq_eif(struct pxa_irda *si, struct net_device *dev)
{
	unsigned int len;
	volatile unsigned long stat, data;
	struct sk_buff *skb = si->rxskb;

	if (!skb) {
		printk(KERN_ERR "pxa_ir: SKB is NULL!\n");
		return;
	}

	/* Get the current data position. */
	len = DTADR(si->rxdma) - si->dma_rx_buff_phy;

	if (len > IRDA_FRAME_SIZE_LIMIT)
		len = IRDA_FRAME_SIZE_LIMIT;
	
	dma_unmap_single(si->dev, si->dma_rx_buff_phy, len, DMA_FROM_DEVICE);

	do {
		/* Read Status, and then Data.   */
		stat = ICSR1;
		rmb();
		data = ICDR;
		if (stat & (ICSR1_CRE | ICSR1_ROR)) {
			si->stats.rx_errors++;
			if (stat & ICSR1_CRE) {
				printk(KERN_DEBUG "pxa_ir: fir receive CRC error\n");
				si->stats.rx_crc_errors++;
			}
			if (stat & ICSR1_ROR) {
				printk(KERN_DEBUG "pxa_ir: fir receive overrun\n");
				si->stats.rx_frame_errors++;
			}
		} else {
			skb->data[len++] = data;
		}

		/* If we hit the end of frame, there's no point in continuing. */
		if (stat & ICSR1_EOF) {
			break;
		}
	} while (ICSR0 & ICSR0_EIF);

	if (stat & ICSR1_EOF) {

		/* end of frame. */
		si->rxskb = NULL;
		skb_put(skb, len);

		/* Feed it to IrLAP  */
		skb->dev = dev;
		skb->mac.raw = skb->data;
		skb->protocol = htons(ETH_P_IRDA);
		netif_rx(skb);
		si->stats.rx_packets++;
		si->stats.rx_bytes += len;
		pxa_irda_rx_alloc(si);
		dev->last_rx = jiffies;
	} 
}

#ifdef	CONFIG_PXA27x_E23
static void read_E23_data(struct pxa_irda *si)
{
	volatile unsigned long data;
	unsigned int len;

	/* Get the current data position. */
	len = DTADR(si->rxdma) - si->dma_rx_buff_phy;

	if (len > IRDA_FRAME_SIZE_LIMIT)
		len = IRDA_FRAME_SIZE_LIMIT;
	
	dma_unmap_single(si->dev, si->dma_rx_buff_phy, len, DMA_FROM_DEVICE);
	
	while ( !(ICSR1 & ICSR1_EOF)) 
		data = ICDR;
}
#endif 	

/* FIR interrupt handler */
static irqreturn_t pxa_irda_fir_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	struct net_device *dev = dev_id;
	struct pxa_irda *si = dev->priv;
	volatile unsigned long icsr0;

	/* stop RX DMA */
	DCSR(si->rxdma) &= ~DCSR_RUN;
	icsr0 = ICSR0;

	if (icsr0 & ICSR0_FRE) {
		printk(KERN_DEBUG "pxa_ir: fir receive frame error\n");
		si->stats.rx_frame_errors++;
		ICSR0 |= ICSR0_FRE;
	}

	if (icsr0 & ICSR0_RAB) {
		/* FIXME: PXA27x E23 */
#ifdef	CONFIG_PXA27x_E23
		if ( !(icsr0 & ICSR0_EIF)) {
			read_E23_data(si);
			ICSR0 |= ICSR0_RAB;
			goto next_skb;
		} else 
#endif
		{
			printk(KERN_DEBUG "pxa_ir: fir receive abort\n");
			si->stats.rx_errors++;
			ICSR0 |= ICSR0_RAB;
		}
	}

	if (icsr0 & ICSR0_EIF) {
		/* An error in FIFO occues, or there is a end of frame */
		pxa_irda_fir_irq_eif(si, dev);
	}

next_skb:
	pxa_irda_fir_dma_rx_start(si);
	return IRQ_HANDLED;
}


/* hard_xmit interface of irda device */
static int pxa_irda_hard_xmit(struct sk_buff *skb, struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;
	int speed = irda_get_next_speed(skb);

	/*
	 * Does this packet contain a request to change the interface
	 * speed?  If so, remember it until we complete the transmission
	 * of this frame.
	 */
	if (speed != si->speed && speed != -1)
		si->newspeed = speed;

	/*
	 * If this is an empty frame, we can bypass a lot.
	 */
	if (skb->len == 0) {
		if (si->newspeed) {
			si->newspeed = 0;
			pxa_irda_set_speed(si, speed);
		}
		dev_kfree_skb(skb);
		return 0;
	}

	if (!IS_FIR(si)) {
		netif_stop_queue(dev);
		si->tx_buff.data = si->tx_buff.head;
		si->tx_buff.len = async_wrap_skb(skb, si->tx_buff.data, si->tx_buff.truesize);

#ifdef DEBUG_LED_ON
		DISCRETE_LED_OFF(DEBUG_LED_RX);
		DISCRETE_LED_ON(DEBUG_LED_TX);

#endif				/*  */

		/* disable IR Receiver, enable IR Transmitter */
		STISR = IrSR_IR_TRANSMIT_ON | IrSR_XMODE_PULSE_1_6;

		/* It is said when GPIO47 switch from normal mode to SIR mode,
		   we must force GPIO low to avoid false start bit.
		   However it seems also ok when skip it */
		/* GPCR1 |= 0x00008000; */
		/*GPCR(47) |= GPIO_bit(47); */

		/* enable STUART and transmit interrupts */
		STIER = IER_UUE | IER_TIE;

		/* reset Transmitter FIFO to fire the interrupt */
		/* STFCR |= FCR_RESETTF; */
		/* but it doesn't alway work. it seems only when FIFO drops,
		 * the interrupt will occur. so we copy the codes
		 * in the interrupt handler here
		 */
		disable_irq(IRQ_STUART);
		while ((si->tx_buff.len) && (!(STLSR & LSR_TEMT) == 0)) {
			STTHR = *si->tx_buff.data++;
			si->tx_buff.len -= 1;
		};

		enable_irq(IRQ_STUART);
		if (si->tx_buff.len == 0) {
			si->stats.tx_packets++;
			si->stats.tx_bytes += si->tx_buff.data - si->tx_buff.head;

			/*
			 * We need to ensure that the transmitter has
			 * finished.
			 */
			while ((STLSR & LSR_TEMT) == 0) {
				rmb();
			}

			/*
			 * Ok, we've finished transmitting.  Now enable
			 * the receiver.  Sometimes we get a receive IRQ
			 * immediately after a transmit...
			 */
#ifdef DEBUG_LED_ON
			DISCRETE_LED_ON(DEBUG_LED_RX);
			DISCRETE_LED_OFF(DEBUG_LED_TX);

#endif				/*  */
			if (si->newspeed) {
				pxa_irda_set_speed(si, si->newspeed);
				si->newspeed = 0;
			} else {

				/* enable IR Receiver, disable IR Transmitter */
				STISR = IrSR_IR_RECEIVE_ON | IrSR_XMODE_PULSE_1_6;

				/* enable STUART and receive interrupts */
				STIER = IER_UUE | IER_RLSE | IER_RAVIE | IER_RTIOE;
			}

			/* I'm hungry! */
			netif_wake_queue(dev);
		}
		dev_kfree_skb(skb);
	} else {
		si->mtt = irda_get_mtt(skb);
		if (si->txskb)
			BUG();
		netif_stop_queue(dev);

#ifdef DEBUG_LED_ON
		DISCRETE_LED_OFF(DEBUG_LED_RX);
		DISCRETE_LED_ON(DEBUG_LED_TX);

#endif				/*  */

		si->txskb = skb;
		si->dma_tx_buff_phy = dma_map_single(si->dev, skb->data, skb->len, DMA_TO_DEVICE);
		pxa_irda_fir_dma_tx_start(si);
		if (si->mtt)
			udelay(si->mtt);
		ICCR0 = ICCR0_ITR | ICCR0_TXE;
	}
	dev->trans_start = jiffies;
	return 0;
}


static int pxa_irda_ioctl(struct net_device *dev, struct ifreq *ifreq, int cmd)
{
	struct if_irda_req *rq = (struct if_irda_req *) ifreq;
	struct pxa_irda *si = dev->priv;
	int ret = -EOPNOTSUPP;

	switch (cmd) {
	case SIOCSBANDWIDTH:
		if (capable(CAP_NET_ADMIN)) {

			/*
			 * We are unable to set the speed if the
			 * device is not running.
			 */
			if (si->open) {
				ret = pxa_irda_set_speed(si, rq->ifr_baudrate);
			} else {
				printk(KERN_INFO "pxa_ir: SIOCSBANDWIDTH: !netif_running\n");
				ret = 0;
			}
		}
		break;
	case SIOCSMEDIABUSY:
		ret = -EPERM;
		if (capable(CAP_NET_ADMIN)) {
			irda_device_set_media_busy(dev, TRUE);
			ret = 0;
		}
		break;
	case SIOCGRECEIVING:
		rq->ifr_receiving = IS_FIR(si) ? 0 : si->rx_buff.state != OUTSIDE_FRAME;
		break;
	default:
		break;
	}
	return ret;
}


static struct net_device_stats *pxa_irda_stats(struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;
	return &si->stats;
}


static int pxa_irda_startup(struct pxa_irda *si)
{
	int ret;

	/* enable STUART interrupt to the processor */
	STMCR = MCR_OUT2;

	/* configure SIR frame format: StartBit - Data 7 ... Data 0 - Stop Bit */
	STLCR = LCR_WLS0 | LCR_WLS1;

	/* enable FIFO, we use FIFO to improve performance */
	STFCR = FCR_TRFIFOE | FCR_ITL_32;

	/* configure FICP ICCR2 */
	ICCR2 = ICCR2_TXP | ICCR2_TRIG_32;

	/* configure DMAC */
	DRCMR17 = si->rxdma | DRCMR_MAPVLD;
	DRCMR18 = si->txdma | DRCMR_MAPVLD;

	/* we start from SIR 9600 baudrate */
	/* disable FICP */
	ICCR0 &= ~ICCR0_ITR;
	CKEN &= ~CKEN13_FICP;
	board_irda_select(BOARD_IRDA_SIR);
	if (machine_is_mainstone())
#ifdef CONFIG_MACH_MAINSTONE
		MST_MSCWR1 &= ~(3 << 4);
#endif
	/* configure GPIO46/47 */
	pxa_gpio_mode(GPIO46_STRXD_MD);
	pxa_gpio_mode(GPIO47_STTXD_MD);

	/* enable the STUART clock */
	CKEN |= CKEN5_STUART;
	ret = pxa_irda_set_speed(si, si->speed = 9600);
	if (ret)
		return ret;

#ifdef DEBUG_LED_ON
	DISCRETE_LED_ON(DEBUG_LED_RX);
	DISCRETE_LED_OFF(DEBUG_LED_TX);

#endif				/*  */
	printk(KERN_INFO "pxa_ir: irda startup\n");
	return 0;
}

static void pxa_irda_shutdown(struct pxa_irda *si)
{

	/* disable the STUART clock */
	CKEN &= ~CKEN5_STUART;

	/* disable STUART and interrupt */
	STIER = 0;

	/* disable STUART SIR mode */
	STISR = 0;

	/* disable the FICP clock */
	CKEN &= ~CKEN13_FICP;

	/* disable FICP */
	ICCR0 = 0;

#ifdef DEBUG_LED_ON
	DISCRETE_LED_OFF(DEBUG_LED_RX);
	DISCRETE_LED_OFF(DEBUG_LED_TX);

#endif				/*  */
	printk(KERN_INFO "pxa_ir: irda shutdown\n");
} 

static int pxa_irda_start(struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;
	int err;

	si->speed = 9600;
	err = request_irq(IRQ_STUART, pxa_irda_sir_irq, 0, dev->name, dev);
	if (err)
		goto err_irq1;
	err = request_irq(IRQ_ICP, pxa_irda_fir_irq, 0, dev->name, dev);
	if (err)
		goto err_irq2;

	/*
	 * The interrupt must remain disabled for now.
	 */
	disable_irq(IRQ_STUART);
	disable_irq(IRQ_ICP);
	si->rxdma = pxa_request_dma("FICP_RX", DMA_PRIO_LOW, pxa_irda_fir_dma_rx_irq, si);
	if (si->rxdma < 0)
		goto err_rx_dma;
	si->txdma = pxa_request_dma("FICP_TX", DMA_PRIO_LOW, pxa_irda_fir_dma_tx_irq, si);
	if (si->txdma < 0)
		goto err_tx_dma;

	/*
	 * Setup the serial port for the specified speed.
	 */
	err = pxa_irda_startup(si);
	if (err)
		goto err_startup;

	/*
	 * Open a new IrLAP layer instance.
	 */
	si->irlap = irlap_open(dev, &si->qos, "pxa");
	err = -ENOMEM;
	if (!si->irlap)
		goto err_irlap;

	/*
	 * Now enable the interrupt and start the queue
	 */
	si->open = 1;
	enable_irq(IRQ_STUART);
	enable_irq(IRQ_ICP);
	netif_start_queue(dev);
	printk(KERN_INFO "pxa_ir: irda driver opened\n");
	return 0;

err_irlap:
	si->open = 0;
	pxa_irda_shutdown(si);
err_startup:
	pxa_free_dma(si->txdma);
err_tx_dma:
	pxa_free_dma(si->rxdma);
err_rx_dma:
	free_irq(IRQ_ICP, dev);
err_irq2:
	free_irq(IRQ_STUART, dev);
err_irq1:
	return err;
}

static int pxa_irda_stop(struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;

	disable_irq(IRQ_STUART);
	disable_irq(IRQ_ICP);
	pxa_irda_shutdown(si);

	/*
	 * If we have been doing DMA receive, make sure we
	 * tidy that up cleanly.
	 */
	if (si->rxskb) {
		dma_unmap_single(si->dev, si->dma_rx_buff_phy, IRDA_FRAME_SIZE_LIMIT, DMA_FROM_DEVICE);
		dev_kfree_skb(si->rxskb);
		si->rxskb = NULL;
	}

	/* Stop IrLAP */
	if (si->irlap) {
		irlap_close(si->irlap);
		si->irlap = NULL;
	}
	netif_stop_queue(dev);
	si->open = 0;
	free_irq(IRQ_STUART, dev);
	free_irq(IRQ_ICP, dev);
	pxa_free_dma(si->rxdma);
	pxa_free_dma(si->txdma);
	printk(KERN_INFO "pxa_ir: irda driver closed\n");
	return 0;
}


#ifdef	CONFIG_PM
/*
 * Suspend the IrDA interface.
 */
static int pxa_irda_suspend(struct device *_dev, u32 state, u32 level)
{
	struct net_device *dev = dev_get_drvdata(_dev);
	struct pxa_irda *si = dev->priv;

	if (!dev || level != SUSPEND_DISABLE)
		return 0;
	printk(KERN_INFO "pxa_ir: irda suspend\n");
	if (si && si->open) {
		/*
		 * Stop the transmit queue
		 */
		netif_device_detach(dev);
		disable_irq(IRQ_STUART);
		disable_irq(IRQ_ICP);
		pxa_irda_shutdown(si);

		/* add here: shutdown IR transceiver on the board */
	}
	return 0;
}


/*
 * Resume the IrDA interface.
 */
static int pxa_irda_resume(struct device *_dev, u32 level)
{
	struct net_device *dev = dev_get_drvdata(_dev);
	struct pxa_irda *si = dev->priv;

	if (!dev || level != RESUME_ENABLE)
		return 0;
	printk(KERN_INFO "pxa_ir: irda resume\n");
	if (si && si->open) {

		/*
		 * If we missed a speed change, initialise at the new speed
		 * directly.  It is debatable whether this is actually
		 * required, but in the interests of continuing from where
		 * we left off it is desireable.  The converse argument is
		 * that we should re-negotiate at 9600 baud again.
		 */
		if (si->newspeed) {
			si->speed = si->newspeed;
			si->newspeed = 0;
		}

		/* add here: restart IR transceiver on the board */
		pxa_irda_startup(si);
		enable_irq(IRQ_STUART);
		enable_irq(IRQ_ICP);

		/*
		 * This automatically wakes up the queue
		 */
		netif_device_attach(dev);
	}
	return 0;
}


#else				/*  */
#define	pxa_irda_suspend	NULL
#define	pxa_irda_resume		NULL
#endif				/*  */

static int pxa_irda_init_iobuf(iobuff_t * io, int size)
{
	io->head = kmalloc(size, GFP_KERNEL | GFP_DMA);
	if (io->head != NULL) {
		io->truesize = size;
		io->in_frame = FALSE;
		io->state = OUTSIDE_FRAME;
		io->data = io->head;
	}
	return io->head ? 0 : -ENOMEM;
}

static int pxa_irda_net_init(struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;
	unsigned int baudrate_mask;
	int err = -ENOMEM;

	/*
	 * Initialise the SIR buffers
	 */
	err = pxa_irda_init_iobuf(&si->rx_buff, 14384);
	if (err)
		goto out;

	err = pxa_irda_init_iobuf(&si->tx_buff, 4000);
	if (err)
		goto out_free_rx;

	/*
	 * Request the regions.
	 */
	err = request_mem_region(__PREG(FICP), MEMRES_LENGTH, "irda") ? 0 : -EBUSY; 
	if (err)
		goto mem_out1;
	
	err = request_mem_region(__PREG(STUART), MEMRES_LENGTH, "irda") ? 0 : -EBUSY;
	if (err)
		goto mem_out2;

	dev->hard_start_xmit = pxa_irda_hard_xmit;
	dev->open = pxa_irda_start;
	dev->stop = pxa_irda_stop;
	dev->do_ioctl = pxa_irda_ioctl;
	dev->get_stats = pxa_irda_stats;
	dev->irq = IRQ_ICP;
	irda_init_max_qos_capabilies(&si->qos);

	/*
	 * We support original IRDA up to 115k2. (we don't currently
	 * support 4Mbps).  Min Turn Time set to 1ms or greater.
	 */
	baudrate_mask = IR_9600 | IR_19200 | IR_38400 | IR_57600 | IR_115200;
	baudrate_mask |= IR_4000000 << 8;
	si->qos.baud_rate.bits &= baudrate_mask;
	si->qos.min_turn_time.bits = 7;	/* 1ms or more */
	irda_qos_bits_to_value(&si->qos);
	printk(KERN_INFO "pxa_ir: irda driver inited\n");
	return 0;

mem_out2:
	release_mem_region(__PREG(FICP), MEMRES_LENGTH);
mem_out1:
	kfree(si->tx_buff.head);
out_free_rx:
	kfree(si->rx_buff.head);
out:
	return err;
}


/*
 * Remove all traces of this driver module from the kernel, so we can't be
 * called.  Note that the device has already been stopped, so we don't have
 * to worry about interrupts or dma.
 */
static void pxa_irda_net_uninit(struct net_device *dev)
{
	struct pxa_irda *si = dev->priv;

	dev->hard_start_xmit = NULL;
	dev->open = NULL;
	dev->stop = NULL;
	dev->do_ioctl = NULL;
	dev->get_stats = NULL;
	dev->priv = NULL;

	release_mem_region(__PREG(STUART), MEMRES_LENGTH);
	release_mem_region(__PREG(FICP), MEMRES_LENGTH);

	kfree(si->tx_buff.head);
	kfree(si->rx_buff.head);
	printk(KERN_INFO "pxa_ir: irda driver uninited\n");
}

static int pxa_irda_probe(struct device *_dev)
{
	struct platform_device *pdev = to_platform_device(_dev);
	struct net_device *dev;
	struct pxa_irda *si;
	int err;

	dev = alloc_irdadev(sizeof(struct pxa_irda));
	if (!dev)
		return -ENOMEM;

	si = dev->priv;
	si->dev = &pdev->dev;
	si->netdev = dev;
	dev->init = pxa_irda_net_init;
	dev->uninit = pxa_irda_net_uninit;

	err = register_netdev(dev);
	if (err == 0)
		dev_set_drvdata(&pdev->dev, dev);

	else
		free_netdev(dev);

	return err;
}

static int pxa_irda_remove(struct device *_dev)
{
	struct net_device *dev = dev_get_drvdata(_dev);

	if (dev) {
		unregister_netdev(dev);
		free_netdev(dev);
	}
	return 0;
}

static struct resource pxa_irda_resources[] = {
        [0] = {                         // fast irda mem resource
                .start  = 0x40800000,
                .end    = 0x4080ffff,
                .flags  = IORESOURCE_MEM,},
        [1] = {                         // stuart mem resource
                .start  = 0x40700000,
                .end    = 0x4070ffff,
                .flags  = IORESOURCE_MEM,},
        [2] = {                         // fast irda irq resource
                .start  = IRQ_ICP,
                .end    = IRQ_ICP,
                .flags  = IORESOURCE_IRQ,},
        [3] = {                         // stuart irq resource
                .start  = IRQ_STUART,
                .end    = IRQ_STUART,
                .flags  = IORESOURCE_IRQ,},
};

static void pxa_irda_dev_release(struct device *pdev)
{
	return;
}

static u64 irda_dma_mask = ~(u64)0;

static struct platform_device pxa_irda_device = {
        .name 		= "pxa-irda",
        .id 		= 0,
        .resource 	= pxa_irda_resources,
        .num_resources 	= ARRAY_SIZE(pxa_irda_resources),
	.dev		= {
		.dma_mask	= &irda_dma_mask,
		.coherent_dma_mask = 0xffffffff,
		.release = pxa_irda_dev_release,
	},
};

static struct device_driver pxa_irda_driver = {
	.name		= "pxa-irda",
	.bus		= &platform_bus_type,
	.probe		= pxa_irda_probe,
	.remove		= pxa_irda_remove,
	.suspend	= pxa_irda_suspend,
	.resume		= pxa_irda_resume,
};

static int __init pxa_irda_init(void)
{
	int ret;
	
	ret = platform_device_register(&pxa_irda_device);
	if (!ret) {
		ret = driver_register(&pxa_irda_driver);
		if (ret) 
			platform_device_unregister(&pxa_irda_device);
	}

	return ret;
}

static void __exit pxa_irda_exit(void)
{
	driver_unregister(&pxa_irda_driver);
	platform_device_unregister(&pxa_irda_device);
} 

module_init(pxa_irda_init);
module_exit(pxa_irda_exit);
MODULE_DESCRIPTION("Bulverde SIR/FIR driver");
MODULE_LICENSE("GPL");
