/*
 * Map driver for the PXA27x-based developer platform.
 *
 * Modified from linux/drivers/mtd/maps/lubbock-flash.c
 *
 * Copyright (C) 2004, Intel Corporation. 
 * Modified and extended by Matthias Ihmig <m.ihmig@mytum.de>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <asm/pgtable.h>
#include <asm/arch/pxa-regs.h>

// Size of complete available flash memory in bytes
#define WINDOW_SIZE 	(CONFIG_MTD_PXA27x_WINDOWSIZE * 1024 * 1024)

/* Size settings (in bytes) and calculations for default flash partition table */
#define BLOB_FLASH_LEN		(256 * 1024)
#define KERNEL_FLASH_LEN	(2 * 1024 * 1024)
// RAMDISK_FLASH_LEN takes up the remainder and is calculated a few lines later
#define PARAM_FLASH_LEN		(256 * 1024)

#define BLOB_FLASH_BASE		(0x00000000)
#define KERNEL_FLASH_BASE	(BLOB_FLASH_BASE + BLOB_FLASH_LEN)
#define RAMDISK_FLASH_BASE	(KERNEL_FLASH_BASE + KERNEL_FLASH_LEN)

#define RAMDISK_FLASH_LEN	(WINDOW_SIZE - KERNEL_FLASH_LEN - PARAM_FLASH_LEN - BLOB_FLASH_LEN)

#define PARAM_FLASH_BASE	(RAMDISK_FLASH_BASE + RAMDISK_FLASH_LEN)


/* Uncomment define to use cached memory access to flash
 *
 * This should not be used, as mtdblock already does caching thus
 * making caching on this level unnecessary!
 * 
 * Also, data may be lost in case of a sudden power-off (which may occur
 * esp. in mobile devices
 */
//#define USE_CACHED_FLASH

struct pxa27x_flash_info {
	struct mtd_info *mtd;
	struct mtd_partition *part;
	struct map_info *map;
	int nparts;
};	

#ifdef USE_CACHED_FLASH
static void pxa27x_map_inval_cache(struct map_info *map, unsigned long from, ssize_t len)
{
	consistent_sync((char *)map->cached + from, len, DMA_FROM_DEVICE);
}
#endif

// don't use spaces within the name or else the cmdline partition table parser won't work
static struct map_info pxa27x_map = {
	.size =		WINDOW_SIZE,
	.phys =		0x00000000,
	.name =		"pxa27x-flash"
};

static struct mtd_partition pxa27x_partitions[] = {
	{
		name:		"Bootloader",
		size:		BLOB_FLASH_LEN,
		offset:		BLOB_FLASH_BASE,
//		mask_flags:	MTD_WRITEABLE  /* uncomment to enforce read-only partition */
	},{
		name:		"Kernel",
		size:		KERNEL_FLASH_LEN,
		offset:		KERNEL_FLASH_BASE
	},{
		name:		"Filesystem",
		size:		RAMDISK_FLASH_LEN,
		offset:		RAMDISK_FLASH_BASE
	} ,{
		name:   	"Param",
		size:  		PARAM_FLASH_LEN,
		offset:		PARAM_FLASH_BASE
	}
};

static struct mtd_info *mymtd;
static struct mtd_partition *parsed_parts;
static int nr_parsed_part;

static const char *probes[] = { "RedBoot", "cmdlinepart", NULL };

static int pxa27x_flash_probe(struct device *dev)
{
	int ret = 0, i;
	int err = 0;
	struct pxa27x_flash_info *info;

	info = kmalloc(sizeof(struct pxa27x_flash_info), GFP_KERNEL);
	if(!info) {
		err = -ENOMEM;
		goto Error;
	}	
	memzero(info, sizeof(struct pxa27x_flash_info));

	dev_set_drvdata(dev, info);

	pxa27x_map.bankwidth = (BOOT_DEF & 1) ? 2 : 4;
	pxa27x_map.cached = NULL;

	pxa27x_map.virt = ioremap(pxa27x_map.phys, pxa27x_map.size);
	if (!pxa27x_map.virt) {
		printk(KERN_WARNING "Failed to ioremap %s\n", pxa27x_map.name);
		err = -ENOMEM;
		goto Error;
	}
#ifdef USE_CACHED_FLASH
	pxa27x_map.inval_cache = pxa27x_map_inval_cache;
	pxa27x_map.cached = __ioremap(pxa27x_map.phys, WINDOW_SIZE, L_PTE_CACHEABLE, 1);
	if (!pxa27x_map.cached)
		printk(KERN_WARNING "Failed to ioremap cached %s\n", pxa27x_map.name);
	else
		printk(KERN_INFO "Using cached access to flash memory! Do you really want to do this?\n");
#endif

	simple_map_init(&pxa27x_map);

	printk(KERN_NOTICE "Probing %s at physical address 0x%08lx (%d-bit bankwidth)\n",
	       pxa27x_map.name, pxa27x_map.phys, pxa27x_map.bankwidth * 8);

	mymtd = do_map_probe("cfi_probe", &pxa27x_map);
		
	if (!mymtd) {
		iounmap((void *)pxa27x_map.virt);
		if (pxa27x_map.cached)
			iounmap(pxa27x_map.cached);
		err = -EIO;
		goto Error;
	}
	mymtd->owner = THIS_MODULE;

	/* Unlock the flash device. */
	for (i = 0; i < mymtd->numeraseregions; i++) {
		int j;
		for( j = 0; j < mymtd->eraseregions[i].numblocks; j++) {
			mymtd->unlock(mymtd, mymtd->eraseregions[i].offset +
			      j * mymtd->eraseregions[i].erasesize,
			      mymtd->eraseregions[i].erasesize);
		}
	}


	ret = parse_mtd_partitions(mymtd, probes, &parsed_parts, 0);

	if (ret > 0) nr_parsed_part = ret;

	if (nr_parsed_part) {
		add_mtd_partitions(mymtd, parsed_parts, nr_parsed_part);
		info->part = parsed_parts;
		info->nparts = nr_parsed_part;
	} else {
		printk("Using static partitions on %s\n", pxa27x_map.name);
		add_mtd_partitions(mymtd, pxa27x_partitions, ARRAY_SIZE(pxa27x_partitions));
		info->nparts = ARRAY_SIZE(pxa27x_partitions);
		info->part = pxa27x_partitions;
	}
	info->mtd = mymtd;
	return 0;
Error:
	if (info)
		kfree(info);
	return err;
}

static int pxa27x_flash_remove(struct device *dev)
{
	struct pxa27x_flash_info *info = dev_get_drvdata(dev);

	dev_set_drvdata(dev, NULL);

	if (!mymtd) return 0;

	if (nr_parsed_part)
		del_mtd_partitions(mymtd);
	else
		del_mtd_device(mymtd);
	
	map_destroy(mymtd);

	if (pxa27x_map.virt)
		iounmap((void *)pxa27x_map.virt);

	if (pxa27x_map.cached) iounmap(pxa27x_map.cached);

	if (parsed_parts) kfree(parsed_parts);
	if (info) kfree(info);
	return 0;
}

static int pxa27x_flash_suspend(struct device *dev, u32 state, u32 level)
{
	struct pxa27x_flash_info *info = dev_get_drvdata(dev);
	struct mtd_info *mtd = info->mtd;
	int ret = 0;
	
	if (!mtd)
		return 0;
	if (level == SUSPEND_POWER_DOWN) {
		if (mtd->suspend)
			ret = mtd->suspend(mtd);
	}
	return ret;
}

static int pxa27x_flash_resume(struct device *dev, u32 level)
{
	struct pxa27x_flash_info *info = dev_get_drvdata(dev);
	struct mtd_info *mtd = info->mtd;
	int i, j;

	if (!mtd)
		return 0;
	if (level == RESUME_POWER_ON) {
		if (mtd->resume) 
			mtd->resume(mtd);
		for (i = 0; i < mtd->numeraseregions; i++) {
			for (j = 0; j < mtd->eraseregions[i].numblocks; j++) {
				mtd->unlock(mtd, mtd->eraseregions[i].offset +
					j * mtd->eraseregions[i].erasesize,
					mtd->eraseregions[i].erasesize);
			}
		}
	}
	return 0;
}

static struct device_driver pxa27x_flash_driver = {
	.name		= "pxa27x-flash",
	.bus		= &platform_bus_type,
	.probe		= pxa27x_flash_probe,
	.remove		= pxa27x_flash_remove,
	.suspend	= pxa27x_flash_suspend,
	.resume		= pxa27x_flash_resume,
};

static void __exit cleanup_pxa27x(void)
{
	driver_unregister(&pxa27x_flash_driver);
}
static int __init init_pxa27x(void)
{
	return driver_register(&pxa27x_flash_driver);
}

module_init(init_pxa27x);
module_exit(cleanup_pxa27x);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Yu Tang <yu.tang@intel.com>");
MODULE_DESCRIPTION("MTD map driver for Intel PXA27x On-Chip ROM");
