/*
 * wm97xx-core.c  --  Touch screen driver core for Wolfson WM9705, WM9712
 *                           and WM9713 AC97 Codecs.
 *
 * Copyright 2003, 2004, 2005 Wolfson Microelectronics PLC.
 * Author: Liam Girdwood
 *         liam.girdwood@wolfsonmicro.com or linux@wolfsonmicro.com
 * Parts Copyright : Ian Molton <spyro@f2s.com>
 *                   Andrew Zabolotny <zap@homelink.ru>
 *                   Russell King <rmk@arm.linux.org.uk>
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Notes:
 *
 *  Features:
 *       - supports WM9705, WM9712, WM9713
 *       - polling mode
 *       - coordinate polling
 *       - continuous mode (arch-dependent)
 *       - adjustable rpu/dpp settings
 *       - adjustable pressure current
 *       - adjustable sample settle delay
 *       - 4 and 5 wire touchscreens (5 wire is WM9712 only)
 *       - pen down detection
 *       - battery monitor
 *       - sample AUX adc's
 *       - power management
 *       - codec GPIO
 *       - codec event notification
 *
 *  Revision history
 *    7th May 2003   Initial version.
 *    6th June 2003  Added non module support and AC97 registration.
 *   18th June 2003  Added AUX adc sampling.
 *   23rd June 2003  Did some minimal reformatting, fixed a couple of
 *                   locking bugs and noted a race to fix.
 *   24th June 2003  Added power management and fixed race condition.
 *   10th July 2003  Changed to a misc device.
 *   31st July 2003  Moved TS_EVENT and TS_CAL to wm97xx.h
 *    8th Aug  2003  Added option for read() calling wm97xx_sample_touch()
 *                   because some ac97_read/ac_97_write call schedule()
 *    7th Nov  2003  Added Input touch event interface, stanley.cai@intel.com
 *   13th Nov  2003  Removed h3600 touch interface, added interrupt based
 *                   pen down notification and implemented continous mode
 *                   on XScale arch.
 *   16th Nov  2003  Ian Molton <spyro@f2s.com>
 *                   Modified so that it suits the new 2.6 driver model.
 *   25th Jan  2004  Andrew Zabolotny <zap@homelink.ru>
 *                   Implemented IRQ-driven pen down detection, implemented
 *                   the private API meant to be exposed to platform-specific
 *                   drivers, reorganized the driver so that it supports
 *                   an arbitrary number of devices.
 *    1st Feb  2004  Moved continuous mode handling to a separate
 *                   architecture-dependent file. For now only PXA
 *                   built-in AC97 controller is supported (pxa-ac97-wm97xx.c).
 *    11th Feb 2004  Reduced CPU usage by keeping a cached copy of both
 *                   digitizer registers instead of reading them every time.
 *                   A reorganization of the whole code for better
 *                   error handling.
 *    17th Apr 2004  Added BMON support.
 *    17th Nov 2004  Added codec GPIO, codec event handling (real and virtual
 *                   GPIOs) and 2.6 power management. 
 *    29th Nov 2004  Added WM9713 support.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/proc_fs.h>
#include <linux/pm.h>
#include <linux/interrupt.h>
#include <linux/bitops.h>
#include <linux/workqueue.h>
#include <linux/device.h>
#include <linux/list.h>
#include <asm/uaccess.h>
#include <asm/io.h>

#include "wm97xx.h"
#include "mcp.h"

#define TS_NAME			"wm97xx"
#define WM_TS_VERSION	"0.30"
#define DEFAULT_PRESSURE	0xb0c0

#define to_wm97xx(x) container_of((x), struct wm97xx, ts)
#define to_wm97xx_ts(x) container_of((x), struct wm97xx_ts, wm97xx_codec)

/*
 * Debug
 */
#if 1
#define dbg(format, arg...) printk(KERN_DEBUG TS_NAME ": " format "\n" , ## arg)
#else
#define dbg(format, arg...)
#endif
#define err(format, arg...) printk(KERN_ERR TS_NAME ": " format "\n" , ## arg)
#define info(format, arg...) printk(KERN_INFO TS_NAME ": " format "\n" , ## arg)
#define warn(format, arg...) printk(KERN_WARNING TS_NAME ": " format "\n" , ## arg)

extern struct class wm97xx_class;

/*
 * Read AUX adc
 */
static int wm97xx_read_aux_adc (struct wm97xx* wm, u16 adcsel)
{
	int power_adc = 0, auxval;
	u16 power = 0;

    /* get codec */
	spin_lock(&wm->lock);

	/* When the touchscreen is not in use, we may have to power up the AUX ADC
	 * before we can use sample the AUX inputs->
	 */
	if (wm->id  == WM9713_ID2 &&
		(power = mcp_reg_read(wm->mcp, AC97_EXTENDED_MID)) & 0x8000) {
		power_adc = 1;
		mcp_reg_write(wm->mcp, AC97_EXTENDED_MID, power & 0x7fff);
	}
	
	/* Prepare the codec for AUX reading */
	wm->wm97xx_codec->digitiser_save(wm);
	wm->wm97xx_codec->aux_prepare(wm);
	
	/* Turn polling mode on to read AUX ADC */
	wm->pen_probably_down = 1;
    wm->wm97xx_codec->poll_sample (wm, adcsel, &auxval);

	if (power_adc)
		mcp_reg_write(wm->mcp, AC97_EXTENDED_MID, power | 0x8000);

	wm->pen_probably_down = 0;
	wm->wm97xx_codec->digitiser_restore(wm);

	spin_unlock(&wm->lock);

	return auxval;
}

#define WM97XX_ATTR(name,input)\
static ssize_t name##_show(struct class_device *dev, char *buf)   \
{                             \
  struct wm97xx *wm = classdev_to_wm97xx(dev);     \
  return sprintf(buf, "%d\n", wm97xx_read_aux_adc(wm, input));  \
}                             \
static CLASS_DEVICE_ATTR(name,0444,name##_show,NULL)

WM97XX_ATTR(aux1, WM97XX_AUX_ID1);
WM97XX_ATTR(aux2, WM97XX_AUX_ID2);
WM97XX_ATTR(aux3, WM97XX_AUX_ID3);
WM97XX_ATTR(aux4, WM97XX_AUX_ID4);

static int wm97xx_aux_adc_add(struct class_device *dev)
{
    class_device_create_file(dev, &class_device_attr_aux1);
    class_device_create_file(dev, &class_device_attr_aux2);
    class_device_create_file(dev, &class_device_attr_aux3);
    class_device_create_file(dev, &class_device_attr_aux4);
    return 0;
}

static void wm97xx_aux_adc_remove(struct class_device *dev)
{
    class_device_remove_file(dev, &class_device_attr_aux1);
    class_device_remove_file(dev, &class_device_attr_aux2);
    class_device_remove_file(dev, &class_device_attr_aux3);
    class_device_remove_file(dev, &class_device_attr_aux4);
}

static struct class_interface wm97xx_aux_adc_interface = {
    .add    = wm97xx_aux_adc_add,
    .remove = wm97xx_aux_adc_remove,
};


/*
 * Get the status of a codec GPIO pin
 */
wm97xx_gpio_status_t wm97xx_get_codec_gpio (struct wm97xx *wm, u32 gpio)
{
	u16 status = mcp_reg_read(wm->mcp, AC97_GPIO_STATUS);
	
	if (status & gpio)
		return WM97XX_GPIO_HIGH;
	else
		return WM97XX_GPIO_LOW;
}
EXPORT_SYMBOL_GPL(wm97xx_get_codec_gpio);

/*
 * Set the status of a codec GPIO pin
 */
void wm97xx_set_codec_gpio (struct wm97xx *wm, u32 gpio, wm97xx_gpio_status_t status)
{
	u16 reg;

	spin_lock(&wm->lock);
	reg = mcp_reg_read(wm->mcp, AC97_GPIO_STATUS);
	if (status & WM97XX_GPIO_HIGH)
		reg |= gpio;
	else
		reg &= ~gpio;
	
	if (wm->id == WM9712_ID2) {
		/* small hack for wm9712 */
		//CODEC->id = 1;
		mcp_reg_write(wm->mcp, AC97_GPIO_STATUS, reg << 1);
		//CODEC->id = 0;
	} else	
		mcp_reg_write(wm->mcp, AC97_GPIO_STATUS, reg);
		
	spin_unlock(&wm->lock);
}
EXPORT_SYMBOL_GPL(wm97xx_set_codec_gpio);

/*
 * Codec GPIO pin configuration, this set's pin direction, polarity,
 * stickyness and wake up.
 */
void wm97xx_config_codec_gpio (struct wm97xx *wm, u32 gpio, 
									wm97xx_gpio_dir_t dir, wm97xx_gpio_pol_t pol,
	 								wm97xx_gpio_sticky_t sticky, wm97xx_gpio_wake_t wake)
{
	u16 reg;
	spin_lock(&wm->lock);
	
	reg = mcp_reg_read(wm->mcp, AC97_GPIO_POLARITY);
	if (pol ==  WM97XX_GPIO_POL_HIGH)
		reg |= gpio;
	else
		reg &= ~gpio;
	mcp_reg_write(wm->mcp, AC97_GPIO_POLARITY, reg);
	
	reg = mcp_reg_read(wm->mcp, AC97_GPIO_STICKY);	
	if (sticky == WM97XX_GPIO_STICKY)
		reg |= gpio;
	else
		reg &= ~gpio;
	mcp_reg_write(wm->mcp, AC97_GPIO_STICKY, reg);
	
	reg = mcp_reg_read(wm->mcp, AC97_GPIO_WAKEUP);	
	if (wake == WM97XX_GPIO_WAKE)
		reg |= gpio;
	else
		reg &= ~gpio;
	mcp_reg_write(wm->mcp, AC97_GPIO_WAKEUP, reg);
	
	reg = mcp_reg_read(wm->mcp, AC97_GPIO_CFG);	
	if (dir == WM97XX_GPIO_IN)
		reg |= gpio;
	else
		reg &= ~gpio;
	mcp_reg_write(wm->mcp, AC97_GPIO_CFG, reg);
	spin_unlock(&wm->lock);
}
EXPORT_SYMBOL_GPL(wm97xx_config_codec_gpio);

/*
 * Allow arch specific driver to use the codec PENDOWN gpio signal 
 * as a pen down interrupt source. The PENDOWN signal is usually 
 * routed via a cpu irq/gpio.
 * 
 * NOTE: The wm9712 also allows the pen down signal to be sent over
 * the AC97 link or over the codec irq pin. Use wm97xx_enable_codec_irq
 * in this case.
 */
int wm97xx_set_pen_irq (struct wm97xx *wm, int irq)
{	
	spin_lock(&wm->lock);
	if (!wm->pen_irq)
		wm->pen_irq = irq;
	
	spin_unlock(&wm->lock);
	return 0;
}
EXPORT_SYMBOL_GPL(wm97xx_set_pen_irq);

/*
 * Allow arch specific drivers to use the codec AC97 interrupt. This 
 * can either be over the AC97 link or using the codec irq pin.
 * This interrupt can be used to notify arch specific drivers of
 * numerous events i.e. battery level, ADC data, pen down, etc
 */
int wm97xx_set_codec_irq (struct wm97xx *wm, int irq, int ac97_link)
{	
	spin_lock(&wm->lock);
	wm->codec_irq_ref_count++;
	if (!wm->codec_irq) {
		wm->codec_irq = irq;
		wm->ac97_link = ac97_link;
	}
	spin_unlock(&wm->lock);
	return 0;
}
EXPORT_SYMBOL_GPL(wm97xx_set_codec_irq);

void wm97xx_clear_pen_irq (struct wm97xx *wm)
{	
	spin_lock(&wm->lock);
	if (wm->pen_irq)
		wm->pen_irq = 0;	

	spin_unlock(&wm->lock);
}
EXPORT_SYMBOL_GPL(wm97xx_clear_pen_irq);

int wm97xx_clear_codec_irq (struct wm97xx *wm)
{
	spin_lock(&wm->lock);
	if (wm->codec_irq_ref_count == 0)
		goto out;
	
	/* last one ? */
	if (--wm->codec_irq_ref_count == 0) {
		wm->codec_irq = 0;	
	}
	
out:	
	spin_unlock(&wm->lock);
	return wm->codec_irq_ref_count;
}
EXPORT_SYMBOL_GPL(wm97xx_clear_codec_irq);

/* 
 * Register arch specific codec event handler. One handler per event.
 */
int wm97xx_request_codec_event (struct wm97xx *wm, int gpio, codec_event_t work)
{
	u16 reg;
	int irq = generic_ffs(gpio) - 1;
		
	if (irq < 0 || irq > WM97XX_MAX_GPIO)
		return -ENODEV;
	
	spin_lock(&wm->lock);
	if (wm->id  == WM9705_ID2)
		goto err;
	
	/* cant use GPIO2 for irq and gpio at the same time */
	if (!wm->ac97_link && gpio == WM97XX_GPIO_2)
		goto err;
	
	if (wm->codec_event[irq])
		goto err;
	
	wm->codec_event[irq] = work;
	
	/* handle pen down internally */
	if (gpio == WM97XX_GPIO_13)
		wm->pen_irq = wm->codec_irq;
		
	/* set pin to GPIO function */
	reg= mcp_reg_read(wm->mcp, AC97_MISC_AFE);
	mcp_reg_write(wm->mcp, AC97_MISC_AFE, reg | gpio);
	spin_unlock(&wm->lock);
	wm97xx_config_codec_gpio (wm, gpio, WM97XX_GPIO_IN, WM97XX_GPIO_POL_HIGH,
		WM97XX_GPIO_STICKY, WM97XX_GPIO_WAKE);
	
	return 0;
	
err:
	spin_unlock(&wm->lock);
	return -ENODEV;	
}
EXPORT_SYMBOL_GPL(wm97xx_request_codec_event);

/*
 * Free codec event handler
 */
void wm97xx_free_codec_event (struct wm97xx *wm, int gpio)
{
	int irq = generic_ffs(gpio) - 1;
	
	spin_lock(&wm->lock);
	if (wm->id == WM9705_ID2)
		goto out;
	
	if (irq >= 0 && irq < WM97XX_MAX_GPIO)
		wm->codec_event[irq] = NULL;
	
	/* handle pen down internally */	
	if (gpio == WM97XX_GPIO_13)
		wm->pen_irq = 0;

out:
	spin_unlock(&wm->lock);
}
EXPORT_SYMBOL_GPL(wm97xx_free_codec_event);


/*
 * Handle a pen down interrupt.
 */
static void wm97xx_pen_irq_worker (void *ptr)
{
	u32 status = 0;
	struct wm97xx* wm = (struct wm97xx*)ptr;

	/* do we need to enable the touch panel reader */
	if (wm->id == WM9705_ID2) {
		wm->pen_is_down++;
		wake_up_interruptible (&wm->pen_irq_wait);
	} else {
		status = mcp_reg_read(wm->mcp, AC97_GPIO_STATUS);
		if (status & WM97XX_GPIO_13) {
			wm->pen_is_down++;
			wake_up_interruptible (&wm->pen_irq_wait);
		}
	}
}

/*
 * Handle a codec event interrupt. 
 * Can also include pen down depending upon configuration.
 */
static void wm97xx_codec_irq_worker (void* ptr)
{
	int i = WM97XX_MAX_GPIO;
	u16 status, sticky = 0;
	struct wm97xx* wm = (struct wm97xx*)ptr;
	
	status = mcp_reg_read(wm->mcp, AC97_GPIO_STATUS);

	/* check for pen down as we handle it internally
	 * MASK pen down the unmask at pen up */
	if (status & WM97XX_GPIO_13)
		wm97xx_pen_irq_worker(wm);
	
	/* reset codec slot 12 sticky bits and disable active interrupt 
	 * sources by clearing the sticky status.
	 */
	if (wm->id != WM9705_ID2) {
		sticky = mcp_reg_read(wm->mcp, AC97_GPIO_STICKY);
        mcp_reg_write(wm->mcp, AC97_GPIO_STICKY, status ^ sticky);
		if (wm->id  == WM9712_ID2) {
			/* small hack for wm9712 */
			((ac97_t*)wm->mcp->me->platform_data)->num = 1;
			mcp_reg_write(wm->mcp, AC97_GPIO_STATUS, status << 1);
			((ac97_t*)wm->mcp->me->platform_data)->num = 0;
		} else
			mcp_reg_write(wm->mcp, AC97_GPIO_STATUS, status);
	}
	
	/* Iterate through archictecture specific codec workers (if they exist) in 
	 * priority order.
	 */
	while(--i >= 0) {
		if (wm->codec_event[i] && status & 1 << i)
			wm->codec_event[i](status);
	}

	/* turn sticky bits back on, except for pen down */
	if (wm->id != WM9705_ID2)
		mcp_reg_write(wm->mcp, AC97_GPIO_STICKY, sticky & ~WM97XX_GPIO_13);
	enable_irq(wm->codec_irq);
}

/*
 * Codec PENDOWN edge signal irq handler
 */
static irqreturn_t wm97xx_pen_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct wm97xx* wm =(struct wm97xx*) dev_id;
	
	wm->pen_irq_count++;
	queue_work(wm->pen_irq_workq, &wm->pen_event_work);
		
	return IRQ_HANDLED;
}

/*
 * Codec PENDOWN/Power/GPIO/status signal irq handler
 */
static irqreturn_t wm97xx_codec_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{	
	struct wm97xx* wm =(struct wm97xx*) dev_id;

	/* disable IRQ so we can reset codec IRQ status */ 
	disable_irq(irq);
	wm->codec_irq_count++;
	queue_work(wm->codec_event_workq, &wm->codec_event_work);
	
	return IRQ_HANDLED;
}

/*
 * initialise codec IRQ handler and workqueue
 */
static int wm97xx_init_codec_irq (struct wm97xx* wm)
{
	u16 reg;

	INIT_WORK(&wm->codec_event_work, wm97xx_codec_irq_worker, wm);
	if ((wm->codec_event_workq = create_singlethread_workqueue("kwm97ev")) == NULL) {
		err("could not create codec event work queue");
        if (wm->codec_irq == wm->pen_irq)
            wm->pen_irq = 0;
		wm->codec_irq = 0;
		return -EINVAL;
	}
	
	wm->codec_irq_count = 0;
	if (request_irq(wm->codec_irq, wm97xx_codec_interrupt, SA_SHIRQ, "wm97xx-ac97", wm)) {
		err("could not register codec interrupt");
		destroy_workqueue(wm->codec_event_workq);
        if (wm->codec_irq == wm->pen_irq)
            wm->pen_irq = 0;
		wm->codec_irq = 0;
		return -EINVAL;
	}
		
	/* wm9712/13 can interrupt using it's IRQ pin or over the AC97 link */
	if (wm->id != WM9705_ID2) {
		if (wm->ac97_link) { /* use link */
			if (wm->id  == WM9712_ID2) {
				reg = mcp_reg_read(wm->mcp, 0x58);
				mcp_reg_write(wm->mcp, 0x58, reg |= 0x2);
			} else {
				reg = mcp_reg_read(wm->mcp, 0x5a);
				mcp_reg_write(wm->mcp, 0x5a, reg |= 0x2);
			}
		} else { /* use pin */
			reg = mcp_reg_read(wm->mcp, AC97_MISC_AFE);
			mcp_reg_write(wm->mcp, AC97_MISC_AFE, reg &= 0xfffb);
			reg = mcp_reg_read(wm->mcp, AC97_GPIO_CFG);
			mcp_reg_write(wm->mcp, AC97_GPIO_CFG, reg &= 0xfffb);
		}
	}
	
	return 0;
}

/*
 * initialise pen IRQ handler and workqueue
 */
static int wm97xx_init_pen_irq (struct wm97xx* wm)
{
	u16 reg;

	INIT_WORK(&wm->pen_event_work, wm97xx_pen_irq_worker, wm);
	if ((wm->pen_irq_workq = create_singlethread_workqueue("kwm97pen")) == NULL) {
		err("could not create pen irq work queue");
		wm->pen_irq = 0;
		return -EINVAL;
	}

	wm->pen_irq_count = 0;
	if (request_irq(wm->pen_irq, wm97xx_pen_interrupt, SA_SHIRQ, "wm97xx-pen", wm)) {
		err("could not register codec pen down interrupt, will poll for pen down");
		destroy_workqueue(wm->pen_irq_workq);
		wm->pen_irq = 0;
		return -EINVAL;
	}
	
	/* enable PEN down on wm9712/13 */
	if (wm->id  != WM9705_ID2) {
		reg = mcp_reg_read(wm->mcp, AC97_MISC_AFE);
	 	mcp_reg_write(wm->mcp, AC97_MISC_AFE, reg & 0xfff7);
	}

	return 0;
}

/* Private struct for communication between struct wm97xx_tshread and wm97xx_read_samples */
struct ts_state {
	int sleep_time;
	int min_sleep_time;
};

/*
 * When we use an irq to detect a pen down event, we must manually check
 * for a pen up by checking the PDEN bit. If the pen goes up, we have to
 * re-enable the codec pen down detection.
 */
static int wm97xx_irq_pen_check(struct wm97xx* wm, int rc)
{
	u16 pen_status, reg;

	pen_status = mcp_reg_read(wm->mcp, AC97_WM97XX_DIGITISER_RD);
	if (!(pen_status & WM97XX_PEN_DOWN)) {
		rc = RC_PENUP;

		/* Pen is now going to the up position */
		if (wm->pen_is_down && (wm->pen_irq == wm->codec_irq)) {
			reg = mcp_reg_read(wm->mcp, AC97_GPIO_STICKY);
			mcp_reg_write(wm->mcp, AC97_GPIO_STICKY, reg | WM97XX_GPIO_13);
		}
	}
	return rc;
}

static int wm97xx_read_samples(struct wm97xx* wm, struct ts_state *state)
{
	struct wm97xx_data data;
	int rc = RC_PENUP;
			
	spin_lock(&wm->lock);

	if (wm->cont_read_sample) {
		rc = wm->cont_read_sample (&wm->wm97xx_input, wm->pen_is_down, wm->pressure);
		rc = wm97xx_irq_pen_check (wm, rc);
	} else {
		switch (wm->mode) {
		case 0:
			rc = wm->wm97xx_codec->poll_touch (wm, &data);
			break;
		case 1:
			rc = wm->wm97xx_codec->poll_coord (wm, &data);
			break;
		}
	}
	
	if (rc & RC_PENUP) {
		if (wm->pen_is_down) {
			wm->pen_is_down = 0;
			dbg("pen up");
			input_report_abs(&wm->wm97xx_input, ABS_PRESSURE, 0);
			input_sync(&wm->wm97xx_input);
		} else if (!(rc & RC_AGAIN)) {
			/* We need high frequency updates only while pen is down,
			 * the user never will be able to touch screen faster than
			 * a few times per second... On the other hand, when the
			 * user is actively working with the touchscreen we don't
			 * want to lose the quick response. So we will slowly
			 * increase sleep time after the pen is up and quicky
			 * restore it to ~one task switch when pen is down again.
			 */
			if (state->sleep_time < HZ/10)
				state->sleep_time++;
		}
	} else if (rc & RC_VALID) {
		printk("pen down: x=%x:%d, y=%x:%d, pressure=%x:%d\n",
		    data.x >> 12, data.x & 0xfff,
		    data.y >> 12, data.y & 0xfff,
		    data.p >> 12, data.p & 0xfff);
	
		input_report_abs (&wm->wm97xx_input, ABS_X, data.x & 0xfff);
		input_report_abs (&wm->wm97xx_input, ABS_Y, data.y & 0xfff);
		input_report_abs (&wm->wm97xx_input, ABS_PRESSURE, data.p & 0xfff);
		input_sync (&wm->wm97xx_input);
		wm->pen_is_down++;
		state->sleep_time = state->min_sleep_time;
	} else if (rc & RC_PENDOWN) {
		dbg("pen down");
		wm->pen_is_down++;
		state->sleep_time = state->min_sleep_time;
	}

	spin_unlock(&wm->lock);
	return rc;
}

/*
 * The touchscreen sample reader thread.
 */
static int wm97xx_ts_read(void* data)
{
	int rc;
	struct ts_state state;
	struct wm97xx* wm =(struct wm97xx*) data;

	/* set up thread context */
	wm->ts_task = current;
	daemonize("kwm97xxts");

	if(wm->wm97xx_codec == NULL) {
		wm->ts_task = NULL;
		printk(KERN_ERR "codec is NULL, bailing\n");
	}
	complete(&wm->ts_init);

	wm->pen_is_down = 0;
	state.min_sleep_time = HZ >= 100 ? HZ/100 : 1;
	if (state.min_sleep_time < 1)
		state.min_sleep_time = 1;
	state.sleep_time = state.min_sleep_time;

	/* touch reader loop */
	while (wm->ts_task) {
		do {
			rc = wm97xx_read_samples (wm, &state);
		} while (rc & RC_AGAIN);
		if (!wm->pen_is_down && wm->pen_irq) {	
			/* Nice, we don't have to poll for pen down event */
			wait_event_interruptible (wm->pen_irq_wait, wm->pen_is_down);
		} else {
			set_task_state(current, TASK_INTERRUPTIBLE);
			schedule_timeout(state.sleep_time);
		}
	}
	complete_and_exit(&wm->ts_exit, 0);
}

/*
 * Start the touchscreen thread and 
 * the touch digitiser.
 */
int wm97xx_ts_input_open(struct input_dev *idev)
{
	int ret = 0;
	struct wm97xx* wm =(struct wm97xx*) idev->private;
    u16 reg = 0;

	if (wm->wm97xx_codec == NULL) {
		printk("error: wm->wm97xx_codec == NULL\n");
		return -EINVAL;
	}

	down(&wm->sem);
	/* first time opened ? */
	if (wm->ts_use_count++ == 0) {

      /* start touchscreen thread */
        init_completion(&wm->ts_init);
        init_completion(&wm->ts_exit);

        ret = kernel_thread(wm97xx_ts_read, wm, CLONE_KERNEL);
        if (ret >= 0) {
            wait_for_completion(&wm->ts_init);
            if (wm->ts_task == NULL)
                ret = -EINVAL;
        }
        if (ret < 0)
            goto out;
        
		/* start digitiser */
		spin_lock(&wm->lock);
		wm->wm97xx_codec->digitiser_start(wm);
		
		/* init pen down/up irq handling */
		if (wm->codec_irq || wm->pen_irq) {
			if (wm->codec_irq == wm->pen_irq) {
                if (wm97xx_init_codec_irq(wm) == 0) {
				    reg = mcp_reg_read(wm->mcp, AC97_GPIO_STICKY);
				    mcp_reg_write(wm->mcp, AC97_GPIO_STICKY, reg | WM97XX_GPIO_13);
                } 
			} else {
				wm97xx_init_pen_irq(wm);
		    }
            if (wm->pen_irq == 0 ) {
                /* we failed to get an irq for pen down events,
                * so we resort to polling. kickstart the reader */
                wm->pen_is_down = 1;
                wake_up_interruptible (&wm->pen_irq_wait);
		    }
        }
		spin_unlock(&wm->lock);
      
    }
        
out:
	up(&wm->sem);
	return ret;
}
EXPORT_SYMBOL_GPL(wm97xx_ts_input_open);

/*
 * Kill the touchscreen thread and stop
 * the touch digitiser.
 */
void wm97xx_ts_input_close(struct input_dev *idev)
{
	struct wm97xx* wm =(struct wm97xx*) idev->private;
	
	down(&wm->sem);
	if (--wm->ts_use_count == 0) {
		/* destroy workqueue and free irq's */
		if (wm->codec_irq || wm->pen_irq) {
			if (wm->pen_irq == wm->codec_irq) {
				free_irq(wm->codec_irq, wm);
				destroy_workqueue(wm->codec_event_workq);
			} else {
				free_irq(wm->pen_irq, wm);
				destroy_workqueue(wm->pen_irq_workq);
			}
		}

		/* kill thread */
		wm->ts_task = NULL;
		wm->pen_is_down = 1;
		wake_up_interruptible (&wm->pen_irq_wait);
		wait_for_completion(&wm->ts_exit);
		wm->pen_is_down = 0;
		
		/* stop digitiser */
		wm->wm97xx_codec->digitiser_stop(wm);
		
	}
	up(&wm->sem);
}
EXPORT_SYMBOL_GPL(wm97xx_ts_input_close);

static int wm97xx_probe(struct mcp *mcp)
{
	struct wm97xx *wm = NULL;
	unsigned int id;
	int ret = -ENODEV;
	dbg("probing..\n");

	mcp_enable(mcp);
	id = mcp_reg_read(mcp, AC97_VENDOR_ID1);
	if (id != WM97XX_ID1) {
		printk(KERN_WARNING "WM97xx ID not found: %04x\n", id);
		goto err_disable;
	}

	wm = kmalloc(sizeof(struct wm97xx), GFP_KERNEL);
	ret = -ENOMEM;
	if (!wm)
		goto err_disable;

	memset(wm, 0, sizeof(struct wm97xx));
	init_MUTEX(&wm->sem);
	spin_lock_init (&wm->lock);
	init_waitqueue_head (&wm->pen_irq_wait);

	wm->cdev.class = &wm97xx_class;
	wm->cdev.dev = mcp->attached_device;
	wm->cdev.class_data = wm;
    wm->id = mcp_reg_read(mcp, AC97_VENDOR_ID2);
	strlcpy(wm->cdev.class_id, "wm97xx", sizeof(wm->cdev.class_id));

	wm->mcp = mcp;
	mcp_set_drvdata(wm->mcp, wm);

	ret = class_device_register(&wm->cdev);
	if (ret)
		kfree(wm);

 err_disable:
	mcp_disable(wm->mcp);
	return ret;
}

static void wm97xx_remove(struct mcp *mcp)
{
	struct wm97xx *wm = mcp_get_drvdata(mcp);

	class_device_unregister(&wm->cdev);
	
	/* Stop touch read thread */
	if (wm->ts_task) {
		wm->ts_task = NULL;
		wm->pen_is_down = 1;
		wake_up_interruptible (&wm->pen_irq_wait);
		wait_for_completion(&wm->ts_exit);
	}
}

static void wm97xx_release(struct class_device *dev)
{
	struct wm97xx *wm = classdev_to_wm97xx(dev);
	kfree(wm);
}

static struct class wm97xx_class = {
	.name		= "wm97xx",
	.release	= wm97xx_release,
};

int wm97xx_register_interface(struct class_interface *intf)
{
	intf->class = &wm97xx_class;
	return class_interface_register(intf);
}
EXPORT_SYMBOL_GPL(wm97xx_register_interface);

void wm97xx_unregister_interface(struct class_interface *intf)
{
	class_interface_unregister(intf);
}
EXPORT_SYMBOL_GPL(wm97xx_unregister_interface);

static struct mcp_driver wm97xx_driver = {
	.drv		= {
		.name	= "wm97xx",
	},
	.probe		= wm97xx_probe,
	.remove		= wm97xx_remove,
};

static int __init wm97xx_init(void)
{
	int ret;
   
    if ((ret = class_register(&wm97xx_class)) != 0)
        goto err;
 
	if ((ret = mcp_driver_register(&wm97xx_driver)) != 0) {
		dbg ("mcp_driver_register failed.\n");
        	goto mcp_err;
	}

    wm97xx_register_interface(&wm97xx_aux_adc_interface);
	dbg ("initialized\n");
    return 0;
    
mcp_err:
    class_unregister(&wm97xx_class);
err:
	return ret;
}

static void __exit wm97xx_exit(void)
{
    wm97xx_unregister_interface(&wm97xx_aux_adc_interface);
	mcp_driver_unregister(&wm97xx_driver);
	class_unregister(&wm97xx_class);
}

module_init(wm97xx_init);
module_exit(wm97xx_exit);

/* Module information */
MODULE_AUTHOR("Liam Girdwood, liam.girdwood@wolfsonmicro.com, www.wolfsonmicro.com");
MODULE_DESCRIPTION("WM97xx Core - Touch Screen / AUX ADC / GPIO Driver");
MODULE_LICENSE("GPL");
