/*
 * glencoe_periph.c: adds entries to /proc fs to provide manual 
 * access&control to peripherals on Glencoe (LEDs, LCD backlight,
 * card detection, ...)
 * 
 * - write 1 to assert, 0 to de-assert pins
 *
 *  Copyright (c) 2005 Matthias Ihmig <m.ihmig@mytum.de>
 *
 * This program is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your
 * option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more
 * details.
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>
#include <asm/uaccess.h>

#include <asm/hardware.h>
#include <asm/arch/pxa-regs.h>
#include <asm/arch/glencoe.h>

#define MODULE_VERS "1.0"
#define MODULE_NAME "glencoe_periph"

static struct proc_dir_entry *gem_dir, *gpio0, *usim, *mmc, *cf1, *cf2, *green, *blue, *bl, *lcd, *lcd_pwm;

static int proc_read_gpio(char *page, char **start, off_t off, int count,
			     int *eof, void *data)
{
	return sprintf(page, "%s\n", (GPIO_lev( (int)data) ? "1" : "0")); // active high signal
}
static int proc_read_ngpio(char *page, char **start, off_t off, int count,
			     int *eof, void *data)
{
	return sprintf(page, "%s\n", (GPIO_lev( (int)data) ? "0" : "1")); // active low signal -> invert
}

static int proc_write_gpio(struct file *file, const char *buffer,
			     unsigned long count, void *data)
{
	int len=0;
	char value='0';

	if(count >= 1)
		len = 1;
	else
		return -EFAULT;

	if(copy_from_user(&value, buffer, len))
		return -EFAULT;

	if (value == '1')
		GPIO_set( (int)data );		// assert pin (active high)
	if (value == '0')
		GPIO_clr( (int)data );		// deassert pin
	return len;
}
static int proc_write_ngpio(struct file *file, const char *buffer,
			     unsigned long count, void *data)
{
	int len=0;
	char value='0';

	if(count >= 1)
		len = 1;
	else
		return -EFAULT;

	if(copy_from_user(&value, buffer, len))
		return -EFAULT;

	if (value == '1')
		GPIO_clr( (int)data );		// assert pin (active low)
	if (value == '0')
		GPIO_set( (int)data );		// deassert pin
	return len;
}

static int proc_read_pwm(char *page, char **start, off_t off, int count,
			     int *eof, void *data)
{
	return sprintf(page, "current:%d max:%d\n", GLN_BACKLIGHT_PWM_PWDUTY, GLN_BACKLIGHT_PWM_PERVAL);
}

static int proc_write_pwm(struct file *file, const char *buffer,
			     unsigned long count, void *data)
{
	int len=count, width;
	char value[10];

	if(count > 9)
		len = 9;
	else
		len = count;

	if(copy_from_user(&value, buffer, len))
		return -EFAULT;
	
	if (sscanf(value, "%d", &width) != 1)
		return -EINVAL;

	if ( (width >= 0) && (width <= GLN_BACKLIGHT_PWM_PERVAL) ) {
		pxa_set_cken(GLN_BACKLIGHT_PWM_CKEN, 1);
		GLN_BACKLIGHT_PWM_CTRL = 0;
		GLN_BACKLIGHT_PWM_PWDUTY = width;
		GLN_BACKLIGHT_PWM_PERVAL = 0x3ff;
	}
	return len;
}

static int __init init_glencoe_periph(void)
{
	int rv = 0;
	
	// Initialize GPIO registers
	pxa_gpio_mode(GLN_GPIO_GPIO0         | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nUSIM_CD_CPU  | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nMMC_CARD_DET | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nGREEN_LED    | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_nBLUE_LED     | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_BL_PWR_ON     | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_LCD_PWR_ON    | GPIO_OUT);
	pxa_gpio_mode(GLN_GPIO_nCF1_CARD_DET | GPIO_IN);
	pxa_gpio_mode(GLN_GPIO_nCF2_CARD_DET | GPIO_IN);
	pxa_gpio_mode(GLN_BACKLIGHT_PWM_MD);

	/* create directory */
	gem_dir = proc_mkdir("driver/" MODULE_NAME, NULL);
	if(gem_dir == NULL) {
		rv = -ENOMEM; goto out;
	}
	gem_dir->owner = THIS_MODULE;
	
	// input files
	gpio0 = create_proc_read_entry("gpio0_state", 0444, gem_dir, proc_read_gpio, (void*)GLN_GPIO_GPIO0);
	if(gpio0 == NULL) {
		rv  = -ENOMEM; goto no_gpio0;
	}
	gpio0->owner = THIS_MODULE;

	usim = create_proc_read_entry("sim_inserted", 0444, gem_dir, proc_read_ngpio, (void*)GLN_GPIO_nUSIM_CD_CPU);
	if(usim == NULL) {
		rv  = -ENOMEM; goto no_usim;
	}
	usim->owner = THIS_MODULE;
	
	mmc = create_proc_read_entry("mmc_inserted", 0444, gem_dir, proc_read_ngpio, (void*)GLN_GPIO_nMMC_CARD_DET);
	if(mmc == NULL) {
		rv  = -ENOMEM; goto no_mmc;
	}
	mmc->owner = THIS_MODULE;

	cf1 = create_proc_read_entry("cf1_inserted", 0444, gem_dir, proc_read_ngpio, (void*)GLN_GPIO_nCF1_CARD_DET);
	if(cf1 == NULL) {
		rv  = -ENOMEM; goto no_cf1;
	}
	cf1->owner = THIS_MODULE;

	cf2 = create_proc_read_entry("cf2_inserted", 0444, gem_dir, proc_read_ngpio, (void*)GLN_GPIO_nCF2_CARD_DET);
	if(cf2 == NULL) {
		rv  = -ENOMEM; goto no_cf2;
	}
	cf2->owner = THIS_MODULE;


	// output files
	green = create_proc_entry("green_led_on", 0644, gem_dir);
	if(green == NULL) {
		rv = -ENOMEM;
		goto no_green;
	}
	green->data = (void*)GLN_GPIO_nGREEN_LED;
	green->read_proc = proc_read_ngpio;
	green->write_proc = proc_write_ngpio;
	green->owner = THIS_MODULE;
	
	blue = create_proc_entry("blue_led_on", 0644, gem_dir);
	if(blue == NULL) {
		rv = -ENOMEM;
		goto no_blue;
	}
	blue->data = (void*)GLN_GPIO_nBLUE_LED;
	blue->read_proc = proc_read_ngpio;
	blue->write_proc = proc_write_ngpio;
	blue->owner = THIS_MODULE;
		
	bl = create_proc_entry("backlight_on", 0644, gem_dir);
	if(bl == NULL) {
		rv = -ENOMEM;
		goto no_bl;
	}
	bl->data = (void*)GLN_GPIO_BL_PWR_ON;
	bl->read_proc = proc_read_gpio;
	bl->write_proc = proc_write_gpio;
	bl->owner = THIS_MODULE;
	
	lcd = create_proc_entry("lcd_on", 0644, gem_dir);
	if(lcd == NULL) {
		rv = -ENOMEM;
		goto no_lcd;
	}
	lcd->data = (void*)GLN_GPIO_LCD_PWR_ON;
	lcd->read_proc = proc_read_gpio;
	lcd->write_proc = proc_write_gpio;
	lcd->owner = THIS_MODULE;

	lcd_pwm = create_proc_entry("lcd_pwm", 0644, gem_dir);
	if(lcd_pwm == NULL) {
		rv = -ENOMEM;
		goto no_lcd_pwm;
	}
	lcd_pwm->data = NULL;
	lcd_pwm->read_proc = proc_read_pwm;
	lcd_pwm->write_proc = proc_write_pwm;
	lcd_pwm->owner = THIS_MODULE;


	/* everything OK */
	printk(KERN_INFO "%s %s proc interface initialised\n",
	       MODULE_NAME, MODULE_VERS);
	return 0;

no_lcd_pwm:
	remove_proc_entry("lcd_on", gem_dir);
no_lcd:
	remove_proc_entry("backlight_on", gem_dir);
no_bl:
	remove_proc_entry("blue_led_on", gem_dir);
no_blue:
	remove_proc_entry("green_led_on", gem_dir);
no_green:

	remove_proc_entry("cf2_inserted", gem_dir);
no_cf2:
	remove_proc_entry("cf1_inserted", gem_dir);
no_cf1:
	remove_proc_entry("mmc_inserted", gem_dir);
no_mmc:
	remove_proc_entry("sim_inserted", gem_dir);
no_usim:
	remove_proc_entry("gpio0_state", gem_dir);
no_gpio0:
	remove_proc_entry(MODULE_NAME, NULL);
out:
	return rv;
}

static void __exit cleanup_glencoe_periph(void)
{
	remove_proc_entry("lcd_pwm", gem_dir);
	remove_proc_entry("lcd_on", gem_dir);
	remove_proc_entry("backlight_on", gem_dir);
	remove_proc_entry("blue_led_on", gem_dir);
	remove_proc_entry("green_led_on", gem_dir);
	remove_proc_entry("cf2_inserted", gem_dir);
	remove_proc_entry("cf1_inserted", gem_dir);
	remove_proc_entry("mmc_inserted", gem_dir);
	remove_proc_entry("sim_inserted", gem_dir);
	remove_proc_entry("gpio0_state", gem_dir);
	remove_proc_entry(MODULE_NAME, NULL);

	printk(KERN_INFO "%s %s interface removed\n",
	       MODULE_NAME, MODULE_VERS);
}

module_init(init_glencoe_periph);
module_exit(cleanup_glencoe_periph);

MODULE_AUTHOR("Matthias Ihmig <m.ihmig@mytum.de>");
MODULE_DESCRIPTION("Glencoe GPIO peripherals /proc interface");
MODULE_LICENSE("GPL");

