/*
 *  linux/drivers/input/keyboard/mainstone_kp.c
 *
 *  Keypad driver for mainstone Platform
 *
 *  Copyright (C) 2004, Intel Corporation (jingqing.xu@intel.com)
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <asm/irq.h>
#include <asm/hardware.h> 
#include <asm/arch/pxa-regs.h>
#include <linux/device.h>
MODULE_AUTHOR("Xu Jingqing <Jingqing.Xu@intel.com>");
MODULE_DESCRIPTION("Mainstone III keypad driver");
MODULE_LICENSE("GPL");
 
static unsigned char mainstone_keycode[0x78] = {
	[0]	 = KEY_A,
	[1]	 = KEY_G,
	[2]	 = KEY_M,
	[3]	 = KEY_S,
	[4]	 = KEY_DOT,
	[5]	 = KEY_HOME,
	[0xA]    = KEY_UP,
        [0xB]    = KEY_DOWN,
        [0XC]    = KEY_KPENTER,
	[0x10]	 = KEY_B,
	[0x11]	 = KEY_H,
	[0x12]	 = KEY_N,
	[0x13]	 = KEY_T,
	[0x14]	 = KEY_EMAIL,
	[0x15]	 = KEY_UP,
	[0x20]	 = KEY_C,
	[0x21]	 = KEY_I,
	[0x22]	 = KEY_O,
	[0x23]	 = KEY_U,
	[0x24]	 = KEY_Y,
	[0x25]	 = KEY_SPACE,
	[0x30]	 = KEY_D,
	[0x31]	 = KEY_J,
	[0x32]	 = KEY_P,
	[0x33]	 = KEY_V,
	[0x34]	 = KEY_Z,
	[0x35]	 = KEY_SPACE,
	[0x40]	 = KEY_E,
	[0x41]	 = KEY_K,
	[0x42]	 = KEY_Q,
	[0x43]	 = KEY_W,
	[0x44]	 = KEY_SLASH,
	[0x45]	 = KEY_KPENTER,
	[0x50]	 = KEY_F,
	[0x51]	 = KEY_L,
	[0x52]	 = KEY_R,
	[0x53]	 = KEY_X,
	[0x54]	 = KEY_BACKSLASH,
	[0x55]	 = KEY_ESC,
};
static struct input_dev mainstone_kp_dev;
/* scan code */
#define ROTARY_DEFAULT			0x7F
#define NO_KEY				0xFF
#define SCAN_CODE_SCROLL_UP		0xA
#define SCAN_CODE_SCROLL_DOWN		0xB
#define SCAN_CODE_ACTION		0xC

static int kp_direct_scan(void)
{
	static int prev;
	u32 curr;
	u32 direct, rotary;
	u8 c;
        direct 	= KPDK;
	rotary 	= KPREC;
	/* Rotary */
	curr = rotary & 0xFF;
	if (rotary & KPREC_OF0) {
		KPREC 	&= ~KPREC_OF0;
		KPREC	|= ROTARY_DEFAULT;
		prev 	= ROTARY_DEFAULT;
		c 	= SCAN_CODE_SCROLL_UP;
	}
	else if (rotary & KPREC_UF0) {
		KPREC 	&= ~KPREC_UF0;
		KPREC	|= ROTARY_DEFAULT;
		prev 	= ROTARY_DEFAULT;
		c 	= SCAN_CODE_SCROLL_DOWN;
	}
	else if (curr > prev) {
		c 	= SCAN_CODE_SCROLL_UP;
		prev	= curr;
	}
	else if (curr < prev) {
		c	= SCAN_CODE_SCROLL_DOWN;
		prev	= curr;
	}

	if (direct & KPDK_DK2) {
		c	= SCAN_CODE_ACTION;
		prev	= curr;
	}

	return 0;
}

static int kp_matrix_scan(void)
{
	u32 autoscan, n;
	u8 c;
	autoscan = KPAS;
	n = (autoscan >> 26) & 0x1f;
	c = autoscan & 0xff;
	/* encode scan-code */
	if ( n == 1 ){
        	input_report_key(&mainstone_kp_dev,mainstone_keycode[c], 1 );
        	input_report_key(&mainstone_kp_dev,mainstone_keycode[c], 0 );
 		return 0;
	}
	else 
		return 1;
}

/* Interrupt Handler for KEYPAD */
static irqreturn_t kp_interrupt(int irq, void *ptr, struct pt_regs *regs) 
{
	unsigned long kpc_val;

	/* ack interrupt */
	kpc_val = KPC;
        
	/* direct interrupt */
	if (kpc_val & KPC_DI) 
                kp_direct_scan();

	/* matrix interrupt */
	if (kpc_val & KPC_MI) 
		kp_matrix_scan();

        return IRQ_HANDLED;

}
static void kpad_configure()
{
	pxa_gpio_mode(93 | GPIO_ALT_FN_1_IN);   /* DKIN0 */
	pxa_gpio_mode(94 | GPIO_ALT_FN_1_IN);   /* DKIN1 */
	pxa_gpio_mode( 98 | GPIO_ALT_FN_3_IN); 	/* MKIN4 */
	pxa_gpio_mode( 99 | GPIO_ALT_FN_3_IN);  /* MKIN5 */
        pxa_gpio_mode(100 | GPIO_ALT_FN_1_IN);  /* MKIN0 */
 	pxa_gpio_mode(101 | GPIO_ALT_FN_1_IN);  /* MKIN1 */
	pxa_gpio_mode(102 | GPIO_ALT_FN_1_IN);  /* MKIN2 */
	pxa_gpio_mode( 97 | GPIO_ALT_FN_3_IN);  /* MKIN3 */
	pxa_gpio_mode(103 | GPIO_ALT_FN_2_OUT); /* MKOUT0 */
	pxa_gpio_mode(104 | GPIO_ALT_FN_2_OUT); /* MKOUT1 */
	pxa_gpio_mode(105 | GPIO_ALT_FN_2_OUT); /* MKOUT2 */
	pxa_gpio_mode(106 | GPIO_ALT_FN_2_OUT); /* MKOUT3 */
	pxa_gpio_mode(107 | GPIO_ALT_FN_2_OUT); /* MKOUT4 */
	pxa_gpio_mode(108 | GPIO_ALT_FN_2_OUT); /* MKOUT5 */

	/* enable unit clock */
	CKEN |= CKEN19_KEYPAD;

//FIXME:PXA27x E11
#ifdef CONFIG_PXA27x_E11
	/* debouce setting: matrix debouce + direct debouce */
   	KPKDI= 10 + (2 << 8);  
#else
	KPKDI= 2 << 8;
#endif 
	/* set scroll wheel value to mid-point value */
	KPREC= ROTARY_DEFAULT;

	/* keypad control register */
	KPC = (KPC_ASACT | (5<<26) | (5<<23)|
		KPC_ME | (2<<6) | KPC_REE0 | KPC_DE | 
		KPC_MS0 | KPC_MS1 | KPC_MS2 | KPC_MS3| 
                KPC_MS4 | KPC_MS5 | KPC_MIE | KPC_DIE | KPC_IMKP);


}
static int mb_kp_resume( struct device * dev, u32 level )
{	
	switch(level){
	case RESUME_POWER_ON:
		kpad_configure();
		break;
	}
	
	return 0;

}

static int mb_kp_suspend(struct device * dev, u32 state, u32 level)
{
	switch(level){
	case SUSPEND_POWER_DOWN:
		CKEN &= ~CKEN19_KEYPAD;
		break;
	}
	
	return 0;


}

static struct device_driver mb_kp_drv = {
		.bus = &platform_bus_type,
		.name = "mb_kp", 
		.resume = mb_kp_resume,
		.suspend = mb_kp_suspend,
		};
static struct platform_device * mb_kp_platform_device;
static int __init mainstone_kp_init( void )
{
        int i,err;
      
	init_input_dev(&mainstone_kp_dev);

	mainstone_kp_dev.evbit[0] = BIT(EV_KEY);
	mainstone_kp_dev.keycode = mainstone_keycode;
	mainstone_kp_dev.keycodesize = sizeof(unsigned char);
	mainstone_kp_dev.keycodemax = ARRAY_SIZE(mainstone_keycode);

	for (i = 0; i < 0x55; i++)
		if (mainstone_keycode[i])
			set_bit(mainstone_keycode[i], mainstone_kp_dev.keybit);

	err = request_irq (IRQ_KEYPAD, kp_interrupt, 0, "Keypad", NULL);

	if (err) {
		printk (KERN_CRIT "Wow!  Can't register IRQ[%d] for Keypad\n", IRQ_KEYPAD);
		return -1;
	}
	kpad_configure();
	mainstone_kp_dev.name = "mainstoneIII keypad";
	mainstone_kp_dev.phys = "mainstoneIII/input0";
	mainstone_kp_dev.id.vendor = 0x0001;
	mainstone_kp_dev.id.product = 0x0001;
	mainstone_kp_dev.id.version = 0x0001;

	input_register_device(&mainstone_kp_dev);
	printk(KERN_INFO "input: mainstoneIII keypad!\n");
	mb_kp_platform_device = platform_device_register_simple("mb_kp",0,0,0);
	printk(KERN_INFO "platform bus: mainstoneIII keypad device!\n");
	driver_register(&mb_kp_drv);
	printk(KERN_INFO "input: mainstoneIII keypad driver!\n");
	return 0;
   
}

static void __exit mainstone_kp_exit( void )
{
	platform_device_unregister(mb_kp_platform_device);
	driver_unregister(&mb_kp_drv);
	input_unregister_device(&mainstone_kp_dev);
	free_irq(IRQ_KEYPAD, NULL);
}
module_init(mainstone_kp_init);
module_exit(mainstone_kp_exit);

 
