/* 
 * linux/arch/arm/mach-pxa/vcs.c
 *
 * Bulverde voltage change sequencer driver.
 *
 * Copyright (C) 2003-2004 Intel Corporation.
 *
 * Author: Cain Yuan <cain.yuan@intel.com>
 * 
 * This software program is licensed subject to the GNU General Public License
 * (GPL).Version 2,June 1991, available at http://www.fsf.org/copyleft/gpl.html
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/init.h>
#include <linux/cpufreq.h>
#include <asm/hardware.h>
#include <asm/arch/pxa-regs.h>
#include <asm/io.h>

/*
 * Functionality: Initialize PWR I2C. 
 * Argument:      None
 * Return:        void
 */
static int __init vcs_init(void)
{
	CKEN |= 0x1 << 15;
	CKEN |= 0x1 << 14;
	PCFR = PCFR_PI2CEN;
//	PCFR = 0x60;
	return 0;
}

#ifdef CONFIG_PXA27x_E28

/*
 * The voltage change process is the workround for 
 * E28: Core hangs during voltage change when there are outstanding
 * transaction on the bus.
 */
static void pxa27x_change_voltage(void)
{
	unsigned long flags;
	volatile int *ramstart;
	unsigned int unused;

	printk("Workround for E28 about Core hangs during voltage change when there are outstanding transaction on the bus.\n If you want to disable workround, DO NOT set \"CONFIG_PXA27x_E28=y\"\n");

	/* map the first page of sdram to an uncached virtual page */
	ramstart = (int *)ioremap(PHYS_OFFSET, 4096);

	local_irq_save(flags);

	__asm__ __volatile__("\n\
		@ WORKAROUND - Core hangs on voltage change at different \n\
		@ alignments and at different core clock frequencies \n\
		@ To ensure that no external fetches occur, we want to \n\
		@ store the next several instructions that occur after the \n\
		@ voltage change inside the cache. The load dependency \n\
		@ stall near the retry label ensures that any outstanding \n\
		@ instruction cacheline loads are complete before \n\
		@ the mcr instruction is executed on the 2nd pass. \n\
		@ This procedure ensures us that the internal bus will not \n\
		@ be busy. \n\
			\n\
		b       2f              \n\
		nop                 \n\
		.align  5               \n\
	2:                  \n\
		ldr     r0, [%1]        @ APB register read and compare \n\
		cmp     r0, #0          @ fence for pending slow apb reads \n\
		\n\
		mov     r0, #8          @ VC bit for PWRMODE \n\
		movs    r1, #1          @ don't execute mcr on 1st pass \n\
		\n\
		@ %1 points to uncacheable memory to force memory read \n\
		\n\
	retry:                  \n\
		ldreq   r3, [%2]        @ only stall on the 2nd pass\n\
		cmpeq   r3, #0          @ cmp causes fence on mem transfers\n\
		cmp     r1, #0          @ is this the 2nd pass? \n\
		mcreq   p14, 0, r0, c7, c0, 0   @ write to PWRMODE on 2nd pass only \n\
		\n\
		@ Read VC bit until it is 0, indicates that the VoltageChange is done.\n\
		@ On first pass, we never set the VC bit, so it will be clear already.\n\
		\n\
	VoltageChange_loop:         \n\
		mrc     p14, 0, r3, c7, c0, 0   \n\
		tst     r3, #0x8            \n\
		bne     VoltageChange_loop      \n\
		\n\
		subs    r1, r1, #1     @ update conditional execution counter\n\
		beq     retry"

		: "=&r" (unused)
		: "r" (&CCCR), "r" (ramstart)
		: "r0", "r1", "r3" );

	local_irq_restore(flags);

	/* unmap the page we used */
	iounmap(ramstart);
}

#else

static void pxa27x_change_voltage(void)
{
	unsigned long flags;

	local_irq_save(flags);

	__asm__ __volatile__(" \n\
		mrc     p14, 0, r0, c7, c0, 0           @ read c7 \n\
		orr     r0, r0, #0x8                    @ Voltage change sequence begins \n\
		mcr     p14, 0, r0, c7, c0, 0           @ set the bit. \n\
		mrc     p14, 0, r0, c7, c0, 0           @ read c7 \n\
		mov             r0,     r0 \n\
		nop \n\
		nop"
		:
		:
		:"r0" );

	local_irq_restore(flags);
}

#endif

static void clr_all_sqc(void)
{
	int i = 0;
	for (i = 0; i < 32; i++)
		PCMD(i) &= ~PCMD_SQC;
}

static void clr_all_mbc(void)
{
	int i = 0;
	for (i = 0; i < 32; i++)
		PCMD(i) &= ~PCMD_MBC;
}

static void clr_all_dce(void)
{
	int i = 0;
	for (i = 0; i < 32; i++)
		PCMD(i) &= ~PCMD_DCE;
}

static void set_mbc_bit(int ReadPointer, int NumOfBytes)
{
	PCMD0 |= PCMD_MBC;
	PCMD1 |= PCMD_MBC;
}

static void set_lc_bit(int ReadPointer, int NumOfBytes)
{
	PCMD0 |= PCMD_LC;
	PCMD1 |= PCMD_LC;
	PCMD2 |= PCMD_LC;
}

static void set_cmd_data(unsigned char *DataArray, int StartPoint, int size)
{
	PCMD0 &= 0xFFFFFF00;
	PCMD0 |= DataArray[0];
	PCMD1 &= 0xFFFFFF00;
	PCMD1 |= DataArray[1];
	PCMD2 &= 0xFFFFFF00;
	PCMD2 |= DataArray[2];
}

static void ipm_power_change_cmd(unsigned int DACValue)
{
	unsigned char dataArray[3];

	dataArray[0] = 0;  /* Command 0 */
	dataArray[1] = (DACValue & 0x000000FF);	/* data LSB */
	dataArray[2] = (DACValue & 0x0000FF00) >> 8; /* data MSB */

	PVCR = 0;

	PCFR &= ~PCFR_FVC;
	PVCR &= 0xFFFFF07F;		/* no delay is necessary */
	PVCR &= 0xFFFFFF80;		/* clear slave address */
	PVCR |= 0x20;			/* set slave address */

	PVCR &= 0xFE0FFFFF;		/* clear read pointer 0	*/
	PVCR |= 0;

	/* DCE and SQC are not necessary for single command */
	clr_all_sqc();	
	clr_all_dce();

	clr_all_mbc();
	set_mbc_bit(0, 2);

	/* indicate the last byte of this command is holded in this register	*/
	PCMD2 &= ~PCMD_MBC;

	/* indicate this is the first command and last command also */
	set_lc_bit(0, 3);

	/* programming the command data bit */
	set_cmd_data(dataArray, 0, 2);
}

void vm_setvoltage(unsigned int DACValue)
{
	printk("vm_setvoltage\n");
	ipm_power_change_cmd(DACValue);
	/* Enable Power I2C */
        PCFR |= PCFR_PI2CEN;
	/* Execute voltage change sequence	*/
	pxa27x_change_voltage();	//set VC on the PWRMODE on CP14
}

/*
 * Prepare for a coupled voltage & frequency change
 */	
void vm_pre_setvoltage(unsigned int DACValue)
{
        ipm_power_change_cmd(DACValue);
        /* Enable Power I2C */
        PCFR |= (PCFR_PI2CEN | PCFR_FVC);
}


module_init(vcs_init);
