/*
 * This function provides the implementation of the access functions to 
 * the Performance Monitoring Unit on all CPUs based on the XScale core.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * Copyright (c) 2003 Intel Corporation.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <asm/atomic.h>
#include <asm/arch/xscale-pmu.h>
#include <asm/arch/irqs.h>

static atomic_t usage = ATOMIC_INIT(0);
static unsigned long id = 0;
struct pmu_results results;

static u32 pmnc;
static u32 evtsel;

#define PMNC 0    /*	PMU Control Register	*/
#define CCNT 1    /*	PMU Clock Counter (CCNT)	*/
#define PMN0 2    /*	PMU Count Register 0 (PMN0)	*/
#define PMN1 3    /*	PMU Count Register 1 (PMN1)	*/
#define PMN2 4
#define PMN3 5
#define INTEN	6
#define FLAG	7
#define EVTSEL	8

extern void pmu_reg_write(unsigned long regno, unsigned long value);
extern unsigned long pmu_reg_read(unsigned long regno);

#define	PMU_ENABLE	0x001	/* Enable counters */
#define PMN_RESET	0x002	/* Reset event counters */
#define	CCNT_RESET	0x004	/* Reset clock counter */
#define	PMU_RESET	CCNT_RESET | PMN_RESET
#define	CLK_DIV		0x008	/* Clock divide enbable */

#define	PMN3_OVERFLOW	0x10	/* Perfromance counter 0 overflow */
#define PMN2_OVERFLOW	0x08	/* Performance counter 1 overflow */
#define PMN1_OVERFLOW	0x04	/* Performance counter 1 overflow */
#define PMN0_OVERFLOW	0x02	/* Performance counter 1 overflow */
#define CCNT_OVERFLOW	0x01	/* Clock counter overflow */

static irqreturn_t pmu_irq_handler(int, void *, struct pt_regs *);

int pmu_claim(void)
{
	int err = 0;

	if(atomic_read(&usage))
		return -EBUSY;

	err = request_irq(IRQ_PMU, pmu_irq_handler, SA_INTERRUPT,
			  NULL, (void *) &results);
	if(err < 0)
	{
		printk(KERN_ERR "unable to request IRQ %d for 80200 PMU: %d\n",
		       IRQ_PMU, err);
		return err;
	}
	
	atomic_inc(&usage);
	pmnc = 0;
	pmu_reg_write(PMNC, pmnc);
	return ++id;
}

int pmu_release(int claim_id)
{
	if(!atomic_read(&usage))
		return 0;

	if(claim_id != id)
		return -EPERM;

	free_irq(IRQ_PMU, (void *)&results);
	atomic_dec(&usage);

	return 0;
}

int pmu_start(u32 pmn0, u32 pmn1, u32 pmn2, u32 pmn3)
{
	memset(&results, 0, sizeof(results));

	evtsel = (pmn3 <<24) | (pmn2 <<16) | (pmn1 <<8) | pmn0;

	pmnc |= PMU_ENABLE | PMU_RESET;

	pmu_reg_write(EVTSEL, evtsel);
	/*	All interrupt are turned on.	*/
	pmu_reg_write(INTEN, 0x1F);
	pmu_reg_write(PMNC, pmnc);

	return 0;
}

int pmu_stop(struct pmu_results *results)
{
	u32 ccnt;
	u32 pmn0;
	u32 pmn1;
	u32 pmn2;
	u32 pmn3;

	if(!pmnc)
		return -ENOSYS;

	ccnt = pmu_reg_read(CCNT);
	pmn0 = pmu_reg_read(PMN0);
	pmn1 = pmu_reg_read(PMN1);
	pmn2 = pmu_reg_read(PMN2);
	pmn3 = pmu_reg_read(PMN3);

	pmnc = 0;

	pmu_reg_write(PMNC, pmnc);

	results->ccnt = ccnt;
	results->pmn0 = pmn0;
	results->pmn1 = pmn1;
	results->pmn2 = pmn2;
	results->pmn3 = pmn3;

	return 0;
}

static irqreturn_t pmu_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	struct pmu_results *results = (struct pmu_results *)dev_id;
	unsigned int flag;

	/* read the status */
	flag = pmu_reg_read(FLAG);
	pmnc = pmu_reg_read(PMNC);

	if(pmnc & PMN0_OVERFLOW) {
		results->pmn0_of++;
	}

	if(pmnc & PMN1_OVERFLOW) {
		results->pmn1_of++;
	}

	if(pmnc & PMN2_OVERFLOW) {
		results->pmn2_of++;
	}

	if(pmnc & PMN3_OVERFLOW) {
		results->pmn3_of++;
	}

	if(pmnc & CCNT_OVERFLOW) {
		results->ccnt_of++;
	}

	pmu_reg_write(FLAG, flag);
	pmu_reg_write(PMNC, pmnc);

	return IRQ_HANDLED;
}

EXPORT_SYMBOL(pmu_claim);
EXPORT_SYMBOL(pmu_release);
EXPORT_SYMBOL(pmu_start);
EXPORT_SYMBOL(pmu_stop);


