/*
 * PXA250/210 Power Management Routines
 *
 * Original code for the SA11x0:
 * Copyright (c) 2001 Cliff Brake <cbrake@accelent.com>
 *
 * Modified for the PXA250 by Nicolas Pitre:
 * Copyright (c) 2002 Monta Vista Software, Inc.
 *
 * Modified for the PXA27x based on Mainstone by Chao Xie <chao.xie@intel.com>
 * Copyright (C) 2004, Intel Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/suspend.h>
#include <linux/errno.h>
#include <linux/time.h>
#include <linux/delay.h>

#include <asm/hardware.h>
#include <asm/memory.h>
#include <asm/system.h>
#include <asm/arch/pxa-regs.h>
#ifdef CONFIG_MACH_MAINSTONE
	#include <asm/arch/mainstone.h>
#endif
#include <asm/mach/time.h>

#include "cpu-freq-voltage-pxa27x.h"

/*
 * Debug macros
 */
#undef DEBUG
extern struct subsystem power_subsys;

extern int pxa27x_set_voltage(unsigned int, unsigned int, unsigned int);
extern unsigned int pxa27x_get_voltage(void);
extern void pxa27x_cpufreq_restore(void);

extern void pxa_cpu_suspend(void);
extern void pxa_cpu_resume(void);
extern void pxa_cpu_standby(void);
static int check_wakeup_src(void);
int pm_updatetimer(int);
void pm_timeout_proc(unsigned long);

#define SAVE(x)		sleep_save[SLEEP_SAVE_##x] = x
#define RESTORE(x)	x = sleep_save[SLEEP_SAVE_##x]

#define RESTORE_GPLEVEL(n) do { \
	GPSR##n = sleep_save[SLEEP_SAVE_GPLR##n]; \
	GPCR##n = ~sleep_save[SLEEP_SAVE_GPLR##n]; \
} while (0)

/* How long we will in sleep mode if duty cycle. */
unsigned int  pm_sleeptime=58;  /* In seconds. */

/* How long we will in run mode if duty cycle. */
unsigned int  pm_waketime = 2;  /* In seconds. */

/* After pm_uitimeout long without input from keypad/touch, we will sleep. */
unsigned int  pm_uitimeout= 30; /* In seconds. */

/* Timer timeout event handled in kernel or in policy maker? */
/* 0 is in kernel and 1 is for policy maker in user space. */
unsigned int  ipm_event_handler = 1;    /* In seconds. */

/*
 * The default sleep level, if we want to use Standy to replace sleep.
 * set this to PM_SUSPEND_STANDBY.
 */
unsigned int ipm_sleep_level = PM_SUSPEND_MEM; /* Default is to Sleep mode. */

/* Shall we use standby as idle? */
unsigned int ipm_sai = 0;

static struct timer_list pm_timer;
struct semaphore pm_timer_wait;

/*
 * List of global PXA peripheral registers to preserve.
 * More ones like CP and general purpose register values are preserved
 * with the stack pointer in sleep.S.
 */
enum {	SLEEP_SAVE_START = 0,

	SLEEP_SAVE_OIER,
	SLEEP_SAVE_OSMR0, SLEEP_SAVE_OSMR1, SLEEP_SAVE_OSMR2, SLEEP_SAVE_OSMR3,

	SLEEP_SAVE_GPLR0, SLEEP_SAVE_GPLR1, SLEEP_SAVE_GPLR2,
	SLEEP_SAVE_GPDR0, SLEEP_SAVE_GPDR1, SLEEP_SAVE_GPDR2,
	SLEEP_SAVE_GRER0, SLEEP_SAVE_GRER1, SLEEP_SAVE_GRER2,
	SLEEP_SAVE_GFER0, SLEEP_SAVE_GFER1, SLEEP_SAVE_GFER2,
	SLEEP_SAVE_GAFR0_L, SLEEP_SAVE_GAFR1_L, SLEEP_SAVE_GAFR2_L,
	SLEEP_SAVE_GAFR0_U, SLEEP_SAVE_GAFR1_U, SLEEP_SAVE_GAFR2_U,

#if defined(CONFIG_PXA27x)
        SLEEP_SAVE_GPDR3, SLEEP_SAVE_GRER3, SLEEP_SAVE_GFER3,
        SLEEP_SAVE_GAFR3_L, SLEEP_SAVE_GAFR3_U,
        SLEEP_SAVE_KPC,
#endif

        SLEEP_SAVE_ICMR,
        SLEEP_SAVE_CKEN,
#ifdef CONFIG_MACH_MAINSTONE
	SLEEP_SAVE_MST_GPSWR,
	SLEEP_SAVE_MST_MSCWR1,
	SLEEP_SAVE_MST_MSCWR2,
	SLEEP_SAVE_MST_MSCWR3,
	SLEEP_SAVE_MST_MSCRD,
	SLEEP_SAVE_MST_INTMSKENA,
	SLEEP_SAVE_MST_INTSETCLR,
#endif

        SLEEP_SAVE_PWER,
        SLEEP_SAVE_PSTR,
        SLEEP_SAVE_OMCR4,
        SLEEP_SAVE_OSCR4,
        SLEEP_SAVE_OSMR4,
        SLEEP_SAVE_MDREFR,
        SLEEP_SAVE_OSSR,
        SLEEP_SAVE_PGSR0,
        SLEEP_SAVE_PGSR1,
        SLEEP_SAVE_PGSR2,
        SLEEP_SAVE_PGSR3,
        SLEEP_SAVE_CKSUM,
        SLEEP_SAVE_SIZE
};


static int pxa_pm_enter(u32 state)
{
	unsigned long sleep_save[SLEEP_SAVE_SIZE];
	unsigned long checksum = 0;
	struct timespec delta, rtc;
	int i;

	if (state != PM_SUSPEND_MEM && state != PM_SUSPEND_STANDBY)
		return -EINVAL;
//	leds_event(led_stop);

	/* preserve current time */
	rtc.tv_sec = RCNR;
	rtc.tv_nsec = 0;
	save_time_delta(&delta, &rtc);

#if defined(CONFIG_MACH_MAINSTONE)
	SAVE(MST_GPSWR);
	SAVE(MST_MSCWR1);
	SAVE(MST_MSCWR2);
	SAVE(MST_MSCWR3);
	SAVE(MST_MSCRD);
	SAVE(MST_INTMSKENA);
	SAVE(MST_INTSETCLR);
#endif
	
	/* For SDRAM sightings. */
        SAVE(MDREFR);

	/* save vital registers */
	SAVE(OSSR);
	SAVE(OSMR0);
	SAVE(OSMR1);
	SAVE(OSMR2);
	SAVE(OSMR3);
	SAVE(OIER);

	SAVE(GPLR0); SAVE(GPLR1); SAVE(GPLR2);
	SAVE(GPDR0); SAVE(GPDR1); SAVE(GPDR2);
	SAVE(GRER0); SAVE(GRER1); SAVE(GRER2);
	SAVE(GFER0); SAVE(GFER1); SAVE(GFER2);
	SAVE(GAFR0_L); SAVE(GAFR0_U);
	SAVE(GAFR1_L); SAVE(GAFR1_U);
	SAVE(GAFR2_L); SAVE(GAFR2_U);
#if defined(CONFIG_PXA27x)
        SAVE(GPDR3);
        SAVE(GRER3);
        SAVE(GFER3);
        SAVE(GAFR3_L); SAVE(GAFR3_U);

        SAVE(KPC);      /* new added. */
#endif

	SAVE(ICMR);
	ICMR = 0;

	SAVE(CKEN);

        SAVE(PGSR0);
        SAVE(PGSR1);
        SAVE(PGSR2);
        SAVE(PGSR3);

#ifdef CONFIG_PXA27x
        if (state == PM_SUSPEND_STANDBY)
                CKEN = CKEN22_MEMC | CKEN9_OSTIMER | CKEN16_LCD |CKEN0_PWM0;
        else
                CKEN = CKEN22_MEMC | CKEN9_OSTIMER;

        PCFR = 0x66;
        /*
         * PSLR |= 0x4 is OK for serial console but not OK for
         * LCD which is configured to use ISRAM as framebuffer when using QVGA.
         *
         * PSLR |= 0xF04 is OK for LCD driver but not OK for serial console,
         * UART doesn't work correctly when PSLR|=0xF04
         */
        /* PSLR |= 0X4;    */
        PSLR |= 0xF04;

        /* For Keypad wakeup. */
#if defined(CONFIG_MACH_MAINSTONE)
        KPC &=~KPC_ASACT;
        KPC |=KPC_AS;
#endif
        PGSR0 = 0x00008800;
        PGSR1 = 0x00000002;
        PGSR2 = 0x0001FC00;
        PGSR3 = 0x00001F81;

        /*      PWER  = 0x80000002;     */
        PWER  = 0xC0000002;
        PRER  = 0x2;
        PFER  = 0x2;

        PKWR  = 0x000FD001;
        /*      Need read PKWR back after set it.       */
        PKWR;
#else
        CKEN = 0;
#endif
	/* clear GPIO transition detect  bits */
	GEDR0 = GEDR0; GEDR1 = GEDR1; GEDR2 = GEDR2;

        /* Clear sleep reset status */
        RCSR = RCSR_SMR;

        /* Clear edge-detect status register. */
        PEDR = 0xCE00FE1B;

        /* set resume return address */
#if defined(CONFIG_PXA27x)
        if (state != PM_SUSPEND_STANDBY)
                PSPR = virt_to_phys(pxa_cpu_resume);

        if (state == PM_SUSPEND_STANDBY) {
                SAVE(PSTR);
                SAVE(OMCR4);
                SAVE(OSCR4);
                SAVE(OSMR4);
                SAVE(OIER);
        }
#else
        PSPR = virt_to_phys(pxa_cpu_resume);
#endif

	/* before sleeping, calculate and save a checksum */
	for (i = 0; i < SLEEP_SAVE_SIZE - 1; i++)
		checksum += sleep_save[i];
	sleep_save[SLEEP_SAVE_CKSUM] = checksum;

	/* Use RTC to wakeup system from sleep. */
	if (state != PM_SUSPEND_STANDBY) { 
		/* Check the sleeptime to see how we need to wake up,in ms. */
		/* come back sometimes later.*/
		if (pm_sleeptime != 0) {  
			/*65s,use periodic interrupt*/
			if (pm_sleeptime < 65) {   
				/* Clear PICE bit in RTSR. */
				RTSR &= ~RTSR_PICE;
				/*set to sleep time,inms.*/
				/* set PIAR */
				PIAR = pm_sleeptime*1000;
				/* Ensure at least 2 CPY cycles pass.*/
				udelay(10);
				/* Clear PIALE bit in RTSR */
				RTSR &= ~RTSR_PIALE;
				RTSR |= RTSR_PIALE;
				RTSR |= RTSR_PICE;
			}
			else {  /*  we need to use RTC. */
                        	/*  Need to use other RTC wakeep methods..  */
                	}

		}
	}

#if defined(CONFIG_PXA27x)
	/* Use OST 4 to wakeup system from standby */
	if (state == PM_SUSPEND_STANDBY) {
		/*
		 * Here we use OS timer 4 as wakeup src. Maybe this is helpful
		 * when we decide to use standby to replace idle.
		 */
		/* All SRAM are on, PI is active with clock running. */
		PSTR = 0x00000F08;
		/* For standby we use OSCR4 as prioridic wakeup source. */
		/* CRES field of OMCR is set to 011, ticks will be 1 seconds.*/
		OMCR4  = 0x0B;
		OSCR4 = 1;
		OSMR4  = OSCR4 + pm_sleeptime;
		OIER = (OIER & 0xFFF) | 0x10;
	}
#endif

	/* go Zzzz */
        switch (state) {
                case PM_SUSPEND_MEM:
                        pxa_cpu_suspend();
			pxa27x_cpufreq_restore();
                        break;
#if defined(CONFIG_PXA27x)
                case PM_SUSPEND_STANDBY:
                        /*  Standby the CPU. */
                        pxa_cpu_standby();
                        break;
#endif
        }       /* end of switch */

	/* after sleeping, validate the checksum */
	checksum = 0;
	for (i = 0; i < SLEEP_SAVE_SIZE - 1; i++)
		checksum += sleep_save[i];

	/* if invalid, display message and wait for a hardware reset */
	if (checksum != sleep_save[SLEEP_SAVE_CKSUM]) {
#ifdef CONFIG_ARCH_LUBBOCK
		LUB_HEXLED = 0xbadbadc5;
#endif
		while (1);
	}

	/* ensure not to come back here if it wasn't intended */
	PSPR = 0;

#ifdef CONFIG_PXA27x
        if (state == PM_SUSPEND_STANDBY ) {
                /* Clear PEDR */
                RESTORE(OIER);
                RESTORE(MDREFR);
                RESTORE(PSTR);
                OSSR =0x10;
                OSMR4 = OSCR4 + 3;
                OSCR4 = OSCR4 + 4;
        }
        /* restore registers */
        RESTORE(PGSR0);
        RESTORE(PGSR1);
        RESTORE(PGSR2);
        RESTORE(PGSR3);
#endif

	/* restore registers */
	RESTORE(GAFR0_L); RESTORE(GAFR0_U);
	RESTORE(GAFR1_L); RESTORE(GAFR1_U);
	RESTORE(GAFR2_L); RESTORE(GAFR2_U);
	RESTORE_GPLEVEL(0); RESTORE_GPLEVEL(1); RESTORE_GPLEVEL(2);
	RESTORE(GPDR0); RESTORE(GPDR1); RESTORE(GPDR2);
	RESTORE(GRER0); RESTORE(GRER1); RESTORE(GRER2);
	RESTORE(GFER0); RESTORE(GFER1); RESTORE(GFER2);

#if defined(CONFIG_PXA27x)
	RESTORE(KPC);   /* new added. */
	RESTORE(GAFR3_L); RESTORE(GAFR3_U);
	RESTORE(GPDR3);
	RESTORE(GRER3);
	RESTORE(GFER3);
	PSSR = PSSR_PH | PSSR_OTGPH;
#else
	PSSR = PSSR_PH;
#endif

	RESTORE(OSMR0);
	RESTORE(OSMR1);
	RESTORE(OSMR2);
	RESTORE(OSMR3);
	RESTORE(OIER);

	/* OSMR0 is the system timer: make sure OSCR is sufficiently behind */
	OSCR = OSMR0 - LATCH;

	RESTORE(CKEN);

	ICLR = 0;
	ICCR = 1;
	RESTORE(ICMR);

#if defined(CONFIG_MACH_MAINSTONE)
	/*  Restore Mainstone Board registers.  */
	RESTORE(MST_GPSWR);
	RESTORE(MST_MSCWR1);
	RESTORE(MST_MSCWR2);
	RESTORE(MST_MSCWR3);
	RESTORE(MST_MSCRD);
	RESTORE(MST_INTMSKENA);
	RESTORE(MST_INTSETCLR);
#endif

	/* restore current time */
	rtc.tv_sec = RCNR;
	restore_time_delta(&delta, &rtc);

#ifdef DEBUG
	printk(KERN_DEBUG "*** made it back from resume\n");
#endif
	//leds_event(led_start);
        /* pm_updatetimer( check_wakeup_src() );*/

	return 0;
}

unsigned long sleep_phys_sp(void *sp)
{
	return virt_to_phys(sp);
}

/*
 * Return the source which bring the system out of sleep.
 */
static int check_wakeup_src(void)
{
	int temp = 0;
	if ( PEDR & 0x80000000) /* wakeup by RTC. */
		temp = 1;
	/* Clear PEDR */
	PEDR = 0xCE00FE1B;
	return temp;
}

/*
 * Let ipm_event_handler to choose whether timer timeout event handle in
 * kernel space or by policy maker.
 * 0 is in kernel and 1 is for policy maker in user space.
 */
void pm_timeout_proc(unsigned long ptr)
{
#if 0
	if (ipm_event_handler == 0)
		up(&pm_timer_wait); /* Let ipm_thread to sleep the system. */
	else /* notice the policy maker. */
		up(&event_sem);
#endif
}

/*
 * Flag is used to say whether it was called from UI update or from
 * sleep call. In UI device call it with flag = 0, in sleep return
 * call it with flag = 1.
 */
int pm_updatetimer(int flag)
{
	if (flag)
		mod_timer(&pm_timer, jiffies + pm_waketime*HZ );
	else
		mod_timer(&pm_timer, jiffies + pm_uitimeout*HZ );
	return 0;
}

EXPORT_SYMBOL_GPL(pm_updatetimer);

/*
 * Called after processes are frozen, but before we shut down devices.
 */
static unsigned int old_core_voltage;
static int pxa_pm_prepare(u32 state)
{
	if (state == PM_SUSPEND_MEM) {
		/* set to highest voltage */
		old_core_voltage = pxa27x_get_voltage();
		pxa27x_set_voltage(PXA27X_MAX_FREQ, 0, 0);
	}
	return 0;
}

/*
 * Called after devices are re-setup, but before processes are thawed.
 */
static int pxa_pm_finish(u32 state)
{
	/* restore voltage, this should after restore CKEN */
	if (state == PM_SUSPEND_MEM) {
		pxa27x_set_voltage(PXA27X_MIN_FREQ, old_core_voltage, 0);
	}
	return 0;
}

/*
 * Set to PM_DISK_FIRMWARE so we can quickly veto suspend-to-disk.
 */
static struct pm_ops pxa_pm_ops = {
	.pm_disk_mode	= PM_DISK_FIRMWARE,
	.prepare	= pxa_pm_prepare,
	.enter		= pxa_pm_enter,
	.finish		= pxa_pm_finish,
};

#define pm_attr(_name, object) \
static ssize_t _name##_store(struct subsystem * subsys, const char * buf, size_t n)									\
{									\
	sscanf(buf, "%u", &object);					\
	return n;							\
}									\
static ssize_t _name##_show(struct subsystem * subsys, char * buf)	\
{									\
	return sprintf(buf, "%u\n", object);				\
}									\
static struct subsys_attribute _name##_attr = { 			\
        .attr   = {                             			\
                .name = __stringify(_name),     			\
                .mode = 0644,                   			\
        },                                      			\
        .show   = _name##_show,                 			\
        .store  = _name##_store,                			\
}


pm_attr(sleeptime, pm_sleeptime);

static int __init pxa_pm_init(void)
{
	pm_set_ops(&pxa_pm_ops);
	sema_init(&pm_timer_wait, 0);
	init_timer(&pm_timer);
	pm_timer.function = pm_timeout_proc;

	if (pm_uitimeout) {
		mod_timer(&pm_timer, jiffies + pm_uitimeout*HZ );
	}
	
	sysfs_create_file(&power_subsys.kset.kobj, &sleeptime_attr.attr);
	return 0;
}

late_initcall(pxa_pm_init);
