/*
 * linux/arch/arm/mach-pxa/ipmc.c
 *
 * Provide ioctl interface to application, to specify more detailed info and change
 * system's behaviour.
 *
 * Copyright (C) 2003-2004 Intel Corporation.
 *
 * Author:  Cain Yuan <cain.yuan@intel.com>
 *
 * This software program is licensed subject to the GNU General Public License
 * (GPL).Version 2,June 1991, available at http://www.fsf.org/copyleft/gpl.html
 *
 * If wana use it please use "mknod /dev/ipmc c 10 90" to create the dev file.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/miscdevice.h>
#include <linux/fs.h>       
#include <linux/fcntl.h>    
#include <linux/errno.h>
#include <linux/cpufreq.h>
#include <linux/wait.h>
#include <linux/poll.h>
#include <linux/pm.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/arch/pxa-regs.h>
#include <asm/current.h>  
#include <asm/system.h> 
#include <asm/semaphore.h>
#include "ipmc.h"    

extern unsigned int pxa27x_validate(unsigned int, unsigned int, unsigned int, unsigned int, unsigned int);
extern unsigned int pxa27x_read_clkcfg(void); 
extern int pxa27x_set_cpufreq(unsigned int, unsigned int, unsigned int, unsigned int, unsigned int);	
extern int pm_updatetimer(int); 
extern int pxa27x_set_voltage(unsigned int, unsigned int, unsigned int);
extern unsigned int pxa27x_get_voltage(void);
#if 0
extern void ipm_start_pmu();
#endif

static int ipmq_get(struct ipm_event *);
static int ipmq_put(struct ipm_event *);
//static int ipmq_empty(void);
static int ipmq_clear(void);

struct ipm_config global_conf;
static DECLARE_MUTEX    (ipm_conf_sem);
static struct ipme_queue ipme_queue;

int ipmc_open(struct inode *inode, struct file *filep)
{
	MOD_INC_USE_COUNT;
	return 0;          /* success */
}

int ipmc_close( struct inode *inode, struct file *filep )
{
	MOD_DEC_USE_COUNT;
	return 0;
}

/*
 *  Return a ipm event or get blocked until there is a event.
 */

static __inline int ipmq_empty(void)
{
	return (ipme_queue.len>0)?0:1;
}

ssize_t ipmc_read(struct file *filep, char *buf, size_t count,
		loff_t *f_pos)
{
	DECLARE_WAITQUEUE(wait, current);
	struct ipm_event data;
	ssize_t retval;

	if (count < sizeof(struct ipm_event))
		return -EINVAL;

	add_wait_queue(&ipme_queue.waitq, &wait);
	set_current_state(TASK_INTERRUPTIBLE);

	for (;;) {
		if (filep->f_flags & O_NONBLOCK) {
			retval = -EAGAIN;
			goto out;
		}

		if (signal_pending(current)) {
			retval = -ERESTARTSYS;
			goto out;
		}
		/*	Get the event out. If no, blocked.*/	
		if ( !ipmq_empty()){
			ipmq_get(&data);
			break;
		}		
		/*	No events and no error, block it.	*/
		schedule();
	}

	/*	pass the event to user space.	*/
	retval = copy_to_user( (struct ipm_event *)buf, &data,
			sizeof(struct ipm_event) );
	if (retval) {
		retval = -EFAULT;
		goto out;
	}
	retval = sizeof(struct ipm_event);	

out:
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&ipme_queue.waitq, &wait);
	return retval;
}

/*	
 *	Write will do nothing. If need to pass down some information use ioctl
 *	interface instead.
 */
ssize_t ipmc_write(struct file *filep, const char *buf, size_t count,
                loff_t *f_pos)
{
    return 0;
}

/* poll for ipm event */
unsigned int ipmc_poll(struct file *file, poll_table *wait) 
{
	poll_wait(file, &ipme_queue.waitq, wait);
	if (!ipmq_empty())
		return POLLIN | POLLRDNORM;

	return 0;
}

#if 0
/*
 * This function will change voltage and freq together using
 * VCS command and set PCMD[FVC] bit.
 */
static void	 combine_vltg_freq_change(struct ipm_config *conf)
{
	unsigned int	l, n;
	unsigned int	turbo_mode, dac_value;

	cpu_voltage_cur = conf->core_vltg;

	l = bulverde_freq_getL(conf->core_freq, *conf);
	n = conf->turbo_mode;
	turbo_mode = (conf->turbo_mode ==0 )? 0:1;
	dac_value =  mv2DAC(conf->core_vltg );		

	combochange(l, n ,turbo_mode, conf->fast_bus_mode, dac_value);
}	

unsigned int check_ipm_config(struct ipm_config *target)
{
		int	i=0;
		struct	ipm_config	cur;

		cur.core_freq = cpufreq_get(0);

		/*	Get current voltage.	*/
		while( ipm_fv_table[i].freq != 0 ) {
			if( cur.core_freq < ipm_fv_table[i].freq ) {
				cur.core_vltg = ipm_fv_table[i].voltage;
				break;
			}
			i++;
		}
		i=0;
		/*	Get target voltage.	*/
		while( ipm_fv_table[i].freq != 0 ) {
			if( target->core_freq < ipm_fv_table[i].freq )  {
				target->core_vltg = ipm_fv_table[i].voltage;
				break;
			}
			i++;
		}
#ifdef DEBUG
		printk(KERN_INFO "Current freq: %d, target freq: %d.\n", cur.core_freq, target->core_freq);
		printk(KERN_INFO "Current vol: %d, target vol: %d.\n",cur.core_vltg, target->core_vltg);
#endif		
		if( target->core_vltg == cur.core_vltg )	return 0;
		if( target->core_vltg > cur.core_vltg )	return HIGHER;
		return LOWER;
}	
#endif


void get_ipm_config(struct ipm_config  *ret_conf)
{
	struct ipm_config conf;
	unsigned long clk_cfg = 0;
	unsigned int l, n2, m, k, run_freq, ccsr, cccr_a;
	
	memset(&conf, 0, sizeof(struct ipm_config));
	conf.cpu_mode =0;	/* definitly in RUN mode.	*/

	ccsr = CCSR;
	l = ccsr & 0x1f;
	n2 = (ccsr>>7) & 0xf;
	m = (l <= 10) ? 1 : (l <= 20) ? 2 : 4;
	k = (l <= 7) ? 1 : (l <= 16) ? 2 : 4;
	cccr_a = CCCR & (1 << 25);

	clk_cfg = pxa27x_read_clkcfg();
	conf.fast_bus_mode = (clk_cfg >> 0x3) & 0x1;
	conf.turbo_mode = (clk_cfg & 0x1)? 2: (clk_cfg & 0x4)? 1: 0;
	conf.n2 = n2;
	conf.l = l;
	conf.mem_clk_conf = cccr_a >> 25;
	run_freq = 13000*l;	
	if (ccsr & (1 << 31)) {
		conf.core_freq = conf.sys_bus_freq = conf.mem_bus_freq = 13000;
		conf.lcd_freq = (CCCR & (1 << 27))? 26000 : 13000;	
		memcpy(ret_conf, &conf,sizeof(struct ipm_config));
		return ;
	}
	conf.core_freq = l*((conf.turbo_mode == 0)? 13000 : 
			((conf.turbo_mode == 2)? 6500*n2 : 3250*n2)); 	
	conf.sys_bus_freq = (conf.fast_bus_mode)? run_freq : run_freq/2;
	conf.mem_bus_freq = (!cccr_a) ? (run_freq/m) : ((conf.fast_bus_mode) ?
				 run_freq : (run_freq/2));
	conf.lcd_freq = run_freq / k;

	memcpy(ret_conf, &conf,sizeof(struct ipm_config));

	return;
}

/*	
 *	Write the desired voltage to
 *	DPM voltage changer and the desired freq to DPM freq changer.
 */
void set_ipm_conf(struct ipm_config *conf)
{
	int mode, ret;
	unsigned int vol = conf->core_vltg;
	unsigned int freq, new_freq;
	unsigned int l = conf->l, n2 = conf->n2;
	unsigned int t = conf->turbo_mode, b = conf->fast_bus_mode;
	unsigned int a = conf->mem_clk_conf;
	int pre_change = 0;

	mode = conf->cpu_mode;
	/* want to change CPU mode? */
	if( mode != 0) {	/* not RUN mode.	*/
		/* allow to use this method to sleep the system...*/
		pm_suspend(mode);
	}	

	ret = pxa27x_validate(l, n2, b, t, a);
	if (ret) {
		printk(KERN_ERR "Error: Wrong L: %d, 2N: %d, B: %d, T: %d, A:%d\n", l, n2, b, t, a);
		return ;	
	}

	freq = cpufreq_get(0);
	new_freq = l*((t == 0)? 13000 : ((t == 2)? 6500*n2 : 3250*n2)); 	
	/*	
	 * It may need to change frequency. The voltage change is also invoked 
	 * according to the frequency. 
	 */	
	if (vol == 0) {
		if (freq < new_freq)
			pre_change = 1;
	}
	else {
		if (pxa27x_get_voltage() < vol)
			pre_change = 1;	
	}
	if (pre_change == 1) {
		if (pxa27x_set_voltage(new_freq, vol, 0) < 0 && vol)
			pxa27x_set_voltage(new_freq, 0, 0);
		pxa27x_set_cpufreq(l, n2, b, t, a);
	}
	else {
		pxa27x_set_cpufreq(l, n2, b, t, a);
		if (pxa27x_set_voltage(new_freq, vol, 0) < 0 && vol)
			pxa27x_set_voltage(new_freq, 0, 0);
	}
	/*
	 * combochange of voltage and frequency
	 */
#if 0
	if (pxa27x_set_voltage(new_freq, vol, 1) < 0 && vol)
		pxa27x_set_voltage(new_freq, 0, 1);
	pxa27x_set_cpufreq(l, n2, b, t);
#endif
	return;
}	

#define	SLEEP_SYSTEM 0
extern unsigned int ipm_sleep_level;
extern unsigned int pm_sleeptime;
extern unsigned int pm_waketime;
extern unsigned int pm_uitimeout;
extern unsigned int ipm_sleep_level;
#if 0
extern unsigned long WindowSize;
#endif
int ipmc_ioctl (struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
{
	struct ipm_config conf, target_conf;
	int result = 0;

	memset(&conf, 0 ,sizeof(struct ipm_config));

	switch (cmd) {
		case IPMC_IOCTL_GET_DPM_CONFIG:	
			get_ipm_config(&conf);
			return (copy_to_user((unsigned long *)arg, &conf,sizeof(conf)))? -EFAULT:0;
			break;

		case IPMC_IOCTL_SET_DPM_CONFIG:	
			if(copy_from_user(&target_conf,(struct ipmc_conf *)arg,sizeof(conf))) {
				printk("Erro from copy_from_user.\n");
				return -EFAULT;
			}	

			set_ipm_conf(&target_conf);
			break;

			/*	Reset the UI timer.	*/	
		case IPMC_IOCTL_RESET_UI_TIMER:
			pm_updatetimer(0);	
			break;

			/*	get rid of those event before.	*/	
		case IPMC_IOCTL_CLEAR_EVENTLIST:
			ipmq_clear();
			break;

		case IPMC_IOCTL_SET_SLEEPTIME:  
			if( copy_from_user(&pm_sleeptime,(unsigned int *)arg,sizeof(int)) )
				return -EFAULT;
			break;
#if 0
		case IPMC_IOCTL_SET_WINDOWSIZE:
			if( copy_from_user(&WindowSize,(unsigned int)arg,sizeof(int)) )
				return -EFAULT;
			break;
#endif
		case IPMC_IOCTL_GET_SLEEPTIME:  
			return copy_to_user((unsigned int *)arg, &pm_sleeptime, sizeof(int));
			break;

		case IPMC_IOCTL_SET_WAKETIME:  
			if (copy_from_user(&pm_waketime,(unsigned int *)arg,sizeof(int)) )
				return -EFAULT;
			break;

		case IPMC_IOCTL_GET_WAKETIME:  
			return copy_to_user((unsigned int *)arg, &pm_waketime, sizeof(int));
			break;

		case IPMC_IOCTL_SET_UITIME:  
			if( copy_from_user(&pm_uitimeout,(unsigned int *)arg,sizeof(int)) )
				return -EFAULT;
			if( pm_uitimeout !=0 )	/*	stop it.	*/
				pm_updatetimer(0);	
			break;

		case IPMC_IOCTL_GET_UITIME:  
			return copy_to_user((unsigned int *)arg, &pm_uitimeout, sizeof(int));
			break;

		case IPMC_IOCTL_SET_SLEEPLEVEL:  
			if( copy_from_user(&ipm_sleep_level,(unsigned int *)arg,sizeof(int)) )
				return -EFAULT;
			break;

		case IPMC_IOCTL_GET_SLEEPLEVEL:  
			return copy_to_user((unsigned int *)arg, &ipm_sleep_level, sizeof(int));
			break;

			/*
			 *	Do we need to decide those two ioctls into four?
			 *	IPMC_IOCTL_CLAIMPMU, 	IPMC_IOCTL_STARTPMU, 
			 *	IPMC_IOCTL_STOPPMU,		IPMC_IOCTL_RELEASEPMU
			 *	or just
			 *	IPMC_IOCTL_STARTPMU,  IPMC_IOCTL_STOPPMU ?
			 */	
			/*	
			 *	Did not need PMU events to policy  maker?	That 's really good. 
			 *	Still keep this bez we need to test PMU driver, right?
			 */	
#if 0		
		case IPMC_IOCTL_STARTPMU:
			ipm_start_pmu();
			break;
			/*	
			 *	When using stop pmu the user application should provide a place
			 *	to save the result.
			 */	
		case IPMC_IOCTL_STOPPMU:
			/*	pmu should already stopped.	*/
			pmu_stop(&presult);
			/*	
			 *	Release it? Since only one user can have it we just
			 *	use the recorded pmu id.
			 */
			pmu_release(pmu_id);
			/*	Shall we check the arg pointer first?	*/
			if( arg!=NULL )
				return copy_to_user( (unsigned int)arg, &presult, 
						sizeof(struct pmu_results) );
			else	return 0;
			break;
#endif

		default:
			result = -ENOIOCTLCMD;
			break;
	}
	return result;
}	

static struct file_operations ipmc_fops = {
	owner:	THIS_MODULE,
	open:	ipmc_open,
	read:	ipmc_read,
	write:	ipmc_write,
	poll:	ipmc_poll,
	ioctl:	ipmc_ioctl,
	release:ipmc_close,
};

static struct miscdevice ipmc_misc_device = {
	minor:	IPMC_MINOR,
	name:	"ipmc",
	fops:	&ipmc_fops
};

/*
 *	Below is the event list maintain functions, the same as keypad.c
 */
static int ipmq_init(void)
{
	/* init ipme queue */
	ipme_queue.head = ipme_queue.tail = 0;
	ipme_queue.len = 0;
	init_waitqueue_head(&ipme_queue.waitq);
	return 0;
}

static int __init ipmc_init( void )
{
	int rc;

	/*	Clear event queue. */
	ipmq_init();

	if ( (rc = misc_register( &ipmc_misc_device )) != 0 ) {
		printk( KERN_INFO "Could not register device ipmc, res = %d.\n ",rc );
		return -EBUSY;
	}
	printk( KERN_INFO "Register device ipmc successful.\n " );
	return 0;
}

static void ipmc_exit( void )
{
	misc_deregister(&ipmc_misc_device);
}


static int ipmq_get(struct ipm_event *ipme)
{
	unsigned long flags;

	local_irq_save(flags);

	if (!ipme_queue.len) {
		local_irq_restore(flags);
		return -1;
	}

	memcpy(ipme, ipme_queue.ipmes + ipme_queue.tail, sizeof(struct ipm_event));

	ipme_queue.len--;
	ipme_queue.tail = (ipme_queue.tail + 1) % MAX_IPME_NUM;

	local_irq_restore(flags);

	return 0;

}

static int ipmq_clear(void)
{
	unsigned long flags;

	local_irq_save(flags);
	ipme_queue.head = ipme_queue.tail = 0;
	ipme_queue.len = 0;
	local_irq_restore(flags);

	return 0;
}

static int ipmq_put(struct ipm_event *ipme)
{
	unsigned long flags;

	local_irq_save(flags);

	if (ipme_queue.len == MAX_IPME_NUM) {
		local_irq_restore(flags);
		return -1;
	}

	memcpy(ipme_queue.ipmes + ipme_queue.head, ipme, sizeof(struct ipm_event));

	ipme_queue.len ++;
	ipme_queue.head = (ipme_queue.head + 1) % MAX_IPME_NUM;

	local_irq_restore(flags);

	/* wake up the waiting process */
	wake_up_interruptible(&ipme_queue.waitq);

	return 0;
}

static struct ipm_event __inline IPM_Events(int type, int kind, int info)
{
	struct ipm_event event;
	event.type = type;
	event.kind = kind;
	event.info = info;
	return event;
}		

/* 
 * IPM event can be posted by this function. 
 * If we need to do some pre-processing of those events we can add code here.
 * Also attach the processed result to info.
 */
void ipm_event_notify(int  type, int kind, int info)
{
	struct ipm_event events;
	events = IPM_Events(type, kind, info);
	ipmq_put(&events);
}

EXPORT_SYMBOL(ipm_event_notify);

module_init(ipmc_init);
module_exit(ipmc_exit);
