/*
 *	IPv6-IPv6 tunneling module
 *
 *	Authors:
 *	Sami Kivisaari		<skivisaa@cc.hut.fi>
 *	Ville Nuorvala          <vnuorval@tml.hut.fi>
 *
 *	$Id: tunnel.c,v 1.15 2002/03/11 09:22:01 jola Exp $
 *
 *	This program is free software; you can redistribute it and/or
 *      modify it under the terms of the GNU General Public License
 *      as published by the Free Software Foundation; either version
 *      2 of the License, or (at your option) any later version.
 *
 */

#include <linux/net.h>
#include <linux/skbuff.h>
#include <linux/ipv6.h>
#include <linux/net.h>
#include <linux/netdevice.h>
#include <linux/init.h>
#include <linux/route.h>
#include <linux/ipv6_route.h>
#include <net/protocol.h>
#include <net/ipv6.h>
#include <net/ip6_route.h>
#include <net/dst.h>
#include <net/addrconf.h>
#include <net/ipv6_tunnel.h>
#include "tunnel.h"
#include "debug.h"
#if 0
void mipv6_initialize_tunnel(unsigned int tunnel_nr)
{
	ipv6_ipv6_tunnel_inc_min_kdev_count(tunnel_nr);
}

void mipv6_shutdown_tunnel(unsigned int tunnel_nr)
{
	ipv6_ipv6_tunnel_dec_max_kdev_count(tunnel_nr);
}


int mipv6_tunnel_add(struct in6_addr *remote, 
		     struct in6_addr *local, 
		     int local_origin) 
{

	struct ipv6_tunnel_parm p;
	int flags = (IPV6_T_F_KERNEL_DEV | IPV6_T_F_MIPV6_DEV |
		     IPV6_T_F_IGN_ENCAP_LIM | 
		     (local_origin ? IPV6_T_F_LOCAL_ORIGIN : 0));
	DEBUG_FUNC();

	DEBUG((DBG_INFO, "mipv6_tunnel_add: adding tunnel from: "
	       "%x:%x:%x:%x:%x:%x:%x:%x to:"
	       "%x:%x:%x:%x:%x:%x:%x:%x", 
	       NIPV6ADDR(local), NIPV6ADDR(remote)));

 	memset(&p, 0, sizeof(p));
	p.proto = IPPROTO_IPV6;
	ipv6_addr_copy(&p.saddr, local);
	ipv6_addr_copy(&p.daddr, remote);
	p.hop_limit = -1;
	p.flags = flags;
	return ipv6_ipv6_kernel_tunnel_add(&p);
}

void mipv6_tunnel_del(struct in6_addr *remote, 
		      struct in6_addr *local) 
{
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(remote, local);
	DEBUG_FUNC();

	DEBUG((DBG_INFO, "mipv6_tunnel_del: deleting tunnel from: "
	       "%x:%x:%x:%x:%x:%x:%x:%x to:"
	       "%x:%x:%x:%x:%x:%x:%x:%x", 
	       NIPV6ADDR(local), NIPV6ADDR(remote)));

	if (t != NULL && (t->parms.flags & IPV6_T_F_MIPV6_DEV)) 
		ipv6_ipv6_kernel_tunnel_del(t);
}

int mipv6_tunnel_route_add(struct in6_addr *home_addr, 
			   struct in6_addr *coa, 
			   struct in6_addr *our_addr) 
{
	struct in6_rtmsg rtmsg;
	int ret;
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(coa, our_addr);
	if (!t) {
		DEBUG((DBG_CRITICAL,"Tunnel missing"));
		return -1;
	}
	memset(&rtmsg, 0, sizeof(rtmsg));
	DEBUG((DBG_INFO, "mipv6_tunnel_route_add: adding route to: %x:%x:%x:%x:%x:%x:%x:%x via tunnel device",	NIPV6ADDR(home_addr)));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, home_addr);
	ipv6_addr_copy(&rtmsg.rtmsg_src, our_addr);
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_HOST;
	rtmsg.rtmsg_dst_len = 128;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = 1;
	if ((ret = ip6_route_add(&rtmsg)) == -EEXIST) {
		DEBUG((DBG_INFO, "Trying to add route twice"));
		return 0;
	}
	else
		return ret;

}

void mipv6_tunnel_route_del(struct in6_addr *home_addr, 
			    struct in6_addr *coa, 
			    struct in6_addr *our_addr) 
{
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(coa, our_addr);
	if (t != NULL) {
		struct rt6_info *rt = rt6_lookup(home_addr, our_addr, 
						 t->dev->ifindex, 0);
		
		if (rt && !ipv6_addr_cmp(&rt->rt6i_dst.addr, home_addr)) {
			DEBUG((DBG_INFO, "mipv6_tunnel_route_del: deleting route to: %x:%x:%x:%x:%x:%x:%x:%x",	NIPV6ADDR(&rt->rt6i_dst.addr)));
			ip6_del_rt(rt);
		}
		else 
			DEBUG((DBG_INFO,"Route to home address not found"));
	}
}
#else
void mipv6_initialize_tunnel(void)
{
	ipv6_ipv6_tunnel_inc_max_kdev_count(mipv6_max_tunnel_nr);
	ipv6_ipv6_tunnel_inc_min_kdev_count(mipv6_min_tunnel_nr);
}

void mipv6_shutdown_tunnel(void)
{
	ipv6_ipv6_tunnel_dec_min_kdev_count(mipv6_min_tunnel_nr);
	ipv6_ipv6_tunnel_dec_max_kdev_count(mipv6_max_tunnel_nr);
}


static int tnl_add(struct in6_addr *remote, 
		   struct in6_addr *local, 
		   int local_origin) 
{
	struct ipv6_tunnel_parm p;
	int ret;

	DEBUG_FUNC();

 	memset(&p, 0, sizeof(p));
	p.proto = IPPROTO_IPV6;
	ipv6_addr_copy(&p.saddr, local);
	ipv6_addr_copy(&p.daddr, remote);
	p.hop_limit = -1;
	p.flags = (IPV6_T_F_KERNEL_DEV | IPV6_T_F_MIPV6_DEV |
		   IPV6_T_F_IGN_ENCAP_LIM | 
		   (local_origin ? IPV6_T_F_LOCAL_ORIGIN : 0));

	ret = ipv6_ipv6_kernel_tunnel_add(&p);
	if (ret > 0) {
		DEBUG((DBG_INFO, "added tunnel from: "
		       "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		       NIPV6ADDR(local), NIPV6ADDR(remote)));
	} else {
		DEBUG((DBG_CRITICAL, "unable to add tunnel from: "
		       "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		       NIPV6ADDR(local), NIPV6ADDR(remote)));		
	}
	return ret;
}

static int tnl_del(struct in6_addr *remote, 
		   struct in6_addr *local) 
{
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(remote, local);
	
	DEBUG_FUNC();
	
	if (t != NULL && (t->parms.flags & IPV6_T_F_MIPV6_DEV)) {
		DEBUG((DBG_INFO, "deleting tunnel from: "
		       "%x:%x:%x:%x:%x:%x:%x:%x to: %x:%x:%x:%x:%x:%x:%x:%x", 
		       NIPV6ADDR(local), NIPV6ADDR(remote)));

		return ipv6_ipv6_kernel_tunnel_del(t);
	}
	return 0;
}

static __inline__ int add_route_to_mn(struct in6_addr *coa, 
				      struct in6_addr *ha_addr, 
				      struct in6_addr *home_addr) 
{
	struct in6_rtmsg rtmsg;
	int err;
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(coa, ha_addr);
	
	DEBUG_FUNC();

	if (!t || !(t->parms.flags & IPV6_T_F_MIPV6_DEV)) {
		DEBUG((DBG_CRITICAL,"Tunnel missing"));
		return -ENODEV;
	}
	
	DEBUG((DBG_INFO, "adding route to: %x:%x:%x:%x:%x:%x:%x:%x via "
	       "tunnel device", NIPV6ADDR(home_addr)));

	memset(&rtmsg, 0, sizeof(rtmsg));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, home_addr);
	rtmsg.rtmsg_dst_len = 128;
#ifdef CONFIG_IPV6_SUBTREES
	ipv6_addr_copy(&rtmsg.rtmsg_src, ha_addr);
	rtmsg.rtmsg_src_len = 128;
#endif
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_NONEXTHOP | RTF_HOST;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((err = ip6_route_add(&rtmsg)) == -EEXIST) {
		err = 0;
	}
	return err;
}


extern int ip6_route_del(struct in6_rtmsg *rtmsg);

static __inline__ void del_route_to_mn(struct in6_addr *coa, 
				       struct in6_addr *ha_addr, 
				       struct in6_addr *home_addr) 
{
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(coa, ha_addr);

	DEBUG_FUNC();

	if (t != NULL && (t->parms.flags & IPV6_T_F_MIPV6_DEV)) {
		struct in6_rtmsg rtmsg;

		DEBUG((DBG_INFO, "deleting route to: %x:%x:%x:%x:%x:%x:%x:%x "
		       " via tunnel device", NIPV6ADDR(home_addr)));

		memset(&rtmsg, 0, sizeof(rtmsg));
		ipv6_addr_copy(&rtmsg.rtmsg_dst, home_addr);
		ipv6_addr_copy(&rtmsg.rtmsg_src, ha_addr);
		rtmsg.rtmsg_dst_len = 128;
#ifdef CONFIG_IPV6_SUBTREES
		ipv6_addr_copy(&rtmsg.rtmsg_src, ha_addr);
		rtmsg.rtmsg_src_len = 128;
#endif
		rtmsg.rtmsg_ifindex = t->dev->ifindex;
		rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
		ip6_route_del(&rtmsg);
	}
}


int mipv6_add_tnl_to_mn(struct in6_addr *coa, 
			struct in6_addr *ha_addr,
			struct in6_addr *home_addr)
{
	int ret;

	DEBUG_FUNC();

	ret = tnl_add(coa, ha_addr, 1);

	if (ret > 0) {
		int err = add_route_to_mn(coa, ha_addr, home_addr);
		if (err && err != -ENODEV) {
			tnl_del(coa, ha_addr);
			return err;
		}
	}
	return ret;
} 

int mipv6_del_tnl_to_mn(struct in6_addr *coa, 
			struct in6_addr *ha_addr,
			struct in6_addr *home_addr)
{
	DEBUG_FUNC();
	del_route_to_mn(coa, ha_addr, home_addr);
	return tnl_del(coa, ha_addr);
} 

static __inline__ int add_reverse_route_to_ha(struct in6_addr *ha_addr,
					      struct in6_addr *coa,
					      struct in6_addr *home_addr) 
{
	struct in6_rtmsg rtmsg;
	int ret;
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(ha_addr, coa);	
	
	DEBUG_FUNC();
	
	if (!t || !(t->parms.flags & IPV6_T_F_MIPV6_DEV)) {
		DEBUG((DBG_CRITICAL,"Tunnel missing"));
		return -ENODEV;
	}
	
	DEBUG((DBG_INFO, "adding route to ha via tunnel device"));
	
	memset(&rtmsg, 0, sizeof(rtmsg));
	ipv6_addr_copy(&rtmsg.rtmsg_dst, ha_addr);
	rtmsg.rtmsg_dst_len = 128;
#ifdef CONFIG_IPV6_SUBTREES
	ipv6_addr_copy(&rtmsg.rtmsg_src, home_addr);
	rtmsg.rtmsg_src_len = 128;
#endif
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_HOST | RTF_NONEXTHOP ;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6 + 1; 
	if ((ret = ip6_route_add(&rtmsg)) == -EEXIST) {
		return 0;
	}
	return ret;	
}


static __inline__ int add_dflt_reverse_route(struct in6_addr *ha_addr,
					     struct in6_addr *coa,
					     struct in6_addr *home_addr) 
{
	struct in6_rtmsg rtmsg;
	int ret;
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(ha_addr, coa);	
	
	DEBUG_FUNC();
	
	if (!t || !(t->parms.flags & IPV6_T_F_MIPV6_DEV)) {
		DEBUG((DBG_CRITICAL,"Tunnel missing"));
		return -ENODEV;
	}
	
	DEBUG((DBG_INFO, "adding default route via tunnel device"));
	
	memset(&rtmsg, 0, sizeof(rtmsg));
#ifdef CONFIG_IPV6_SUBTREES
	ipv6_addr_copy(&rtmsg.rtmsg_src, home_addr);
	rtmsg.rtmsg_src_len = 128;
#endif
	ipv6_addr_copy(&rtmsg.rtmsg_gateway, ha_addr);
	rtmsg.rtmsg_type = RTMSG_NEWROUTE;
	rtmsg.rtmsg_flags = RTF_UP | RTF_GATEWAY | RTF_DEFAULT;
	rtmsg.rtmsg_ifindex = t->dev->ifindex;
	rtmsg.rtmsg_metric = IP6_RT_PRIO_MIPV6;
	if ((ret = ip6_route_add(&rtmsg)) == -EEXIST) {
		return 0;
	}
	return ret;	
}

static int reverse_route_cleanup(struct rt6_info *rt, void *arg)
{
	if (((void*)rt->rt6i_dev == arg) && rt != &ip6_null_entry) {
		return -1;
	}
	return 0;
}

static __inline__ void del_reverse_routes(struct in6_addr *ha_addr,
					  struct in6_addr *coa) 
{
	struct ipv6_tunnel *t = ipv6_ipv6_tunnel_lookup(ha_addr, coa);
	
	DEBUG_FUNC();

	if (t != NULL && atomic_read(&t->refcnt) == 1) {
		write_lock_bh(&rt6_lock);
		fib6_clean_tree(&ip6_routing_table, 
				reverse_route_cleanup, 
				0, t->dev);
		write_unlock_bh(&rt6_lock);
	}
}

int mipv6_add_tnl_to_ha(struct in6_addr *ha_addr, 
			struct in6_addr *coa,
			struct in6_addr *home_addr)
{
	int ret;
	
	DEBUG_FUNC();
	
	ret = tnl_add(ha_addr, coa, mipv6_reverse_tunnel);
	
	if (ret > 0 && mipv6_reverse_tunnel) {
		int err = add_reverse_route_to_ha(ha_addr, coa, home_addr);
		if (err && err != -ENODEV) {
			tnl_del(ha_addr, coa);
			return err;
		} else {
			err = add_dflt_reverse_route(ha_addr, coa, home_addr);
			if (err && err != -ENODEV) {
				tnl_del(ha_addr, coa);
				return err;
			}
		}
	}
	return ret;
} 

int mipv6_del_tnl_to_ha(struct in6_addr *ha_addr, 
			struct in6_addr *coa,
			struct in6_addr *home_addr)
{
	DEBUG_FUNC();
	
	if (mipv6_reverse_tunnel) {
		del_reverse_routes(ha_addr, coa);
	}
	return tnl_del(ha_addr, coa);
} 

#endif
