/**
 * Generic icmp routines
 *
 * Authors:
 * Jaakko Laine <jola@niksula.hut.fi>
 *
 * $Id: mipv6_icmp.c,v 1.1 2002/03/22 18:34:46 ayer Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/icmpv6.h>
#include <linux/spinlock.h>
#include <net/checksum.h>
#include <net/ipv6.h>

#include "debug.h"

struct icmpv6_msg {
	struct icmp6hdr icmph;
	__u8 *data;
	struct in6_addr *daddr;
	int len;
	__u32 csum;
};

#define MIPV6_SOCK_HOP_LIMIT 64

static struct socket *icmpv6_socket = NULL;
extern struct net_proto_family inet6_family_ops;

static int icmpv6_xmit_holder = -1;

static __u16 identifier = 0;

static int icmpv6_xmit_lock_bh(void)
{
	if (!spin_trylock(&icmpv6_socket->sk->lock.slock)) {
		if (icmpv6_xmit_holder == smp_processor_id())
			return -EAGAIN;
		spin_lock(&icmpv6_socket->sk->lock.slock);
	}
	icmpv6_xmit_holder = smp_processor_id();
	return 0;
}

static __inline__ int icmpv6_xmit_lock(void)
{
	int ret;
	local_bh_disable();
	ret = icmpv6_xmit_lock_bh();
	if (ret)
		local_bh_enable();
	return ret;
}

static void icmpv6_xmit_unlock_bh(void)
{
	icmpv6_xmit_holder = -1;
	spin_unlock(&icmpv6_socket->sk->lock.slock);
}
 
static __inline__ void icmpv6_xmit_unlock(void)
{
	icmpv6_xmit_unlock_bh();
	local_bh_enable();
}

static int icmpv6_getfrag(const void *data, struct in6_addr *saddr, 
			   char *buff, unsigned int offset, unsigned int len)
{
	struct icmpv6_msg *msg = (struct icmpv6_msg *) data;
	struct icmp6hdr *icmph;
	__u32 csum;

	csum = csum_partial_copy_nocheck((void *) &msg->icmph, buff,
					 sizeof(struct icmp6hdr), msg->csum);

	csum = csum_partial_copy_nocheck((void *) msg->data, 
					 buff + sizeof(struct icmp6hdr),
					 len - sizeof(struct icmp6hdr), csum);

	icmph = (struct icmp6hdr *) buff;

	icmph->icmp6_cksum = csum_ipv6_magic(saddr, msg->daddr, msg->len,
					     IPPROTO_ICMPV6, csum);
	return 0; 
}

/**
 * mipv6_icmpv6_send - generic icmpv6 message send
 * @daddr: destination address
 * @saddr: source address
 * @type: icmp type
 * @code: icmp code
 * @id: packet identifier. If null, uses internal counter to get new id
 * @data: packet data
 * @datalen: length of data in bytes
 */
void mipv6_icmpv6_send(struct in6_addr *daddr, struct in6_addr *saddr, int type,
		       int code, __u16 *id, void *data, int datalen, int iif)
{
	struct sock *sk = icmpv6_socket->sk;
	struct icmpv6_msg msg;
	struct flowi fl;
	int len;

	memset(&msg, 0, sizeof(struct icmpv6_msg));

	fl.proto = IPPROTO_ICMPV6;
	fl.fl6_dst = daddr;
	fl.fl6_src = saddr;
	fl.oif = iif;
	fl.fl6_flowlabel = 0;
	fl.uli_u.icmpt.type = type;
	fl.uli_u.icmpt.code = code;

	if (icmpv6_xmit_lock())
		return;

	msg.icmph.icmp6_type = type;
	msg.icmph.icmp6_code = code;
	msg.icmph.icmp6_cksum = 0;

	if (id)
		msg.icmph.icmp6_identifier = htons(*id);
	else
		msg.icmph.icmp6_identifier = htons(identifier++);

	msg.data = data;
	msg.csum = 0;
	msg.daddr = daddr;

	if ((len = datalen + sizeof(struct icmp6hdr)) < 0) {
		DEBUG((DBG_ERROR, "icmp: len problem"));
		goto out;
	}

	msg.len = len;

	ip6_build_xmit(sk, icmpv6_getfrag, &msg, &fl, len, NULL, -1,
		       MSG_DONTWAIT);

	ICMP6_INC_STATS_BH(Icmp6OutMsgs);

 out:
	icmpv6_xmit_unlock();
}


int mipv6_initialize_icmpv6(void)
{
	struct net_proto_family *ops = &inet6_family_ops;
	struct sock *sk;
	int err;

	if ((icmpv6_socket = sock_alloc()) == NULL) {
		DEBUG((DBG_CRITICAL, "Cannot allocate icmpv6_socket"));
		return -1;
	}

	icmpv6_socket->inode->i_uid = 0;
	icmpv6_socket->inode->i_gid = 0;
	icmpv6_socket->type = SOCK_RAW;

	if ((err = ops->create(icmpv6_socket, NEXTHDR_NONE)) < 0) {
		DEBUG((DBG_CRITICAL, "Cannot initialize icmpv6_socket"));
		sock_release(icmpv6_socket);
		icmpv6_socket = NULL; /* For safety */
		return err;
	}

	sk = icmpv6_socket->sk;
	sk->allocation = GFP_ATOMIC;
	sk->net_pinfo.af_inet6.hop_limit = MIPV6_SOCK_HOP_LIMIT;
	sk->prot->unhash(sk);

	return 0;
}

void mipv6_shutdown_icmpv6(void)
{
	if (icmpv6_socket)
		sock_release(icmpv6_socket);
	icmpv6_socket = NULL; /* For safety */
}
