/*
 * linux/drivers/video/epson1356fb.c -- Epson 1356 LCD Controller Frame Buffer Device
 *
 *  Copyright (C) 2001 MIT
 *
 * Edited from sa1100fb.c
 *  Copyright (C) 1999 Eric A. Thomas
 *   Based on acornfb.c Copyright (C) Russell King.
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/ctype.h>
#include <linux/init.h>
#include <linux/fb.h>
#include <linux/delay.h>
#include <linux/wrapper.h>
#include <asm/uaccess.h>
#include <asm/proc/pgtable.h>
#include <video/fbcon.h>
#include <video/fbcon-cfb16.h>

#define EPSON1356_NAME	"EPSON1356"
#define REGISTER_OFFSET	((unsigned char *) 0xf0000000) /* SED1356 registers */
#define DISP_MEM_OFFSET	((unsigned char *) 0xf1000000) /* display buffer */
#define DISP_MEM_OFFSET_PHYS ((unsigned char *)0x48200000) /* display buffer */
#define ALLOCATED_FB_MEM_SIZE 0x80000 /* display memory size (512kb) */

#define MAX_BPP		16
#define MAX_XRES	640
#define MAX_YRES	240
#define MIN_XRES	64
#define MIN_YRES	64

#define GET_DISPLAY(arg_con) ((arg_con>=0)?&fb_display[arg_con]:&global_disp)

static struct display global_disp;	/* Initial (default) Display Settings */
static unsigned short dummy_cfb16[16];
static int current_contrast = 0x60;

/* ** The SED1356 lookup table used only the upper four bits.  */
unsigned char LUT8[256*3] = {
/* Primary and secondary colors */
0x00, 0x00, 0x00,  0x00, 0x00, 0xA0,  0x00, 0xA0, 0x00,  0x00, 0xA0, 0xA0,
0xA0, 0x00, 0x00,  0xA0, 0x00, 0xA0,  0xA0, 0xA0, 0x00,  0xA0, 0xA0, 0xA0,
0x50, 0x50, 0x50,  0x00, 0x00, 0xF0,  0x00, 0xF0, 0x00,  0x00, 0xF0, 0xF0,
0xF0, 0x00, 0x00,  0xF0, 0x00, 0xF0,  0xF0, 0xF0, 0x00,  0xF0, 0xF0, 0xF0};
static char lut_base[] = {
/*red    green  blue   rinc   ginc   binc  */
  0x00,  0x00,  0x00,  0x10,  0x10,  0x10, /* Gray shades */
  0x00,  0x00,  0x00,  0x10,  0x00,  0x00, /* Black to red */
  0x00,  0x00,  0x00,  0x00,  0x10,  0x00, /* Black to green */
  0x00,  0x00,  0x00,  0x00,  0x00,  0x10, /* Black to blue */
  0x00,  0x00,  0xF0,  0x00,  0x10,  0x00, /* Blue to cyan (blue and green) */
  0x00,  0xf0,  0xf0,  0x00,  0x00, -0x10, /* Cyan (blue and green) to green */
  0x00,  0xf0,  0x00,  0x10,  0x00,  0x00, /* Green to yellow (red and green)*/
  0xf0,  0xf0,  0x00,  0x00, -0x10,  0x00, /* Yellow (red and green) to red */
  0xf0,  0x00,  0x00,  0x00,  0x00,  0x10, /* Red to magenta (blue and red) */
  0xf0,  0x00,  0xf0, -0x10,  0x00,  0x00, /* Magenta (blue and red) to blue */
  0x00,  0x00,  0x00,  0x10,  0x00,  0x10, /* Black to magenta (blue and red)*/
  0x00,  0x00,  0x00,  0x00,  0x10,  0x10, /* Black to cyan (blue and green) */
  0xf0,  0x00,  0x00,  0x00,  0x10,  0x10, /* Red to white */
  0x00,  0xf0,  0x00,  0x10,  0x00,  0x10, /* Green to white */
  0x00,  0x00,  0xf0,  0x10,  0x10,  0x00, /* Blue to white */
};

static int
e1356fb_get_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
printk("[%s:%d] %s()**************** con %x \n", __FILE__, __LINE__, __FUNCTION__, con);
	*var = GET_DISPLAY(con)->var;
	return 0;
}

/*
 * e1356fb_set_var():
 *	Set the user defined part of the display for the specified console
 */
long idummy;
static int
e1356fb_set_var(struct fb_var_screeninfo *var, int con, struct fb_info *info)
{
struct display *display = GET_DISPLAY(con); /* Display settings for console */
int chgvar = (var->activate & FB_ACTIVATE_MASK);

printk("[%s:%d] %s() entry\n", __FILE__, __LINE__, __FUNCTION__);

	if (var->bits_per_pixel != MAX_BPP)  /* RGB 565 */
		return -EINVAL;
	if (var->xres < MIN_XRES)
		var->xres = MIN_XRES;
	if (var->yres < MIN_YRES)
		var->yres = MIN_YRES;
	if (var->xres > MAX_XRES)
		var->xres = MAX_XRES;
	if (var->yres > MAX_YRES)
		var->yres = MAX_YRES;
	if (var->xres_virtual < var->xres)
		var->xres_virtual = var->xres;
	if (var->yres_virtual < var->yres)
		var->yres_virtual = var->yres;
	var->red.length = 5;
	var->blue.length   = 5;
	var->green.length  = 6;
	var->transp.length = 0;
	var->red.offset = 11;
	var->green.offset  = 5;
	var->blue.offset   = 0;
	var->transp.offset = 0;

	if (chgvar == FB_ACTIVATE_TEST)
		return 0;
	else if ((chgvar != FB_ACTIVATE_NOW) && (chgvar != FB_ACTIVATE_NXTOPEN))
		return -EINVAL;

	chgvar = 0;
	if ((display->var.xres != var->xres) ||
		(display->var.yres != var->yres) ||
		(display->var.xres_virtual != var->xres_virtual) ||
		(display->var.yres_virtual != var->yres_virtual) )
		chgvar = 1;

	display->var = *var;
	display->screen_base = global_disp.screen_base;
	display->visual = FB_VISUAL_TRUECOLOR;
	display->type = FB_TYPE_PACKED_PIXELS;
	display->type_aux = 0;
	display->ypanstep = 0;
	display->ywrapstep = 0;
	display->line_length =
	display->next_line = (var->xres * var->bits_per_pixel) / 8;
	display->can_soft_blank = 1;
	display->inverse = 0;
	display->dispsw = &fbcon_cfb16;
	display->dispsw_data = &dummy_cfb16;

	/* If the console has changed and the console has defined */
	/* a changevar function, call that function. */
	if (con >= 0 && chgvar && info && info->changevar)
		info->changevar(con);
printk("[%s:%d] %s() exit\n", __FILE__, __LINE__, __FUNCTION__);
	return 0;
}

static int
e1356fb_get_fix(struct fb_fix_screeninfo *fix, int con, struct fb_info *info)
{
struct display *display = GET_DISPLAY(con); /* Display settings for console */

//printk("[%s:%d] %s()\n", __FILE__, __LINE__, __FUNCTION__);
	memset(fix, 0, sizeof(struct fb_fix_screeninfo));
	strcpy(fix->id, EPSON1356_NAME);
	fix->smem_start = (unsigned long)DISP_MEM_OFFSET_PHYS;
	fix->smem_len = (MAX_XRES * MAX_YRES * MAX_BPP)/8;
	fix->type = FB_TYPE_PACKED_PIXELS;
	fix->visual = FB_VISUAL_TRUECOLOR;
	fix->line_length = (display->var.xres * MAX_BPP) / 8;
	fix->accel = FB_ACCEL_NONE;
	return 0;
}

/*
 *  e1356fb_switch():
 *	Change to the specified console.
 */
static int e1356fb_switch(int con, struct fb_info *info)
{
struct display *display = GET_DISPLAY(con); /* Display settings for console */
	display->var.activate = FB_ACTIVATE_NOW;
	e1356fb_set_var(&display->var, con, info);
	return 0;
}

static int e1356fb_updatevar(int con, struct fb_info *info)
{
printk("e1356fb_updatevar: entered\n");
	return 0;
}

/*
 * e1356fb_blank():
 */
static void e1356fb_blank(int blank, struct fb_info *info)
{
int i;
printk("e1356fb_blank: blank=%d info->modename=%s\n", blank, info->modename);
	switch (blank) {
	case VESA_POWERDOWN:
	case VESA_VSYNC_SUSPEND:
	case VESA_HSYNC_SUSPEND:
		/* turn light off */
		jornada_contrast(0);
		jornada_brightness(0);
		break;
	case VESA_NO_BLANKING:
		/* turn light on */
		i = jornada_brightness(8);
printk("brightness %x\n", i);
		jornada_contrast(current_contrast);
	}
}

static int
e1356fb_get_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
printk("[%s:%d] %s()\n", __FILE__, __LINE__, __FUNCTION__);
	return 0;
}

static int
e1356fb_set_cmap(struct fb_cmap *cmap, int kspc, int con, struct fb_info *info)
{
printk("[%s:%d] %s()\n", __FILE__, __LINE__, __FUNCTION__);
	return 0;
}
static struct fb_info my_fb_info;
static struct fb_ops e1356fb_ops = {
	owner: THIS_MODULE,
	fb_get_fix: e1356fb_get_fix,
	fb_get_var: e1356fb_get_var,
	fb_set_var: e1356fb_set_var,
	fb_get_cmap: e1356fb_get_cmap,
	fb_set_cmap: e1356fb_set_cmap,
};
/* Fake monspecs to fill in fbinfo structure */
static struct fb_monspecs monspecs __initdata = {
	30000, 70000, 50, 65, 0	/* Generic */
};

int __init e1356fb_init(void)
{
struct page *page;
unsigned char *pRegs = REGISTER_OFFSET;
unsigned char *pMem = DISP_MEM_OFFSET;
unsigned char *pLUT = LUT8;
unsigned char *pseed = lut_base;
unsigned char plast[3];
long lCnt;
int i, j, rgb;

	printk("Configuring Jornada LCD\n");
	/* Remap the fb memory to a non-buffered, non-cached region */
	global_disp.screen_base = (u_char *)__ioremap((u_long)DISP_MEM_OFFSET_PHYS,
		ALLOCATED_FB_MEM_SIZE, L_PTE_PRESENT | L_PTE_YOUNG | L_PTE_DIRTY | L_PTE_WRITE);
	/* Set reserved flag for fb memory to allow it to be remapped into */
	/* user space by the common fbmem driver using remap_page_range(). */
	for(page = virt_to_page(global_disp.screen_base);
		page < virt_to_page(global_disp.screen_base + ALLOCATED_FB_MEM_SIZE); page++)
	mem_map_reserve(page);

	memset(&global_disp.var, 0, sizeof(global_disp.var));
	global_disp.var.bits_per_pixel = MAX_BPP;
	global_disp.var.xres = MAX_XRES;
	global_disp.var.yres = MAX_YRES;
	global_disp.var.height = -1;
	global_disp.var.width = -1;
	global_disp.var.vmode = FB_VMODE_NONINTERLACED;
	e1356fb_switch(-1, NULL);

	PPSR &= ~(PPC_LDD0 | PPC_LDD1 | PPC_LDD2);
	PPDR |= PPC_LDD0 | PPC_LDD1 | PPC_LDD2;
	LCCR3 = 0;
	LCCR2 = 0;
	LCCR1 = 0;
	LCCR0 = 0;
	DBAR1 = 0;
	DBAR2 = 0;

	/* Enable access to SED1356 by setting memory/register select bit to 0. */
	pRegs[0x001] = 0;

	/* Disable display outputs during initialization) */
	pRegs[0x1FC] = 0;

	/* Set the GPIOs to input. Should GPIO bits in register [004] get switched
	then the GPIO outputs, according to register [008], are driven low. */
	pRegs[0x004] = 0;
	pRegs[0x008] = 0;

	/* Program the LCD panel type and panel timing registers.
	*	The horizontal and vertical non-display times have been
	*	calculated for a 78 Hz frame rated.
	*								LCD PCLK
	*			FrameRate = -----------------------------
	*						(HDP + HNDP) * (VDP/2 + VNDP)
	*
	*								20,000,000
	* = -----------------------------
	*						(640 + 256) * (480 / 2 + 45)
	*
	* = 78 Hz */

	pRegs[0x10] = 0x1;  /* Memory Clock Configuration Register */
	pRegs[0x14] = 0x11; /* LCD Pixel Clock Configuration Register */
	pRegs[0x18] = 0x1;  /* CRT/TV Pixel Clock Configuration Register */
	pRegs[0x1c] = 0x1;  /* MediaPlug Clock Configuration Register */
	pRegs[0x1e] = 0x1;  /* CPU To Memory Wait State Select Register */
	pRegs[0x20] = 0;    /* Memory Configuration Register */
	pRegs[0x21] = 0x45; /* DRAM Refresh Rate Register */
	pRegs[0x2a] = 0x1;  /* DRAM Timings Control Register 0 */
	pRegs[0x2b] = 0x1;  /* DRAM Timings Control Register 1 */
	pRegs[0x30] = 0x1c; /* Panel Type Register */
	pRegs[0x31] = 0;    /* MOD Rate Register */
	pRegs[0x32] = 0x4f;   /* LCD Horizontal Display Width Register */
	pRegs[0x34] = 0x7;   /* LCD Horizontal Non-Display Period Register */
	pRegs[0x35] = 0x1;   /* TFT FPLINE Start Position Register */
	pRegs[0x36] = 0xb;   /* TFT FPLINE Pulse Width Register */
	pRegs[0x38] = 0xef;   /* LCD Vertical Display Height Register 0 */
	pRegs[0x39] = 0;   /* LCD Vertical Display Height Register 1 */
	pRegs[0x3a] = 0x13;   /* LCD Vertical Non-Display Period Register */
	pRegs[0x3b] = 0xb;   /* TFT FPFRAME Start Position Register */
	pRegs[0x3c] = 0x1;   /* TFT FPFRAME Pulse Width Register */
	pRegs[0x40] = 0x5;   /* LCD Display Mode Register */
	pRegs[0x41] = 0;   /* LCD Miscellaneous Register */
	pRegs[0x42] = 0;   /* LCD Display Start Address Register 0 */
	pRegs[0x43] = 0;   /* LCD Display Start Address Register 1 */
	pRegs[0x44] = 0;   /* LCD Display Start Address Register 2 */
	pRegs[0x46] = 0x80;   /* LCD Memory Address Offset Register 0 */
	pRegs[0x47] = 0x2;   /* LCD Memory Address Offset Register 1 */
	pRegs[0x48] = 0;   /* LCD Pixel Panning Register */
	pRegs[0x4a] = 0;   /* LCD Display FIFO High Threshold Control Register */
	pRegs[0x4b] = 0;   /* LCD Display FIFO Low Threshold Control Register */
	pRegs[0x50] = 0x4f;   /* CRT/TV Horizontal Display Width Register */
	pRegs[0x52] = 0x13;   /* CRT/TV Horizontal Non-Display Period Register */
	pRegs[0x53] = 0x1;   /* CRT/TV HRTC Start Position Register */
	pRegs[0x54] = 0xb;   /* CRT/TV HRTC Pulse Width Register */
	pRegs[0x56] = 0xdf;   /* CRT/TV Vertical Display Height Register 0 */
	pRegs[0x57] = 0x1;   /* CRT/TV Vertical Display Height Register 1 */
	pRegs[0x58] = 0x2b;   /* CRT/TV Vertical Non-Display Period Register */
	pRegs[0x59] = 0x9;   /* CRT/TV VRTC Start Position Register */
	pRegs[0x5a] = 0x1;   /* CRT/TV VRTC Pulse Width Register */
	pRegs[0x5b] = 0x10;   /* TV Output Control Register */
	pRegs[0x60] = 0x3;   /* CRT/TV Display Mode Register */
	pRegs[0x62] = 0;   /* CRT/TV Display Start Address Register 0 */
	pRegs[0x63] = 0;   /* CRT/TV Display Start Address Register 1 */
	pRegs[0x64] = 0;   /* CRT/TV Display Start Address Register 2 */
	pRegs[0x66] = 0x40;   /* CRT/TV Memory Address Offset Register 0 */
	pRegs[0x67] = 0x1;   /* CRT/TV Memory Address Offset Register 1 */
	pRegs[0x68] = 0;   /* CRT/TV Pixel Panning Register */
	pRegs[0x6a] = 0;   /* CRT/TV Display FIFO High Threshold Control Register */
	pRegs[0x6b] = 0;   /* CRT/TV Display FIFO Low Threshold Control Register */
	pRegs[0x70] = 0;   /* LCD Ink/Cursor Control Register */
	pRegs[0x71] = 0x1;   /* LCD Ink/Cursor Start Address Register */
	pRegs[0x72] = 0;   /* LCD Cursor X Position Register 0 */
	pRegs[0x73] = 0;   /* LCD Cursor X Position Register 1 */
	pRegs[0x74] = 0;   /* LCD Cursor Y Position Register 0 */
	pRegs[0x75] = 0;   /* LCD Cursor Y Position Register 1 */
	pRegs[0x76] = 0;   /* LCD Ink/Cursor Blue Color 0 Register */
	pRegs[0x77] = 0;   /* LCD Ink/Cursor Green Color 0 Register */
	pRegs[0x78] = 0;   /* LCD Ink/Cursor Red Color 0 Register */
	pRegs[0x7a] = 0x1f;   /* LCD Ink/Cursor Blue Color 1 Register */
	pRegs[0x7b] = 0x3f;   /* LCD Ink/Cursor Green Color 1 Register */
	pRegs[0x7c] = 0x1f;   /* LCD Ink/Cursor Red Color 1 Register */
	pRegs[0x7e] = 0;   /* LCD Ink/Cursor FIFO Threshold Register */
	pRegs[0x80] = 0;   /* CRT/TV Ink/Cursor Control Register */
	pRegs[0x81] = 0x1;   /* CRT/TV Ink/Cursor Start Address Register */
	pRegs[0x82] = 0;   /* CRT/TV Cursor X Position Register 0 */
	pRegs[0x83] = 0;   /* CRT/TV Cursor X Position Register 1 */
	pRegs[0x84] = 0;   /* CRT/TV Cursor Y Position Register 0 */
	pRegs[0x85] = 0;   /* CRT/TV Cursor Y Position Register 1 */
	pRegs[0x86] = 0;   /* CRT/TV Ink/Cursor Blue Color 0 Register */
	pRegs[0x87] = 0;   /* CRT/TV Ink/Cursor Green Color 0 Register */
	pRegs[0x88] = 0;   /* CRT/TV Ink/Cursor Red Color 0 Register */
	pRegs[0x8a] = 0x1f;   /* CRT/TV Ink/Cursor Blue Color 1 Register */
	pRegs[0x8b] = 0x3f;   /* CRT/TV Ink/Cursor Green Color 1 Register */
	pRegs[0x8c] = 0x1f;   /* CRT/TV Ink/Cursor Red Color 1 Register */
	pRegs[0x8e] = 0;   /* CRT/TV Ink/Cursor FIFO Threshold Register */

	/* Set the 2D acceleration (BitBLT) registers to a known state */
	for (i = 0x100; i <= 0x119; i++)
		if (i != 0x107 && i != 0x10b && i != 0x10e && i != 0x10f && i != 0x117)
			pRegs[i] = 0x00;	/* 0000 0000 */
	/* Program the look-up table to a known state.  */
	pRegs[0x1E0] = 0x01;	/* Enable the LCD LUT for read/write. */
	pRegs[0x1E2] = 0;		/* Reset the LUT address. */
	for (i = 0; i < 16 * 3; i++)
		pRegs[0x1E4] = *pLUT++;	/* non-regular color template */
	for (i = 0; i < 15; i++) {
		for (rgb = 0; rgb < 3; rgb++) {
			plast[rgb] = *pseed++;		/* base color value */
			pRegs[0x1E4] = plast[rgb];
		}
		for (j = 0; j < 15; j++)
			for (rgb = 0; rgb < 3; rgb++) {
				plast[rgb] += pseed[rgb];	/* increment through color values*/
				pRegs[0x1E4] = plast[rgb];
			}
		pseed += 3;
	}
	pRegs[0x1e4] = 0;   /* Look-Up Table Data Register */
	pRegs[0x1f0] = 0;   /* Power Save Configuration Register */
	pRegs[0x1f1] = 0;   /* Power Save Status Register */
	pRegs[0x1f4] = 0;   /* CPU-to-Memory Access Watchdog Timer Register */

	for (lCnt = 0; lCnt < ALLOCATED_FB_MEM_SIZE; lCnt++)
		*pMem++ = 0;		/* Clear display memory */

	PPSR |= PPC_LDD0;
	for (idummy = 0; idummy < 1000000; idummy++)
		; /* Wait for 100ms */
	/* Turn off power save mode.  */
	pRegs[0x1F0] = 0;		/* 000 000 - Disable Power Save Mode. */
	/* Disable the watchdog timer.  */
	pRegs[0x1F4] = 0;		/* 0000 0000 */
	/* Enable the display.  */
	pRegs[0x1FC] = 0x01;	/* 0000 0001 - Disable Power Save Mode. */

	jornada_contrast(0x36);
	PPSR |= PPC_LDD2;
	for (idummy = 0; idummy < 1000000; idummy++)
		; /* Wait for 100ms */
	e1356fb_blank(VESA_NO_BLANKING, NULL);
	PPSR |= PPC_LDD1;

	my_fb_info.node = -1;
	my_fb_info.flags = FBINFO_FLAG_DEFAULT;
	my_fb_info.fbops = &e1356fb_ops;
	my_fb_info.monspecs = monspecs;
	my_fb_info.disp = &global_disp;
	my_fb_info.changevar = NULL;
	my_fb_info.switch_con = e1356fb_switch;
	my_fb_info.updatevar = e1356fb_updatevar;
	my_fb_info.blank = e1356fb_blank;
	register_framebuffer(&my_fb_info);
printk("[%s:%d] %s() exit\n", __FILE__, __LINE__, __FUNCTION__);
	return 0;
}

int __init e1356fb_setup(char *options)
{
	/* No parameters for now */
	return 0;
}

void e1356fb_cleanup(struct fb_info *info)
{
    unregister_framebuffer(info);
}

#ifdef MODULE
MODULE_LICENSE("GPL");
int init_module(void)
{
    return e1356fb_init();
}

void cleanup_module(void)
{
    e1356fb_cleanup(void);
}
#endif /* MODULE */
