/*
 *	Video for Linux Two
 *	Backward Compatibility Layer
 *
 *	Support subroutines for providing V4L2 drivers with backward
 *	compatibility with applications using the old API.
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 * Author:	Bill Dirks <bdirks@pacbell.net>
 *		et al.
 *
 */

#ifndef __KERNEL__
#define __KERNEL__
#endif


#include <linux/config.h>
#ifndef EXPORT_SYMTAB
#define EXPORT_SYMTAB
#endif


#include <linux/version.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/mm.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <videodevx/videodev.h>

#if LINUX_VERSION_CODE >= 0x020100
#include <asm/uaccess.h>
#endif
#include <asm/system.h>
#include <asm/pgtable.h>
#include <asm/io.h>

#ifdef CONFIG_KMOD
#include <linux/kmod.h>
#endif

#if LINUX_VERSION_CODE == 0x020409
#undef min
#define min(a,b) (((a)<(b))?(a):(b))
#endif

#ifdef MODULE_LICENSE
MODULE_LICENSE("GPL");
#endif


/*
 *	M E M O R Y   M A P P I N G
 *
 *	V4l apps pass zero for the offset on mmap() calls. The V4L2
 *	driver needs its special offset value associated with the
 *	buffer. Get the special value and fix up the vma argument so
 *	that it is now suitable to pass to a V4L2 driver.
 */
static void
v4l_compat_fix_offset(struct file *file,
		      struct v4l2_device *vfl,
		      struct vm_area_struct *vma)
{
#if LINUX_VERSION_CODE >= 0x020300
	if (vma->vm_pgoff == 0 && vfl->ioctl)
#else
	if (vma->vm_offset == 0 && vfl->ioctl)
#endif
	{
		struct v4l2_buffer buf;
		buf.index = 0;
		buf.type = V4L2_BUF_TYPE_CAPTURE;
		if (vfl->ioctl(file->private_data,
			       VIDIOC_QUERYBUF, &buf) == 0)
#if LINUX_VERSION_CODE >= 0x020300
			vma->vm_pgoff = buf.offset / PAGE_SIZE;
#else
			vma->vm_offset = buf.offset;
#endif
	}
}


/*
 *	I O C T L   T R A N S L A T I O N
 *
 *	From here on down is the code for translating the numerous
 *	ioctl commands from the old API to the new API.
 */

static int
get_v4l_control(struct v4l2_device	*vfl,
		void			*context,
		int			cid)
{
	struct v4l2_queryctrl	qctrl2;
	struct v4l2_control	ctrl2;
	int			err;

	qctrl2.id = cid;
	err = vfl->ioctl(context, VIDIOC_QUERYCTRL, &qctrl2);
	if (err == 0 &&
	    !(qctrl2.flags & V4L2_CTRL_FLAG_DISABLED))
	{
		ctrl2.id = qctrl2.id;
		vfl->ioctl(context, VIDIOC_G_CTRL, &ctrl2);
		return ((ctrl2.value - qctrl2.minimum) * 65535
			 + (qctrl2.maximum - qctrl2.minimum) / 2)
			/ (qctrl2.maximum - qctrl2.minimum);
	}
	return 0;
}

static int
set_v4l_control(struct v4l2_device	*vfl,
		void			*context,
		int			cid,
		int			value)
{
	struct v4l2_queryctrl	qctrl2;
	struct v4l2_control	ctrl2;
	int			err;

	qctrl2.id = cid;
	err = vfl->ioctl(context, VIDIOC_QUERYCTRL, &qctrl2);
	if (err == 0 &&
	    !(qctrl2.flags & V4L2_CTRL_FLAG_DISABLED) &&
	    !(qctrl2.flags & V4L2_CTRL_FLAG_GRABBED))
	{
		if (value < 0)
			value = 0;
		if (value > 65535)
			value = 65535;
		if (value && qctrl2.type == V4L2_CTRL_TYPE_BOOLEAN)
			value = 65535;
		ctrl2.id = qctrl2.id;
		ctrl2.value = 
			(value * (qctrl2.maximum - qctrl2.minimum)
			 + 32767)
			/ 65535;
		ctrl2.value += qctrl2.minimum;
		vfl->ioctl(context, VIDIOC_S_CTRL, &ctrl2);
	}
	return 0;
}

static int
find_tuner(struct v4l2_device	*vfl,
	   void			*context,
	   int			n)
{
	struct v4l2_input	inp2;
	int	i;
	int	err=0;

	/*  Find the input number of the n'th tuner  */
	for (i = 0; i < 100/*arbitrary*/; ++i)
	{
		inp2.index = i;
		err = vfl->ioctl(context, VIDIOC_ENUMINPUT, &inp2);
		if (err < 0)
			break;
		if (inp2.type != V4L2_INPUT_TYPE_TUNER)
			continue;
		if (n == 0)
			break;
		--n;
	}
	if (err < 0)
		return err;
	return i;
}

static int
palette_to_pixelformat(int palette)
{
	int	pixelformat = 0;
	switch (palette)
	{
	case VIDEO_PALETTE_GREY:
		pixelformat = V4L2_PIX_FMT_GREY;
		break;
	case VIDEO_PALETTE_RGB555:
		pixelformat = V4L2_PIX_FMT_RGB555;
		break;
	case VIDEO_PALETTE_RGB565:
		pixelformat = V4L2_PIX_FMT_RGB565;
		break;
	case VIDEO_PALETTE_RGB24:
		pixelformat = V4L2_PIX_FMT_BGR24;
		break;
	case VIDEO_PALETTE_RGB32:
		pixelformat = V4L2_PIX_FMT_BGR32;
		break;
	case VIDEO_PALETTE_YUYV:
	case VIDEO_PALETTE_YUV422:
		pixelformat = V4L2_PIX_FMT_YUYV;
		break;
	case VIDEO_PALETTE_UYVY:
		pixelformat = V4L2_PIX_FMT_UYVY;
		break;
	case VIDEO_PALETTE_YUV420:
	case VIDEO_PALETTE_YUV420P:
		pixelformat = V4L2_PIX_FMT_YUV420;
		break;
	case VIDEO_PALETTE_YUV422P:
		pixelformat = V4L2_PIX_FMT_YUV422P;
		break;
	case VIDEO_PALETTE_YUV411P:
	case VIDEO_PALETTE_YUV411:
		pixelformat = V4L2_PIX_FMT_YUV411P;
		break;
	case VIDEO_PALETTE_YUV410P:
		pixelformat = V4L2_PIX_FMT_YUV410;
		break;
	}
	return pixelformat;
}

static int
pixelformat_to_palette(int pixelformat)
{
	int	palette = 0;
	switch (pixelformat)
	{
	case V4L2_PIX_FMT_GREY:
		palette = VIDEO_PALETTE_GREY;
		break;
	case V4L2_PIX_FMT_RGB555:
		palette = VIDEO_PALETTE_RGB555;
		break;
	case V4L2_PIX_FMT_RGB565:
		palette = VIDEO_PALETTE_RGB565;
		break;
	case V4L2_PIX_FMT_BGR24:
		palette = VIDEO_PALETTE_RGB24;
		break;
	case V4L2_PIX_FMT_BGR32:
		palette = VIDEO_PALETTE_RGB32;
		break;
	case V4L2_PIX_FMT_YUYV:
		palette = VIDEO_PALETTE_YUYV;
		break;
	case V4L2_PIX_FMT_UYVY:
		palette = VIDEO_PALETTE_UYVY;
		break;
	case V4L2_PIX_FMT_YUV420:
		palette = VIDEO_PALETTE_YUV420P;
		break;
	case V4L2_PIX_FMT_YUV422P:
		palette = VIDEO_PALETTE_YUV422P;
		break;
	case V4L2_PIX_FMT_YUV411P:
		palette = VIDEO_PALETTE_YUV411P;
		break;
	}
	return palette;
}

/*  Do an 'in' (wait for input) select on a single file descriptor  */
/*  This stuff plaigarized from linux/fs/select.c     */
#define __FD_IN(fds, n)		(fds->in + n)
#define BIT(i)		(1UL << ((i)&(__NFDBITS-1)))
#define SET(i,m)	(*(m) |= (i))

#ifndef HAVE_DO_SELECT

/*  Note: This code, inside the #ifndef HAVE_DO_SELECT ... #endif,
    is copied from fs/select.c, and is only included
    here because do_select() is not exported to modules. In the future
    export do_select() to modules and delete this code down to the
    #endif. Thank you.
*/

#if LINUX_VERSION_CODE >= 0x020300
#include <linux/slab.h>
#include <linux/smp_lock.h>
#include <linux/file.h>

#define DEFAULT_POLLMASK (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM)

#define my__IN(fds, n)		(fds->in + n)
#define my__OUT(fds, n)		(fds->out + n)
#define my__EX(fds, n)		(fds->ex + n)
#define my__RES_IN(fds, n)	(fds->res_in + n)
#define my__RES_OUT(fds, n)	(fds->res_out + n)
#define my__RES_EX(fds, n)	(fds->res_ex + n)

#define BITS(fds, n)		(*my__IN(fds, n)|*my__OUT(fds, n)|*my__EX(fds, n))

static int max_select_fd(unsigned long n, fd_set_bits *fds)
{
	unsigned long *open_fds;
	unsigned long set;
	int max;

	/* handle last in-complete long-word first */
	set = ~(~0UL << (n & (__NFDBITS-1)));
	n /= __NFDBITS;
	open_fds = current->files->open_fds->fds_bits+n;
	max = 0;
	if (set) {
		set &= BITS(fds, n);
		if (set) {
			if (!(set & ~*open_fds))
				goto get_max;
			return -EBADF;
		}
	}
	while (n) {
		open_fds--;
		n--;
		set = BITS(fds, n);
		if (!set)
			continue;
		if (set & ~*open_fds)
			return -EBADF;
		if (max)
			continue;
get_max:
		do {
			max++;
			set >>= 1;
		} while (set);
		max += n * __NFDBITS;
	}

	return max;
}

#define BIT(i)		(1UL << ((i)&(__NFDBITS-1)))
#define MEM(i,m)	((m)+(unsigned)(i)/__NFDBITS)
#define ISSET(i,m)	(((i)&*(m)) != 0)
#define SET(i,m)	(*(m) |= (i))

#define POLLIN_SET (POLLRDNORM | POLLRDBAND | POLLIN | POLLHUP | POLLERR)
#define POLLOUT_SET (POLLWRBAND | POLLWRNORM | POLLOUT | POLLERR)
#define POLLEX_SET (POLLPRI)

int my_do_select(int n, fd_set_bits *fds, long *timeout)
{
	poll_table table, *wait;
	int retval, i, off;
	long __timeout = *timeout;

 	read_lock(&current->files->file_lock);
	retval = max_select_fd(n, fds);
	read_unlock(&current->files->file_lock);

	if (retval < 0)
		return retval;
	n = retval;

	poll_initwait(&table);
	wait = &table;
	if (!__timeout)
		wait = NULL;
	retval = 0;
	for (;;) {
		set_current_state(TASK_INTERRUPTIBLE);
		for (i = 0 ; i < n; i++) {
			unsigned long bit = BIT(i);
			unsigned long mask;
			struct file *file;

			off = i / __NFDBITS;
			if (!(bit & BITS(fds, off)))
				continue;
			file = fget(i);
			mask = POLLNVAL;
			if (file) {
				mask = DEFAULT_POLLMASK;
				if (file->f_op && file->f_op->poll)
					mask = file->f_op->poll(file, wait);
				fput(file);
			}
			if ((mask & POLLIN_SET) && ISSET(bit, my__IN(fds,off))) {
				SET(bit, my__RES_IN(fds,off));
				retval++;
				wait = NULL;
			}
			if ((mask & POLLOUT_SET) && ISSET(bit, my__OUT(fds,off))) {
				SET(bit, my__RES_OUT(fds,off));
				retval++;
				wait = NULL;
			}
			if ((mask & POLLEX_SET) && ISSET(bit, my__EX(fds,off))) {
				SET(bit, my__RES_EX(fds,off));
				retval++;
				wait = NULL;
			}
		}
		wait = NULL;
		if (retval || !__timeout || signal_pending(current))
			break;
		if(table.error) {
			retval = table.error;
			break;
		}
		__timeout = schedule_timeout(__timeout);
	}
	current->state = TASK_RUNNING;

	poll_freewait(&table);

	/*
	 * Up-to-date the caller timeout.
	 */
	*timeout = __timeout;
	return retval;
}


#else /* 2.2 */

#include <linux/smp_lock.h>
#include <linux/file.h>


#define BITS(fds, n)		(*__FD_IN(fds, n))
#define DEFAULT_POLLMASK (POLLIN | POLLOUT | POLLRDNORM | POLLWRNORM)
#define POLLIN_SET (POLLRDNORM | POLLRDBAND | POLLIN | POLLHUP | POLLERR)
#define ISSET(i,m)	(((i)&*(m)) != 0)
static void free_wait(poll_table * p)
{
        struct poll_table_entry * entry;
        poll_table *old;

        while (p) {
                entry = p->entry + p->nr;
                while (p->nr > 0) {
                        p->nr--;
                        entry--;
                        remove_wait_queue(entry->wait_address,&entry->wait);
                        fput(entry->filp);
                }
                old = p;
                p = p->next;
                free_page((unsigned long) old);
        }
}
static int 
my_do_select(int n, fd_set_bits *fds, long *timeout)
{
	poll_table *wait_table, *wait;
	int retval, i, off;
	long __timeout = *timeout;

	wait = wait_table = NULL;
	if (__timeout) {
		wait_table = (poll_table *) __get_free_page(GFP_KERNEL);
		if (!wait_table)
			return -ENOMEM;

		wait_table->nr = 0;
		wait_table->entry = (struct poll_table_entry *)(wait_table + 1);
		wait_table->next = NULL;
		wait = wait_table;
	}

	lock_kernel();

	//	retval = max_select_fd(n, fds);
	//if (retval < 0)
	//	goto out;
	//n = retval;
	retval = 0;
	for (;;) {
		current->state = TASK_INTERRUPTIBLE;
		for (i = 0 ; i < n; i++) {
			unsigned long bit = BIT(i);
			unsigned long mask;
			struct file *file;

			off = i / __NFDBITS;
			if (!(bit & BITS(fds, off)))
				continue;
			/*
			 * The poll_wait routine will increment f_count if
			 * the file is added to the wait table, so we don't
			 * need to increment it now.
			 */
			file = fcheck(i);
			mask = POLLNVAL;
			if (file) {
				mask = DEFAULT_POLLMASK;
				if (file->f_op && file->f_op->poll)
					mask = file->f_op->poll(file, wait);
			}
			if ((mask & POLLIN_SET) && ISSET(bit, __FD_IN(fds,off))) {
				retval++;
				wait = NULL;
			}
		}
		wait = NULL;
		if (retval || !__timeout || signal_pending(current))
			break;
		__timeout = schedule_timeout(__timeout);
	}
	current->state = TASK_RUNNING;

//out:
	if (*timeout)
		free_wait(wait_table);

	/*
	 * Up-to-date the caller timeout.
	 */
	*timeout = __timeout;
	unlock_kernel();
	return retval;
}
#endif /* 2.3 */


#endif // HAVE_DO_SELECT


static int
simple_select(struct file *file)
{
	fd_set_bits fds;
	char *bits;
	long timeout;
	int i, fd, n, ret, size;

	for (i = 0; i < current->files->max_fds; ++i)
		if (file == current->files->fd[i])
			break;
	if (i == current->files->max_fds)
		return -EINVAL;
	fd = i;
	n = fd + 1;

	timeout = MAX_SCHEDULE_TIMEOUT;
	/*
	 * We need 6 bitmaps (in/out/ex for both incoming and outgoing),
	 * since we used fdset we need to allocate memory in units of
	 * long-words. 
	 */
	ret = -ENOMEM;
	size = FDS_BYTES(n);
	bits = kmalloc(6 * size, GFP_KERNEL);
	if (!bits)
		goto out_nofds;
	fds.in      = (unsigned long *)  bits;
	fds.out     = (unsigned long *) (bits +   size);
	fds.ex      = (unsigned long *) (bits + 2*size);
	fds.res_in  = (unsigned long *) (bits + 3*size);
	fds.res_out = (unsigned long *) (bits + 4*size);
	fds.res_ex  = (unsigned long *) (bits + 5*size);

	/*  All zero except our one file descriptor bit, for input  */
	memset(bits, 0, 6 * size);
	SET(BIT(fd), __FD_IN((&fds), fd / __NFDBITS));

	ret = my_do_select(n, &fds, &timeout);

	if (ret < 0)
		goto out;
	if (!ret) {
		ret = -ERESTARTNOHAND;
		if (signal_pending(current))
			goto out;
		ret = 0;
	}
out:
	kfree(bits);
out_nofds:
	return ret;
}


/*
 *	This function is exported.
 */
static int
v4l_compat_translate_ioctl(struct file		*file,
			   struct v4l2_device	*vfl,
			   void			*context,
			   int			cmd,
			   void			*arg)
{
	int	err	= -ENOIOCTLCMD;

	switch (cmd)
	{
	case VIDIOCGCAP:	/* capability */
	{
		struct video_capability *cap = arg;
		struct v4l2_capability cap2;
		struct v4l2_framebuffer fbuf2;

		err = vfl->ioctl(context, VIDIOC_QUERYCAP, &cap2);
		if (err < 0)
			break;
		if (cap2.flags & V4L2_FLAG_PREVIEW)
		{
			err = vfl->ioctl(context, VIDIOC_G_FBUF, &fbuf2);
			if (err < 0)
				memset(&fbuf2, 0, sizeof(fbuf2));
			err = 0;
		}
		memset(cap, 0, sizeof(cap));
		memcpy(cap->name, cap2.name, 
		       min(sizeof(cap->name), sizeof(cap2.name)));
		cap->name[sizeof(cap->name) - 1] = 0;
		if (cap2.type == V4L2_TYPE_CAPTURE)
			cap->type = VID_TYPE_CAPTURE;
		if (cap2.flags & V4L2_FLAG_TUNER)
			cap->type |= VID_TYPE_TUNER;
		if (cap2.flags & V4L2_FLAG_DATA_SERVICE)
			cap->type |= VID_TYPE_TELETEXT;
		if (cap2.flags & V4L2_FLAG_PREVIEW)
			cap->type |= VID_TYPE_OVERLAY;
		if (cap2.flags & V4L2_FLAG_MONOCHROME)
			cap->type |= VID_TYPE_MONOCHROME;
		if (fbuf2.flags & V4L2_FBUF_FLAG_PRIMARY)
			cap->type |= VID_TYPE_FRAMERAM;
		if (fbuf2.capability & V4L2_FBUF_CAP_CHROMAKEY)
			cap->type |= VID_TYPE_CHROMAKEY;
		if (fbuf2.capability & V4L2_FBUF_CAP_CLIPPING)
			cap->type |= VID_TYPE_CLIPPING;
		if (fbuf2.capability & V4L2_FBUF_CAP_SCALEUP ||
		    fbuf2.capability & V4L2_FBUF_CAP_SCALEDOWN)
			cap->type |= VID_TYPE_SCALES;
		cap->channels  = cap2.inputs;
		cap->audios    = cap2.audios;
		cap->maxwidth  = cap2.maxwidth;
		cap->maxheight = cap2.maxheight;
		cap->minwidth  = cap2.minwidth;
		cap->minheight = cap2.minheight;
		break;
	}
	case VIDIOCGFBUF: /*  get frame buffer  */
	{
		struct video_buffer	*buffer = arg;
		struct v4l2_framebuffer	fbuf2;

		err = vfl->ioctl(context, VIDIOC_G_FBUF, &fbuf2);
		if (err < 0)
			break;
		buffer->base = fbuf2.base[0];
		buffer->height = fbuf2.fmt.height;
		buffer->width = fbuf2.fmt.width;
		buffer->depth = fbuf2.fmt.depth;
		if (fbuf2.fmt.flags & V4L2_FMT_FLAG_BYTESPERLINE)
			buffer->bytesperline = fbuf2.fmt.bytesperline;
		else
		{
			buffer->bytesperline = 
				(fbuf2.fmt.width * fbuf2.fmt.depth + 7) & 7;
			buffer->bytesperline >>= 3;
		}
		if (fbuf2.fmt.pixelformat == V4L2_PIX_FMT_RGB555)
			buffer->depth = 15;
		break;
	}
	case VIDIOCSFBUF: /*  set frame buffer  */
	{
		struct video_buffer	*buffer = arg;
		struct v4l2_framebuffer	fbuf2;

		memset(&fbuf2, 0, sizeof(fbuf2));
		fbuf2.base[0] = buffer->base;
		fbuf2.fmt.height = buffer->height;
		fbuf2.fmt.width = buffer->width;
		fbuf2.fmt.depth = buffer->depth;
		switch (fbuf2.fmt.depth)
		{
		case 8:
			fbuf2.fmt.pixelformat = V4L2_PIX_FMT_RGB332;
			break;
		case 15:
			fbuf2.fmt.depth = 16;
			fbuf2.fmt.pixelformat = V4L2_PIX_FMT_RGB555;
			break;
		case 16:
			fbuf2.fmt.pixelformat = V4L2_PIX_FMT_RGB565;
			break;
		case 24:
			fbuf2.fmt.pixelformat = V4L2_PIX_FMT_BGR24;
			break;
		case 32:
			fbuf2.fmt.pixelformat = V4L2_PIX_FMT_BGR32;
			break;
		}
		fbuf2.fmt.flags |= V4L2_FMT_FLAG_BYTESPERLINE;
		fbuf2.fmt.bytesperline = buffer->bytesperline;
		fbuf2.flags = V4L2_FBUF_FLAG_PRIMARY;
		err = vfl->ioctl(context, VIDIOC_S_FBUF, &fbuf2);
		break;
	}
	case VIDIOCGWIN: /*  get window or capture dimensions  */
	{
		struct video_window	*win = arg;
		struct v4l2_window	win2;
		struct v4l2_format	fmt2;

		err = vfl->ioctl(context, VIDIOC_G_WIN, &win2);
		if (err == 0)
		{
			win->x = win2.x;
			win->y = win2.y;
			win->width = win2.width;
			win->height = win2.height;
			win->chromakey = win2.chromakey;
			win->clips = NULL;
			win->clipcount = 0;
			break;
		}
		fmt2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_G_FMT, &fmt2);
		if (err < 0)
			break;
		win->x = 0;
		win->y = 0;
		win->width = fmt2.fmt.pix.width;
		win->height = fmt2.fmt.pix.height;
		win->chromakey = 0;
		win->clips = NULL;
		win->clipcount = 0;
		break;
	}
	case VIDIOCSWIN: /*  set window and/or capture dimensions  */
	{
		struct video_window	*win = arg;
		struct v4l2_window	win2;
		struct v4l2_format	fmt2;

		fmt2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_G_FMT, &fmt2);
		if (err == 0)
		{
			fmt2.fmt.pix.width = win->width;
			fmt2.fmt.pix.height = win->height;
			fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
			err = vfl->ioctl(context, VIDIOC_S_FMT, &fmt2);
			win->width = fmt2.fmt.pix.width;
			win->height = fmt2.fmt.pix.height;
		}
		win2.x = win->x;
		win2.y = win->y;
		win2.width = win->width;
		win2.height = win->height;
		win2.chromakey = win->chromakey;
		win2.clips = (void *)win->clips;
		win2.clipcount = win->clipcount;
		vfl->ioctl(context, VIDIOC_S_WIN, &win2);
		break;
	}
	case VIDIOCCAPTURE: /*  turn on/off preview  */
	{
		err = vfl->ioctl(context, VIDIOC_PREVIEW, arg);
		break;
	}
	case VIDIOCGCHAN: /*  get input information  */
	{
		struct video_channel	*chan = arg;
		struct v4l2_input	input2;
		struct v4l2_standard	std2;
		int			sid;

		input2.index = chan->channel;
		err = vfl->ioctl(context, VIDIOC_ENUMINPUT, &input2);
		if (err < 0)
			break;
		chan->channel = input2.index;
		memcpy(chan->name, input2.name,
		       min(sizeof(chan->name), sizeof(input2.name)));
		chan->name[sizeof(chan->name) - 1] = 0;
		chan->tuners = (input2.type == V4L2_INPUT_TYPE_TUNER) ? 1 : 0;
		chan->flags = (chan->tuners) ? VIDEO_VC_TUNER : 0;
		if (input2.capability & V4L2_INPUT_CAP_AUDIO)
			chan->flags |= VIDEO_VC_AUDIO;
		switch (input2.type)
		{
		case V4L2_INPUT_TYPE_TUNER:
			chan->type = VIDEO_TYPE_TV;
			break;
		default:
		case V4L2_INPUT_TYPE_CAMERA:
			chan->type = VIDEO_TYPE_CAMERA;
			break;
		}
		chan->norm = 0;
		err = vfl->ioctl(context, VIDIOC_G_STD, &std2);
		if (err == 0)
		{
			sid = v4l2_video_std_confirm(&std2);
			switch (sid)
			{
			case V4L2_STD_NTSC:
				chan->norm = VIDEO_MODE_NTSC;
				break;
			case V4L2_STD_PAL:
				chan->norm = VIDEO_MODE_PAL;
				break;
			case V4L2_STD_SECAM:
				chan->norm = VIDEO_MODE_SECAM;
				break;
			}
		}
		break;
	}
	case VIDIOCSCHAN: /*  set input  */
	{
		err = vfl->ioctl(context, VIDIOC_S_INPUT, arg);
		break;
	}
	case VIDIOCGPICT: /*  get tone controls & partial capture format  */
	{
		struct video_picture	*pict = arg;
		struct v4l2_format	fmt2;

		pict->brightness = get_v4l_control(vfl, context, 
						   V4L2_CID_BRIGHTNESS);
		pict->hue = get_v4l_control(vfl, context, 
					    V4L2_CID_HUE);
		pict->contrast = get_v4l_control(vfl, context, 
						 V4L2_CID_CONTRAST);
		pict->colour = get_v4l_control(vfl, context, 
					       V4L2_CID_SATURATION);
		pict->whiteness = get_v4l_control(vfl, context, 
						  V4L2_CID_WHITENESS);
		fmt2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_G_FMT, &fmt2);
		if (err < 0)
			break;
		pict->depth = fmt2.fmt.pix.depth;
		pict->palette = pixelformat_to_palette(
			fmt2.fmt.pix.pixelformat);
		if (pict->palette == VIDEO_PALETTE_RGB555)
			pict->depth = 15;
		break;
	}
	case VIDIOCSPICT: /*  set tone controls & partial capture format  */
	{
		struct video_picture	*pict = arg;
		struct v4l2_format	fmt2;

		set_v4l_control(vfl, context,
				V4L2_CID_BRIGHTNESS, pict->brightness);
		set_v4l_control(vfl, context,
				V4L2_CID_HUE, pict->hue);
		set_v4l_control(vfl, context,
				V4L2_CID_CONTRAST, pict->contrast);
		set_v4l_control(vfl, context,
				V4L2_CID_SATURATION, pict->colour);
		set_v4l_control(vfl, context,
				V4L2_CID_WHITENESS, pict->whiteness);
		err = 0;
		fmt2.type = V4L2_BUF_TYPE_CAPTURE;
		vfl->ioctl(context, VIDIOC_G_FMT, &fmt2);
		if (fmt2.fmt.pix.pixelformat != 
		    palette_to_pixelformat(pict->palette))
		{
			fmt2.fmt.pix.pixelformat = palette_to_pixelformat(
				pict->palette);
			fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
			err = vfl->ioctl(context, VIDIOC_S_FMT, &fmt2);
		}
		break;
	}
	case VIDIOCGTUNER: /*  get tuner information  */
	{
		struct video_tuner	*tun = arg;
		struct v4l2_tuner	tun2;
		int			i;
		int			sid;

		i = find_tuner(vfl, context, tun->tuner);
		if (i < 0)
		{
			err = i;
			break;
		}
		tun2.input = i;
		err = vfl->ioctl(context, VIDIOC_G_TUNER, &tun2);
		if (err < 0)
			break;
		memcpy(tun->name, tun2.name,
		       min(sizeof(tun->name), sizeof(tun2.name)));
		tun->name[sizeof(tun->name) - 1] = 0;
		tun->rangelow = tun2.rangelow;
		tun->rangehigh = tun2.rangehigh;
		tun->flags = 0;
		tun->mode = VIDEO_MODE_AUTO;
		sid = v4l2_video_std_confirm(&tun2.std);
		switch (sid)
		{
		case V4L2_STD_NTSC:
			tun->flags = VIDEO_TUNER_NTSC;
			tun->mode = VIDEO_MODE_NTSC;
			break;
		case V4L2_STD_PAL:
			tun->flags = VIDEO_TUNER_PAL;
			tun->mode = VIDEO_MODE_PAL;
			break;
		case V4L2_STD_SECAM:
			tun->flags = VIDEO_TUNER_SECAM;
			tun->mode = VIDEO_MODE_SECAM;
			break;
		}
		if (tun2.capability & V4L2_TUNER_CAP_LOW)
			tun->flags |= VIDEO_TUNER_LOW;
		if (tun2.rxsubchans & V4L2_TUNER_SUB_STEREO)
			tun->flags |= VIDEO_TUNER_STEREO_ON;
		tun->signal = tun2.signal;
		break;
	}
	case VIDIOCSTUNER: /*  select a tuner input  */
	{
		int	i;

		i = find_tuner(vfl, context, (int)arg);
		if (i < 0)
		{
			err = i;
			break;
		}
		err = vfl->ioctl(context, VIDIOC_S_INPUT, &i);
		break;
	}
	case VIDIOCGFREQ: /*  get frequency  */
	{
		err = vfl->ioctl(context, VIDIOC_G_FREQ, arg);
		break;
	}
	case VIDIOCSFREQ: /*  set frequency  */
	{
		err = vfl->ioctl(context, VIDIOC_S_FREQ, arg);
		break;
	}
	case VIDIOCGAUDIO: /*  get audio properties/controls  */
	{
		struct video_audio	*aud = arg;
		struct v4l2_audio	aud2;
		struct v4l2_queryctrl	qctrl2;
		struct v4l2_tuner	tun2;
		int			v;

		err = vfl->ioctl(context, VIDIOC_G_AUDIO, &aud2);
		if (err < 0)
			break;
		memcpy(aud->name, aud2.name,
		       min(sizeof(aud->name), sizeof(aud2.name)));
		aud->name[sizeof(aud->name) - 1] = 0;
		aud->audio = aud2.audio;
		aud->flags = 0;
		v = get_v4l_control(vfl, context, V4L2_CID_AUDIO_VOLUME);
		if (v >= 0)
		{
			aud->volume = v;
			aud->flags |= VIDEO_AUDIO_VOLUME;
		}
		v = get_v4l_control(vfl, context, V4L2_CID_AUDIO_BASS);
		if (v >= 0)
		{
			aud->bass = v;
			aud->flags |= VIDEO_AUDIO_BASS;
		}
		v = get_v4l_control(vfl, context, V4L2_CID_AUDIO_TREBLE);
		if (v >= 0)
		{
			aud->treble = v;
			aud->flags |= VIDEO_AUDIO_TREBLE;
		}
		v = get_v4l_control(vfl, context, V4L2_CID_AUDIO_BALANCE);
		if (v >= 0)
		{
			aud->balance = v;
			aud->flags |= VIDEO_AUDIO_BALANCE;
		}
		v = get_v4l_control(vfl, context, V4L2_CID_AUDIO_MUTE);
		if (v >= 0)
		{
			if (v)
				aud->flags |= VIDEO_AUDIO_MUTE;
			aud->flags |= VIDEO_AUDIO_MUTABLE;
		}
		aud->step = 1;
		qctrl2.id = V4L2_CID_AUDIO_VOLUME;
		if (vfl->ioctl(context, VIDIOC_QUERYCTRL, &qctrl2) == 0 &&
		    !(qctrl2.flags & V4L2_CTRL_FLAG_DISABLED))
			aud->step = qctrl2.step;
		aud->mode = 0;
		err = vfl->ioctl(context, VIDIOC_G_TUNER, &tun2);
		if (err < 0)
		{
			err = 0;
			break;
		}
		if (tun2.rxsubchans & V4L2_TUNER_SUB_LANG2)
			aud->mode = VIDEO_SOUND_LANG1 | VIDEO_SOUND_LANG2;
		else if (tun2.rxsubchans & V4L2_TUNER_SUB_STEREO)
			aud->mode = VIDEO_SOUND_STEREO;
		else if (tun2.rxsubchans & V4L2_TUNER_SUB_MONO)
			aud->mode = VIDEO_SOUND_MONO;
		break;
	}
	case VIDIOCSAUDIO: /*  set audio controls  */
	{
		struct video_audio	*aud = arg;
		struct v4l2_audio	aud2;
		struct v4l2_tuner	tun2;
		int			i;

		aud2.audio = aud->audio;
		err = vfl->ioctl(context, VIDIOC_S_AUDIO, &aud2);
		if (err < 0)
			break;

		set_v4l_control(vfl, context, V4L2_CID_AUDIO_VOLUME, 
				aud->volume);
		set_v4l_control(vfl, context, V4L2_CID_AUDIO_BASS,
				aud->bass);
		set_v4l_control(vfl, context, V4L2_CID_AUDIO_TREBLE,
				aud->treble);
		set_v4l_control(vfl, context, V4L2_CID_AUDIO_BALANCE,
				aud->balance);
		set_v4l_control(vfl, context, V4L2_CID_AUDIO_MUTE,
				!!(aud->flags & VIDEO_AUDIO_MUTE));

		err = vfl->ioctl(context, VIDIOC_G_INPUT, &i);
		if (err < 0)
		{
			err = 0;
			break;
		}
		tun2.input = i;
		err = vfl->ioctl(context, VIDIOC_G_TUNER, &tun2);
		if (err == 0)
		{
			switch (aud->mode)
			{
			default:
			case VIDEO_SOUND_MONO:
			case VIDEO_SOUND_LANG1:
				tun2.audmode = V4L2_TUNER_MODE_MONO;
				break;
			case VIDEO_SOUND_STEREO:
				tun2.audmode = V4L2_TUNER_MODE_STEREO;
				break;
			case VIDEO_SOUND_LANG2:
				tun2.audmode = V4L2_TUNER_MODE_LANG2;
				break;
			}
			vfl->ioctl(context, VIDIOC_S_TUNER, &tun2);
		}
		err = 0;
		break;
	}
	case VIDIOCGMBUF: /*  get mmap parameters  */
	{
		struct video_mbuf		*mbuf = arg;
		struct v4l2_requestbuffers	reqbuf2;
		struct v4l2_buffer		buf2;
		struct v4l2_format		fmt2, fmt2o;
		struct v4l2_capability		cap2;
		int				i;

		/*  Set the format to maximum dimensions  */
		if ((err = vfl->ioctl(context, VIDIOC_QUERYCAP, &cap2)) < 0)
			break;
		fmt2o.type = V4L2_BUF_TYPE_CAPTURE;
		if ((err = vfl->ioctl(context, VIDIOC_G_FMT, &fmt2o)) < 0)
			break;
		fmt2 = fmt2o;
		fmt2.fmt.pix.width = cap2.maxwidth;
		fmt2.fmt.pix.height = cap2.maxheight;
		fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
		if ((err = vfl->ioctl(context, VIDIOC_S_FMT, &fmt2)) < 0)
			break;
		reqbuf2.count = 2; /* v4l always used two buffers */
		reqbuf2.type = V4L2_BUF_TYPE_CAPTURE | V4L2_BUF_REQ_CONTIG;
		err = vfl->ioctl(context, VIDIOC_REQBUFS, &reqbuf2);
		if (err < 0 || reqbuf2.count < 2 || reqbuf2.type
		    != (V4L2_BUF_TYPE_CAPTURE | V4L2_BUF_REQ_CONTIG))
		{/*	Driver doesn't support v4l back-compatibility  */
			fmt2o.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
			vfl->ioctl(context, VIDIOC_S_FMT, &fmt2o);
			reqbuf2.count = 1;
			reqbuf2.type = V4L2_BUF_TYPE_CAPTURE;
			err = vfl->ioctl(context, VIDIOC_REQBUFS, &reqbuf2);
			if (err < 0)
			{
				err = -EINVAL;
				break;
			}
			printk(KERN_INFO"V4L2: Device \"%s\" doesn't support"
			       " v4l memory mapping\n", vfl->name);
		}
		buf2.index = 0;
		buf2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
		mbuf->size = buf2.length * reqbuf2.count;
		mbuf->frames = reqbuf2.count;
		memset(mbuf->offsets, 0, sizeof(mbuf->offsets));
		for (i = 0; i < mbuf->frames; ++i)
			mbuf->offsets[i] = i * buf2.length;
		break;
	}
	case VIDIOCMCAPTURE: /*  capture a frame  */
	{
		struct video_mmap	*mm = arg;
		struct v4l2_buffer	buf2;
		struct v4l2_format	fmt2;

		fmt2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_G_FMT, &fmt2);
		if (err < 0)
			break;
		if (mm->width != fmt2.fmt.pix.width || 
		    mm->height != fmt2.fmt.pix.height ||
		    palette_to_pixelformat(mm->format) != 
		    fmt2.fmt.pix.pixelformat)
		{/*	New capture format...  */
			fmt2.fmt.pix.width = mm->width;
			fmt2.fmt.pix.height = mm->height;
			fmt2.fmt.pix.pixelformat =
				palette_to_pixelformat(mm->format);
			fmt2.fmt.pix.flags |= V4L2_FMT_FLAG_INTERLACED;
			err = vfl->ioctl(context, VIDIOC_S_FMT, &fmt2);
			if (err < 0)
				break;
		}
		buf2.index = mm->frame;
		buf2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
		if (err < 0)
			break;
		err = vfl->ioctl(context, VIDIOC_QBUF, &buf2);
		if (err < 0)
			break;
		vfl->ioctl(context, VIDIOC_STREAMON, &buf2.type);
		break;
	}
	case VIDIOCSYNC: /*  wait for a frame  */
	{
		int			*i = arg;
		struct v4l2_buffer	buf2;

		buf2.index = *i;
		buf2.type = V4L2_BUF_TYPE_CAPTURE;
		err = vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
		if (err < 0)	/*  No such buffer */
			break;
		if (!(buf2.flags & V4L2_BUF_FLAG_MAPPED))
		{/*	Buffer is not mapped  */
			err = -EINVAL;
			break;
		}

		/*  Loop as long as the buffer is queued, but not done  */
		while ((buf2.flags &
			(V4L2_BUF_FLAG_QUEUED | V4L2_BUF_FLAG_DONE))
		       == V4L2_BUF_FLAG_QUEUED)
		{
			err = simple_select(file);
			if (err < 0 ||	/* error or sleep was interrupted  */
			    err == 0)	/* timeout? Shouldn't occur.  */
				break;
			vfl->ioctl(context, VIDIOC_QUERYBUF, &buf2);
		}
		if (!(buf2.flags & V4L2_BUF_FLAG_DONE)) /* not done */
			break;
		do
		{
			err = vfl->ioctl(context, VIDIOC_DQBUF, &buf2);
		}	while (err == 0 && buf2.index != *i);
		break;
	}
	case VIDIOCGUNIT: /*  get related device minors  */
		/*  No translation  */
		break;
	case VIDIOCGCAPTURE: /*    */
		/*  No translation, yet...  */
		printk(KERN_INFO"V4L2: VIDIOCGCAPTURE not implemented. "
		       "Send patches to bdirks@pacbell.net :-)\n");
		break;
	case VIDIOCSCAPTURE: /*    */
		/*  No translation, yet...  */
		printk(KERN_INFO"V4L2: VIDIOCSCAPTURE not implemented. "
		       "Send patches to bdirks@pacbell.net :-)\n");
		break;
	}
	return err;
}

/*
 *	Module init and cleanup
 */

struct v4l2_v4l_compat v4l_compat =
{
	translate_ioctl:	v4l_compat_translate_ioctl,
	fix_offset:		v4l_compat_fix_offset,
};


#ifdef MODULE
int init_module(void)
{
	return v4l2_v4l_compat_register(&v4l_compat);
}

void cleanup_module(void)
{
	v4l2_v4l_compat_unregister(&v4l_compat);
}
#endif
