/*
 * cpu_energy-aware.c: code to support energy-aware voltage-scaling 
 * algorithm on processor
 *
 * Real-time and Multimedia Systems Laboratory
 * Copyright (c) 2000 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Real-Time and Multimedia Systems Laboratory
 *  Attn: Prof. Raj Rajkumar
 *  Electrical and Computer Engineering, and Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 *  or via email to raj@ece.cmu.edu
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */

///!!!!!! NOTE:::: IPAQ coding is not finished yet !!!

#ifdef ENERGY_AWARE

#define MAX_ENERGY_BUFFER 50

#include <rk/rk_linux.h>
#include <rk/rk_error.h>
#include <rk/rk.h>
#include <rk/timespec.h>
#include <linux/time.h>
#include <asm/processor.h>
#include <asm/uaccess.h>
#include <asm/signal.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/types.h>
#include <asm/hardware.h>
#ifdef CONFIG_MEASURE
#include <linux/measure.h>
#endif				/* CONFIG_MEASURE */
#if defined(__brh__)
#include "VSCALE_BRH.h"
#endif

/* For XScale Energy Monitor */
static int history_time[MAX_ENERGY_BUFFER];
static int history_freq[MAX_ENERGY_BUFFER];
static int history_volt[MAX_ENERGY_BUFFER];
static int history_power[MAX_ENERGY_BUFFER];
int history_index; 
int history_round;



extern int cpu_reserves_scheduling_policy;
extern struct list_head cpu_reserve_head;

#define NODVS                           0
#define SYS_CLOCK                       1
#define PM_CLOCK                        2
#define DPM_CLOCK1                      3
#define DPM_CLOCK2                      4
#define NUM_ENERGY_AWARE_POLICIES       (DPM_CLOCK2 + 1)

struct cpu_energy_hw_action {
  void (*cpu_energy_hw_init)(void);
  void (*cpu_energy_hw_update_freq)(int freqIndex);
};


/* The available operating voltage/frequency points */
#if defined (__i386__)
#elif defined (__ipaq__)
#define NUM_ENERGY_OPS 10

static int frequency[NUM_ENERGY_OPS] = {40,   46,   53,   59,   66,
				 73,   79,   87,   93,  100}; /* f respected to f_max */
static int abs_frequency[NUM_ENERGY_OPS] = { 885, 1032, 1180, 1327, 1475, 
				     1622, 1769, 1917, 2064, 2212 }; /* MHz/10 */

#elif defined (__brh__)
#define SEVENOP

#ifdef SEVENOP
#define NUM_ENERGY_OPS 7 /* The number of voltage/frequency operating points */
static int frequency[NUM_ENERGY_OPS]  =        {45,   54,   63,   72,   81,   90,  100}; /* f respected to f_max */
static int abs_frequency[NUM_ENERGY_OPS] = { 3330, 4000, 4660, 5330, 6000, 6660, 7330}; /* MHz/10 */
static int voltage[NUM_ENERGY_OPS] =       {1000, 1100, 1150, 1350, 1400, 1450, 1500}; /* mV */
static  int power[NUM_ENERGY_OPS] = {120, 160, 210, 255, 345, 430, 500}; /* mW - based on http://www.ee.ucla.edu/~shalabh/pads/power.html */
static u32 freq_set[NUM_ENERGY_OPS]  =          {3,    4,    5,    6,    7,    8,    9}; /* CCLKCFG */
static u32 volt_set[NUM_ENERGY_OPS]  = {BRH_VOLT_100V, BRH_VOLT_110V, BRH_VOLT_115V, BRH_VOLT_135V, BRH_VOLT_140V, BRH_VOLT_145V, BRH_VOLT_150V};  
#endif
#ifdef FOUROP
#define NUM_ENERGY_OPS 4
static int frequency[NUM_ENERGY_OPS]  =        {45,   54,  81,  100};
//static int abs_frequency[NUM_ENERGY_OPS] = {3330, 4000,6000, 7330}; /* MHz/10 */
//static int voltage[NUM_ENERGY_OPS] =       {1000, 1100,1300, 1500}; /* mV */
static u32 freq_set[NUM_ENERGY_OPS]  =          {3,    4,   7,    9}; /* CCLKCFG */
static u32 volt_set[NUM_ENERGY_OPS]  = {BRH_VOLT_100V, BRH_VOLT_110V, BRH_VOLT_130V, BRH_VOLT_150V};  
#endif

static volatile u32 freq_set_value;
static volatile u32 current_freq_set_value;




#endif

static int freqIndex; // The index of current system frequency
static int SYSfreqIndex; // The index of system clock frequency assigned by SYS_CLOCK algorithm
static struct cpu_energy_hw_action *energy_action;
static int cpu_energy_aware_policy = NODVS;



extern int efficient_timespec_ceil(struct timespec dividend,
				   struct timespec divider);
extern int prio_lt(cpu_reserve_attr_t r_attr, cpu_reserve_attr_t curr_attr);
extern int prio_le(cpu_reserve_attr_t r_attr, cpu_reserve_attr_t curr_attr);

static int efficient_strict_timespec_ceil(struct timespec dividend,
					  struct timespec divider);
static int cpu_energy_choose_index(int freq);
static struct timespec timespec_mult(struct timespec multiplicand,
				     int multiplier);

static struct timespec timespec_div(struct timespec dividend, 
				    int divider);
static struct timespec div(unsigned long dividend, unsigned long divider);
static struct timespec efficient_timespec_div(struct timespec dividend, 
					      int divider);

#define cpu_reserve_entry(list)	list_entry((list), struct cpu_reserve, cpu_link)


/**** HARDWARE DEPENDENT CODE ****/

#if defined(__brh__)
void brh_energy_hw_update_freq(int freqIndex) {
  static int old_index = 0;
  static int changeVolt;
  static u32 volt_set_value;

  if (freqIndex == old_index) return;

  if ((freqIndex > NUM_ENERGY_OPS-1) || (old_index > NUM_ENERGY_OPS-1))  {
    printk("Error!!! \n");
    return;
  }
  if ( volt_set_value != volt_set[freqIndex]) {
    changeVolt = 1;
    volt_set_value = volt_set[freqIndex];
  }
  if (freqIndex > old_index) {
    /* set voltage first to avoid overclock scenario*/
    if (changeVolt) {
      writeb(volt_set_value, BRH_VOLT_CONTROL);
    }
    freq_set_value = freq_set[freqIndex];
    current_freq_set_value = -1;
    // set the frequency based on freq_set_value
    asm("ldr  r3, =freq_set_value \n"
	"ldr  r3, [r3, #0] \n"
	"MCR P14, 0, r3, C6, C0, 0\n"
	"MRC P14, 0, r3, C6, C0, 0\n"
	"ldr r4, =current_freq_set_value \n"
	"str r3, [r4]\n"
	);    
    // just checking if the hw is really updated 
    if (current_freq_set_value != freq_set_value) {
      printk("Set new frequency error!!: current_freq_set_value = %x , freq_set_value = %x \n", 
	   current_freq_set_value, 
	   freq_set_value);
    }
  }
  else {
    /* scale down the freq first */
    freq_set_value = freq_set[freqIndex];
    current_freq_set_value = -1;
    // set the frequency based on freq_set_value
    asm("ldr  r3, =freq_set_value \n"
	"ldr  r3, [r3, #0] \n"
	"MCR P14, 0, r3, C6, C0, 0\n"
	"MRC P14, 0, r3, C6, C0, 0\n"
	"ldr r4, =current_freq_set_value \n"
	"str r3, [r4]\n"
	);    
    // just checking if the hw is really updated 
    if (current_freq_set_value != freq_set_value) {
      printk("Set new frequency error!!: current_freq_set_value = %x , freq_set_value = %x \n", 
	     current_freq_set_value, 
	     freq_set_value);
    }

    if (changeVolt) {
      writeb(volt_set_value, BRH_VOLT_CONTROL);
    }

  }

  old_index = freqIndex;

}

void brh_energy_hw_init(void) {
  cpu_energy_aware_policy = PM_CLOCK;
  /* to enable the voltage-scaling circuit , Do not change the bottom line */
  writeb(BRH_VOLT_150V, BRH_VOLT_CONTROL);
  /**** You can update from here *******/


  /* set to minimum  voltage and  freq */
  freqIndex = 0;
  SYSfreqIndex = 0;
  writeb(volt_set[freqIndex], BRH_VOLT_CONTROL);
  brh_energy_hw_update_freq(freqIndex);
}


static struct cpu_energy_hw_action brh_energy_action = {
  cpu_energy_hw_init:        brh_energy_hw_init,
  cpu_energy_hw_update_freq: brh_energy_hw_update_freq
};

#endif /* __brh__ */

#if defined (__ipaq__)
void ipaq_energy_hw_init(void) {
  cpu_energy_aware_policy = SYS_CLOCK;
  /* set to minimum  voltage and  freq */
  freqIndex = 0; 
  brh_energy_hw_update_freq(freqIndex); 
}

void ipaq_energy_hw_update_freq(int freqIndex) {
  
}


static struct cpu_energy_hw_action ipaq_energy_action = {
  cpu_energy_hw_init:        ipaq_energy_hw_init,
  cpu_energy_hw_update_freq: ipaq_energy_hw_update_freq
};

#endif /* __ipaq__ */

/********HARDWARE INDEPENDENT CODE ******/

void rk_cpu_energy_init(void) {
  int i;
  for (i = 0; i < MAX_ENERGY_BUFFER; i++) {
    history_time[i] = 0;
    history_freq[i] = 0;
    history_volt[i] = 0;
    history_power[i] = 0;
  }  
  history_index = 0;
  history_round = 0;

#if defined (__brh__) 
  energy_action = &brh_energy_action;
#elif defined (__ipaq__) 
  energy_action = &ipaq_energy_action;
#else 
  return;
#endif  

  energy_action->cpu_energy_hw_init();
}

/* cpu_energy_sys_clock_freq:
 * This function is to computer the minimum frequency needed by each task 
 * assuming that the higher priority tasks and itself use the same frequency.
 */
static int cpu_energy_sys_clock_freq(cpu_reserve_t cpu) {
  struct timespec deadline = cpu->cpu_res_attr.deadline;
  struct timespec completion_time, new_completion_time, result;
  struct timespec slack, idle_duration, load;
  struct list_head *p;
  cpu_reserve_t r;
  cpu_reserve_attr_t curr_attr = &(cpu->cpu_res_attr);
  int ceil_value;
  int IN_BZP = 1;
  int f, minf = 100; /* frequency % respected to f_max */
  struct timespec zero_time;

  timespec_zero(zero_time);
  timespec_set(completion_time, curr_attr->compute_time);
  timespec_add(completion_time, curr_attr->blocking_time);
  timespec_zero(slack);
  while (timespec_lt(completion_time, deadline) ){
    if (IN_BZP) {
      struct timespec DELTA_i;
      
      timespec_set(DELTA_i, deadline);
      timespec_sub(DELTA_i, completion_time);

      while (timespec_lt(completion_time, deadline) && 
	     timespec_gt(DELTA_i, zero_time)) {
	timespec_zero(new_completion_time);

	for (p = cpu_reserve_head.next;
	     p != &cpu_reserve_head; p = p->next) {
	  cpu_reserve_attr_t r_attr;
	  r = cpu_reserve_entry(p);
	  /* We compute preemption time from the HIGHER priority reserves and
	   * any EQUAL priority reserves.  This is necessary because the
	   * deadlines can be different for reserves with the same period.
	   */
	  if (r == cpu) {
	    /* self */
	    continue;
	  }
	  
	  r_attr = &(r->cpu_res_attr);
	  
	  if (prio_lt(r_attr, curr_attr)) {
	    break;
	  }
	  
	  ceil_value = efficient_strict_timespec_ceil(completion_time,
						      r_attr->period);

	  result = timespec_mult(r_attr->compute_time, ceil_value);
	  timespec_add(new_completion_time, result);
	}
	
	/* add computation time and blocking time of self */
	timespec_add(new_completion_time, slack);
	timespec_add(new_completion_time, curr_attr->compute_time);
	timespec_add(new_completion_time, curr_attr->blocking_time); 
	timespec_set(DELTA_i, new_completion_time);
	timespec_sub(DELTA_i, completion_time);
	timespec_set(completion_time, new_completion_time);

      }


      IN_BZP = 0;

    } else {
      

      timespec_set(idle_duration, deadline);
      timespec_sub(idle_duration, completion_time);
      for (p = cpu_reserve_head.next;
	   p != &cpu_reserve_head; p = p->next) {
	cpu_reserve_attr_t r_attr;
	
	r = cpu_reserve_entry(p);
	
	/* We compute preemption time from the HIGHER priority reserves and
	   * any EQUAL priority reserves.  This is necessary because the
	   * deadlines can be different for reserves with the same period.
	   */
	if (r == cpu) {
	  /* self */
	  continue;
	}
	  
	r_attr = &(r->cpu_res_attr);
	  
	if (prio_lt(r_attr, curr_attr)) {
	  break;
	}
	
	ceil_value = efficient_timespec_ceil(completion_time, 
					     r_attr->period);

	
	result = timespec_mult(r_attr->period, ceil_value);
	timespec_sub(result, completion_time);
	if (timespec_lt(result, idle_duration)) {
	  timespec_set(idle_duration, result);
	}
      }

      timespec_add(completion_time, idle_duration);
      timespec_add(slack, idle_duration);
      
      timespec_set(load, completion_time);
      timespec_sub(load, slack);
      load = timespec_mult(load, 100);
      
      f =  efficient_timespec_ceil(load, completion_time);

      if (f < minf) { minf = f; }
      IN_BZP = 1;
    }
    
  }

  return minf;
}

/* cpu_energy_pm_clock_adjust_freq: Recalculate the minimum frequency needed by 
 * each task with inflated frequency effect.
 */
void cpu_energy_pm_clock_adjust_freq(void) {
  struct timespec deadline;
  struct timespec completion_time, new_completion_time, result;
  struct timespec slack, idle_duration, load, fixed_load, scalable_duration;
  struct list_head *p, *pp;
  cpu_reserve_t r;
  cpu_reserve_attr_t curr_attr;
  int ceil_value;
  int IN_BZP = 1;
  int f, minf = 100; /* frequency % respected to f_max */
  struct timespec zero_time;

  for (pp = cpu_reserve_head.next;
       pp != &cpu_reserve_head; pp = pp->next) {
  
    cpu_reserve_t cpu = cpu_reserve_entry(pp);
    if (cpu->adjustFreq) {
      continue;
    }
    deadline = cpu->cpu_res_attr.deadline;
    curr_attr = &(cpu->cpu_res_attr);
    timespec_zero(zero_time);
    timespec_set(completion_time, curr_attr->compute_time);
    timespec_add(completion_time, curr_attr->blocking_time);
    timespec_zero(slack);
    while (timespec_lt(completion_time, deadline) ){
      if (IN_BZP) {
	struct timespec DELTA_i;
      
	timespec_set(DELTA_i, deadline);
	timespec_sub(DELTA_i, completion_time);

	while (timespec_lt(completion_time, deadline) && 
	       timespec_gt(DELTA_i, zero_time)) {
	  timespec_zero(new_completion_time);
	  timespec_zero(load);
	  timespec_zero(fixed_load);
	  for (p = cpu_reserve_head.next;
	       p != &cpu_reserve_head; p = p->next) {
	    cpu_reserve_attr_t r_attr;
	    r = cpu_reserve_entry(p);
	    /* We compute preemption time from the HIGHER priority reserves and
	     * any EQUAL priority reserves.  This is necessary because the
	     * deadlines can be different for reserves with the same period.
	     */
	    if (r == cpu) {
	      /* self */
	      continue;
	    }
	  
	    r_attr = &(r->cpu_res_attr);
	  
	    if (prio_lt(r_attr, curr_attr)) 
	      break;

	    
	    ceil_value = efficient_strict_timespec_ceil(completion_time,
							r_attr->period);
	    result = timespec_mult(r_attr->compute_time, ceil_value);
	    
	    if (r->adjustFreq) {
	      result = timespec_mult(result, 100);
	      result = efficient_timespec_div(result, frequency[r->PMfreqIndex]);
	      timespec_add(fixed_load, result);

	    } else {
	      timespec_add(load, result);
	    }
	  }
	
	  /* add computation time and blocking time of self */
	  timespec_add(load, curr_attr->compute_time);
	  timespec_add(load, curr_attr->blocking_time);
	  timespec_set(new_completion_time, load);
	  timespec_add(new_completion_time, fixed_load);
	  timespec_add(new_completion_time, slack);
	  timespec_set(DELTA_i, new_completion_time);
	  timespec_sub(DELTA_i, completion_time);
	  timespec_set(completion_time, new_completion_time);
	}

	IN_BZP = 0;

      } else {
      
	timespec_set(idle_duration, deadline);
	timespec_sub(idle_duration, completion_time);
	for (p = cpu_reserve_head.next;
	     p != &cpu_reserve_head; p = p->next) {
	  cpu_reserve_attr_t r_attr;
	
	  r = cpu_reserve_entry(p);
	
	  /* We compute preemption time from the HIGHER priority reserves and
	   * any EQUAL priority reserves.  This is necessary because the
	   * deadlines can be different for reserves with the same period.
	   */
	  if (r == cpu) {
	    /* self */
	    continue;
	  }
	  
	  r_attr = &(r->cpu_res_attr);
	  
	  if (prio_lt(r_attr, curr_attr)) 
	    break;
	
	  ceil_value = efficient_timespec_ceil(completion_time, 
					       r_attr->period);
	
	  result = timespec_mult(r_attr->period, ceil_value);
	  timespec_sub(result, completion_time);
	  if (timespec_lt(result, idle_duration)) {
	    timespec_set(idle_duration, result);
	  }
	}

	timespec_add(completion_time, idle_duration);
	timespec_add(slack, idle_duration);
	timespec_set(scalable_duration, completion_time);
	timespec_sub(scalable_duration, fixed_load);
	load = timespec_mult(load, 100);      
	f =  efficient_timespec_ceil(load, scalable_duration);
	if (f < minf) { minf = f; }
	IN_BZP = 1;
      }

      cpu->tmpF = minf;
    
    }
   
  }
}





/* cpu_energy_aware_determine_freq: Calculate the frequency needed.
 * This is based on the chosen energy policy  
 */
void cpu_energy_aware_determine_freq(cpu_reserve_t cpu) {
  struct list_head *p, *p2;
  cpu_reserve_t r, r2;
  cpu_reserve_attr_t curr_attr = &(cpu->cpu_res_attr);

  switch(cpu_energy_aware_policy) {
  case NODVS:
    /* do nothing */
    return;
  case SYS_CLOCK:
    {
      int sysFreq = 0;

      for (p = cpu_reserve_head.next;
	   p != &cpu_reserve_head; p = p->next) {

	cpu_reserve_attr_t r_attr;
	r = cpu_reserve_entry(p);
	r_attr = &(r->cpu_res_attr);	  
	if (prio_le(r_attr, curr_attr)) {
	  r->ownFreq = cpu_energy_sys_clock_freq(r);
	}
	r->tmpF = r->ownFreq;
	r->adjustFreq = 0; 
      }

      // Find the minimum speed that satisfies all reserves
      for (p = cpu_reserve_head.next;
	   p != &cpu_reserve_head; p = p->next) {

	r = cpu_reserve_entry(p);
	if (r->tmpF > sysFreq) {
	  sysFreq = r->tmpF;
	}
      }

#ifdef DEBUG_BRH
      printk("System clock frequency = %d \n", sysFreq);
#endif /* DEBUG_BRH */
      // Find the suitable operating point
      SYSfreqIndex = cpu_energy_choose_index(sysFreq);
#ifdef DEBUG_BRH
      printk("System freq Index = %d freq = %d \n", freqIndex, frequency[freqIndex]);
#endif /* DEBUG_BRH */
    }
    return;
	    
  case PM_CLOCK:      
    {
      int recompute;
      for (p = cpu_reserve_head.next;
	   p != &cpu_reserve_head; p = p->next) {
	
	cpu_reserve_attr_t r_attr;
	r = cpu_reserve_entry(p);
	r_attr = &(r->cpu_res_attr);	  
	if (prio_le(r_attr, curr_attr)) {
	  r->ownFreq = cpu_energy_sys_clock_freq(r);
	}
	r->tmpF = r->ownFreq;
#ifdef DEBUG_BRH
	printk("reserve (%lu.%lu,%lu.%lu) own_f = %d \n", r_attr->compute_time.tv_sec, r_attr->compute_time.tv_nsec, r_attr->period.tv_sec , r_attr->period.tv_nsec, r->tmpF);
#endif /* DEBUG_BRH */	
	r->adjustFreq = 0; 
      }
      
      // Find the minimum speed of "each" task that maintains 
      // the schedulability of the task set
      
      for (p = cpu_reserve_head.next;
	   p != &cpu_reserve_head; p = p->next) {
	
	r = cpu_reserve_entry(p);
	int maxAlpha = 0;
	int tmpIndex;
	for (p2 = cpu_reserve_head.next;
	     p2 != &cpu_reserve_head; p2 = p2->next) {
	  r2 = cpu_reserve_entry(p2);
	  if ( prio_le(&(r2->cpu_res_attr), &(r->cpu_res_attr))  && 
	       (r2->tmpF > maxAlpha) ) {
	    maxAlpha = r2->tmpF;
	  }
	}

	tmpIndex = cpu_energy_choose_index(maxAlpha);
	if ( (p != cpu_reserve_head.next) && 
	     (cpu_reserve_entry(p->prev)->PMfreqIndex > tmpIndex) ){
	  recompute = 1;
	}
	else if ( (p != cpu_reserve_head.next) &&
		  (p->next == &cpu_reserve_head) && 
		  (frequency[cpu_reserve_entry(p->prev)->PMfreqIndex] > cpu_reserve_entry(p)->tmpF) ) {
	  // this is the last guy
	  recompute = 1;
	}
	else recompute = 0;


	if (recompute) {
#ifdef DEBUG_BRH
	  printk("Adjust speed \n");
#endif /* DEBUG_BRH */	  
	  //adjust speed
	  cpu_energy_pm_clock_adjust_freq();
	  maxAlpha = 0;
	  for (p2 = cpu_reserve_head.next;
	       p2 != &cpu_reserve_head; p2 = p2->next) {
	    r2 = cpu_reserve_entry(p2);
	    if ( prio_le(&(r2->cpu_res_attr), &(r->cpu_res_attr))  && 
		 (r2->tmpF > maxAlpha) ) {
	      maxAlpha = r2->tmpF;
	    }
	  }
	}

	r->PMfreqIndex = cpu_energy_choose_index(maxAlpha);
	r->adjustFreq = 1;

       }

    }

#ifdef DEBUG_BRH
    printk("Final result: \n");
    for (p = cpu_reserve_head.next;
	 p != &cpu_reserve_head; p = p->next) {
      cpu_reserve_t cpu = cpu_reserve_entry(p);
      cpu_reserve_attr_t cpu_attr;
      cpu_attr = &(cpu->cpu_res_attr);	  
      printk("PM freq (%lu.%lu, %lu.%lu) = %d \n", cpu_attr->compute_time.tv_sec, cpu_attr->compute_time.tv_nsec, cpu_attr->period.tv_sec, cpu_attr->period.tv_nsec, frequency[cpu->PMfreqIndex]);
    }
#endif /* DEBUG_BRH */
    return;
  case DPM_CLOCK1:
    /* do nothing for now */
    return;
  case DPM_CLOCK2:
    /* do nothing for now */
    return;
  }
}

cpu_tick_data_t cpu_energy_set_new_freq(rk_reserve_t rsv, 
					signed_cpu_tick_data_t next) {
  unsigned long adjusted_ticks = (unsigned long) next;
  cpu_tick_data_t result;
  struct timespec now;
  int time_ms;

  rk_cur_time(&now);

  time_ms = now.tv_sec * 1000 + now.tv_nsec/1000000;

  if (rsv == NULL || !rk_valid_rsv(rsv)) {
    freqIndex = 0;
    adjusted_ticks = next;
  }
  else {
    switch(cpu_energy_aware_policy) {
    case NODVS:
      freqIndex = NUM_ENERGY_OPS -1;
      adjusted_ticks = next;
      break;
    case SYS_CLOCK:
      freqIndex = SYSfreqIndex;
      adjusted_ticks =  (adjusted_ticks /frequency[freqIndex]) * 100;
      break;
    case PM_CLOCK:
      freqIndex = ( (cpu_reserve_t)rsv->rsv_rsv )->PMfreqIndex;
      adjusted_ticks =  (adjusted_ticks / frequency[freqIndex]) * 100;
      break;
    case DPM_CLOCK1:
      /* compute new frequency here */
      break;
    case DPM_CLOCK2:
      /* compute new frequency here */
      break;
    }
  }


  /* Set voltage and frequency */
  energy_action->cpu_energy_hw_update_freq(freqIndex);
  
  history_time[history_index] = time_ms;
  history_freq[history_index] = abs_frequency[freqIndex];
  history_volt[history_index] = voltage[freqIndex];
  history_power[history_index] = power[freqIndex];
  history_index++;
  if (history_index == MAX_ENERGY_BUFFER) {
    history_index = 0;
    history_round++;
  }


  result = adjusted_ticks;

  return result;

}

cpu_tick_data_t cpu_energy_used_ticks(cpu_reserve_t cpu, 
				      cpu_tick_data_t now) {
  cpu_tick_data_t used_ticks = (now-cpu->cpu_period_start_ticks);
  unsigned long adjusted_ticks = (unsigned long) used_ticks;

  switch(cpu_energy_aware_policy) {
  case NODVS:
    return used_ticks;
  case SYS_CLOCK:
    adjusted_ticks = (adjusted_ticks  * frequency[freqIndex])/ 100;
    used_ticks = (cpu_tick_data_t) adjusted_ticks;
    return used_ticks;
  case PM_CLOCK:
  case DPM_CLOCK1:
  case DPM_CLOCK2:
    adjusted_ticks = (adjusted_ticks * frequency[freqIndex])/ 100;
    used_ticks = (cpu_tick_data_t) adjusted_ticks;
    return used_ticks;
  }
  return used_ticks;
}



/* cpu_energy_choose_freq_index
 * This function is to find the closest available operating frequency that
 * can satisfy all tasks' timing constraints.
 */
static int cpu_energy_choose_index(int freq) {
  int i;

  for (i = 0; i < NUM_ENERGY_OPS; i++) {
    if (frequency[i] >= freq) return i;
  }
  return NUM_ENERGY_OPS-1;
}
	 



/* For system calls */


int rk_cpu_energy_aware_set_policy(int policy)
{
#ifdef ENERGY_AWARE
  if ((policy >= 0) && (policy < NUM_ENERGY_AWARE_POLICIES)) {
    cpu_energy_aware_policy = policy;
    return RK_SUCCESS;
  }
#endif /* ENERGY_AWARE */
  return -1;
}

int rk_cpu_energy_aware_get_policy(void) 
{
#ifdef ENERGY_AWARE
  return cpu_energy_aware_policy;
#endif /* ENERGY_AWARE */
  return -1;
}

asmlinkage int sys_rk_cpu_energy_aware_set_policy(int policy) 
{
#ifdef DEBUG_RK
  printk("cpu_energy_aware_set_policy: pol(0x%x)\n", (int) policy);
#endif
  return rk_cpu_energy_aware_set_policy(policy);
}

asmlinkage int sys_rk_cpu_energy_aware_get_policy(void) 
{
#ifdef DEBUG_RK
  printk("cpu_energy_aware_get_policy\n");
#endif
  return rk_cpu_energy_aware_get_policy();
}


/* For local use */
static struct timespec timespec_div(struct timespec dividend, 
					   int divider) {
  struct timespec divider1;
  struct timespec result;

  divider1.tv_sec = 0;
  divider1.tv_nsec = divider;
  timespec_zero(result);
  while (timespec_ge(dividend, divider1)) {
    timespec_add(result, divider1);
  }
  return (result);

} 


static struct timespec div(unsigned long dividend, unsigned long divider) {
    int quotient;
    struct timespec result;
    quotient = dividend / divider;

    ++quotient;

#define    MICROSECS_PER_SECOND          1000000
    /* convert back to timespec */
    result.tv_sec = quotient / MICROSECS_PER_SECOND;
    result.tv_nsec = (quotient - (result.tv_sec * MICROSECS_PER_SECOND)) * 1000;
    return result;
}

/* This dividen computation works faster by assuming that the seconds
 * field is less than 1000 seconds, and by truncating the nanoseconds field.
 * If the seconds field is greater than 1000 seconds, a valid but VERY 
 * inefficient response will result! 
 */

static struct timespec efficient_timespec_div(struct timespec dividend, 
					       int divider) {
    unsigned long dividend_microsecs, divider_microsecs;

    if (divider == 0) {
      // do nothing
      return dividend;
    }

    if (dividend.tv_sec >= 1000) {
	/* be ready to pay a BIG penalty now! */
	return (timespec_div(dividend, divider));
    }

#define    MICROSECS_PER_SECOND          1000000

    /* truncate nanoseconds */
    dividend_microsecs = (dividend.tv_sec % 1000) * MICROSECS_PER_SECOND +
      (dividend.tv_nsec / 1000);
    divider_microsecs = divider;
    return div(dividend_microsecs, divider_microsecs);
}



static int strict_ceiling(unsigned long dividend, unsigned long divider)
{
    int quotient;
    quotient = dividend / divider;

    return (quotient+1);
}

static int strict_timespec_ceiling(struct timespec dividend, 
				   struct timespec divider)
{
    int quotient = 1;
    struct timespec divider1;

    divider1 = divider;
    while (timespec_ge(dividend, divider1)) {
	timespec_add(divider1, divider);
	quotient++;
    }
    return (quotient);
}

/* this ceiling computation works faster by assuming that the seconds
 * field is less than 1000 seconds, and by truncating the nanoseconds field.
 * If the seconds field is greater than 1000 seconds, a valid but VERY 
 * inefficient response will result!
 * This function gives the result of floor function + 1
 */
static int efficient_strict_timespec_ceil(struct timespec dividend,
				   struct timespec divider)
{
    unsigned long dividend_microsecs, divider_microsecs;

    if ((divider.tv_sec == 0) && (divider.tv_nsec == 0)) {
	return -1;
    }
    if ((dividend.tv_sec >= 1000) || (divider.tv_sec >= 1000)) {
	/* be ready to pay a BIG penalty now! */
	return (strict_timespec_ceiling(dividend, divider));
    }
#define    MICROSECS_PER_SECOND          1000000

    /* truncate nanoseconds */
    dividend_microsecs = (dividend.tv_sec % 1000) * MICROSECS_PER_SECOND +
	(dividend.tv_nsec / 1000);
    divider_microsecs = (divider.tv_sec % 1000) * MICROSECS_PER_SECOND +
	(divider.tv_nsec / 1000);
    return (strict_ceiling(dividend_microsecs, divider_microsecs));
}



struct timespec timespec_mult(struct timespec multiplicand, int multiplier)
{
    struct timespec result;

    result.tv_sec = 0;
    result.tv_nsec = 0;

    do {
	timespec_add(result, multiplicand);
    } while (--multiplier);

    return (result);
}

int cpu_energy_status_proc(char *buf)
{
char *p = buf;
    int i;
    
    // First line ; the size of the buffer, the number of overflows, the current index
    p += sprintf(p, "%d %d %d \n", MAX_ENERGY_BUFFER, history_round, history_index);
    // Show the history of frequencym, voltage and power
    // The remote monitor will need to figure out the order later
    for (i = 0; i < MAX_ENERGY_BUFFER; i++ ) {
      p += sprintf(p, "%d %d %d %d \n", history_time[i], history_freq[i], history_volt[i], history_power[i]);
    }
    return (p - buf);
}



#endif /* ENERGY_AWARE */
