/* 
 * Real-Time and Multimedia Systems Laboratory
 * Copyright (c) 2000 Carnegie Mellon University
 * All Rights Reserved.
 * 
 * Permission to use, copy, modify and distribute this software and its
 * documentation is hereby granted, provided that both the copyright
 * notice and this permission notice appear in all copies of the
 * software, derivative works or modified versions, and any portions
 * thereof, and that both notices appear in supporting documentation.
 * 
 * CARNEGIE MELLON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS"
 * CONDITION.  CARNEGIE MELLON DISCLAIMS ANY LIABILITY OF ANY KIND FOR
 * ANY DAMAGES WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
 * 
 * Carnegie Mellon requests users of this software to return to
 * 
 *  Real-Time and Multimedia Systems Laboratory
 *  Attn: Prof. Raj Rajkumar
 *  Electrical and Computer Engineering, and Computer Science
 *  Carnegie Mellon University
 *  Pittsburgh PA 15213-3890
 *
 *  or via email to raj@ece.cmu.edu
 * 
 * any improvements or extensions that they make and grant Carnegie Mellon
 * the rights to redistribute these changes.
 */
/*
 * Portable QoS manager, implementing the comcepts described in
 * ``Adaptive Bandwidth Reservation for Multimedia Computing''
 * by L. Abeni and G. Buttazzo
 * IEEE Real Time Computing Systems and Applications 1999
 *
 *			program developed by Luca Abeni
 *					luca@sssup.it
 *					http://hartik.sssup.it/~luca
 */

/************************************************************************/
/*	This file implements the proportional compression function	*/
/************************************************************************/
#include <stdlib.h>
#include <arch/basetype.h>
#include <arch/funcs.h>

#include "prop.h"
#ifdef __PROP_STATS__
#include "stats.h"
#endif

#define MAXPROP 64

static float W = 0;
static float B = 0;
static float BM = 0;
static float BMAX = 0;
static struct propparm *propq;
/*#define QOSDBG*/

#define max(a,b)  (((a) > (b)) ? (a) : (b))

#ifdef __PROP_STATS__
struct t1stat propstats[10000][10];
static int sk = 0;
#endif

/*
 * Initializes the compression function
 */
void prop_init(float freebw)
{
    BMAX = freebw;
    propq = NULL;
}

/*
 * Creates and initializes a compression structure
 */
void *prop_create(float u, float w)
{
    struct propparm *parms;

    parms = malloc(sizeof(struct propparm));
    W += w;
    parms->weight = w;
    parms->bmin = u;
    if (u < parms->bmin) {
	u = parms->bmin;
    }
    parms->util = u;
    parms->reqb = u;
    B += u;
    BM += parms->bmin;
    parms->next = propq;
    propq = parms;

    return parms;
}

/*
 * Frees a compression structure
 */
void prop_destroy(void *pv)
{
    struct propparm *p1;
    int done;
    struct propparm *p = pv;

    if (pv != propq) {
	p1 = propq;
	done = 0;
	while ((p1 != NULL) && (!done)) {
	    if (p1->next == p) {
		done = 1;
	    } else {
		p1 = p1->next;
	    }
	}

	if (p1 == NULL) {
	    return;
	}
	p1->next = p->next;
    } else {
	propq = p->next;
    }

    BM -= p->bmin;
    B -= p->reqb;
    W -= p->weight;
    free(p);
}

/*
 * Returns the compressed bandwidth
 */
float prop_getband(void *pv)
{
    struct propparm *p = pv;

    return p->util;
}

/*
 * Sets a new requested bandwidth
 */
void prop_setband(void *pv, float u)
{
    struct propparm *p = pv;

    if (u < p->bmin) {
	u = p->bmin;
    }
    B -= p->reqb;
    p->reqb = u;
    B += p->reqb;
}

/*
 * Hey!!! This is the real work... Performing the bandwidth compression!!!
 */
void prop_compress(void)
{
    float fact;
    float sum;
    struct propparm *p;

#ifdef __PROP_STATS__
    propstats[sk][6].time = qos_time();
    propstats[sk][6].p1 = B;
    propstats[sk][6].p2 = BMAX;
#endif
    if (B >= BMAX) {
	/* The system is overloaded... Compress the bandwidths */

	/* First of all, compute S = sum((B_i - B^min_i) * w_i) */
	sum = 0;
	p = propq;
	while (p != NULL) {
#ifdef __PROP_STATS__
	    propstats[sk][i].time = qos_time();
	    propstats[sk][i].band = tpar[i].reqb;
#endif
	    sum += (p->reqb - p->bmin) * p->weight;
	    p = p->next;
	}
#ifdef QOSDBG
	cprintf("BMAX= %f BMin= %f\n", BMAX, BM);
	cprintf("Sum= %f       W= %f\n", sum, W);
#endif
	/* ...Then, F = B^max - B^min) / S ... */
	fact = (BMAX - BM) / sum;
#ifdef QOSDBG
	cprintf("B= %f ScaleFact: %f\n", B, fact);
#endif

	/* 
	 * ... And now, B'_i = B^min_i + (B_i - B^min_i) * F * w_i ...
	 * ... Isn't it simple? :)
	 */
	p = propq;
	while (p != NULL) {
#ifdef QOSDBG
	    cprintf
		("Compressing Task 0x%p (bmin %f share %f): band from %f (req %f)",
		 p, p->bmin, p->weight / W, p->util, p->reqb);
#endif
	    B -= p->reqb;
	    /* This is the stupid scaling... */
/*	    p->util = p->bmin + (p->util - p->bmin) * fact * p->weight / W; */
	    p->util = p->bmin + (p->reqb - p->bmin) * fact * p->weight;
#ifdef QOSDBG
	    cprintf(" to %f\n", p->util);
#endif

#if 0
	    /* This avoid that an ``inactive'' task requires too much bandwdth
	       (remember: only the compression decrease bandwidths...
	     */
	    p->reqb = p->util;
#endif

	    B += p->reqb;

#ifdef __PROP_STATS__
	    propstats[sk][i].band = p->util;
#endif
	    p = p->next;
	}
#ifdef QOSDBG
	cprintf("B= %f DONE!!!\n", B);
#endif
    } else {
	/* No compression has to be performed... Bi'=Bi */
	p = propq;
	while (p != NULL) {
	    p->util = p->reqb;
	    p = p->next;
	}
    }

#ifdef __PROP_STATS__
    sk++;
#endif
}
