/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * redirect.c: redirect requests to other server sockets (such as Apache).
 */

#include <net/tux.h>

static void dummy_destructor(struct open_request *req)
{
}

static struct or_calltable dummy = 
{
	0,
 	NULL,
 	NULL,
 	&dummy_destructor,
 	NULL
};

static int redirect_sock (tux_req_t *req, const int port)
{
	struct socket *sock = req->sock;
	struct open_request *tcpreq;
	struct sock *sk, *oldsk;
	int err = -1;

	/*
	 * Look up (optional) listening user-space socket.
	 */
	local_bh_disable();
	sk = tcp_v4_lookup_listener(INADDR_ANY, port, 0);
	local_bh_enable();

	/* No secondary server found */
	if (!sk)
		goto out;

	/*
	 * Requeue the 'old' socket as an accept-socket of
	 * the listening socket. This way we can shuffle
	 * a socket around. Since we've read the input data
	 * via the non-destructive MSG_PEEK, the secondary
	 * server can be used transparently.
	 */
	oldsk = sock->sk;
	lock_sock(sk);

	if (sk->state != TCP_LISTEN)
		goto out_unlock;

	tcpreq = tcp_openreq_alloc();
	if (!tcpreq)
		goto out_unlock;

	unlink_tux_socket(req);

	sock->sk = NULL;
	sock->state = SS_UNCONNECTED;

	tcpreq->class = &dummy;
	write_lock_irq(&oldsk->callback_lock);
	oldsk->socket = NULL;
        oldsk->sleep = NULL;
	write_unlock_irq(&oldsk->callback_lock);

	tcp_acceptq_queue(sk, tcpreq, oldsk);

	sk->tp_pinfo.af_tcp.nonagle = 0;
	sk->data_ready(sk, 0);

	/*
	 * It's now completely up to the secondary
	 * server to handle this request.
	 */
	sock_release(req->sock);
	req->sock = NULL;
	req->parsed_len = 0;
	req->input_skb = NULL;

	err = 0;

out_unlock:
	release_sock(sk);
	sock_put(sk);
out:
	return err;
}

void redirect_request (tux_req_t *req, int cachemiss)
{
	if (cachemiss)
		TUX_BUG();
	if (req->error == 3)
		goto out_flush;
	if (!req->sock)
		TUX_BUG();

	Dprintk("redirecting request (headers: {%s})\n", req->headers);
	if (!req->status)
		req->status = -1;
	/*
	 * 301 redirects are special and do not go to
	 * the secondary server.
	 */
	if ((req->status == 301) || redirect_sock(req, tux_clientport)) {
		if (req->parsed_len)
			trunc_headers(req);
		if ((req->status != 301) && req->proto) {
			req->proto->illegal_request(req, cachemiss);
			return;
		}
		goto out_flush;
	} else {
		if (req->ftp_data_sock)
			BUG();
	}
out_flush:
	if (req->status != 301) {
		clear_keepalive(req);
		if (!tux_redirect_logging)
			req->status = 0;
	}
	flush_request(req, cachemiss);
}

