/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * proto_http.c: HTTP application protocol support
 *
 * Right now we detect simple GET headers, anything more
 * subtle gets redirected to secondary server port.
 */

#include <net/tux.h>
#include "parser.h"

/* 
 * Parse the HTTP message and put results into the request structure.
 * CISAPI extensions do not see the actual message buffer.
 *
 * Any perceived irregularity is honored with a redirect to the
 * secondary server - which in most cases should be Apache. So
 * if TUX gets confused by some strange request we fall back
 * to Apache to be RFC-correct.
 *
 * The parser is 'optimistic', ie. it's optimized for the case where
 * the whole message is available and correct. The parser is also
 * supposed to be 'robust', ie. it can be called multiple times with
 * an incomplete message, as new packets arrive.
 */

#ifdef CONFIG_ARCH_S390
# error Add EBCDIC support to the parser first!
#endif

static inline int TOHEX (unsigned char c)
{
	switch (c) {
		case '0' ... '9': c -= '0'; break;
		case 'a' ... 'f': c -= 'a'-10; break;
		case 'A' ... 'F': c -= 'A'-10; break;
	default:
		c = -1;
	}
	return c;
}

/*
 * This function determines whether the client supports
 * gzip-type content-encoding.
 */
static int may_gzip (const char *str, int len)
{
	const char *tmp, *curr;
	int i;

	if (len <= 4)
		return 0;
	tmp = str;
	for (i = 0; i <= len-6; i++) {
		Dprintk("gzip-checking: {%s}\n", tmp);
		if (memcmp(tmp, " gzip", 5)) {
			tmp++;
			continue;
		}
		curr = tmp + 5;

		if (*curr == ',' || *curr == '\r')
			return 1;
		if (memcmp(curr, ";q=", 3))
			return 0;
		curr += 3;
		/*
		 * Every qvalue except explicitly zero is accepted.
		 * Zero values are "q=0.0", "q=0.00", "q=0.000".
		 * Parsing is optimized.
		 */
		if (*curr == '0') {
			curr += 2;
			if (*curr == '0') {
				curr++;
				if (*curr == ' ' || *curr == '\r')
					return 0;
				if (*curr == '0') {
					curr++;
					if (*curr == ' ' || *curr == '\r')
						return 0;
					if (*curr == '0') {
						curr++;
						if (*curr == ' ' ||
								*curr == '\r')
							return 0;
					}
				}
			}
		}
		return 1;
	}
	return 0;
}

static void http_process_message (tux_req_t *req, int cachemiss);

int parse_http_message (tux_req_t *req, const int total_len)
{
	int hexhex = 0, hex_val_0 = 0, hex_val_1 = 0;
	const char *curr, *uri, *message;
	int objectname_len, left;
	int have_r = 0;
	char c;

	left = total_len;
	message = req->headers;
	Dprintk("parsing request:\n---\n%s\n---\n", message);
/*
 * RFC 2616, 5.1:
 *
 *	 Request-Line   = Method SP Request-URI SP HTTP-Version CRLF
 */

	if (!total_len)
		TUX_BUG();

	curr = message;

#define GOTO_INCOMPLETE do { Dprintk("incomplete at %s:%d.\n", __FILE__, __LINE__); goto incomplete_message; } while (0)
#define GOTO_REDIR do { TDprintk("redirect secondary at %s:%d.\n", __FILE__, __LINE__); goto error; } while (0)

#define PRINT_MESSAGE_LEFT \
    Dprintk("message left (%d) at %s:%d:\n--->{%s}<---\n", left, __FILE__, __LINE__, curr)

	switch (*curr) {
		case 'G':
			if (PARSE_METHOD(req,curr,GET,left))
				break;
			GOTO_REDIR;

		case 'H':
			if (PARSE_METHOD(req,curr,HEAD,left))
				break;
			GOTO_REDIR;

		case 'P':
			if (PARSE_METHOD(req,curr,POST,left))
				break;
			if (PARSE_METHOD(req,curr,PUT,left))
				break;
			GOTO_REDIR;

		default:
			GOTO_REDIR;
	}

	req->method_str = message;
	req->method_len = curr-message-1;

	Dprintk("got method %d\n", req->method);

	PRINT_MESSAGE_LEFT;

	/*
	 * Ok, we got one of the methods we can handle, parse
	 * the URI:
	 */

	{
		// Do not allow leading "../" and intermediate "/../"
		int dotdot = 1;
		char *tmp = req->objectname;
		int slashcheck = 1;

		req->uri_str = uri = curr;

		for (;;) {
			c = get_c(curr,left);
			if (slashcheck) {
				if (c == '/')
					continue;
				slashcheck = 0;
			}

			PRINT_MESSAGE_LEFT;
			if (c == ' ' || c == '?' || c == '\r' || c == '\n')
				break;
			if (c == '#')
				GOTO_REDIR;

			/*
			 * First handle HEX HEX encoding
			 */
			switch (hexhex) {
				case 0:
					if (c == '%') {
						hexhex = 1;
						goto continue_parsing;
					}
					break;
				case 1:
					hex_val_0 = TOHEX(c);
					if (hex_val_0 < 0)
						GOTO_REDIR;
					hexhex = 2;
					goto continue_parsing;
				case 2:
					hex_val_1 = TOHEX(c);
					if (hex_val_1 < 0)
						GOTO_REDIR;
					c = (hex_val_0 << 4) | hex_val_1;
					if (!c)
						GOTO_REDIR;
					hexhex = 0;
				default:
					TUX_BUG();
			}
			if (hexhex)
				TUX_BUG();

			switch (dotdot) {
				case 0:
					break;
				case 1:
					if (c == '.')
						dotdot = 2;
					else
						dotdot = 0;
					break;
				case 2:
					if (c == '.')
						dotdot = 3;
					else
						dotdot = 0;
					break;
				case 3:
					if (c == '/')
						GOTO_REDIR;
					else
						dotdot = 0;
					break;
				default:
					TUX_BUG();
			}
			if (!dotdot && (c == '/'))
				dotdot = 1;

			*(tmp++) = c;
continue_parsing:
			if (curr - uri >= MAX_OBJECTNAME_LEN)
				GOTO_REDIR;
		}
		PRINT_MESSAGE_LEFT;
		*tmp = 0;

		// handle trailing "/.."
		if (dotdot == 3)
			GOTO_REDIR;

		objectname_len = tmp - req->objectname;
		req->objectname_len = objectname_len;
	}
	Dprintk("got filename %s (%d)\n", req->objectname, req->objectname_len);

	PRINT_MESSAGE_LEFT;

	/*
	 * Parse optional query string. Copy until end-of-string or space.
	 */
	if (c == '?') {
		int query_len;
		const char *query;

		req->query_str = query = curr;

		for (;;) {
			c = get_c(curr,left);
			if (c == ' ')
				break;
			if (c == '#')
				GOTO_REDIR;
		}
		query_len = curr-query-1;
		req->query_len = query_len;
	}
	if (req->query_len)
		Dprintk("got query string %s (%d)\n", req->query_str, req->query_len);
	req->uri_len = curr-uri-1;
	if (!req->uri_len)
		GOTO_REDIR;
	Dprintk("got URI %s (%d)\n", req->uri_str, req->uri_len);

	PRINT_MESSAGE_LEFT;
	/*
	 * Parse the HTTP version field:
	 */
	req->version_str = curr;
	if (!PARSE_TOKEN(curr,"HTTP/1.",left))
		GOTO_REDIR;

	switch (get_c(curr,left)) {
		case '0':
			req->version = HTTP_1_0;
			break;
		case '1':
			req->version = HTTP_1_1;
			break;
		default:
			GOTO_REDIR;
	}
	/*
	 * If this connection had other requests already (ie. it came from
	 * a keepalive-aware client) then default to keep-alive, except if
	 * the client specifically requests a close or max_keepalives is
	 * set to 0.
	 *
	 * Otherwise we default to keepalive in the HTTP/1.1 case and default
	 * to non-keepalive in the HTTP/1.0 case.
	 */
	clear_keepalive(req);
	if (tux_max_keepalives && (req->nr_keepalives ||
					(req->version == HTTP_1_1)))
		req->keep_alive = 1;
	req->version_len = curr - req->version_str;

	if (get_c(curr,left) != '\r')
		GOTO_REDIR;
	if (get_c(curr,left) != '\n')
		GOTO_REDIR;

	Dprintk("got version %d [%d]\n", req->version, req->version_len);
	PRINT_MESSAGE_LEFT;

	/*
	 * Now parse (optional) request header fields:
	 */
	for (;;) {
		char c;

		c = get_c(curr,left);
		switch (c) {
		case '\r':
			if (have_r)
				GOTO_REDIR;
			have_r = 1;
			continue;
		case '\n':
			if (!have_r)
				GOTO_REDIR;
			goto out;
		default:
			if (have_r)
				GOTO_REDIR;
		}

#define PARSE_STR_FIELD(char,field,str,len) 				\
	if (PARSE_TOKEN(curr,field,left)) {				\
		req->str = curr;					\
		SKIP_LINE(curr,left);					\
		req->len = curr - req->str - 2;				\
		Dprintk(char field "field: %s.\n", req->str);		\
		break;							\
	}

#define ALLOW_UNKNOWN_FIELDS 1
#ifdef ALLOW_UNKNOWN_FIELDS
# define UNKNOWN_FIELD { SKIP_LINE(curr,left); break; }
#else
# define UNKNOWN_FIELD GOTO_REDIR
#endif

		switch (c) {
		case 'A':
			PARSE_STR_FIELD("A","ccept: ",
				accept_str,accept_len);
			if (PARSE_TOKEN(curr,"ccept-Encoding: ",left)) {
				const char *str = curr-1;

				req->accept_encoding_str = curr;
				SKIP_LINE(curr,left);
				req->accept_encoding_len = curr - req->accept_encoding_str - 2;
				Dprintk("Accept-Encoding field: {%s}.\n", str);

				if (tux_compression && may_gzip(str,curr-str)) {
					Dprintk("client accepts gzip!.\n");
					req->may_send_gzip = 1;
				}
				break;
			}
			PARSE_STR_FIELD("A","ccept-Charset: ",
				accept_charset_str,accept_charset_len);
			PARSE_STR_FIELD("A","ccept-Language: ",
				accept_language_str,accept_language_len);
			UNKNOWN_FIELD;

		case 'C':
			if (PARSE_TOKEN(curr,"onnection: ",left)) {
				switch (get_c(curr,left)) {
				case 'K':
					if (!PARSE_TOKEN(curr,"eep-Alive",left))
						GOTO_REDIR;
					req->keep_alive = 1;
					break;

				case 'c':
					if (!PARSE_TOKEN(curr,"lose",left))
						GOTO_REDIR;
					clear_keepalive(req);
					break;

				case 'k':
					if (!PARSE_TOKEN(curr,"eep-alive",left))
						GOTO_REDIR;
					req->keep_alive = 1;
					break;

				default:
					GOTO_REDIR;
				}
				// allow other tokens.
				SKIP_LINE(curr,left);
				break;
			}

			PARSE_STR_FIELD("C","ookie: ",
				cookies_str,cookies_len);
			PARSE_STR_FIELD("C","ontent-Type: ",
				content_type_str,content_type_len);

			if (PARSE_TOKEN(curr,"ontent-Length: ",left) ||
			 PARSE_TOKEN(curr,"ontent-length: ",left)) {
				const char *tmp;
				req->contentlen_str = curr;
				SKIP_LINE(curr,left);
				req->contentlen_len = curr - req->contentlen_str - 2;
				if (req->contentlen_len) {
					tmp = req->contentlen_str;
					req->content_len = simple_strtoul(tmp, NULL, 10);
				}
				Dprintk("Content-Length field: %s [%d].\n", req->contentlen_str, req->contentlen_len);
				Dprintk("Content-Length value: %d.\n", req->content_len);
				break;
			}
			PARSE_STR_FIELD("C","ache-Control: ",
				cache_control_str,cache_control_len);
			UNKNOWN_FIELD;

		case 'H':
			if (PARSE_TOKEN(curr,"ost: ",left)) {
				const char *tmp = curr;
				char *tmp2 = req->host;

				COPY_LINE_TOLOWER(curr, tmp2, left);
				req->host_len = curr - tmp - 2;
				req->host[req->host_len] = 0;
				Dprintk("Host field: %s [%d].\n", req->host, req->host_len);
				break;
			}
			UNKNOWN_FIELD;

		case 'I':
			PARSE_STR_FIELD("I","f-Modified-Since: ",
				if_modified_since_str,if_modified_since_len);
			UNKNOWN_FIELD;

		case 'N':
			PARSE_STR_FIELD("N","egotiate: ",
				negotiate_str,negotiate_len);
			UNKNOWN_FIELD;

		case 'P':
			PARSE_STR_FIELD("P","ragma: ",
				pragma_str,pragma_len);
			UNKNOWN_FIELD;

		case 'R':
			PARSE_STR_FIELD("R","eferer: ",
				referer_str,referer_len);
			UNKNOWN_FIELD;

		case 'U':
			PARSE_STR_FIELD("U","ser-Agent: ",
				user_agent_str,user_agent_len);
			UNKNOWN_FIELD;

		default:
			UNKNOWN_FIELD;
		}
		PRINT_MESSAGE_LEFT;
	}
out:
	/*
	 * POST data.
	 */
	if ((req->method == METHOD_POST) && req->content_len) {
		PRINT_MESSAGE_LEFT;
		if (curr + req->content_len > message + total_len)
			GOTO_INCOMPLETE;
		req->post_data_str = curr;
		req->post_data_len = req->content_len;
		curr += req->content_len;
		left -= req->content_len;
		Dprintk("POST-ed data: {%s}\n", req->post_data_str);
	}

	switch (req->method) {
		default:
			GOTO_REDIR;
		case METHOD_GET:
		case METHOD_HEAD:
		case METHOD_POST:
		case METHOD_PUT:
	}

#define TUX_SCHEME "http://"
#define TUX_SCHEME_LEN (sizeof(TUX_SCHEME)-1)

	if (!memcmp(req->objectname, TUX_SCHEME, TUX_SCHEME_LEN)) {

		/* http://user:password@host:port/object */

		char *head, *tail, *end, *host, *port;
		int host_len, objectname_len;

		head = req->objectname + TUX_SCHEME_LEN;
		end = req->objectname + req->objectname_len;

		tail = memchr(head, '/', end - head);
		if (!tail)
			GOTO_REDIR;
		host = memchr(head, '@', tail - head);
		if (!host)
			host = head;
		else
			host++;
		if (!*host)
			GOTO_REDIR;
		port = memchr(host, ':', tail - host);
		if (port)
			host_len = port - host;
		else
			host_len = tail - host;
		if (host_len >= MAX_HOST_LEN)
			GOTO_REDIR;
		memcpy(req->host, host, host_len);
		req->host_len = host_len;
		req->host[host_len] = 0;


		tail++;
		while (*tail == '/')
			tail++;

		objectname_len = end - tail;
		memcpy(req->objectname, tail, objectname_len);
		req->objectname_len = objectname_len;
		req->objectname[objectname_len] = 0;
	}

	if ((req->version == HTTP_1_1) && !req->host_len)
		GOTO_REDIR;
	if (req->objectname[0] == '/')
		GOTO_REDIR;
	/*
	 * Lets make sure nobody plays games with the host
	 * header in a virtual hosting environment:
	 */
	if (virtual_server && req->host_len) {
		if (memchr(req->host, '/', req->host_len))
			GOTO_REDIR;
		if (req->host[0] == '.') {
			if (req->host_len == 1)
				GOTO_REDIR;
			if ((req->host_len == 2) && (req->host[0] == '.'))
				GOTO_REDIR;
		}
	}
	/*
	 * From this point on the request is for the main TUX engine:
	 */
	Dprintk("ok, request accepted.\n");

	if (req->keep_alive) {
		req->nr_keepalives++;
		if (req->nr_keepalives == KEEPALIVE_HIST_SIZE)
			req->nr_keepalives--;
		kstat.nr_keepalive_reqs++;
	} else
		kstat.nr_nonkeepalive_reqs++;
	kstat.keepalive_hist[req->nr_keepalives]++;

	PRINT_MESSAGE_LEFT;
	req->parsed_len = curr-message;
	if (req->dentry)
		TUX_BUG();
	add_tux_atom(req, http_process_message);

	return req->parsed_len;

incomplete_message:
	Dprintk("incomplete message!\n");
	PRINT_MESSAGE_LEFT;

	return 0;

error:
	if (total_len > 0)
		req->parsed_len = total_len;
	else
		req->parsed_len = 0;
	TDprintk("redirecting message to secondary server!\n");
	PRINT_MESSAGE_LEFT;
	return -1;
}

int handle_gzip_req (tux_req_t *req, unsigned int flags)
{
	char *curr = req->objectname + req->objectname_len;
	struct dentry *dentry;
	struct inode *inode, *orig_inode;
	ssize_t size, orig_size;

	*curr++ = '.';
	*curr++ = 'g';
	*curr++ = 'z';
	*curr++ = 0;
	req->objectname_len += 3;

	dentry = tux_lookup(req, req->objectname, flags);

	req->objectname_len -= 3;
	req->objectname[req->objectname_len] = 0;

	if (!dentry)
		return 0;
	if (IS_ERR(dentry)) {
		if (PTR_ERR(dentry) == -EWOULDBLOCKIO)
			return 1;
		return 0;
	}

	inode = dentry->d_inode;
	size = inode->i_size;
	orig_inode = req->dentry->d_inode;
	orig_size = orig_inode->i_size;

	if (!url_permission(inode)
			&& (size < orig_size)
			&& (inode->i_mtime >= orig_inode->i_mtime)) {

		release_req_dentry(req);
		install_req_dentry(req, dentry);
		req->filelen = size;
		Dprintk("content WILL be gzipped!\n");
		req->content_gzipped = 1;
	} else
		dput(dentry);

	return 0;
}

static spinlock_t mimetypes_lock = SPIN_LOCK_UNLOCKED;

static LIST_HEAD(mimetypes_head);

static mimetype_t default_mimetype = { type: "text/html", type_len: 9 };

void add_mimetype (char *new_ext, char *new_type)
{
	int type_len = strlen(new_type);
	int ext_len = strlen(new_ext);
	mimetype_t *mime;
	char *ext, *type;

	mime = kmalloc(sizeof(*mime), GFP_KERNEL);
	memset(mime, 0, sizeof(*mime));
	ext = kmalloc(ext_len + 1, GFP_KERNEL);
	type = kmalloc(type_len + 1, GFP_KERNEL);
	strcpy(ext, new_ext);
	strcpy(type, new_type);

	mime->ext = ext;
	mime->ext_len = ext_len;

	mime->type = type;
	mime->type_len = type_len;

	mime->special = NORMAL_MIME_TYPE;
	if (!strcmp(type, "TUX/redirect"))
		mime->special = MIME_TYPE_REDIRECT;
	if (!strcmp(type, "TUX/CGI"))
		mime->special = MIME_TYPE_CGI;
	if (!strcmp(type, "TUX/module"))
		mime->special = MIME_TYPE_MODULE;

	spin_lock(&mimetypes_lock);
	list_add(&mime->list, &mimetypes_head);
	spin_unlock(&mimetypes_lock);
}

static inline int ext_matches (char *file, int len, char *ext, int extlen)
{
	int i;
	char *tmp = file + len-1;
	char *tmp2 = ext + extlen-1;

	if (len < extlen)
		return 0;

	for (i = 0; i < extlen; i++) {
		if (*tmp != *tmp2)
			return 0;
		tmp--;
		tmp2--;
	}
	return 1;
}

/*
 * Overhead is not a problem, we cache the MIME type
 * in the dentry.
 */
static mimetype_t * lookup_mimetype (tux_req_t *req)
{
	char *objectname = req->objectname;
	int len = req->objectname_len;
	mimetype_t *mime = NULL;
	struct list_head *head, *tmp, *tmp1, *tmp2, *tmp3;

	if (!memchr(objectname, '.', len))
		goto out;

	spin_lock(&mimetypes_lock);
	head = &mimetypes_head;
	tmp = head->next;

	while (tmp != head) {
		mime = list_entry(tmp, mimetype_t, list);
		if (ext_matches(objectname, len, mime->ext, mime->ext_len)) {
			/*
			 * Percolate often-used mimetypes down:
			 */
			if (tmp->prev != &mimetypes_head) {
				tmp1 = tmp;
				tmp2 = tmp->prev;
				tmp3 = tmp->prev->prev;
				list_del(tmp1);
				list_del(tmp2);
				list_add(tmp, tmp3);
				list_add(tmp2, tmp);
			}
			break;
		} else
			mime = NULL;
		tmp = tmp->next;
	}
	spin_unlock(&mimetypes_lock);

out:
	if (!mime)
		mime = &default_mimetype;
	return mime;
}

void free_mimetypes (void)
{
	struct list_head *head, *tmp, *next;
	mimetype_t *mime;

	spin_lock(&mimetypes_lock);
	head = &mimetypes_head;
	tmp = head->next;

	while (tmp != head) {
		next = tmp->next;
		mime = list_entry(tmp, mimetype_t, list);
		list_del(tmp);

		kfree(mime->ext);
		mime->ext = NULL;
		kfree(mime->type);
		mime->type = NULL;
		kfree(mime);

		tmp = next;
	}
	spin_unlock(&mimetypes_lock);
}

/*
 * Various constant HTTP responses:
 */

static const char forbidden[] =
	"HTTP/1.1 403 Forbidden\r\n"
	"Content-Length: 24\r\n\r\n"
	"<HTML> Forbidden </HTML>";

static const char not_found[] =
	"HTTP/1.1 404 Not Found\r\n"
	"Content-Length: 29\r\n\r\n"
	"<HTML> Page Not Found </HTML>";

static const char timed_out[] =
	"HTTP/1.1 408 Request Timeout\r\n"
	"Content-Length: 34\r\n\r\n"
	"<HTML> Request Timed Out </HTML>";

#define REDIRECT_1 \
	"HTTP/1.1 301 Moved Permanently\r\n" \
	"Location: http://"

#define REDIRECT_1_LEN (sizeof(REDIRECT_1) - 1)

#define REDIRECT_2 \
	"/\r\nContent-Length: 36\r\n" \
	"Connection: Keep-Alive\r\n" \
	"Content-Type: text/html\r\n\r\n" \
	"<HTML> 301 Moved Permanently </HTML>"

#define REDIRECT_2_LEN (sizeof(REDIRECT_2) - 1)

void send_async_timed_out (tux_req_t *req)
{
	__send_async_message(req, timed_out, 408);
}

void send_async_err_forbidden (tux_req_t *req)
{
	__send_async_message(req, forbidden, 403);
}

void send_async_err_not_found (tux_req_t *req)
{
	__send_async_message(req, not_found, 404);
}

static void send_ret_redirect (tux_req_t *req, int cachemiss)
{
	char *buf;
	int size;
	int uts_len = 0;

	size = REDIRECT_1_LEN;
	if (req->host_len)
		size += req->host_len;
	else {
		down_read(&uts_sem);
		uts_len = strlen(system_utsname.nodename);
		size += uts_len;
	}
	if (req->objectname[0] != '/')
		size++;
	size += req->objectname_len;
	size += REDIRECT_2_LEN;

	if (size > PAGE_SIZE) {
		zap_request(req, cachemiss);
		return;
	}

	buf = get_abuf(req, size);

	memcpy(buf, REDIRECT_1, REDIRECT_1_LEN);
	buf += REDIRECT_1_LEN;

	Dprintk("req %p, host: %s, host_len: %d.\n", req, req->host, req->host_len);
	if (req->host_len) {
		memcpy(buf, req->host, req->host_len);
		buf += req->host_len;
	} else {
		memcpy(buf, system_utsname.nodename, uts_len);
		up_read(&uts_sem);
		buf += uts_len;
	}
	if (req->objectname[0] != '/') {
		buf[0] = '/';
		buf++;
	}

	memcpy(buf, req->objectname, req->objectname_len);
	buf += req->objectname_len;

	memcpy(buf, REDIRECT_2, REDIRECT_2_LEN);
	buf += REDIRECT_2_LEN;

	req->status = 301;
	send_abuf(req, size, MSG_DONTWAIT);
	add_req_to_workqueue(req);
}

static void http_got_request (tux_req_t *req)
{
	add_tux_atom(req, parse_request);
	add_req_to_workqueue(req);
}


tux_attribute_t * lookup_tux_attribute (tux_req_t *req)
{
	tux_attribute_t *attr;
	struct inode *inode;
	mimetype_t *mime;

	attr = kmalloc(sizeof(*attr), GFP_KERNEL);
	if (!attr)
		TUX_BUG();
	memset(attr, 0, sizeof(*attr));

	mime = lookup_mimetype(req);

	inode = req->dentry->d_inode;
	if (!inode->i_uid && !inode->i_gid) {
		if (mime->special == MIME_TYPE_MODULE) {
			attr->tcapi = lookup_tuxmodule(req->objectname);
			if (!attr->tcapi) {
				req_err(req);
				mime = &default_mimetype;
			}
		}
	} else {
		if (mime->special && (mime->special != MIME_TYPE_REDIRECT))
			mime = &default_mimetype;
	}
	attr->mime = mime;

	return attr;
}

static void http_pre_header (tux_req_t *req, int push);
static void http_post_header (tux_req_t *req, int cachemiss);
static void http_send_body (tux_req_t *req, int cachemiss);

static void http_process_message (tux_req_t *req, int cachemiss)
{
	tux_attribute_t *attr;
	int missed;
	unsigned int lookup_flag = cachemiss ? 0 : LOOKUP_ATOMIC;

	Dprintk("handling req %p, cachemiss: %d.\n", req, cachemiss);

	/*
	 * URL redirection support - redirect all valid requests
	 * to the first userspace module.
	 */
	if (tux_all_userspace) {
		tcapi_template_t *tcapi = get_first_usermodule();
		if (tcapi) {
			req->usermode = 1;
			req->usermodule_idx = tcapi->userspace_id;
			goto usermode;
		}
	}
	missed = lookup_url(req, lookup_flag);
	if (missed == 2) {
		send_ret_redirect(req, cachemiss);
		return;
	}
	if (req->error)
		goto error;

	if (missed) {
cachemiss:
		if (cachemiss)
			TUX_BUG();
		Dprintk("uncached request.\n");
		INC_STAT(static_lookup_cachemisses);
		if (req->dentry)
			TUX_BUG();
		add_tux_atom(req, http_process_message);
		queue_cachemiss(req);
		return;
	}

	attr = req->dentry->d_tux_data;
	if (!attr) {
		attr = lookup_tux_attribute(req);
		if (!attr)
			TUX_BUG();
		req->dentry->d_tux_data = attr;
	}
	req->attr = attr;
	if (attr->mime)
		Dprintk("using MIME type %s:%s, %d.\n", attr->mime->type, attr->mime->ext, attr->mime->special);
	if (attr->tcapi) {
		req->usermode = 1;
		req->usermodule_idx = attr->tcapi->userspace_id;
		if (req->module_dentry)
			TUX_BUG();
		req->module_dentry = dget(req->dentry);
		release_req_dentry(req);
		goto usermode;
	}

	switch (attr->mime->special) {
		case MIME_TYPE_MODULE:
			req->usermode = 1;
			goto usermode;

		case MIME_TYPE_REDIRECT:
			req->error = 1;
			goto error;

		case MIME_TYPE_CGI:
#if CONFIG_TUX_EXTCGI
			Dprintk("CGI request %p.\n", req);
			query_extcgi(req);
			return;
#endif

		default:
	}
	if (req->usermode)
		TUX_BUG();

	if (req->may_send_gzip)
		if (handle_gzip_req(req, lookup_flag))
			goto cachemiss;
	if (req->parsed_len)
		trunc_headers(req);

	if (req->error)
		goto error;

	add_tux_atom(req, http_send_body);
	add_tux_atom(req, http_post_header);

	http_pre_header(req, req->method == METHOD_HEAD ? 1 : 0);

	add_req_to_workqueue(req);
	return;

error:
	if (req->error)
		zap_request(req, cachemiss);
	return;

usermode:
	add_req_to_workqueue(req);
}

static void http_post_header (tux_req_t *req, int cachemiss)
{
#if CONFIG_TUX_DEBUG
	req->bytes_expected = req->filelen;
#endif
	req->bytes_sent = 0; // data comes now.

	add_req_to_workqueue(req);
}

static void http_send_body (tux_req_t *req, int cachemiss)
{
	int ret;

	Dprintk("SEND req %p <%p> (sock %p, sk %p) (keepalive: %d, status: %d)\n", req, __builtin_return_address(0), req->sock, req->sock->sk, req->keep_alive, req->status);

	SET_TIMESTAMP(req->output_timestamp);

	if (req->error) {
#if CONFIG_TUX_DEBUG
		req->bytes_expected = 0;
#endif
		req->in_file.f_pos = 0;
		/*
		 * We are in the middle of a file transfer,
		 * zap it immediately:
		 */
		TDprintk("req->error = 3.\n");
		req->error = 3;
		zap_request(req, cachemiss);
		return;
	}

repeat:
	ret = 0;
	if (!req->status)
		req->status = 200;
	if (req->method != METHOD_HEAD)
		ret = generic_send_file(req, 1, !cachemiss, req->sock);
	else {
#if CONFIG_TUX_DEBUG
		req->bytes_expected = 0;
#endif
	}

	switch (ret) {
		case -5:
			add_tux_atom(req, http_send_body);
			output_timeout(req);
			break;
		case -4:
			add_tux_atom(req, http_send_body);
			if (add_output_space_event(req, req->sock)) {
				del_tux_atom(req);
				goto repeat;
			}
			break;
		case -3:
			INC_STAT(static_sendfile_cachemisses);
			add_tux_atom(req, http_send_body);
			queue_cachemiss(req);
			break;
		default:
			req->in_file.f_pos = 0;
			add_req_to_workqueue(req);
			break;
	}
}

#define DEFAULT_DATE "Wed, 01 Jan 1970 00:00:01 GMT"

char tux_date [DATE_LEN] = DEFAULT_DATE;

/*
 * HTTP header
 */

#define HEADER_PART1A \
		"HTTP/1.1 200 OK\r\n" \
		"Content-Type: "

#define HEADER_PART1B \
		"HTTP/1.1 200 OK" \

#define HEADER_PART1C \
		"HTTP/1.1 404 Page Not Found\r\n" \
		"Content-Type: "

#define MAX_MIMETYPE_LEN 20

#define HEADER_PART2_keepalive "\r\nConnection: Keep-Alive\r\nDate: "

#define HEADER_PART2_close "\r\nConnection: close\r\nDate: "

#define HEADER_PART2_none "\r\nDate: "

// date "%s"

#define HEADER_PART3A "\r\nContent-Encoding: gzip"
#define HEADER_PART3BX "\r\nContent-Length: "

/*
 * Please acknowledge our hard work by not changing this define, or
 * at least please acknowledge us by leaving "TUX/2.0 (Linux)" in
 * the ID string. Thanks! :-)
 */
#define HEADER_PART3BY "\r\nServer: TUX/2.0 (Linux)\r\nContent-Length: "
#define HEADER_PART4 "\r\n\r\n"

#define MAX_OUT_HEADER_LEN (sizeof(HEADER_PART1A) + MAX_MIMETYPE_LEN + \
		sizeof(HEADER_PART2_keepalive) + DATE_LEN + \
		sizeof(HEADER_PART3A) + sizeof(HEADER_PART3BY) + \
		12 + sizeof(HEADER_PART4))

static void http_pre_header (tux_req_t *req, int push)
{
	unsigned long flags;
	char *buf, *curr;
	mimetype_t *mime;
	int size;

	if (MAX_OUT_HEADER_LEN > PAGE_SIZE)
		TUX_BUG();
	if (req->attr->tcapi || req->usermode)
		TUX_BUG();

#define COPY_STATIC_PART(nr,curr)					\
	do {								\
		memcpy(curr, HEADER_PART##nr, sizeof(HEADER_PART##nr)-1); \
		curr += sizeof(HEADER_PART##nr)-1;			\
	} while (0)

	buf = curr = get_abuf(req, MAX_OUT_HEADER_LEN);

	mime = req->attr->mime;
	if (!mime)
		TUX_BUG();

	if (req->status == 404) {
		COPY_STATIC_PART(1C, curr);
		memcpy(curr, mime->type, mime->type_len);
		curr += mime->type_len;
	} else {
		if (tux_noid && (mime == &default_mimetype))
			COPY_STATIC_PART(1B, curr);
		else {
			COPY_STATIC_PART(1A, curr);
			memcpy(curr, mime->type, mime->type_len);
			curr += mime->type_len;
		}
	}

	if (req->keep_alive && (req->version == HTTP_1_0))
		COPY_STATIC_PART(2_keepalive, curr);
	else if (!req->keep_alive && (req->version == HTTP_1_1))
		COPY_STATIC_PART(2_close, curr);
	else
		COPY_STATIC_PART(2_none, curr);

	memcpy(curr, tux_date, DATE_LEN-1);
	curr += DATE_LEN-1;

	if (req->content_gzipped)
		COPY_STATIC_PART(3A, curr);

	if (tux_noid)
		COPY_STATIC_PART(3BX, curr);
	else
		COPY_STATIC_PART(3BY, curr);

	// "%d" req->filelen

	{
		unsigned int num = req->filelen;
		int nr_digits = 0;
		char digits [10];

		do {
			digits[nr_digits++] = '0' + num % 10;
			num /= 10;
		} while (num);

		while (nr_digits)
			*curr++ = digits[--nr_digits];
	}

	COPY_STATIC_PART(4, curr);

	size = curr-buf;

#if CONFIG_TUX_DEBUG
	*curr = 0;
	Dprintk("{%s} [%d/%d]\n", buf, size, strlen(buf));
#endif

	flags = MSG_DONTWAIT;
	if (!push)
		flags |= MSG_MORE;
	send_abuf(req, size, flags);
}

void http_illegal_request (tux_req_t *req, int cachemiss)
{
	clear_keepalive(req);
	if (req->error == 2)
		/*
		 * Just zap timed out connections
		 */
		add_req_to_workqueue(req);
	else {
		if (req->status == 403)
			send_async_err_forbidden(req);
		else
			send_async_err_not_found(req);
	}
}

tux_proto_t tux_proto_http = {
	defer_accept: 1,
	got_request: http_got_request,
	parse_message: parse_http_message,
	illegal_request: http_illegal_request,
};

