/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * ftp_proto.c: FTP application protocol support
 */

#define __KERNEL_SYSCALLS__
#include <net/tux.h>

#define HELLO		"220 Linux TUX FTP server welcomes you!\r\n"
#define WRITE_DONE	"226 Transfer complete.\r\n"
#define BAD_FILENAME	"550 No such file or directory.\r\n"
#define GOOD_DIR	"250 CWD command successful.\r\n"
#define WRITE_FILE	"150 Opening BINARY mode data connection.\r\n"
#define LIST_ERR	"503 LIST without PORT! Closing connection.\r\n"
#define WRITE_LIST	"150 Opening ASCII mode data connection.\r\n"
#define RETR_ERR	"503 RETR without PORT! Closing connection.\r\n"
#define PORT_OK		"200 PORT command successful.\r\n"
#define LOGIN_OK	"230-There are currently %d users logged in, out of %d maximum.\r\n230 TUX Guest login ok.\r\n"
#define LOGIN_FORBIDDEN	"530 Sorry, Login Denied!\r\n"
#define TYPE_OK		"200 Type set to I.\r\n"
#define BYE		"221 Thank You for using TUX!\r\n"
#define NOT_IMPLEMENTED	"502 Command not implemented.\r\n"
#define CLOSE_2		"221 Cannot handle request, closing connection!\r\n"
#define CLOSE		"500 Command not understood.\r\n"
#define CLOSE_TIMEOUT	"421 Timeout, closing connection!\r\n"
#define LINUX_SYST	"215 UNIX Type: L8, Linux 2.4 TUX/FTP 2.0 Server\r\n"
#define NO_EXTRA_FEATURES \
			"211 No Extra Features\r\n"
#define COMMAND_OK	"200 Command OK.\r\n"
#define WRITE_ABORTED	"426 Transfer aborted. Data connection closed.\r\n"
#define SITE		"214 No SITE commands are recognized.\r\n"

static void ftp_got_request (tux_req_t *req)
{
	add_tux_atom(req, parse_request);
	__send_async_message(req, HELLO, 220);
}

#define GOTO_ERR { TDprintk("FTP protocol error at: %s:%d\n", \
			__FILE__, __LINE__); goto error; }

static void zap_data_socket (tux_req_t *req)
{
	if (req->ftp_data_sock) {
		Dprintk("zapping req %p's data socket %p.\n",
			req, req->ftp_data_sock);
		unlink_tux_ftp_data_socket(req);
		sock_release(req->ftp_data_sock);
		req->ftp_data_sock = NULL;
	}
}
static void ftp_execute_command (tux_req_t *req, int cachemiss);

static int parse_ftp_message (tux_req_t *req, const int total_len)
{
	int comm, comm1 = 0, comm2 = 0, comm3 = 0, comm4 = 0;
	int newline_pos, i;
	char *mess, *curr;

	curr = mess = req->headers;

	Dprintk("FTP parser got %d bytes: --->{%s}<---\n", total_len, curr);

	newline_pos = -1;
	for (i = 0; i < total_len; i++, curr++) {
		if (!*curr)
			GOTO_ERR;
		if (!(*curr == '\r') || !(*(curr+1) == '\n'))
			continue;
		newline_pos = i;
		break;
	}
	Dprintk("Newline pos: %d\n", newline_pos);
	if (newline_pos == -1) {
		Dprintk("incomplete mess on req %p!\n", req);
		return 0;
	}
	if (newline_pos < 3)
		GOTO_ERR;

#define STRING_VAL(c1,c2,c3,c4) \
		((c1) + ((c2) << 8) + ((c3) << 16) + ((c4) << 24))
#define STRING_VAL_STR(str) \
		STRING_VAL(str[0], str[1], str[2], str[3])

	Dprintk("string val (%c%c%c%c): %08x\n",
		mess[0], mess[1], mess[2], mess[3],
		STRING_VAL_STR(mess));

#define PARSE_FTP_COMM(c1,c2,c3,c4,name,num)			\
	if (STRING_VAL_STR(mess) == STRING_VAL(c1,c2,c3,c4))	\
	{							\
		Dprintk("parsed "#name".\n");			\
		comm##num = FTP_COMM_##name;			\
	}

	PARSE_FTP_COMM('P','A','S','S', PASS,1);
	PARSE_FTP_COMM('A','C','C','T', ACCT,2);
	PARSE_FTP_COMM('C','D','U','P', CDUP,3);
	PARSE_FTP_COMM('S','M','N','T', SMNT,4);
	PARSE_FTP_COMM('Q','U','I','T', QUIT,1);
	PARSE_FTP_COMM('R','E','I','N', REIN,2);
	PARSE_FTP_COMM('P','A','S','V', PASV,3);
	PARSE_FTP_COMM('S','T','R','U', STRU,4); 
	PARSE_FTP_COMM('M','O','D','E', MODE,1); 
	PARSE_FTP_COMM('S','T','O','R', STOR,2); 
	PARSE_FTP_COMM('S','T','O','U', STOU,3); 
	PARSE_FTP_COMM('A','P','P','E', APPE,4); 
	PARSE_FTP_COMM('A','L','L','O', ALLO,1); 
	PARSE_FTP_COMM('R','N','F','R', RNFR,2); 
	PARSE_FTP_COMM('R','N','T','O', RNTO,3); 
	PARSE_FTP_COMM('A','B','O','R', ABOR,4); 
	PARSE_FTP_COMM('D','E','L','E', DELE,1); 
	PARSE_FTP_COMM('R','M','D',' ', RMD, 2); 
	PARSE_FTP_COMM('M','K','D',' ', MKD, 3); 
	PARSE_FTP_COMM('P','W','D',' ', PWD, 4); 
	PARSE_FTP_COMM('S','Y','S','T', SYST,2); 
	PARSE_FTP_COMM('N','O','O','P', NOOP,3); 
	PARSE_FTP_COMM('F','E','A','T', FEAT,4); 

	comm = comm1 | comm2 | comm3 | comm4;

	if (comm) {
		if (newline_pos != 4)
			GOTO_ERR;
		req->ftp_command = comm;
		goto out;
	}
	
	switch (STRING_VAL(mess[0], mess[1], mess[2], mess[3])) {

#define PARSE_FTP_COMM_3CHAR(c1,c2,c3,name)				\
		case STRING_VAL(c1,c2,c3,'\r'):				\
		{							\
			Dprintk("parsed "#name".\n");			\
			req->ftp_command = FTP_COMM_##name;		\
			if (newline_pos != 3)				\
				GOTO_ERR;				\
		}

#define PARSE_FTP_3CHAR_COMM_IGNORE(c1,c2,c3,name)			\
		case STRING_VAL(c1,c2,c3,' '):				\
		{							\
			Dprintk("parsed "#name".\n");			\
			req->ftp_command = FTP_COMM_##name;		\
		}

#define PARSE_FTP_COMM_IGNORE(c1,c2,c3,c4,name)				\
		case STRING_VAL(c1,c2,c3,c4):				\
		{							\
			Dprintk("parsed "#name".\n");			\
			req->ftp_command = FTP_COMM_##name;		\
		}

#define PARSE_FTP_3CHAR_COMM_1_FIELD(c1,c2,c3,name,field,field_len,max)	\
		case STRING_VAL(c1,c2,c3,' '):				\
		{							\
			Dprintk("parsed "#name".\n");			\
			req->ftp_command = FTP_COMM_##name;		\
			if (newline_pos == 4)				\
				GOTO_ERR;				\
			if (newline_pos >= 5) {				\
				curr = mess + 3;			\
				if (*curr++ != ' ')			\
					GOTO_ERR;			\
				*(field_len) = newline_pos-4;		\
				if (*(field_len) >= max)		\
					GOTO_ERR;			\
				memcpy(field, curr, *(field_len));	\
				(field)[*(field_len)] = 0;		\
			}						\
		}

#define PARSE_FTP_COMM_1_FIELD(c1,c2,c3,c4,name,field,field_len,max)	\
		case STRING_VAL(c1,c2,c3,c4):				\
		{							\
			Dprintk("parsed "#name".\n");			\
			req->ftp_command = FTP_COMM_##name;		\
			if (newline_pos == 5)				\
				GOTO_ERR;				\
			if (newline_pos >= 6) {				\
				curr = mess + 4;			\
				if (*curr++ != ' ')			\
					GOTO_ERR;			\
				*(field_len) = newline_pos-5;		\
				if (*(field_len) >= max)		\
					GOTO_ERR;			\
				memcpy(field, curr, *(field_len));	\
				(field)[*(field_len)] = 0;		\
			}						\
		}
		PARSE_FTP_COMM_1_FIELD('U','S','E','R', USER,
			req->username, &req->username_len,
			MAX_USERNAME_LEN-1);
		if (!req->username_len)
			GOTO_ERR;
		break;

		PARSE_FTP_3CHAR_COMM_1_FIELD('C','W','D', CWD,
			req->objectname, &req->objectname_len,
			MAX_OBJECTNAME_LEN-1);
		if (!req->objectname_len)
			GOTO_ERR;
		break;

		PARSE_FTP_COMM_3CHAR('P','W','D', PWD); break;

		{
			char type[3];
			unsigned int type_len;

			PARSE_FTP_COMM_1_FIELD('T','Y','P','E', TYPE,
				type, &type_len, 2);
			if (!type_len)
				GOTO_ERR;
			if ((type[0] != 'I') && (type[0] != 'A'))
				GOTO_ERR;
		}
		break;

		PARSE_FTP_COMM_1_FIELD('R','E','T','R', RETR,
			req->objectname, &req->objectname_len,
			MAX_OBJECTNAME_LEN-1);
		if (!req->objectname_len)
			GOTO_ERR;
		break;

		PARSE_FTP_COMM_IGNORE('S','T','A','T', STAT);
		break;

		PARSE_FTP_COMM_IGNORE('S','I','T','E', SITE);
		break;

		PARSE_FTP_COMM_IGNORE('L','I','S','T', LIST);
		break;

		PARSE_FTP_COMM_IGNORE('N','L','S','T', NLST);
		break;

		PARSE_FTP_COMM_IGNORE('H','E','L','P', HELP);
		break;

		PARSE_FTP_COMM_IGNORE('C','L','N','T', CLNT);
		break;

#define IS_NUM(n) (((n) >= '0') && ((n) <= '9'))

#define GET_DIGIT(curr,n)				\
	n += (*curr) - '0';				\
	curr++;						\
	if (IS_NUM(*curr)) {				\
		n *= 10;

#define PARSE_PORTNUM(curr,n)				\
do {							\
	Dprintk("PORT NUM parser:--->{%s}<---\n", curr);\
	if (!IS_NUM(*curr))				\
		GOTO_ERR;				\
	n = 0;						\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	}}}						\
	if (n > 255)					\
		GOTO_ERR;				\
	Dprintk("PORT NUM parser:--->{%s}<---\n", curr);\
	Dprintk("PORT NUM parser parsed %d.\n", n);	\
} while (0)

#define PARSE_NUM(curr,n)				\
do {							\
	Dprintk("NUM parser:--->{%s}<---\n", curr);	\
	if (!IS_NUM(*curr))				\
		GOTO_ERR;				\
	n = 0;						\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	GET_DIGIT(curr,n);				\
	}}}}}}}}}}					\
	Dprintk("NUM parser:--->{%s}<---\n", curr);	\
	Dprintk("NUM parser parsed %d.\n", n);		\
} while (0)

		case STRING_VAL('P','O','R','T'):
		{
			unsigned int h1, h2, h3, h4, p1, p2;
			if (req->ftp_data_sock)
				zap_data_socket(req);
			/*
			 * Minimum size: "PORT 0,0,0,0,0,0", 16 bytes.
			 */
			if (newline_pos < 16)
				GOTO_ERR;
			Dprintk("parsed PORT.\n");
			if (req->ftp_data_sock)
				GOTO_ERR;
			curr = mess + 4;
			if (*curr++ != ' ')
				GOTO_ERR;
			PARSE_PORTNUM(curr,h1);
			if (*curr++ != ',')
				GOTO_ERR;
			PARSE_PORTNUM(curr,h2);
			if (*curr++ != ',')
				GOTO_ERR;
			PARSE_PORTNUM(curr,h3);
			if (*curr++ != ',')
				GOTO_ERR;
			PARSE_PORTNUM(curr,h4);
			if (*curr++ != ',')
				GOTO_ERR;
			PARSE_PORTNUM(curr,p1);
			if (*curr++ != ',')
				GOTO_ERR;
			PARSE_PORTNUM(curr,p2);
			if (curr-mess != newline_pos)
				GOTO_ERR;
			req->ftp_command = FTP_COMM_PORT;
			req->ftp_user_addr = (h1<<24) + (h2<<16) + (h3<<8) + h4;
			req->ftp_user_port = (p1<<8) + p2;
			Dprintk("FTP PORT got: %d.%d.%d.%d:%d.\n",
				h1, h2, h3, h4, req->ftp_user_port);
			Dprintk("FTP user-addr: %08x (htonl: %08x), socket: %08x.\n",
				req->ftp_user_addr, htonl(req->ftp_user_addr),
				req->sock->sk->daddr);
			/*
			 * Do not allow arbitrary redirection
			 * of connections ...
			 */
			if (req->sock->sk->daddr != htonl(req->ftp_user_addr))
				GOTO_ERR;

			break;
		}
		case STRING_VAL('R','E','S','T'):
		{
			unsigned int offset;

			/*
			 * Minimum size: "REST 0", 6 bytes.
			 */
			if (newline_pos < 6)
				GOTO_ERR;
			Dprintk("parsed REST.\n");
			curr = mess + 4;
			if (*curr++ != ' ')
				GOTO_ERR;
			PARSE_NUM(curr,offset);
			if (curr-mess != newline_pos)
				GOTO_ERR;
			req->ftp_command = FTP_COMM_REST;
			req->ftp_offset = offset;
			Dprintk("FTP REST got: %d bytes offset.\n", offset);

			break;
		}
		default:
			req->ftp_command = FTP_COMM_NONE;
			break;
	}

out:
	req->parsed_len = newline_pos + 2;
	add_tux_atom(req, ftp_execute_command);

	req->lookup_dir = 1; // disable index.html auto-transfer

	return req->parsed_len;
error:
	clear_keepalive(req);
	TDprintk("rejecting FTP session!\n");
	TDprintk("mess     :--->{%s}<---\n", mess);
	TDprintk("mess left:--->{%s}<---\n", curr);
	req_err(req);
	return -1;
}

#define data_sock_err(req) \
	((req)->ftp_data_sock && (req)->ftp_data_sock->sk && \
		((req)->ftp_data_sock->sk->state >= TCP_FIN_WAIT1))

void ftp_send_file (tux_req_t *req, int cachemiss)
{
	int ret;


	SET_TIMESTAMP(req->output_timestamp);

repeat:
	if (req->error || data_sock_err(req)) {
#if CONFIG_TUX_DEBUG
		req->bytes_expected = 0;
#endif
		req->in_file.f_pos = 0;
		TDprintk("zapping, data sock state: %d\n",
			req->ftp_data_sock->sk->state);
		/*
		 * We are in the middle of a file transfer,
		 * zap it immediately:
		 */
		req->error = 3;
		zap_request(req, cachemiss);
		return;
	}
	if (req->sock->sk->tp_pinfo.af_tcp.urg_data) {
		req->in_file.f_pos = 0;
		zap_data_socket(req);
		__send_async_message(req, WRITE_ABORTED, 426);
		return;
	}

	ret = generic_send_file(req, 1, !cachemiss, req->ftp_data_sock);
	switch (ret) {
		case -5:
			add_tux_atom(req, ftp_send_file);
			output_timeout(req);
			break;
		case -4:
			add_tux_atom(req, ftp_send_file);
			if (add_output_space_event(req, req->ftp_data_sock)) {
				del_tux_atom(req);
				goto repeat;
			}
			break;
		case -3:
			add_tux_atom(req, ftp_send_file);
			queue_cachemiss(req);
			break;
		default:
			req->in_file.f_pos = 0;
			Dprintk("FTP send file req %p finished!\n", req);
			zap_data_socket(req);
			__send_async_message(req, WRITE_DONE, 200);
			break;
	}
}

void ftp_create_host (tux_req_t *req)
{
#define IP(n) ((unsigned char *)& req->sock->sk->rcv_saddr)[n]

	req->host_len = sprintf(req->host, "%d.%d.%d.%d",
			 			IP(0), IP(1), IP(2), IP(3));
	Dprintk("created FTP virtual hostname: {%s}\n", req->host);

#undef IP
}

void ftp_get_file (tux_req_t *req, int cachemiss)
{
	int missed;

	if (!req->dentry) {
		if (virtual_server)
			ftp_create_host(req);
		missed = lookup_url(req, cachemiss ? 0 : LOOKUP_ATOMIC);
		if ((!missed && !req->dentry) || (missed == 2)) {
//			zap_data_socket(req);
			__send_async_message(req, BAD_FILENAME, 200);
			return;
		}
		if (missed) {
			if (cachemiss)
				TUX_BUG();
			add_tux_atom(req, ftp_get_file);
			queue_cachemiss(req);
			return;
		}
	}
	req->in_file.f_pos = 0;
	if (req->ftp_offset <= req->filelen)
		req->in_file.f_pos = req->ftp_offset;
	req->ftp_offset = 0;
	Dprintk("ftp_send_file %p, f_pos: %d.\n", req, (int)req->in_file.f_pos);
	add_tux_atom(req, ftp_send_file);
	__send_async_message(req, WRITE_FILE, 200);
}

static void ftp_get_listfile (tux_req_t *req, int cachemiss)
{
	int missed;

	if (!req->dentry) {
		if (virtual_server)
			ftp_create_host(req);
		missed = lookup_url(req, cachemiss ? 0 : LOOKUP_ATOMIC);
		Dprintk("get_listfile(): missed: %d, dentry: %p, error: %d.\n",
			missed, req->dentry, req->error);
		if ((!missed && !req->dentry) || (missed == 2)) {
//			zap_data_socket(req);
			req->error = 0;
			__send_async_message(req, BAD_FILENAME, 200);
			return;
		}
		if (missed) {
			if (cachemiss)
				TUX_BUG();
			add_tux_atom(req, ftp_get_listfile);
			queue_cachemiss(req);
			return;
		}
	}
	req->ftp_offset = 0;
	req->in_file.f_pos = 0;
	add_tux_atom(req, ftp_send_file);
	__send_async_message(req, WRITE_LIST, 200);
}

static void ftp_chdir (tux_req_t *req, int cachemiss)
{
	unsigned int lookupflag = cachemiss ? 0 : LOOKUP_ATOMIC;
	struct nameidata nd;
	int err;

	Dprintk("ftp_chdir(%p, %d, {%s})\n", req, cachemiss, req->objectname);
	lookupflag |= LOOKUP_POSITIVE|LOOKUP_FOLLOW|LOOKUP_DIRECTORY;
	nd.flags = lookupflag;
	nd.last_type = LAST_ROOT;
	if ((req->objectname[0] == '/') && req->cwd) {
		dput(req->cwd);
		mntput(req->cwdmnt);
		req->cwd = NULL;
		req->cwdmnt = NULL;
	}
	if (!req->cwd) {
		req->cwd = dget(docroot.dentry);
		req->cwdmnt = mntget(docroot.mnt);
	}
	nd.dentry = req->cwd;
	nd.mnt = req->cwdmnt;
	dget(nd.dentry);
	mntget(nd.mnt);

	err = path_walk(req->objectname, &nd);
	if (err == -EWOULDBLOCKIO) {
		if (cachemiss)
			TUX_BUG();
		add_tux_atom(req, ftp_chdir);
		queue_cachemiss(req);
		return;
	}
	if (err) {
		mntput(nd.mnt);
		goto out_err;
	}
	err = permission(nd.dentry->d_inode, MAY_EXEC);
	if (err)
		goto out_err_put;

	req->cwd = nd.dentry;
	req->cwdmnt = nd.mnt;
	
	__send_async_message(req, GOOD_DIR, 200);
	return;

out_err_put:
	path_release(&nd);
out_err:
	__send_async_message(req, BAD_FILENAME, 550);
}

void ftp_accept_pasv (tux_req_t *req, int cachemiss)
{
	struct socket *sock, *new_sock = NULL;
	struct tcp_opt *tp1, *tp2;
	int err;

	tp1 = &req->ftp_data_sock->sk->tp_pinfo.af_tcp;

	Dprintk("PASV accept on req %p, accept_queue: %p.\n",
			req, tp1->accept_queue);
	if (req->error || (req->ftp_data_sock->sk->state != TCP_LISTEN))
		goto error;
new_socket:
	if (!tp1->accept_queue) {
		spin_lock_irq(&req->ti->work_lock);
		add_keepalive_timer(req);
		if (test_and_set_bit(0, &req->idle_input))
			TUX_BUG();
		spin_unlock_irq(&req->ti->work_lock);
		if (!tp1->accept_queue) {
			add_tux_atom(req, ftp_accept_pasv);
			return;
		}
		unidle_req(req);
	}
	new_sock = sock_alloc();
	if (!new_sock)
		goto error;
	sock = req->ftp_data_sock;
	new_sock->type = sock->type;
	new_sock->ops = sock->ops;

	err = sock->ops->accept(sock, new_sock, O_NONBLOCK);
	Dprintk("PASV accept() returned %d.\n", err);
	if (err < 0)
		goto error;
	if (new_sock->sk->state != TCP_ESTABLISHED)
		goto error;
	/*
	 * Do not allow other clients to steal the FTP connection!
	 */
	if (new_sock->sk->daddr != req->sock->sk->daddr) {
		Dprintk("PASV: ugh, unauthorized connect?\n");
		unlink_tux_listen_socket(req);
		sock_release(new_sock);
		new_sock = NULL;
		goto new_socket;
	}
	/*
	 * Zap the listen socket:
	 */
	unlink_tux_listen_socket(req);
	sock_release(sock);
	req->ftp_data_sock = NULL;

	tp2 = &new_sock->sk->tp_pinfo.af_tcp;
	tp2->nonagle = tux_nonagle;
	tp2->ack.pingpong = tux_ack_pingpong;
	new_sock->sk->reuse = 1;
	new_sock->sk->urginline = 1;

	link_tux_ftp_accept_socket(req, new_sock);

	add_req_to_workqueue(req);
	return;

error:
	if (new_sock)
		sock_release(new_sock);
	req_err(req);
	zap_data_socket(req);
	__send_async_message(req, CLOSE, 500);
}

static void ftp_execute_command (tux_req_t *req, int cachemiss)
{
	if (!req->parsed_len)
		TUX_BUG();
	trunc_headers(req);
	req->keep_alive = 1;

	switch (req->ftp_command) {

#define ABORTED \
	"226 Abort successful.\r\n"

	case FTP_COMM_ABOR:
	{
		zap_data_socket(req);
		__send_async_message(req, ABORTED, 226);
		break;
	}

	case FTP_COMM_PWD:
	{
		struct dentry *cwd, *root;
		struct vfsmount *cwdmnt, *rootmnt;
		char *buf, *path;

		if (!req->cwd) {
			req->cwd = dget(docroot.dentry);
			req->cwdmnt = mntget(docroot.mnt);
		}
		buf = (char *)__get_free_page(GFP_KERNEL);

// "257 "/" is current directory.\r\n"

#define PART_1 "257 \""
#define PART_1_LEN (sizeof(PART_1)-1)

#define PART_3 "\" is current directory.\r\n"
#define PART_3_LEN sizeof(PART_3)

		cwd = dget(req->cwd);
		cwdmnt = mntget(req->cwdmnt);
		root = dget(docroot.dentry);
		rootmnt = mntget(docroot.mnt);

		spin_lock(&dcache_lock);
		path = __d_path(cwd, cwdmnt, root, rootmnt,
			buf+PART_1_LEN, PAGE_SIZE - PART_3_LEN - PART_1_LEN);
		spin_unlock(&dcache_lock);

		dput(cwd);
		mntput(cwdmnt);
		dput(root);
		mntput(rootmnt);

		if (path < buf + PART_1_LEN)
			BUG();

		memcpy(path - PART_1_LEN, PART_1, PART_1_LEN);
		memcpy(buf + PAGE_SIZE-PART_3_LEN-1, PART_3, PART_3_LEN);

		__send_async_message(req, path - PART_1_LEN, 226);
		free_page((unsigned long)buf);
		break;
	}

	case FTP_COMM_CDUP:
	{
		memcpy(req->objectname, "..", 3);
		req->objectname_len = 2;

		// fall through to CWD:
	}
	case FTP_COMM_CWD:
	{
		ftp_chdir(req, cachemiss);
		break;
	}

	case FTP_COMM_NLST:
	{
		#define NLST_FILENAME ".TUX-NLST"
		#define NLST_FILENAME_LEN sizeof(NLST_FILENAME)

		memcpy(req->objectname, NLST_FILENAME, NLST_FILENAME_LEN);
		req->objectname_len = NLST_FILENAME_LEN-1;
		goto get_listfile;
	}

	case FTP_COMM_LIST:
	{
		#define LIST_FILENAME ".TUX-LIST"
		#define LIST_FILENAME_LEN sizeof(LIST_FILENAME)

		memcpy(req->objectname, LIST_FILENAME, LIST_FILENAME_LEN);
		req->objectname_len = LIST_FILENAME_LEN-1;

get_listfile:
		if (!req->ftp_data_sock) {
			req_err(req);
			__send_async_message(req, LIST_ERR, 200);
			GOTO_ERR;
		}
		add_tux_atom(req, ftp_get_listfile);
		add_req_to_workqueue(req);
		break;
	}

	case FTP_COMM_RETR:
	{
		if (!req->ftp_data_sock) {
			req_err(req);
			__send_async_message(req, RETR_ERR, 200);
			GOTO_ERR;
		}
		ftp_get_file(req, cachemiss);
		break;
	}

	case FTP_COMM_PASV:
	{
		char buf [36 + 4*3 + 5 + 10];
		struct socket *data_sock;
		struct sockaddr_in addr;
		struct tcp_opt *tp;
		u32 local_addr;
		int err;

		if (req->ftp_data_sock)
			zap_data_socket(req);
		/*
		 * Create FTP data connection to client:
		 */
		err = sock_create(AF_INET, SOCK_STREAM, IPPROTO_IP, &data_sock);
		if (err < 0) {
			Dprintk("sock create err: %d\n", err);
			req_err(req);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}
			
#define IP(n) ((unsigned char *)&req->sock->sk->daddr)[n]

		local_addr = req->sock->sk->rcv_saddr;
		addr.sin_family = AF_INET;
		addr.sin_port = 0;
		addr.sin_addr.s_addr = local_addr;
		Dprintk("client address: (%d,%d,%d,%d).\n",
				IP(0), IP(1), IP(2), IP(3));
#undef IP

		data_sock->sk->reuse = 1;

		err = data_sock->ops->bind(data_sock,
				(struct sockaddr*)&addr, sizeof(addr));
	       	Dprintk("PASV bind() ret: %d.\n", err);
		if (err < 0) {
			req_err(req);
			sock_release(data_sock);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}

		data_sock->sk->linger = 0;
		data_sock->sk->urginline = 1;

		tp = &data_sock->sk->tp_pinfo.af_tcp;
		tp->ack.pingpong = tux_ack_pingpong;
		if (!tux_keepalive_timeout)
			tp->linger2 = 0;
		else
			tp->linger2 = tux_keepalive_timeout * HZ;

		err = data_sock->ops->listen(data_sock, 1);
		Dprintk("PASV listen() ret: %d\n", err);
		if (err) {
			req_err(req);
			sock_release(data_sock);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}
		link_tux_ftp_data_socket(req, data_sock);

		Dprintk("FTP PASV listen sock state: %d, sk state: %d\n",
			data_sock->state, data_sock->sk->state);

#define IP(n) ((unsigned char *)&local_addr)[n]
		sprintf(buf,
			"227 Entering Passive Mode (%d,%d,%d,%d,%d,%d).\r\n",
				IP(0), IP(1), IP(2), IP(3),
				ntohs(data_sock->sk->sport) / 256,
				ntohs(data_sock->sk->sport) & 255 );
#undef IP
		Dprintk("PASV mess: {%s}\n", buf);

		add_tux_atom(req, ftp_accept_pasv);
		__send_async_message(req, buf, 227);
		break;
	}

	case FTP_COMM_PORT:
	{
		struct socket *data_sock;
		struct sockaddr_in addr;
		kernel_cap_t saved_cap;
		u32 local_addr;
		int err;

		/*
		 * Create FTP data connection to client:
		 */
		err = sock_create(AF_INET, SOCK_STREAM, IPPROTO_IP, &data_sock);
		if (err < 0) {
			Dprintk("sock create err: %d\n", err);
			req_err(req);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}

		local_addr = req->sock->sk->rcv_saddr;
		addr.sin_family = AF_INET;
		addr.sin_port = htons(20);
		addr.sin_addr.s_addr = local_addr;

#define IP(n) ((unsigned char *)&local_addr)[n]
		Dprintk("data socket address: (%d,%d,%d,%d).\n",
				IP(0), IP(1), IP(2), IP(3));
#undef IP
		data_sock->sk->reuse = 1;

		saved_cap = current->cap_effective;
		cap_raise (current->cap_effective, CAP_NET_BIND_SERVICE);
		err = data_sock->ops->bind(data_sock,
				(struct sockaddr*)&addr, sizeof(addr));
		current->cap_effective = saved_cap;

	       	Dprintk("ACTIVE bind() ret: %d.\n", err);
		if (err) {
			sock_release(data_sock);
			req_err(req);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}

		link_tux_ftp_data_socket(req, data_sock);

		addr.sin_family = AF_INET;
		addr.sin_port = htons(req->ftp_user_port);
		addr.sin_addr.s_addr = htonl(req->ftp_user_addr);

		err = data_sock->ops->connect(data_sock, (struct sockaddr *) &addr, sizeof(addr), O_RDWR|O_NONBLOCK);
		if (err && (err != -EINPROGRESS)) {
			Dprintk("connect error: %d\n", err);
			zap_data_socket(req);
			req_err(req);
			__send_async_message(req, CLOSE, 500);
			GOTO_ERR;
		}
		Dprintk("FTP data sock state: %d, sk state: %d\n", data_sock->state, data_sock->sk->state);
		__send_async_message(req, PORT_OK, 200);
		break;
	}

	case FTP_COMM_USER:
	{
		if (!strcmp(req->username, "ftp")
			 || !strcmp(req->username, "FTP")
			 || !strcmp(req->username, "anonymous")
			 || !strcmp(req->username, "ANONYMOUS")
			 || !strcmp(req->username, "guest")
			 || !strcmp(req->username, "GUEST")) {
			char login_ok [200];

			sprintf(login_ok, LOGIN_OK, nr_requests_used(), tux_max_connect);
			__send_async_message(req, login_ok, 200);
		} else {
			clear_keepalive(req);
			__send_async_message(req, LOGIN_FORBIDDEN, 530);
		}
		break;
	}
	case FTP_COMM_SITE:
	{
		__send_async_message(req, SITE, 214);
		break;
	}
	case FTP_COMM_SYST:
	{
		__send_async_message(req, LINUX_SYST, 200);
		break;
	}
	case FTP_COMM_TYPE:
	{
		__send_async_message(req, TYPE_OK, 200);
		break;
	}
	case FTP_COMM_FEAT:
	{
		__send_async_message(req, NO_EXTRA_FEATURES, 211);
		break;
	}
	case FTP_COMM_HELP:
	case FTP_COMM_CLNT:
	case FTP_COMM_NOOP:
	{
		__send_async_message(req, COMMAND_OK, 200);
		break;
	}
	case FTP_COMM_REST:
	{
		__send_async_message(req, COMMAND_OK, 200);
		break;
	}
	case FTP_COMM_QUIT:
	{
		clear_keepalive(req);
		__send_async_message(req, BYE, 200);
		break;
	}

	default:
	{
		req->keep_alive = 1;
		__send_async_message(req, CLOSE, 500);
		break;
	}
	}
	return;
error:
	Dprintk("rejecting FTP session!\n");
	return;
}


static void ftp_close (tux_req_t *req, int cachemiss)
{
	if (req->error == 2)
		__send_async_message(req, CLOSE_TIMEOUT, 421);
	else
		__send_async_message(req, CLOSE, 500);
}

tux_proto_t tux_proto_ftp = {
	defer_accept: 0,
	got_request: ftp_got_request,
	parse_message: parse_ftp_message,
	illegal_request: ftp_close,
};

