/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * proc.c: /proc/sys/tux handling
 */

#include <net/tux.h>

char tux_docroot[200] = "/var/www/tux/";
char tux_logfile[200] = "/var/log/tux";
char tux_cgiroot[200] = "/var/www/tux/cgiroot/";
char tux_404_page[200] = "404.html";

int tux_cgi_uid = -1;
int tux_cgi_gid = -1;
int tux_clientport = 8080;
int tux_logging = 0;
int tux_serverport= 80;
int tux_threads = 2;
int tux_max_connect = 10000;
int tux_max_keepalives = 10000;
int tux_max_backlog = 2048;
int tux_keepalive_timeout = 0;
int tux_max_output_bandwidth = 0;
int tux_defer_accept = 1;
int tux_mode_forbidden = 0 /*S_IXUGO*/; /* do not allow executable (CGI) files */
int tux_mode_allowed = S_IROTH; /* allow access if read-other is set */
int multifragment_api = 1;
int pagecache_special = 0;
int virtual_server = 0;
int tux_max_object_size = 0;
unsigned int log_cpu_mask = ~0;
int tux_compression = 0;
int tux_noid = 0;
int tux_cgi_inherit_cpu = 0;
unsigned int tux_cgi_cpu_mask = ~0;
int tux_zerocopy_header = 1;
unsigned int tux_max_free_requests = 1000;
int tux_all_userspace = 0;
int tux_redirect_logging = 1;
unsigned int tux_max_header_len = 3000;
int tux_application_protocol = 0;
int tux_referer_logging = 0;

static struct ctl_table_header *tux_table_header;

static ctl_table tux_table[] = {
	{	NET_TUX_DOCROOT,
		"documentroot",
		&tux_docroot,
		sizeof(tux_docroot),
		0644,
		NULL,
		proc_dostring,
		&sysctl_string,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_LOGFILE,
		"logfile",
		&tux_logfile,
		sizeof(tux_logfile),
		0644,
		NULL,
		proc_dostring,
		&sysctl_string,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_THREADS,
		"threads",
		&tux_threads,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_KEEPALIVE_TIMEOUT,
		"keepalive_timeout",
		&tux_keepalive_timeout,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_KEEPALIVE_BW,
		"max_output_bandwidth",
		&tux_max_output_bandwidth,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_DEFER_ACCEPT,
		"defer_accept",
		&tux_defer_accept,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_BACKLOG,
		"max_backlog",
		&tux_max_backlog,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_CONNECT,
		"max_connect",
		&tux_max_connect,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_KEEPALIVES,
		"max_keepalives",
		&tux_max_keepalives,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MODE_FORBIDDEN,
		"mode_forbidden",
		&tux_mode_forbidden,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MODE_ALLOWED,
		"mode_allowed",
		&tux_mode_allowed,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CGI_UID,
		"cgi_uid",
		&tux_cgi_uid,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CGI_GID,
		"cgi_gid",
		&tux_cgi_gid,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CGIROOT,
		"cgiroot",
		&tux_cgiroot,
		sizeof(tux_cgiroot),
		0644,
		NULL,
		proc_dostring,
		&sysctl_string,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_404_PAGE,
		"404_page",
		&tux_404_page,
		sizeof(tux_404_page),
		0644,
		NULL,
		proc_dostring,
		&sysctl_string,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CLIENTPORT,
		"clientport",
		&tux_clientport,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
#if CONFIG_TUX_DEBUG
	{	NET_TUX_LOGGING,
		"Dprintk",
		&tux_Dprintk,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
#endif
	{	NET_TUX_LOGGING,
		"logging",
		&tux_logging,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_SERVERPORT,
		"serverport",
		&tux_serverport,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_LOGENTRY_ALIGN_ORDER,
		"logentry_align_order",
		&tux_logentry_align_order,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_NONAGLE,
		"nonagle",
		&tux_nonagle,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_ACK_PINGPONG,
		"ack_pingpong",
		&tux_ack_pingpong,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_PUSH_ALL,
		"push_all",
		&tux_push_all,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_ZEROCOPY_PARSE,
		"zerocopy_parse",
		&tux_zerocopy_parse,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_VIRTUAL_SERVER,
		"virtual_server",
		&virtual_server,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_OBJECT_SIZE,
		"max_object_size",
		&tux_max_object_size,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_COMPRESSION,
		"compression",
		&tux_compression,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_NOID,
		"noid",
		&tux_noid,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CGI_INHERIT_CPU,
		"cgi_inherit_cpu",
		&tux_cgi_inherit_cpu,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_CGI_CPU_MASK,
		"cgi_cpu_mask",
		&tux_cgi_cpu_mask,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_ZEROCOPY_HEADER,
		"zerocopy_header",
		&tux_zerocopy_header,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_ZEROCOPY_SENDFILE,
		"zerocopy_sendfile",
		&tux_zerocopy_sendfile,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_FREE_REQUESTS,
		"max_free_requests",
		&tux_max_free_requests,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_ALL_USERSPACE,
		"all_userspace",
		&tux_all_userspace,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_REDIRECT_LOGGING,
		"redirect_logging",
		&tux_redirect_logging,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_REFERER_LOGGING,
		"referer_logging",
		&tux_referer_logging,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_MAX_HEADER_LEN,
		"max_header_len",
		&tux_max_header_len,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{	NET_TUX_APPLICATION_PROTOCOL,
		"application_protocol",
		&tux_application_protocol,
		sizeof(int),
		0644,
		NULL,
		proc_dointvec,
		&sysctl_intvec,
		NULL,
		NULL,
		NULL
	},
	{0,0,0,0,0,0,0,0,0,0,0}	};
	
	
static ctl_table tux_dir_table[] = {
	{NET_TUX, "tux", NULL, 0, 0555, tux_table,0,0,0,0,0},
	{0,0,0,0,0,0,0,0,0,0,0}
};

static ctl_table tux_root_table[] = {
	{CTL_NET, "net", NULL, 0, 0555, tux_dir_table,0,0,0,0,0},
	{0,0,0,0,0,0,0,0,0,0,0}
};


static struct proc_dir_entry * root_tux_dir;
static struct proc_dir_entry * log_cpu_mask_entry;
static struct proc_dir_entry * tux_dir [CONFIG_TUX_NUMTHREADS];
static struct proc_dir_entry * listen_dir [CONFIG_TUX_NUMTHREADS];
static struct proc_dir_entry * listen_entries [CONFIG_TUX_NUMTHREADS][CONFIG_TUX_NUMSOCKETS];

unsigned int tux_listen [CONFIG_TUX_NUMTHREADS][CONFIG_TUX_NUMSOCKETS] =
 { [0 ... CONFIG_TUX_NUMTHREADS-1] = { 0, [1 ... CONFIG_TUX_NUMSOCKETS-1] = -1 } };

#define HEX_DIGITS 8

static int hex_read_proc (char *page, char **start, off_t off,
			int count, int *eof, void *data)
{
	if (count < HEX_DIGITS+1)
		return -EINVAL;
	return sprintf (page, "%08x\n", *(unsigned int *)data);
}

static int hex_write_proc (struct file *file, const char *buffer,
					unsigned long count, void *data)
{
	unsigned char hexnum [HEX_DIGITS];
	unsigned int new_value;
	int i, full_count = count;

	if (!count)
		return -EINVAL;
	if (count > HEX_DIGITS)
		count = HEX_DIGITS;
	if (copy_from_user(hexnum, buffer, count))
		return -EFAULT;

	/*
	 * Parse the first 8 characters as a hex string, any non-hex char
	 * is end-of-string. '00e1', 'e1', '00E1', 'E1' are the same.
	 */
	new_value = 0;

	for (i = 0; i < count; i++) {
		unsigned int c = hexnum[i];

		switch (c) {
			case '0' ... '9': c -= '0'; break;
			case 'a' ... 'f': c -= 'a'-10; break;
			case 'A' ... 'F': c -= 'A'-10; break;
		default:
			goto out;
		}
		new_value = (new_value << 4) | c;
	}
out:
	*(int *)data = new_value;

	return full_count;
}

#define MAX_NAMELEN 10

static void register_tux_proc (unsigned int nr)
{
	struct proc_dir_entry *entry;
	char name [MAX_NAMELEN];
	int i;

	if (!root_tux_dir)
		TUX_BUG();

	sprintf(name, "%d", nr);

	/* create /proc/net/tux/1234/ */
	tux_dir[nr] = proc_mkdir(name, root_tux_dir);

	/* create /proc/net/tux/1234/listen/ */
	listen_dir[nr] = proc_mkdir("listen", tux_dir[nr]);

	/* create /proc/net/tux/1234/listen/ */
	for (i = 0; i < CONFIG_TUX_NUMSOCKETS; i++) {
		sprintf(name, "%d", i);
		entry = create_proc_entry(name, 0700, listen_dir[nr]);

		entry->nlink = 1;
		entry->data = (void *)&tux_listen[nr][i];
		entry->read_proc = hex_read_proc;
		entry->write_proc = hex_write_proc;

		listen_entries[nr][i] = entry;
	}
}

static void unregister_tux_proc (unsigned int nr)
{
	int i;

	for (i = 0; i < CONFIG_TUX_NUMSOCKETS; i++)
		remove_proc_entry(listen_entries[nr][i]->name, listen_dir[nr]);

	remove_proc_entry(listen_dir[nr]->name, tux_dir[nr]);

	remove_proc_entry(tux_dir[nr]->name, root_tux_dir);
}

static void cleanup_tux_proc (void)
{
	int i;

	Dprintk("cleaning up /proc/net/tux/\n");

	for (i = 0; i < CONFIG_TUX_NUMTHREADS; i++)
		unregister_tux_proc(i);
	remove_proc_entry(log_cpu_mask_entry->name, root_tux_dir);
	remove_proc_entry(root_tux_dir->name, proc_net);
}

static void init_tux_proc (void)
{
	struct proc_dir_entry *entry;
	int i;

	if (root_tux_dir)
		return;

	/* create /proc/net/tux */
	root_tux_dir = proc_mkdir("tux", proc_net);

	entry = create_proc_entry("log_cpu_mask", 0700, root_tux_dir);

	entry->nlink = 1;
	entry->data = (void *)&log_cpu_mask;
	entry->read_proc = hex_read_proc;
	entry->write_proc = hex_write_proc;

	log_cpu_mask_entry = entry;

	/*
	 * Create entries for all existing threads.
	 */
	for (i = 0; i < CONFIG_TUX_NUMTHREADS; i++)
		register_tux_proc(i);
}

void start_sysctl(void)
{
	init_tux_proc();
	tux_table_header = register_sysctl_table(tux_root_table,1);
}

void end_sysctl(void)
{
	cleanup_tux_proc();
	unregister_sysctl_table(tux_table_header);
}
