/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * output.c: Send data to clients
 */

#include <net/tux.h>

int send_sync_buf (tux_req_t *req, struct socket *sock, const char *buf, const size_t length, unsigned long flags)
{
	struct msghdr msg;
	struct iovec iov;
	int len, written = 0, left = length;

	msg.msg_name     = 0;
	msg.msg_namelen  = 0;
	msg.msg_iov	 = &iov;
	msg.msg_iovlen   = 1;
	msg.msg_control  = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags    = flags | MSG_NOSIGNAL;
repeat_send:
	msg.msg_iov->iov_len = left;
	msg.msg_iov->iov_base = (char *) buf + written;

	len = sock_sendmsg(sock, &msg, left);

	Dprintk("sendmsg ret: %d, written: %d, left: %d.\n", len,written,left);
	if ((len == -ERESTARTSYS) || (!(flags & MSG_DONTWAIT) &&
			 (len == -EAGAIN))) {
		reap_kids();
		goto repeat_send;
	}
	if (len > 0) {
		written += len;
		left -= len;
		if (left)
			goto repeat_send;
	}
	if (len >= 0) {
		if (written != length)
			TUX_BUG();
		if (left)
			TUX_BUG();
	}
	if (req && (written > 0))
		req->bytes_sent += written;
	Dprintk("sendmsg FINAL ret: %d, written: %d, left: %d.\n", len,written,left);
	return written ? written : len;
}

int tux_zerocopy_sendfile = 1;

typedef struct sock_send_desc
{
	struct socket *sock;
	int push;
} sock_send_desc_t;

int sock_send_actor (read_descriptor_t * desc, struct page *page,
				unsigned long offset, unsigned long size)
{
	sock_send_desc_t *sock_desc = (sock_send_desc_t *)desc->buf;
	struct socket *sock = sock_desc->sock;
	int push = sock_desc->push;
	unsigned int flags;
	ssize_t written;

	if (desc->count < size)
		size = desc->count;
	Dprintk("sock_send_actor(), page: %p, offset: %ld, size: %ld, sock: %p, desc->count: %d, desc->written: %d.\n", page, offset, size, sock, desc->count, desc->written);
	flags = MSG_DONTWAIT | MSG_NOSIGNAL;
	if (!push || !(desc->count == size))
		flags |= MSG_MORE;
	if (tux_zerocopy_sendfile && sock->ops->sendpage &&
	    (sock->sk->route_caps&NETIF_F_SG)) {
		written = sock->ops->sendpage(sock, page, offset, size, flags);
	} else {
		struct msghdr msg;
		struct iovec iov;
		char *kaddr;
		mm_segment_t oldmm;

		if (offset+size > PAGE_SIZE)
			return -EFAULT;

		kaddr = kmap(page);
	
		msg.msg_name = NULL;
		msg.msg_namelen = 0;
		msg.msg_iov = &iov;
		msg.msg_iovlen = 1;
		msg.msg_control = NULL;
		msg.msg_controllen = 0;
		msg.msg_flags = flags;
		iov.iov_base = kaddr + offset;
		iov.iov_len = size;

		oldmm = get_fs(); set_fs(KERNEL_DS);
		written = sock_sendmsg(sock, &msg, size);
		set_fs(oldmm);

		Dprintk("kaddr: %p, offset: %ld, size: %ld, written: %d.\n", kaddr, offset, size, written);
		kunmap(page);
	}
	if (written < 0) {
		desc->error = written;
		written = 0;
	}
	Dprintk("desc->count: %d, desc->written: %d, written: %d.\n", desc->count, desc->written, written);
	desc->count -= written;
	if (desc->count < 0)
		TUX_BUG();
	desc->written += written;

	return written;
}

/*
 * Return 1 if the output space condition went away
 * before adding the handler.
 */
int add_output_space_event (tux_req_t *req, struct socket *sock)
{
	struct sock *sk = sock->sk;
	/*
	 * blocked due to socket IO?
	 */
	spin_lock_irq(&req->ti->work_lock);
	add_keepalive_timer(req);
	if (test_and_set_bit(0,&req->wait_output_space))
		TUX_BUG();
	INC_STAT(nr_output_space_pending);

	if ((sk->state == TCP_ESTABLISHED) && enough_wspace(sk)) {
		if (test_and_clear_bit(0, &req->wait_output_space)) {
			DEC_STAT(nr_output_space_pending);
			del_keepalive_timer(req);
			spin_unlock_irq(&req->ti->work_lock);
//			tux_push_pending(sk);
			return 1;
		}
	}
	spin_unlock_irq(&req->ti->work_lock);

	return 0;
}

#define SEND_BLOCKSIZE 8192

int generic_send_file (tux_req_t *req, int push, int nonblock,
						struct socket *sock)
{
	sock_send_desc_t sock_desc;
	int len, want;

repeat:
	Dprintk("generic_send_file(%p,%d,%d,%p) called, f_pos: %d.\n", req, push, nonblock, sock, (int)req->in_file.f_pos);
	if (connection_too_fast(req) == 2) {
		len = -5;
		goto out;
	}
	if (req->filelen < req->in_file.f_pos)
		TUX_BUG();

	req->desc.written = 0;
	want = req->filelen - req->in_file.f_pos;
	if (want > SEND_BLOCKSIZE)
		want = SEND_BLOCKSIZE;
	req->desc.count = want;
	sock_desc.sock = sock;
	sock_desc.push = push;
	req->desc.buf = (char *) &sock_desc;
	req->desc.error = 0;
	Dprintk("sendfile(), desc.count: %d.\n", req->desc.count);
	do_generic_file_read(&req->in_file, &req->in_file.f_pos, &req->desc, sock_send_actor, nonblock);
	if (req->desc.written > 0)
		req->bytes_sent += req->desc.written;
	if (!nonblock && (req->desc.error == -EWOULDBLOCKIO))
		TUX_BUG();
	Dprintk("sendfile() wrote: %d bytes.\n", req->desc.written);

	switch (req->desc.error) {

	case -EWOULDBLOCKIO:
		len = -3;
		break;
	case -EAGAIN:
		Dprintk("sk->wmem_queued: %d, sk->sndbuf: %d.\n",
			sock->sk->wmem_queued, sock->sk->sndbuf);
		len = -4;
		break;
	default:
		len = req->desc.written;
#if CONFIG_TUX_DEBUG
		if (req->desc.error)
			printk(KERN_ERR "TUX: sendfile() returned error %d (signals pending: %08lx)!\n", req->desc.error, current->pending.signal.sig[0]);
#endif
		if (!req->desc.error && (req->filelen > req->in_file.f_pos))
			goto repeat;
#if CONFIG_TUX_DEBUG
		if (req->desc.written != want)
			printk(KERN_ERR "TUX: sendfile() wrote %d bytes, wanted %d! (pos %d) (signals pending: %08lx).\n", req->desc.written, want, (int)req->in_file.f_pos, current->pending.signal.sig[0]);
		else
			Dprintk("TUX: sendfile() FINISHED for req %p, wrote %d bytes.\n", req, req->desc.written);
		req->bytes_expected = 0;
#endif
		req->in_file.f_pos = 0;
		break;
	}

out:
	Dprintk("sendfile() wrote %d bytes.\n", len);

	return len;
}

static int file_fetch_actor (read_descriptor_t * desc, struct page *page,
				unsigned long offset, unsigned long size)
{
	if (desc->count < size)
		size = desc->count;

	desc->count -= size;
	desc->written += size;

	return size;
}

int tux_fetch_file (tux_req_t *req, int nonblock)
{
	int len;

	req->desc.written = 0;
	req->desc.count = req->filelen - req->in_file.f_pos;
	req->desc.buf = NULL;
	req->desc.error = 0;

	do_generic_file_read(&req->in_file, &req->in_file.f_pos, &req->desc,
					file_fetch_actor, nonblock);
	if (nonblock && (req->desc.error == -EWOULDBLOCKIO))
		return 1;
	len = req->desc.written;
	if (req->desc.error)
		Dprintk("fetchfile() returned %d error!\n", req->desc.error);
	Dprintk("fetchfile() fetched %d bytes.\n", len);
	return 0;
}

