/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * cachemiss.c: handle the 'slow IO path' by queueing not-yet-cached
 * requests to the IO-thread pool. Dynamic load balancing is done
 * between IO threads, based on the number of requests they have pending.
 */

#include <net/tux.h>
#include <linux/delay.h>

void queue_cachemiss (tux_req_t *req)
{
	iothread_t *iot = req->ti->iot;

	if (req->idle_input || req->wait_output_space)
		TUX_BUG();
	req->had_cachemiss = 1;
	if (!list_empty(&req->work))
		TUX_BUG();
	spin_lock(&iot->async_lock);
	if (connection_too_fast(req))
		list_add_tail(&req->work, &iot->async_queue);
	else
		list_add(&req->work, &iot->async_queue);
	iot->nr_async_pending++;
	INC_STAT(nr_cachemiss_pending);
	spin_unlock(&iot->async_lock);

	wake_up(&iot->async_sleep);
}

static tux_req_t * get_cachemiss (iothread_t *iot)
{
	struct list_head *tmp;
	tux_req_t *req = NULL;

	spin_lock(&iot->async_lock);
	if (!list_empty(&iot->async_queue)) {

		tmp = iot->async_queue.next;
		req = list_entry(tmp, tux_req_t, work);

		list_del(tmp);
		DEBUG_DEL_LIST(tmp);
		iot->nr_async_pending--;
		DEC_STAT(nr_cachemiss_pending);

		if (req->ti->iot != iot)
			TUX_BUG();
	}
	spin_unlock(&iot->async_lock);
	return req;
}

struct file * tux_open_file (char *filename, int mode)
{
	struct file *filp;

	if (!filename)
		TUX_BUG();

	/* Rule no. 3 -- Does the file exist ? */

	filp = filp_open(filename, mode, 0600);

	if (IS_ERR(filp) || !filp || !filp->f_dentry)
		goto err;

out:
	return filp;
err:
	Dprintk("filp_open() error: %d.\n", (int)filp);
	filp = NULL;
	goto out;
}

static int cachemiss_thread (void *data)
{
	tux_req_t *req;
	struct k_sigaction *ka;
	DECLARE_WAITQUEUE(wait, current);
	iothread_t *iot = data;
	int nr = iot->ti->cpu;

//	daemonize();
	tux_chroot(tux_docroot);
	drop_permissions();

	spin_lock(&iot->async_lock);
	iot->threads++;
	sprintf(current->comm, "async IO %d/%d", nr, iot->threads);

#if CONFIG_SMP
	{
		unsigned int mask;

		mask = 1 << nr; 
		if (cpu_online_map & mask)
			current->cpus_allowed = mask;
	}
#endif

	spin_lock_irq(&current->sigmask_lock);
	ka = current->sig->action + SIGCHLD-1;
	ka->sa.sa_handler = SIG_IGN;
	siginitsetinv(&current->blocked, sigmask(SIGCHLD));
	recalc_sigpending(current);
	spin_unlock_irq(&current->sigmask_lock);

	spin_unlock(&iot->async_lock);

	for (;;) {
		while (!list_empty(&iot->async_queue) &&
				(req = get_cachemiss(iot))) {

			tux_schedule_atom(req, 1);
			if (signal_pending(current)) {
				flush_all_signals();
				while (sys_wait4(-1, NULL, WNOHANG, NULL) > 0)
					/* nothing */;
			}
		}
		if (signal_pending(current)) {
			flush_all_signals();
			while (sys_wait4(-1, NULL, WNOHANG, NULL) > 0)
				/* nothing */;
		}
		if (!list_empty(&iot->async_queue))
			continue;
		if (iot->shutdown)
			break;
		add_wait_queue_exclusive(&iot->async_sleep, &wait);
		__set_current_state(TASK_INTERRUPTIBLE);
		if (list_empty(&iot->async_queue))
			schedule();
		__set_current_state(TASK_RUNNING);
		remove_wait_queue(&iot->async_sleep, &wait);
	}
	spin_lock(&iot->async_lock);
	if (!--iot->threads)
		wake_up(&iot->wait_shutdown);
	spin_unlock(&iot->async_lock);

	return 0;
}

static void __stop_cachemiss_threads (iothread_t *iot)
{
	DECLARE_WAITQUEUE(wait, current);

	Dprintk("stopping async IO threads %p.\n", iot);
	add_wait_queue(&iot->wait_shutdown, &wait);

	spin_lock(&iot->async_lock);
	if (iot->shutdown)
		TUX_BUG();
	if (!iot->threads)
		TUX_BUG();
	iot->shutdown = 1;
	wake_up_all(&iot->async_sleep);
	spin_unlock(&iot->async_lock);
		
	__set_current_state(TASK_UNINTERRUPTIBLE);
	schedule();
	remove_wait_queue(&iot->wait_shutdown, &wait);

	if (iot->threads)
		TUX_BUG();
	if (iot->nr_async_pending)
		TUX_BUG();
	Dprintk("stopped async IO threads %p.\n", iot);
}

void stop_cachemiss_threads (threadinfo_t *ti)
{
	iothread_t *iot = ti->iot;

	if (!iot)
		TUX_BUG();
	if (iot->nr_async_pending)
		TUX_BUG();
	__stop_cachemiss_threads(iot);
	ti->iot = NULL;
	kfree(iot);
}

int start_cachemiss_threads (threadinfo_t *ti)
{
	int i, pid;

	iothread_t *iot;

	iot = kmalloc(sizeof(*iot), GFP_KERNEL);
	if (!iot)
		return -ENOMEM;
	memset(iot, 0, sizeof(*iot));

	iot->ti = ti;
	iot->async_lock = SPIN_LOCK_UNLOCKED;
	iot->nr_async_pending = 0;
	INIT_LIST_HEAD(&iot->async_queue);
	init_waitqueue_head(&iot->async_sleep);
	init_waitqueue_head(&iot->wait_shutdown);
		
	for (i = 0; i < NR_IO_THREADS; i++) {
		pid = kernel_thread(cachemiss_thread, (void *)iot, 0);
		if (pid < 0) {
			printk(KERN_ERR "TUX: error %d creating IO thread!\n",
					pid);
			__stop_cachemiss_threads(iot);
			kfree(iot);
			return pid;
		}
	}
	ti->iot = iot;
	/*
	 * Wait for all cachemiss threads to start up:
	 */
	while (iot->threads != NR_IO_THREADS) {
		__set_current_state(TASK_INTERRUPTIBLE);
		schedule_timeout(HZ/10);
	}
	return 0;
}

