#ifndef _NET_TUX_H
#define _NET_TUX_H

/*
 * TUX - Integrated Application Protocols Layer and Object Cache
 *
 * Copyright (C) 2000, 2001, Ingo Molnar <mingo@redhat.com>
 *
 * tux.h: main structure definitions and function prototypes
 */

#define __KERNEL_SYSCALLS__

#include <linux/mm.h>
#include <linux/net.h>
#include <linux/wait.h>
#include <linux/file.h>
#include <linux/mman.h>
#include <linux/swap.h>
#include <linux/ctype.h>
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/unistd.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/pagemap.h>
#include <linux/vmalloc.h>
#include <linux/utsname.h>
#include <linux/smp_lock.h>
#include <linux/kernel_stat.h>
#include <linux/kernel_stat.h>

#include <asm/unaligned.h>

#include <net/tcp.h>
#include <net/tux_u.h>

/* Maximum number of threads: */
#define CONFIG_TUX_NUMTHREADS 8

/* Number of cachemiss/IO threads: */
#define NR_IO_THREADS 8

/* Maximum number of listen sockets per thread: */
#define CONFIG_TUX_NUMSOCKETS 4

extern unsigned int tux_listen [CONFIG_TUX_NUMTHREADS][CONFIG_TUX_NUMSOCKETS];

extern spinlock_t tux_module_lock;
extern struct module *tux_module;
extern int (*sys_tux_ptr) (unsigned int action, user_req_t *u_info);

#undef Dprintk

extern int tux_Dprintk;

#define TUX_DEBUG CONFIG_TUX_DEBUG
#if CONFIG_TUX_DEBUG
# define TUX_BUG() BUG()

# define INC_STAT(x) atomic_inc((atomic_t *)&kstat.x)
# define DEC_STAT(x) atomic_dec((atomic_t *)&kstat.x)
# define ADD_STAT(y,x) atomic_add(y,(atomic_t *)&kstat.x)
# define SUB_STAT(y,x) atomic_sub(y,(atomic_t *)&kstat.x)
# define TUX_DPRINTK 1
# define TDprintk(x...) do { printk("<%ld:%s:%d>: ", jiffies, __FILE__, __LINE__); printk(x); } while (0)
# define Dprintk(x...) do { if (tux_Dprintk == 1) TDprintk(x); } while (0)
#else
# define TUX_DPRINTK 0
# define Dprintk(x...) do { } while (0)
# define TDprintk(x...) do { } while (0)
# define INC_STAT(x) do { } while (0)
# define DEC_STAT(x) do { } while (0)
# define ADD_STAT(x,y) do { } while (0)
# define SUB_STAT(x,y) do { } while (0)
# define TUX_BUG() do { } while (0)
#endif

// lru needs this:

# define DEBUG_DEL_LIST(x...) do { INIT_LIST_HEAD((x)); } while (0)


#define LOG_LEN (8*1024*1024UL)

struct tux_req_struct;
typedef struct tux_req_struct tux_req_t;
typedef struct tux_threadinfo threadinfo_t;

extern struct address_space_operations url_aops;

typedef struct tcapi_template_s {
	char *vfs_name;
	struct list_head modules;
	int (*query) (tux_req_t *req);
	struct module *mod;
	int userspace_id;
} tcapi_template_t;

typedef struct mimetype_s {
	struct list_head list;

	char *ext;
	int ext_len;
	char *type;
	int type_len;

	int special;
} mimetype_t;

typedef struct tux_attribute_s {
	mimetype_t *mime;
	tcapi_template_t *tcapi;
} tux_attribute_t;

#define MAX_TUX_ATOMS 8

typedef void (atom_func_t)(tux_req_t *req, int cachemiss);

typedef struct tux_proto_s
{
	int defer_accept;
	void (*got_request) (tux_req_t *req);
	int (*parse_message) (tux_req_t *req, const int total_len);
	atom_func_t *illegal_request;

} tux_proto_t;

typedef struct abuf_s {
	struct page *page;
	char *buf;
	unsigned int size;
	unsigned int max_len;
	unsigned int offset;
	unsigned int left;
	unsigned long flags;
} abuf_t;

#define TUX_MAGIC 0x12457801

#define MAX_TUX_ATOMS 8

struct tux_req_struct
{
	tux_proto_t *proto;

	int atom_idx;
	atom_func_t *atoms [MAX_TUX_ATOMS];
	struct list_head work;

	struct list_head all;
	struct list_head free;
	struct list_head lru;

	unsigned int idle_input;
	unsigned int wait_output_space;

	struct socket *sock;
	struct dentry *dentry;
	struct file in_file;
	int fd;
	read_descriptor_t desc;

	unsigned int filelen;

	char usermode;
	int usermodule_idx;
	struct dentry *module_dentry;
	char *userbuf;
	unsigned int userlen;

	tux_attribute_t *attr;

	threadinfo_t *ti;
	wait_queue_t sleep;
	wait_queue_t ftp_sleep;

	struct sk_buff *input_skb;

	abuf_t abuf;
	/*
	 * Parsed request fields. In-line strings are zero-delimited.
	 */

	char *headers;
	char *headers_buf;
	int headers_len;

	int parsed_len;

	// FTP part
	ftp_command_t ftp_command;
	u32 ftp_user_addr;
	u16 ftp_user_port;

	struct socket *ftp_data_sock;
	unsigned int ftp_offset;

#define MAX_USERNAME_LEN 16
	char username[MAX_USERNAME_LEN];
	unsigned int username_len;

	// HTTP part
	http_method_t method;
	const char *method_str;
	int method_len;

	http_version_t version;
	const char *version_str;
	int version_len;

	/* requested URI: */

	const char *uri_str;
	int uri_len;

	/* Objectname (filename/scriptname) this URI refers to: */

#define MAX_OBJECTNAME_LEN 256
	char objectname[MAX_OBJECTNAME_LEN + 4]; // space for .gz as well
	unsigned int objectname_len;

	/* Query string within the URI: */

	const char *query_str;
	int query_len;

	/* Cookies: */

	const char *cookies_str;
	int cookies_len;
	int parse_cookies;

	/* Content-TYpe */
	const char *content_type_str;
	int content_type_len;

	/* Content-Length: */

	const char *contentlen_str;
	int contentlen_len;
	int content_len;

	/* User-Agent: */

	const char *user_agent_str;
	int user_agent_len;

	/* Accept: */

	const char *accept_str;
	int accept_len;

	/* Accept-Charset: */

	const char *accept_charset_str;
	int accept_charset_len;

	/* Accept-Language: */

	const char *accept_language_str;
	int accept_language_len;

	/* Cache-Control: */

	const char *cache_control_str;
	int cache_control_len;

	/* If-Modified-Since: */

	const char *if_modified_since_str;
	int if_modified_since_len;

	/* Negotiate: */

	const char *negotiate_str;
	int negotiate_len;

	/* Pragma: */

	const char *pragma_str;
	int pragma_len;

	/* Referer: */

	const char *referer_str;
	int referer_len;

	/* Accept-Encoding: */

	const char *accept_encoding_str;
	int accept_encoding_len;
	int may_send_gzip;
	int content_gzipped;

	/* Host */

#define MAX_HOST_LEN 128
	char host[MAX_HOST_LEN];
	int host_len;

	struct dentry *cwd;
	struct vfsmount *cwdmnt;

	/* POSTed data: */

	const char *post_data_str;
	int post_data_len;

	int status;

	/* the file being sent */

	int bytes_sent;
#if CONFIG_TUX_DEBUG
	int bytes_expected;
#endif
	unsigned long first_timestamp;
	int body_len;

	char error;
	char postponed;

	char had_cachemiss;
	char lookup_dir;
	char lookup_404;

	char keep_alive;
	timer_t keepalive_timer;
	unsigned int total_bytes;
	timer_t output_timer;

	int nr_keepalives;

	int event;
	void *private;

	unsigned int magic;
	void (*real_data_ready)(struct sock *sk, int space);
	void (*real_state_change)(struct sock *sk);
	void (*real_write_space)(struct sock *sk);
	void (*real_error_report)(struct sock *sk);
	void (*real_destruct)(struct sock *sk);

	void (*ftp_real_data_ready)(struct sock *sk, int space);
	void (*ftp_real_state_change)(struct sock *sk);
	void (*ftp_real_write_space)(struct sock *sk);
	void (*ftp_real_error_report)(struct sock *sk);
	void (*ftp_real_destruct)(struct sock *sk);

#if CONFIG_TUX_EXTENDED_LOG
	unsigned long accept_timestamp;
	unsigned long parse_timestamp;
	unsigned long output_timestamp;
	unsigned long flush_timestamp;
# define SET_TIMESTAMP(x) do { (x) = jiffies; } while (0)
#else
# define SET_TIMESTAMP(x) do { } while (0)
#endif

};

extern void add_tux_atom (tux_req_t *req, atom_func_t *event_done);
extern void del_tux_atom (tux_req_t *req);
extern void tux_schedule_atom (tux_req_t *req, int cachemiss);
extern void add_req_to_workqueue (tux_req_t *req);


typedef struct iothread_s
{
	spinlock_t async_lock;
	threadinfo_t *ti;
	struct list_head async_queue;
	wait_queue_head_t async_sleep;
	int nr_async_pending;
	int threads;
	int shutdown;
	wait_queue_head_t wait_shutdown;
} iothread_t;

typedef struct tux_listen_s
{
	tux_proto_t *proto;
	struct socket *sock;
	int cloned;
} tux_listen_t;

struct tux_threadinfo
{
	tux_req_t *userspace_req;
	int started;
	struct task_struct *thread;
	iothread_t *iot;
	wait_queue_t wait_event [CONFIG_TUX_NUMSOCKETS];
	wait_queue_t stop;
	int pid;

	struct page *header_cache;
	int header_offset;

	int nr_requests;
	struct list_head all_requests;

	int nr_free_requests;
	spinlock_t free_requests_lock;
	struct list_head free_requests;

	spinlock_t work_lock;
	struct list_head work_pending;
	struct list_head lru;
	unsigned int nr_lru;

	int listen_error;
	tux_listen_t listen[CONFIG_TUX_NUMSOCKETS];

	int cpu;
	unsigned int __padding[16];
};

typedef enum special_mimetypes {
	NORMAL_MIME_TYPE,
	MIME_TYPE_REDIRECT,
	MIME_TYPE_CGI,
	MIME_TYPE_MODULE,
} special_mimetypes_t;

extern struct nameidata docroot;

#if CONFIG_TUX_DEBUG
extern inline void url_hist_hit (int size)
{
	unsigned int idx = size/1024;

	if (idx >= URL_HIST_SIZE)
		idx = URL_HIST_SIZE-1;
	kstat.url_hist_hits[idx]++;
}
extern inline void url_hist_miss (int size)
{
	unsigned int idx = size/1024;
 
	if (idx >= URL_HIST_SIZE)
		idx = URL_HIST_SIZE-1;
	kstat.url_hist_misses[idx]++;
}
extern void __check_req_list (tux_req_t *req, struct list_head *list);
# define check_req_list __check_req_list
#else
# define url_hist_hit(size) do { } while (0)
# define url_hist_miss(size) do { } while (0)
# define check_req_list(req, list) do { } while (0)
#endif


extern char tux_docroot[200];
extern char tux_logfile[200];
extern char tux_cgiroot[200];
extern char tux_404_page[200];
extern int tux_cgi_uid;
extern int tux_cgi_gid;
extern int tux_clientport;
extern int tux_logging;
extern int tux_serverport;
extern int tux_threads;
extern int tux_keepalive_timeout;
extern int tux_max_output_bandwidth;
extern int tux_max_backlog;
extern int tux_max_connect;
extern int tux_mode_forbidden;
extern int tux_mode_allowed;
extern int tux_logentry_align_order;
extern int tux_nonagle;
extern int tux_ack_pingpong;
extern int tux_push_all;
extern int tux_zerocopy_parse;
extern int virtual_server;
extern int tux_max_object_size;
extern unsigned int tux_max_free_requests;
extern int tux_defer_accept;


#if CONFIG_TUX_DEBUG
# undef FASTCALL
# define FASTCALL(x) x
#endif

extern struct socket * FASTCALL(start_listening(u16 port, u32 addr, int defer, int nr));
extern void FASTCALL(stop_listening(struct socket **sock));
extern void FASTCALL(start_sysctl(void));
extern void FASTCALL(end_sysctl(void));
extern void flush_request (tux_req_t *req, int cachemiss);
extern void unlink_tux_socket (tux_req_t *req);
extern void unlink_tux_ftp_data_socket (tux_req_t *req);
extern void unlink_tux_listen_socket (tux_req_t *req);
extern void link_tux_ftp_accept_socket (tux_req_t *req, struct socket *sock);
extern void link_tux_ftp_data_socket (tux_req_t *req, struct socket *sock);
extern void FASTCALL(push_frames (tux_req_t *req));
extern int FASTCALL(send_sync_buf (tux_req_t *req, struct socket *sock, const char *buf, const size_t length, unsigned long flags));
extern void FASTCALL(__send_async_message (tux_req_t *req, const char *message, int status));
extern void FASTCALL(send_success (tux_req_t *req, struct socket *sock));
extern void FASTCALL(send_async_err_not_found (tux_req_t *req));
extern void FASTCALL(send_async_timed_out (tux_req_t *req));

extern void FASTCALL(kfree_req (tux_req_t *req));
extern int FASTCALL(accept_requests (threadinfo_t *ti));
extern int FASTCALL(process_requests (threadinfo_t *ti, tux_req_t **user_req));
extern int FASTCALL(flush_freequeue (threadinfo_t * ti));
extern int FASTCALL(flush_workqueue (threadinfo_t *ti));
extern tux_req_t * FASTCALL(pick_userspace_req (threadinfo_t *ti));
extern atom_func_t redirect_request;
extern atom_func_t parse_request;
extern int FASTCALL(accept_pending (threadinfo_t *ti));
extern int FASTCALL(work_pending (threadinfo_t *ti));
extern void FASTCALL(queue_cachemiss (tux_req_t *req));
extern int FASTCALL(start_cachemiss_threads (threadinfo_t *ti));
extern void FASTCALL(stop_cachemiss_threads (threadinfo_t *ti));
struct file * FASTCALL(tux_open_file(char *filename, int mode));
extern void FASTCALL(init_log_thread (void));
extern void FASTCALL(stop_log_thread (void));
extern void add_mimetype (char *new_ext, char *new_type);
extern void free_mimetypes (void);
extern int FASTCALL(lookup_url (tux_req_t *req, const unsigned int flag));
extern int FASTCALL(handle_gzip_req (tux_req_t *req, unsigned int flags));
extern struct dentry * FASTCALL(tux_lookup (tux_req_t *req, const char *filename, const unsigned int flag));
extern tcapi_template_t * lookup_tuxmodule (const char *filename);
extern int register_tuxmodule (tcapi_template_t *tcapi);
extern tcapi_template_t * unregister_tuxmodule (char *vfs_name);
extern tcapi_template_t * get_first_usermodule (void);
extern int user_register_module (user_req_t *u_info);
extern int user_unregister_module (user_req_t *u_info);
extern void unregister_all_tuxmodules (void);

typedef struct exec_param_s {
	char *command;
	char **argv;
	char **envp;
	int pipe_fds;
} exec_param_t;

extern pid_t tux_exec_process (char *command, char **argv, char **envp, int pipe_fds, exec_param_t *param, int wait);

extern void start_external_cgi (tux_req_t *req);
extern tcapi_template_t extcgi_tcapi;

extern void queue_output_req (tux_req_t *req, threadinfo_t *ti);
extern void queue_userspace_req (tux_req_t *req, threadinfo_t *ti);


extern void FASTCALL(__log_request (tux_req_t *req));
extern inline void log_request (tux_req_t *req)
{
	if (tux_logging)
		__log_request(req);
}

extern int FASTCALL(connection_too_fast (tux_req_t *req));
extern void FASTCALL(trunc_headers (tux_req_t *req));
extern int FASTCALL(generic_send_file (tux_req_t *req, int push, int nonblock, struct socket *sock));
extern int FASTCALL(tux_fetch_file (tux_req_t *req, int nonblock));

extern void FASTCALL(postpone_request (tux_req_t *req));
extern int FASTCALL(continue_request (int fd));
extern void FASTCALL(tux_push_pending (struct sock *sk));
extern void FASTCALL(zap_request (tux_req_t *req, int cachemiss));
extern int FASTCALL(add_output_space_event (tux_req_t *req, struct socket *sock));

extern void reap_kids (void);
extern void unuse_frag (struct sk_buff *skb, skb_frag_t *frag);
extern skb_frag_t * build_dynbuf_frag (tux_req_t *req, int size);
extern int url_permission (struct inode *inode);
extern void flush_all_signals (void);

extern int multifragment_api;

extern int tux_print_retransmit;

#define D() Dprintk("{%s:%d}\n", __FILE__, __LINE__)

#define tux_sleep(n) \
	do {						\
		current->state = TASK_INTERRUPTIBLE;	\
		schedule_timeout(HZ * (n));		\
	} while (0)

#define tux_file file

#define tux_write_file(file, buf, len) \
		({ unsigned int __ret; mm_segment_t oldmm = get_fs(); set_fs(KERNEL_DS); __ret = ((file)->f_op->write(file, buf, len, &(file)->f_pos)); set_fs(oldmm); __ret; })

#define tux_read_file(file, buf, len) \
		({ unsigned int __ret; mm_segment_t oldmm = get_fs(); set_fs(KERNEL_DS); __ret = ((file)->f_op->read(file, buf, len, &(file)->f_pos)); set_fs(oldmm); __ret; })

#define tux_close_file(file) \
		(fput(file))

#define TUX_DECLARE_MUTEX DECLARE_MUTEX
#define tux_down down
#define tux_up up

#define tux_time() CURRENT_TIME

#define tux_direntry dentry
#define tux_direntry_open(d,r,fl) \
	({ struct file *__f; lock_kernel(); __f = dentry_open(d,r,fl); unlock_kernel(); __f; })
#define tux_lookup_direntry(f,r,fl) \
	({ struct dentry *__d; lock_kernel(); __d = tux_lookup(f,r,fl); unlock_kernel(); __d; })
#define tux_file_size(file) ((file)->f_dentry->d_inode->i_size)

#define tux_mmap_page(file, virt, offset) \
({ \
	struct page *page = NULL; \
	page = grab_cache_page((file)->f_dentry->d_inode->i_mapping, 0); \
	if (page) { \
		virt = (char *)kmap(page); \
		UnlockPage(page); \
	} \
	page; \
})

#define tux_direntry_error(dentry) \
		(!(dentry) || IS_ERR(dentry) || !(dentry)->d_inode)
#define tux_dput(d) do { lock_kernel(); dput(d); unlock_kernel(); } while (0)
#define tux_mtime(dentry) \
		((dentry)->d_inode->i_mtime)
#define tux_file_error(file) \
		((!file) || !(file)->f_dentry || !(file)->f_dentry->d_inode)

#define tux_getpid() (current->pid)
#define tux_client_addr(req) ((req)->sock->sk->daddr)

#define tux_page page

extern int nr_async_io_pending (void);

extern void __add_keepalive_timer (tux_req_t *req);
#define add_keepalive_timer(req)					\
do {									\
	if (tux_keepalive_timeout) {					\
		Dprintk("add_keepalive_timer(%p).\n", (req));		\
		__add_keepalive_timer(req);				\
	}								\
} while (0)
extern void __del_keepalive_timer (tux_req_t *req);
#define del_keepalive_timer(req)					\
do {									\
	if (tux_keepalive_timeout) {					\
		Dprintk("del_keepalive_timer(%p).\n", (req));		\
		__del_keepalive_timer(req);				\
	}								\
} while (0)

extern void del_output_timer (tux_req_t *req);
extern void output_timeout (tux_req_t *req);

extern void print_req (tux_req_t *req);

extern char tux_date [DATE_LEN];


extern int nr_async_io_pending (void);
extern void tux_exit (void);
extern char * FASTCALL(get_abuf (tux_req_t *req, int max_size));
extern void FASTCALL(send_abuf (tux_req_t *req, int size, unsigned long flags));


extern int idle_event (tux_req_t *req);
extern int output_space_event (tux_req_t *req);
extern unsigned int log_cpu_mask;
extern int tux_compression;
extern int tux_noid;
extern int tux_cgi_inherit_cpu;
extern int tux_zerocopy_header;
extern int tux_zerocopy_sendfile;
extern unsigned int tux_cgi_cpu_mask;
extern tux_proto_t tux_proto_http;
extern tux_proto_t tux_proto_ftp;
extern int tux_all_userspace;
extern int tux_redirect_logging;
extern int tux_referer_logging;
extern unsigned int tux_max_header_len;
extern int tux_application_protocol;

extern void drop_permissions (void);
extern int query_extcgi (tux_req_t *req);
extern int tux_chroot (char *dir);

extern void install_req_dentry (tux_req_t *req, struct dentry *dentry);
extern void release_req_dentry (tux_req_t *req);
extern void unidle_req (tux_req_t *req);
extern int nr_requests_used (void);

#define req_err(req) do { (req)->error = 1; TDprintk("request %p error at %s:%d.\n", req, __FILE__, __LINE__); } while (0)

#define enough_wspace(sk) (tcp_wspace(sk) >= tcp_min_write_space(sk))
#define clear_keepalive(req) do { (req)->keep_alive = 0; Dprintk("keepalive cleared for req %p.\n", req); } while (0)

extern int print_all_requests (threadinfo_t *ti);
extern int tux_max_keepalives;

#endif
