/*		a driver to encapsulate SCSI in USB
*
 * (C) Copyright 2000 John Fremlin <vii@penguinpowered.com>
 * (C) Copyright 2000 Oliver Neukum <Oliver.Neukum@lrz.uni-muenchen.de>
 * (C) Copyright 2000 Jeremy Hall <JHall@UU.NET>
 *
 * Parts shamelessly stolen from usb-storage and microtek and copyright by their
 * authors. Thanks to Matt Dharm for giving us permission!
 *
 * This driver implements a SCSI host controller driver and a USB
 * device driver. To avoid confusion, all the USB related stuff is
 * prefixed by usc_usb_ and all the SCSI stuff by usc_scsi_.
 *
 * Guessed protocol:
 *
 *	Send raw SCSI command to EP 0x1
 *
 *	If there is data to receive:
 *		If the command was READ datatype=image:
 *			Read a lot of data from EP 0x83
 *		Else:
 *			Read data from EP 0x82
 *	Else:
 *		If there is data to transmit:
 *			Write it to EP 0x1
 *
 *	Read status byte from EP 0x82
 *
 * References:
 *
 * The SCSI command set for the scanner is available from
 *	ftp://ftp.microtek.com/microtek/devpack/
 *
 * We'll fix the comments later
 * 20001006 16:53 EDT work begins
 * 20001007 00:58:35 first test run
 * 20001008 02:24:40 FIRST IMAGE SCANNED!
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/random.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/spinlock.h>
#include <linux/smp_lock.h>
#include <linux/usb.h>
#include <linux/proc_fs.h>

#include <asm/atomic.h>
#include <linux/blk.h>
#include "../scsi/scsi.h"
#include "../scsi/hosts.h"
#include "../scsi/sd.h"

#include "hp5300.h"

/* Constants */

#define USC_ABORT_TIMEOUT HZ /*jiffies*/


/* Should we do debugging? */

#define USC_DO_DEBUG


static struct usb_device_id hp5300_usb_id [] =
{
        {  USB_DEVICE(0x3f0, 0x0701) },
	{  }
};


/* USB layer driver interface */

static void *usc_usb_probe(struct usb_device *dev, unsigned int interface, const struct usb_device_id *id);
static void usc_usb_disconnect(struct usb_device *dev, void *ptr);

static struct usb_driver usc_usb_driver = {
	name:                   "hp5300",
	probe:                  usc_usb_probe,
	disconnect:             usc_usb_disconnect,
	id_table:               hp5300_usb_id,
};


/* Internal driver stuff */

#define USC_VERSION	"0.0.1"
#define USC_NAME	"USB-SCSI: "

#define USC_WARNING(x...) \
	printk( KERN_WARNING USC_NAME x )
#define USC_ERROR(x...) \
	printk( KERN_ERR USC_NAME x )
#define USC_INT_ERROR(x...) \
	USC_ERROR(x)
#define USC_MESSAGE(x...) \
	printk( KERN_INFO USC_NAME x )

#if defined USC_DO_DEBUG

#define USC_DEBUG(x...) \
	printk( KERN_DEBUG USC_NAME x )

#define USC_DEBUG_GOT_HERE() \
	USC_DEBUG("got to %s:%d (%s)\n", __FILE__, (int)__LINE__, __PRETTY_FUNCTION__ )
#define USC_DEBUG_INT() \
	do { USC_DEBUG_GOT_HERE(); \
	     USC_DEBUG("transfer = %x context = %x\n",(int)transfer,(int)context ); \
	     USC_DEBUG("transfer->status = %x data-length = %x sent = %x\n",(int)transfer->status,(int)context->data_length, (int)transfer->actual_length ); \
             usc_debug_dump(context->instance);\
	   } while(0)
#else

#define USC_NUL_STATEMENT do { } while(0)

#define USC_DEBUG(x...)	USC_NUL_STATEMENT
#define USC_DEBUG_GOT_HERE() USC_NUL_STATEMENT
#define USC_DEBUG_INT() USC_NUL_STATEMENT
#define USC_DEBUG_HERE USC_NUL_STATEMENT

#endif


static void usc_transfer_cleanup( struct urb *transfer );

#define USC_INT_INIT()\
	do {\
	context = (struct usc_transfer_context*)transfer->context; \
	if (atomic_read(&context->do_abort)) {\
		usc_transfer_cleanup(transfer);\
		return;\
	}\
	USC_DEBUG_INT();\
	} while (0)

static inline void usc_debug_dump(struct usc_desc* desc) {
	USC_DEBUG("desc at 0x%x: halted = %x%x, toggle = %x%x\n",
		  (int)desc,(int)desc->usb_dev->halted[1],(int)desc->usb_dev->halted[0],
		  (int)desc->usb_dev->toggle[1],(int)desc->usb_dev->toggle[0]
		);
	USC_DEBUG("ep_out=%x ep_response=%x ep_image=%x\n",
		  desc->out_pipe,
		  desc->status_pipe,
		  desc->image_pipe
		);
}


static inline void usc_show_command(Scsi_Cmnd *srb)
{
	char *what = NULL;

	switch (srb->cmnd[0]) {
	case TEST_UNIT_READY: what = "TEST_UNIT_READY"; break;
	case REZERO_UNIT: what = "REZERO_UNIT"; break;
	case REQUEST_SENSE: what = "REQUEST_SENSE"; break;
	case FORMAT_UNIT: what = "FORMAT_UNIT"; break;
	case READ_BLOCK_LIMITS: what = "READ_BLOCK_LIMITS"; break;
	case REASSIGN_BLOCKS: what = "REASSIGN_BLOCKS"; break;
	case READ_6: what = "READ_6"; break;
	case WRITE_6: what = "WRITE_6"; break;
	case SEEK_6: what = "SEEK_6"; break;
	case READ_REVERSE: what = "READ_REVERSE"; break;
	case WRITE_FILEMARKS: what = "WRITE_FILEMARKS"; break;
	case SPACE: what = "SPACE"; break;
	case INQUIRY: what = "INQUIRY"; break;
	case RECOVER_BUFFERED_DATA: what = "RECOVER_BUFFERED_DATA"; break;
	case MODE_SELECT: what = "MODE_SELECT"; break;
	case RESERVE: what = "RESERVE"; break;
	case RELEASE: what = "RELEASE"; break;
	case COPY: what = "COPY"; break;
	case ERASE: what = "ERASE"; break;
	case MODE_SENSE: what = "MODE_SENSE"; break;
	case START_STOP: what = "START_STOP"; break;
	case RECEIVE_DIAGNOSTIC: what = "RECEIVE_DIAGNOSTIC"; break;
	case SEND_DIAGNOSTIC: what = "SEND_DIAGNOSTIC"; break;
	case ALLOW_MEDIUM_REMOVAL: what = "ALLOW_MEDIUM_REMOVAL"; break;
	case SET_WINDOW: what = "SET_WINDOW"; break;
	case READ_CAPACITY: what = "READ_CAPACITY"; break;
	case READ_10: what = "READ_10"; break;
	case WRITE_10: what = "WRITE_10"; break;
	case SEEK_10: what = "SEEK_10"; break;
	case WRITE_VERIFY: what = "WRITE_VERIFY"; break;
	case VERIFY: what = "VERIFY"; break;
	case SEARCH_HIGH: what = "SEARCH_HIGH"; break;
	case SEARCH_EQUAL: what = "SEARCH_EQUAL"; break;
	case SEARCH_LOW: what = "SEARCH_LOW"; break;
	case SET_LIMITS: what = "SET_LIMITS"; break;
	case READ_POSITION: what = "READ_POSITION"; break;
	case SYNCHRONIZE_CACHE: what = "SYNCHRONIZE_CACHE"; break;
	case LOCK_UNLOCK_CACHE: what = "LOCK_UNLOCK_CACHE"; break;
	case READ_DEFECT_DATA: what = "READ_DEFECT_DATA"; break;
	case MEDIUM_SCAN: what = "MEDIUM_SCAN"; break;
	case COMPARE: what = "COMPARE"; break;
	case COPY_VERIFY: what = "COPY_VERIFY"; break;
	case WRITE_BUFFER: what = "WRITE_BUFFER"; break;
	case READ_BUFFER: what = "READ_BUFFER"; break;
	case UPDATE_BLOCK: what = "UPDATE_BLOCK"; break;
	case READ_LONG: what = "READ_LONG"; break;
	case WRITE_LONG: what = "WRITE_LONG"; break;
	case CHANGE_DEFINITION: what = "CHANGE_DEFINITION"; break;
	case WRITE_SAME: what = "WRITE_SAME"; break;
	case READ_TOC: what = "READ_TOC"; break;
	case LOG_SELECT: what = "LOG_SELECT"; break;
	case LOG_SENSE: what = "LOG_SENSE"; break;
	case MODE_SELECT_10: what = "MODE_SELECT_10"; break;
	case MODE_SENSE_10: what = "MODE_SENSE_10"; break;
	case MOVE_MEDIUM: what = "MOVE_MEDIUM"; break;
	case READ_12: what = "READ_12"; break;
	case WRITE_12: what = "WRITE_12"; break;
	case WRITE_VERIFY_12: what = "WRITE_VERIFY_12"; break;
	case SEARCH_HIGH_12: what = "SEARCH_HIGH_12"; break;
	case SEARCH_EQUAL_12: what = "SEARCH_EQUAL_12"; break;
	case SEARCH_LOW_12: what = "SEARCH_LOW_12"; break;
	case READ_ELEMENT_STATUS: what = "READ_ELEMENT_STATUS"; break;
	case SEND_VOLUME_TAG: what = "SEND_VOLUME_TAG"; break;
	case WRITE_LONG_2: what = "WRITE_LONG_2"; break;
	default:
		USC_DEBUG("can't decode command\n");
		goto out;
		break;
	}
	USC_DEBUG( "Command %s (%d bytes)\n", what, srb->cmd_len);

 out:
	USC_DEBUG( "  %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
	       srb->cmnd[0], srb->cmnd[1], srb->cmnd[2], srb->cmnd[3], srb->cmnd[4], srb->cmnd[5], 
	       srb->cmnd[6], srb->cmnd[7], srb->cmnd[8], srb->cmnd[9]);
}

static inline int usc_is_aborting(struct usc_desc* desc) {
	return (atomic_read(&desc->context.do_abort));
}

static inline void usc_request_abort(struct usc_desc* desc, struct urb *urb) {
	USC_DEBUG_GOT_HERE();
	usc_debug_dump(desc);
	atomic_set(&desc->context.do_abort,1);
}

static inline void usc_urb_abort(struct usc_desc* desc, struct urb* urb) {
	USC_DEBUG_GOT_HERE();
	usc_debug_dump(desc);
	if ( urb->status == USB_ST_URB_PENDING ) {
		usb_unlink_urb( urb );
	}
}

static inline void usc_wait_abort(struct usc_desc* desc, struct urb *urb, int which_urb)
{
	usc_request_abort(desc, urb);

	if (which_urb) {
	while( !atomic_read(&desc->status_lock.count) ) {
/* Is there a function to check if the semaphore is locked? */
		schedule_timeout( USC_ABORT_TIMEOUT ); 
		USC_DEBUG_GOT_HERE();
		usc_urb_abort(desc, urb);
	}
	} else {
	while( !atomic_read(&desc->cmd_lock.count) ) {
/* Is there a function to check if the semaphore is locked? */
		schedule_timeout( USC_ABORT_TIMEOUT ); 
		USC_DEBUG_GOT_HERE();
		usc_urb_abort(desc, urb);
	}
	}

}


static struct usc_desc * usc_list; /* list of active scanners */
struct semaphore usc_list_semaphore;

/* Internal list operations */

static
void usc_remove_nolock( struct usc_desc* to_remove )
{
	USC_DEBUG( "removing 0x%x from list\n",
		   (int)to_remove );

	USC_DEBUG("line %d: locking kernel!\n",__LINE__);
	lock_kernel();
	USC_DEBUG("line %d: trying to abort cmd_urb.\n",__LINE__);
	usc_wait_abort(to_remove, &to_remove->cmd_urb, 0);
	USC_DEBUG("line %d: Trying to abort status urb.\n",__LINE__);
up(&to_remove->status_lock);
//	usc_wait_abort(to_remove, &to_remove->status_urb, 1);
USC_DEBUG("line %d: unlinking status_urb which currently has status 0x%x\n",__LINE__,to_remove->status_urb.status);
			usb_unlink_urb(&to_remove->status_urb);
	USC_DEBUG("Finished aborting my urbs!\n");

	USC_DEBUG_GOT_HERE();
	
	if ( to_remove != usc_list ) {
		USC_DEBUG_GOT_HERE();
		if (to_remove->prev && to_remove->next)
			to_remove->prev->next = to_remove->next;
	} else {
		USC_DEBUG_GOT_HERE();
		usc_list = to_remove->next;
		if (usc_list) {
			USC_DEBUG_GOT_HERE();
			usc_list->prev = 0;
		}
	}
		
	if ( to_remove->next ) {
		USC_DEBUG_GOT_HERE();
		to_remove->next->prev = to_remove->prev;
	}

	USC_DEBUG_GOT_HERE();
	scsi_unregister_module(MODULE_SCSI_HA, &(to_remove->ctempl));
	USC_DEBUG_GOT_HERE();
	unlock_kernel();
	USC_DEBUG_GOT_HERE();
	
	kfree( to_remove );
	USC_DEBUG_GOT_HERE();
}

static
void usc_add_nolock( struct usc_desc* to_add )
{
	USC_DEBUG( "adding 0x%x to list\n", (int)to_add );
	
	to_add->prev = 0;
	to_add->next = usc_list;
	if ( usc_list ) {
		usc_list->prev = to_add;
	}
			
	usc_list = to_add;
}
		

static void usc_get_status( struct usc_desc *desc, struct urb *status_transfer, int pipe, void *status, int want_init );


/* SCSI driver interface */

/* scsi related functions - dummies for now mostly */

static int usc_scsi_release(struct Scsi_Host *psh)
{
	USC_DEBUG_GOT_HERE();
	
	return 0;
}

static int usc_scsi_abort (Scsi_Cmnd *srb)
/* interrupt context (!) */ /* FIXME this is about to become task context */
{
	struct usc_desc* desc = (struct usc_desc*)(srb->host->hostdata[0]);

	USC_DEBUG_GOT_HERE();
	
	usc_request_abort(desc, &desc->cmd_urb);
	usc_urb_abort(desc, &desc->cmd_urb);
	
	return SCSI_ABORT_PENDING;
}

static int usc_scsi_host_reset (Scsi_Cmnd *srb)
{
	struct usc_desc* desc = (struct usc_desc*)(srb->host->hostdata[0]);

	USC_DEBUG_GOT_HERE();
	usc_debug_dump(desc);

	usb_reset_device(desc->usb_dev); /*FIXME: untested on new reset code */
	return 0;  /* RANT why here 0 and not SUCCESS */
}

/* the core of the scsi part */

/* faking a detection - which can't fail :-) */

static int usc_scsi_detect (struct SHT * sht)
{
	/* Whole function stolen from usb-storage */
	
	struct usc_desc * desc = (struct usc_desc *)sht->proc_dir;
	/* What a hideous hack! */
	  
	char local_name[48];

	USC_DEBUG_GOT_HERE();

	/* set up the name of our subdirectory under /proc/scsi/ */
	sprintf(local_name, "usb-scsi-%d", desc->host_number);
	sht->proc_name = kmalloc (strlen(local_name) + 1, GFP_KERNEL);
	/* FIXME: where is this freed ? */

	if (!sht->proc_name) {
		USC_ERROR( "unable to allocate memory for proc interface!!\n" );
		return 0;
	}
	
	strcpy(sht->proc_name, local_name);

 	sht->proc_dir = NULL;

	/* In host->hostdata we store a pointer to desc */
	desc->host = scsi_register(sht, sizeof(desc));
	desc->host->hostdata[0] = (unsigned long)desc;
/* FIXME: what if sizeof(void*) != sizeof(unsigned long)? */ 

	return 1;
}



/* Main entrypoint: SCSI commands are dispatched to here */



static
int usc_scsi_queuecommand (Scsi_Cmnd *srb, usc_scsi_cmnd_callback callback );



inline static
void usc_int_submit_urb (struct urb* transfer,
			int pipe,
			void* data,
			unsigned length,
			usc_usb_urb_callback callback )
/* Interrupt context! */

/* Holding transfer->context->lock! */
{
	int res;
	struct usc_transfer_context* context;
	
	USC_INT_INIT();
	
		USC_DEBUG("Line %d: filling the bulk urb.\n",__LINE__);
	FILL_BULK_URB(transfer,
		      context->instance->usb_dev,
		      pipe,
		      data,
		      length,
		      callback,
		      context
		);

/*	transfer->transfer_flags = USB_DISABLE_SPD;*/
	transfer->transfer_flags = USB_ASYNC_UNLINK;
	transfer->status = 0;
	transfer->timeout = 100;

		USC_DEBUG("about to submit the bulk urb.\n");
	res = usb_submit_urb( transfer );
	USC_DEBUG("submitted! res is %d\n",res);
	if ( res ) {
		USC_INT_ERROR( "could not submit URB! Error was %d\n",(int)res );
		context->srb->result = DID_ERROR << 16;
		usc_transfer_cleanup(transfer);
	}
	return;
}


static void usc_transfer_cleanup( struct urb *transfer )
/* Interrupt context! */
{
	usc_scsi_cmnd_callback callback;
	struct usc_transfer_context* context = (struct usc_transfer_context*)transfer->context;
	
		USC_DEBUG_GOT_HERE();
	if ( context->lock)
		return;
		context->lock = 1;
		USC_DEBUG("not locked.\n");
	if ( context->final_callback ) {
		callback = context->final_callback;
		USC_DEBUG("running srb->callback.\n");
		context->final_callback = 0;
		wmb();
		callback(context->srb);
	}
		USC_DEBUG("releasing lock.\n");
	up( &context->instance->cmd_lock );

}

static void usc_status_done( struct urb *transfer )
/* interrupt context */
{
//	struct usc_transfer_context* context;

//	USC_INT_INIT();
	struct usc_transfer_context* context = (struct usc_transfer_context*)transfer->context;

USC_DEBUG_GOT_HERE();
 if (context->status) {
USC_DEBUG_GOT_HERE();
USC_DEBUG ("context->status is 0x%x\n",context->status);
	context->srb->result = DID_ERROR << 16;
	context->srb->result &= USC_SCSI_ERR_MASK;
	context->srb->result |= (unsigned)context->status<<1;
	usc_request_abort(context->instance, &context->instance->cmd_urb);
	usc_urb_abort(context->instance, &context->instance->cmd_urb);
USC_DEBUG_GOT_HERE();
		usc_transfer_cleanup(&context->instance->cmd_urb);
//	usc_get_status( context->instance, &context->instance->status_urb, context->instance->status_pipe, &context->status, 0);
return;
	} else {
USC_DEBUG_GOT_HERE();
		USC_DEBUG("status == 0\n");
	context->srb->result &= USC_SCSI_ERR_MASK;
	context->srb->result |= (unsigned)context->status<<1;
USC_DEBUG_GOT_HERE();
USC_DEBUG("context->srb->result = 0x%x\n",context->srb->result);
		usc_transfer_cleanup(&context->instance->cmd_urb);

//	usc_get_status( context->instance, &context->instance->status_urb, context->instance->status_pipe, &context->status, 0);
return;
	}

}

static void usc_transfer_done( struct urb *transfer )
{
	struct usc_transfer_context* context;

	USC_INT_INIT();

	context->srb->result &= USC_SCSI_ERR_MASK;
	context->srb->result |= (unsigned)context->status<<1;

	usc_transfer_cleanup(transfer);

	return;
}


static void usc_get_status( struct usc_desc *desc, struct urb *status_transfer, int pipe, void *status, int want_init )
/* Interrupt context! */
{
//	struct usc_transfer_context* context;

//	USC_INT_INIT();
int res;

	
		if (want_init) {
USC_DEBUG("line %d: filling the status urb.\n",__LINE__);
	FILL_INT_URB(status_transfer,
			desc->usb_dev,
			pipe,
			status,
			1,
			usc_status_done,
			&desc->context,
			desc->interrupt_interval
			);
		status_transfer->transfer_flags = USB_ASYNC_UNLINK;
	}
	USC_DEBUG("line %d: sending the status urb.\n",__LINE__);
	res = usb_submit_urb(status_transfer);
		if (res)
		USC_ERROR ("ERROR! unable to transmit status urb. res == %d status_transfer->status == 0x%x\n",res, status_transfer->status);
	
	
	return;
}

static void usc_data_done( struct urb* transfer )
/* Interrupt context! */
{
	struct usc_transfer_context* context;

	USC_INT_INIT();

USC_DEBUG_GOT_HERE();
if ( transfer->status ) {
		context->srb->result = DID_ERROR<<16;
USC_DEBUG ("JHALL: line %d: status %d error %x hope status_done takes this.\n",__LINE__, (int)transfer->status, context->srb->result);
		usc_transfer_cleanup(transfer);
	} else if ( context->data_length != transfer->actual_length ) {
		context->srb->resid = context->data_length - transfer->actual_length;
USC_DEBUG_GOT_HERE();
		usc_transfer_cleanup(transfer);
	}
//usc_transfer_done(transfer);

	return;
}


static void usc_command_done( struct urb *transfer )
/* Interrupt context! */
{
	struct usc_transfer_context* context;

	USC_INT_INIT();

		USC_DEBUG_GOT_HERE();
	if ( transfer->status ) {
		USC_DEBUG_GOT_HERE();
		context->srb->result = DID_ERROR<<16;
		usc_transfer_cleanup(transfer);
		
		return;
	}
	
		USC_DEBUG_GOT_HERE();
	if ( context->data ) {
		USC_DEBUG_GOT_HERE();
		usc_int_submit_urb(transfer,
				   context->data_pipe,
				   context->data,
				   context->data_length,
				   usc_data_done);
//	} else usc_transfer_done(transfer);
}
	
	return;
}



	static const unsigned char usc_direction[256/8] = {
		0x28, 0x81, 0x14, 0x14, 0x20, 0x01, 0x90, 0x77, 
		0x0C, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};


#define USC_DIRECTION_IS_IN(x) ((usc_direction[x>>3] >> (x & 7)) & 1)

static void
usc_build_transfer_context( Scsi_Cmnd *srb, struct usc_desc* desc )
{

	int pipe;


	USC_DEBUG_GOT_HERE();

	desc->context.instance = desc;
	desc->context.srb = srb;
	atomic_set(&desc->context.do_abort,0);
	
	if ( !srb->bufflen ){
		desc->context.data = 0;
		desc->context.data_length = 0;
		return;
	} else {
		desc->context.data = srb->buffer;
		desc->context.data_length = srb->bufflen;
	}
	
	/* can't rely on srb->sc_data_direction */

	/* Brutally ripped from usb-storage */

		if ( USC_DIRECTION_IS_IN(srb->cmnd[0]) ) {
 		pipe = desc->image_pipe;
		USC_DEBUG( "transfering from desc->ep_image == %d\n",
			   (int)desc->ep_image );
	} else {
		USC_DEBUG("transfering to desc->ep_out == %d\n",
			  (int)desc->ep_out);
		pipe = desc->out_pipe;
	}
	desc->context.data_pipe = pipe;
}


static
int usc_scsi_queuecommand( Scsi_Cmnd *srb, usc_scsi_cmnd_callback callback )
{
	struct usc_desc* desc = (struct usc_desc*)(srb->host->hostdata[0]);
	int err = 0;
	int res;

	USC_DEBUG_GOT_HERE();
	usc_show_command(srb);
	usc_debug_dump(desc);
	
	if ( srb->device->lun || srb->device->id || srb->device->channel ) {

		USC_DEBUG("Command to LUN=%d ID=%d CHANNEL=%d from SCSI layer\n",(int)srb->device->lun,(int)srb->device->id, (int)srb->device->channel );
		
		USC_DEBUG("this device doesn't exist\n");

		srb->result = DID_BAD_TARGET << 16;

		if(callback)
			callback(srb);

		goto out;
	}	

USC_DEBUG("Line %d: trying to engage the cmd_lock.\n",__LINE__);
	down(&desc->cmd_lock);
		desc->context.lock = 0;

	USC_DEBUG_GOT_HERE();
	usc_show_command(srb);
	

	FILL_BULK_URB(&desc->cmd_urb,
		      desc->usb_dev,
		      desc->out_pipe,
		      srb->cmnd,
		      srb->cmd_len,
		      usc_command_done,
		      &desc->context
		      );
	
		
	usc_build_transfer_context( srb, desc );
	desc->context.final_callback = callback;
	desc->cmd_urb.timeout = 100;
	desc->cmd_urb.transfer_flags = USB_ASYNC_UNLINK;
	
/*	desc->cmd_urb.transfer_flags = USB_DISABLE_SPD;*/

	res=usb_submit_urb(&desc->cmd_urb);
	
	if(res){
		USC_ERROR("error %d submitting URB\n",(int)res);
		srb->result = DID_ERROR << 16;
		USC_DEBUG_GOT_HERE();
desc->context.lock = 1;

		if(callback)
			callback(srb);
	up(&desc->cmd_lock); /* no further cleanup is done */

		goto out;
	}	
	
	USC_DEBUG_GOT_HERE();

 out:
	return err;
}
/*
 * this defines our 'host'
 */

/* NOTE: This is taken from usb-storage, should be right. */


static Scsi_Host_Template usc_scsi_host_template = {
	name:           "hp5300",
	detect:		usc_scsi_detect,
	release:	usc_scsi_release,
	command:	0,
	queuecommand:	usc_scsi_queuecommand,

	eh_abort_handler:	usc_scsi_abort,
	eh_device_reset_handler:0,
	eh_bus_reset_handler:	0,
	eh_host_reset_handler:	usc_scsi_host_reset,

	can_queue:		1,
	this_id:		-1,
	cmd_per_lun:		1,
	present:		0,
	unchecked_isa_dma:	FALSE,
	use_clustering:		FALSE,
	use_new_eh_code:	TRUE,
	emulated:		TRUE
};


/* USB layer driver interface implementation */

static void usc_usb_disconnect (struct usb_device *dev, void *ptr)
{
	struct usc_desc* to_remove = (struct usc_desc*)ptr;

	USC_DEBUG_GOT_HERE();

	/* leave the list - lock it */
	down(&usc_list_semaphore);

	usc_remove_nolock(to_remove);

	up(&usc_list_semaphore);
}

struct vendor_product
{
	u16 idVendor;
	u16 idProduct;
	char* name;
	enum
	{
		usc_sup_unknown=0,
		usc_sup_alpha,
		usc_sup_full
	}
	support_status;
} ;


/* These are taken from the msmUSB.inf file on the Windows driver CD */
const static struct vendor_product usc_supported_products[] =
{
	{
		0x3f0, 0x701,"HP 5300C",usc_sup_unknown
	}
}
;


MODULE_DEVICE_TABLE (usb, hp5300_usb_id);

const static struct vendor_product* usc_last_product = &usc_supported_products[ sizeof(usc_supported_products) / sizeof(struct vendor_product) ];
		/* Must never be derefed, points to one after last entry */


static void * usc_usb_probe (struct usb_device *dev, unsigned int interface, const struct usb_device_id *id)
{
	int i;
	int result;
	int ep_out = -1;
	int interrupt_interval = 0;
	int ep_in_set[3]; /* this will break if we have more than three endpoints
			   which is why we check */
	int *ep_in_current = ep_in_set;

	struct usc_desc * new_desc;
	struct vendor_product const* p;

	/* the altsettting 0 on the interface we're probing */
	struct usb_interface_descriptor *altsetting;

	USC_DEBUG_GOT_HERE();
	USC_DEBUG( "usb-device descriptor at %x\n", (int)dev );
	
	USC_DEBUG( "product id = 0x%x, vendor id = 0x%x\n",
		   (int)dev->descriptor.idProduct,
		   (int)dev->descriptor.idVendor );

	USC_DEBUG_GOT_HERE();
	
	/* checking IDs */
	for( p = usc_supported_products; p != usc_last_product; p++ )
		if ( dev->descriptor.idVendor == p->idVendor &&
		     dev->descriptor.idProduct == p->idProduct )
			goto is_supported;
		else
			USC_DEBUG( "doesn't appear to be model %s\n", p->name );

	USC_DEBUG( "returning NULL: unsupported\n" );
	
	return NULL;
	
 is_supported:

	USC_DEBUG_GOT_HERE();

	USC_DEBUG( "found model %s\n", p->name );
	if ( p->support_status != usc_sup_full )
		USC_MESSAGE( "model %s is not known to be fully supported, reports welcome!\n",
			     p->name );
	
	/* the altsettting 0 on the interface we're probing */
	altsetting =
		&(dev->actconfig->interface[interface].altsetting[0]);	

	
	/* Check if the config is sane */

	if ( altsetting->bNumEndpoints != USC_EP_TOTAL ) {
		USC_WARNING( "expecting %d got %d endpoints! Bailing out.\n",
			     (int)USC_EP_TOTAL, (int)altsetting->bNumEndpoints );
		return NULL;
	}

	for( i = 0; i < altsetting->bNumEndpoints; i++ ) {
		if ((altsetting->endpoint[i].bmAttributes & 
		     USB_ENDPOINT_XFERTYPE_MASK) == USB_ENDPOINT_XFER_INT)
			interrupt_interval = altsetting->endpoint[i].bInterval;
		if (((altsetting->endpoint[i].bmAttributes & 
		     USB_ENDPOINT_XFERTYPE_MASK) != USB_ENDPOINT_XFER_BULK) && 
		((altsetting->endpoint[i].bmAttributes & 
		     USB_ENDPOINT_XFERTYPE_MASK) != USB_ENDPOINT_XFER_INT)) {
			
			USC_WARNING( "can only deal with 2 bulk and 1 interrupt endpoints; endpoint %d is not either.\n",
			     (int)altsetting->endpoint[i].bEndpointAddress );
		} else {
			if (altsetting->endpoint[i].bEndpointAddress & 
			    USB_DIR_IN)
				*ep_in_current++
					= altsetting->endpoint[i].bEndpointAddress &
					USB_ENDPOINT_NUMBER_MASK;
			else {
				if ( ep_out != -1 ) {
					USC_WARNING( "can only deal with one output endpoints. Bailing out." );
					return NULL;
				}
		
				ep_out = altsetting->endpoint[i].bEndpointAddress &
					USB_ENDPOINT_NUMBER_MASK;
			}
		}
		
	}
	

	if ( ep_out == -1 ) {
		USC_WARNING( "couldn't find an output bulk endpoint. Bailing out.\n" );
		return NULL;
	}
	
	
USC_DEBUG_GOT_HERE();
	/* I don't understand the following fully (it's from usb-storage) -- John */

	/* set the interface -- STALL is an acceptable response here */
	result = usb_set_interface(dev, altsetting->bInterfaceNumber, 0);
	
	USC_DEBUG("usb_set_interface returned %d.\n",result);
	switch( result )
	{
	case 0: /* no error */
		break;
		
	case -EPIPE:
		usb_clear_halt(dev, usb_sndctrlpipe(dev, 0));
		USC_DEBUG( "clearing clearing stall on control interface\n" );
		break;
		
	default:
		USC_DEBUG( "unknown error %d from usb_set_interface\n",
			(int)result );
 		return NULL;
	}
	
	
	/* allocating a new descriptor */
	new_desc = (struct usc_desc *)kmalloc(sizeof(struct usc_desc), GFP_KERNEL);
	if (new_desc == NULL)
	{
		USC_ERROR("couldn't allocate scanner desc, bailing out!\n");
		return NULL;
	}

	/* As done by usb_alloc_urb */
	memset( new_desc, 0, sizeof(*new_desc) );
	spin_lock_init(&new_desc->cmd_urb.lock);
	spin_lock_init(&new_desc->status_urb.lock);
	
		
	/* initialising that descriptor */
	new_desc->usb_dev = dev;
	new_desc->interface = interface;

	init_MUTEX(&new_desc->cmd_lock);
	init_MUTEX(&new_desc->status_lock);

	if(usc_list){
		new_desc->host_number = usc_list->host_number+1;
	} else {
		new_desc->host_number = 0;
	}
	
	/* endpoints */
	
	new_desc->ep_out = ep_out;
	new_desc->out_pipe = usb_sndbulkpipe(dev, ep_out);
	new_desc->ep_response = ep_in_set[0];
	new_desc->status_pipe = usb_rcvintpipe(dev, ep_in_set[0]);
	new_desc->ep_image = ep_in_set[1];
	new_desc->image_pipe = usb_rcvbulkpipe(dev, ep_in_set[1]);
	new_desc->interrupt_interval = interrupt_interval;
	

	if ( new_desc->ep_out != USC_EP_OUT )
		USC_WARNING( "will this work? Command EP is not usually %d\n",
			     (int)new_desc->ep_out );

	if ( new_desc->ep_response != USC_EP_RESPONSE )
		USC_WARNING( "will this work? Response EP is not usually %d\n",
			     (int)new_desc->ep_response );

	if ( new_desc->ep_image != USC_EP_IMAGE )
		USC_WARNING( "will this work? Image data EP is not usually %d\n",
			     (int)new_desc->ep_image );

/* start the status handler */
down(&new_desc->status_lock);
usc_get_status(new_desc, &new_desc->status_urb, new_desc->status_pipe, &new_desc->context.status, 1);	

	/* Initialize the host template based on the default one */
	memcpy(&(new_desc->ctempl), &usc_scsi_host_template, sizeof(usc_scsi_host_template)); 	
	/* HACK from usb-storage - this is needed for scsi detection */
	(struct usc_desc *)new_desc->ctempl.proc_dir = new_desc; /* FIXME */
	
	USC_DEBUG("registering SCSI module\n");
	
	new_desc->ctempl.module = THIS_MODULE;
	result = scsi_register_module(MODULE_SCSI_HA, &(new_desc->ctempl));	
	/* Will get hit back in microtek_detect by this func */
	if ( result )
	{
		USC_ERROR( "error %d from scsi_register_module! Help!\n",
			   (int)result );
			   usb_unlink_urb(&new_desc->status_urb);

		/* FIXME: need more cleanup? */
		kfree( new_desc );
		return NULL;
	}
	USC_DEBUG_GOT_HERE();

	/* FIXME: the bomb is armed, must the host be registered under lock ? */
	/* join the list - lock it */
	down(&usc_list_semaphore);

	usc_add_nolock( new_desc );
	
	up(&usc_list_semaphore);
	
	
	USC_DEBUG("completed probe and exiting happily\n");
	
	return (void *)new_desc;
}



/* get us noticed by the rest of the kernel */

int __init usbscsibulkint_drv_init(void)
{
	int result;
	
	USC_DEBUG_GOT_HERE();
	init_MUTEX(&usc_list_semaphore);

	if ((result = usb_register(&usc_usb_driver)) < 0) {
		USC_DEBUG("usb_register returned %d\n", result );
		return -1;
	} else {
		USC_DEBUG("driver registered.\n");
	}
	
	return 0;
}

void __exit usbscsibulkint_drv_exit(void)
{
	struct usc_desc* next;
	
	USC_DEBUG_GOT_HERE();

	usb_deregister(&usc_usb_driver);

	down(&usc_list_semaphore);

	while (usc_list) {
		/* keep track of where the next one is */
		next = usc_list->next;

		usc_remove_nolock( usc_list );

		/* advance the list pointer */
		usc_list = next;
	}
	
	up(&usc_list_semaphore);
}

module_init(usbscsibulkint_drv_init);
module_exit(usbscsibulkint_drv_exit);

MODULE_AUTHOR("Jeremy Hall, <JHall@UU.NET>");
MODULE_DESCRIPTION("SCSI over USB driver for scanners");
