/********************************************************************************
* QLogic QLA2x00 device driver for Linux 2.2.x and 2.4.x 
* Copyright (C) 2000,2001 Qlogic Corporation 
* (www.qlogic.com)
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation; either version 2, or (at your option) any
* later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
**
******************************************************************************/
/****************************************************************************
Revision History:
    Rev. 1.3 Beta       February 20, 2001   BN QLogic
        - Zero the sp used for IOCTL SCSI Passthru operations
    Rev. 1.2 Beta       January 31, 2001   BN QLogic
        - Added setting of scsi completion to pext->DetailStatus
          and set pext->Status to EXT_STATUS_SCSI_STATUS.
        - Added details to Statistics IOCTL for ispaborts,
          lip_count, isr_count.

    Rev. 1.2 Beta       January 8, 2001    BN QLogic
        - Added loop-back diagnostic IOCTL support.

    Rev. 1.1 Beta       October 27, 2000    BN QLogic
        - Updated return status from ioctl function.

    Rev. 1.0 Beta       October 20, 2000    BN QLogic
        - Initial version check in
  

*/

void copy_up_EXT( PEXT_IOCTL pext, void *arg) {
uint32_t i;
uint8_t  *usr_temp, *kernel_tmp;

  /* copy up the EXT_IOCTL to application (api library) */
  for (i=0 ; i < sizeof(EXT_IOCTL) ; i++) {
    usr_temp   = (uint8_t *)arg + i;
    kernel_tmp = (uint8_t *)pext + i;
    __put_user(*kernel_tmp, usr_temp);
  }
  
  return;
}


/*************************************************************************
 *   ioctl_scsi_pt_done
 *
 * Description:
 *   Sets completion flag.
 *
 * Returns:
 *************************************************************************/
void  ioctl_scsi_pt_done(Scsi_Cmnd *pscsi_cmd) {
  struct Scsi_Host *host;
  scsi_qla_host_t  *ha;

      host = pscsi_cmd->host;
      ha = (scsi_qla_host_t *) host->hostdata; 

      ha->IoctlPassThru_InProgress = 0;
      return;
}


/*************************************************************************
 *   ioctl_fcct_done
 *
 * Description:
 *   Sets completion flag.
 *
 * Returns:
 *************************************************************************/
void  ioctl_fcct_done(Scsi_Cmnd *pscsi_cmd) {
  struct Scsi_Host *host;
  scsi_qla_host_t  *ha;

      host = pscsi_cmd->host;
      ha = (scsi_qla_host_t *) host->hostdata; 
      /* printk("ioctl_fcct_done post function called OK\n"); */
      DEBUG(sprintf(debug_buff,"ioctl_fcct_done post function called OK\n"));
      DEBUG(qla2100_print(debug_buff));
      ha->IoctlPassFCCT_InProgress = 0;
      return;
}


/*************************************************************************
 *   qla2100_ioctl
 *
 * Description:
 *   Performs ioctl requests not satified by the upper levels.
 *
 * Returns:
 *   ret  = 0    Success
 *   ret != 0    Failed; detailed status copied to EXT_IOCTL structure
 *               if applicable     
 *************************************************************************/
int
qla2100_ioctl(Scsi_Device *dev, int cmd, void *arg){

  static  EXT_IOCTL    ext;
  PEXT_IOCTL   pext = &ext;
  struct Scsi_Host   *host;
  scsi_qla_host_t  *ha, *search_ha; 
  scsi_lu_t        *q;
#if BITS_PER_LONG <= 32
  uint32_t    handle;
#else
  uint64_t    handle;
#endif
  uint32_t    cnt, i, b, t, l, port_cnt, status;
  uint32_t    tgt_cnt, tgt, transfer_size, inst;
  uint8_t     *extptr, *usrsrc, *usr_temp, *kernel_tmp, *kernel_tmp1;
  static Scsi_Cmnd          scsi_cmd;
  Scsi_Cmnd   *pscsi_cmd = &scsi_cmd;
  static Scsi_Device scsi_device;
  static srb_t             ioctl_sp;
  srb_t             *sp = &ioctl_sp;
  static         EXT_SCSI_PASSTHRU   scsi_pass;
  EXT_SCSI_PASSTHRU   *pscsi_pass = &scsi_pass;
  static EXT_HBA_NODE      tmp_hba_node;
  static EXT_HBA_PORT      tmp_hba_port;
  static EXT_DISC_PORT     tmp_disc_port;
  static EXT_DISC_TARGET   tmp_disc_target;
  static EXT_CHIP          tmp_isp;
  static EXT_HBA_PORT_STAT tmp_stat;
  uint16_t    mb[MAILBOX_REGISTER_COUNT];
  unsigned long cpu_flags = 0;
  uint16_t       scratch;
  uint16_t       *wptr = &scratch;
  qla_boards_t   *bdp;
  uint8_t *temp;
  uint8_t tempbuf[8];
  cmd_ms_iocb_entry_t  *pkt;
 
  int  ret = EINVAL;  
 
  BZERO((caddr_t)sp,sizeof(srb_t));

  host = dev->host;
  ha = (scsi_qla_host_t *) host->hostdata; /* midlayer chosen instance */

  ret = verify_area(VERIFY_READ, (void *)arg, sizeof(EXT_IOCTL)); 
  if (ret) {
    DEBUG2(printk("[qla2100_ioctl: ERROR in verify_area READ ha=%8x]\n",(uint32_t)ha);)
    return(ret);
  }
  /* copy in application layer EXT_IOCTL */
  for (i=0 ; i < sizeof(EXT_IOCTL) ; i++) {
    usrsrc = (uint8_t *)(arg + i);
    extptr = (uint8_t *)pext + i;
    __get_user(*extptr, usrsrc);
  }

  /* printk("[GOT QLA2100 IOCTL sig=%s cmd=%x]\n",
     (char *)&pext->Signature,cmd); */

  /* check signature of this ioctl */
  temp = (uint8_t *) &pext->Signature;
#if BITS_PER_LONG <= 32
for (i=0 ; i<4 ; i++,temp++) tempbuf[i] = *temp;
  if( (tempbuf[0] == 'Q') && (tempbuf[1] == 'L') && 
      (tempbuf[2] == 'O') && (tempbuf[3] == 'G')) status = 0;
  else status = 1;
#else
  if( (tempbuf[0] == 'Q') && (tempbuf[1] == 'L') && 
      (tempbuf[2] == 'O') && (tempbuf[3] == 'G') &&
      (tempbuf[4] == 'I') && (tempbuf[5] == 'C')) status = 0;
      else status = 1; 
#endif 
  if ( status != 0) {
    DEBUG2(printk("[GOT QLA2100 IOCTL but signature did not match ha=%8x]\n",
          (uint32_t)ha);)
    ret = EXT_STATUS_ERR;
    return(ret);
  }
  /* check version of this ioctl */
  if (pext->Version  !=  EXT_VERSION) {
    pext->Status       = EXT_STATUS_ERR;
    pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
    copy_up_EXT(pext,arg);
    printk("[GOT QLA2100 IOCTL but version did not match]\n");
    ret = EXT_STATUS_ERR;
    return(ret);
  }
  /* check for API setting HBA Instance for subsequent operations */
  if (cmd == (int)EXT_CC_SETINSTANCE) {
      /* since API opens devices once and uses handle for subsequent calls */
      /* we keep a parameter to designate the "active HBA" for ioctls */
      if (pext->HbaSelect < num_hosts) {
         apiHBAInstance     = pext->HbaSelect;
         pext->Status       = EXT_STATUS_OK;
         pext->DetailStatus = EXT_STATUS_OK;
         copy_up_EXT(pext,arg);
         ret = EXT_STATUS_OK;
      } else {
         pext->Status       = EXT_STATUS_ERR;
         pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
         copy_up_EXT(pext,arg);
         DEBUG2(printk("[qla2100_ioctl: ERROR in EXT_SETINSTANCE ha=%8x]\n",(uint32_t)ha);)
         ret = EXT_STATUS_ERR;
      }
      return(ret); /* Instance for subsequent IOCTLs are not set */
  }

  /* check for valid  apiHBAInstance (set previously by EXT_SETINSTANCE 
     or default 0)  and set   ha   context for this IOCTL */    
  for (search_ha=qla2100_hostlist; 
      (search_ha != NULL) && search_ha->instance != apiHBAInstance;
       search_ha = search_ha->next) ;
  if ( !search_ha ) { 
     pext->Status       = EXT_STATUS_ERR;
     pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
     copy_up_EXT(pext,arg);
     DEBUG2(printk("[qla2100_ioctl: ERROR in matching apiHBAInstance to an HBA Instance]\n");)
     ret = EXT_STATUS_ERR;
     return(ret);
  }
  /* IOCTL ha context is ready to be set from apiHBAInstance */
  ha = search_ha;
  /* set EXT_IOCTL.HbaSelect  for reference by IOCTL caller */
  pext->HbaSelect = apiHBAInstance;

  switch (cmd) { /* switch on EXT IOCTL COMMAND CODE */
    case EXT_CC_STARTIOCTL:
      pext->Instance     = num_hosts;
      pext->Status       = EXT_STATUS_OK;
      pext->DetailStatus = EXT_STATUS_OK;
      copy_up_EXT(pext,arg);
      /*printk("[QLA2100 EXT_STARTIOCTL did OK ]\n");*/ 
      ret = EXT_STATUS_OK;
      break;
    case EXT_CC_QUERY:
      /* All Query type ioctls are done here */
      switch(pext->SubCode) {
        case EXT_SC_QUERY_HBA_NODE:
        /* fill all available HBA NODE Information */
          bdp = &QLBoardTbl_fc[ha->devnum];
          for (i=0; i < 8 ;i++)   tmp_hba_node.WWNN[i] = ha->node_name[i];
          sprintf((char *)(tmp_hba_node.Manufacturer),"Qlogic Corp.");
          sprintf((char *)(tmp_hba_node.Model),(char *)&bdp->bdName[0]);
          tmp_hba_node.SerialNum[0] = ha->node_name[5];
          tmp_hba_node.SerialNum[1] = ha->node_name[6];
          tmp_hba_node.SerialNum[2] = ha->node_name[7];
          sprintf((char *)(tmp_hba_node.DriverVersion),QLA2100_VERSION);
          sprintf((char *)(tmp_hba_node.FWVersion),"%2d.%02d.%02d",
                  bdp->fwver[0], bdp->fwver[1], bdp->fwver[2]);
          /* tmp_hba_node.NvramVersion = ha->nvram_version; */  
          sprintf((char *)(tmp_hba_node.OptRomVersion),"0");
          tmp_hba_node.InterfaceType = EXT_DEF_FC_INTF_TYPE;
          tmp_hba_node.PortCount = 1;
          
          ret = verify_area(VERIFY_WRITE, (void  *)pext->ResponseAdr,
                sizeof(EXT_HBA_NODE)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }
          /* now copy up the HBA_NODE to user */
          if (pext->ResponseLen < sizeof(EXT_HBA_NODE)) transfer_size = pext->ResponseLen;
          else       transfer_size = sizeof(EXT_HBA_NODE);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_hba_node + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /*printk("[finished QLA2100 IOCTL QUERY_HBA_NODE ]\n"); */
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        case EXT_SC_QUERY_HBA_PORT:
          /* reflect all HBA PORT related info */
          tmp_hba_port.WWPN[7] = ha->init_cb->port_name[7];  
          tmp_hba_port.WWPN[6] = ha->init_cb->port_name[6];
          tmp_hba_port.WWPN[5] = ha->init_cb->port_name[5];
          tmp_hba_port.WWPN[4] = ha->init_cb->port_name[4];
          tmp_hba_port.WWPN[3] = ha->init_cb->port_name[3];
          tmp_hba_port.WWPN[2] = ha->init_cb->port_name[2];
          tmp_hba_port.WWPN[1] = ha->init_cb->port_name[1];
	  tmp_hba_port.WWPN[0] = ha->init_cb->port_name[0];
          tmp_hba_port.Id[0] = ha->port_id[0];
          tmp_hba_port.Id[1] = ha->port_id[1];
          tmp_hba_port.Id[2] = ha->port_id[2];
          tmp_hba_port.Type =  EXT_DEF_INITIATOR_DEV;

          port_cnt = 0; 
          tgt_cnt  = 0;
          for (tgt = 0; tgt < MAX_FIBRE_DEVICES ; tgt++) {
          if (ha->fc_db[tgt].loop_id > LAST_SNS_LOOP_ID) 
               continue;
            port_cnt++;

          /* at this point the linux driver does not differentiate */
          /* Initiator devices (ie HBAs) from Targets; so the count */
          /* is same for both */
          /*      if (ha->fc_db[tgt] & FC_INITIATOR_DEVICE) 
                  continue;  */
            tgt_cnt++;
          }
          tmp_hba_port.DiscPortCount   = port_cnt;
          tmp_hba_port.DiscTargetCount = tgt_cnt;

          if ( ha->loop_down_timer == 0 && ha->loop_state == LOOP_DOWN) {
                  tmp_hba_port.State    = EXT_DEF_HBA_LOOP_DOWN;
          } else  tmp_hba_port.State    = EXT_DEF_HBA_OK;
          tmp_hba_port.DiscPortNameType = EXT_DEF_USE_PORT_NAME;

          ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                sizeof(EXT_HBA_PORT)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }
          /* now copy up the HBA_PORT to user */
          if (pext->ResponseLen < sizeof(EXT_HBA_PORT)) transfer_size = pext->ResponseLen;
          else         transfer_size = sizeof(EXT_HBA_PORT);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_hba_port + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /*printk("[finished QLA2100 IOCTL QUERY_HBA_PORT ports=%x tgts=%x]\n",port_cnt,tgt_cnt); */
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        case EXT_SC_QUERY_DISC_PORT:
          for (tgt=0, inst=0; tgt < MAX_FIBRE_DEVICES; tgt++) {
            if (ha->fc_db[tgt].loop_id > LAST_SNS_LOOP_ID)
               continue; 
            if( inst != pext->Instance ) {
              inst++;
              continue;
            }
            break;
          }
          if (tgt == MAX_FIBRE_DEVICES) {
            pext->Status       = ret = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
            copy_up_EXT(pext,arg);
            return(ret); 
          }

            tmp_disc_port.WWNN[0] =  ha->fc_db[tgt].name[1] & 0x000000ff;
            tmp_disc_port.WWNN[1] = (ha->fc_db[tgt].name[1] & 0x0000ff00) >> 8;
            tmp_disc_port.WWNN[2] =(ha->fc_db[tgt].name[1] & 0x00ff0000) >> 16;
            tmp_disc_port.WWNN[3] =(ha->fc_db[tgt].name[1] & 0xff000000) >> 24;

            tmp_disc_port.WWNN[4] = (ha->fc_db[tgt].name[0] & 0x000000ff);
            tmp_disc_port.WWNN[5] = (ha->fc_db[tgt].name[0] & 0x0000ff00) >> 8;
            tmp_disc_port.WWNN[6] =(ha->fc_db[tgt].name[0] & 0x00ff0000) >> 16;
            tmp_disc_port.WWNN[7] =(ha->fc_db[tgt].name[0] & 0xff000000) >> 24;

            tmp_disc_port.WWPN[0] =  ha->fc_db[tgt].wwn[1] & 0x000000ff;
            tmp_disc_port.WWPN[1] = (ha->fc_db[tgt].wwn[1] & 0x0000ff00) >> 8;
            tmp_disc_port.WWPN[2] = (ha->fc_db[tgt].wwn[1] & 0x00ff0000) >> 16;
            tmp_disc_port.WWPN[3] = (ha->fc_db[tgt].wwn[1] & 0xff000000) >> 24;

            tmp_disc_port.WWPN[4] = (ha->fc_db[tgt].wwn[0] & 0x000000ff);
            tmp_disc_port.WWPN[5] = (ha->fc_db[tgt].wwn[0] & 0x0000ff00) >> 8;
            tmp_disc_port.WWPN[6] = (ha->fc_db[tgt].wwn[0] & 0x00ff0000) >> 16;
            tmp_disc_port.WWPN[7] = (ha->fc_db[tgt].wwn[0] & 0xff000000) >> 24;

          tmp_disc_port.Id[0] = ha->fc_db[tgt].port_id[0];
          tmp_disc_port.Id[1] = ha->fc_db[tgt].port_id[1];
          tmp_disc_port.Id[2] = ha->fc_db[tgt].port_id[2];
          tmp_disc_port.Type  = EXT_DEF_TARGET_DEV;

          tmp_disc_port.Status   = EXT_STATUS_OK;
          tmp_disc_port.Bus      = 0;
          tmp_disc_port.TargetId = tgt;
 
          ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                sizeof(EXT_DISC_PORT)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }

          /* now copy up the DISC_PORT to user */
          if (pext->ResponseLen < sizeof(EXT_DISC_PORT)) transfer_size = pext->ResponseLen;
          else        transfer_size = sizeof(EXT_DISC_PORT);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_disc_port + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /*printk("[finished QLA2100 IOCTL QUERY_DISC_PORT ]\n"); */
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        case EXT_SC_QUERY_DISC_TGT:
	  /* (for multipath merge) to do: search the ha->tgt[][] data base 
             instead of ha->fc_db[] so that we give details on Targets that
             are visible to the OS. */
          for (tgt=0, inst=0; tgt < MAX_FIBRE_DEVICES; tgt++) {
            if (ha->fc_db[tgt].loop_id > LAST_SNS_LOOP_ID)
               continue; 
            if( inst != pext->Instance ) {
              inst++;
              continue;
            }
            break;
          }
          if (tgt == MAX_FIBRE_DEVICES) {
            pext->Status       = ret = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
            copy_up_EXT(pext,arg);
            return(ret); 
          }
            tmp_disc_target.WWNN[0]=ha->fc_db[tgt].name[1] & 0x000000ff;
            tmp_disc_target.WWNN[1]=(ha->fc_db[tgt].name[1] & 0x0000ff00) >> 8;
            tmp_disc_target.WWNN[2]=(ha->fc_db[tgt].name[1] & 0x00ff0000) >> 16;
            tmp_disc_target.WWNN[3]=(ha->fc_db[tgt].name[1] & 0xff000000) >> 24;

            tmp_disc_target.WWNN[4]=(ha->fc_db[tgt].name[0] & 0x000000ff);
            tmp_disc_target.WWNN[5]=(ha->fc_db[tgt].name[0] & 0x0000ff00) >> 8;
            tmp_disc_target.WWNN[6]=(ha->fc_db[tgt].name[0] & 0x00ff0000) >> 16;
            tmp_disc_target.WWNN[7]=(ha->fc_db[tgt].name[0] & 0xff000000) >> 24;


            tmp_disc_target.WWPN[0]=ha->fc_db[tgt].wwn[1] & 0x000000ff;
            tmp_disc_target.WWPN[1]=(ha->fc_db[tgt].wwn[1] & 0x0000ff00) >> 8;
            tmp_disc_target.WWPN[2]=(ha->fc_db[tgt].wwn[1] & 0x00ff0000) >> 16;
            tmp_disc_target.WWPN[3]=(ha->fc_db[tgt].wwn[1] & 0xff000000) >> 24;

            tmp_disc_target.WWPN[4]=(ha->fc_db[tgt].wwn[0] & 0x000000ff);
            tmp_disc_target.WWPN[5]=(ha->fc_db[tgt].wwn[0] & 0x0000ff00) >> 8;
            tmp_disc_target.WWPN[6]=(ha->fc_db[tgt].wwn[0] & 0x00ff0000) >> 16;
            tmp_disc_target.WWPN[7]=(ha->fc_db[tgt].wwn[0] & 0xff000000) >> 24;
            tmp_disc_target.Id[0] = ha->fc_db[tgt].port_id[0];
            tmp_disc_target.Id[1] = ha->fc_db[tgt].port_id[1];
            tmp_disc_target.Id[2] = ha->fc_db[tgt].port_id[2];
            tmp_disc_target.Type  = EXT_DEF_TARGET_DEV;

          tmp_disc_target.Status   = EXT_STATUS_OK;
          tmp_disc_target.Bus      = 0;
          tmp_disc_target.TargetId = tgt;
 
          cnt = 0;
          for (i=0; i < MAX_LUNS ; i++) {
          if ((ha->tgt[0][tgt])->luns[i] !=0) 
              cnt++;
          }
          tmp_disc_target.LunCount = cnt;

          ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                sizeof(EXT_DISC_TARGET)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }
          /* now copy up the DISC_TGT to user */
          if (pext->ResponseLen < sizeof(EXT_DISC_PORT)) transfer_size = pext->ResponseLen;
          else     transfer_size = sizeof(EXT_DISC_TARGET);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_disc_target + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /* printk("[finished QLA2100 IOCTL QUERY_DISC_TGT Luns=%x pid0=%x pid1=%x pid2=%x]\n",cnt,ha->fc_db[tgt].port_id[0],ha->fc_db[tgt].port_id[1],ha->fc_db[tgt].port_id[2]);*/
 
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        case EXT_SC_QUERY_CHIP:
          host = ha->host;
	  tmp_isp.VendorId       = QLA2100_VENDOR_ID;
	  tmp_isp.DeviceId       = ha->device_id;
	  tmp_isp.SubVendorId    = QLA2100_VENDOR_ID;
          tmp_isp.SubSystemId    = 0;	
	  tmp_isp.PciBusNumber   = ha->pci_bus;	
	  tmp_isp.PciSlotNumber  = (ha->pci_device_fn & 0xf8) >> 3;
	  tmp_isp.IoAddr         = host->io_port;	
	  tmp_isp.IoAddrLen      = 512;	
          tmp_isp.MemAddr        = 0; 
          tmp_isp.MemAddrLen     = 0; 	
          tmp_isp.ChipType       = 0; 	
          tmp_isp.InterruptLevel = host->irq;
          for (i=0;i<8;i++)      tmp_isp.OutMbx[i] = 0;		

          ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                sizeof(EXT_CHIP)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }
          /* now copy up the ISP to user */
          if (pext->ResponseLen < sizeof(EXT_CHIP)) transfer_size = pext->ResponseLen;
          else         transfer_size = sizeof(EXT_CHIP);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_isp + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /* printk("[finished QLA2100 IOCTL QUERY_CHIP]\n"); */ 
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;

        case EXT_SC_QUERY_DISC_LUN:
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        default:
          ret = EXT_STATUS_ERR;
          break;
      }
      break;
    case EXT_CC_GET_DATA:
      switch(pext->SubCode) {
        case EXT_SC_GET_STATISTICS:

  	  tmp_stat.ControllerErrorCount   =  ha->isp_aborts;
	  tmp_stat.DeviceErrorCount       =  0;
	  tmp_stat.TotalIoCount           =  ha->qthreads;
	  tmp_stat.TotalMBytes            =  0;
	  tmp_stat.TotalLipResets         =  ha->lip_count;
	  tmp_stat.TotalInterrupts        =  ha->isr_count;

          ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr,
                sizeof(EXT_HBA_PORT_STAT)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE ha=%8x]\n",(uint32_t)ha);)
            return(ret);
          }
          /* now copy up the STATISTICS to user */
          if (pext->ResponseLen < sizeof(EXT_HBA_PORT_STAT)) transfer_size = pext->ResponseLen;
          else       transfer_size = sizeof(EXT_HBA_PORT_STAT);
          for (i=0 ; i < transfer_size ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)&tmp_stat + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /*printk("[finished QLA2100 IOCTL STATISTICS ]\n");*/ 
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
          break;
        default:
          break;
      }
      break;
    case EXT_CC_SEND_FCCT_PASSTHRU:
    DEBUG(sprintf(debug_buff,"qla2100_ioctl: start EXT_CC_SEND_FCCT_PASSTHRU\n"));
    DEBUG(qla2100_print(debug_buff));
    /* Management Server type (fc switch) pass thru ioctl */
    /* same as EXT_FCSCSI_REQ but it is going to the FC switch */
    /* clear ioctl_sp and scsi_cmd to be used */
    kernel_tmp = (uint8_t *)sp;
    for (i=0;i<sizeof(srb_t);i++)  *(kernel_tmp+i) = 0; 
    kernel_tmp = (uint8_t *)ha->ioctl_mem;
    for (i=0;i<PAGE_SIZE;i++)      *(kernel_tmp+i) = 0; 
    kernel_tmp = (uint8_t *)pscsi_cmd;
    for (i=0;i<sizeof(Scsi_Cmnd);i++)   *(kernel_tmp+i) = 0; 
    kernel_tmp = (uint8_t *)&scsi_device;
    for (i=0;i<sizeof(Scsi_Device);i++) *(kernel_tmp+i) = 0; 

      /*printk("[start EXT_CC_SEND_FCCT_PASSTHRU]\n"); */
      if (pext->ResponseLen > PAGE_SIZE)  pext->ResponseLen = PAGE_SIZE;
      if (pext->RequestLen  > PAGE_SIZE)  {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
        copy_up_EXT(pext,arg);
        /*printk("[EXT_CC_SEND_FCCT_PASSTHRU too big ResponseLen=%x ReqLen=%x]\n",pext->ResponseLen,pext->RequestLen); */
        DEBUG2(printk("[qla2100_ioctl: ERROR size of requested Resp_len in EXT_CC_SEND_FCCT_PASSTHRU]\n");)
        return(ret);
      }
      ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
                        pext->RequestLen); 
      if (ret){
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_COPY_ERR;
        copy_up_EXT(pext,arg);
        /*printk("[EXT_CC_SEND_FCCT_PASSTHRU verify read error]\n");*/ 
        DEBUG2(printk("[qla2100_ioctl: ERROR verify_area READ of EXT_CC_SEND_FCCT_PASSTHRU]\n");)
        return(ret);
      }
      for (i=0 ; i < pext->RequestLen ; i++) {
        /* copy in from user space the fcct command to be sent */
        usr_temp   = (uint8_t *)pext->RequestAdr + i;
        kernel_tmp = (uint8_t *)ha->ioctl_mem + i;
        __get_user(*kernel_tmp, usr_temp);
        /* printk("{%x}",*kernel_tmp); */
      }
      /* check on current topology or loop down */
      if ((ha->current_topology != ISP_CFG_F)  && 
          (ha->current_topology != ISP_CFG_FL ) ) {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
        copy_up_EXT(pext,arg);
        /*printk("[EXT_CC_SEND_FCCT_PASSTHRU wrong topology current=%x]\n",
          ha->current_topology); */
        DEBUG2(printk("[qla2100_ioctl: ERROR EXT_CC_SEND_FCCT_PASSTHRU not in F-Port or FL-Port mode]\n");)
        return(ret);
      }
      /* check on loop down */
      if ( ha->loop_down_timer == 0 && ha->loop_state == LOOP_DOWN)  {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
        copy_up_EXT(pext,arg);
        /* printk("[EXT_CC_SEND_FCCT_PASSTHRU loop down]\n"); */
        DEBUG2(printk("[qla2100_ioctl: ERROR EXT_CC_SEND_FCCT_PASSTHRU not in F-Port mode]\n");)
        return(ret);
      }
      /* login to management server device */
      if (ha->flags.managment_server_logged_in == 0) {
          mb[0] = MBC_LOGIN_FABRIC_PORT;
          mb[1] = MANAGEMENT_SERVER << 8;
          mb[2] = 0xff;
          mb[3] = 0xfffa;

          ret = qla2100_mailbox_command(ha, BIT_3|BIT_2|BIT_1|BIT_0,
                                &mb[0]);
          if ( (ret != 0) || (mb[0] == 0x4006)  ||
               (mb[0] == 0x4009) || (mb[0] != 0x4000) ) {
              pext->Status       = EXT_STATUS_ERR;
              pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
              copy_up_EXT(pext,arg);
              /* printk("[EXT_CC_SEND_FCCT_PASSTHRU could not login to sns]\n"); */
              DEBUG2(printk("[qla2100_ioctl: ERROR could not login to Management Server]\n");)
              DEBUG(sprintf(debug_buff,"FCCT ioctl: call qla2100_ms_req_pkt\n"));
              DEBUG(qla2100_print(debug_buff));
              return(ret);
          }
          ha->flags.managment_server_logged_in = 1;
      }

      /* setup  sp  for this FCCT pass thru */
      pscsi_cmd->host = ha->host;
      sp->cmd         = pscsi_cmd;
      sp->flags       = SRB_WATCHDOG;

      /* mark this as a special delivery and collection command */
      scsi_cmd.flags     = 0;      
      scsi_cmd.scsi_done = ioctl_fcct_done;

      DEBUG(sprintf(debug_buff,"FCCT ioctl: FABRIC_LOGIN OK, call qla2100_ms_req_pkt\n"));
      DEBUG(qla2100_print(debug_buff));

      /* get spin lock for this operation */
      spin_lock_irqsave(&io_request_lock, cpu_flags);

      /* Get MS request packet. */
      if( (pkt = (cmd_ms_iocb_entry_t *)qla2100_ms_req_pkt(ha, sp) ) ) {
            pkt->entry_type  = MS_IOCB_TYPE;
            pkt->entry_count = 1;
            pkt->loop_id     = MANAGEMENT_SERVER;
            pkt->timeout     = 4;
            pkt->DSDcount    = 1;
            pkt->RespDSDcount = 2;
            pkt->Response_bytecount = pext->ResponseLen;
            pkt->Request_bytecount  = pext->RequestLen;

            /* loading command payload address */
            pkt->dseg_req_address[0] = LS_64BITS(ha->ioctl_mem_phys);
            pkt->dseg_req_address[1] = MS_64BITS(ha->ioctl_mem_phys);
            pkt->dseg_req_length = pext->RequestLen;
  
            /* loading command response address */
            pkt->dseg_rsp_address[0] = LS_64BITS(ha->ioctl_mem_phys);
            pkt->dseg_rsp_address[1] = MS_64BITS(ha->ioctl_mem_phys);
            pkt->dseg_rsp_length = pext->ResponseLen;

            /* set flag to indicate IOCTL FCCT PassThru in progress */
            ha->IoctlPassFCCT_InProgress = 1;

            ha->ioctl_timer = 6; /* 6 ticks of 1 second timer max wait */

            /* Issue command to ISP */
            qla2100_isp_cmd(ha);
      }
      /* release spin lock since command is issued */  
      spin_unlock_irqrestore(&io_request_lock, cpu_flags);

      DEBUG(sprintf(debug_buff,
        "FCCT ioctl: Command issued and released spin lock\n"));
      DEBUG(qla2100_print(debug_buff));

      if (!pkt) {
          pext->Status       = EXT_STATUS_ERR;
          pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
          copy_up_EXT(pext,arg);
          /* printk("[EXT_CC_SEND_FCCT_PASSTHRU could not get Request Packet]\n"); */
          DEBUG2(printk("[qla2100_ioctl:EXT_CC_SEND_FCCT_PASSTHRU could not get Request Packet]\n");)
          return(ret);
      }

      /* wait for post function or timer to zero the InProgress flag */
      while (ha->IoctlPassFCCT_InProgress == 1)  {
        udelay(35);
      }

      DEBUG(sprintf(debug_buff,
        "FCCT ioctl: finished while(InProgress) wait loop \n"));
      DEBUG(qla2100_print(debug_buff));
      /* printk("[FCCT IOCTL finished while(InProgress) wait loop ]\n");*/

        if (ha->IoctlPassFCCT_InProgress == 1) {
          /* We waited and post function did not get called */ 
          DEBUG(printk("[FCCT IOCTL post function not called]\n");)
          DEBUG(sprintf(debug_buff,
           "FCCT ioctl: post function not called \n"));
          DEBUG(qla2100_print(debug_buff));
          pext->Status       = EXT_STATUS_ERR;
          pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;   
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_ERR;
        } else {
             /* getting device data and putting in pext->ResponseAdr */
             /* printk("[post function called; start FCCT IOCTL returning up data ]\n");*/
             ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                               pext->ResponseLen); 
             if (ret) {
               pext->Status       = EXT_STATUS_ERR;
               pext->DetailStatus = EXT_STATUS_COPY_ERR;
               copy_up_EXT(pext,arg);
               DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE for IOCTL PT ha=%8x]\n",
                     (uint32_t)ha);)
               return(ret);
             }
             /* sending back data returned from Management Server */
             for (i=0 ; i < pext->ResponseLen ; i++) {
                usr_temp   = (uint8_t *)pext->ResponseAdr + i;
                kernel_tmp = (uint8_t *)ha->ioctl_mem + i;
                /*printk("[%x]",*kernel_tmp);*/
                __put_user(*kernel_tmp, usr_temp);
             }
          /*printk("[finished QLA2100 IOCTL EXT_CC_SEND_FCCT_PASSTHRU]\n");*/
          pext->Status       = EXT_STATUS_SCSI_STATUS;
          pext->DetailStatus = sp->scode;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
        }
      break;

    case EXT_CC_SEND_SCSI_PASSTHRU: 
      /* printk("[start EXT_CC_SEND_SCSI_PASSTHRU]\n"); */
      ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
            sizeof(EXT_SCSI_PASSTHRU)); 
      if (ret) {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_COPY_ERR;
        copy_up_EXT(pext,arg);
        DEBUG2(printk("[qla2100_ioctl: ERROR verify_area READ of EXT_SCSI_PASSTHRU]\n");)
        return(ret);
      }
      for (i=0 ; i < sizeof(EXT_SCSI_PASSTHRU) ; i++) {
        usr_temp   = (uint8_t *)pext->RequestAdr + i;
        kernel_tmp = (uint8_t *)pscsi_pass + i;
        __get_user(*kernel_tmp, usr_temp);
      }
      if (pext->ResponseLen > PAGE_SIZE) {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
        copy_up_EXT(pext,arg);
        DEBUG2(printk("[qla2100_ioctl: ERROR size of requested EXT_SCSI_PASSTHRU]\n");)
        return(ret);
      }
      /* clear ioctl_sp and scsi_cmd and scsi_device to be used */
      kernel_tmp = (uint8_t *)sp;
      for (i=0;i<sizeof(srb_t);i++)       *(kernel_tmp+i) = 0; 
      kernel_tmp = (uint8_t *)pscsi_cmd;
      for (i=0;i<sizeof(Scsi_Cmnd);i++)   *(kernel_tmp+i) = 0; 
      kernel_tmp = (uint8_t *)&scsi_device;
      for (i=0;i<sizeof(Scsi_Device);i++) *(kernel_tmp+i) = 0; 
      kernel_tmp = (uint8_t *)ha->ioctl_mem;
      for (i=0;i<PAGE_SIZE;i++) *kernel_tmp = 0; 

      pscsi_cmd->host = ha->host;
      sp->cmd         = pscsi_cmd;
      sp->flags       = SRB_WATCHDOG;

      /* mark this as a special delivery and collection command */
      scsi_cmd.flags     = 0;      
      scsi_cmd.scsi_done = ioctl_scsi_pt_done;

      scsi_cmd.device               = &scsi_device;     
      scsi_cmd.device->tagged_queue = 0;
      scsi_cmd.use_sg               = 0; /* no ScatterGather */
      scsi_cmd.target               = pscsi_pass->TargetAddr.Target;
      scsi_cmd.lun                  = pscsi_pass->TargetAddr.Lun;
      scsi_cmd.request_bufflen      = pext->ResponseLen;
      scsi_cmd.request_buffer       = ha->ioctl_mem;
      scsi_cmd.timeout_per_command  = 0x300;

      // printk("[start FCSCSI IOCTL look at direction t=%x l=%x]\n",
      // scsi_cmd.target,scsi_cmd.lun);
      if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_OUT) {
          /* sending user data from pext->ResponseAdr to device */
          ret = verify_area(VERIFY_READ, (void *)pext->ResponseAdr,
                            pext->ResponseLen); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area READ EXT_SCSI_PASSTHRU]\n");)
            return(ret);
          }
          for (i=0 ; i < pext->ResponseLen ; i++) {
            usr_temp   = (uint8_t *)pext->ResponseAdr + i;
            kernel_tmp = (uint8_t *)ha->ioctl_mem    + i;
            __get_user(*kernel_tmp, usr_temp);
          }
      }
      if (pscsi_pass->CdbLength == 6) {
        scsi_cmd.cmd_len = 6;
        scsi_cmd.data_cmnd[0] = scsi_cmd.cmnd[0] = pscsi_pass->Cdb[0];
        scsi_cmd.data_cmnd[1] = scsi_cmd.cmnd[1] = pscsi_pass->Cdb[1];
        scsi_cmd.data_cmnd[2] = scsi_cmd.cmnd[2] = pscsi_pass->Cdb[2];
        scsi_cmd.data_cmnd[3] = scsi_cmd.cmnd[3] = pscsi_pass->Cdb[3];
        scsi_cmd.data_cmnd[4] = scsi_cmd.cmnd[4] = pscsi_pass->Cdb[4];
        scsi_cmd.data_cmnd[5] = scsi_cmd.cmnd[5] = pscsi_pass->Cdb[5];
        scsi_cmd.data_cmnd[6] = scsi_cmd.cmnd[6] = 0;
        scsi_cmd.data_cmnd[7] = scsi_cmd.cmnd[7] = 0;
        scsi_cmd.data_cmnd[8] = scsi_cmd.cmnd[8] = 0;
        scsi_cmd.data_cmnd[9] = scsi_cmd.cmnd[9] = 0;
      } else if (pscsi_pass->CdbLength == 10) {
        scsi_cmd.cmd_len = 0x0A;
        scsi_cmd.data_cmnd[0] = scsi_cmd.cmnd[0] = pscsi_pass->Cdb[0];
        scsi_cmd.data_cmnd[1] = scsi_cmd.cmnd[1] = pscsi_pass->Cdb[1];
        scsi_cmd.data_cmnd[2] = scsi_cmd.cmnd[2] = pscsi_pass->Cdb[2];
        scsi_cmd.data_cmnd[3] = scsi_cmd.cmnd[3] = pscsi_pass->Cdb[3];
        scsi_cmd.data_cmnd[4] = scsi_cmd.cmnd[4] = pscsi_pass->Cdb[4];
        scsi_cmd.data_cmnd[5] = scsi_cmd.cmnd[5] = pscsi_pass->Cdb[5];
        scsi_cmd.data_cmnd[6] = scsi_cmd.cmnd[6] = pscsi_pass->Cdb[6];
        scsi_cmd.data_cmnd[7] = scsi_cmd.cmnd[7] = pscsi_pass->Cdb[7];
        scsi_cmd.data_cmnd[8] = scsi_cmd.cmnd[8] = pscsi_pass->Cdb[8];
        scsi_cmd.data_cmnd[9] = scsi_cmd.cmnd[9] = pscsi_pass->Cdb[9];
      } else if (pscsi_pass->CdbLength == 12) {
        scsi_cmd.cmd_len = 0x0C;
        scsi_cmd.data_cmnd[0] = scsi_cmd.cmnd[0] = pscsi_pass->Cdb[0];
        scsi_cmd.data_cmnd[1] = scsi_cmd.cmnd[1] = pscsi_pass->Cdb[1];
        scsi_cmd.data_cmnd[2] = scsi_cmd.cmnd[2] = pscsi_pass->Cdb[2];
        scsi_cmd.data_cmnd[3] = scsi_cmd.cmnd[3] = pscsi_pass->Cdb[3];
        scsi_cmd.data_cmnd[4] = scsi_cmd.cmnd[4] = pscsi_pass->Cdb[4];
        scsi_cmd.data_cmnd[5] = scsi_cmd.cmnd[5] = pscsi_pass->Cdb[5];
        scsi_cmd.data_cmnd[6] = scsi_cmd.cmnd[6] = pscsi_pass->Cdb[6];
        scsi_cmd.data_cmnd[7] = scsi_cmd.cmnd[7] = pscsi_pass->Cdb[7];
        scsi_cmd.data_cmnd[8] = scsi_cmd.cmnd[8] = pscsi_pass->Cdb[8];
        scsi_cmd.data_cmnd[9] = scsi_cmd.cmnd[9] = pscsi_pass->Cdb[9];
        scsi_cmd.data_cmnd[10] = scsi_cmd.cmnd[10] = pscsi_pass->Cdb[10];
        scsi_cmd.data_cmnd[11] = scsi_cmd.cmnd[10] = pscsi_pass->Cdb[11];
       }
      /* Generate LU queue on bus, target, LUN */
      b = SCSI_BUS_32(pscsi_cmd);
      t = SCSI_TCN_32(pscsi_cmd);
      l = SCSI_LUN_32(pscsi_cmd);

      /* set sp->target for 32bit/64bit delivery */
      sp->wdg_time                   = 0;

      /* check presense of requested target and other conditions */
      if( TGT_Q(ha,b,t) == NULL ||
        ( TGT_Q(ha,b,t) && TGT_Q(ha,b,t)->loop_id > LAST_SNS_LOOP_ID)  ||
        ( ha->loop_down_timer == 0 && ha->loop_state == LOOP_DOWN)) {  
            printk("[FCSCSI IOCTL Target MISSING]\n");
            DEBUG2(printk("scsi(%2d:%2d:%2d:%2d): SCSI PT port unavailable\n",
                  (int)ha->host_no,b,t,l);)
           pext->Status       = EXT_STATUS_ERR;
           pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
           copy_up_EXT(pext,arg);
           return(ret);
       }  
      /* get spin lock for this operation */
      spin_lock_irqsave(&io_request_lock, cpu_flags);

      /* Allocate a LUN/DEVICE queue from this request */
        if( (q = GET_LU_Q(ha, b, t,l)) == NULL ) {
            DRIVER_LOCK
            if( (q = qla2100_lun_alloc()) == NULL ) {
               pext->Status       = EXT_STATUS_ERR;
               pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
               copy_up_EXT(pext,arg);
               DEBUG2(printk("[qla2100_ioctl: ERROR in GET_LU_Q for SCSI_PASSTHRU ha=%8x]\n",
                     (uint32_t)ha);)
               ret = EXT_STATUS_ERR;
               return(ret);
            }
            LU_Q(ha, b, t, l) = q;
            DEBUG(sprintf(debug_buff,"Allocate new device queue 0x%x\n",q));
            DEBUG(qla2100_print(debug_buff));
            DRIVER_UNLOCK
        }
        /* Set an invalid handle until we issue the command to ISP */
        /* then we will set the real handle value.                 */
        handle = INVALID_HANDLE;  
        CMD_HANDLE(pscsi_cmd) = (unsigned char *)handle;

        if( sp->flags ) {
            sp->port_down_retry_count = ha->port_down_retry_count - 1;
            sp->retry_count = ha->retry_count; 
            DEBUG3(sprintf(debug_buff,"qla2100: PT Set retry counts =0x%x,0x%x\n\r",
                   sp->port_down_retry_count,sp->retry_count));
            DEBUG3(qla2100_print(debug_buff));
        }
        qla2100_putq_t(q,sp); 

        /* set flag to indicate IOCTL SCSI PassThru in progress */
        ha->IoctlPassThru_InProgress = 1;

        //printk("[start FCSCSI IOCTL restart queues]\n");
        /* send command to adapter */
        qla2100_restart_queues(ha,FALSE);

        /* release spin lock since command is queued */  
        spin_unlock_irqrestore(&io_request_lock, cpu_flags);

        ha->ioctl_timer = 6; /* 6 ticks of 1 second timer max wait */

        /* wait for post function or timer to zero the InProgress flag */
        while (ha->IoctlPassThru_InProgress == 1) {
             udelay(35);
        }

        //printk("[start FCSCSI IOCTL finished while]\n");
        if (ha->IoctlPassThru_InProgress == 1) {
          /* We waited and post function did not get called */ 
          DEBUG(printk("[FCSCSI IOCTL post function not called]\n");)
          pext->Status       = EXT_STATUS_ERR;
          pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;   
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_ERR;
        } else {
          if (pscsi_pass->Direction == EXT_DEF_SCSI_PASSTHRU_DATA_IN) {
             /* getting device data and putting in pext->ResponseAdr */
             ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                               pext->ResponseLen); 
             if (ret) {
               pext->Status       = EXT_STATUS_ERR;
               pext->DetailStatus = EXT_STATUS_COPY_ERR;
               copy_up_EXT(pext,arg);
               DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE for IOCTL PT ha=%8x]\n",
                     (uint32_t)ha);)
               return(ret);
             }
             /* now copy up the READ data to user */
             for (i=0 ; i < pext->ResponseLen ; i++) {
                usr_temp   = (uint8_t *)pext->ResponseAdr + i;
                kernel_tmp = (uint8_t *)ha->ioctl_mem + i;
                /*printk("[%x]",*kernel_tmp);*/
                __put_user(*kernel_tmp, usr_temp);
             }
          }
          //printk("[[sense0=%x sense2=%x]]\n",
          //pscsi_cmd->sense_buffer[0],
          //pscsi_cmd->sense_buffer[2]);
          /* copy up structure to make sense data available to user */
          for (i=0;i<16;i++) 
               pscsi_pass->SenseData[i] = pscsi_cmd->sense_buffer[i];
          ret = verify_area(VERIFY_WRITE, (void *)pext->RequestAdr,
                            sizeof(EXT_SCSI_PASSTHRU)); 
          if (ret) {
            pext->Status       = EXT_STATUS_ERR;
            pext->DetailStatus = EXT_STATUS_COPY_ERR;
            copy_up_EXT(pext,arg);
            DEBUG2(printk("[qla2100_ioctl: ERROR verify_area WRITE of EXT_SCSI_PASSTHRU]\n");)
            return(ret);
          }
          for (i=0 ; i < sizeof(EXT_SCSI_PASSTHRU) ; i++) {
            usr_temp   = (uint8_t *)pext->RequestAdr + i;
            kernel_tmp = (uint8_t *)pscsi_pass + i;
            __put_user(*kernel_tmp, usr_temp);
          }
          /* printk("[finished QLA2100 IOCTL EXT_FCSCSI_REQ]\n"); */ 
          pext->Status       = EXT_STATUS_SCSI_STATUS;
          pext->DetailStatus = sp->scode;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
        }
      break;
 
   case EXT_CC_READ_NVRAM:
      if (pext->ResponseLen < sizeof(nvram21_t)) transfer_size = pext->ResponseLen;
      else      transfer_size = sizeof(nvram21_t)/2;
      for( i = 0, cnt = 0; cnt < transfer_size ; cnt++ , i++ ) {
          *wptr = qla2100_get_nvram_word(ha, cnt);

          usr_temp    = (uint8_t *)pext->ResponseAdr;
          kernel_tmp  = (uint8_t *)wptr;

          usr_temp += i; /* even byte */
          __put_user(*kernel_tmp, usr_temp);

          i++;
          usr_temp    = (uint8_t *)pext->ResponseAdr;
          kernel_tmp1 = (uint8_t *)wptr + 1;
          usr_temp += i; /* odd byte */
          __put_user(*kernel_tmp1, usr_temp);
      }

      /* printk("[finished QLA2100 IOCTL EXT_NVR_RD]\n"); */ 
      pext->Status       = EXT_STATUS_OK;
      pext->DetailStatus = EXT_STATUS_OK;
      copy_up_EXT(pext,arg);
      ret = EXT_STATUS_OK;
      break;

    case EXT_CC_LOOPBACK:
    DEBUG(sprintf(debug_buff,"qla2100_ioctl: start EXT_CC_LOOPBACK\n"));
    DEBUG(qla2100_print(debug_buff));

    /* printk("[start EXT_CC_LOOPBACK]\n"); */

    if (ha->device_id == QLA2100_DEVICE_ID) {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
        copy_up_EXT(pext,arg);
        DEBUG2(printk("[EXT_CC_SEND_LOOP_BACK not supported on QLA2100]\n");)
        return(ret);
    }

      if (pext->ResponseLen > PAGE_SIZE)  pext->ResponseLen = PAGE_SIZE;
      if (pext->RequestLen  > PAGE_SIZE)  {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_INVALID_PARAM;
        copy_up_EXT(pext,arg);
        /*printk("[EXT_CC_LOOPBACK too big ResponseLen=%x ReqLen=%x]\n",pext->ResponseLen,pext->RequestLen); */
        DEBUG2(printk("[qla2100_ioctl: ERROR size of requested Resp_len in EXT_CC_SEND_LOOP_BACK]\n");)
        return(ret);
      }
      ret = verify_area(VERIFY_READ, (void *)pext->RequestAdr,
                        pext->RequestLen); 
      if (ret){
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_COPY_ERR;
        copy_up_EXT(pext,arg);
        /*printk("[EXT_CC_LOOPBACK verify read error]\n");*/ 
        DEBUG2(printk("[qla2100_ioctl: ERROR verify_area READ of EXT_CC_LOOPBACK]\n");)
        return(ret);
      }
      for (i=0 ; i < pext->ResponseLen ; i++) {
        /* copy in from user space the user data pattern to be sent */
        usr_temp   = (uint8_t *)pext->ResponseAdr + i;
        kernel_tmp = (uint8_t *)ha->ioctl_mem + i;
        __get_user(*kernel_tmp, usr_temp);
	/*  printk("{%x}",*kernel_tmp); */ 
      }
      /* check on loop down */
      if ( ha->loop_down_timer == 0 && ha->loop_state == LOOP_DOWN)  {
        pext->Status       = EXT_STATUS_ERR;
        pext->DetailStatus = EXT_STATUS_DEV_NOT_FOUND;
        copy_up_EXT(pext,arg);
        /* printk("[EXT_CC_LOOPBACK loop down]\n"); */
        return(ret);
      }

      /* get spin lock for this operation */
      spin_lock_irqsave(&io_request_lock, cpu_flags);

      mb[0] = MBC_DIAGNOSTIC_LOOP_BACK;
      mb[1] = 0x72; /* options: 64 bit, use buffer for snd/rcv on ext. loop */
      mb[2] = mb[3] = mb[4] = mb[5] = mb[6] = mb[7] = mb[8] = mb[9] = 0;
      mb[10] = LSW(pext->ResponseLen);
      mb[11] = MSW(pext->ResponseLen);
      mb[12] = 0; /* transfer segment count */
      mb[13] = 0; /* receive segment count */
      mb[14] = LSW(ha->ioctl_mem_phys); /* send data address */
      mb[15] = MSW(ha->ioctl_mem_phys);
      mb[20] = QL21_64BITS_3RDWD(ha->ioctl_mem_phys);
      mb[21] = QL21_64BITS_4THWD(ha->ioctl_mem_phys);
      mb[16] = LSW(ha->ioctl_mem_phys); /* recv data address */
      mb[17] = MSW(ha->ioctl_mem_phys);
      mb[6]  = QL21_64BITS_3RDWD(ha->ioctl_mem_phys);
      mb[7]  = QL21_64BITS_4THWD(ha->ioctl_mem_phys); 
      mb[18] = 1; /* iteration count lsb */
      mb[19] = 0; /* iteration count msb */

      DEBUG(sprintf(debug_buff,
            "LoopBack ioctl: issue loop back mailbox command\n"));
      DEBUG(qla2100_print(debug_buff));

      // wait for 64 bit loopback ready firmware
      status = qla2100_mailbox_command(ha,BIT_21|BIT_20|BIT_19|BIT_18|
                                       BIT_17|BIT_16|BIT_15|BIT_14|
                                       BIT_13|BIT_12|BIT_11|
                                       BIT_10|BIT_7|BIT_6|BIT_1|BIT_0,
                                       &mb[0]);

      /* release spin lock since command is issued */  
      spin_unlock_irqrestore(&io_request_lock, cpu_flags);

        if (status) {
          DEBUG(sprintf(debug_buff,
           "LoopBack ioctl: issue loop back mailbox command FAILED\n"));
          DEBUG(qla2100_print(debug_buff));
          pext->Status       = EXT_STATUS_ERR;
          pext->DetailStatus = EXT_STATUS_ERR;   
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_ERR;
        } else {
             /* put looped back data in pext->ResponseAdr */
             /* printk("[LoopBack ioctl: LoopBack was OK ]\n");*/
             ret = verify_area(VERIFY_WRITE, (void *)pext->ResponseAdr ,
                               pext->ResponseLen); 
             if (ret) {
               pext->Status       = EXT_STATUS_ERR;
               pext->DetailStatus = EXT_STATUS_COPY_ERR;
               copy_up_EXT(pext,arg);
               DEBUG2(printk("[LoopBack ioctl: ERROR verify_area WRITE ha=%x]\n",
                     (uint32_t)ha);)
               return(ret);
             }
             /* sending back looped back data */
             for (i=0 ; i < pext->ResponseLen ; i++) {
                usr_temp   = (uint8_t *)pext->ResponseAdr + i;
                kernel_tmp = (uint8_t *)ha->ioctl_mem + i;
                /*printk("[%x]",*kernel_tmp);*/
                __put_user(*kernel_tmp, usr_temp);
             }
          /*printk("[finished QLA2100 IOCTL EXT_CC_LOOPBACK]\n");*/
          pext->Status       = EXT_STATUS_OK;
          pext->DetailStatus = EXT_STATUS_OK;
          copy_up_EXT(pext,arg);
          ret = EXT_STATUS_OK;
        }
      break;

/* case EXT_CC_RSCN_REG:
      break;
    case EXT_CC_RSCN_GET:
      break; 
    case EXT_CC_ELS_RNID_SEND:
      break;
    case EXT_CC_ELS_RTIN_SEND:
      break;
    case EXT_CC_PLATFORM_REG:
      break;

all others go here */

    default:
      break;
  }

  return(ret);
  
}


