/******************************************************************************
*                  QLOGIC LINUX SOFTWARE
*
* QLogic  QLA1280 (Ultra2)  and  QLA12160 (Ultra3) SCSI driver
* Copyright (C) 2000 Qlogic Corporation 
* (www.qlogic.com)
*
* This program is free software; you can redistribute it and/or modify it
* under the terms of the GNU General Public License as published by the
* Free Software Foundation; either version 2, or (at your option) any
* later version.
*
* This program is distributed in the hope that it will be useful, but
* WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* General Public License for more details.
**
******************************************************************************/
#define QLA1280_VERSION      "3.24 Beta"
/******************************************************************************
    Revision History:
    Rev  3.24 Beta April 24, Jes Sorensen
        - Remove pre 2.2 kernel support
        - clean up 64 bit DMA setting to use 2.4 API (provide backwards compat)
        - Fix MMIO access to use readl/writel instead of directly
          dereferencing pointers
        - Nuke MSDOS debugging code
        - Change true/false data types to int from uint8_t
        - Use int for counters instead of uint8_t etc.
        - Clean up size & byte order conversion macro usage
    Rev  3.23 Beta January 11, 2001 BN Qlogic
        - Added check of device_id when handling non 
          QLA12160s during detect().
    Rev  3.22 Beta January 5, 2001 BN Qlogic
        - Changed queue_task() to schedule_task()
          for kernels 2.4.0 and higher. 
          Note: 2.4.0-testxx kernels released prior to
                the actual 2.4.0 kernel release on January 2001
                will get compile/link errors with schedule_task().
                Please update your kernel to released 2.4.0 level,
                or comment lines in this file flagged with  3.22 
                to resolve compile/link error of schedule_task().
        - Added -DCONFIG_SMP in addition to -D__SMP__ 
          in Makefile for 2.4.0 builds of driver as module.
    Rev  3.21 Beta January 4, 2001 BN Qlogic
        - Changed criteria of 64/32 Bit mode of HBA
          operation according to BITS_PER_LONG rather
          than HBA's NVRAM setting of >4Gig memory bit;
          so that the HBA auto-configures without the need
          to setup each system individually.
    Rev  3.20 Beta December 5, 2000 BN Qlogic
        - Added priority handling to IA-64  onboard SCSI
          ISP12160 chip for kernels greater than 2.3.18.
        - Added irqrestore for qla1280_intr_handler.
        - Enabled /proc/scsi/qla1280 interface.
        - Clear /proc/scsi/qla1280 counters in detect().
    Rev  3.19 Beta October 13, 2000 BN Qlogic
        - Declare driver_template for new kernel
          (2.4.0 and greater) scsi initialization scheme.
        - Update /proc/scsi entry for 2.3.18 kernels and
          above as qla1280
    Rev  3.18 Beta October 10, 2000 BN Qlogic
        - Changed scan order of adapters to map 
          the QLA12160 followed by the QLA1280.
    Rev  3.17 Beta September 18, 2000 BN Qlogic
        - Removed warnings for 32 bit 2.4.x compiles
        - Corrected declared size for request and response
          DMA addresses that are kept in each ha
    Rev. 3.16 Beta  August 25, 2000   BN  Qlogic
        - Corrected 64 bit addressing issue on IA-64
          where the upper 32 bits were not properly
          passed to the RISC engine.
    Rev. 3.15 Beta  August 22, 2000   BN  Qlogic
        - Modified qla1280_setup_chip to properly load
          ISP firmware for greater that 4 Gig memory on IA-64
    Rev. 3.14 Beta  August 16, 2000   BN  Qlogic
        - Added setting of dma_mask to full 64 bit
          if flags.enable_64bit_addressing is set in NVRAM
    Rev. 3.13 Beta  August 16, 2000   BN  Qlogic
        - Use new PCI DMA mapping APIs for 2.4.x kernel
    Rev. 3.12       July 18, 2000    Redhat & BN Qlogic
        - Added check of pci_enable_device to detect() for 2.3.x
        - Use pci_resource_start() instead of 
          pdev->resource[0].start in detect() for 2.3.x
        - Updated driver version
    Rev. 3.11       July 14, 2000    BN  Qlogic
	- Updated SCSI Firmware to following versions:
	  qla1x80:   8.13.08
	  qla1x160:  10.04.08
	- Updated driver version to 3.11
    Rev. 3.10    June 23, 2000   BN Qlogic
        - Added filtering of AMI SubSys Vendor ID devices 
    Rev. 3.9 
        - DEBUG_QLA1280 undefined and  new version  BN Qlogic 
    Rev. 3.08b      May 9, 2000    MD Dell
        - Added logic to check against AMI subsystem vendor ID
	Rev. 3.08       May 4, 2000    DG  Qlogic
        - Added logic to check for PCI subsystem ID.
	Rev. 3.07       Apr 24, 2000    DG & BN  Qlogic
	   - Updated SCSI Firmware to following versions:
	     qla12160:   10.01.19
		 qla1280:     8.09.00
	Rev. 3.06       Apr 12, 2000    DG & BN  Qlogic
	   - Internal revision; not released
    Rev. 3.05       Mar 28, 2000    DG & BN  Qlogic
       - Edit correction for virt_to_bus and PROC.
    Rev. 3.04       Mar 28, 2000    DG & BN  Qlogic
       - Merge changes from ia64 port.
    Rev. 3.03       Mar 28, 2000    BN  Qlogic
       - Increase version to reflect new code drop with compile fix
         of issue with inclusion of linux/spinlock for 2.3 kernels
    Rev. 3.02       Mar 15, 2000    BN  Qlogic
       - Merge qla1280_proc_info from 2.10 code base
    Rev. 3.01       Feb 10, 2000    BN  Qlogic
       - Corrected code to compile on a 2.2.x kernel.
    Rev. 3.00       Jan 17, 2000    DG  Qlogic
	   - Added 64-bit support.
    Rev. 2.07       Nov 9, 1999     DG  Qlogic
	   - Added new routine to set target parameters for ISP12160. 
    Rev. 2.06       Sept 10, 1999     DG  Qlogic
       - Added support for ISP12160 Ultra 3 chip.
    Rev. 2.03       August 3, 1999    Fred Lewis, Intel DuPont
	- Modified code to remove errors generated when compiling with
	  Cygnus IA64 Compiler.
        - Changed conversion of pointers to unsigned longs instead of integers.
        - Changed type of I/O port variables from uint32_t to unsigned long.
        - Modified OFFSET macro to work with 64-bit as well as 32-bit.
        - Changed sprintf and printk format specifiers for pointers to %p.
        - Changed some int to long type casts where needed in sprintf & printk.
        - Added l modifiers to sprintf and printk format specifiers for longs.
        - Removed unused local variables.
    Rev. 1.20       June 8, 1999      DG,  Qlogic
         Changes to support RedHat release 6.0 (kernel 2.2.5).  
       - Added SCSI exclusive access lock (io_request_lock) when accessing 
         the adapter.
       - Added changes for the new LINUX interface template. Some new error
         handling routines have been added to the template, but for now we
         will use the old ones.
    -   Initial Beta Release.  
*****************************************************************************/


#include <linux/config.h>       
#ifdef MODULE
#include <linux/module.h>
#endif

#include <linux/version.h>
#include <linux/types.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/pci.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/tqueue.h>
#include <linux/stat.h>
#include <linux/slab.h> 

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/segment.h>
#include <asm/byteorder.h>

#ifndef KERNEL_VERSION
#  define KERNEL_VERSION(x,y,z) (((x)<<16)+((y)<<8)+(z))
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
#include <linux/pci_ids.h>
#endif

#include "sd.h"
#include "scsi.h"
#include "hosts.h"
#define UNIQUE_FW_NAME
#include "qla1280.h"
#include "ql12160_fw.h"   /* ISP RISC codes */
#include "ql1280_fw.h"


/*
 * Compile time Options: 
 *            0 - Disable and 1 - Enable 
 */
#define  QL1280_TARGET_MODE_SUPPORT    0   /* Target mode support */
#define  QL1280_LUN_SUPPORT            0
#define  WATCHDOGTIMER                 0
#define  MEMORY_MAPPED_IO              0
#define  DEBUG_QLA1280_INTR            0
#define  USE_NVRAM_DEFAULTS	       0
#define  DEBUG_PRINT_NVRAM             0
#define  LOADING_RISC_ACTIVITY         0
#define  AUTO_ESCALATE_RESET           0   /* Automatically escalate resets */
#define  AUTO_ESCALATE_ABORT           0   /* Automatically escalate aborts */
#define  STOP_ON_ERROR                 0   /* Stop on aborts and resets  */
#define  STOP_ON_RESET                 0 
#define  STOP_ON_ABORT                 0
#define  QLA1280_PROFILE               1   /* 3.20 */ 
#define  DEBUG_QLA1280                 0

#define	SYS_DELAY(x)		udelay(x);barrier()
#define QLA1280_DELAY(sec)	mdelay(sec * 1000)

#if (BITS_PER_LONG == 64) 
#define QLA_64BIT_PTR	1
#endif

/* 3.16 */
#ifdef QLA_64BIT_PTR
#define pci_dma_lo32(a)		(a & 0xffffffff)
#define pci_dma_hi32(a)		(a >> 32)
#else
#define pci_dma_lo32(a)		(a & 0xffffffff)
#define pci_dma_hi32(a)		0
#endif


#define NVRAM_DELAY()		udelay(500) /* 2 microsecond delay */

#define CACHE_FLUSH(a)		RD_REG_WORD(a)
#define INVALID_HANDLE		(MAX_OUTSTANDING_COMMANDS + 1)

#ifdef QLA_64BIT_PTR
#define MS_64BITS(x)		0
#else
#ifdef __LP64__
#define MS_64BITS(x)		(x >> 32)
#else
#define MS_64BITS(x)		0
#endif
#endif


/*
 * Compat macros
 */
#ifndef pci_set_dma_mask
#define pci_set_dma_mask(dev, mask)		dev->dma_mask = mask;
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
typedef unsigned long dma_addr_t;

static inline void *pci_alloc_consistent(struct pci_dev *hwdev, size_t size,
					 dma_addr_t *dma_handle)
{
	void *virt_ptr;

	virt_ptr = kmalloc(size, GFP_KERNEL);
	if (!virt_ptr)
		return NULL;
	*dma_handle = virt_to_bus(virt_ptr);
	return virt_ptr;
}
#define pci_free_consistent(cookie, size, ptr, dma_ptr)	kfree(ptr)
#define pci_map_single(cookie, address, size, dir)	virt_to_bus(address)
#define pci_map_sg(cookie, scatter, ents, dir)		ents
#define pci_unmap_single(cookie, address, size, dir)
#define pci_unmap_sg(cookie, scatter, ents, dir)

#define pci_resource_start(dev, i)			dev->base_address[i]
#endif


/*
 *  QLogic Driver Support Function Prototypes.
 */
static void qla1280_done(scsi_qla_host_t *, srb_t **, srb_t **);
static void qla1280_next(scsi_qla_host_t *, scsi_lu_t *, uint8_t);
static void qla1280_putq_t(scsi_lu_t *, srb_t *);
static void qla1280_done_q_put(srb_t *, srb_t **, srb_t **);
static void qla1280_device_queue_depth(scsi_qla_host_t *, Scsi_Device *);
static void qla1280_select_queue_depth(struct Scsi_Host *, Scsi_Device *);
#if  STOP_ON_ERROR 
static void qla1280_panic(char *, struct Scsi_Host *host);
#endif
static void qla1280_abort_queue_single(scsi_qla_host_t *,uint32_t,uint32_t,uint32_t,uint32_t);

static int qla1280_return_status( sts_entry_t *sts, Scsi_Cmnd       *cp);
static void qla1280_removeq(scsi_lu_t *q, srb_t *sp);
static void qla1280_mem_free(scsi_qla_host_t *ha);
void qla1280_do_dpc(void *p);
static char *qla1280_get_token(char *, char *);
static inline void qla1280_enable_intrs(scsi_qla_host_t *);
static inline void qla1280_disable_intrs(scsi_qla_host_t *);

/*
 *  QLogic ISP1280 Hardware Support Function Prototypes.
 */
static int qla1280_initialize_adapter(struct scsi_qla_host   *ha);
static int qla1280_enable_tgt(scsi_qla_host_t *, uint8_t);
static int qla1280_isp_firmware(scsi_qla_host_t *);
static int qla1280_pci_config(scsi_qla_host_t *);
static int qla1280_chip_diag(scsi_qla_host_t *);
static int qla1280_setup_chip(scsi_qla_host_t *);
static int qla1280_init_rings(scsi_qla_host_t *);
static int qla1280_nvram_config(scsi_qla_host_t *);
static int qla1280_mailbox_command(scsi_qla_host_t *, uint8_t, uint16_t *);
static int qla1280_bus_reset(scsi_qla_host_t *, u8);
static int qla1280_device_reset(scsi_qla_host_t *, uint8_t, uint32_t);
static int qla1280_abort_device(scsi_qla_host_t *, u8, u32, u32);
static int qla1280_abort_command(scsi_qla_host_t *, srb_t *);
static int qla1280_abort_isp(scsi_qla_host_t *);
static int qla1280_64bit_start_scsi(scsi_qla_host_t *, srb_t *);
static int qla1280_32bit_start_scsi(scsi_qla_host_t *, srb_t *);
static void qla1280_nv_write(scsi_qla_host_t *, uint16_t);
static void qla1280_poll(scsi_qla_host_t *);
static void qla1280_reset_adapter(scsi_qla_host_t *);
static void qla1280_marker(scsi_qla_host_t *, u8, u32, u32, u8);
static void qla1280_isp_cmd(scsi_qla_host_t *);
static void qla1280_isr(scsi_qla_host_t *, srb_t **, srb_t **);
static void qla1280_rst_aen(scsi_qla_host_t *);
static void qla1280_status_entry(scsi_qla_host_t *, sts_entry_t *,
                                 srb_t **, srb_t **);
static void qla1280_error_entry(scsi_qla_host_t *, response_t *,
                                srb_t **, srb_t **),
                 qla1280_restart_queues(scsi_qla_host_t *),
                 qla1280_abort_queues(scsi_qla_host_t *);
static uint16_t  qla1280_get_nvram_word(scsi_qla_host_t *, uint32_t),
                 qla1280_nvram_request(scsi_qla_host_t *, uint32_t),
                 qla1280_debounce_register(volatile uint16_t *);
static request_t *qla1280_req_pkt(scsi_qla_host_t *);
int  qla1280_check_for_dead_scsi_bus(scsi_qla_host_t *ha, srb_t *sp);
static int	 qla1280_mem_alloc(scsi_qla_host_t *ha);
static int	 qla1280_register_with_Linux(scsi_qla_host_t *ha,
                                             unsigned int maxchannels);

static int	 qla12160_set_target_parameters(scsi_qla_host_t *, uint32_t, uint32_t, uint32_t, nvram160_t *);
static void	 qla12160_get_target_parameters(scsi_qla_host_t *, uint32_t, uint32_t, uint32_t);

#if QL1280_LUN_SUPPORT
static void      qla1280_enable_lun(scsi_qla_host_t *, uint8_t, uint32_t);
#endif

#if QL1280_TARGET_MODE_SUPPORT
static void      qla1280_notify_ack(scsi_qla_host_t *, notify_entry_t *),
                 qla1280_immed_notify(scsi_qla_host_t *, notify_entry_t *),
                 qla1280_accept_io(scsi_qla_host_t *, ctio_ret_entry_t *),
                 qla1280_64bit_continue_io(scsi_qla_host_t *, 
                                 atio_entry_t *, uint32_t,
                                 paddr32_t *),
                 qla1280_32bit_continue_io(scsi_qla_host_t *, 
                                 atio_entry_t *, uint32_t,
                                 paddr32_t *),
                 qla1280_atio_entry(scsi_qla_host_t *, atio_entry_t *),
                 qla1280_notify_entry(scsi_qla_host_t *, notify_entry_t *);
#endif  /* QLA1280_TARGET_MODE_SUPPORT */

#if DEBUG_QLA1280
static void	qla1280_print(char *);
static void	qla1280_output_number(uint32_t, int);
static int ql_debug_print = 1;
#endif

#ifdef QL_DEBUG_ROUTINES
/*
 *  Driver Debug Function Prototypes.
 */
static u8 qla1280_getbyte(u8 *);
static u16 qla1280_getword(u16 *);
static u32 qla1280_getdword(u32 *);
static void qla1280_putbyte(u8 *, u8);
static void qla1280_putword(u16 *, u8);
static void qla1280_putdword(u32 *, u32);
static void qla1280_dump_buffer(char *, u32);
static void qla1280_print_scsi_cmd(Scsi_Cmnd *cmd);
#endif

/*
 * insmod needs to find the variable and make it point to something
 */
#ifdef MODULE
static char *options = NULL;

/* insmod qla1280 options=verbose" */
MODULE_PARM(options, "s");  
/*
 * Just in case someone uses commas to separate items on the insmod
 * command line, we define a dummy buffer here to avoid having insmod
 * write wild stuff into our code segment
 */
static char dummy_buffer[60] = "Please don't add commas in your insmod command!!\n";
#endif


#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
/*
 * Our directory Entry in /proc/scsi for the user to 
 * access the driver. 
 */
/* Need to add in proc_fs.h     PROC_SCSI_QL1280 */
#define PROC_SCSI_QL1280  PROC_SCSI_QLOGICISP

struct proc_dir_entry proc_scsi_qla1280 = {
    PROC_SCSI_QL1280, 7, "qla1280",                           
    S_IFDIR | S_IRUGO | S_IXUGO, 2,
    0, 0, 0, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};
#endif

/* We use the Scsi_Pointer structure that's included with each command
 * SCSI_Cmnd as a scratchpad for our SRB.
 *
 * SCp will always point to the SRB structure (defined in qla1280.h).
 * It is define as follows:
 *  - SCp.ptr  -- > pointer back to the cmd
 *  - SCp.this_residual --> used as forward pointer to next srb
 *  - SCp.buffer --> used as backward pointer to next srb
 *  - SCp.buffers_residual --> used as flags field
 *  - SCp.have_data_in --> not used
 *  - SCp.sent_command --> not used
 *  - SCp.phase --> not used
 */

#define	CMD_SP(Cmnd)		(&(Cmnd)->SCp)
#define	CMD_XFRLEN(Cmnd)	(Cmnd)->request_bufflen
#define	CMD_CDBLEN(Cmnd)	(Cmnd)->cmd_len
#define	CMD_CDBP(Cmnd)		(Cmnd)->cmnd
#define	CMD_SNSP(Cmnd)		(Cmnd)->sense_buffer
#define	CMD_SNSLEN(Cmnd)	(sizeof (Cmnd)->sense_buffer)
#define	CMD_RESULT(Cmnd)	((Cmnd)->result)
#define	CMD_HANDLE(Cmnd)	((Cmnd)->host_scribble)

/*****************************************/
/*   ISP Boards supported by this driver */
/*****************************************/
#define QLA1280_VENDOR_ID   0x1077
#define QLA1080_DEVICE_ID   0x1080
#define QLA1240_DEVICE_ID   0x1240
#define QLA1280_DEVICE_ID   0x1280
#define QLA12160_DEVICE_ID  0x1216
#define QLA10160_DEVICE_ID  0x1016
#define NUM_OF_ISP_DEVICES        6

typedef struct _qlaboards 
{
   unsigned char   bdName[9];       /* Board ID String */
   unsigned long   device_id;       /* Device PCI ID   */
   int   numPorts;                  /* Number of SCSI ports */
   unsigned short   *fwcode;        /* pointer to FW array         */
   unsigned long    *fwlen;         /* number of words in array    */
   unsigned short   *fwstart;       /* start address for F/W       */
   unsigned char   *fwver;          /* Ptr to F/W version array    */
} qla_boards_t;

struct _qlaboards   QL1280BoardTbl[NUM_OF_ISP_DEVICES] = 
{
   /* Name ,  Board PCI Device ID,         Number of ports */
  {"QLA12160 ", QLA12160_DEVICE_ID,          2,      
               &fw12160i_code01[0],  (unsigned long *)&fw12160i_length01,&fw12160i_addr01, &fw12160i_version_str[0] },       
  {"QLA1080 ", QLA1080_DEVICE_ID,           1,        
               &fw1280ei_code01[0],  (unsigned long *)&fw1280ei_length01,&fw1280ei_addr01, &fw1280ei_version_str[0] },       
  {"QLA1240 ", QLA1240_DEVICE_ID,           2,       
               &fw1280ei_code01[0],  (unsigned long *)&fw1280ei_length01,&fw1280ei_addr01, &fw1280ei_version_str[0] },       
  {"QLA1280 ", QLA1280_DEVICE_ID,           2,      
               &fw1280ei_code01[0],  (unsigned long *)&fw1280ei_length01,&fw1280ei_addr01, &fw1280ei_version_str[0] },       
  {"QLA10160 ", QLA10160_DEVICE_ID,          1,      
               &fw12160i_code01[0],  (unsigned long *)&fw12160i_length01,&fw12160i_addr01, &fw12160i_version_str[0] },       
  {"        ",                 0,           0}
};

static int qla1280_verbose = 1;
static scsi_qla_host_t *qla1280_hostlist = NULL;
#if QLA1280_PROFILE
static int qla1280_buffer_size = 0;
static char *qla1280_buffer = NULL;
#endif

#if  DEBUG_QLA1280  
char          debug_buff[80];
#define COMTRACE(x)	do{}while(0)
/* #define COMTRACE(x)     qla1280_print(x) */
#define DEBUG(x)	x
#else
#define DEBUG(x)
#define COMTRACE(x)	do{}while(0)
#endif

#ifdef QL_DEBUG_LEVEL_3 
#define ENTER(x)	sprintf(debug_buff,"qla1280 : Entering %s()\n\r", x); \
                        qla1280_print(debug_buff);
#define LEAVE(x)	sprintf(debug_buff,"qla1280 : Leaving %s()\n\r", x); \
                        qla1280_print(debug_buff);
#define ENTER_INTR(x)	sprintf(debug_buff,"qla1280 : Entering %s()\n\r", x); \
                        qla1280_print(debug_buff);
#define LEAVE_INTR(x)	sprintf(debug_buff,"qla1280 : Leaving %s()\n\r", x); \
                        qla1280_print(debug_buff);
#define DEBUG3(x)	x
#else
#define ENTER(x)
#define LEAVE(x)
#define ENTER_INTR(x)
#define LEAVE_INTR(x)
#define DEBUG3(x)
#endif

#ifdef QL_DEBUG_LEVEL_2 
#define DEBUG2(x)	x
#else
#define DEBUG2(x)
#endif
#define DEBUG5(x)

#define OFFSET(w)	(((unsigned long) &w) & 0xFF)   /* 256 byte offsets */

#define SCSI_BUS_32(scp)   ((scp)->channel)
#define SCSI_TCN_32(scp)    ((scp)->target)
#define SCSI_LUN_32(scp)    ((scp)->lun)

/****************************************************************************/
/*  LINUX -  Loadable Module Functions.                                     */
/****************************************************************************/


/*************************************************************************
 *   qla1280_set_info
 *
 * Description:
 *   Set parameters for the driver from the /proc filesystem.
 *
 * Returns:
 *************************************************************************/
int
qla1280_set_info(char *buffer, int length, struct Scsi_Host *HBAptr)
{
  return (-ENOSYS);  /* Currently this is a no-op */
}

/*************************************************************************
 * qla1280_proc_info
 *
 * Description:
 *   Return information to handle /proc support for the driver.
 *
 * buffer - ptrs to a page buffer
 *
 * Returns:
 *************************************************************************/
#define	PROC_BUF	(&qla1280_buffer[len])
int
qla1280_proc_info( char *buffer, char **start, off_t offset, int length, 
                    int hostno, int inout) {
#if QLA1280_PROFILE
  struct Scsi_Host *host;
  scsi_qla_host_t *ha;
  int    size = 0;
  scsi_lu_t  *up;
  int   len = 0;
  qla_boards_t   *bdp; 
  uint32_t        b, t, l;
  uint8_t  *temp;
  host = NULL;

    /* Find the host that was specified */
    for( ha=qla1280_hostlist; (ha != NULL) && ha->host->host_no != hostno; ha=ha->next )
    ;

    /* if host wasn't found then exit */
    if( !ha ) {
        size = sprintf(buffer, "Can't find adapter for host number %d\n", hostno);
        if( size > length ) {
      return (size);
        } else {
            return (0);
    }
  }

  host = ha->host;

  if( inout == TRUE ) /* Has data been written to the file? */
  {
        printk("qla1280_proc: has data been written to the file. \n");
        return (qla1280_set_info(buffer, length, host));
  }
  
  /* 
   * if our old buffer is the right size use it otherwise 
   * allocate a new one.
   */
    size = 4096;  /* get a page */
    if( qla1280_buffer_size != size ) {
    /* deallocate this buffer and get a new one */
        if( qla1280_buffer != NULL ) {
      kfree(qla1280_buffer);
      qla1280_buffer_size = 0;
    }
    qla1280_buffer = kmalloc(size, GFP_KERNEL);
  }
    if( qla1280_buffer == NULL ) {
    size = sprintf(buffer, "qla1280 - kmalloc error at line %d\n",
        __LINE__);
    return size;
  }
    /* save the size of our buffer */
  qla1280_buffer_size = size;

  /* 3.20 clear the buffer we use for proc display */
  temp = qla1280_buffer;
  for (b=0 ; b < size; b++)     *(temp+b) = 0; 

    /* start building the print buffer */ 
    bdp = &QL1280BoardTbl[ha->devnum];
    size =  sprintf(PROC_BUF,
            "QLogic PCI to SCSI Adapter for ISP 1280/12160:\n"
            "        Firmware version: %2d.%02d.%02d, Driver version %s\n",                bdp->fwver[0], bdp->fwver[1], bdp->fwver[2], QLA1280_VERSION);

    len += size;

    size = sprintf(PROC_BUF, "SCSI Host Adapter Information: %s\n", bdp->bdName);
    len += size;
    size = sprintf(PROC_BUF, "Request Queue = 0x%p, Response Queue = 0x%p\n",
                        (void *)ha->request_dma,
                        (void *)ha->response_dma);
    len += size;
    size = sprintf(PROC_BUF, "Request Queue count= 0x%x, Response Queue count= 0x%x\n",
                        REQUEST_ENTRY_CNT,
                        RESPONSE_ENTRY_CNT);
    len += size; 
    size = sprintf(PROC_BUF, "Number of pending commands = 0x%lx\n", ha->actthreads);
    len += size;
    size = sprintf(PROC_BUF, "Number of queued commands = 0x%lx\n", ha->qthreads);
    len += size;
    size = sprintf(PROC_BUF, "Number of free request entries = %d\n",ha->req_q_cnt);
    len += size;
    size = sprintf(PROC_BUF, "\n");                                             /* 1       */
    len += size;
                        
    size = sprintf(PROC_BUF, "SCSI device Information:\n");
    len += size; 
  /* scan for all equipment stats */ 
	for (b = 0; b < MAX_BUSES; b++) 
	for (t = 0; t < MAX_TARGETS; t++) {
        for( l = 0; l < MAX_LUNS; l++ ) {
           up = (scsi_lu_t *) LU_Q(ha, b, t, l);
           if( up == NULL ) 
           continue;
           /* unused device/lun */
           if( up->io_cnt == 0 || up->io_cnt < 2 )
           continue;
      /* total reads since boot */
      /* total writes since boot */
      /* total requests since boot  */
           size = sprintf(PROC_BUF, "(%2d:%2d:%2d): Total reqs %ld,",b,t,l,up->io_cnt);
           len += size;
      /* current number of pending requests */
           size = sprintf(PROC_BUF, " Pend reqs %d,",up->q_outcnt);
           len += size;
#if 0
      /* avg response time */
           size = sprintf(PROC_BUF, " Avg resp time %ld%%,",(up->resp_time/up->io_cnt)*100);
           len += size;
      
      /* avg active time */
           size = sprintf(PROC_BUF, " Avg active time %ld%%\n",(up->act_time/up->io_cnt)*100);
#else
           size = sprintf(PROC_BUF, "\n");
#endif
           len += size; 
        }
        if( len >= qla1280_buffer_size ) 
          break;
  }

    if( len >= qla1280_buffer_size ) {
    printk(KERN_WARNING "qla1280: Overflow buffer in qla1280_proc.c\n");
  }

    if( offset > len - 1 ) {
    kfree(qla1280_buffer);
    qla1280_buffer = NULL;
    qla1280_buffer_size = length = 0;
    *start = NULL;
    } else {
    *start = &qla1280_buffer[offset];   /* Start of wanted data */
        if( len - offset < length ) {
            length = len - offset;
    }
  }
    return (length);
#else
    return (0);
#endif

}

/**************************************************************************
 * qla1280_detect
 *    This routine will probe for Qlogic 1280 SCSI host adapters.
 *    It returns the number of host adapters of a particular
 *    type that were found.	 It also initialize all data necessary for 
 *    the driver.  It is passed-in the host number, so that it
 *    knows where its first entry is in the scsi_hosts[] array.
 *
 * Input:
 *     template - pointer to SCSI template
 *
 * Returns:
 *  num - number of host adapters found.  
 **************************************************************************/
int
qla1280_detect(Scsi_Host_Template *template)
{
    int num_hosts = 0;
    struct Scsi_Host *host;
    scsi_qla_host_t *ha, *cur_ha;
    struct _qlaboards  *bdp;
    int i,j;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
    unsigned short subsys;
#endif
    device_reg_t   *reg;
    char   *cp;
    struct pci_dev *pdev = NULL;

#ifndef PCI_VENDOR_ID_AMI
#define PCI_VENDOR_ID_AMI               0x101e
#endif

    ENTER("qla1280_detect");

    if (sizeof(srb_t) > sizeof(Scsi_Pointer) )
      printk("qla1280_detect: [WARNING] srb_t Must Be Redefined");

#ifdef CHECKSRBSIZE
    if (sizeof(srb_t) > sizeof(Scsi_Pointer) )
    {
      printk("qla1280_detect:  srb_t Must Be Redefined - its too big");
      return 0;
    }
#endif

#ifdef MODULE
	DEBUG(sprintf(debug_buff,"DEBUG: qla1280_detect starts at address = %p\n",qla1280_detect);)
	DEBUG(qla1280_print(debug_buff);)
    /*
    * If we are called as a module, the qla1280 pointer may not be null
    * and it would point to our bootup string, just like on the lilo
    * command line.  IF not NULL, then process this config string with
    * qla1280_setup
    *
    * Boot time Options
    * To add options at boot time add a line to your lilo.conf file like:
    * append="qla1280=verbose,max_tags:{{255,255,255,255},{255,255,255,255}}"
    * which will result in the first four devices on the first two
    * controllers being set to a tagged queue depth of 32.
    */
    if(options)
        qla1280_setup(options, NULL);
    if(dummy_buffer[0] != 'P')
        printk(KERN_WARNING "qla1280: Please read the file /usr/src/linux/drivers"
                "/scsi/README.qla1280\n"
                "qla1280: to see the proper way to specify options to the qla1280 "
                "module\n"
                "qla1280: Specifically, don't use any commas when passing arguments to\n"
                "qla1280: insmod or else it might trash certain memory areas.\n");
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
    if ((int) !pcibios_present())
#else
    if (!pci_present())
#endif
    {
	printk("scsi: PCI not present\n");
	return 0;
    } 

    bdp = &QL1280BoardTbl[0];
    qla1280_hostlist = NULL;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
    template->proc_dir = &proc_scsi_qla1280;
#else
    template->proc_name = "qla1280";
#endif

        /* 3.20 */
        /* First Initialize QLA12160 on PCI Bus 1 Dev 2 */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
        while ((pdev = pci_find_subsys(QLA1280_VENDOR_ID,
                               bdp->device_id,  /* QLA12160 first in list */ 
                               PCI_ANY_ID, 
                               PCI_ANY_ID,pdev))) {

                /* find QLA12160 device on PCI bus=1 slot=2 */
                if ((pdev->bus->number     != 1)   ||
                    (PCI_SLOT(pdev->devfn) != 2))  continue;

                if (pci_enable_device(pdev)) goto find_devices;
                printk("qla1x160: Initializing ISP12160 on PCI Bus 1, Dev 2\n");
		host = scsi_register(template, sizeof(scsi_qla_host_t));
		ha = (scsi_qla_host_t *) host->hostdata;
		/* Clear our data area */
		for( j =0, cp = (char *)ha;  j < sizeof(scsi_qla_host_t); j++)
			*cp++ = 0;
		/* Sanitize the information from PCI BIOS.  */
		host->irq = pdev->irq;
		host->io_port = pci_resource_start(pdev, 0);
		ha->pci_bus = pdev->bus->number;
		ha->pci_device_fn = pdev->devfn;
		ha->pdev = pdev;
		ha->device_id = bdp->device_id; /* QLA12160 first in list */
    
                ha->devnum = 0; // This priority ISP12160 is always devnum zero
		if( qla1280_mem_alloc(ha) ) {
  	          printk(KERN_INFO "qla1x160: Failed to get memory\n");
		}                
                ha->ports = bdp->numPorts; 
                /* following needed for all cases of OS versions */
                host->io_port &= PCI_BASE_ADDRESS_IO_MASK; 
                ha->iobase = (device_reg_t *) host->io_port;
                ha->host = host;
                ha->host_no = host->host_no;
                /* 3.20 zero out /proc/scsi/qla1280 counters */
                ha->actthreads = 0;
                ha->qthreads = 0;
                ha->isr_count = 0; 

                /* load the F/W, read paramaters, and init the H/W */
                ha->instance = num_hosts;
                if (qla1280_initialize_adapter(ha))
                {
                   printk(KERN_INFO "qla1x160: Failed to initialize QLA12160 on PCI Bus 1 Dev 2 \n");
                   qla1280_mem_free(ha);
                   scsi_unregister(host);
                   goto find_devices;
                }                
                host->max_channel = bdp->numPorts-1; 
		/* Register our resources with Linux */
		if( qla1280_register_with_Linux(ha, bdp->numPorts-1) ) {
		  printk(KERN_INFO "qla1x160: Failed to register resources for QLA12160 on PCI Bus 1 Dev 2\n");
		  qla1280_mem_free(ha);
		  scsi_unregister(host);
		  goto find_devices;
		}
                reg = ha->iobase;
                /* Disable ISP interrupts. */
		qla1280_disable_intrs(ha);
                /* Insure mailbox registers are free. */
                WRT_REG_WORD(&reg->semaphore, 0);
                WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
                WRT_REG_WORD(&reg->host_cmd, HC_CLR_HOST_INT);

                /* Enable chip interrupts. */
		qla1280_enable_intrs(ha);
                /* Insert new entry into the list of adapters */
                ha->next = NULL;
                /* this preferred device will always be the first one found */
                cur_ha = qla1280_hostlist = ha;
                num_hosts++;
        }
#endif

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
 find_devices:
#endif

        pdev = NULL;
	/* Try and find each different type of adapter we support */
	for(i=0;bdp->device_id != 0 && i < NUM_OF_ISP_DEVICES;i++,bdp++) {
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
                /* PCI_SUBSYSTEM_IDS supported */ 
		while ((pdev = pci_find_subsys(QLA1280_VENDOR_ID,
			bdp->device_id, PCI_ANY_ID, PCI_ANY_ID, pdev) )) {
 			if (pci_enable_device(pdev))
                          continue;
#else
		while ((pdev = pci_find_device(QLA1280_VENDOR_ID,
			bdp->device_id, pdev ) ))  {
#endif /* 2,3,18 */
                /* found a adapter */
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
		  /* If it's an AMI SubSys Vendor ID adapter, skip it. */
                  if (pdev->subsystem_vendor == PCI_VENDOR_ID_AMI) 
                    { 
                      printk("qla1x160: Skip AMI SubSys Vendor ID Chip\n");
                      continue;
                    }

                  /* 3.20 and 3.23 */
		  /* skip QLA12160 already initialized on PCI Bus 1 Dev 2 */
                  /* since we already initialized and presented it */
                  if ((bdp->device_id        == QLA12160_DEVICE_ID) &&
                      (pdev->bus->number     == 1)   &&
                      (PCI_SLOT(pdev->devfn) == 2))
                    continue;

 		  printk("qla1x160: Supported Device Found VID=%x DID=%x SSVID=%x SSDID=%x\n",
			pdev->vendor, pdev->device,
			pdev->subsystem_vendor, pdev->subsystem_device); 

#else
                  printk("qla1x160: Supported Device Found\n");
		  pci_read_config_word(pdev, PCI_SUBSYSTEM_VENDOR_ID,
                                       &subsys);
		  /* Bypass all AMI SUBSYS VENDOR IDs */
                  if (subsys == PCI_VENDOR_ID_AMI) 
                    { 
                      printk("qla1x160: Skip AMI SubSys Vendor ID  Chip\n");
                      continue;
                    }
#endif /* 2,3,18 */
		host = scsi_register(template, sizeof(scsi_qla_host_t));
		ha = (scsi_qla_host_t *) host->hostdata;
		/* Clear our data area */
		for( j =0, cp = (char *)ha;  j < sizeof(scsi_qla_host_t); j++)
			*cp++ = 0;
		/* Sanitize the information from PCI BIOS.  */
		host->irq = pdev->irq;
		host->io_port = pci_resource_start(pdev, 0);
		ha->pci_bus = pdev->bus->number;
		ha->pci_device_fn = pdev->devfn;
		ha->pdev = pdev;
		ha->device_id = bdp->device_id;
                ha->devnum = i; // specifies microcode load address

		if( qla1280_mem_alloc(ha) ) {
  	          printk(KERN_INFO "qla1x160: Failed to get memory\n");
		}
                
                ha->ports = bdp->numPorts; 
                /* following needed for all cases of OS versions */
                host->io_port &= PCI_BASE_ADDRESS_IO_MASK; 
                ha->iobase = (device_reg_t *) host->io_port;
                ha->host = host;
                ha->host_no = host->host_no;

                /* load the F/W, read paramaters, and init the H/W */
                ha->instance = num_hosts;
                if (qla1280_initialize_adapter(ha))
                {
                   printk(KERN_INFO "qla1x160:Failed to initialize adapter\n");
                   qla1280_mem_free(ha);
                   scsi_unregister(host);
                   continue;
                }
                
                host->max_channel = bdp->numPorts-1; 
		/* Register our resources with Linux */
		if( qla1280_register_with_Linux(ha, bdp->numPorts-1) ) {
		  printk(KERN_INFO "qla1x160: Failed to register resources\n");
		  qla1280_mem_free(ha);
		  scsi_unregister(host);
		  continue;
		}

                reg = ha->iobase;
                /* Disable ISP interrupts. */
		qla1280_disable_intrs(ha);

                /* Insure mailbox registers are free. */
                WRT_REG_WORD(&reg->semaphore, 0);
                WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
                WRT_REG_WORD(&reg->host_cmd, HC_CLR_HOST_INT);

                /* Enable chip interrupts. */
		qla1280_enable_intrs(ha);
               
                /* Insert new entry into the list of adapters */
                ha->next = NULL;
                if( qla1280_hostlist == NULL )
                {
                    cur_ha = qla1280_hostlist = ha;
                }
                else
                {
                    cur_ha = qla1280_hostlist;
                    while( cur_ha->next != NULL )
                        cur_ha = cur_ha->next;
                    cur_ha->next = ha;
                }
                num_hosts++;
            }  /* end of WHILE */
        } /* end of FOR */

    LEAVE("qla1280_detect");
    return num_hosts; 
}

/**************************************************************************
*   qla1280_register_with_Linux
*
* Description:
*   Free the passed in Scsi_Host memory structures prior to unloading the
*   module.
*
* Input:
*     ha - pointer to host adapter structure
*     maxchannels - MAX number of channels.
*
* Returns:
*  0 - Sucessfully reserved resources.
*  1 - Failed to reserved a resource.
**************************************************************************/
static int qla1280_register_with_Linux(scsi_qla_host_t *ha,
                                       unsigned int maxchannels)
{
    struct Scsi_Host *host = ha->host;

    host->can_queue = 0xfffff;  /* unlimited  */
    host->cmd_per_lun = 1;
    host->select_queue_depths = qla1280_select_queue_depth;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
    host->base = (unsigned char *) ha->mmpbase;
#else
    host->base = (unsigned long) ha->mmpbase;
#endif
    host->max_channel = maxchannels; 
    host->max_lun = MAX_LUNS-1; 
    host->unique_id = ha->instance;
    host->max_id = MAX_TARGETS; 
    host->unique_id = ha->instance;

    /* set our host ID  (need to do something about our two IDs) */
    host->this_id = ha->bus_settings[0].id;
    /* Register the IRQ with Linux (sharable) */
    if ( request_irq(host->irq, qla1280_intr_handler, SA_INTERRUPT| SA_SHIRQ, "qla1280", ha))
    {
      printk("qla1280 : Failed to reserved interrupt %d already in use\n", host->irq);
      qla1280_mem_free(ha);
      scsi_unregister(host);
      return 1;
    }

    /* Register the I/O space with Linux */
    if (check_region(host->io_port, 0xff))
    {
      printk("qla1280 : Failed to reserved i/o region 0x%04lx-0x%04lx already"
             " in use\n", host->io_port, host->io_port + 0xff);
      free_irq(host->irq, NULL);
      qla1280_mem_free(ha);
      scsi_unregister(host);
      return 1;
    }

    request_region(host->io_port, 0xff, "qla1280");

    return 0;
}


/**************************************************************************
 *   qla1280_release
 *   Free the passed in Scsi_Host memory structures prior to unloading the
 *   module.
 **************************************************************************/
int
qla1280_release(struct Scsi_Host *host)
{
    scsi_qla_host_t *ha = (scsi_qla_host_t *) host->hostdata;

    ENTER("qla1280_release");

    if( !ha->flags.online )
        return(0);

    /* turn-off interrupts on the card */
    WRT_REG_WORD(&ha->iobase->ictrl, 0);

    /* Detach interrupts */
    if(host->irq)
        free_irq(host->irq, ha);

    /* release io space registers  */
    if( host->io_port )
        release_region(host->io_port, 0xff);

#if MEMORY_MAPPED_IO
    if(ha->mmpbase)
        iounmap((void *) (((unsigned long) ha->mmpbase) & PAGE_MASK));
#endif /* MEMORY_MAPPED_IO */
    qla1280_mem_free(ha);

    ENTER("qla1280_release");
    return(0);
}

/**************************************************************************
 *   qla1280_info
 *     Return a string describing the driver.
 **************************************************************************/
const char *
qla1280_info(struct Scsi_Host *host)
{
    static char qla1280_buffer[125];
    char *bp;
    scsi_qla_host_t *ha;
    qla_boards_t   *bdp; 

    bp = &qla1280_buffer[0];
    ha = (scsi_qla_host_t *)host->hostdata;
    bdp = &QL1280BoardTbl[ha->devnum];
    memset(bp, 0, sizeof(qla1280_buffer));
    sprintf(bp,
            "QLogic %s PCI to SCSI Host Adapter: bus %d device %d irq %d\n"
            "       Firmware version: %2d.%02d.%02d, Driver version %s",
            (char *)&bdp->bdName[0], ha->pci_bus, (ha->pci_device_fn & 0xf8) >> 3, host->irq,
            bdp->fwver[0],bdp->fwver[1],bdp->fwver[2],
            QLA1280_VERSION);
    return(bp);
}

/**************************************************************************
 *   qla1200_queuecommand
 *     Queue a command to the controller.
 *
 * Note:
 * The mid-level driver tries to ensures that queuecommand never gets invoked
 * concurrently with itself or the interrupt handler (although the
 * interrupt handler may call this routine as part of request-completion
 * handling).   Unfortunely, it sometimes calls the scheduler in interrupt
 * context which is a big NO! NO!.
 **************************************************************************/
int
qla1280_queuecommand(Scsi_Cmnd *cmd, void (*fn)(Scsi_Cmnd *))
{
    scsi_qla_host_t *ha;
    srb_t  *sp;
    struct Scsi_Host *host;
    uint32_t        b, t, l;
    scsi_lu_t       *q;
    u_long          handle;

    /*ENTER("qla1280_queuecommand");
      COMTRACE('C');*/

    host = cmd->host;
    ha = (scsi_qla_host_t *) host->hostdata;

    /* send command to adapter */
    sp = (srb_t *) CMD_SP(cmd);
    sp->cmd = cmd;
    cmd->scsi_done = fn;
    if (cmd->flags == 0)  /* new command */
    {
        sp->flags = 0;
    }

    DEBUG5(qla1280_print_scsi_cmd(cmd));

    /* Generate LU queue on bus, target, LUN */
    b = SCSI_BUS_32(cmd);
    t = SCSI_TCN_32(cmd);
    l = SCSI_LUN_32(cmd);
    if((q = LU_Q(ha, b, t, l)) == NULL )
    {
        if( (q = (scsi_lu_t *)kmalloc(sizeof(struct scsi_lu), GFP_ATOMIC)))
        {
           LU_Q(ha, b, t, l) = q;
           memset(q, 0, sizeof(struct scsi_lu));
           DEBUG(sprintf(debug_buff,"Allocate new device queue 0x%p\n\r",(void *)q));
           DEBUG(qla1280_print(debug_buff));
        }
        else
        {
            CMD_RESULT(cmd) = (int) (DID_BUS_BUSY << 16);
            qla1280_done_q_put(sp, &ha->done_q_first, &ha->done_q_last);
/* 3.22 */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0) /* 3.22 */
            queue_task(&ha->run_qla_bh,&tq_scheduler);
#else  /* 3.22 */ 
            schedule_task(&ha->run_qla_bh); /* 3.22 */
#endif /* 3.22 */
            ha->flags.dpc_sched = TRUE;
            return(0);
        }
    }
    /* Set an invalid handle until we issue the command to ISP */
    /* then we will set the real handle value.                 */
    handle = INVALID_HANDLE;  
    CMD_HANDLE(cmd) = (unsigned char *)handle;

    /* add the command to our queue */
    ha->qthreads++;
    qla1280_putq_t(q,sp);

    DEBUG(sprintf(debug_buff,
     "qla1280_QC: t=%x CDB=%x I/OSize=0x%x haQueueCount=0x%lx\n\r",
     t,cmd->cmnd[0],CMD_XFRLEN(cmd),ha->qthreads));
    DEBUG(qla1280_print(debug_buff));

    /* send command to adapter */
    if (q->q_outcnt == 0)
      qla1280_restart_queues(ha);

    /*LEAVE("qla1280_queuecommand");*/
    return (0);
}

/**************************************************************************
 *   qla1200_abort
 *     Abort the speciifed SCSI command(s).
 **************************************************************************/
int
qla1280_abort(Scsi_Cmnd *cmd)
{
    scsi_qla_host_t *ha;
    srb_t  *sp;
    struct Scsi_Host *host;
    uint32_t        b, t, l;
    scsi_lu_t       *q;
    int return_status = SCSI_ABORT_SUCCESS;
    int found = 0;
    int i;
    unsigned char *handle;
    u16 data;

    ENTER("qla1280_abort");
    COMTRACE('A');
    ha = (scsi_qla_host_t *) cmd->host->hostdata;
    host = cmd->host;

    /* Get the SCSI request ptr */
    sp = (srb_t *) CMD_SP(cmd);
    handle = CMD_HANDLE(cmd);
    if (qla1280_verbose)
        printk("scsi(%li): ABORT Command=0x%p, handle=0x%p\n",
               ha->host_no, (void *)cmd, (void *)handle);

    /* Check for pending interrupts. */
    if( handle == NULL )
    {
      COMTRACE('a');
      /* we never got this command */
      printk(KERN_INFO "qla1280: Aborting a NULL handle\n");
      return(SCSI_ABORT_NOT_RUNNING);  /* no action - we don't have command */
    }
    data = qla1280_debounce_register(&ha->iobase->istatus);
    if( !(ha->flags.in_isr) && (data & RISC_INT) )
    {
        /* put any pending command in done queue */
        qla1280_isr(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
    }

    /*
     * This seems unnecessary, it's not used below! / Jes
     */
#ifdef UNUSED
    handle = CMD_HANDLE(cmd);
#endif

    /* Generate LU queue on bus, target, LUN */
    b = SCSI_BUS_32(cmd);
    t = SCSI_TCN_32(cmd);
    l = SCSI_LUN_32(cmd);
    if((q = LU_Q(ha, b, t, l)) == NULL )
    {
      COMTRACE('a');
      /* No lun queue -- command must not be active */
      printk(KERN_WARNING "qla1280 (%d:%d:%d): No LUN queue for the specified device\n",(int)b,(int)t,(int)l);
      return(SCSI_ABORT_NOT_RUNNING);  /* no action - we don't have command */
    }

#if AUTO_ESCALATE_ABORT
    if ( (sp->flags & SRB_ABORTED) )
    {
        DEBUG(qla1280_print("qla1280_abort: Abort escalayted - returning SCSI_ABORT_SNOOZE.\n\r"));
        return(SCSI_ABORT_SNOOZE);
    }
#endif

    if ( (sp->flags & SRB_ABORT_PENDING) )
    {
      COMTRACE('a');
      if( qla1280_verbose  )
        printk("scsi(): Command has a pending abort message - ABORT_PENDING.\n");
        DEBUG(qla1280_print("qla1280: Command has a pending abort message - ABORT_PENDING.\n\r"));
        return(SCSI_ABORT_PENDING);
    }

#if  STOP_ON_ABORT 
    printk("Scsi layer issued a ABORT command= 0x%x\n",(int)cmd);
    DEBUG2(qla1280_print_scsi_cmd(cmd));
#endif

    ha->flags.in_abort = TRUE;
    /*
    * Normally, would would need to search our queue for the specified command
    * but; since our sp contains the cmd ptr, we can just remove it from our
    * LUN queue.
    */
    if(!(sp->flags & SRB_SENT))
    { 
        found++;
        if( qla1280_verbose  )
            printk("scsi(): Command returned from queue aborted.\n");
        DEBUG(qla1280_print("qla1280: Command returned from queue aborted.\n\r"));
        /* Remove srb from SCSI LU queue. */
        qla1280_removeq(q, sp);
        sp->flags |=  SRB_ABORTED;
        CMD_RESULT(cmd) = DID_ABORT << 16; 
        qla1280_done_q_put(sp, &ha->done_q_first, &ha->done_q_last);
        return_status = SCSI_ABORT_SUCCESS;
    }
    else
    {  /* find the command in our active list */
        for (i = 1; i < MAX_OUTSTANDING_COMMANDS; i++)
        {
            if( sp == ha->outstanding_cmds[i] )
            {
                found++;
                DEBUG(qla1280_print("qla1280: RISC aborting command.\n\r"));
                qla1280_abort_command(ha,sp);
                return_status = SCSI_ABORT_PENDING;
                break;
            }
        }
    }

#if  STOP_ON_ABORT 
    qla1280_panic("qla1280_abort",ha->host);
#endif
    if ( found == 0 )
        return_status = SCSI_ABORT_NOT_RUNNING;  /* no action - we don't have command */

    DEBUG(sprintf(debug_buff, "qla1280_abort: Aborted status returned = 0x%x.\n\r",return_status));
    DEBUG(qla1280_print(debug_buff));

    if( ha->done_q_first )
       qla1280_done(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
    if ( found )
    {
        qla1280_restart_queues(ha);
    }
    ha->flags.in_abort = FALSE;

    LEAVE("qla1280_abort");
    COMTRACE('a');
    return(return_status);
}

/**************************************************************************
 * qla1200_reset
 *    The reset function will reset the SCSI bus and abort any executing
 *    commands. 
 *
 * Input:
 *      cmd = Linux SCSI command packet of the command that cause the
 *            bus reset.
 *      flags = SCSI bus reset option flags (see scsi.h)
 *
 * Returns:
 *      DID_RESET in cmd.host_byte of aborted command(s) 
 *
 * Note:
 *      Resetting the bus always succeeds - is has to, otherwise the
 *      kernel will panic! Try a surgical technique - sending a BUS
 *      DEVICE RESET message - on the offending target before pulling
 *      the SCSI bus reset line.
 **************************************************************************/
int
qla1280_reset(Scsi_Cmnd *cmd, unsigned int flags)
{
    scsi_qla_host_t *ha;
    uint32_t        b, t, l;
    srb_t  *sp;
    typedef enum
    {
        ABORT_DEVICE = 1, 
                DEVICE_RESET = 2, 
                BUS_RESET    = 3, 
                ADAPTER_RESET= 4,
                RESET_DELAYED= 5,
                FAIL         = 6
    } action_t;
    action_t     action = ADAPTER_RESET;
    u_short    data;
    scsi_lu_t       *q;
    int result;


    ENTER("qla1280_reset");
    COMTRACE('R');
    if (qla1280_verbose)
        printk("scsi(): Resetting Cmnd=0x%lx, Handle=0x%lx, flags=0x%x\n",(long)cmd,(long)CMD_HANDLE(cmd),flags);
    if ( cmd == NULL )
    {
        printk(KERN_WARNING "(scsi?:?:?:?) Reset called with NULL Scsi_Cmnd "
                "pointer, failing.\n");
        return(SCSI_RESET_SNOOZE);
    }
    ha = (scsi_qla_host_t *) cmd->host->hostdata;
    sp = (srb_t *) CMD_SP(cmd);

#if  STOP_ON_RESET 
    qla1280_panic("qla1280_reset",ha->host);
#endif 

    /* Check for pending interrupts. */
    data = qla1280_debounce_register(&ha->iobase->istatus);
    if( !(ha->flags.in_isr) && (data & RISC_INT) )
      qla1280_isr(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);

    /*
    * Determine the suggested action that the mid-level driver wants
    * us to perform.
    */
    if( CMD_HANDLE(cmd) == NULL )
    {
        /* 
        * if mid-level driver called reset with a orphan SCSI_Cmnd 
        * (i.e. a command that's not pending ), so perform the 
        * function specified.
        */
        if( (flags & SCSI_RESET_SUGGEST_HOST_RESET) )
            action = ADAPTER_RESET;
        else 
            action = BUS_RESET;
    }
    else
    { /* 
        * Mid-level driver has called reset with this SCSI_Cmnd and 
        * its pending.
        */
        if( flags & SCSI_RESET_SUGGEST_HOST_RESET )
            action = ADAPTER_RESET;
        else if( flags & SCSI_RESET_SUGGEST_BUS_RESET )
            action = BUS_RESET;
        else 
            action = DEVICE_RESET;
    }

    b = SCSI_BUS_32(cmd);
    t = SCSI_TCN_32(cmd);
    l = SCSI_LUN_32(cmd);
    q = LU_Q(ha, b, t, l);

#if AUTO_ESCALATE_RESET
    if ( (action & DEVICE_RESET) && (q->q_flag & QLA1280_QRESET) )
    {
        printk(KERN_INFO "qla1280(%d): Bus device reset already sent to " "device, escalating.\n", (int)ha->host_no);
        action = BUS_RESET;
    }
    if ( (action & DEVICE_RESET) && (sp->flags & SRB_ABORT_PENDING) )
    {
        printk(KERN_INFO "qla1280(%d):Have already attempted to reach " "device with abort device\n", (int)ha->host_no);
        printk(KERN_INFO "qla1280(%d):message, will escalate to BUS " "RESET.\n",(int) ha->host_no);
        action = BUS_RESET;
    }
#endif

    /*
    *  By this point, we want to already know what we are going to do,
    *  so we only need to perform the course of action.
    */
    result = SCSI_RESET_ERROR;
    switch (action)
    {
        case FAIL:
            break;

        case RESET_DELAYED:
            result = SCSI_RESET_PENDING;
            break;

        case ABORT_DEVICE:
            ha->flags.in_reset = TRUE;
            if (qla1280_verbose)
                printk(KERN_INFO "scsi(%d:%d:%d:%d): Queueing abort device command.\n", (int)ha->host_no,(int)b,(int)t,(int)l); 
            qla1280_abort_queue_single(ha,b,t,l,DID_ABORT);
            if( qla1280_abort_device(ha, b, t, l) == 0)
                result = SCSI_RESET_PENDING;
            break;

        case DEVICE_RESET:
            if (qla1280_verbose)
                printk(KERN_INFO "scsi(%d:%d:%d:%d): Queueing device reset command.\n",(int) ha->host_no,(int)b,(int)t,(int)l); 
            ha->flags.in_reset = TRUE;
            for (l = 0; l < MAX_LUNS; l++)
                qla1280_abort_queue_single(ha,b,t,l,DID_ABORT);
            if( qla1280_device_reset(ha, b, t) == 0 ) 
                result = SCSI_RESET_PENDING;
            q->q_flag |= QLA1280_QRESET;
            break;

        case BUS_RESET:
                if (qla1280_verbose)
                    printk(KERN_INFO "qla1280(%d:%d:%d:%d): Issuing BUS DEVICE RESET.\n",(int) ha->host_no,(int)b,(int)t,(int)l); 
            ha->flags.in_reset = TRUE;
            for (t = 0; t < MAX_TARGETS; t++)
                for (l = 0; l < MAX_LUNS; l++)
                    qla1280_abort_queue_single(ha,b,t,l,DID_RESET);
                qla1280_bus_reset(ha, b);
                /*
                * The bus reset routine returns all the outstanding commands back
                * with "DID_RESET" in the status field after a short delay
                * by the firmware. If the mid-level time out the SCSI reset before
                * our delay we may need to ignore it.
                */
                /* result = SCSI_RESET_PENDING | SCSI_RESET_BUS_RESET; */
                result = SCSI_RESET_SUCCESS | SCSI_RESET_BUS_RESET;
                mdelay(4 * 1000); barrier();
                if( flags & SCSI_RESET_SYNCHRONOUS )
                {
                  CMD_RESULT(cmd) = (int) (DID_BUS_BUSY << 16);
                  (*(cmd)->scsi_done)(cmd); 
                }
                /* ha->reset_start = jiffies; */
                break;

            case ADAPTER_RESET:
            default:
                if (qla1280_verbose)
                {
                    printk(KERN_INFO "scsi(%d:%d:%d:%d): Issued an ADAPTER RESET.\n",(int) ha->host_no,(int)b,(int)t,(int)l); 
                    printk(KERN_INFO "scsi(%d:%d:%d:%d): I/O processing will continue automatically.\n",(int) ha->host_no,(int)b,(int)t,(int)l); 
                }
                ha->flags.reset_active = TRUE;
                /* 
                * We restarted all of the commands automatically, so the mid-level code can expect 
                * completions momentitarily.
                */
                if( qla1280_abort_isp(ha) == 0 )
                    result = SCSI_RESET_SUCCESS | SCSI_RESET_HOST_RESET;

                        ha->flags.reset_active = FALSE;
    }

    if( ha->done_q_first ) 
        qla1280_done(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
    qla1280_restart_queues(ha);
    ha->flags.in_reset = FALSE;
    
    DEBUG(printk("RESET returning %d\n", result)); 

    COMTRACE('r');
    LEAVE("qla1280_reset");
    return( result );
}

/**************************************************************************
 * qla1200_biosparam
 *   Return the disk geometry for the given SCSI device.
 **************************************************************************/
int
qla1280_biosparam(Disk *disk, kdev_t dev, int geom[])
{
    int heads, sectors, cylinders;

            heads = 64;
    sectors = 32;
    cylinders = disk->capacity / (heads * sectors);
    if (cylinders > 1024)
    {
        heads = 255;
        sectors = 63;
        cylinders = disk->capacity / (heads * sectors);
        /* if (cylinders > 1023)
        cylinders = 1023; */
    }

    geom[0] = heads;
    geom[1] = sectors;
    geom[2] = cylinders;

    return (0);
}
/**************************************************************************
 * qla1280_intr_handler
 *   Handles the H/W interrupt
 **************************************************************************/
void qla1280_intr_handler(int irq, void *dev_id, struct pt_regs *regs)
{
    scsi_qla_host_t *ha;
    device_reg_t *reg;
    unsigned long cpu_flags;
    u_short    data;

    ENTER_INTR("qla1280_intr_handler");
    COMTRACE('I');
    ha = (scsi_qla_host_t *) dev_id;
    if(!ha)
    {
        printk(KERN_INFO "scsi(): Interrupt with NULL host ptr\n");
        COMTRACE('X');
        return;
    }
    spin_lock_irqsave(&io_request_lock, cpu_flags);
    if(test_and_set_bit(QLA1280_IN_ISR_BIT, &ha->flags))
    {
      COMTRACE('X');
      spin_unlock_irqrestore(&io_request_lock, cpu_flags);
      return;
    }
    ha->isr_count++;
    reg = ha->iobase;
     /* disable our interrupt. */
    WRT_REG_WORD(&reg->ictrl, 0); 
    data = qla1280_debounce_register(&reg->istatus);
    /* Check for pending interrupts. */
    if ( !(data & RISC_INT) )
    {
        /* spurious interrupts can happen legally */
        DEBUG(printk("scsi(%d): Spurious interrupt - ignoring\n",(int)ha->host_no));
        COMTRACE('X');
    }
    else
      qla1280_isr(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
    if (ha->done_q_first)
        qla1280_done(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);

    clear_bit(QLA1280_IN_ISR_BIT, &ha->flags);
    spin_unlock_irqrestore(&io_request_lock, cpu_flags);

     /* enable our interrupt. */
    WRT_REG_WORD(&reg->ictrl, (ISP_EN_INT | ISP_EN_RISC));

    COMTRACE('i');
    LEAVE_INTR("qla1280_intr_handler");
}

/**************************************************************************
 *   qla1280_do_dpc
 *
 * Description:
 * This routine is a task that is schedule by the interrupt handler 
 * to perform the background processing for interrupts.  We put it 
 * on a task queue that is consumed whenever the scheduler runs; that's
 * so you can do anything (i.e. put the process to sleep etc).  In fact, the 
 * mid-level tries to sleep when it reaches the driver threshold 
 * "host->can_queue". This can cause a panic if we were in our interrupt
 * code .
 **************************************************************************/
void qla1280_do_dpc(void *p)
{
    scsi_qla_host_t *ha = (scsi_qla_host_t *) p;
    unsigned long cpu_flags;

    COMTRACE('p');
    spin_lock_irqsave(&io_request_lock, cpu_flags);

    if (ha->flags.isp_abort_needed)
        qla1280_abort_isp(ha);

    if (ha->flags.reset_marker)
        qla1280_rst_aen(ha);

    if (ha->done_q_first)
        qla1280_done(ha, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
    ha->flags.dpc_sched = FALSE;

    spin_unlock_irqrestore(&io_request_lock, cpu_flags);
}

/**************************************************************************
 *   qla1280_device_queue_depth
 *
 * Description:
 *   Determines the queue depth for a given device.  There are two ways
 *   a queue depth can be obtained for a tagged queueing device.  One
 *   way is the default queue depth which is determined by whether
 *   If it is defined, then it is used
 *   as the default queue depth.  Otherwise, we use either 4 or 8 as the
 *   default queue depth (dependent on the number of hardware SCBs).
 **************************************************************************/
static void qla1280_device_queue_depth(scsi_qla_host_t *p, Scsi_Device *device)
{
    int default_depth = 3;
    int bus = device->channel;
    int target = device->id;

    device->queue_depth = default_depth;

    if (device->tagged_supported &&
        (p->bus_settings[bus].qtag_enables & (BIT_0 << target)) )
    {
        device->tagged_queue = 1;
        device->current_tag = 0;
        device->queue_depth = p->bus_settings[bus].hiwat; 
        /* device->queue_depth = 20; */
        printk(KERN_INFO "scsi(%d:%d:%d:%d): Enabled tagged queuing, queue depth %d.\n",
                (int)p->host_no, device->channel, device->id,
                device->lun, device->queue_depth);
    }
    qla12160_get_target_parameters(p, bus, target, device->lun);

}

/**************************************************************************
 *   qla1280_select_queue_depth
 *
 *   Sets the queue depth for each SCSI device hanging off the input
 *   host adapter.  We use a queue depth of 2 for devices that do not
 *   support tagged queueing.
 **************************************************************************/
static void
qla1280_select_queue_depth(struct Scsi_Host *host, Scsi_Device *scsi_devs)
{
    Scsi_Device *device;
    scsi_qla_host_t  *p = (scsi_qla_host_t *) host->hostdata;

    ENTER("qla1280_select_queue_depth");
    for (device = scsi_devs; device != NULL; device = device->next)
    {
        if (device->host == host)
            qla1280_device_queue_depth(p, device);
    }
    LEAVE("qla1280_select_queue_depth");
}

/*--------------------------**
** Driver Support Routines  **
**--------------------------*/


/*
 * qla1280_done
 *      Process completed commands.
 *
 * Input:
 *      ha           = adapter block pointer.
 *      done_q_first = done queue first pointer.
 *      done_q_last  = done queue last pointer.
 */
static void
qla1280_done(scsi_qla_host_t *ha, srb_t **done_q_first, srb_t **done_q_last)
{
    srb_t           *sp;
    scsi_lu_t       *q;
    uint32_t        b, t, l;
    Scsi_Cmnd  *cmd;

    ENTER("qla1280_done");
    COMTRACE('D');

    while (*done_q_first !=  NULL)
    {
        /* remove command from done list */
                sp = *done_q_first;
        if (!(*done_q_first = sp->s_next))
            *done_q_last = NULL;
        else
            (*done_q_first)->s_prev = NULL;

        cmd = sp->cmd;
        b = SCSI_BUS_32(cmd);
        t = SCSI_TCN_32(cmd);
        l = SCSI_LUN_32(cmd);
        q = LU_Q(ha, b, t, l);

        /* Decrement outstanding commands on device. */
        if (q->q_outcnt)
            q->q_outcnt--;
        if (q->q_outcnt < ha->bus_settings[b].hiwat)
        {
            q->q_flag &= ~QLA1280_QBUSY;
        }
        
        q->io_cnt++;
        if( sp->dir & BIT_5 )
         q->r_cnt++;
        else
         q->w_cnt++;

        switch ( (CMD_RESULT(cmd)>>16))
        {
            case DID_RESET:
                q->q_flag &= ~QLA1280_QRESET;
                /* Issue marker command. */
                qla1280_marker(ha, b, t, 0, MK_SYNC_ID); 
                break;
            case DID_ABORT:
                sp->flags &= ~SRB_ABORT_PENDING;
                sp->flags |= SRB_ABORTED;
                if (sp->flags & SRB_TIMEOUT)
                    CMD_RESULT(sp->cmd)= DID_TIME_OUT << 16;
                break;
            default:
                break;
        }

        /* 3.13   64 and 32 bit */
        /* Release memory used for this I/O */
        if (cmd->use_sg) {
            DEBUG(sprintf(debug_buff,
            "S/G unmap_sg cmd=%p\n\r",(void *)cmd);)
            DEBUG(qla1280_print(debug_buff));
            pci_unmap_sg(ha->pdev, cmd->request_buffer,
                         cmd->use_sg,
                         scsi_to_pci_dma_dir(cmd->sc_data_direction));
        }
        else if (cmd->request_bufflen) {
          /*DEBUG(sprintf(debug_buff,
                 "No S/G unmap_single cmd=%x saved_dma_handle=%lx\n\r",
                  cmd,sp->saved_dma_handle);)
                  DEBUG(qla1280_print(debug_buff);)*/

                  pci_unmap_single(ha->pdev,sp->saved_dma_handle,
                                  cmd->request_bufflen,
                                  scsi_to_pci_dma_dir(cmd->sc_data_direction));
       }

        /* Call the mid-level driver interrupt handler */
        CMD_HANDLE(sp->cmd) = NULL;
        ha->actthreads--;

        (*(cmd)->scsi_done)(cmd);

        qla1280_next(ha, q, b);
    }
    COMTRACE('d');
    LEAVE("qla1280_done");
}

/*
 * Translates a ISP error to a Linux SCSI error
 */
static int qla1280_return_status( sts_entry_t *sts, Scsi_Cmnd       *cp)
{
    int host_status = DID_ERROR;
#if DEBUG_QLA1280_INTR
    static char *reason[] =
    {
        "DID_OK",
                "DID_NO_CONNECT",
                "DID_BUS_BUSY",
                "DID_TIME_OUT",
                "DID_BAD_TARGET",
                "DID_ABORT",
                "DID_PARITY",
                "DID_ERROR",
                "DID_RESET",
                "DID_BAD_INTR"
    };
#endif /* DEBUG_QLA1280_INTR */

    ENTER("qla1280_return_status");

#if DEBUG_QLA1280_INTR
    /*
    DEBUG(printk("qla1280_return_status: compl status = 0x%04x\n", sts->comp_status));
    */
#endif
    switch(sts->comp_status)
    {
        case CS_COMPLETE:
            host_status = DID_OK;
            break;
        case CS_INCOMPLETE:
            if (!(sts->state_flags & SF_GOT_BUS))
                host_status = DID_NO_CONNECT;
            else if (!(sts->state_flags & SF_GOT_TARGET))
                host_status = DID_BAD_TARGET;
            else if (!(sts->state_flags & SF_SENT_CDB))
                host_status = DID_ERROR;
            else if (!(sts->state_flags & SF_TRANSFERRED_DATA))
                host_status = DID_ERROR;
            else if (!(sts->state_flags & SF_GOT_STATUS))
                host_status = DID_ERROR;
            else if (!(sts->state_flags & SF_GOT_SENSE))
                host_status = DID_ERROR;
            break;
        case CS_RESET:
            host_status = DID_RESET;
            break;
        case CS_ABORTED:
            host_status = DID_ABORT;
            break;
        case CS_TIMEOUT:
            host_status = DID_TIME_OUT;
            break;
        case CS_DATA_OVERRUN:
#ifdef QL_DEBUG_LEVEL_2 
            printk("Data overrun 0x%x\n",(int)sts->residual_length);
            qla1280_print(
                        "\n\rqla1280_isr: response packet data\n\r");
                        qla1280_dump_buffer((char *)sts, RESPONSE_ENTRY_SIZE); 
#endif
            host_status = DID_ERROR;
            break;
        case CS_DATA_UNDERRUN:
            if ( (CMD_XFRLEN(cp) - sts->residual_length) < cp->underflow)    
            { 
              printk("scsi: Underflow detected - retrying command.\n");
              host_status = DID_ERROR;
            }
            else
                host_status = DID_OK;
            break;
        default:
            host_status = DID_ERROR;
            break;
    }

#if DEBUG_QLA1280_INTR
    sprintf(debug_buff, "qla1280 ISP status: host status (%s) scsi status %x\n\r", reason[host_status], sts->scsi_status);
    qla1280_print(debug_buff);
#endif

    LEAVE("qla1280_return_status");

    return (sts->scsi_status & 0xff) | (host_status << 16);
}

/*
 * qla1280_done_q_put
 *      Place SRB command on done queue.
 *
 * Input:
 *      sp           = srb pointer.
 *      done_q_first = done queue first pointer.
 *      done_q_last  = done queue last pointer.
 */
static void
qla1280_done_q_put(srb_t *sp, srb_t **done_q_first, srb_t **done_q_last)
{
#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_put_done_q");
#endif
    /* Place block on done queue */
    sp->s_next = NULL;
    sp->s_prev = *done_q_last;
    if (!*done_q_first)
        *done_q_first = sp;
    else
        (*done_q_last)->s_next = sp;
    *done_q_last = sp;

#ifdef QL_DEBUG_LEVEL_3
            LEAVE("qla1280_put_done_q");
#endif
}

/*
 * qla1280_next
 *      Retrieve and process next job in the queue.
 *
 * Input:
 *      ha = adapter block pointer.
 *      q  = SCSI LU pointer.
 *      b  = SCSI bus number.
 *      SCSI_LU_Q lock must be already obtained and no other locks.
 *
 * Output:
 *      Releases SCSI_LU_Q upon exit.
 */
static void
qla1280_next(scsi_qla_host_t *ha, scsi_lu_t *q, uint8_t b)
{
    srb_t   *sp;
    int cnt, status;

    ENTER("qla1280_next");

    while ( ((sp = q->q_first) != NULL) &&     /* we have a queue pending */
        !(q->q_flag &  QLA1280_QBUSY) &&      /* device not busy */
        !ha->flags.abort_isp_active &&      /* not resetting the adapter */
        !(q->q_flag & QLA1280_QSUSP) )      /* device not suspended */
    {
        /* Remove srb from SCSI LU queue. */
        qla1280_removeq(q, sp);

        DEBUG(sprintf(debug_buff,"starting request 0x%p<-(0x%p)\n\r",
                      (void *)q, (void *)sp));
        DEBUG(qla1280_print(debug_buff));
        {
            /* Set busy flag if reached high water mark. */
            q->q_outcnt++;
            if (q->q_outcnt >= ha->bus_settings[b].hiwat)
                q->q_flag |= QLA1280_QBUSY;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
            if (ha->flags.enable_64bit_addressing)
                status = qla1280_64bit_start_scsi(ha, sp);
            else
#endif
                status = qla1280_32bit_start_scsi(ha, sp);

            if (status)  /* if couldn't start the request */
            {
                if (q->q_outcnt == 1)
                {
                    /* Release SCSI LU queue specific lock */
                    QLA1280_SCSILU_UNLOCK(q);

                    /* Wait for 30 sec for command to be accepted. */
                    for (cnt = 6000000; cnt; cnt--)
                    {
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
                        if (ha->flags.enable_64bit_addressing)
                            status = qla1280_64bit_start_scsi(ha, sp);
                        else
#endif
                            status = qla1280_32bit_start_scsi(ha, sp);

                        if (!status)
                        {
                            break;
                        }

                        /* Go check for pending interrupts. */
                        qla1280_poll(ha);

                        SYS_DELAY(5);  /* 10 */
                    }
                    if (!cnt)
                    {
                        /* Set timeout status */
                        CMD_RESULT(sp->cmd) = DID_TIME_OUT << 16;

#if WATCHDOGTIMER
                        /* Remove command from watchdog queue. */
                        if (sp->flags & SRB_WATCHDOG)
                            qla1280_timeout_remove(ha, sp);
#endif
                        COMTRACE('M');
                        CMD_HANDLE(sp->cmd) = NULL;

                        /* Call the mid-level driver interrupt handler */
                        (*(sp->cmd)->scsi_done)(sp->cmd);

                        /* Acquire LU queue specific lock */
                        QLA1280_SCSILU_LOCK(q);

                        if (q->q_outcnt)
                            q->q_outcnt--;
                    }
                    else
                        /* Acquire LU queue specific lock */
                        QLA1280_SCSILU_LOCK(q);
                }
                else
                {   /* Place request back on top of device queue. */
                    qla1280_putq_t(q, sp);

                    if (q->q_outcnt)
                        q->q_outcnt--;
                    if (q->q_outcnt < ha->bus_settings[b].hiwat)
                        q->q_flag &= ~QLA1280_QBUSY;
                    break;
                }
            }
        }
    }

    /* Release SCSI LU queue specific lock */
    QLA1280_SCSILU_UNLOCK(q);

    LEAVE("qla1280_next");
}

/*
 * qla1280_putq_t
 *      Add the standard SCB job to the top of standard SCB commands.
 *
 * Input:
 *      q  = SCSI LU pointer.
 *      sp = srb pointer.
 *      SCSI_LU_Q lock must be already obtained.
 */
static void
qla1280_putq_t(scsi_lu_t *q, srb_t *sp)
{
    srb_t *srb_p;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_putq_t");
#endif
    DEBUG(sprintf(debug_buff,"Adding to device q=0x%p<-(0x%p)sp\n\r",
                  (void *)q, (void *)sp));
    DEBUG(qla1280_print(debug_buff));
    sp->s_next = NULL;
    if (!q->q_first)                  /* If queue empty */
    {
        sp->s_prev = NULL;
        q->q_first = sp;
        q->q_last = sp;
    }
    else
    {      
        srb_p = q->q_first;
        while (srb_p )
            srb_p = srb_p->s_next;

        if (srb_p)
        {
            sp->s_prev = srb_p->s_prev;
            if (srb_p->s_prev)
                srb_p->s_prev->s_next = sp;
            else
                q->q_first = sp;
            srb_p->s_prev = sp;
            sp->s_next = srb_p;
        }
        else
        {
            sp->s_prev = q->q_last;
            q->q_last->s_next = sp;
            q->q_last = sp;
        }
    }

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_putq_t");
#endif
}

/*
 * qla1280_removeq
 *      Function used to remove a command block from the
 *      LU queue.
 *
 * Input:
 *      q  = SCSI LU pointer.
 *      sp = srb pointer.
 *      SCSI_LU_Q lock must be already obtained.
 */
static void
qla1280_removeq(scsi_lu_t *q, srb_t *sp)
{
    DEBUG(sprintf(debug_buff,"Removing from device_q (0x%p)->(0x%p)\n\r",
                  (void *)q, (void *)sp));
    DEBUG(qla1280_print(debug_buff));
            if (sp->s_prev)
    {
        if ((sp->s_prev->s_next = sp->s_next) != NULL)
            sp->s_next->s_prev = sp->s_prev;
        else
            q->q_last = sp->s_prev;
    }
    else if (!(q->q_first = sp->s_next))
        q->q_last = NULL;
    else
        q->q_first->s_prev = NULL;
}

/*
* qla1280_mem_alloc
*      Allocates adapter memory.
*
* Returns:
*      0  = success.
*      1  = failure.
*/
static int
qla1280_mem_alloc(scsi_qla_host_t *ha)
{

	uint8_t   status = 1;
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
        dma_addr_t     dma_handle;
#endif

#ifdef QL_DEBUG_LEVEL_3
	ENTER("qla1280_mem_alloc");
#endif

        /* 3.13 */
        /* get consistent memory allocated for request and response rings */
        ha->request_ring  = pci_alloc_consistent(ha->pdev,
                               ((REQUEST_ENTRY_CNT+1)*(sizeof(request_t))),
                               &dma_handle);
        ha->request_dma    = dma_handle;
        ha->response_ring  = pci_alloc_consistent(ha->pdev,
                               ((RESPONSE_ENTRY_CNT+1)*(sizeof(response_t))),
                               &dma_handle);
        ha->response_dma   = dma_handle;
        status = 0; 

	if(status) {
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
		qla1280_print("qla1280_mem_alloc: **** FAILED ****\n");
#endif
	}
#ifdef QL_DEBUG_LEVEL_3
	else
		LEAVE("qla1280_mem_alloc");
#endif
	return(status);
}

/*
 * qla1280_mem_free
 *      Frees adapter allocated memory.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void
qla1280_mem_free(scsi_qla_host_t *ha)
{
    scsi_lu_t *q;
    uint32_t  b, t, l;

    ENTER("qlc1280_mem_free");
    if (ha)
    {
        /* Free device queues. */
        for (b = 0; b < MAX_BUSES; b++)
        {
            q = LU_Q(ha, b, ha->bus_settings[b].id, 0);
            for (t = 0; t < MAX_TARGETS; t++)
                for (l = 0; l < MAX_LUNS; l++)
                    if (LU_Q(ha, b, t, l) != NULL && LU_Q(ha, b, t, l) != q)
                        kfree(LU_Q(ha, b, t, l));
                    kfree(q);
        }
        for( b =0; b < MAX_EQ; b++ )
            ha->dev[b] =  (scsi_lu_t  *)NULL;
    }

    /* 3.13 */
    /* free consistent memory allocated for request and response rings */
    pci_free_consistent(ha->pdev, ((REQUEST_ENTRY_CNT+1)*(sizeof(request_t))),
                        ha->request_ring, ha->request_dma);

    pci_free_consistent(ha->pdev,((RESPONSE_ENTRY_CNT+1)*(sizeof(response_t))),
    ha->response_ring, ha->response_dma); 

    LEAVE("qlc1280_mem_free");
}




/****************************************************************************/
/*                QLogic ISP1280 Hardware Support Functions.                */
/****************************************************************************/

 /*
 * qla2100_enable_intrs
 * qla2100_disable_intrs
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      None      
 */
static inline void qla1280_enable_intrs(scsi_qla_host_t *ha) {
  device_reg_t *reg;

  reg = ha->iobase;
  ha->flags.interrupts_on = 1;
  /* enable risc and host interrupts */
  WRT_REG_WORD(&reg->ictrl, (ISP_EN_INT | ISP_EN_RISC));
}

static inline void qla1280_disable_intrs(scsi_qla_host_t *ha) {
  device_reg_t *reg;

  reg = ha->iobase;
  ha->flags.interrupts_on = 0;
  /* disable risc and host interrupts */
  WRT_REG_WORD(&reg->ictrl, 0);
}

/*
 * qla1280_initialize_adapter
 *      Initialize board.
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_initialize_adapter(scsi_qla_host_t *ha)
{
    device_reg_t *reg;
    int	status;
    /* int      cnt; */
    uint8_t      b;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_initialize_adapter");
#endif

    /* Clear adapter flags. */
    ha->flags.online = FALSE;
    ha->flags.isp_abort_needed = FALSE;
    ha->flags.disable_host_adapter = FALSE;
    ha->flags.reset_active = FALSE;
    ha->flags.abort_isp_active = FALSE;
    ha->flags.watchdog_enabled = FALSE;

    DEBUG(printk("Configure PCI space for adapter...\n"));
    if (!(status = qla1280_pci_config(ha)))
    {
        reg = ha->iobase;

        /* Disable ISP interrupts. */
        WRT_REG_WORD(&reg->ictrl, 0);

        /* Insure mailbox registers are free. */
        WRT_REG_WORD(&reg->semaphore, 0);
        WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
        WRT_REG_WORD(&reg->host_cmd, HC_CLR_HOST_INT);

        /* If firmware needs to be loaded */
        if (qla1280_verbose)
         printk("scsi(%d): Determining if RISC is loaded...\n",(int)ha->host_no);
        if (qla1280_isp_firmware(ha))
        {
            if (qla1280_verbose)
              printk("scsi(%d): Verifying chip...\n",(int)ha->host_no);
            if (!(status = qla1280_chip_diag(ha)))
            {
                if (qla1280_verbose)
                  printk("scsi(%d): Setup chip...\n",(int)ha->host_no);
                status = qla1280_setup_chip(ha); 
            }
        }
        else
          printk("initialize: isp_firmware() failed!\n");

        if (!status)
        {
            /* Setup adapter based on NVRAM parameters. */
            if (qla1280_verbose)
              printk("scsi(%d): Configure NVRAM parameters...\n",(int)ha->host_no);
            qla1280_nvram_config(ha);

            if (!ha->flags.disable_host_adapter &&
                !qla1280_init_rings(ha))
            {
                /* Issue SCSI reset. */
                for (b = 0; b < ha->ports; b++)
                    if (!ha->bus_settings[b].disable_scsi_reset)
                    {
                      /* dg 03/13 if we can't reset twice then bus is dead */
                        if( qla1280_bus_reset(ha, b) )
                           if( qla1280_bus_reset(ha, b) )
                           {
                               ha->bus_settings[b].scsi_bus_dead = TRUE;
                            }
                    }

                    do
                    {
                        /* Issue marker command. */
                        ha->flags.reset_marker = FALSE;
                        for (b = 0; b < ha->ports; b++)
                        {
                            ha->bus_settings[b].reset_marker = FALSE;
                            qla1280_marker(ha, b, 0, 0, MK_SYNC_ALL);
                        }
                    }while (ha->flags.reset_marker);

                    ha->flags.online = TRUE;

                    /* Enable host adapter target mode. */
                    for (b = 0; b < ha->ports; b++)
                    {
                        if (!(status = qla1280_enable_tgt(ha, b)))
                        {
                            /* for (cnt = 0; cnt < MAX_LUNS; cnt++)
                            {
                                qla1280_enable_lun(ha, b, cnt);
                                 qla1280_poll(ha);
                            }*/
                        }
                        else
                            break;
                    }
            }
            else
                status = 1;
        }
    }
    else
      printk("initialize: pci probe failed!\n");

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_initialize_adapter: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_initialize_adapter");
#endif
    return(status);
}

/*
 * qla1280_enable_tgt
 *      Enable target mode.
 *
 * Input:
 *      ha = adapter block pointer.
 *      b  = SCSI bus number.
 *
 * Returns:
 *      0 = success.
 */
static int
qla1280_enable_tgt(scsi_qla_host_t *ha, uint8_t b)
{
    int status = 0;
    /*  uint16_t    mb[MAILBOX_REGISTER_COUNT]; */

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_enable_tgt: entered\n\r");
#endif

    /* Enable target mode. */
    /*
    mb[0] = MBC_ENABLE_TARGET_MODE;
    mb[1] = BIT_15;
    mb[2] = (uint16_t)(b << 15);
    status = qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);
    */
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_enable_tgt: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_enable_tgt: exiting normally\n\r");
#endif
    return(status);
}

/*
 * ISP Firmware Test
 *      Checks if present version of RISC firmware is older than
 *      driver firmware.
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      0 = firmware does not need to be loaded.
 */
static int
qla1280_isp_firmware(scsi_qla_host_t *ha)
{
    nvram_t     *nv    = (nvram_t *)ha->response_ring; 
    uint16_t    *wptr;
    int		status = 0;    /* dg 2/27 always loads RISC */
    int		cnt;
    uint8_t     chksum;
    uint16_t    mb[MAILBOX_REGISTER_COUNT];

    ENTER("qla1280_isp_firmware");

    /* Verify valid NVRAM checksum. */
    wptr = (uint16_t *)ha->response_ring;
    DEBUG(printk("qla1280_isp_firmware: Reading NVRAM\n"));
    chksum = 0;
    for (cnt = 0; cnt < sizeof(nvram_t)/2; cnt++)
    {
        *wptr = qla1280_get_nvram_word(ha, cnt);
        chksum += (uint8_t)*wptr;
        chksum += (uint8_t)(*wptr >> 8);
        wptr++;
    }
    DEBUG(printk("qla1280_isp_firmware: Completed Reading NVRAM\n"));

#if defined(QL_DEBUG_LEVEL_3)
    sprintf(debug_buff,"qla1280_isp_firmware: NVRAM Magic ID= %c %c %c\n\r",(char *) nv->id[0],nv->id[1],nv->id[2]);
    qla1280_print(debug_buff);
#endif

    /* Bad NVRAM data, load RISC code. */
    if (chksum || nv->id[0] != 'I' || nv->id[1] != 'S' ||
        nv->id[2] != 'P' || nv->id[3] != ' ' || nv->version < 1)
    {
        printk(KERN_INFO "qla1280_isp_firmware: Bad checksum or magic number or version in NVRAM.\n");
        ha->flags.disable_risc_code_load = FALSE;
    }
    else
        ha->flags.disable_risc_code_load = nv->cntr_flags_1.disable_loading_risc_code;

    if (ha->flags.disable_risc_code_load)
    {
#if defined(QL_DEBUG_LEVEL_3)
        qla1280_print("qla1280_isp_firmware: Telling RISC to verify checksum of loaded BIOS code.\n\r");
#endif
        /* Verify checksum of loaded RISC code. */
        mb[0] = MBC_VERIFY_CHECKSUM;
        /* mb[1] = ql12_risc_code_addr01; */
        mb[1] = *QL1280BoardTbl[ha->devnum].fwstart;  

        if (!(status = qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0])))
        {
            /* Start firmware execution. */
#if defined(QL_DEBUG_LEVEL_3)
            qla1280_print("qla1280_isp_firmware: Startng F/W execution.\n\r");
#endif
            mb[0] = MBC_EXECUTE_FIRMWARE;
            /* mb[1] = ql12_risc_code_addr01; */
            mb[1] = *QL1280BoardTbl[ha->devnum].fwstart;  
            qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);
        }
        else
            printk(KERN_INFO "qla1280: RISC checksum failed.\n");
    }
    else
    {
        DEBUG(printk("qla1280: NVRAM configured to load RISC load.\n"));
        status = 1;
     }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print(
                "qla1280_isp_firmware: **** Load RISC code ****\n\r");
#endif
    LEAVE("qla1280_isp_firmware");
    return(status);
}

/*
 * PCI configuration
 *      Setup device PCI configuration registers.
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      0 = success.
 */
static int
qla1280_pci_config(scsi_qla_host_t *ha)
{
#if MEMORY_MAPPED_IO
    uint32_t  page_offset, base;
    uint32_t   mmapbase;
#endif
    uint16_t     buf_wd;
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
    uint32_t     command;
    config_reg_t *creg = 0;
#endif
    int status = 1;


    ENTER("qla1280_pci_config");

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
    /*
     * Set Bus Master Enable, Memory Address Space Enable and
     * reset any error bits, in the command register.
     */
    pci_read_config_word(ha->pdev, PCI_COMMAND, &buf_wd);
    buf_wd &= ~0x7;
#if MEMORY_MAPPED_IO
    DEBUG(printk("qla1280: MEMORY MAPPED IO is enabled.\n"));
    buf_wd |= BIT_2 + BIT_1 + BIT_0;
#else
    buf_wd |= BIT_2 + BIT_0;
#endif
    pci_write_config_word(ha->pdev, PCI_COMMAND, buf_wd);
    /*
     * Reset expansion ROM address decode enable.
     */
    pci_read_config_word(ha->pdev, PCI_ROM_ADDRESS, &buf_wd);
    buf_wd &= ~PCI_ROM_ADDRESS_ENABLE;
    pci_write_config_word(ha->pdev, PCI_ROM_ADDRESS, buf_wd);
#if MEMORY_MAPPED_IO
    /*
     * Get memory mapped I/O address. 
     */
    pci_read_config_word(ha->pdev, PCI_BASE_ADDRESS_1, &mmapbase);
    mmapbase &= PCI_BASE_ADDRESS_MEM_MASK;

    /*
     * Find proper memory chunk for memory map I/O reg.
     */
    base = mmapbase & PAGE_MASK;
    page_offset = mmapbase - base;
    /*
     * Get virtual address for I/O registers.
     */
    ha->mmpbase = ioremap_nocache(base, page_offset + 256);
    if( ha->mmpbase )
    {
        ha->mmpbase += page_offset;
        /* ha->iobase = ha->mmpbase; */
        status = 0;
    }
#else /* MEMORY_MAPPED_IO */
    status = 0;
#endif /* MEMORY_MAPPED_IO */

#else /*LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18) */

    /* Get command register. */
    if (pcibios_read_config_word(ha->pci_bus,ha->pci_device_fn, OFFSET(creg->command), &buf_wd) == PCIBIOS_SUCCESSFUL)
    {
        command = buf_wd;
        /*
        * Set Bus Master Enable, Memory Address Space Enable and
        * reset any error bits.
        */
        buf_wd &= ~0x7;
#if MEMORY_MAPPED_IO
        DEBUG(printk("qla1280: MEMORY MAPPED IO is enabled.\n"));
        buf_wd |= BIT_2 + BIT_1 + BIT_0;
#else
        buf_wd |= BIT_2 + BIT_0;
#endif
        if( pcibios_write_config_word(ha->pci_bus,ha->pci_device_fn, OFFSET(creg->command), buf_wd) )
        {
            printk(KERN_WARNING "qla1280: Could not write config word.\n");
        }
        /* Get expansion ROM address. */
        if (pcibios_read_config_word(ha->pci_bus,ha->pci_device_fn, OFFSET(creg->expansion_rom), &buf_wd) == PCIBIOS_SUCCESSFUL)
        {
            /* Reset expansion ROM address decode enable. */
            buf_wd &= ~BIT_0;
            if (pcibios_write_config_word(ha->pci_bus,ha->pci_device_fn, OFFSET(creg->expansion_rom), buf_wd) == PCIBIOS_SUCCESSFUL)
            {
#if MEMORY_MAPPED_IO
                /* Get memory mapped I/O address. */
                pcibios_read_config_dword(ha->pci_bus, ha->pci_device_fn,OFFSET(cfgp->mem_base_addr), &mmapbase);
                mmapbase &= PCI_BASE_ADDRESS_MEM_MASK;

                /* Find proper memory chunk for memory map I/O reg. */
                base = mmapbase & PAGE_MASK;
                page_offset = mmapbase - base;

                /* Get virtual address for I/O registers. */
                ha->mmpbase = ioremap_nocache(base, page_offset + 256);
                if( ha->mmpbase )
                {
                    ha->mmpbase += page_offset;
                    /* ha->iobase = ha->mmpbase; */
                    status = 0;
                }
#else /* MEMORY_MAPPED_IO */
                status = 0;
#endif /* MEMORY_MAPPED_IO */
            }
        }
    }
#endif /* LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18) */

    LEAVE("qla1280_pci_config");
    return(status);
}

/*
 * Chip diagnostics
 *      Test chip for proper operation.
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      0 = success.
 */
static int
qla1280_chip_diag(scsi_qla_host_t *ha)
{
    uint16_t     mb[MAILBOX_REGISTER_COUNT];
    device_reg_t *reg   = ha->iobase;
    int		 status = 0;
    int		 cnt;
    uint16_t     data;

#ifdef QL_DEBUG_LEVEL_3
    sprintf(debug_buff, "qla1280_chip_diag: testing device at 0x%p \n\r",&reg->id_l);
    qla1280_print(debug_buff);
#endif

    /* Soft reset chip and wait for it to finish. */
    WRT_REG_WORD(&reg->ictrl, ISP_RESET); 
    data = qla1280_debounce_register(&reg->ictrl);
    for (cnt = 6000000; cnt && data & ISP_RESET; cnt--)
    {
        SYS_DELAY(5);
        data = RD_REG_WORD(&reg->ictrl);
    }
    if (cnt)
    {
        /* Reset register not cleared by chip reset. */
#if defined(QL_DEBUG_LEVEL_3)
        qla1280_print("qla1280_chip_diag: reset register cleared by chip reset\n\r");
#endif
        WRT_REG_WORD(&reg->cfg_1, 0);

        /* Reset RISC and disable BIOS which
        allows RISC to execute out of RAM. */
        WRT_REG_WORD(&reg->host_cmd, HC_RESET_RISC);
        WRT_REG_WORD(&reg->host_cmd, HC_RELEASE_RISC);
        WRT_REG_WORD(&reg->host_cmd, HC_DISABLE_BIOS);
        data = qla1280_debounce_register(&reg->mailbox0);
        for (cnt = 6000000; cnt && data == MBS_BUSY; cnt--)
        {
            SYS_DELAY(5);
            data = RD_REG_WORD(&reg->mailbox0);
        }

        if (cnt)
        {
            /* Check product ID of chip */
#if defined(QL_DEBUG_LEVEL_3)
            qla1280_print("qla1280_chip_diag: Checking product ID of chip\n\r");
#endif
            if (RD_REG_WORD(&reg->mailbox1) != PROD_ID_1 ||
                (RD_REG_WORD(&reg->mailbox2) != PROD_ID_2 &&
                RD_REG_WORD(&reg->mailbox2) != PROD_ID_2a) ||
                RD_REG_WORD(&reg->mailbox3) != PROD_ID_3 ||
                RD_REG_WORD(&reg->mailbox4) != PROD_ID_4)
            {
                printk(KERN_INFO "qla1280: Wrong product ID = 0x%x,0x%x,0x%x,0x%x\n",
                        RD_REG_WORD(&reg->mailbox1),
                        RD_REG_WORD(&reg->mailbox2),
                        RD_REG_WORD(&reg->mailbox3),
                        RD_REG_WORD(&reg->mailbox4) );
                status = 1;
            }
            else
            {
                DEBUG(printk("qla1280_chip_diag: Checking mailboxes of chip\n"));
                /* Wrap Incoming Mailboxes Test. */
                mb[0] = MBC_MAILBOX_REGISTER_TEST;
                mb[1] = 0xAAAA;
                mb[2] = 0x5555;
                mb[3] = 0xAA55;
                mb[4] = 0x55AA;
                mb[5] = 0xA5A5;
                mb[6] = 0x5A5A;
                mb[7] = 0x2525;
                if (!(status = qla1280_mailbox_command(ha,
                    (BIT_7|BIT_6|BIT_5|BIT_4|BIT_3|BIT_2|BIT_1|BIT_0),
                    &mb[0])))
                {
                    if (mb[1] != 0xAAAA || mb[2] != 0x5555 ||
                        mb[3] != 0xAA55 || mb[4] != 0x55AA)
                        status = 1;
                    if (mb[5] != 0xA5A5 || mb[6] != 0x5A5A ||
                        mb[7] != 0x2525)
                        status = 1;
                    if( status == 1 )
                        printk(KERN_INFO "qla1280: Failed mailbox check\n");
                }
            }
        }
        else
            status = 1;
    }
    else
        status = 1;

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_chip_diag: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_chip_diag: exiting normally\n\r");
#endif
    return(status);
}

/*
 * Setup chip
 *      Load and start RISC firmware.
 *
 * Input:
 *      ha = adapter block pointer.
 *
 * Returns:
 *      0 = success.
 */
#define DUMP_IT_BACK 0   /* for debug of RISC loading */
static int
qla1280_setup_chip(scsi_qla_host_t *ha)
{
    int		 status = 0;
    uint16_t     risc_address;
    uint16_t     *risc_code_address;
    long         risc_code_size;
    uint16_t     mb[MAILBOX_REGISTER_COUNT];
    uint16_t	 cnt;
    int          num;
#if DUMP_IT_BACK
    int     i;
    uint8_t *sp;
    uint8_t    *tbuf;
#ifdef QLA_64BIT_PTR
    dma_addr_t     p_tbuf;
#else
    uint32_t   p_tbuf;
#endif
#endif

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_setup_chip");
#endif

    /* 3.13 */
#if DUMP_IT_BACK
    /* get consistent memory allocated for setup_chip */
    tbuf   = pci_alloc_consistent(ha->pdev, 8000, &p_tbuf);
#endif

    /* Load RISC code. */
    /* 
    risc_address      = ql12_risc_code_addr01;
    risc_code_address = &ql12_risc_code01[0];
    risc_code_size    = ql12_risc_code_length01;
    */
    risc_address = *QL1280BoardTbl[ha->devnum].fwstart;  
    risc_code_address = QL1280BoardTbl[ha->devnum].fwcode;  
    risc_code_size    = (long)(*QL1280BoardTbl[ha->devnum].fwlen & 0xffff);  

    DEBUG(printk("qla1280_setup_chip: DMA RISC code (%d) words\n",
         (int)risc_code_size));
    DEBUG(sprintf(debug_buff,
       "qla1280_setup_chip: DMA RISC code (%li) words\n\r",risc_code_size);)
    DEBUG(qla1280_print(debug_buff));
    num =0;
    while (risc_code_size > 0 && !status)
    {
        cnt = 2000 >> 1;

        if ( cnt > risc_code_size ) 
            cnt = risc_code_size;

        DEBUG(sprintf(debug_buff,
         "qla1280_setup_chip:  loading risc @ =(0x%p),%d,%d(0x%x).\n\r",
          (void *)risc_code_address, cnt, num, risc_address);)
        DEBUG(qla1280_print(debug_buff));
        memcpy(ha->request_ring, risc_code_address, (cnt <<1));

        flush_cache_all();

        mb[0] = MBC_LOAD_RAM; 
        mb[1] = risc_address;
        mb[4] = cnt;
        mb[3] = ha->request_dma & 0xffff;
        mb[2] = (ha->request_dma >> 16) & 0xffff;
        mb[7] = MS_64BITS(ha->request_dma) & 0xffff;
        mb[6] = MS_64BITS(ha->request_dma) >> 16;
        DEBUG(printk("qla1280_setup_chip: op=%d  0x%p = 0x%4x,0x%4x,0x%4x,0x%4x\n",mb[0],(void *)ha->request_dma,mb[6],mb[7],mb[2],mb[3]));
        if( (status = qla1280_mailbox_command(ha, BIT_4|BIT_3|BIT_2|BIT_1|BIT_0,
            &mb[0]))  )
        {
            printk("Failed to load partial segment of f/w\n");
            break;
        }
#if DUMP_IT_BACK
        mb[0] = MBC_READ_RAM_WORD;
        mb[1] = risc_address;
        mb[4] = cnt;
        mb[3] = (uint16_t)  p_tbuf & 0xffff;
        mb[2] = (uint16_t) (p_tbuf >> 16) & 0xffff;
        mb[7] = (uint16_t) (p_tbuf >> 32) & 0xffff;
        mb[6] = (uint16_t) (p_tbuf >> 48) & 0xffff;

        if( (status = qla1280_mailbox_command(ha, 
                      BIT_4|BIT_3|BIT_2|BIT_1|BIT_0,&mb[0]))  )
        {
            printk("Failed to dump partial segment of f/w\n");
            DEBUG(sprintf(debug_buff,
               "setup_chip: Failed to dump partial segment of f/w\n\r");)
            DEBUG(qla1280_print(debug_buff));
            break;
        }
        sp =  (uint8_t *)ha->request_ring;
        for (i = 0; i < (cnt<< 1) ; i++)
        {
            if( tbuf[i] != sp[i] )
            {
               printk("qla1280_setup_chip: FW compare error @ byte(0x%x) loop#=%x\n",i,num);
               printk("setup_chip: FWbyte=%x  FWfromChip=%x\n",sp[i],tbuf[i]); 
               DEBUG(sprintf(debug_buff,
                "qla1280_setup_chip: FW compare error @ byte(0x%x) loop#=%x\n\r",i);)
               DEBUG(qla1280_print(debug_buff);)
               /*break;*/
            }
        }
#endif
        risc_address += cnt;
        risc_code_size = risc_code_size - cnt;
        risc_code_address = risc_code_address + cnt;
        num++;
    }

    /* Verify checksum of loaded RISC code. */
    if (!status)
    {
        DEBUG(printk("qla1280_setup_chip: Verifying checksum of loaded RISC code.\n");)
        mb[0] = MBC_VERIFY_CHECKSUM;
        /* mb[1] = ql12_risc_code_addr01; */
        mb[1] = *QL1280BoardTbl[ha->devnum].fwstart;  
        
        if (!(status = qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0])))
        {
            /* Start firmware execution. */
            DEBUG(qla1280_print("qla1280_setup_chip: start firmware running.\n\r");)
            mb[0] = MBC_EXECUTE_FIRMWARE;
            mb[1] = *QL1280BoardTbl[ha->devnum].fwstart;  
            qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);
        }
        else
            printk("qla1280_setup_chip: Failed checksum.\n");
    }

    /* 3.13 */
#if DUMP_IT_BACK
    /* free consistent memory allocated for setup_chip */
    pci_free_consistent(ha->pdev, 8000, tbuf, p_tbuf);
#endif

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_setup_chip: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_setup_chip");
#endif
    return(status);
}

/*
 * Initialize rings
 *
 * Input:
 *      ha                = adapter block pointer.
 *      ha->request_ring  = request ring virtual address
 *      ha->response_ring = response ring virtual address
 *      ha->request_dma   = request ring physical address
 *      ha->response_dma  = response ring physical address
 *
 * Returns:
 *      0 = success.
 */
static int
qla1280_init_rings(scsi_qla_host_t *ha)
{
    uint16_t    mb[MAILBOX_REGISTER_COUNT];
    int		status = 0;
    int		cnt;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_init_rings");
#endif
    /* Clear outstanding commands array. */
    for (cnt = 0; cnt < MAX_OUTSTANDING_COMMANDS; cnt++)
        ha->outstanding_cmds[cnt] = 0;

    /* Initialize request queue. */
    ha->request_ring_ptr = ha->request_ring;
    ha->req_ring_index   = 0;
    ha->req_q_cnt        = REQUEST_ENTRY_CNT;
    /* mb[0] = MBC_INIT_REQUEST_QUEUE; */
    mb[0] = MBC_INIT_REQUEST_QUEUE_A64;
    mb[1] = REQUEST_ENTRY_CNT;
    mb[3] = ha->request_dma & 0xffff;
    mb[2] = (ha->request_dma >> 16) & 0xffff;
    mb[4] = 0;
    mb[7] = MS_64BITS(ha->request_dma) & 0xffff;
    mb[6] = MS_64BITS(ha->request_dma) >> 16;
    if (!(status = qla1280_mailbox_command(ha,
        BIT_7|BIT_6|BIT_4|BIT_3|BIT_2|BIT_1|BIT_0,
        &mb[0])))
    {
        /* Initialize response queue. */
        ha->response_ring_ptr = ha->response_ring;
        ha->rsp_ring_index    = 0;
        /* mb[0] = MBC_INIT_RESPONSE_QUEUE; */
        mb[0] = MBC_INIT_RESPONSE_QUEUE_A64;
        mb[1] = RESPONSE_ENTRY_CNT;
        mb[3] = ha->response_dma & 0xffff;
        mb[2] = (ha->response_dma >> 16) & 0xffff;
        mb[5] = 0;
        mb[7] = MS_64BITS(ha->response_dma) & 0xffff;
        mb[6] = MS_64BITS(ha->response_dma) >> 16;
        status = qla1280_mailbox_command(ha,
                BIT_7|BIT_6|BIT_5|BIT_3|BIT_2|BIT_1|BIT_0,
                &mb[0]);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_init_rings: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_init_rings");
#endif
    return(status);
}

/*
 * NVRAM configuration.
 *
 * Input:
 *      ha                = adapter block pointer.
 *      ha->request_ring  = request ring virtual address
 *
 * Output:
 *      host adapters parameters in host adapter block
 *
 * Returns:
 *      0 = success.
 */
static int
qla1280_nvram_config(scsi_qla_host_t *ha)
{
    device_reg_t *reg   = ha->iobase;
    nvram_t      *nv    = (nvram_t *)ha->response_ring; 
    int		 status = 0;
    int		 cnt;
    uint32_t     b, t, l;
    uint16_t     *wptr;
    uint16_t     mb[MAILBOX_REGISTER_COUNT];
    uint8_t      chksum;
    uint32_t     nvsize;

#if defined(QL_DEBUG_ROUTINES) && !defined(QL_DEBUG_LEVEL_4)
    int		 saved_print_status = ql_debug_print;
#endif
    ENTER("qla1280_nvram_config");
#if defined(QL_DEBUG_ROUTINES) && !defined(QL_DEBUG_LEVEL_4)
    ql_debug_print = FALSE;
#endif

    /* Verify valid NVRAM checksum. */
#if  USE_NVRAM_DEFAULTS
    chksum = 1;
#else
    wptr = (uint16_t *)ha->response_ring;
    chksum = 0;
    if( ha->device_id == QLA12160_DEVICE_ID ||  
        ha->device_id == QLA10160_DEVICE_ID )  
    nvsize = sizeof(nvram160_t)/2;
    else
    nvsize = sizeof(nvram_t)/2;
    for( cnt = 0; cnt < nvsize; cnt++ ) 
    {
        *wptr = qla1280_get_nvram_word(ha, cnt); 
        chksum += (uint8_t)*wptr;
        chksum += (uint8_t)(*wptr >> 8);
        wptr++;
    }
#endif


    /* Bad NVRAM data, set defaults parameters. */
    if (chksum || nv->id[0] != 'I' || nv->id[1] != 'S' ||
        nv->id[2] != 'P' || nv->id[3] != ' ' || nv->version < 1)
    {
#if  USE_NVRAM_DEFAULTS
        DEBUG(printk("Using defaults for NVRAM\n"));
#else
        DEBUG(printk("Using defaults for NVRAM: \n"));
        DEBUG(printk("checksum=0x%x, Id=%c, version=0x%x\n",chksum,nv->id[0],nv->version));
#if defined(QL_DEBUG_LEVEL_3)
        /* ql_debug_print = 1;
        qla1280_dump_buffer((char *)ha->response_ring, REQUEST_ENTRY_SIZE);
        ql_debug_print = 0; */
#endif
                wptr = (uint16_t *)ha->response_ring;
        for (cnt = 0; cnt < sizeof(nvram_t)/2; cnt++)
            *wptr++ = 0;
#endif


        /* nv->cntr_flags_1.disable_loading_risc_code = 1; */
        nv->firmware_feature.w = BIT_0;
        nv->termination.f.scsi_bus_0_control = 3;
        nv->termination.f.scsi_bus_1_control = 3;
        nv->termination.f.auto_term_support = 1;

        for (b = 0; b < MAX_BUSES; b++)
        {
            nv->bus[b].config_1.initiator_id = 7;
            nv->bus[b].bus_reset_delay = 5;
            nv->bus[b].config_2.async_data_setup_time = 9;
            nv->bus[b].config_2.req_ack_active_negation = 1;
            nv->bus[b].config_2.data_line_active_negation = 1;
            nv->bus[b].selection_timeout = 250;
            nv->bus[b].max_queue_depth = 256;

            for (t = 0; t < MAX_TARGETS; t++)
            {
                nv->bus[b].target[t].parameter.f.auto_request_sense = 1;
                nv->bus[b].target[t].parameter.f.disconnect_allowed = 1;
                nv->bus[b].target[t].parameter.f.tag_queuing = 1;
                nv->bus[b].target[t].flags.device_enable = 1;
            }
        }

#if  USE_NVRAM_DEFAULTS
        status = 0;
#else
        status = 1;
#endif
    }
    else
    {
        /* Always force AUTO sense for LINUX SCSI */
        for (b = 0; b < MAX_BUSES; b++)
            for (t = 0; t < MAX_TARGETS; t++)
            {
                nv->bus[b].target[t].parameter.f.auto_request_sense = 1;
            }
    }
#if  DEBUG_PRINT_NVRAM
    ql_debug_print = 1;
    sprintf(debug_buff,"qla1280 : initiator scsi id bus[0]=%d\n\r",
            nv->bus[0].config_1.initiator_id);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : initiator scsi id bus[1]=%d\n\r",
            nv->bus[1].config_1.initiator_id);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : bus reset delay[0]=%d\n\r",
            nv->bus[0].bus_reset_delay);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : bus reset delay[1]=%d\n\r",
            nv->bus[1].bus_reset_delay);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : retry count[0]=%d\n\r",
            nv->bus[0].retry_count);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : retry delay[0]=%d\n\r",
            nv->bus[0].retry_delay);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : retry count[1]=%d\n\r",
            nv->bus[1].retry_count);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : retry delay[1]=%d\n\r",
            nv->bus[1].retry_delay);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : async data setup time[0]=%d\n\r",
            nv->bus[0].config_2.async_data_setup_time);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : async data setup time[1]=%d\n\r",
            nv->bus[1].config_2.async_data_setup_time);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : req/ack active negation[0]=%d\n\r",
            nv->bus[0].config_2.req_ack_active_negation);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : req/ack active negation[1]=%d\n\r",
            nv->bus[1].config_2.req_ack_active_negation);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : data line active negation[0]=%d\n\r",
            nv->bus[0].config_2.data_line_active_negation);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : data line active negation[1]=%d\n\r",
            nv->bus[1].config_2.data_line_active_negation);
    qla1280_print(debug_buff);


    sprintf(debug_buff,"qla1280 : disable loading risc code=%d\n\r",
            nv->cntr_flags_1.disable_loading_risc_code);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : enable 64bit addressing=%d\n\r",
            nv->cntr_flags_1.enable_64bit_addressing);
    qla1280_print(debug_buff);

    sprintf(debug_buff,"qla1280 : selection timeout limit[0]=%d\n\r",
            nv->bus[0].selection_timeout);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : selection timeout limit[1]=%d\n\r",
            nv->bus[1].selection_timeout);

    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : max queue depth[0]=%d\n\r",
            nv->bus[0].max_queue_depth);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"qla1280 : max queue depth[1]=%d\n\r",
            nv->bus[1].max_queue_depth);
    qla1280_print(debug_buff);
#endif

    DEBUG(ql_debug_print = 0;)

    /* Disable RISC load of firmware. */
    ha->flags.disable_risc_code_load =
            nv->cntr_flags_1.disable_loading_risc_code;

#ifdef QLA_64BIT_PTR
    /* Enable 64bit addressing for OS/System combination supporting it   */
    /* actual NVRAM bit is: nv->cntr_flags_1.enable_64bit_addressing     */ 
    /* but we will ignore it and use BITS_PER_LONG macro to setup for    */
    /* 64 or 32 bit access of host memory in all x86/ia-64/Alpha systems */
    ha->flags.enable_64bit_addressing = 1;
#else
    ha->flags.enable_64bit_addressing = 0;
#endif

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
    if (ha->flags.enable_64bit_addressing) {
      printk("[[[ qla1x160: 64 Bit PCI Addressing Enabled ]]]\n");

      pci_set_dma_mask(ha->pdev, (dma_addr_t)~0ULL);
    }
#endif

    /* Set ISP hardware DMA burst */
    mb[0] = nv->isp_config.c;
    WRT_REG_WORD(&reg->cfg_1, mb[0]);

    /* Set SCSI termination. */
    WRT_REG_WORD(&reg->gpio_enable, (BIT_3 + BIT_2 + BIT_1 + BIT_0));
    mb[0] = nv->termination.c & (BIT_3 + BIT_2 + BIT_1 + BIT_0);
    WRT_REG_WORD(&reg->gpio_data, mb[0]);

    /* ISP parameter word. */
    mb[0] = MBC_SET_SYSTEM_PARAMETER;
    mb[1] = nv->isp_parameter;
    status |= qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);

    /* Firmware feature word. */
    mb[0] = MBC_SET_FIRMWARE_FEATURES;
    mb[1] = nv->firmware_feature.w & (BIT_1|BIT_0);
    status |= qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);

    /* Retry count and delay. */
    mb[0] = MBC_SET_RETRY_COUNT;
    mb[1] = nv->bus[0].retry_count;
    mb[2] = nv->bus[0].retry_delay;
    mb[6] = nv->bus[1].retry_count;
    mb[7] = nv->bus[1].retry_delay;
    status |= qla1280_mailbox_command(ha, BIT_7|BIT_6|BIT_2|BIT_1|BIT_0, &mb[0]);

    /* ASYNC data setup time. */
    mb[0] = MBC_SET_ASYNC_DATA_SETUP;
    mb[1] = nv->bus[0].config_2.async_data_setup_time;
    mb[2] = nv->bus[1].config_2.async_data_setup_time;
    status |= qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);

    /* Active negation states. */
    mb[0] = MBC_SET_ACTIVE_NEGATION;
    mb[1] = 0;
    if (nv->bus[0].config_2.req_ack_active_negation)
        mb[1] |= BIT_5;
    if (nv->bus[0].config_2.data_line_active_negation)
        mb[1] |= BIT_4;
    mb[2] = 0;
    if (nv->bus[1].config_2.req_ack_active_negation)
        mb[2] |= BIT_5;
    if (nv->bus[1].config_2.data_line_active_negation)
        mb[2] |= BIT_4;
    status |= qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);

    /* Selection timeout. */
    mb[0] = MBC_SET_SELECTION_TIMEOUT;
    mb[1] = nv->bus[0].selection_timeout;
    mb[2] = nv->bus[1].selection_timeout;
    status |= qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);

    for (b = 0; b < ha->ports; b++)
    {
        /* SCSI Reset Disable. */
        ha->bus_settings[b].disable_scsi_reset = nv->bus[b].config_1.scsi_reset_disable;

        /* Initiator ID. */
        ha->bus_settings[b].id = nv->bus[b].config_1.initiator_id;
        mb[0] = MBC_SET_INITIATOR_ID;
        mb[1] = b ? ha->bus_settings[b].id | BIT_7 : ha->bus_settings[b].id;
        status |= qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);

        /* Reset Delay. */
        ha->bus_settings[b].bus_reset_delay = nv->bus[b].bus_reset_delay;

        /* Command queue depth per device. */
        ha->bus_settings[b].hiwat = nv->bus[b].max_queue_depth - 1;

        /* Set target parameters. */
        for (t = 0; t < MAX_TARGETS; t++)
        {
            if( ha->device_id == QLA12160_DEVICE_ID ||  
                ha->device_id == QLA10160_DEVICE_ID )  
            {
                status = qla12160_set_target_parameters(ha,b,t,0,(nvram160_t *)nv);
            }
			else
            {
                /* Set Target Parameters. */
                mb[0] = MBC_SET_TARGET_PARAMETERS;
                mb[1] = (uint16_t)(b ? t | BIT_7 :t);
                mb[1] <<= 8;
                mb[2] = nv->bus[b].target[t].parameter.c << 8;
                mb[2] |= TP_AUTO_REQUEST_SENSE;
                mb[2] &= ~TP_STOP_QUEUE;
                mb[3] = nv->bus[b].target[t].flags.sync_offset << 8;
                mb[3] |= nv->bus[b].target[t].sync_period;
                status |= qla1280_mailbox_command(ha, BIT_3|BIT_2|BIT_1|BIT_0,
                    &mb[0]);
            }

            /* Save Tag queuing enable flag. */
            mb[0] = BIT_0 << t;
            if (nv->bus[b].target[t].parameter.f.tag_queuing)
                ha->bus_settings[b].qtag_enables |= mb[0];

            /* Save Device enable flag. */
            if (nv->bus[b].target[t].flags.device_enable)
                ha->bus_settings[b].device_enables |= mb[0];

            /* Save LUN disable flag. */
            if (nv->bus[b].target[t].flags.lun_disable)
                ha->bus_settings[b].lun_disables |= mb[0];

            /* Set Device Queue Parameters. */
            for (l = 0; l < MAX_LUNS; l++)
            {
                mb[0] = MBC_SET_DEVICE_QUEUE;
                mb[1] = (uint16_t)(b ? t | BIT_7 :t);
                mb[1] = mb[1] << 8 | l;
                mb[2] = nv->bus[b].max_queue_depth;
                mb[3] = nv->bus[b].target[t].execution_throttle;
                status |= qla1280_mailbox_command(ha, BIT_3|BIT_2|BIT_1|BIT_0,
                        &mb[0]);
            }
        }
    }
    DEBUG(ql_debug_print = 0;)

#if defined(QL_DEBUG_ROUTINES) && !defined(QL_DEBUG_LEVEL_4)
    ql_debug_print = saved_print_status;
#endif

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    DEBUG(if (status))
        DEBUG(qla1280_print("qla1280_nvram_config: **** FAILED ****\n\r");)
#endif
    LEAVE("qla1280_nvram_config");
    return(status);
}

/*
 * Get NVRAM data word
 *      Calculates word position in NVRAM and calls request routine to
 *      get the word from NVRAM.
 *
 * Input:
 *      ha      = adapter block pointer.
 *      address = NVRAM word address.
 *
 * Returns:
 *      data word.
 */
static uint16_t
qla1280_get_nvram_word(scsi_qla_host_t *ha, uint32_t address)
{
    uint32_t nv_cmd;
    uint16_t data;

#ifdef QL_DEBUG_ROUTINES
    uint8_t  saved_print_status = ql_debug_print;
#endif
#ifdef QL_DEBUG_LEVEL_4
    ENTER("qla1280_get_nvram_word");
#endif

    nv_cmd = address << 16;
    nv_cmd |= NV_READ_OP;

#ifdef QL_DEBUG_ROUTINES
    ql_debug_print = FALSE;
#endif
    data = qla1280_nvram_request(ha, nv_cmd);
#ifdef QL_DEBUG_ROUTINES
    ql_debug_print = saved_print_status;
#endif

#ifdef QL_DEBUG_LEVEL_4
    qla1280_print("qla1280_get_nvram_word: exiting normally NVRAM data = ");
    qla1280_output_number((uint32_t)data, 16);
    qla1280_print("\n\r");
#endif
    return(data);
}

/*
 * NVRAM request
 *      Sends read command to NVRAM and gets data from NVRAM.
 *
 * Input:
 *      ha     = adapter block pointer.
 *      nv_cmd = Bit 26     = start bit
 *               Bit 25, 24 = opcode
 *               Bit 23-16  = address
 *               Bit 15-0   = write data
 *
 * Returns:
 *      data word.
 */
static uint16_t
qla1280_nvram_request(scsi_qla_host_t *ha, uint32_t nv_cmd)
{
    device_reg_t *reg = ha->iobase;
    int		 cnt;
    uint16_t     data = 0;
    uint16_t     reg_data;

    /* Send command to NVRAM. */

    nv_cmd <<= 5;
    for (cnt = 0; cnt < 11; cnt++)
    {
        if (nv_cmd & BIT_31)
            qla1280_nv_write(ha, NV_DATA_OUT);
        else
            qla1280_nv_write(ha, 0);
        nv_cmd <<= 1;
    }

    /* Read data from NVRAM. */

    for (cnt = 0; cnt < 16; cnt++)
    {
        WRT_REG_WORD(&reg->nvram, NV_SELECT+NV_CLOCK);
        NVRAM_DELAY();
        data <<= 1;
        reg_data = RD_REG_WORD(&reg->nvram);
        if (reg_data & NV_DATA_IN)
            data |= BIT_0;
        WRT_REG_WORD(&reg->nvram, NV_SELECT);
        NVRAM_DELAY();
    }

    /* Deselect chip. */

    WRT_REG_WORD(&reg->nvram, NV_DESELECT);
    NVRAM_DELAY();

    return(data);
}

static void
qla1280_nv_write(scsi_qla_host_t *ha, uint16_t data)
{
    device_reg_t *reg = ha->iobase;

    WRT_REG_WORD(&reg->nvram, data | NV_SELECT);
    NVRAM_DELAY();
    WRT_REG_WORD(&reg->nvram, data | NV_SELECT | NV_CLOCK);
    NVRAM_DELAY();
    WRT_REG_WORD(&reg->nvram, data | NV_SELECT);
    NVRAM_DELAY();
}


/*
 * Mailbox Command
 *      Issue mailbox command and waits for completion.
 *
 * Input:
 *      ha = adapter block pointer.
 *      mr = mailbox registers to load.
 *      mb = data pointer for mailbox registers.
 *
 * Output:
 *      mb[MAILBOX_REGISTER_COUNT] = returned mailbox data.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_mailbox_command(scsi_qla_host_t *ha, uint8_t mr, uint16_t *mb)
{
    device_reg_t *reg   = ha->iobase;
    srb_t        *done_q_first = 0;
    srb_t        *done_q_last = 0;
    int		 status = 0;
    int		 cnt;
    uint16_t     *optr, *iptr;
    uint16_t     data;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_mailbox_command");
#endif

    /* Acquire interrupt specific lock */
    QLA1280_INTR_LOCK(ha);
    ha->flags.mbox_busy = TRUE;

    /* Load mailbox registers. */
    optr = (uint16_t *)&reg->mailbox0;
    iptr = mb;
    for (cnt = 0; cnt < MAILBOX_REGISTER_COUNT; cnt++)
    {
        if (mr & BIT_0)
        {
            WRT_REG_WORD(optr, (*iptr));
        }

        mr >>= 1;
        optr++;
        iptr++;
    }
    /* Issue set host interrupt command. */
    ha->flags.mbox_int = FALSE;
    WRT_REG_WORD(&reg->host_cmd, HC_SET_HOST_INT);
    data = qla1280_debounce_register(&reg->istatus);

    /* Wait for 30 seconds for command to finish. */
    for (cnt = 30000000; cnt > 0 && !ha->flags.mbox_int; cnt--)
    {
        /* Check for pending interrupts. */
        if (data & RISC_INT)
        {
            qla1280_isr(ha, (srb_t **)&done_q_first, (srb_t **)&done_q_last);
        }
        SYS_DELAY(1);
        data = RD_REG_WORD(&reg->istatus);
    }

    /* Check for mailbox command timeout. */
    if ( !cnt )
    {
#ifdef QL_DEBUG_LEVEL_2
        qla1280_print(
                "qla1280_mailbox_command: **** Command Timeout, mailbox0 = ");
        qla1280_output_number((uint32_t)mb[0], 16);
        qla1280_print(" ****\n\r");
#endif
        ha->flags.isp_abort_needed = TRUE; 
        status = 1;
    }
    else if (ha->mailbox_out[0] != MBS_CMD_CMP)
        status = 1;

    /* Load return mailbox registers. */
    optr = mb;
    iptr = (uint16_t *)&ha->mailbox_out[0];
    mr = MAILBOX_REGISTER_COUNT;
    while (mr--)
        *optr++ = *iptr++;

    /* Go check for any response interrupts pending. */
    ha->flags.mbox_busy = FALSE;
    qla1280_isr(ha, (srb_t **)&done_q_first, (srb_t **)&done_q_last);

    /* Release interrupt specific lock */
    QLA1280_INTR_UNLOCK(ha);

    if (ha->flags.isp_abort_needed)
        qla1280_abort_isp(ha);

    if (ha->flags.reset_marker)
        qla1280_rst_aen(ha);

    if (done_q_first)
        qla1280_done(ha, (srb_t **)&done_q_first, (srb_t **)&done_q_last);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
    {
        qla1280_print("qla1280_mailbox_command: **** FAILED, mailbox0 = ");
        qla1280_output_number((uint32_t)mb[0], 16);
        qla1280_print(" ****\n\r");
    }
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_mailbox_command");
#endif
    return(status);
}

/*
 * qla1280_poll
 *      Polls ISP for interrupts.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void
qla1280_poll(scsi_qla_host_t *ha)
{
    device_reg_t    *reg   = ha->iobase;
    uint16_t        data;
    srb_t           *done_q_first = 0;
    srb_t           *done_q_last = 0;

#ifdef QL_DEBUG_LEVEL_3
    /* ENTER("qla1280_poll"); */
#endif

    /* Acquire interrupt specific lock */
    QLA1280_INTR_LOCK(ha);

    /* Check for pending interrupts. */
    data = RD_REG_WORD(&reg->istatus);
    if (data & RISC_INT)
        qla1280_isr(ha, (srb_t **)&done_q_first, (srb_t **)&done_q_last);

    /* Release interrupt specific lock */
    QLA1280_INTR_UNLOCK(ha);

    if (!ha->flags.mbox_busy)
    {
        if (ha->flags.isp_abort_needed)
            qla1280_abort_isp(ha);
        if (ha->flags.reset_marker)
            qla1280_rst_aen(ha);
    }

    if (done_q_first)
        qla1280_done(ha, (srb_t **)&done_q_first, (srb_t **)&done_q_last);

#ifdef QL_DEBUG_LEVEL_3
    /* LEAVE("qla1280_poll"); */
#endif
}

/*
 * qla1280_bus_reset
 *      Issue SCSI bus reset.
 *
 * Input:
 *      ha = adapter block pointer.
 *      b  = SCSI bus number.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_bus_reset(scsi_qla_host_t *ha, u8 b)
{
    uint16_t    mb[MAILBOX_REGISTER_COUNT];
    int status;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_bus_reset: entered\n\r");
#endif
    if( qla1280_verbose )
    {
      printk("scsi(%li): Resetting SCSI BUS (%i)\n", ha->host_no, b);
    }

    mb[0] = MBC_BUS_RESET;
    mb[1] = ha->bus_settings[b].bus_reset_delay;
    mb[2] = (uint16_t)b;
    status = qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);

    if (status)
    {
        if (ha->bus_settings[b].failed_reset_count > 2)                  /* dg - 03/13/99 */
            ha->bus_settings[b].scsi_bus_dead = TRUE;
        ha->bus_settings[b].failed_reset_count++;
    }
	else
    {
       QLA1280_DELAY(4);
       ha->bus_settings[b].scsi_bus_dead = FALSE;                         /* dg - 03/13/99 */
       ha->bus_settings[b].failed_reset_count = 0;
       /* Issue marker command. */
       qla1280_marker(ha, b, 0, 0, MK_SYNC_ALL);
    }
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_bus_reset: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_bus_reset: exiting normally\n\r");
#endif
    return(status);
}

/*
 * qla1280_device_reset
 *      Issue bus device reset message to the target.
 *
 * Input:
 *      ha = adapter block pointer.
 *      b  = SCSI BUS number.
 *      t  = SCSI ID.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_device_reset(scsi_qla_host_t *ha, uint8_t b, uint32_t t)
{
    uint16_t    mb[MAILBOX_REGISTER_COUNT];
    int status;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_device_reset");
#endif

    mb[0] = MBC_ABORT_TARGET;
    mb[1] = (b ? (t | BIT_7) : t) << 8;
    mb[2] = 1;
    status = qla1280_mailbox_command(ha, BIT_2|BIT_1|BIT_0, &mb[0]);

    /* Issue marker command. */
    qla1280_marker(ha, b, t, 0, MK_SYNC_ID);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_device_reset: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_device_reset");
#endif
    return(status);
}

/*
 * qla1280_abort_device
 *      Issue an abort message to the device
 *
 * Input:
 *      ha = adapter block pointer.
 *      b  = SCSI BUS.
 *      t  = SCSI ID.
 *      l  = SCSI LUN.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_abort_device(scsi_qla_host_t *ha, u8 b, u32 t, u32 l)
{
    uint16_t    mb[MAILBOX_REGISTER_COUNT];
    int status;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_abort_device");
#endif

    mb[0] = MBC_ABORT_DEVICE;
    mb[1] = (b ? t | BIT_7 : t) << 8 | l;
    status = qla1280_mailbox_command(ha, BIT_1|BIT_0, &mb[0]);

    /* Issue marker command. */
    qla1280_marker(ha, b, t, l, MK_SYNC_ID_LUN);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_abort_device: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_abort_device");
#endif
    return(status);
}

/*
 * qla1280_abort_command
 *      Abort command aborts a specified IOCB.
 *
 * Input:
 *      ha = adapter block pointer.
 *      sp = SB structure pointer.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_abort_command(scsi_qla_host_t *ha, srb_t *sp)
{
    uint16_t	mb[MAILBOX_REGISTER_COUNT];
    uint32_t	b, t, l;
    uint32_t	handle;
    int		status;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_abort_command");
#endif

    /* Locate handle number. */
    for (handle = 0; handle < MAX_OUTSTANDING_COMMANDS; handle++)
        if (ha->outstanding_cmds[handle] == sp)
            break;

                b  = SCSI_BUS_32(sp->cmd);
        t  = SCSI_TCN_32(sp->cmd);
        l  = SCSI_LUN_32(sp->cmd);

        mb[0] = MBC_ABORT_COMMAND;
        mb[1] = (b ? t | BIT_7 : t) << 8 | l;
        mb[2] = handle >> 16;
        mb[3] = handle & 0xffff;
        status = qla1280_mailbox_command(ha, BIT_3|BIT_2|BIT_1|BIT_0, &mb[0]);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
        if (status)
            qla1280_print("qla1280_abort_command: **** FAILED ****\n\r");
#endif
        sp->flags |= SRB_ABORT_PENDING;

        LEAVE("qla1280_abort_command");
        return(status);
}

/*
 * qla1280_reset_adapter
 *      Reset adapter.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void
qla1280_reset_adapter(scsi_qla_host_t *ha)
{
    device_reg_t *reg = ha->iobase;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_reset_adapter");
#endif

    /* Disable ISP chip */
    ha->flags.online = FALSE;
    WRT_REG_WORD(&reg->ictrl, ISP_RESET);
    WRT_REG_WORD(&reg->host_cmd, HC_RESET_RISC);
    WRT_REG_WORD(&reg->host_cmd, HC_RELEASE_RISC);
    WRT_REG_WORD(&reg->host_cmd, HC_DISABLE_BIOS);

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_reset_adapter");
#endif
}

/*
 *  Issue marker command.
 *      Function issues marker IOCB.
 *
 * Input:
 *      ha   = adapter block pointer.
 *      bus  = SCSI BUS number
 *      id   = SCSI ID
 *      lun  = SCSI LUN
 *      type = marker modifier
 */
static void
qla1280_marker(scsi_qla_host_t *ha, u8 bus, u32 id, u32 lun, u8 type)
{
    mrk_entry_t *pkt;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_marker");
#endif

    /* Get request packet. */
    if ( (pkt = (mrk_entry_t *)qla1280_req_pkt(ha)) )
    {
        pkt->entry_type = MARKER_TYPE;
        pkt->lun = (uint8_t)lun;
        pkt->target = (uint8_t)(bus ? (id | BIT_7) : id);
        pkt->modifier = type;

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_marker");
#endif
}

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,3,18)
/*
 * qla1280_64bit_start_scsi
 *      The start SCSI is responsible for building request packets on
 *      request ring and modifying ISP input pointer.
 *
 * Input:
 *      ha = adapter block pointer.
 *      sp = SB structure pointer.
 *
 * Returns:
 *      0 = success, was able to issue command.
 */
static int
qla1280_64bit_start_scsi(scsi_qla_host_t *ha, srb_t *sp)
{
    device_reg_t *reg = ha->iobase;
    Scsi_Cmnd *cmd = sp->cmd;
    cmd_a64_entry_t *pkt;
    struct scatterlist *sg = (struct scatterlist *) NULL;
    u32	*dword_ptr;
    dma_addr_t	    dma_handle;
    int	status = 0;
    int cnt;
    u16 req_cnt;
    u16	seg_cnt;

    ENTER("qla1280_64bit_start_scsi:");

    DEBUG(sprintf(debug_buff,
                  "64bit_start: cmd=%p sp=%p CDB=%x\n\r",
                  (void *)cmd, (void *)sp, cmd->cmnd[0]);)
    DEBUG(qla1280_print(debug_buff));

    if( qla1280_check_for_dead_scsi_bus(ha, sp) )
    {
        return(0);
    }

    /* Calculate number of entries and segments required. */
    seg_cnt = 0;
    req_cnt = 1;
    if (cmd->use_sg)
    {   /* 3.13 64 bit */
        sg = (struct scatterlist *) cmd->request_buffer;
        seg_cnt = pci_map_sg(ha->pdev,sg,cmd->use_sg, 
                    scsi_to_pci_dma_dir(cmd->sc_data_direction));
            
        if (seg_cnt > 2)
        {
           req_cnt += (seg_cnt - 2) / 5;
           if ((seg_cnt - 2) % 5)
               req_cnt++;
        }
    }
    else if (cmd->request_bufflen)  /* If data transfer. */
    {
      /*DEBUG(printk("Single data transfer len=0x%x\n",cmd->request_bufflen));*/
        seg_cnt = 1;
    }

    /* Acquire ring specific lock */
    QLA1280_RING_LOCK(ha);

    if ((req_cnt + 2) >= ha->req_q_cnt)
    {
        /* Calculate number of free request entries. */
        cnt = RD_REG_WORD(&reg->mailbox4);
        if (ha->req_ring_index < cnt)
            ha->req_q_cnt = cnt - ha->req_ring_index;
        else
            ha->req_q_cnt = REQUEST_ENTRY_CNT - (ha->req_ring_index - cnt);
    }

    /* If room for request in request ring. */
    if ((req_cnt + 2) < ha->req_q_cnt)
    {
        /* Check for room in outstanding command list. */
        for (cnt = 1; cnt < MAX_OUTSTANDING_COMMANDS &&
            ha->outstanding_cmds[cnt] != 0; cnt++)
            ;

        if (cnt < MAX_OUTSTANDING_COMMANDS)
        {
            ha->outstanding_cmds[cnt] = sp;
            ha->req_q_cnt -= req_cnt;
            CMD_HANDLE(sp->cmd) = (unsigned char *) (u_long) cnt;

            /*
            * Build command packet.
            */
            pkt = (cmd_a64_entry_t *)ha->request_ring_ptr;

            pkt->entry_type = COMMAND_A64_TYPE;
            pkt->entry_count = (uint8_t)req_cnt;
            pkt->sys_define = (uint8_t)ha->req_ring_index;
            pkt->handle = (uint32_t)cnt;

            /* Zero out remaining portion of packet. */
            dword_ptr = (uint32_t *)pkt + 2;
            for (cnt = 2; cnt < REQUEST_ENTRY_SIZE/4; cnt++)
                *dword_ptr++ = 0;

            /* Set ISP command timeout. */
            pkt->timeout = (uint16_t)30;

            /* Set device target ID and LUN */
            pkt->lun = SCSI_LUN_32(cmd);
            pkt->target = SCSI_BUS_32(cmd) ?
                    (SCSI_TCN_32(cmd) | BIT_7) : SCSI_TCN_32(cmd);

            /* Enable simple tag queuing if device supports it. */
            if (cmd->device->tagged_queue )
                pkt->control_flags |= BIT_3;

            /* Load SCSI command packet. */
            pkt->cdb_len = (uint16_t)CMD_CDBLEN(cmd);
            memcpy(pkt->scsi_cdb, &(CMD_CDBP(cmd)), pkt->cdb_len);
            //DEBUG(printk("Build packet for command[0]=0x%x\n",pkt->scsi_cdb[0]));

            /*
            * Load data segments.
            */
            if (seg_cnt)                /* If data transfer. */
            {
                /* Set transfer direction. */
                if ( (cmd->data_cmnd[0] == WRITE_6) )
                    pkt->control_flags |= BIT_6;
                else
                    pkt->control_flags |= (BIT_5|BIT_6);
                    
                sp->dir = pkt->control_flags & (BIT_5|BIT_6);

                /* Set total data segment count. */
                pkt->dseg_count = seg_cnt;

                /* Setup packet address segment pointer. */
                dword_ptr = (uint32_t *)&pkt->dseg_0_address;

                if (cmd->use_sg)              /* If scatter gather */
                {
                    /* Load command entry data segments. */
                    for (cnt = 0; cnt < 2 && seg_cnt; cnt++, seg_cnt--)
                    {
                        /* 3.13 64 bit */
                        *dword_ptr++ = cpu_to_le32(pci_dma_lo32(sg_dma_address(sg)));
                        *dword_ptr++ = cpu_to_le32(pci_dma_hi32(sg_dma_address(sg)));
                        *dword_ptr++ = cpu_to_le32(sg_dma_len(sg));
                        sg++;
                        DEBUG(sprintf(debug_buff,
                         "S/G Segment phys_addr=%x %x, len=0x%x\n\r",
                          cpu_to_le32(pci_dma_hi32(sg_dma_address(sg))),
                          cpu_to_le32(pci_dma_lo32(sg_dma_address(sg))),
                                      cpu_to_le32(sg_dma_len(sg)));)
                        DEBUG(qla1280_print(debug_buff));
                    }
#ifdef QL_DEBUG_LEVEL_5
                    qla1280_print(
                            "qla1280_64bit_start_scsi: Scatter/gather command packet data - ");
                    qla1280_print("b ");
                    qla1280_output_number((uint32_t)SCSI_BUS_32(cmd), 10);
                    qla1280_print(" t ");
                    qla1280_output_number((uint32_t)SCSI_TCN_32(cmd), 10);
                    qla1280_print(" d ");
                    qla1280_output_number((uint32_t)SCSI_LUN_32(cmd), 10);
                    qla1280_print("\n\r");
                    qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
                    /*
                    * Build continuation packets.
                    */
                    DEBUG(sprintf(debug_buff,
                    "S/G Building Continuation...seg_cnt=0x%x remains\n\r",
                    seg_cnt);)
                    DEBUG(qla1280_print(debug_buff));
                    while (seg_cnt > 0)
                    {
                        /* Adjust ring index. */
                        ha->req_ring_index++;
                        if (ha->req_ring_index == REQUEST_ENTRY_CNT)
                        {
                            ha->req_ring_index = 0;
                            ha->request_ring_ptr = ha->request_ring;
                        }
                        else
                            ha->request_ring_ptr++;

                        pkt = (cmd_a64_entry_t *)ha->request_ring_ptr;

                        /* Zero out packet. */
                        dword_ptr = (uint32_t *)pkt;
                        for (cnt = 0;cnt < REQUEST_ENTRY_SIZE/4; cnt++)
                            *dword_ptr++ = 0;

                        /* Load packet defaults. */
                        ((cont_a64_entry_t *)pkt)->entry_type =
                                CONTINUE_A64_TYPE;
                        ((cont_a64_entry_t *)pkt)->entry_count = 1;
                        ((cont_a64_entry_t *)pkt)->sys_define = (uint8_t)
                                ha->req_ring_index;

                        /* Setup packet address segment pointer. */
                        dword_ptr = (uint32_t *)
                                &((cont_a64_entry_t *)pkt)->dseg_0_address;

                        /* Load continuation entry data segments. */
                        for (cnt = 0; cnt < 5 && seg_cnt; cnt++, seg_cnt--)
                        {
                           /* 3.13 64 bit */
                           *dword_ptr++ = cpu_to_le32(pci_dma_lo32(sg_dma_address(sg)));
                           *dword_ptr++ = cpu_to_le32(pci_dma_hi32(sg_dma_address(sg)));
                           *dword_ptr++ = cpu_to_le32(sg_dma_len(sg));
                           DEBUG(sprintf(debug_buff,
                            "S/G Segment Cont. phys_addr=%x %x, len=0x%x\n\r",
                             cpu_to_le32(pci_dma_hi32(sg_dma_address(sg))),
                             cpu_to_le32(pci_dma_lo32(sg_dma_address(sg))),
                             cpu_to_le32(sg_dma_len(sg)));)
                           DEBUG(qla1280_print(debug_buff));
                           sg++;
                        }
#ifdef QL_DEBUG_LEVEL_5
                        qla1280_print(
                                "qla1280_64bit_start_scsi: continuation packet data - c");
                        qla1280_print(" b ");
                        qla1280_output_number((uint32_t)SCSI_BUS_32(cmd), 10);

                        qla1280_print(" t ");
                        qla1280_output_number((uint32_t)SCSI_TCN_32(cmd), 10);
                        qla1280_print(" d ");
                        qla1280_output_number((uint32_t)SCSI_LUN_32(cmd), 10);
                        qla1280_print("\n\r");
                        qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
                    }
                }
                else         /* No scatter gather data transfer */
                {   /* 3.13 64 bit */
                    dma_handle = pci_map_single(ha->pdev, 
                                                cmd->request_buffer,
                                                cmd->request_bufflen,
                                                scsi_to_pci_dma_dir(cmd->sc_data_direction));
                    /* save dma_handle for pci_unmap_single */
                    sp->saved_dma_handle = dma_handle;

                    *dword_ptr++ = cpu_to_le32(pci_dma_lo32(dma_handle));
                    *dword_ptr++ = cpu_to_le32(pci_dma_hi32(dma_handle));
                    *dword_ptr   = (uint32_t) cmd->request_bufflen;
                    /*DEBUG(sprintf(debug_buff,
                    "No S/G map_single saved_dma_handle=%lx\n\r",dma_handle));
                    DEBUG(qla1280_print(debug_buff));*/
#ifdef QL_DEBUG_LEVEL_5
                    qla1280_print(
                            "qla1280_64bit_start_scsi: No scatter/gather command packet data - c");
                    qla1280_print(" b ");
                    qla1280_output_number((uint32_t)SCSI_BUS_32(cmd), 10);
                    qla1280_print(" t ");
                    qla1280_output_number((uint32_t)SCSI_TCN_32(cmd), 10);
                    qla1280_print(" d ");
                    qla1280_output_number((uint32_t)SCSI_LUN_32(cmd), 10);
                    qla1280_print("\n\r");
                    qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
                }
            }
            else         /* No data transfer */
            {
                *dword_ptr++ = (uint32_t) 0;
                *dword_ptr++ = (uint32_t) 0;
                *dword_ptr = (uint32_t)  0;
#ifdef QL_DEBUG_LEVEL_5
                qla1280_print(
                        "qla1280_64bit_start_scsi: No data, command packet data - c");
                qla1280_print(" b ");
                qla1280_output_number((uint32_t)SCSI_BUS_32(cmd), 10);
                qla1280_print(" t ");
                qla1280_output_number((uint32_t)SCSI_TCN_32(cmd), 10);
                qla1280_print(" d ");
                qla1280_output_number((uint32_t)SCSI_LUN_32(cmd), 10);
                qla1280_print("\n\r");
                qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
            }
            /* Adjust ring index. */
            ha->req_ring_index++;
            if (ha->req_ring_index == REQUEST_ENTRY_CNT)
            {
                ha->req_ring_index = 0;
                ha->request_ring_ptr = ha->request_ring;
            }
            else
                ha->request_ring_ptr++;

            /* Set chip new ring index. */
            DEBUG(qla1280_print("qla1280_64bit_start_scsi: Wakeup RISC for pending command\n\r"));
            ha->qthreads--;
            sp->flags |= SRB_SENT;
            ha->actthreads++;
            WRT_REG_WORD(&reg->mailbox4, ha->req_ring_index);
        }
        else
        {
            status = 1;
#ifdef QL_DEBUG_LEVEL_2
            qla1280_print(
                    "qla1280_64bit_start_scsi: NO ROOM IN OUTSTANDING ARRAY\n\r");
            qla1280_print(" req_q_cnt=");
            qla1280_output_number((u32)ha->req_q_cnt, 16);
#endif
        }
    }
    else
    {
        status = 1;
#ifdef QL_DEBUG_LEVEL_2
        qla1280_print("qla1280_64bit_start_scsi: in-ptr=");
        qla1280_output_number((uint32_t)ha->req_ring_index, 16);
        qla1280_print(" req_q_cnt=");
        qla1280_output_number((u32)ha->req_q_cnt, 16);
        qla1280_print(" req_cnt=");
        qla1280_output_number(req_cnt, 16);
        qla1280_print("\n\r");
#endif
    }

    /* Release ring specific lock */
    QLA1280_RING_UNLOCK(ha);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (status)
        qla1280_print("qla1280_64bit_start_scsi: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_64bit_start_scsi: exiting normally\n\r");
#endif
    return(status);
}
#endif

/*
 * qla1280_32bit_start_scsi
 *      The start SCSI is responsible for building request packets on
 *      request ring and modifying ISP input pointer.
 *
 *      The Qlogic firmware interface allows every queue slot to have a SCSI
 *      command and up to 4 scatter/gather (SG) entries.  If we need more
 *      than 4 SG entries, then continuation entries are used that can 
 *      hold another 7 entries each.  The start routine determines if there
 *      is eought empty slots then build the combination of requests to 
 *      fulfill the OS request.
 *
 * Input:
 *      ha = adapter block pointer.
 *      sp = SCSI Request Block structure pointer.
 *
 * Returns:
 *      0 = success, was able to issue command.
 */
static int
qla1280_32bit_start_scsi(scsi_qla_host_t *ha, srb_t *sp)
{
    device_reg_t    *reg   = ha->iobase;
    Scsi_Cmnd       *cmd = sp->cmd;
    cmd_entry_t     *pkt;
    struct scatterlist    *sg = (struct scatterlist *) NULL;
    uint8_t        *data_ptr;
    uint32_t        *dword_ptr;
    int status = 0;
    int cnt;
    int req_cnt;
    uint16_t        seg_cnt;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
    dma_addr_t       dma_handle;
#endif

    ENTER("qla1280_32bit_start_scsi");

    DEBUG(sprintf(debug_buff,
                  "32bit_start: cmd=%p sp=%p CDB=%x\n\r",
                  (void *)cmd, (void *)sp, cmd->cmnd[0]));
    DEBUG(qla1280_print(debug_buff));

    if( qla1280_check_for_dead_scsi_bus(ha, sp) )
    {
        return(0);
    }

    /* Calculate number of entries and segments required. */
    req_cnt = 1;
    if (cmd->use_sg)
    {
        /*
         * We must build an SG list in adapter format, as the kernel's SG list
         * cannot be used directly because of data field size (__alpha__)
         * differences and the kernel SG list uses virtual addresses where
         * we need physical addresses.
         */
        sg = (struct scatterlist *) cmd->request_buffer;
        /* 3.13 32 bit */
        seg_cnt = pci_map_sg(ha->pdev, sg, cmd->use_sg, 
                             scsi_to_pci_dma_dir(cmd->sc_data_direction));

        /* 
         * if greater than four sg entries then we need to allocate
         * continuation entries
         */
        if (seg_cnt > 4)
        {
            req_cnt += (uint16_t)(seg_cnt - 4) / 7;
            if ((uint16_t)(seg_cnt - 4) % 7)
                req_cnt++;
        }
        DEBUG(sprintf(debug_buff,
         "S/G Transfer cmd=%p seg_cnt=0x%x, req_cnt=%x\n\r",
          (void *)cmd,seg_cnt,req_cnt));
        DEBUG(qla1280_print(debug_buff));
    }
    else if (cmd->request_bufflen)  /* If data transfer. */
    {
        DEBUG(sprintf(debug_buff,
         "No S/G transfer t=%x cmd=%p len=%x CDB=%x\n\r",
         SCSI_TCN_32(cmd), (void *)cmd,cmd->request_bufflen,cmd->cmnd[0]));
        DEBUG(qla1280_print(debug_buff));
        seg_cnt = 1;
    }
    else
    {
      //DEBUG(printk("No data transfer \n"));
        seg_cnt = 0;
    }

    /* Acquire ring specific lock */
    QLA1280_RING_LOCK(ha);

    if ((req_cnt + 2) >= ha->req_q_cnt)
    {
        /* Calculate number of free request entries. */
        cnt = RD_REG_WORD(&reg->mailbox4);
        if (ha->req_ring_index < cnt)
            ha->req_q_cnt = cnt - ha->req_ring_index;
        else
            ha->req_q_cnt = REQUEST_ENTRY_CNT - (ha->req_ring_index - cnt);
    }

    DEBUG(sprintf(debug_buff,"Number of free entries=(%d) seg_cnt=0x%x\n\r",
          ha->req_q_cnt,seg_cnt));
    DEBUG(qla1280_print(debug_buff));
    /* If room for request in request ring. */
    if ((req_cnt + 2) < ha->req_q_cnt)
    {
        /* Check for empty slot in outstanding command list. */
        for (cnt = 1; cnt < MAX_OUTSTANDING_COMMANDS &&
            (ha->outstanding_cmds[cnt] != 0); cnt++)
            ;

        if (cnt < MAX_OUTSTANDING_COMMANDS)
        {
            CMD_HANDLE(sp->cmd) = (unsigned char *)(unsigned long)cnt;
            ha->outstanding_cmds[cnt] = sp;
            ha->req_q_cnt -= req_cnt;

            /*
            * Build command packet.
            */
            pkt = (cmd_entry_t *)ha->request_ring_ptr;

            pkt->entry_type = COMMAND_TYPE;
            pkt->entry_count = (uint8_t)req_cnt;
            pkt->sys_define = (uint8_t)ha->req_ring_index;
            pkt->handle = (uint32_t)cnt;

            /* Zero out remaining portion of packet. */
            dword_ptr = (uint32_t *)pkt + 2;
            for (cnt = 2; cnt < REQUEST_ENTRY_SIZE/4; cnt++)
                *dword_ptr++ = 0;

            /* Set ISP command timeout. */
            pkt->timeout = (uint16_t)30;

            /* Set device target ID and LUN */
            pkt->lun = SCSI_LUN_32(cmd);
            pkt->target = SCSI_BUS_32(cmd) ?
                    (SCSI_TCN_32(cmd) | BIT_7) : SCSI_TCN_32(cmd);

            /* Enable simple tag queuing if device supports it. */
            if (cmd->device->tagged_queue )
                pkt->control_flags |= BIT_3;

            /* Load SCSI command packet. */
            pkt->cdb_len = (uint16_t)CMD_CDBLEN(cmd);
            data_ptr = (uint8_t *) &(CMD_CDBP(cmd));
            for (cnt = 0; cnt < pkt->cdb_len; cnt++)
                pkt->scsi_cdb[cnt] = *data_ptr++;
            //DEBUG(printk("Build packet for command[0]=0x%x\n",pkt->scsi_cdb[0]));
            /*
            * Load data segments.
            */
            if (seg_cnt)
            {
                /* Set transfer direction (READ and WRITE) */
                /* Linux doesn't tell us                   */
                /*
                * For block devices, cmd->request.cmd has the operation 
                * For character devices, this isn't always set properly, so
                * we need to check data_cmnd[0].  This catches the conditions
                * for st.c, but not sg. Generic commands are pass down to us.
                */
                if ( (cmd->data_cmnd[0] == WRITE_6) )
                    pkt->control_flags |= BIT_6;
                else
                    pkt->control_flags |= (BIT_5|BIT_6);
                    
                sp->dir = pkt->control_flags & (BIT_5|BIT_6);

                /* Set total data segment count. */
                pkt->dseg_count = seg_cnt;

                /* Setup packet address segment pointer. */
                dword_ptr = (uint32_t *)&pkt->dseg_0_address;

                if (cmd->use_sg)     /* If scatter gather */
                {
#if QL_DEBUG_ROUTINES
                    DEBUG(qla1280_print("Building S/G data segments..\n\r"));
                    DEBUG(qla1280_dump_buffer((char *)sg, 4*16 ));
#endif
                    /* Load command entry data segments. */
                    for (cnt = 0; cnt < 4 && seg_cnt; cnt++, seg_cnt--)
                    {
                        /* 3.13 32 bit */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
                        *dword_ptr++ = cpu_to_le32(virt_to_bus(sg->address));
                        *dword_ptr++ = sg->length;
                        DEBUG(sprintf(debug_buff,
                         "S/G Segment phys_addr=0x%x, len=0x%x\n\r",
                          cpu_to_le32(virt_to_bus(sg->address)),sg->length));
                        DEBUG(qla1280_print(debug_buff));
#else
                        *dword_ptr++ = cpu_to_le32(pci_dma_lo32(sg_dma_address(sg)));
                        *dword_ptr++ = cpu_to_le32(sg_dma_len(sg));
                        DEBUG(sprintf(debug_buff,
                         "S/G Segment phys_addr=0x%x, len=0x%x\n\r",
                          cpu_to_le32(pci_dma_lo32(sg_dma_address(sg))),
                          cpu_to_le32(sg_dma_len(sg)));)
                        DEBUG(qla1280_print(debug_buff));
#endif 
                        sg++;
                    }
                    /*
                    * Build continuation packets.
                    */
                    DEBUG(sprintf(debug_buff,
                    "S/G Building Continuation...seg_cnt=0x%x remains\n\r",
                    seg_cnt);)
                    DEBUG(qla1280_print(debug_buff));
                    while (seg_cnt > 0)
                    {
                        /* Adjust ring index. */
                        ha->req_ring_index++;
                        if (ha->req_ring_index == REQUEST_ENTRY_CNT)
                        {
                            ha->req_ring_index = 0;
                            ha->request_ring_ptr = ha->request_ring;
                        }
                        else
                            ha->request_ring_ptr++;

                        pkt = (cmd_entry_t *)ha->request_ring_ptr;

                        /* Zero out packet. */
                        dword_ptr = (uint32_t *)pkt;
                        for (cnt = 0;cnt < REQUEST_ENTRY_SIZE/4; cnt++)
                            *dword_ptr++ = 0;

                        /* Load packet defaults. */
                        ((cont_entry_t *)pkt)->entry_type =
                                CONTINUE_TYPE;
                        ((cont_entry_t *)pkt)->entry_count = 1;

                        ((cont_entry_t *)pkt)->sys_define = (uint8_t)
                                ha->req_ring_index;

                        /* Setup packet address segment pointer. */
                        dword_ptr = (uint32_t *)
                                &((cont_entry_t *)pkt)->dseg_0_address;

                        /* Load continuation entry data segments. */
                        for (cnt = 0; cnt < 7 && seg_cnt; cnt++, seg_cnt--)
                        {
                           /* 3.13 32 bit */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
                           *dword_ptr++ = cpu_to_le32(virt_to_bus(sg->address));
                           *dword_ptr++ = sg->length;
                           DEBUG(sprintf(debug_buff,
                           "S/G Segment Cont. phys_addr=0x%x, len=0x%x\n\r",
                           cpu_to_le32(pci_dma_lo32(virt_to_bus(sg->address))),
                           sg->length);)
                           DEBUG(qla1280_print(debug_buff));
#else
                           *dword_ptr++ = cpu_to_le32(pci_dma_lo32(sg_dma_address(sg)));
                           *dword_ptr++ = cpu_to_le32(sg_dma_len(sg));
                           DEBUG(sprintf(debug_buff,
                            "S/G Segment Cont. phys_addr=0x%x, len=0x%x\n\r",
                             cpu_to_le32(pci_dma_lo32(sg_dma_address(sg))),
                             cpu_to_le32(sg_dma_len(sg)));)
                           DEBUG(qla1280_print(debug_buff));
#endif
                           sg++;
                        }
#ifdef QL_DEBUG_LEVEL_5
                        qla1280_print(
                                "qla1280_32bit_start_scsi: continuation packet data - scsi(");
                        qla1280_output_number((uint32_t)SCSI_BUS_32(cmd), 10);
                        qla1280_print(":");
                        qla1280_output_number((uint32_t)SCSI_TCN_32(cmd), 10);
                        qla1280_print(":");
                        qla1280_output_number((uint32_t)SCSI_LUN_32(cmd), 10);
                        qla1280_print(")\n\r");
                        qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
                    }
                }
                else     /* No S/G data transfer */
                {
                    /* 3.13 32 bit */
#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,18)
                    *dword_ptr++ = cpu_to_le32(virt_to_bus(cmd->request_buffer));
                    *dword_ptr = (uint32_t) cmd->request_bufflen;
#else
                    dma_handle = pci_map_single(ha->pdev, 
                                                cmd->request_buffer,
                                                cmd->request_bufflen,
                                                scsi_to_pci_dma_dir(cmd->sc_data_direction));
                    sp->saved_dma_handle = dma_handle;

                    *dword_ptr++ = cpu_to_le32(pci_dma_lo32(dma_handle));
                    *dword_ptr   = (uint32_t) cmd->request_bufflen;
                    /*DEBUG(sprintf(debug_buff,
                    "No S/G map_single saved_dma_handle=%lx\n\r",dma_handle));
                    DEBUG(qla1280_print(debug_buff));*/
#endif 
                }
            }
            else         /* No data transfer at all */
            {
                *dword_ptr++ = (uint32_t) 0;
                *dword_ptr = (uint32_t)  0;
#ifdef QL_DEBUG_LEVEL_5
                qla1280_print(
                        "qla1280_32bit_start_scsi: No data, command packet data - ");
                qla1280_print("\n\r");
                qla1280_dump_buffer((char *)pkt, REQUEST_ENTRY_SIZE);
#endif
            }
#ifdef QL_DEBUG_LEVEL_5
            qla1280_print("qla1280_32bit_start_scsi: First IOCB block:\n\r");
            qla1280_dump_buffer((char *)ha->request_ring_ptr, REQUEST_ENTRY_SIZE);
#endif
            /* Adjust ring index. */
            ha->req_ring_index++;
            if (ha->req_ring_index == REQUEST_ENTRY_CNT)
            {
                ha->req_ring_index = 0;
                ha->request_ring_ptr = ha->request_ring;
            }
            else
                ha->request_ring_ptr++;

            /* Set chip new ring index. */
            DEBUG(qla1280_print("qla1280_32bit_start_scsi: Wakeup RISC for pending command\n\r"));
            ha->qthreads--;
            sp->flags |= SRB_SENT;
            ha->actthreads++;
            WRT_REG_WORD(&reg->mailbox4, ha->req_ring_index);
        }
        else
        {
            status = 1;
#ifdef QL_DEBUG_LEVEL_2
            qla1280_print(
            "qla1280_32bit_start_scsi: NO ROOM IN OUTSTANDING ARRAY\n\r");
            qla1280_print(" req_q_cnt=");
            qla1280_output_number((u32)ha->req_q_cnt, 16);
            qla1280_print("\n\r");
#endif
        }
    }
    else
    {
        status = 1;
#ifdef QL_DEBUG_LEVEL_2
        /*  qla1280_print("qla1280_32bit_start_scsi: in-ptr=");
        qla1280_output_number((uint32_t)ha->req_ring_index, 16);
        qla1280_print(" req_q_cnt=");
        qla1280_output_number((u32)ha->req_q_cnt, 16);
        qla1280_print(" req_cnt=");
        qla1280_output_number(req_cnt, 16);
        qla1280_print("\n\r"); */
#endif
    }

    /* Release ring specific lock */
    QLA1280_RING_UNLOCK(ha);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    /* if (status)
    qla1280_print("qla1280_32bit_start_scsi: **** FAILED ****\n\r"); */
#endif
#ifdef QL_DEBUG_LEVEL_3
            LEAVE("qla1280_32bit_start_scsi");
#endif
    return(status);
}


/*
 * qla1280_req_pkt
 *      Function is responsible for locking ring and
 *      getting a zeroed out request packet.
 *
 * Input:
 *      ha  = adapter block pointer.
 *
 * Returns:
 *      0 = failed to get slot.
 */
static request_t *
qla1280_req_pkt(scsi_qla_host_t *ha)
{
    device_reg_t    *reg = ha->iobase;
    request_t       *pkt = 0;
    int		    cnt;
    uint32_t        *dword_ptr;
    uint32_t        timer;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_req_pkt");
#endif

    /* Wait for 30 seconds for slot. */
    for (timer = 15000000; timer; timer--)
    {
        /* Acquire ring specific lock */
        QLA1280_RING_LOCK(ha);

        if (ha->req_q_cnt > 0)
        {
            /* Calculate number of free request entries. */
            cnt = RD_REG_WORD(&reg->mailbox4);
            if (ha->req_ring_index < cnt)
                ha->req_q_cnt = cnt - ha->req_ring_index;
            else
                ha->req_q_cnt = REQUEST_ENTRY_CNT - (ha->req_ring_index - cnt);
        }

        /* Found empty request ring slot? */
        if (ha->req_q_cnt > 0)
        {
            ha->req_q_cnt--;
            pkt = ha->request_ring_ptr;

            /* Zero out packet. */
            dword_ptr = (uint32_t *)pkt;
            for (cnt = 0; cnt < REQUEST_ENTRY_SIZE/4; cnt++)
                *dword_ptr++ = 0;

            /* Set system defined field. */
            pkt->sys_define = (uint8_t)ha->req_ring_index;

            /* Set entry count. */
            pkt->entry_count = 1;

            break;
        }

        /* Release ring specific lock */
        QLA1280_RING_UNLOCK(ha);

        SYS_DELAY(2);   /* 10 */ 

        /* Check for pending interrupts. */
        qla1280_poll(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_req_pkt: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_req_pkt: exiting normally\n\r");
#endif
    return(pkt);
}

/*
 * qla1280_isp_cmd
 *      Function is responsible for modifying ISP input pointer.
 *      Releases ring lock.
 *
 * Input:
 *      ha  = adapter block pointer.
 */
static void
qla1280_isp_cmd(scsi_qla_host_t *ha)
{
    device_reg_t    *reg = ha->iobase;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_isp_cmd");
#endif

#ifdef QL_DEBUG_LEVEL_5
    qla1280_print("qla1280_isp_cmd: IOCB data:\n\r");
    qla1280_dump_buffer((char *)ha->request_ring_ptr, REQUEST_ENTRY_SIZE); 
#endif

    /* Adjust ring index. */
    ha->req_ring_index++;
    if (ha->req_ring_index == REQUEST_ENTRY_CNT)
    {
        ha->req_ring_index = 0;
        ha->request_ring_ptr = ha->request_ring;
    }
    else
        ha->request_ring_ptr++;

    /* Set chip new ring index. */
    WRT_REG_WORD(&reg->mailbox4, ha->req_ring_index);

    /* Release ring specific lock */
    QLA1280_RING_UNLOCK(ha);

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_isp_cmd");
#endif
}

#if QL1280_LUN_SUPPORT
/*
 * qla1280_enable_lun
 *      Issue enable LUN entry IOCB.
 *
 * Input:
 *      ha = adapter block pointer.
 *      b  = SCSI BUS number.
 *      l  = LUN number.
 */
static void
qla1280_enable_lun(scsi_qla_host_t *ha, uint8_t b, uint32_t l)
{
    elun_entry_t    *pkt;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_enable_lun: entered\n\r");
#endif

    /* Get request packet. */
    /*
    if (pkt = (elun_entry_t *)qla1280_req_pkt(ha))
    {
    pkt->entry_type = ENABLE_LUN_TYPE;
    pkt->lun = (uint16_t)(b ? l | BIT_15 : l);
    pkt->command_count = 32;
    pkt->immed_notify_count = 1;
    pkt->group_6_length = MAX_CMDSZ;
    pkt->group_7_length = MAX_CMDSZ;
    pkt->timeout = 0x30;

    qla1280_isp_cmd(ha);
    }
    */
    pkt = (elun_entry_t *)1;

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_enable_lun: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_enable_lun: exiting normally\n\r");
#endif
}
#endif

#if QL1280_TARGET_MODE_SUPPORT
/****************************************************************************/
/*                      Target Mode Support Functions.                      */
/****************************************************************************/

/*
 * qla1280_notify_ack
 *      Issue notify acknowledge IOCB.
 *      If sequence ID is zero, acknowledgement of
 *      SCSI bus reset or bus device reset is assumed.
 *
 * Input:
 *      ha      = adapter block pointer.
 *      inotify = immediate notify entry pointer.
 */
static void
qla1280_notify_ack(scsi_qla_host_t *ha, notify_entry_t *inotify)
{
    nack_entry_t    *pkt;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_notify_ack: entered\n\r");
#endif

    /* Get request packet. */
    if (pkt = (nack_entry_t *)qla1280_req_pkt(ha))
    {
        pkt->entry_type = NOTIFY_ACK_TYPE;
        pkt->lun = inotify->lun;
        pkt->initiator_id = inotify->initiator_id;
        pkt->target_id = inotify->target_id;
        if (inotify->seq_id == 0)
            pkt->event = BIT_7;
        else
            pkt->seq_id = inotify->seq_id;

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_notify_ack: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_notify_ack: exiting normally\n\r");
#endif
}

/*
 * qla1280_immed_notify
 *      Issue immediate notify IOCB for LUN 0.
 *
 * Input:
 *      ha      = adapter block pointer.
 *      inotify = immediate notify entry pointer.
 */
static void
qla1280_immed_notify(scsi_qla_host_t *ha, notify_entry_t *inotify)
{
    notify_entry_t    *pkt;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_immed_notify: entered\n\r");
#endif

    /* Get request packet. */
    if (pkt = (notify_entry_t *)qla1280_req_pkt(ha))
    {
        pkt->entry_type = IMMED_NOTIFY_TYPE;
        pkt->lun = inotify->lun;
        pkt->initiator_id = inotify->initiator_id;
        pkt->target_id = inotify->target_id;
        pkt->status = 1;

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_immed_notify: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_immed_notify: exiting normally\n\r");
#endif
}

/*
 * qla1280_accept_io
 *      Issue accept target I/O IOCB for LUN 0.
 *
 * Input:
 *      ha = adapter block pointer.
 *      ctio = ctio returned entry pointer.
 */
static void
qla1280_accept_io(scsi_qla_host_t *ha, ctio_ret_entry_t *ctio)
{
    atio_entry_t    *pkt;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_accept_io: entered\n\r");
#endif

    /* Get request packet. */
    if (pkt = (atio_entry_t *)qla1280_req_pkt(ha))
    {
        pkt->entry_type = ACCEPT_TGT_IO_TYPE;
        pkt->lun = ctio->lun;
        pkt->initiator_id = ctio->initiator_id;
        pkt->target_id = ctio->target_id;
        pkt->tag_value = ctio->tag_value;
        pkt->status = 1;

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_accept_io: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_accept_io: exiting normally\n\r");
#endif
}

/*
 * qla1280_64bit_continue_io
 *      Issue continue target I/O IOCB.
 *
 * Input:
 *      ha   = adapter block pointer.
 *      atio = atio pointer.
 *      len  = total bytecount.
 *      addr = physical address pointer.
 */
static void
qla1280_64bit_continue_io(scsi_qla_host_t *ha, atio_entry_t *atio, uint32_t len,
                    paddr32_t *addr)
{
    ctio_a64_entry_t *pkt;
    uint32_t         *dword_ptr;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_64bit_continue_io: entered\n\r");
#endif

    /* Get request packet. */
    if (pkt = (ctio_a64_entry_t *)qla1280_req_pkt(ha))
    {
        pkt->entry_type = CTIO_A64_TYPE;
        pkt->lun = atio->lun;
        pkt->initiator_id = atio->initiator_id;
        pkt->target_id = atio->target_id;
        pkt->option_flags = atio->option_flags;
        pkt->tag_value = atio->tag_value;
        pkt->scsi_status = atio->scsi_status;

        if (len)
        {
            pkt->dseg_count = 1;
            pkt->transfer_length = len;
            pkt->dseg_0_length = len;
            dword_ptr = (uint32_t *)addr;
            pkt->dseg_0_address[0] = *dword_ptr++;
            pkt->dseg_0_address[1] = *dword_ptr;
        }

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_64bit_continue_io: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_64bit_continue_io: exiting normally\n\r");
#endif
}

/*
 * qla1280_32bit_continue_io
 *      Issue continue target I/O IOCB.
 *
 * Input:
 *      ha   = adapter block pointer.
 *      atio = atio pointer.
 *      len  = total bytecount.
 *      addr = physical address pointer.
 */
static void
qla1280_32bit_continue_io(scsi_qla_host_t *ha, atio_entry_t *atio, uint32_t len,
                    paddr32_t *addr)
{
    ctio_entry_t *pkt;
    uint32_t     *dword_ptr;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_32bit_continue_io: entered\n\r");
#endif

    /* Get request packet. */
    if (pkt = (ctio_entry_t *)qla1280_req_pkt(ha))
    {
        pkt->entry_type = CONTINUE_TGT_IO_TYPE;
        pkt->lun = atio->lun;
        pkt->initiator_id = atio->initiator_id;
        pkt->target_id = atio->target_id;
        pkt->option_flags = atio->option_flags;
        pkt->tag_value = atio->tag_value;
        pkt->scsi_status = atio->scsi_status;

        if (len)
        {
            pkt->dseg_count = 1;
            pkt->transfer_length = len;
            pkt->dseg_0_length = len;
            dword_ptr = (uint32_t *)addr;
            pkt->dseg_0_address = *dword_ptr;
        }

        /* Issue command to ISP */
        qla1280_isp_cmd(ha);
    }

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
    if (!pkt)
        qla1280_print("qla1280_32bit_continue_io: **** FAILED ****\n\r");
#endif
#ifdef QL_DEBUG_LEVEL_3
    else
        qla1280_print("qla1280_32bit_continue_io: exiting normally\n\r");
#endif
}
#endif /* QL1280_TARGET_MODE_SUPPORT */

/****************************************************************************/
/*                        Interrupt Service Routine.                        */
/****************************************************************************/

/****************************************************************************
 *  qla1280_isr
 *      Calls I/O done on command completion.
 *
 * Input:
 *      ha           = adapter block pointer.
 *      done_q_first = done queue first pointer.
 *      done_q_last  = done queue last pointer.
 *      INTR_LOCK must be already obtained.
 ****************************************************************************/
static void
qla1280_isr(scsi_qla_host_t *ha, srb_t **done_q_first, srb_t **done_q_last)
{
    device_reg_t    *reg = ha->iobase;
    response_t      *pkt;
    srb_t           *sp = 0; 
    uint16_t        mailbox[MAILBOX_REGISTER_COUNT];
    uint16_t        *wptr;
    uint32_t        index;

    ENTER("qla1280_isr");


    /* Save mailbox register 5 */
    mailbox[5] = RD_REG_WORD(&reg->mailbox5);

    /* Check for mailbox interrupt. */

    mailbox[0] = RD_REG_WORD(&reg->semaphore);

    if (mailbox[0] & BIT_0)
    {
        /* Get mailbox data. */
        //DEBUG(qla1280_print("qla1280_isr: In Get mailbox data \n\r");)

        wptr = &mailbox[0];
        *wptr++ = RD_REG_WORD(&reg->mailbox0);
        *wptr++ = RD_REG_WORD(&reg->mailbox1);
        *wptr = RD_REG_WORD(&reg->mailbox2);
        if (mailbox[0] != MBA_SCSI_COMPLETION)
        {
            wptr++;
            *wptr++ = RD_REG_WORD(&reg->mailbox3);
            *wptr++ = RD_REG_WORD(&reg->mailbox4);
            wptr++;
            *wptr++ = RD_REG_WORD(&reg->mailbox6);
            *wptr   = RD_REG_WORD(&reg->mailbox7);
        }

        /* Release mailbox registers. */

        WRT_REG_WORD(&reg->semaphore, 0);
        WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);

#ifdef QL_DEBUG_LEVEL_5
        qla1280_print("qla1280_isr: mailbox interrupt mailbox[0] = ");
        qla1280_output_number((uint32_t)mailbox[0], 16);
        qla1280_print("\n\r");
#endif

        /* Handle asynchronous event */

        switch (mailbox[0])
        {
            case MBA_SCSI_COMPLETION:   /* Response completion */
#ifdef QL_DEBUG_LEVEL_5
                qla1280_print("qla1280_isr: mailbox SCSI response completion\n\r");
#endif
                if (ha->flags.online)
                {
                    /* Get outstanding command index. */
                    index = (uint32_t)(mailbox[2] << 16 | mailbox[1]);

                    /* Validate handle. */
                    if (index < MAX_OUTSTANDING_COMMANDS)
                        sp = ha->outstanding_cmds[index];
                    else
                        sp = 0;

                    if (sp)
                    {
                        /* Free outstanding command slot. */
                        ha->outstanding_cmds[index] = 0;

                        /* Save ISP completion status */
                        CMD_RESULT(sp->cmd) = 0;

                        /* Place block on done queue */
                        sp->s_next = NULL;
                        sp->s_prev = *done_q_last;
                        if (!*done_q_first)
                            *done_q_first = sp;
                        else
                            (*done_q_last)->s_next = sp;
                        *done_q_last = sp;

                    }
                    else
                    {

#ifdef QL_DEBUG_LEVEL_2
                            qla1280_print("qla1280_isr: ISP invalid handle\n\r");
#endif
                            printk(KERN_WARNING "qla1280: ISP invalid handle");
                            ha->flags.isp_abort_needed = TRUE;
                    }
                }
                break;
            case MBA_BUS_RESET:         /* SCSI Bus Reset */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: asynchronous BUS_RESET\n\r");
#endif
                ha->flags.reset_marker = TRUE;
                index = mailbox[6] & BIT_0;
                ha->bus_settings[index].reset_marker = TRUE;
                break;
            case MBA_SYSTEM_ERR:        /* System Error */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: ISP System Error - mbx1=");
                qla1280_output_number((uint32_t)mailbox[1], 16);
                qla1280_print(", mbx2=");
                qla1280_output_number((uint32_t)mailbox[2], 16);
                qla1280_print(", mbx3=");
                qla1280_output_number((uint32_t)mailbox[3], 16);
                qla1280_print("\n\r");
#endif
                printk(KERN_WARNING
                        "qla1280: ISP System Error - mbx1=%xh, mbx2=%xh, mbx3=%xh\n",
                        mailbox[1], mailbox[2], mailbox[3]);
                ha->flags.isp_abort_needed = TRUE;
                break;
            case MBA_REQ_TRANSFER_ERR:  /* Request Transfer Error */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: ISP Request Transfer Error\n\r");
#endif
                printk(KERN_WARNING "qla1280: ISP Request Transfer Error\n");
                ha->flags.isp_abort_needed = TRUE;
                break;
            case MBA_RSP_TRANSFER_ERR:  /* Response Transfer Error */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: ISP Response Transfer Error\n\r");
#endif
                printk(KERN_WARNING "qla1280: ISP Response Transfer Error\n");
                ha->flags.isp_abort_needed = TRUE;
                break;
            case MBA_WAKEUP_THRES:      /* Request Queue Wake-up */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: asynchronous WAKEUP_THRES\n\r");
#endif
                break;
            case MBA_TIMEOUT_RESET:     /* Execution Timeout Reset */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print("qla1280_isr: asynchronous TIMEOUT_RESET\n\r");
#endif
                break;
            case MBA_DEVICE_RESET:         /* Bus Device Reset */
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print(
                        "qla1280_isr: asynchronous BUS_DEVICE_RESET\n\r");
#endif
                ha->flags.reset_marker = TRUE;
                index = mailbox[6] & BIT_0;
                ha->bus_settings[index].reset_marker = TRUE;
                break;
            case MBA_BUS_MODE_CHANGE:
#ifdef QL_DEBUG_LEVEL_2
                qla1280_print(
                        "qla1280_isr: asynchronous BUS_MODE_CHANGE\n\r");
#endif
                break;
            default:
              //DEBUG(qla1280_print("qla1280_isr: default case of switch MB \n\r");)
                if (mailbox[0] < MBA_ASYNC_EVENT)
                {
                        wptr = &mailbox[0];
                        ha->mailbox_out[0] = *wptr++;
                        ha->mailbox_out[1] = *wptr++;
                        ha->mailbox_out[2] = *wptr++;
                        ha->mailbox_out[3] = *wptr++;
                        ha->mailbox_out[4] = *wptr++;
                        ha->mailbox_out[5] = *wptr++;
                        ha->mailbox_out[6] = *wptr++;
                        ha->mailbox_out[7] = *wptr;
                        ha->flags.mbox_int = TRUE;
                }
                break;
        }
    }
    else {
        WRT_REG_WORD(&reg->host_cmd, HC_CLR_RISC_INT);
    }
    /*
    * Response ring
    */
    if (ha->flags.online && !ha->flags.mbox_busy)
    {
        if (mailbox[5] < RESPONSE_ENTRY_CNT)
        {
            while (ha->rsp_ring_index != mailbox[5])
            {
                pkt = ha->response_ring_ptr;

#ifdef QL_DEBUG_LEVEL_5
                qla1280_print("qla1280_isr: ha->rsp_ring_index = ");
                qla1280_output_number((uint32_t)ha->rsp_ring_index, 16);
                qla1280_print(" mailbox[5] = ");
                qla1280_output_number((uint32_t)mailbox[5], 16);
                qla1280_print("\n\rqla1280_isr: response packet data\n\r");
                qla1280_dump_buffer((char *)pkt, RESPONSE_ENTRY_SIZE);
#endif

#if defined(QL_DEBUG_LEVEL_2) && !defined(QL_DEBUG_LEVEL_5)
                if (pkt->entry_type == STATUS_TYPE)
                {
                    if ((uint8_t)(pkt->scsi_status) || pkt->comp_status ||
                        pkt->entry_status)
                    {
                        DEBUG(qla1280_print("qla1280_isr: ha->rsp_ring_index = ");)
                        DEBUG(qla1280_output_number((uint32_t)ha->rsp_ring_index,
                                16);)
                        DEBUG(qla1280_print(" mailbox[5] = ");)
                        DEBUG(qla1280_output_number((uint32_t)mailbox[5], 16);)
                        DEBUG(qla1280_print( "\n\r comp_status = ");)
                        DEBUG(qla1280_output_number((uint32_t)pkt->comp_status,16);)
                        DEBUG(qla1280_print( ", ");)
                        DEBUG(qla1280_print( " scsi_status = ");)
                        DEBUG(qla1280_output_number((uint32_t)pkt->scsi_status,16);)
                        DEBUG(qla1280_print( "\n\r");)
                        /* qla1280_print(
                        "\n\rqla1280_isr: response packet data\n\r");
                        qla1280_dump_buffer((char *)pkt,
                        RESPONSE_ENTRY_SIZE); */
                    }
                }
                else
                {
                    qla1280_print("qla1280_isr: ha->rsp_ring_index = ");
                    qla1280_output_number((uint32_t)ha->rsp_ring_index, 16);
                    qla1280_print(" mailbox[5] = ");
                    qla1280_output_number((uint32_t)mailbox[5], 16);
                    qla1280_print(
                            "\n\rqla1280_isr: response packet data\n\r");
                    qla1280_dump_buffer((char *)pkt, RESPONSE_ENTRY_SIZE);
                }
#endif
                if (pkt->entry_type == STATUS_TYPE || pkt->entry_status)
                {
                    if (pkt->entry_type == STATUS_TYPE)
                        qla1280_status_entry(ha, (sts_entry_t *)pkt,
                                done_q_first, done_q_last);
                    else
                        qla1280_error_entry(ha, pkt,
                                done_q_first, done_q_last);


                    /* Adjust ring index. */
                    ha->rsp_ring_index++;
                    if (ha->rsp_ring_index == RESPONSE_ENTRY_CNT)
                    {
                        ha->rsp_ring_index = 0;
                        ha->response_ring_ptr = ha->response_ring;
                    }
                    else
                        ha->response_ring_ptr++;
                    WRT_REG_WORD(&reg->mailbox5, ha->rsp_ring_index);
                }
#if QLA1280_TARGET_MODE_SUPPORT
                else
                {
                    pkt = &response_entry;

                    /* Copy packet. */
                    dptr1 = (uint32_t *)ha->response_ring_ptr;
                    dptr2 = (uint32_t *)pkt;
                    for (index = 0; index < RESPONSE_ENTRY_SIZE/4; index++)
                        *dptr2++ = *dptr1++;

                    /* Adjust ring index. */
                    ha->rsp_ring_index++;
                    if (ha->rsp_ring_index == RESPONSE_ENTRY_CNT)
                    {
                        ha->rsp_ring_index = 0;
                        ha->response_ring_ptr = ha->response_ring;
                    }
                    else
                        ha->response_ring_ptr++;
                    WRT_REG_WORD(&reg->mailbox5, ha->rsp_ring_index);

                    /* Release interrupt specific lock */
                    QLA1280_INTR_UNLOCK(ha);

                    switch (pkt->entry_type)
                    {
                        case ACCEPT_TGT_IO_TYPE:
                            qla1280_atio_entry(ha, (atio_entry_t *)pkt);
                            break;
                        case IMMED_NOTIFY_TYPE:
                            qla1280_notify_entry(ha, (notify_entry_t *)pkt);
                            break;
                        case CTIO_RET_TYPE:
                            qla1280_accept_io(ha, (ctio_ret_entry_t *)pkt);
                            break;
                        default:
                            break;
                    }

                    /* Acquire interrupt specific lock */
                    QLA1280_INTR_LOCK(ha);
                }
#endif
            }
        }
        else
        {
            ha->flags.isp_abort_needed = TRUE;
#ifdef QL_DEBUG_LEVEL_2
            qla1280_print("qla1280_isr: Response pointer Error\n");
#endif
        }
    }

    LEAVE("qla1280_isr");
}

/*
 *  qla1280_rst_aen
 *      Processes asynchronous reset.
 *
 * Input:
 *      ha  = adapter block pointer.
 */
static void
qla1280_rst_aen(scsi_qla_host_t *ha)
{
#if QL1280_TARGET_MODE_SUPPORT
    notify_entry_t  nentry;
#endif
    uint8_t         b;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_rst_aen");
#endif

    if (ha->flags.online && !ha->flags.reset_active &&
        !ha->flags.abort_isp_active)
    {
        ha->flags.reset_active = TRUE;
        while (ha->flags.reset_marker)
        {
            /* Issue marker command. */
            ha->flags.reset_marker = FALSE;
            for (b = 0; b < ha->ports && !ha->flags.reset_marker; b++)
            {
                if (ha->bus_settings[b].reset_marker)
                {
                    ha->bus_settings[b].reset_marker = FALSE;
                    qla1280_marker(ha, b, 0, 0, MK_SYNC_ALL);

                    if (!ha->flags.reset_marker)
                    {
#if QL1280_TARGET_MODE_SUPPORT
                        /* Issue notify acknowledgement command. */
                        memset(&nentry, 0, sizeof(notify_entry_t));

                        nentry.initiator_id = nentry.target_id = b ?
                                ha->bus_settings[b].id | BIT_7 :
                        ha->bus_settings[b].id;
                        qla1280_notify_entry(ha, &nentry);
#endif

                        /* Asynchronous event notification */
                    }
                }
            }
        }
    }

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_rst_aen");
#endif
}

#if QL1280_TARGET_MODE_SUPPORT
/*
 *  qla1280_atio_entry
 *      Processes received ISP accept target I/O entry.
 *
 * Input:
 *      ha  = adapter block pointer.
 *      pkt = entry pointer.
 */
static void
qla1280_atio_entry(scsi_qla_host_t *ha, atio_entry_t *pkt)
{
    uint64_t    *a64;
    uint64_t    *end_a64;
    paddr32_t   phy_addr[2];
    paddr32_t   end_addr[2];
    uint32_t    len;
    uint32_t    offset;
    uint8_t     t;
    uint8_t     *sense_ptr;

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_atio_entry: entered\n\r");
#endif

    t = pkt->initiator_id;
    sense_ptr = ha->tsense + t * TARGET_SENSE_SIZE;
    a64 = (uint64_t *)&phy_addr[0];
    end_a64 = (uint64_t *)&end_addr[0];

    switch (pkt->status & ~BIT_7)
    {
        case 7:                         /* Path invalid */
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
            qla1280_print("qla1280_atio_entry: Path invalid\n\r");
#endif
            break;
        case 0x14:                  /* Target Bus Phase Sequence Failure */
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
            qla1280_print(
                    "qla1280_atio_entry: Target Bus Phase Sequence Failure\n\r");
#endif
            if (pkt->status & BIT_7)
            {
                memcpy(sense_ptr, &pkt->sense_data, TARGET_SENSE_SIZE);
            }
            else
            {
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    *sense_ptr = 0x70;
                    *(sense_ptr+2) = SD_HARDERR;
                    *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                    *(sense_ptr+12) = SC_SELFAIL;
            }
            pkt->scsi_status = S_CKCON;
            pkt->option_flags |= (uint32_t)OF_SSTS | (uint32_t)OF_NO_DATA;

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
            if (ha->flags.enable_64bit_addressing)
                qla1280_64bit_continue_io(ha, pkt, 0, 0);
            else
#endif
                qla1280_32bit_continue_io(ha, pkt, 0, 0);
            break;
        case 0x16:                  /* Requested Capability Not Available */
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
            qla1280_print(
                    "qla1280_atio_entry: Target Bus Phase Sequence Failure\n\r");
#endif
            break;
        case 0x17:                  /* Bus Device Reset Message Received */
#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
            qla1280_print(
                    "qla1280_atio_entry: Target Bus Phase Sequence Failure\n\r");
#endif
            break;
        case 0x3D:                  /* CDB Received */

            /* Check for invalid LUN */
            if (pkt->lun && pkt->cdb[0] != SS_INQUIR &&
                pkt->cdb[0] != SS_REQSEN)
                pkt->cdb[0] = SS_TEST;

            switch (pkt->cdb[0])
            {
                case SS_TEST:
#ifdef QL_DEBUG_LEVEL_3
                    qla1280_print("qla1280_atio_entry: SS_TEST\n\r");
#endif
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    len = 0;
                    if (pkt->lun == 0)
                        pkt->scsi_status = S_GOOD;
                    else
                    {
                        *sense_ptr = 0x70;
                        *(sense_ptr+2) = SD_ILLREQ;
                        *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                        *(sense_ptr+12) = SC_INVLUN;
                        pkt->scsi_status = S_CKCON;
                    }

                    pkt->option_flags |= (uint32_t)OF_SSTS |
                            (uint32_t)OF_NO_DATA;
                    break;
                case SS_REQSEN:
#ifdef QL_DEBUG_LEVEL_3
                    qla1280_print("qla1280_atio_entry: SS_REQSEN\n\r");
#endif
                    phy_addr[0] = ha->tsense_dma;
                    phy_addr[1] = 0;
                    *a64 += t * TARGET_SENSE_SIZE;
                    if (pkt->cdb[4] > TARGET_SENSE_SIZE)
                        len = TARGET_SENSE_SIZE;
                    else
                        len = pkt->cdb[4];
                    pkt->scsi_status = S_GOOD;
                    pkt->option_flags |= (uint32_t)OF_SSTS |
                            (uint32_t)OF_DATA_IN;
                    break;
                case SS_INQUIR:
#ifdef QL_DEBUG_LEVEL_3
                    qla1280_print("qla1280_atio_entry: SS_INQUIR\n\r");
#endif
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    phy_addr[0] = ha->tbuf_dma;
                    phy_addr[1] = 0;
                    *a64 += TARGET_INQ_OFFSET;

                    if (pkt->lun == 0)
                    {
                            ha->tbuf->inq.id_type = ID_PROCESOR;
                            ha->tbuf->inq.id_pqual = ID_QOK;
                    }
                    else
                    {
                            ha->tbuf->inq.id_type = ID_NODEV;
                            ha->tbuf->inq.id_pqual = ID_QNOLU;
                    }

                    if (pkt->cdb[4] > sizeof(struct ident))
                        len = sizeof(struct ident);
                    else
                        len = pkt->cdb[4];
                    pkt->scsi_status = S_GOOD;
                    pkt->option_flags |= (uint32_t)OF_SSTS |
                            (uint32_t)OF_DATA_IN;
                    break;
                case SM_WRDB:
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    offset = pkt->cdb[5];
                    offset |= pkt->cdb[4] << 8;
                    offset |= pkt->cdb[3] << 16;
                    len = pkt->cdb[8];
                    len |= pkt->cdb[7] << 8;
                    len |= pkt->cdb[6] << 16;
                    end_addr[0] = phy_addr[0] = ha->tbuf_dma;
                    end_addr[1] = phy_addr[1] = 0;
                    *end_a64 += TARGET_DATA_OFFSET + TARGET_DATA_SIZE;
                    switch (pkt->cdb[1] & 7)
                    {
                        case RW_BUF_HDATA:
#ifdef QL_DEBUG_LEVEL_3
                            qla1280_print("qla1280_atio_entry: SM_WRDB, RW_BUF_HDATA\n\r");
#endif
                            if (len > TARGET_DATA_SIZE + 4)
                            {
#ifdef QL_DEBUG_LEVEL_2
                                qla1280_print("qla1280_atio_entry: SM_WRDB, length > buffer size\n\r");
#endif
                                *sense_ptr = 0x70;
                                *(sense_ptr+2) = SD_ILLREQ;
                                *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                                *(sense_ptr+12) = SC_ILLCDB;
                                pkt->scsi_status = S_CKCON;
                                pkt->option_flags |= (uint32_t)OF_SSTS |
                                        (uint32_t)OF_NO_DATA;
                                len = 0;
                            }
                            else if (len)
                            {
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_DATA_OUT;
#ifdef QL_DEBUG_LEVEL_3
                                    qla1280_print("qla1280_atio_entry: Issuing SDI_TARMOD_WRCOMP\n\r");
#endif
                                    sdi_xaen(SDI_TARMOD_WRCOMP, ha->cntlr,
                                            pkt->target_id, pkt->lun, 0, offset);
                            }
                            else
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_WRDB, zero length\n\r");
#endif
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }

                            break;
                        case RW_BUF_DATA:
#ifdef QL_DEBUG_LEVEL_3
                            qla1280_print("qla1280_atio_entry: SM_WRDB, RW_BUF_DATA\n\r");
#endif
                            *a64 += offset + TARGET_DATA_OFFSET;
                            if (pkt->cdb[2] != 0 || *a64 >= *end_a64 ||
                                *a64 + len > *end_a64)
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_WRDB, RW_BUF_DATA BAD\n\r");
                                    qla1280_print("buf_id=");
                                    qla1280_output_number((uint32_t)pkt->cdb[2], 16);
                                    qla1280_print(", offset=");
                                    qla1280_output_number((uint32_t)offset, 16);
                                    qla1280_print(", length=");
                                    qla1280_output_number((uint32_t)len, 16);
                                    qla1280_print("\n\r");
#endif
                                    *sense_ptr = 0x70;
                                    *(sense_ptr+2) = SD_ILLREQ;
                                    *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                                    *(sense_ptr+12) = SC_ILLCDB;
                                    len = 0;
                                    pkt->scsi_status = S_CKCON;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }
                            else if (len)
                            {
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_DATA_OUT;
#ifdef QL_DEBUG_LEVEL_3
                                    qla1280_print("qla1280_atio_entry: Issuing SDI_TARMOD_WRCOMP\n\r");
#endif
                                    sdi_xaen(SDI_TARMOD_WRCOMP, ha->cntlr,
                                            pkt->target_id, pkt->lun, 0, offset);
                            }
                            else
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_WRDB, zero length\n\r");
#endif
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }
                            break;
                        default:
#ifdef QL_DEBUG_LEVEL_2
                            qla1280_print("qla1280_atio_entry: SM_WRDB unknown mode\n\r");
#endif
                            *sense_ptr = 0x70;
                            *(sense_ptr+2) = SD_ILLREQ;
                            *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                            *(sense_ptr+12) = SC_ILLCDB;
                            len = 0;
                            pkt->scsi_status = S_CKCON;
                            pkt->option_flags |= (uint32_t)OF_SSTS |
                                    (uint32_t)OF_NO_DATA;
                            break;
                    }
                    break;
                case SM_RDDB:
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    offset = pkt->cdb[5];
                    offset |= pkt->cdb[4] << 8;
                    offset |= pkt->cdb[3] << 16;
                    len = pkt->cdb[8];
                    len |= pkt->cdb[7] << 8;
                    len |= pkt->cdb[6] << 16;
                    end_addr[0] = phy_addr[0] = ha->tbuf_dma;
                    end_addr[1] = phy_addr[1] = 0;
                    *end_a64 += TARGET_DATA_OFFSET + TARGET_DATA_SIZE;
                    switch (pkt->cdb[1] & 7)
                    {
                        case RW_BUF_HDATA:
#ifdef QL_DEBUG_LEVEL_3
                            qla1280_print("qla1280_atio_entry: SM_RDDB, RW_BUF_HDATA\n\r");
#endif
                            if (len)
                            {
                                ha->tbuf->hdr[0] = 0;
                                ha->tbuf->hdr[1] =
                                        (uint8_t)(TARGET_DATA_SIZE >> 16);
                                ha->tbuf->hdr[2] =
                                        (uint8_t)(TARGET_DATA_SIZE >> 8);
                                ha->tbuf->hdr[3] = (uint8_t)TARGET_DATA_SIZE;
                                if (len > TARGET_DATA_SIZE + 4)
                                    len = TARGET_DATA_SIZE + 4;
                                pkt->scsi_status = S_GOOD;
                                pkt->option_flags |= (uint32_t)OF_SSTS |
                                        (uint32_t)OF_DATA_IN;
                            }
                            else
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_RDDB, zero length\n\r");
#endif
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }
                            break;
                        case RW_BUF_DATA:
#ifdef QL_DEBUG_LEVEL_3
                            qla1280_print("qla1280_atio_entry: SM_RDDB, RW_BUF_DATA\n\r");
#endif
                            *a64 += offset + TARGET_DATA_OFFSET;
                            if (pkt->cdb[2] != 0 || *a64 >= *end_a64)
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_RDDB, RW_BUF_DATA BAD\n\r");
                                    qla1280_print("buf_id=");
                                    qla1280_output_number((uint32_t)pkt->cdb[2], 16);
                                    qla1280_print(", offset=");
                                    qla1280_output_number((uint32_t)offset, 16);
                                    qla1280_print("\n\r");
#endif
                                    *sense_ptr = 0x70;
                                    *(sense_ptr+2) = SD_ILLREQ;
                                    *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                                    *(sense_ptr+12) = SC_ILLCDB;
                                    len = 0;
                                    pkt->scsi_status = S_CKCON;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }
                            else
                            {
                                    if (*a64 + len > *end_a64)
                                        len = *end_a64 - *a64;
                                    if (len)
                                    {
                                        pkt->scsi_status = S_GOOD;
                                        pkt->option_flags |= (uint32_t)OF_SSTS |
                                                (uint32_t)OF_DATA_IN;
                                    }
                                    else
                                    {
#ifdef QL_DEBUG_LEVEL_2
                                            qla1280_print("qla1280_atio_entry: SM_RDDB, zero length\n\r");
#endif
                                            pkt->scsi_status = S_GOOD;
                                            pkt->option_flags |= (uint32_t)OF_SSTS |
                                                    (uint32_t)OF_NO_DATA;
                                    }
                            }
                            break;
                        case RW_BUF_DESC:
#ifdef QL_DEBUG_LEVEL_3
                            qla1280_print("qla1280_atio_entry: SM_RDDB, RW_BUF_DESC\n\r");
#endif
                            if (len)
                            {
                                    if (len > 4)
                                        len = 4;

                                    ha->tbuf->hdr[0] = 0;
                                    if (pkt->cdb[2] != 0)
                                    {
                                        ha->tbuf->hdr[1] = 0;
                                        ha->tbuf->hdr[2] = 0;
                                        ha->tbuf->hdr[3] = 0;
                                    }
                                    else
                                    {
                                            ha->tbuf->hdr[1] =
                                                    (uint8_t)(TARGET_DATA_SIZE >> 16);
                                            ha->tbuf->hdr[2] =
                                                    (uint8_t)(TARGET_DATA_SIZE >> 8);
                                            ha->tbuf->hdr[3] =
                                                    (uint8_t)TARGET_DATA_SIZE;
                                    }
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_DATA_IN;
                            }
                            else
                            {
#ifdef QL_DEBUG_LEVEL_2
                                    qla1280_print("qla1280_atio_entry: SM_RDDB, zero length\n\r");
#endif
                                    pkt->scsi_status = S_GOOD;
                                    pkt->option_flags |= (uint32_t)OF_SSTS |
                                            (uint32_t)OF_NO_DATA;
                            }
                            break;
                        default:
#ifdef QL_DEBUG_LEVEL_2
                            qla1280_print("qla1280_atio_entry: SM_RDDB unknown mode\n\r");
#endif
                            *sense_ptr = 0x70;
                            *(sense_ptr+2) = SD_ILLREQ;
                            *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                            *(sense_ptr+12) = SC_ILLCDB;
                            len = 0;
                            pkt->scsi_status = S_CKCON;
                            pkt->option_flags |= (uint32_t)OF_SSTS |
                                    (uint32_t)OF_NO_DATA;
                            break;
                    }
                    break;
                default:
#ifdef QL_DEBUG_LEVEL_2
                    qla1280_print("qla1280_atio_entry: Unknown SCSI command\n\r");
                    qla1280_dump_buffer(&pkt->cdb[0], pkt->cdb_len);
#endif
                    memset(sense_ptr, 0, TARGET_SENSE_SIZE);
                    *sense_ptr = 0x70;
                    *(sense_ptr+2) = SD_ILLREQ;
                    *(sense_ptr+7) = TARGET_SENSE_SIZE-8;
                    *(sense_ptr+12) = SC_INVOPCODE;
                    len = 0;
                    pkt->scsi_status = S_CKCON;
                    pkt->option_flags |= (uint32_t)OF_SSTS |
                            (uint32_t)OF_NO_DATA;
                    break;
            }
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,18)
            if (ha->flags.enable_64bit_addressing)
               qla1280_64bit_continue_io(ha, pkt, len, (paddr32_t *)&phy_addr);
            else
#endif
               qla1280_32bit_continue_io(ha, pkt, len, (paddr32_t *)&phy_addr);
            break;
        default:
            break;
    }

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_atio_entry: exiting normally\n\r");
#endif
}

/*
 *  qla1280_notify_entry
 *      Processes received ISP immediate notify entry.
 *
 * Input:
 *      ha  = adapter block pointer.
 *      pkt = entry pointer.
 */
static void
qla1280_notify_entry(scsi_qla_host_t *ha, notify_entry_t *pkt)
{
#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_notify_entry: entered\n\r");
#endif

    /* Acknowledge immediate notify */
    qla1280_notify_ack(ha, pkt);

    /* Issue notify entry to increment resource count */
    qla1280_immed_notify(ha, pkt);

#ifdef QL_DEBUG_LEVEL_3
    qla1280_print("qla1280_notify_entry: exiting normally\n\r");
#endif
}

#endif  /* QLA1280_TARGET_MODE_SUPPORT */
/*
 *  qla1280_status_entry
 *      Processes received ISP status entry.
 *
 * Input:
 *      ha           = adapter block pointer.
 *      pkt          = entry pointer.
 *      done_q_first = done queue first pointer.
 *      done_q_last  = done queue last pointer.
 */
static void
qla1280_status_entry(scsi_qla_host_t *ha, sts_entry_t *pkt, srb_t **done_q_first,
                     srb_t **done_q_last)
{
    uint32_t        b, t, l;
    uint8_t         sense_sz = 0;
    srb_t           *sp;
    scsi_lu_t       *q;
    Scsi_Cmnd       *cp;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_status_entry");
#endif

    /* Validate handle. */
    if (pkt->handle < MAX_OUTSTANDING_COMMANDS)
        sp = ha->outstanding_cmds[pkt->handle];
    else
        sp = 0;

    if (sp)
    {
        /* Free outstanding command slot. */
        ha->outstanding_cmds[pkt->handle] = 0;

        cp = sp->cmd;

        /* Generate LU queue on cntrl, target, LUN */
        b = SCSI_BUS_32(cp);
        t = SCSI_TCN_32(cp);
        l = SCSI_LUN_32(cp);
        q = LU_Q(ha, b, t, l);

        if( pkt->comp_status || pkt->scsi_status )
        {
            DEBUG(qla1280_print( "scsi: comp_status = ");)
            DEBUG(qla1280_output_number((uint32_t)pkt->comp_status,16);)
            DEBUG(qla1280_print( ", ");)
            DEBUG(qla1280_print( " scsi_status = ");)
            DEBUG(qla1280_output_number((uint32_t)pkt->scsi_status,16);)
            DEBUG(qla1280_print( "\n\r");)
            DEBUG(qla1280_print(", handle = ");)
            DEBUG(qla1280_output_number((uint32_t)pkt->handle, 16);)
            DEBUG(qla1280_print("\n\r");)
        }

        /* Target busy */
        if ( pkt->scsi_status & SS_BUSY_CONDITION &&
            pkt->scsi_status != SS_RESERVE_CONFLICT   )
        {
            CMD_RESULT(cp) = (int) (DID_BUS_BUSY << 16) |
                    (pkt->scsi_status & 0xff);
        }
        else
        {

            /* Save ISP completion status */
            CMD_RESULT(cp) = qla1280_return_status( pkt, cp );

            if (pkt->scsi_status & SS_CHECK_CONDITION)
            {
                memset(cp->sense_buffer, 0, CMD_SNSLEN(cp));
                if (pkt->comp_status != CS_ARS_FAILED)
                {
                    if ( pkt->req_sense_length < CMD_SNSLEN(cp)  )
                        sense_sz = pkt->req_sense_length;
                    else
                        sense_sz = CMD_SNSLEN(cp) - 1;

                    memcpy(cp->sense_buffer, &pkt->req_sense_data, sense_sz);

                }
#ifdef QL_DEBUG_LEVEL_2
                DEBUG(qla1280_print(
                        "qla1280_status_entry: Check condition Sense data, b");)
                DEBUG(qla1280_output_number((uint32_t)b, 10);)
                DEBUG(qla1280_print("t");)
                DEBUG(qla1280_output_number((uint32_t)t, 10);)
                DEBUG(qla1280_print("d");)
                DEBUG(qla1280_output_number((uint32_t)l, 10);)
                DEBUG(qla1280_print("\n\r");)
                DEBUG(if (sense_sz))
                    DEBUG(qla1280_dump_buffer((char *)cp->sense_buffer, sense_sz);)
#endif
            }
        }
        /* Place command on done queue. */
        qla1280_done_q_put(sp, done_q_first, done_q_last);
    }
    else
    {
#ifdef QL_DEBUG_LEVEL_2
        qla1280_print("qla1280_status_entry: ISP Invalid handle\n\r");
#endif
        printk(KERN_WARNING "qla1280: Status Entry invalid handle\n");
        ha->flags.isp_abort_needed = TRUE;
    }
#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_status_entry");
#endif
}

/*
 *  qla1280_error_entry
 *      Processes error entry.
 *
 * Input:
 *      ha           = adapter block pointer.
 *      pkt          = entry pointer.
 *      done_q_first = done queue first pointer.
 *      done_q_last  = done queue last pointer.
 */
static void
qla1280_error_entry(scsi_qla_host_t *ha, response_t *pkt, srb_t **done_q_first,
                    srb_t **done_q_last)
{
    srb_t   *sp;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_error_entry");
#endif

#ifdef QL_DEBUG_LEVEL_2
    if (pkt->entry_status & BIT_3)
        qla1280_print("qla1280_error_entry: BAD PAYLOAD flag error\n\r");
    else if (pkt->entry_status & BIT_2)
        qla1280_print("qla1280_error_entry: BAD HEADER flag error\n\r");
    else if (pkt->entry_status & BIT_1)
        qla1280_print("qla1280_error_entry: FULL flag error\n\r");
    else
        qla1280_print("qla1280_error_entry: UNKNOWN flag error\n\r");
#endif

    /* Validate handle. */
    if (pkt->handle < MAX_OUTSTANDING_COMMANDS)
        sp = ha->outstanding_cmds[pkt->handle];
    else
        sp = 0;

    if (sp)
    {
        /* Free outstanding command slot. */
        ha->outstanding_cmds[pkt->handle] = 0;

        /* Bad payload or header */
        if (pkt->entry_status & (BIT_3 + BIT_2))
        {
            /* Bad payload or header, set error status. */
            /* CMD_RESULT(sp->cmd) = CS_BAD_PAYLOAD; */
            CMD_RESULT(sp->cmd) = (int) DID_ERROR << 16;
        }
        else if (pkt->entry_status & BIT_1 ) /* FULL flag */
        {
            CMD_RESULT(sp->cmd) = (int) DID_BUS_BUSY << 16;
        }
        else
        {
            /* Set error status. */
            CMD_RESULT(sp->cmd) =(int)  DID_ERROR << 16;
        }
        /* Place command on done queue. */
        qla1280_done_q_put(sp, done_q_first, done_q_last);
    }
#ifdef QLA_64BIT_PTR
    else if (pkt->entry_type == COMMAND_A64_TYPE)
    {
#ifdef QL_DEBUG_LEVEL_2
        qla1280_print("qla1280_error_entry: ISP Invalid handle\n\r");
#endif
        printk(KERN_WARNING "!qla1280: Error Entry invalid handle");
        ha->flags.isp_abort_needed = TRUE;
    }
#endif

#ifdef QL_DEBUG_LEVEL_3
    LEAVE("qla1280_error_entry");
#endif
}

/*
 *  qla1280_abort_isp
 *      Resets ISP and aborts all outstanding commands.
 *
 * Input:
 *      ha           = adapter block pointer.
 *
 * Returns:
 *      0 = success
 */
static int
qla1280_abort_isp(scsi_qla_host_t *ha)
{
    device_reg_t    *reg = ha->iobase;
    srb_t           *sp;
    scsi_lu_t       *q;
    int		    status = 0;
    int		    cnt;
    uint32_t        b, t, l;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_abort_isp");
#endif

    ha->flags.isp_abort_needed = FALSE;
    if (!ha->flags.abort_isp_active && ha->flags.online)
    {
        ha->flags.abort_isp_active = TRUE;

        /* Disable ISP interrupts. */
        WRT_REG_WORD(&reg->ictrl, 0);

        /* Dequeue all commands in outstanding command list. */
        for (cnt = 1; cnt < MAX_OUTSTANDING_COMMANDS; cnt++)
        {
            sp = ha->outstanding_cmds[cnt];
            if (sp)
            {
                ha->outstanding_cmds[cnt] = 0;

                /* Generate LU queue on controller, target, LUN */
                b = SCSI_BUS_32(sp->cmd);
                t = SCSI_TCN_32(sp->cmd);
                l = SCSI_LUN_32(sp->cmd);

                q = (scsi_lu_t       *)LU_Q(ha, b, t, l);

                /* Reset outstanding command count. */
                q->q_outcnt = 0;
                q->q_flag &= ~QLA1280_QBUSY;
                q->q_flag =  0;

                /* Adjust watchdog timer for command. */
                /* if (sp->flags & SRB_WATCHDOG)
                sp->timeout += 2; */

                /* Place request back on top of device queue. */
                sp->flags = 0;
                qla1280_putq_t(q, sp);
            }
        }

        /* If firmware needs to be loaded */
        if (qla1280_isp_firmware(ha))
        {
            if (!(status = qla1280_chip_diag(ha)))
                status = qla1280_setup_chip(ha); 
        }

        if (!status)
        {
            /* Setup adapter based on NVRAM parameters. */
            qla1280_nvram_config(ha); 

            if (!(status = qla1280_init_rings(ha)))
            {
                /* Issue SCSI reset. */
                for (b = 0; b < ha->ports; b++)
                {
                    qla1280_bus_reset(ha, b);
                }
                do
                {
                    /* Issue marker command. */
                    ha->flags.reset_marker = FALSE;
                    for (b = 0; b < ha->ports; b++)
                    {
                        ha->bus_settings[b].reset_marker = FALSE;
                        qla1280_marker(ha, b, 0, 0, MK_SYNC_ALL);
                    }
                }while (ha->flags.reset_marker);

                /* Enable host adapter target mode. */
                for (b = 0; b < ha->ports; b++)
                {
                    if (!(status = qla1280_enable_tgt(ha, b)))
                    {
                        for (cnt = 0; cnt < MAX_LUNS; cnt++)
                        {
                            /* qla1280_enable_lun(ha, b, cnt); */
                            qla1280_poll(ha);
                        }
                    }
                    else
                        break;
                }

                if (!status)
                { 
                    /* Enable ISP interrupts. */
                    WRT_REG_WORD(&reg->ictrl, (ISP_EN_INT | ISP_EN_RISC));
                    ha->flags.abort_isp_active = FALSE;
                    /* Restart queues that may have been stopped. */
                    qla1280_restart_queues(ha);
                }
            }
        }
    }

    if (status)
    {
        printk(KERN_WARNING
                "qla1280: ISP error recovery failed, board disabled");
        qla1280_reset_adapter(ha);
        qla1280_abort_queues(ha);

#if defined(QL_DEBUG_LEVEL_2) || defined(QL_DEBUG_LEVEL_3)
        qla1280_print("qla1280_abort_isp: **** FAILED ****\n\r");
#endif
    }
#ifdef QL_DEBUG_LEVEL_3
    else
        LEAVE("qla1280_abort_isp");
#endif

    return(status);
}

/*
 *  qla1280_restart_queues
 *      Restart all device queues.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void
qla1280_restart_queues(scsi_qla_host_t *ha)
{
    scsi_lu_t *q;
    uint32_t  b, t, l;

#ifdef QL_DEBUG_LEVEL_3
    ENTER("qla1280_restart_queues");
#endif

    for (b = 0; b < ha->ports; b++)
        for (t = 0; t < MAX_TARGETS; t++)
            for (l = 0; l < MAX_LUNS; l++)
            {
                q = (scsi_lu_t *) LU_Q(ha, b, t, l);
                if (q != NULL)
                {
                    /* Acquire LU queue specific lock */
                    QLA1280_SCSILU_LOCK(q);

                    if (q->q_first)
                        qla1280_next(ha, q, b);
                    else
                        /* Release LU queue specific lock */
                        QLA1280_SCSILU_UNLOCK(q);
                }
            }
#ifdef QL_DEBUG_LEVEL_3
            qla1280_print("qla1280_restart_queues: exiting normally\n\r");
#endif
}

/*
 *  qla1280_abort_queue_single
 *      Abort all commands on a device queues.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void qla1280_abort_queue_single(scsi_qla_host_t *ha,uint32_t b,uint32_t t,uint32_t l,uint32_t stat)
{
    scsi_lu_t *q;
    srb_t     *sp, *sp_next; 

    ENTER("qla1280_abort_queue_single");
    q = (scsi_lu_t * )LU_Q(ha, b, t, l);
    if (q != NULL)
    {
        /* Acquire LU queue specific lock */
        QLA1280_SCSILU_LOCK(q);

        sp = q->q_first;
        q->q_first = q->q_last = NULL;

        QLA1280_SCSILU_UNLOCK(q);

        while (sp)
        {
            sp_next = sp->s_next;
            CMD_RESULT(sp->cmd) = stat;
            qla1280_done_q_put(sp, (srb_t **)&ha->done_q_first, (srb_t **)&ha->done_q_last);
            sp = sp_next;
        }
    }
    LEAVE("qla1280_abort_queue_single");
}

/*
 *  qla1280_abort_queues
 *      Abort all commands on device queues.
 *
 * Input:
 *      ha = adapter block pointer.
 */
static void
qla1280_abort_queues(scsi_qla_host_t *ha)
{
    uint32_t  b, t, l;

    ENTER("qla1280_abort_queues");

    for (b = 0; b < ha->ports; b++)
        for (t = 0; t < MAX_TARGETS; t++)
            for (l = 0; l < MAX_LUNS; l++)
                qla1280_abort_queue_single(ha,b,t,l,DID_RESET);

            LEAVE("qla1280_abort_queues");
}

/*
 * qla1280_debounce_register
 *      Debounce register.
 *
 * Input:
 *      port = register address.
 *
 * Returns:
 *      register value.
 */
static u16
qla1280_debounce_register(volatile u16 *addr)
{
    volatile u16 ret;
    volatile u16 ret2;

    do
    {
        ret = RD_REG_WORD(addr);
        ret2 = RD_REG_WORD(addr);
    }while (ret != ret2);

    return(ret);
}

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,4,0)
#ifdef MODULE
Scsi_Host_Template driver_template = QLA1280_LINUX_TEMPLATE;
#include "scsi_module.c"
#endif
#else   /* new kernel scsi initialization scheme */
static  Scsi_Host_Template driver_template = QLA1280_LINUX_TEMPLATE;
#include "scsi_module.c"
#endif

/************************************************************************
 * qla1280_check_for_dead_scsi_bus                                      *
 *                                                                      *
 *    This routine checks for a dead SCSI bus                           *
 ************************************************************************/
#define SET_SXP_BANK            0x0100
#define SCSI_PHASE_INVALID      0x87FF
int qla1280_check_for_dead_scsi_bus(scsi_qla_host_t *ha, srb_t *sp)
{
    uint16_t  config_reg, scsi_control;
    device_reg_t    *reg = ha->iobase;
    uint32_t  b;
    Scsi_Cmnd       *cp;

    /*
     * If SCSI Bus is Dead because of bad termination,
     * we will return a status of Selection timeout.
     */
 
     cp = sp->cmd;
     b = SCSI_BUS_32(cp);
    if (ha->bus_settings[b].scsi_bus_dead)
    {
        WRT_REG_WORD(&reg->host_cmd, HC_PAUSE_RISC);
        config_reg = RD_REG_WORD(&reg->cfg_1);
        WRT_REG_WORD(&reg->cfg_1,SET_SXP_BANK);
        scsi_control = RD_REG_WORD(&reg->scsiControlPins);
        WRT_REG_WORD(&reg->cfg_1,config_reg);
        WRT_REG_WORD(&reg->host_cmd, HC_RELEASE_RISC);

        if (scsi_control == SCSI_PHASE_INVALID)
        {
            CMD_RESULT(cp) = DID_NO_CONNECT << 16;
            CMD_HANDLE(cp) = NULL;
            /* ha->actthreads--; */

            (*(cp)->scsi_done)(cp);

            return(TRUE);   /* bus is dead */
        }
        else
        {
            ha->bus_settings[b].scsi_bus_dead = FALSE;
            ha->bus_settings[b].failed_reset_count= 0; 
        }
    }
    return(FALSE);   /* bus is not dead */
}

static int
qla12160_set_target_parameters(scsi_qla_host_t *ha, uint32_t b,
                               uint32_t t, uint32_t l, nvram160_t *nv)
{
    uint16_t        mb[MAILBOX_REGISTER_COUNT];

    /* Set Target Parameters. */
    mb[0] = MBC_SET_TARGET_PARAMETERS;
    mb[1] = (uint16_t)(b ? t | BIT_7 :t);
    mb[1] <<= 8;
    mb[2] = nv->bus[b].target[t].parameter.c << 8;
    mb[2] |= TP_AUTO_REQUEST_SENSE;
    mb[2] &= ~TP_STOP_QUEUE;
    mb[2] |=  (nv->bus[b].target[t].flags.enable_ppr << 5);
    mb[3] = nv->bus[b].target[t].flags.sync_offset << 8;
    mb[3] |= nv->bus[b].target[t].sync_period;

    mb[6] = nv->bus[b].target[t].flags.ppr_options << 8;
    mb[6] |= nv->bus[b].target[t].flags.ppr_bus_width;
    return( qla1280_mailbox_command(ha, BIT_6|BIT_3|BIT_2|BIT_1|BIT_0, &mb[0]) ) ;
}

static void
qla12160_get_target_parameters(scsi_qla_host_t *ha, uint32_t b, uint32_t t, uint32_t l)
{
    uint16_t        mb[MAILBOX_REGISTER_COUNT];

    mb[0] = MBC_GET_TARGET_PARAMETERS;
    mb[1] = (uint16_t)(b ? t | BIT_7 :t);
    mb[1] <<= 8;
    qla1280_mailbox_command(ha, BIT_6|BIT_3|BIT_2|BIT_1|BIT_0, &mb[0]);
    if( mb[3] != 0 )
    printk(KERN_INFO "scsi(%d:%d:%d:%d): Synchronous tranfer at period %d, offset %d. \n",
        (int)ha->host_no, b, t, l, (mb[3] &0xff), (mb[3] >> 8));

    if ( (mb[2] & BIT_5) &&  ((mb[6] >> 8) & 0xff) >= 2 )
            printk(KERN_INFO "scsi(%d:%d:%d:%d): Dual Transition enabled.\n",
                          (int)ha->host_no, b, t, l);
}


#if DEBUG_QLA1280
/*
 *  Out NULL terminated string to COM port.
 */
static void
qla1280_print(char *s)
{
    if (ql_debug_print)
      printk("%s", s);
}

/*
 *  Output long number to COM port.
 */
static void
qla1280_output_number(u32 n, int base)
{
    char buf[12];
    if (base == 10)
      sprintf(buf, " %i", n);
    else
      sprintf(buf, " 0x%x", n);
    qla1280_print(buf);
}
#endif

#ifdef QL_DEBUG_ROUTINES
/****************************************************************************/
/*                         Driver Debug Functions.                          */
/****************************************************************************/

/*
 *  Get byte from I/O port
 */
static u8
qla1280_getbyte(u8 *port)
{
    u8 ret;

#if MEMORY_MAPPED_IO
    ret = readb(unsigned long)port);
#else
    ret = inb((unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_getbyte: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number((u32)ret, 16);
        qla1280_print("\n\r");
    }

    return(ret);
}

/*
 *  Get word from I/O port
 */
static u16
qla1280_getword(u16 *port)
{
    u16 ret;

#if MEMORY_MAPPED_IO
    ret = readw(unsigned long)port;
#else
    ret = inw((unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_getword: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number((u32)ret, 16);
        qla1280_print("\n\r");
    }

    return(ret);
}

/*
 *  Get double word from I/O port
 */
static u32
qla1280_getdword(u32 *port)
{
    u32 ret;

#if MEMORY_MAPPED_IO
    ret = readl((unsigned long)port);
#else
    ret = inl((unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_getdword: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number(ret, 16);
        qla1280_print("\n\r");
    }

    return(ret);
}

/*
 *  Send byte to I/O port
 */
static void
qla1280_putbyte(u8 *port, u8 data)
{
#if MEMORY_MAPPED_IO
    writeb(data, (unsigned long)port);
#else
    outb(data, (unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_putbyte: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number((u32)data, 16);
        qla1280_print("\n\r");
    }
}

/*
 *  Send word to I/O port
 */
static void
qla1280_putword(u16 *port, u16 data)
{
#if MEMORY_MAPPED_IO
    writew(data, (unsigned long)port);
#else
    outw(data, (unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_putword: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number((u32)data, 16);
        qla1280_print("\n\r");
    }
}

/*
 *  Send double word to I/O port
 */
static void
qla1280_putdword(u32 *port, u32 data)
{
#if MEMORY_MAPPED_IO
    writel(data, (unsigned long)port);
#else
    outl(data, (unsigned long)port);
#endif

    if (ql_debug_print)
    {
        qla1280_print("qla1280_putdword: address = ");
        qla1280_output_number((unsigned long)port, 16);
        qla1280_print(" data = 0x");
        qla1280_output_number(data, 16);
        qla1280_print("\n\r");
    }
}

/*
 * Dummy function to prevent warnings for
 * declared and unused debug functions
 */
void
qla1280_debug(void)
{
    qla1280_getbyte(0);
    qla1280_getword(0);
    qla1280_getdword(0);
    qla1280_putbyte(0, 0);
    qla1280_putword(0, 0);
    qla1280_putdword(0, 0);
}

static void
qla1280_dump_buffer(char *b, int size)
{
    int cnt;
    u8 c;

    if (ql_debug_print)
    {
        qla1280_print(
                " 0   1   2   3   4   5   6   7   8   9   Ah  Bh  Ch  Dh  Eh  Fh\n\r");
        qla1280_print(
                "---------------------------------------------------------------\n\r");

        for (cnt = 0; cnt < size; )
        {
            c = *b++;
            if (c < 16)
                qla1280_print(' ');
            qla1280_output_number((u32)c, 16);
            cnt++;
            if (!(cnt % 16))
                qla1280_print("\n\r");
            else if (c < 10)
                qla1280_print("  ");
            else
                qla1280_print(' ');
        }
        if (cnt % 16)
            qla1280_print("\n\r");
    }
}

/**************************************************************************
 *   ql1280_print_scsi_cmd
 *
 **************************************************************************/
static void qla1280_print_scsi_cmd(Scsi_Cmnd *cmd)
{
    scsi_qla_host_t *ha;
    struct Scsi_Host  *host = cmd->host;
    srb_t           *sp;
   /* struct scatterlist *sg; */

    int i;
    ha = (scsi_qla_host_t *) host->hostdata;

    ql_debug_print = 1;
    sp = (srb_t *) CMD_SP(cmd);
    sprintf(debug_buff,"SCSI Command @= 0x%p, Handle=0x%p\n\r", cmd, CMD_HANDLE(cmd));
    qla1280_print(debug_buff);
    sprintf(debug_buff,"  chan=%d, target = 0x%02x, lun = 0x%02x, cmd_len = 0x%02x\n\r",
            cmd->channel, cmd->target, cmd->lun, cmd->cmd_len);
    qla1280_print(debug_buff);
    qla1280_print(" CDB = ");
    for (i = 0; i < cmd->cmd_len; i++)
    {
        sprintf(debug_buff,"0x%02x ", cmd->cmnd[i]);
        qla1280_print(debug_buff);
    }
    sprintf(debug_buff,"  seg_cnt =%d\n\r",cmd->use_sg);
    qla1280_print(debug_buff);
    sprintf(debug_buff,"  request buffer=0x%p, request buffer len=0x%x\n\r",cmd->request_buffer,cmd->request_bufflen);
    qla1280_print(debug_buff);
    /* if( cmd->use_sg )
    {
       sg = (struct scatterlist *) cmd->request_buffer;
       qla1280_print("  SG buffer: \n\r");
       qla1280_dump_buffer((char *)sg, (cmd->use_sg*sizeof(struct scatterlist)) );
    } */
    sprintf(debug_buff,"  tag=%d, flags=0x%x, transfersize=0x%x \n\r", 
            cmd->tag, cmd->flags,cmd->transfersize );
    qla1280_print(debug_buff);
    sprintf(debug_buff,"  Pid=%d, SP=0x%p\n\r", (int)cmd->pid, CMD_SP(cmd));
    qla1280_print(debug_buff);
    sprintf(debug_buff," underflow size = 0x%x, direction=0x%x, req.cmd=0x%x \n\r", cmd->underflow, sp->dir,cmd->request.cmd);    
    qla1280_print(debug_buff);
}
/**************************************************************************
 *   ql1280_dump_device
 *
 **************************************************************************/
void
ql1280_dump_device(scsi_qla_host_t *ha)
{

    Scsi_Cmnd       *cp;
    srb_t           *sp;
    int i;
    qla1280_print("Outstanding Commands on controller:\n\r");   
    for ( i=0; i < MAX_OUTSTANDING_COMMANDS; i++ )
    {
        if( (sp = ha->outstanding_cmds[i]) == NULL )
            continue;
        if( (cp = sp->cmd) == NULL )
            continue;
        qla1280_print_scsi_cmd(cp);
    }

}
#endif



#if  STOP_ON_ERROR 
/**************************************************************************
 *   ql1280_panic
 *
 **************************************************************************/
static void qla1280_panic(char *cp, struct Scsi_Host *host)
{
    scsi_qla_host_t *ha;
    long  *fp;

    ha = (scsi_qla_host_t *) host->hostdata;
    printk("qla1280 - PANIC:  %s\n",cp);
    printk("Current time=0x%lx\n", jiffies);
    printk("Number of pending commands =0x%lx\n", ha->actthreads);
    printk("Number of SCSI queued commands =0x%lx\n", ha->qthreads);
    printk("Number of free entries = (%d)\n",ha->req_q_cnt);
    printk("Request Queue @ 0x%lx, Response Queue @ 0x%lx\n",
                        ha->request_dma,
                        ha->response_dma);
    printk("Request In Ptr %d\n", ha->req_ring_index );
    fp = (long *) &ha->flags;
    printk("HA flags =0x%lx\n", *fp);
    DEBUG2(ql_debug_print = 1;)
    /* DEBUG2(ql1280_dump_device((scsi_qla_host_t *) host->hostdata)); */
    sti();  
    panic("Ooops");  
    /* cli(); 
    for(;;)
    { 
        barrier();
    sti();  
    }
    */
}
#endif


/**************************************************************************
 *   qla1280_setup
 *
 *   Handle Linux boot parameters. This routine allows for assigning a value
 *   to a parameter with a ':' between the parameter and the value.
 *   ie. qla1280=max_reqs:0x0A,verbose
 **************************************************************************/
void
qla1280_setup(char *s, int *dummy)
{
    char *end, *str, *cp;

    printk("scsi: Processing Option str = %s\n", s);
    end = strchr(s, '\0');
    /* locate command */
    str = s;
    for( cp = s; *cp && cp != end; cp++ ) 
    {
       cp = qla1280_get_token(cp, str);
       printk("scsi: token str = %s\n", str);
       /* if found execute routine */

    }
    
}

static char	*qla1280_get_token(char *cmdline, char *str )
{
    register	char 	*cp = cmdline;

        /* skip preceeding spaces */
        while(strchr(cp,' '))
            ++cp;
        /* symbol starts here */
        str = cp;
        /* skip char if not a space or : */
        while (*cp && !( strchr(cp,' ') || strchr(cp,':'))  )
            cp++;
        *cp = '\0';
        return( cp );
}

/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 2
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -2
 * c-argdecl-indent: 2
 * c-label-offset: -2
 * c-continued-statement-offset: 2
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
