/*
 * iSCSI driver for Linux
 * Copyright (C) 2001 Cisco Systems, Inc.
 * maintained by linux-iscsi@cisco.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * See the file COPYING included with this distribution for more details.
 *
 * $Id: iscsi.c,v 1.9 2001/08/08 19:04:02 smferris Exp $ 
 *
 */

/* there's got to be a better way to wait for child processes created by kernel_thread */
static int errno = 0;
#define __KERNEL_SYSCALLS__

#include <linux/config.h>
#include <linux/version.h>
#include <linux/module.h>

#include <linux/sched.h>
#include <asm/io.h>
#include <asm/byteorder.h>
#include <linux/stddef.h>
#include <linux/version.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/file.h>
#include <linux/kernel.h>
#include <linux/ioport.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/proc_fs.h>
#include <linux/blk.h>
#include <linux/types.h>
#include <linux/stat.h>
#include <linux/slab.h>
#include <linux/config.h>
#include <linux/poll.h>
#include <linux/smp_lock.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/net.h>
#include <net/sock.h>
#include <linux/socket.h>
#include <linux/errno.h>
#include <linux/unistd.h>

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
#include <asm/semaphore.h>
#else
#include <asm/spinlock.h>
#endif
#include <asm/uaccess.h>
#include <scsi/sg.h>
#include <endian.h>

#include "sd.h"
#include "scsi.h"
#include "hosts.h"



/* #define DEBUG_INIT   1 */
/* #define DEBUG_FLOW   1 */
#define DEBUG_ERROR 1
/* #define DEBUG_TRACE  1 */

#define USE_DYNAMIC_DEVICE

#include "iscsi-common.h"
#include "iscsi-protocol.h"
#include "iscsi-ioctl.h"
#include "iscsi-trace.h"
#include "iscsi.h"



MODULE_AUTHOR("Cisco Systems, Inc.");
MODULE_DESCRIPTION("iSCSI Driver");

typedef enum boolean {
    false= 0,
    true = 1
} bool;

/* useful 2.4-ism */
#ifndef set_current_state
#define set_current_state(state_value) do { current->state = state_value; mb(); } while(0)
#endif

static int ControlOpen( struct inode *inode, struct file *file );
static int ControlClose( struct inode *inode, struct file *file );
static int ControlIoctl( struct inode *inode,
                         struct file *file,
                         unsigned int cmd,
                         unsigned long arg );

static int ControlMajor;
static const char *ControlName = "iscsictl";

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
static struct file_operations ControlFops = {
    owner: THIS_MODULE,
    ioctl: ControlIoctl,    /* ioctl */
    open: ControlOpen,      /* open */
    release: ControlClose,  /* release */
};
#else
static struct file_operations ControlFops = {
    NULL,                   /* lseek */
    NULL,                   /* read */
    NULL,                   /* write */
    NULL,                   /* readdir */
    NULL,                   /* poll */
    ControlIoctl,       /* ioctl */
    NULL,                   /* mmap */
    ControlOpen,        /* open */
    NULL,                   /* flush */
    ControlClose,       /* release */
};
#endif

spinlock_t hba_list_lock = SPIN_LOCK_UNLOCKED;
static iscsi_hba_t *hba_list = NULL;

static unsigned int init_module_complete = 0;
static int TimerActive = 0;
static atomic_t NumTargets = ATOMIC_INIT(0);

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
static wait_queue_head_t WaitQ;
#else
static struct wait_queue *WaitQ = NULL;
#endif


#ifdef DEBUG_TRACE
spinlock_t TraceSpinLock = SPIN_LOCK_UNLOCKED;
#endif



#ifdef DEBUG_TRACE
static iscsiTraceEntry TraceTable[ISCSI_TRACE_COUNT];
static int TraceIndex=0;
#define ISCSI_TRACE( P_FLG, P_LUN, P_CMD, P_TGT, P_ITT, P_DAT ) \
    iscsi_fill_trace( P_FLG, P_LUN, P_CMD, P_TGT, P_ITT, P_DAT );
#else
#define ISCSI_TRACE( P_FLG, P_LUN, P_CMD, P_TGT, P_ITT, P_DAT )
#endif

#if ( LINUX_VERSION_CODE < KERNEL_VERSION(2,3,27) )
/* note change modeled per linux2.4 drivers/scsi/ips.c */
struct proc_dir_entry proc_dir_iscsi = {
#ifdef PROC_SCSI_ISCSI
    PROC_SCSI_ISCSI,
#else
    PROC_SCSI_NOT_PRESENT,
#endif
    9,
    "iscsi",
    S_IFDIR|S_IRUGO|S_IXUGO,
    2
};
#endif


/* wake up the tx_thread without ever losing the wakeup event */
static void wake_tx_thread(int control_bit, iscsi_session_t *session)
{
    /* tell the Tx thread what to do when it wakes up. */
    set_bit(control_bit, &session->control_bits);

    /* We make a condition variable out of a wait queue and atomic test&clear.
     * May get spurious wake-ups, but no wakeups will be lost.
     * this is cv_signal().  wait_event_interruptible is cv_wait().
     */
    set_bit(TX_WAKE, &session->control_bits);
    wake_up_interruptible(&session->tx_wait_q);
}

static void iscsi_task_done(iscsi_task_t *task)
{
#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
    unsigned long cpu_flags;
#else
    unsigned int cpu_flags;
#endif

    if (task && task->scsi_cmnd) {
        spin_lock_irqsave( &io_request_lock, cpu_flags );
        if (task->scsi_cmnd && test_bit(TASK_ABORTING, &task->flags) == 0) {
#ifdef DEBUG_FLOW
            printk("iSCSI: completing command %p, (%u %u %u %u), Cmd 0x%x, result 0x%x\n", 
                   task->scsi_cmnd, 
                   task->scsi_cmnd->host->host_no, task->scsi_cmnd->channel, task->scsi_cmnd->target, task->scsi_cmnd->lun, 
                   task->scsi_cmnd->cmnd[0], task->scsi_cmnd->result);
#endif
            task->scsi_cmnd->scsi_done(task->scsi_cmnd);
            atomic_dec(&task->session->num_tasks);
        }
        else {
            DEBUG_FLOW1("iSCSI: not completing aborted task %p\n", task);
        }
        spin_unlock_irqrestore( &io_request_lock, cpu_flags );
    }
    else {
        DEBUG_ERR0("iSCSI: tried to complete NULL scsi cmd\n");
    }
}


/* compare against 2^31 */
#define SNA32_CHECK 2147483648UL

/* Serial Number Arithmetic, 32 bits, less than, RFC1982 */
static int sna_lt(uint32_t n1, uint32_t n2)
{
    return ((n1 != n2) && 
            (((n1 < n2) && ((n2 - n1) < SNA32_CHECK)) || ((n1 > n2) && ((n2 - n1) < SNA32_CHECK))));
}

/* Serial Number Arithmetic, 32 bits, less than, RFC1982 */
static int sna_lte(uint32_t n1, uint32_t n2)
{
    return ((n1 == n2) ||
            (((n1 < n2) && ((n2 - n1) < SNA32_CHECK)) || ((n1 > n2) && ((n2 - n1) < SNA32_CHECK))));
}

/* possibly update the ExpCmdSN and MaxCmdSN */
static void updateSN(iscsi_session_t *session, UINT32 expcmdsn, UINT32 maxcmdsn) 
{
    /* standard specifies this check for when to update expected and max sequence numbers */
    if (!sna_lt(maxcmdsn, expcmdsn - 1)) {
        if (! sna_lt(expcmdsn, session->ExpCmdSn)) {
            session->ExpCmdSn = expcmdsn;
        }
        if (! sna_lt(maxcmdsn, session->MaxCmdSn)) {
            session->MaxCmdSn = maxcmdsn;
            /* wake the tx thread to try sending more commands */
            wake_tx_thread(TX_SCSI_COMMAND, session);
        }
    }

    DEBUG_FLOW2("iSCSI: ExpCmdSN %u, MaxCmdSN %u\n", session->ExpCmdSn, session->MaxCmdSn);
}

static iscsi_session_t *find_session_for_cmnd(Scsi_Cmnd *sc)
{
    iscsi_session_t *session = NULL;
    iscsi_hba_t *hba;

    if (!sc->host)
        return NULL;

    if (!sc->host->hostdata)
        return NULL;

    hba = (iscsi_hba_t *)sc->host->hostdata;

    /* find the session for this command */
    spin_lock(&hba->session_lock);
    session = hba->session_list_head;
    while (session && (session->channel != sc->channel || session->target_id != sc->target))
        session = session->next;
    spin_unlock(&hba->session_lock);

    return session;
}

static iscsi_task_t *find_task(iscsi_session_t *session, uint32_t itt)
{
    iscsi_task_t *task;
    
    /* search for it, rather than use session->hba->task_mem + (itt % ISCSI_TASKS_PER_HBA),
     * to make sure task is really in use by the specified session, and to make
     * sure all caller's sync with the task_lock.
     */
    spin_lock(&session->task_lock);
    task = session->task_head;
    while (task && task->itt != itt)
        task = task->next;

    if (task && task->itt == itt) {
        DEBUG_FLOW2("iSCSI: itt %u is task %p\n", itt, task);
    }
    else {
        DEBUG_ERR3("iSCSI: no task for itt %u, task[%u] is itt %u\n", itt, (itt % ISCSI_TASKS_PER_HBA), task->itt);
    }
    spin_unlock(&session->task_lock);
    return task;
}

static iscsi_task_t *alloc_task(iscsi_session_t *session )
{
    iscsi_hba_t *hba = session->hba;
    iscsi_task_t *task;
    uint32_t task_index;
    uint32_t itt_index;

    /* get one from the HBA's free list */
    spin_lock( &hba->task_lock);
    task = hba->free_task_head;
    if ( task ) {
        hba->free_task_head = task->next;
        memset(task, 0, sizeof(iscsi_task_t) );
        hba->active_tasks++;

        /* put it on the session's task list */
        spin_lock(&session->task_lock);
        if ( session->task_head ) {
            session->task_tail->next = task;
            task->prev = session->task_tail;
        } else {
            session->task_head = task;
        }
        session->task_tail = task;
        task->jiffies = jiffies;
        task->session = session;
        spin_unlock(&session->task_lock);

        /* compute an appropriate ITT, such that:
         *   the ITT chosen is always >= the session's next available ITT value
         *   the ITT chosen % ISCSI_TASKS_PER_HBA is the index of the task allocated.
         */
        task_index = task - hba->task_mem; /* index into pre-allocated tasks */
        itt_index = session->itt % ISCSI_TASKS_PER_HBA; /* where the session's next available ITT maps to */
        if (task_index < itt_index)
            session->itt = session->itt + ISCSI_TASKS_PER_HBA - task_index;
        else
            session->itt =  session->itt + task_index - itt_index;

        /* allocate the ITT */
        task->itt = session->itt++;
        DEBUG_FLOW2("iSCSI: allocated task %p, itt %u\n", task, task->itt);
    }
    spin_unlock(&hba->task_lock);

    return task;
}

static void free_task( iscsi_task_t *task )
{
    iscsi_session_t *session = task->session;
    iscsi_hba_t *hba;

    if (! session) {
        DEBUG_ERR1("iSCSI: free_task couldn't find session for task %p\n", task);
        return;
    }
    hba = session->hba;
    if ( ! hba ) {
        DEBUG_ERR1("iSCSI: free_task couldn't find hba for task %p\n", task);
        return;
    }

    /* remove the task from the session's task list */
    spin_lock(&session->task_lock);
    if (task == session->task_head) {
        session->task_head = task->next;
    }
    else if (task == session->task_tail) {
        session->task_tail = task->prev;
    }
    if ( task->prev )
        task->prev->next = task->next;
    if ( task->next )
        task->next->prev = task->prev;
    task->prev = NULL;
    task->next = NULL;
    task->session = NULL;
    task->itt = 0;
    spin_unlock(&session->task_lock);

    /* FIXME: make sure nobody else is using the task before we free it */
    
    /* disconnect the Scsi_Cmnd and the task */
    if (task->scsi_cmnd && (iscsi_task_t *)task->scsi_cmnd->host_scribble == task)
        task->scsi_cmnd->host_scribble = NULL;
    task->scsi_cmnd = NULL;
    
    /* put the task on the HBA's free list */
    spin_lock( &hba->task_lock);
    if ( hba->free_task_head )
        hba->free_task_tail->next = task;
    else
        hba->free_task_head = task;
    hba->free_task_tail = task;
    hba->active_tasks--;
    spin_unlock( &hba->task_lock);
}

static int iscsi_recvmsg( iscsi_session_t *session, struct msghdr *msg, int len )
{
    int rc;
    mm_segment_t oldfs;

    oldfs = get_fs();
    set_fs( get_ds() );
    
    /* FIXME: ought to loop handling short reads */
    rc = sock_recvmsg( session->socket, msg, len, MSG_WAITALL);
    if (rc > 0) {
        session->last_rx = jiffies;
        mb();
    }

    set_fs( oldfs );
    return rc;
}

static int iscsi_sendmsg( iscsi_session_t *session, struct msghdr *msg, int len )
{
    int rc;
    mm_segment_t oldfs;

    current->flags |= PF_MEMALLOC; /* ??? */
    oldfs = get_fs();
    set_fs( get_ds() );

    /* FIXME: loop trying to send, until all is sent, or a signal/error occurs */
    rc = sock_sendmsg( session->socket, msg, len );

    set_fs( oldfs );
    current->flags &= ~PF_MEMALLOC;

    return rc;
}


#ifdef DEBUG_TRACE
static void
iscsi_fill_trace( unsigned char flag,
                    unsigned char lun,
                    unsigned char cmd,
                    unsigned char target,
                    unsigned int itt,
                    unsigned int data )
{
    iscsiTraceEntry *te;

    spin_lock( &TraceSpinLock);
    te = &TraceTable[TraceIndex];
    TraceIndex += 1;
    if ( TraceIndex >= ISCSI_TRACE_COUNT ) {
        TraceIndex = 0;
    }
    spin_unlock(&TraceSpinLock);

    te->lun = lun;
    te->cmd = cmd;
    te->itt = itt;
    te->flag = flag;
    te->data = data;
    te->target = target;
    te->activecmds = hba_list->active_tasks;
    te->jiffies = jiffies;

    return;
}
#endif

static int
SetDirection( unsigned char cmd )
{
    switch( cmd ) {
        case TEST_UNIT_READY:
        case START_STOP:
        case REZERO_UNIT:
        case WRITE_FILEMARKS:
        case SPACE:
        case ERASE:
        case ALLOW_MEDIUM_REMOVAL:
            /* just control commands */
            return ISCSI_CMD_CONTROL;
      case WRITE_6:           case WRITE_10:          case WRITE_12:
      case WRITE_LONG:        case WRITE_SAME:        case WRITE_BUFFER:
      case WRITE_VERIFY:      case WRITE_VERIFY_12:
      case COMPARE:           case COPY:              case COPY_VERIFY:
      case SEARCH_EQUAL:      case SEARCH_HIGH:       case SEARCH_LOW:
      case SEARCH_EQUAL_12:   case SEARCH_HIGH_12:    case SEARCH_LOW_12:
      case FORMAT_UNIT:       case REASSIGN_BLOCKS:   case RESERVE:
      case MODE_SELECT:       case MODE_SELECT_10:    case LOG_SELECT:
      case SEND_DIAGNOSTIC:   case CHANGE_DEFINITION: case UPDATE_BLOCK:
      case SET_WINDOW:        case MEDIUM_SCAN:       case SEND_VOLUME_TAG:
      case 0xea:
            return ISCSI_CMD_WRITE;
      default:
         return ISCSI_CMD_READ;
    }

    return -1;
}

static void
SetTag( Scsi_Cmnd *cmd, struct IscsiScsiCmdHdr *hdr )
{
    if ( cmd->device->tagged_supported ) {
        switch( cmd->tag ) {
            case HEAD_OF_QUEUE_TAG:
                hdr->flags.attr = ISCSI_ATTR_HEAD_OF_QUEUE;
                break;
            case ORDERED_QUEUE_TAG:
                hdr->flags.attr = ISCSI_ATTR_ORDERED;
                break;
            default:
                hdr->flags.attr = ISCSI_ATTR_SIMPLE;
                break;
        }
    }
    else
        hdr->flags.attr = ISCSI_ATTR_UNTAGGED;
}


static void
iscsi_xmit_data( iscsi_session_t *session, iscsi_task_t *task )
{
    struct msghdr msg;
    struct IscsiDataHdr stdh;
    Scsi_Cmnd *sc = NULL;
    struct scatterlist *sglist = NULL;
    int wlen, rc, index, iovn;
    int i, segOffset=0, remain, xfrlen, data_offset, segSN = 0;
    int bytes_to_fill, bytes_from_segment;
    
    sc = task->scsi_cmnd;
    remain = task->data_credit;
    
    memset( &stdh, 0, sizeof(stdh) );
    stdh.opcode = ISCSI_OP_SCSI_DATA;
    stdh.itt = htonl(task->itt);
    if( task->ttt != RSVD_TASK_TAG ) {         
        /* R2T data */
        stdh.ttt = task->ttt;
        stdh.offset = task->data_offset;
        data_offset = stdh.offset;
    } 
    else {                    
        /* Immediate, or Unsolicited data */
        stdh.ttt = htonl(RSVD_TASK_TAG);
        data_offset = 0;
    }
    
    session->TxIov[0].iov_base = &stdh;
    session->TxIov[0].iov_len  = sizeof(stdh);
    
    /* Find the segment and offset within the segment to start writing from.
     * We use the offset to find it (amg)
     */
    index = 0;
    if ( sc->use_sg ) {
        sglist = (struct scatterlist *)sc->buffer;
        segOffset = data_offset;
        for(i = 0; i < sc->use_sg; i++) {
            if(segOffset < sglist->length)
                break;
            segOffset -= sglist->length;
            sglist += 1;
        }
        index = i;
    }
   
    /* Our starting point is now segOffset within segment index.
     * Start sending the data 
     */
    DEBUG_FLOW4("iSCSI: xmit_data cmd %p, bufflen %d, credit %d @ %d\n",
                task, sc->bufflen, remain, data_offset);
    while (remain) {
        stdh.datasn = htonl(segSN);
        stdh.offset = htonl(data_offset);
        
        if (remain > session->dataPDULength) {
            bytes_to_fill = session->dataPDULength;
        } else {
            bytes_to_fill = remain;
            stdh.final = 1;
            stdh.expstatsn = htonl(session->ExpStatSn);
        }
        
        if ( sc->use_sg ) {
            iovn = 1;
            
            xfrlen = 0;
            while( index < sc->use_sg && bytes_to_fill) {
                bytes_from_segment = sglist->length - segOffset;
                if ( bytes_from_segment > bytes_to_fill ) {
                    xfrlen += bytes_to_fill;
                    session->TxIov[iovn].iov_base = sglist->address + segOffset;
                    session->TxIov[iovn].iov_len  = bytes_to_fill;
                    iovn += 1;
                    segOffset += bytes_to_fill;
                    break;
                } else {
                    xfrlen += bytes_from_segment;
                    session->TxIov[iovn].iov_base = sglist->address + segOffset;
                    session->TxIov[iovn].iov_len  = bytes_from_segment;
                    bytes_to_fill -= bytes_from_segment;
                    iovn += 1;
                    index += 1;
                    sglist += 1;
                    segOffset = 0;
                }
            }
        } else {
            session->TxIov[1].iov_base = sc->buffer + data_offset;
            session->TxIov[1].iov_len = xfrlen = bytes_to_fill;
            iovn = 2;
        }
        
        hton24(stdh.dlength, xfrlen);
        
        memset( &msg, 0, sizeof(msg) );
        msg.msg_iov = &session->TxIov[0];
        msg.msg_iovlen = iovn;
        /* msg.msg_flags = MSG_DONTWAIT; */
        
        wlen = sizeof(stdh) + xfrlen;
        rc = iscsi_sendmsg( session, &msg, wlen );
        if ( rc != wlen ) {
            DEBUG_ERR2("iSCSI: Error XmitImmData %d, %d\n",wlen,rc);
            /* close the connection */
            if (session->rx_pid)
                kill_proc(session->rx_pid, SIGPIPE, 1);
            return;
        }
        
        sc->request_bufflen -= xfrlen;

        task->data_credit -= xfrlen;

        DEBUG_FLOW3("iSCSI: XmitData sent %d @ %d, remaining credit %d\n",
                    xfrlen, data_offset, task->data_credit);
        /*
         * We shouldn't send more data than we are allowed to. Flag an error
         * if this happens
         */
        if (task->data_credit < 0) {
                DEBUG_ERR1("iSCSI: Error XmitImmData. Exceeded credit (%d)\n",
                            task->data_credit);
                task->data_credit = 0;
        }

        /* We can send up to data_credit without waiting for R2T */
        if (!task->data_credit && sc->request_bufflen) {
            set_bit(TASK_NEED_R2T, &task->flags);
            break;
        }

        remain  -= xfrlen;
        data_offset += xfrlen;
        if (++segSN == 0xffffffff)
            segSN++;
        
        if (!remain)
            clear_bit(TASK_NEED_R2T, &task->flags);
    }
    
    ISCSI_TRACE( ISCSI_TRACE_TxData, sc->lun, sc->cmnd[0], sc->target,
                 (unsigned int)task, task->ttt );
    
    return;
}

static iscsi_task_t *iscsi_xmit_cmd( iscsi_session_t *session, Scsi_Cmnd *sc)
{
    struct msghdr msg;
    struct iovec iov;
    struct IscsiScsiCmdHdr stsch;
    int rc, wlen, directx;
    iscsi_task_t *task;

    DEBUG_FLOW5("iSCSI: xmit_cmd (%u %u %u %u), Cmd 0x%x\n",
                sc->host->host_no, sc->channel, sc->target,sc->lun, sc->cmnd[0]);
    DEBUG_FLOW3("       cmd_len 0x%x, request_len 0x%x, buflen 0x%x\n",
                sc->cmd_len, sc->request_bufflen, sc->bufflen);

    wlen = sizeof(stsch);
    memset( &stsch, 0, sizeof(stsch) );

    directx = SetDirection( sc->cmnd[0] );
    if ( directx < 0 ) {
        printk("iSCSI: xmit_cmd - unsupported SCSI command, 0x%x\n",sc->cmnd[0]);
        return NULL;
    }

    task = alloc_task(session);
    if (!task) {
        DEBUG_ERR0("iSCSI: xmit_cmd couldn't allocate task!\n");
        return NULL;
    }
    task->direction = directx;
    task->scsi_cmnd = sc;
    task->cmdsn = session->CmdSn;

    if ( directx != ISCSI_CMD_CONTROL ) {
        /* read or write type command */
        if ( directx == ISCSI_CMD_READ ) {
            /* read */
            stsch.flags.read_data = 1;
        }
        else {
            /* write */
            stsch.flags.write_data = 1;                                        
        }
        stsch.data_length = htonl(sc->bufflen);
    }

    SetTag( sc, &stsch );
    stsch.opcode = ISCSI_OP_SCSI_CMD;
    stsch.itt = htonl(task->itt);
    stsch.cmdsn = htonl(session->CmdSn);
    stsch.expstatsn = htonl(session->ExpStatSn);
    if (stsch.flags.write_data && session->initialR2T)
        stsch.flags.final = 1;   
                                    
    stsch.lun[1] = sc->lun;
    memcpy( stsch.scb, sc->cmnd, sc->cmd_len );

    iov.iov_base = &stsch;
    iov.iov_len = sizeof(stsch);
    memset( &msg, 0, sizeof(struct msghdr) );
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;

    rc = iscsi_sendmsg( session, &msg, wlen );
    if ( rc != wlen ) {
        DEBUG_ERR2("iSCSI: xmit_cmd sendmsg %d failed, %d\n",wlen,rc);
        /* free the task */
        free_task(task);
    
        /* close the connection */
        if (session->rx_pid)
            kill_proc(session->rx_pid, SIGPIPE, 1);
        return NULL;
    }

    session->CmdSn++;
    atomic_inc(&session->num_tasks);

    ISCSI_TRACE( ISCSI_TRACE_TxCmd, sc->lun, sc->cmnd[0], sc->target,
                 (unsigned int)task, sc->bufflen );

    return task;
}


static void iscsi_xmit_queued_cmnds( iscsi_session_t *session)
{
    Scsi_Cmnd *sc;
    iscsi_task_t *task;

    while (session && sna_lte(session->CmdSn, session->MaxCmdSn)) {
        /* try to grab one SCSI command off the queue */
        spin_lock(&session->scsi_cmnd_lock);
        if ((sc = session->scsi_cmnd_head)) {
            session->scsi_cmnd_head = (Scsi_Cmnd *)sc->host_scribble;
        }
        spin_unlock(&session->scsi_cmnd_lock);
        if ( sc == NULL) {
            return;
        }

        DEBUG_FLOW4("iSCSI: sending CmdSN %u, connection %p, ExpCmdSn %u, MaxCmdSn %u\n", 
                    session->CmdSn, session, session->ExpCmdSn, session->MaxCmdSn);
        sc->host_scribble = NULL;

        task = iscsi_xmit_cmd(session, sc);
        if (task) {
            /* once issued, a Scsi_Cmds's host_scribble is our task rather than a linked list pointer */
            sc->host_scribble = (unsigned char *)task;
            if ( task->direction == ISCSI_CMD_WRITE) {
                if (session->initialR2T == 0) {
                    task->data_credit = MIN(session->firstBurstSize,
                                            sc->bufflen);
                    iscsi_xmit_data( session, task );
                } else {
                    task->data_credit = 0;
                    set_bit(TASK_NEED_R2T, &task->flags);
                }
            }
            DEBUG_FLOW2("iSCSI: issued cmnd %p, task %p\n", sc, task);
        }
        else {
            DEBUG_FLOW1("iSCSI: failed to send cmnd %p, returning to queue\n", sc);

            /* put it back on the head of the Scsi_Cmnd queue, we couldn't send it */
            spin_lock(&session->scsi_cmnd_lock);
            if ( session->scsi_cmnd_head ) {
                sc->host_scribble = (unsigned char *)session->scsi_cmnd_head;
                session->scsi_cmnd_head = sc;
            }
            else {
                session->scsi_cmnd_head = session->scsi_cmnd_tail = sc;
            }
            spin_unlock(&session->scsi_cmnd_lock);

            /* don't try to send any more right now */
            return;
        }
    }
}


static void
iscsi_xmit_nop( iscsi_session_t *session, struct IscsiNopInHdr *stnih )
{
    struct IscsiNopOutHdr stnoh;
    struct msghdr msg;
    struct iovec iov;
    int rc;

    memset( &stnoh, 0, sizeof(stnoh) );
    stnoh.opcode = ISCSI_OP_NOOP_OUT;
    stnoh.opcode |= ISCSI_OP_IMMEDIATE;
    if (stnih) {
        /* NOP reply */
        stnoh.itt  = RSVD_TASK_TAG;
        stnoh.ttt  = stnih->ttt;
        memcpy(stnoh.lun, stnih->lun, sizeof(stnoh.lun));
        /* FIXME: we need to reply with the data in the NopIn, if there is any */
    }
    else {
        /* initiator sending a Nop */
        session->itt += ISCSI_TASKS_PER_HBA;
        stnoh.itt = htonl(session->itt); 
        session->itt += ISCSI_TASKS_PER_HBA;
        stnoh.ttt  = RSVD_TASK_TAG;;
        memset(stnoh.lun, 0, sizeof(stnoh.lun));
    }
    stnoh.cmdsn  = htonl(session->CmdSn); /* don't increment after immediate cmds */
    stnoh.expstatsn = htonl(session->ExpStatSn);

    iov.iov_base = &stnoh;
    iov.iov_len = sizeof(stnoh);
    memset( &msg, 0, sizeof(msg) );
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    
    rc = iscsi_sendmsg( session, &msg, sizeof(stnoh) );
    if ( rc != sizeof(stnoh) ) {
        DEBUG_ERR1("iSCSI: XmitNop error %d\n",rc);
        /* close the connection */
        if (session->rx_pid)
            kill_proc(session->rx_pid, SIGPIPE, 1);
    }
    
    ISCSI_TRACE( ISCSI_TRACE_TxNop, 0, 0, 0, 0, 0 );
    
    return;
}

static int
iscsi_xmit_abort(iscsi_task_t *task)
{
    struct msghdr msg;
    struct iovec iov;
    int rc;
    iscsi_session_t *session;
    struct IscsiScsiTaskMgtHdr ststmh;
    Scsi_Cmnd *sc = task->scsi_cmnd;

    session = task->session;
    if ( ! session ) {
        printk("iSCSI: no session for command %p, can't abort\n", sc);
        return 1;
    }

    memset( &ststmh, 0, sizeof(ststmh) );
    ststmh.opcode = ISCSI_OP_SCSI_TASK_MGT_MSG;
    /* Flag it as an Immediate CMD */
    ststmh.opcode |= ISCSI_OP_IMMEDIATE;
    session->itt += ISCSI_TASKS_PER_HBA;
    ststmh.itt = htonl(session->itt); 
    session->itt += ISCSI_TASKS_PER_HBA;
    ststmh.rtt = htonl(task->itt);

    ststmh.lun[1] = sc->lun;
    ststmh.function = ISCSI_TM_FUNC_ABORT_TASK;
    ststmh.cmdsn = htonl(session->CmdSn);   /* CmdSn not incremented after imm cmd */
    ststmh.expstatsn = htonl(session->ExpStatSn);

    iov.iov_base = &ststmh;
    iov.iov_len = sizeof(ststmh);
    memset( &msg, 0, sizeof(msg) );
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;

    DEBUG_FLOW3("iSCSI: tx itt %u, abort for task %p, rtt %u\n", ntohl(ststmh.itt), task, task->itt);
    rc = iscsi_sendmsg( session, &msg, sizeof(ststmh) );
    if ( rc != sizeof(ststmh) ) {
        DEBUG_INIT1("iSCSI: sendmsg error %d\n",rc);
        /* close the connection */
        if (session->rx_pid)
            kill_proc(session->rx_pid, SIGPIPE, 1);
        return 1;
    }

    set_bit(TASK_ABORT_SENT, &task->flags);
    ISCSI_TRACE( ISCSI_TRACE_TxAbort, sc->lun, sc->cmnd[0], sc->target,
                 (unsigned int)task, task->rxdata);
    return 0;
}

/* send aborts for every task that needs one */
static void iscsi_xmit_aborts(iscsi_session_t *session) 
{
    iscsi_task_t *task;

    spin_lock(&session->task_lock);
    task = session->task_head;
    while (task) {
        if (test_bit(TASK_ABORTING, &task->flags) && test_bit(TASK_ABORT_SENT, &task->flags) == 0) {
            iscsi_xmit_abort(task);
        }
        task = task->next;
    }
    spin_unlock(&session->task_lock);
}

static void
iscsi_xmit_ping( iscsi_session_t *session )
{
    struct IscsiNopOutHdr stph;
    struct msghdr msg;
    struct iovec iov;
    int rc;
    
    memset( &stph, 0, sizeof(stph) );
    stph.opcode = ISCSI_OP_NOOP_OUT;
    stph.opcode |= ISCSI_OP_IMMEDIATE;
    stph.poll = 1;
    session->itt += ISCSI_TASKS_PER_HBA;
    stph.itt  = htonl(session->itt);
    session->itt += ISCSI_TASKS_PER_HBA;
    stph.cmdsn = htonl(session->CmdSn);
    stph.expstatsn = htonl(session->ExpStatSn);

    iov.iov_base = &stph;
    iov.iov_len = sizeof(stph);
    memset( &msg, 0, sizeof(msg) );
    msg.msg_iov = &iov;
    msg.msg_iovlen = 1;
    
    rc = iscsi_sendmsg( session, &msg, sizeof(stph) );
    if ( rc != sizeof(stph) ) {
        DEBUG_ERR1("iSCSI: XmitPing error %d\n",rc);
        /* close the connection */
        if (session->rx_pid)
            kill_proc(session->rx_pid, SIGPIPE, 1);
    }
    
    ISCSI_TRACE( ISCSI_TRACE_TxPing, 0, 0, 0, 0, 0 );
}


/* the writer thread */
static int
iscsi_tx_thread( void *vtaskp )
{
    iscsi_session_t *session;
    iscsi_task_t *task;

    if ( ! vtaskp ) {
        DEBUG_ERR0("iSCSI: tx thread task parameter NULL\n");
        return 0;
    }

    session = (iscsi_session_t *)vtaskp;

    DEBUG_INIT1("iSCSI: tx thread to %s\n", session->TargetName);

    while (!signal_pending(current)) {
        /* sleep (no timeout) until any of the following occurs:
         *     a new SCSI command is queued for this connection
         *     the connection's MaxCmdSN is incremented by the target.
         *     an R2T is received for a command previously issued on this connection.
         *     a timeout expires, and we need to send a Nop/poll to check the connection.
         *     a Nop with poll is received from the target, and we need to send a Nop in reply.
         *     a signal occurs.
         */
        wait_event_interruptible(session->tx_wait_q, test_and_clear_bit(TX_WAKE, &session->control_bits));
        DEBUG_FLOW1("iSCSI: tx thread %s is awake\n", session->TargetName);

        if ( signal_pending(current) ) {
            break;
        }

        /* See if we should send a ping (Nop with poll bit set) */
        if (test_and_clear_bit(TX_PING, &session->control_bits)) {
            DEBUG_FLOW1("iSCSI: sending Nop/poll to %s\n", session->TargetName);
            iscsi_xmit_ping( session );
        }

        /* See if we should send one or more Nops (no poll bit) */
        if (test_and_clear_bit(TX_NOP, &session->control_bits)) {
            /* FIXME: need to get Nop ttt, LUN, and data from a queue somewhere,
             * if these are replies to Nop/poll from the target.
             */
            DEBUG_FLOW1("iSCSI: sending Nop reply to %s\n", session->TargetName);
            iscsi_xmit_nop( session, NULL );
        }

        /* See if we should abort any tasks */
        if (test_and_clear_bit(TX_ABORT, &session->control_bits)) {
            iscsi_xmit_aborts(session);
        }

        /* See if we need to send more data after receiving an R2T */
        if (test_and_clear_bit(TX_R2T_DATA, &session->control_bits)) {
            spin_lock(&session->task_lock);
            task = session->task_head;
            while (task) {
                if ( test_bit(TASK_NEED_R2T, &task->flags) && test_and_clear_bit(TASK_GOT_R2T, &task->flags)) {
                    /* if new commands will need an R2T immediately, try to start them now, 
                     * so that we can overlap the R2T latency with the time it takes to send R2T 
                     * data for commands already issued.  Command PDUs are small, so this increases
                     * throughput without significantly increasing the completion time of commands 
                     * already issued.
                     */
                    if (session->initialR2T && !session->immediateData) {
                        iscsi_xmit_queued_cmnds(session);
                    }
                    iscsi_xmit_data( session, task );
                }
                task = task->next;
            }
            spin_unlock(&session->task_lock);
        }

        /* New SCSI command received, or MaxCmdSN incremented, or task freed */
        if (test_and_clear_bit(TX_SCSI_COMMAND, &session->control_bits)) {
            /* if possible, issue new commands */
            iscsi_xmit_queued_cmnds(session);
        }
    }

    DEBUG_INIT1("iSCSI: tx thread to %s shutting down\n", session->TargetName);

    /* kill the reader thread, if there still is one */
    if ( session->rx_pid > 1) {
        DEBUG_INIT2("iSCSI: tx thread %d killing rx thread %d\n", session->tx_pid, session->rx_pid);
        kill_proc(session->rx_pid, SIGPIPE, 1);
    }

    set_current_state(TASK_RUNNING);
    DEBUG_INIT1("iSCSI: tx thread for %s exiting\n",
                session->TargetName);
    
    return 0;
}

static void iscsi_recv_nop(iscsi_session_t *session, struct IscsiNopInHdr *stnih)
{
    ISCSI_TRACE( ISCSI_TRACE_RxPing, 0, 0, 0, 0, 0 );
    DEBUG_FLOW2("iSCSI: RxNop for session %p from %s\n", session, session->TargetName);
    session->ExpStatSn = ntohl(stnih->statsn)+1;

    updateSN(session, ntohl(stnih->expcmdsn), ntohl(stnih->maxcmdsn));

    if (stnih->poll) {
        /* FIXME: need to make the tx thread send a poll reply, 
         * with whatever data was in this ping.  Since the tx thread
         * may be busy, we may need to queue up the ping (and it's data)
         * til later.
         */
        printk("iSCSI: received Nop with poll, unimplemented feature\n");
        wake_tx_thread(TX_NOP, session);
    }
}

static void
iscsi_recv_cmd(iscsi_session_t *session, struct IscsiScsiRspHdr *stsrh, unsigned char *xbuf )
{
    iscsi_task_t *task;
    Scsi_Cmnd *sc;
    unsigned char *respbuf;
    int senselen;

    updateSN(session, ntohl(stsrh->expcmdsn), ntohl(stsrh->maxcmdsn));

    task = find_task(session, ntohl(stsrh->itt));
    if (!task) {
        /* odd, there should be a task. */
        DEBUG_ERR1("iSCSI: RxCmd - response for itt %u, but no such task\n", ntohl(stsrh->itt));
        return;
    }
    if (!task->scsi_cmnd || test_bit(TASK_ABORTING, &task->flags)) {
        /* the command it aborting, so just throw away the command response PDU,
         * since we will have told Linux that we aborted the command, and
         * we can't call the done() function if we told Linux we aborted the command.
         */
        DEBUG_FLOW2("iSCSI: RxCmd - response for itt %u, but task %p already aborted\n", ntohl(stsrh->itt), task);
        return;
    }

    sc = task->scsi_cmnd;

    DEBUG_FLOW3("iSCSI: RxCmd - cmd_len %d, Cmd 0x%x, rsp dlength %d\n",
                sc->cmd_len, sc->cmnd[0], ntoh24(stsrh->dlength));

    if (!stsrh->flags.rsp_selector ) {      
        /*
         * If this is not giving the right status, remove this
         * check and let the cmd_status check handle it.
         */
        ISCSI_TRACE( ISCSI_TRACE_RxCmdStatus, sc->lun, sc->cmnd[0], sc->target,
                     (unsigned int)task, stsrh->cmd_status );
        DEBUG_INIT1("iSCSI: RxCmd iscsi status 0x%x\n",stsrh->cmd_status);
        sc->result = DID_NO_CONNECT << 16;
    } 
    else {
        sc->result = stsrh->cmd_status;
        if ( stsrh->cmd_status ) {
            senselen = ntoh24(stsrh->dlength);
            /* senselen = ntohl(stsrh->length); */
            ISCSI_TRACE( ISCSI_TRACE_RxCmdCmdStatus, sc->lun, sc->cmnd[0],
                         sc->target, (unsigned int)task, stsrh->cmd_status );
            DEBUG_INIT3("iSCSI: RxCmd (0x%x) bad CmdStatus 0x%x, RxLen 0x%x\n",
                        sc->cmnd[0],stsrh->cmd_status,senselen);
            if ( senselen ) {
                printk("iSCSI: (%u %u %u %u), %s, Cmd 0x%x\n       Sense %02x%02x%02x%02x %02x%02x%02x%02x\n",
                       sc->host->host_no, sc->channel, sc->target, sc->lun, session->TargetName, sc->cmnd[0],
                       xbuf[0],xbuf[1],xbuf[2],xbuf[3],
                       xbuf[4],xbuf[5],xbuf[6],xbuf[7]);
                if ( xbuf[7] == 10 ) {
                    printk("       %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x\n",
                           xbuf[8],xbuf[9],xbuf[10],xbuf[11],
                           xbuf[12],xbuf[13],xbuf[14],xbuf[15],
                           xbuf[16],xbuf[17]);
                } else {
                    printk("       %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x\n",
                           xbuf[8],xbuf[9],xbuf[10],xbuf[11],
                           xbuf[12],xbuf[13],xbuf[14],xbuf[15],
                           xbuf[16],xbuf[17],xbuf[18],xbuf[19],
                           xbuf[20],xbuf[21],xbuf[22],xbuf[23]);
                }
                if ( senselen > sizeof(sc->sense_buffer) ) {
                    senselen = sizeof(sc->sense_buffer);
                }
                memcpy( sc->sense_buffer, xbuf, senselen );
            }
        }
    }

    if ( (task->direction == ISCSI_CMD_READ) &&
         (task->rxdata != sc->bufflen) ) {

        set_bit(TASK_RX_RSP, &task->flags);

        if (stsrh->flags.overflow || stsrh->flags.underflow ) {
            if (stsrh->flags.overflow)
                set_bit(TASK_RX_OVERFLOW, &task->flags);
            else
                clear_bit(TASK_RX_OVERFLOW, &task->flags);
            
            if (stsrh->flags.underflow)
                set_bit(TASK_RX_UNDERFLOW, &task->flags);
            else
                clear_bit(TASK_RX_UNDERFLOW, &task->flags);

            DEBUG_INIT3("iSCSI: Cmd 0x%x, OvrFlw %d, UndFlw %d\n",sc->cmnd[0],
                        stsrh->flags.overflow,stsrh->flags.underflow);
            DEBUG_INIT3("           RxData 0x%x, ResCnt 0x%x, bufflen 0x%x\n",
                        task->rxdata,ntohl(stsrh->residual_count),sc->bufflen);
            if ( stsrh->flags.underflow ) {
                ISCSI_TRACE( ISCSI_TRACE_RxCmdFlow, sc->lun, sc->cmnd[0],
                             sc->target, (unsigned int)task, ntohl(stsrh->residual_count) );
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,19)
                sc->resid = ntohl(stsrh->residual_count);
#else
                if ( task->rxdata < sc->underflow ) {
                    sc->result = DID_ERROR;
                }
#endif
            } else {
                ISCSI_TRACE( ISCSI_TRACE_RxCmdFlow, sc->lun, sc->cmnd[0],
                             sc->target, (unsigned int)task, 0 );
            }
        } 
        else {
            /* All the data did not arrive */
            sc->result = DID_ERROR;
        }
    }

    ISCSI_TRACE( ISCSI_TRACE_RxCmd, sc->lun, sc->cmnd[0], sc->target,
                 (unsigned int)task, task->rxdata );

    if ( sc->cmnd[0] == TEST_UNIT_READY ) {
        if ( stsrh->cmd_status == 0 ) {
            /* when a LUN becomes ready */
            if ( test_and_set_bit(sc->lun, session->lun_bitmap)) {
                printk("iSCSI: Unit still Ready, (%u %u %u %u), TargetName %s\n",
                       sc->host->host_no, sc->channel, sc->target, sc->lun, session->TargetName);
            }
            else {
                printk("iSCSI: Unit Ready, (%u %u %u %u), TargetName %s\n",
                       sc->host->host_no, sc->channel, sc->target, sc->lun, session->TargetName);
            }
        }
        else {
            if (test_and_clear_bit(sc->lun, session->lun_bitmap)) {
                /* was ready, but isn't anymore */
                printk("iSCSI: Unit no longer Ready, (%u %u %u %u), TargetName %s\n",
                       sc->host->host_no, sc->channel, sc->target, sc->lun, session->TargetName);
            }
            else {
                printk("iSCSI: Unit Not Ready, (%u %u %u %u), TargetName %s\n",
                       sc->host->host_no, sc->channel, sc->target, sc->lun, session->TargetName);
            }
        }

        respbuf = sc->buffer;
        memset( respbuf, 0, sc->bufflen );
    }

    /* we're done with the command & task */
    if (test_bit(TASK_ABORTING, &task->flags) == 0) {
        iscsi_task_done(task);
        free_task(task);
        /* wake the tx thread after freeing the task, in case the tx
         * thread blocked after running out of available tasks.
         */
        wake_tx_thread(TX_SCSI_COMMAND, session);
        sc->host_scribble = NULL;
        ISCSI_TRACE( ISCSI_TRACE_CmdDone, sc->lun, sc->cmnd[0], sc->target,
                     (unsigned int)task, 0 );
    }
    else {
        DEBUG_FLOW2("iSCSI: RxCmd - response for itt %u, but task %p already aborted\n", ntohl(stsrh->itt), task);
    }
}

static void
iscsi_recv_r2t(iscsi_session_t *session, struct IscsiRttHdr *strh )
{
    Scsi_Cmnd *sc;
    iscsi_task_t *task;

    task = find_task(session, ntohl(strh->itt));
    if ( !task) {
        DEBUG_ERR1("iSCSI: R2T for itt %u, but no such task\n", ntohl(strh->itt));
        return;
    }
    if (!task->scsi_cmnd || test_bit(TASK_ABORTING, &task->flags)) {
        /* this is normal if the commands was aborted on our side */
        if (test_bit(TASK_ABORTING, &task->flags) == 0)
            DEBUG_ERR1("iSCSI: RxR2T - response for itt %u, but no Scsi_Cmnd\n", ntohl(strh->itt));
        updateSN(session, ntohl(strh->expcmdsn), ntohl(strh->maxcmdsn));
        return;
    }
    else 
        sc = task->scsi_cmnd;

    task->ttt = strh->ttt;
    task->data_credit  = ntohl(strh->data_length);
    task->data_offset = ntohl(strh->data_offset);
    set_bit(TASK_GOT_R2T, &task->flags);

    updateSN(session, ntohl(strh->expcmdsn), ntohl(strh->maxcmdsn));

    DEBUG_FLOW4("iSCSI: R2T cmd %p (itt 0x%08x) Len %d bytes @ %d\n", task,
                ntohl(strh->itt), task->data_credit, task->data_offset);

    ISCSI_TRACE( ISCSI_TRACE_RttData, sc->lun, sc->cmnd[0], sc->target,
              (unsigned int)task, task->ttt );

    /* wake up the Tx thread for this connection */
    wake_tx_thread(TX_R2T_DATA, session);
}


static void
iscsi_recv_data( iscsi_session_t *session, struct IscsiDataRspHdr *stdrh )
{
    iscsi_task_t *task;
    Scsi_Cmnd *sc = NULL;
    struct scatterlist *sglist;
    int i, dlength, offset, relative_offset, segNum=0, iovn, rc, pad;
    struct msghdr msg;
    struct iovec iov;
    bool found_segment = false;

    updateSN(session, ntohl(stdrh->expcmdsn), ntohl(stdrh->maxcmdsn));

    dlength = ntoh24( stdrh->dlength );
    offset = ntohl( stdrh->offset );

    /* If there are padding bytes after the data, get them */
    pad = dlength % PAD_WORD_LEN;
    if (pad) {
        pad = PAD_WORD_LEN - pad;
    }

    task = find_task(session, ntohl(stdrh->itt));
    if (!task) {
        DEBUG_ERR1("iSCSI: RxData - no task for itt %u, ignoring received data\n", ntohl(stdrh->itt));
        sc = NULL;
    }
    else if (!task->scsi_cmnd || test_bit(TASK_ABORTING, &task->flags)) {
            DEBUG_ERR1("iSCSI: RxData - task aborting, itt %u, ignoring received data\n", ntohl(stdrh->itt));
            sc = NULL;
    }
    else
        sc = task->scsi_cmnd;

    /* no SCSI command, just throw away the PDU */
    if (!sc) {
        int bytes_read = 0;

        while (!signal_pending(current) && (bytes_read < dlength + pad)) {
            int num_bytes = MIN(dlength + pad - bytes_read, sizeof(session->RxBuf));
            iov.iov_base = session->RxBuf;
            iov.iov_len = sizeof(session->RxBuf);;
            memset( &msg, 0, sizeof(struct msghdr) );
            msg.msg_iov = &iov;
            msg.msg_iovlen = 1;
            rc = iscsi_recvmsg( session, &msg, num_bytes);
            if ( rc != num_bytes ) {
                DEBUG_ERR2("iSCSI: failed to recv %d data PDU bytes, rc %d\n", num_bytes, rc);
                if (session->rx_pid)
                    kill_proc(session->rx_pid, SIGPIPE, 1);
            }
            bytes_read += num_bytes;
        }
        return;
    }

    relative_offset = offset;
    
    if (sc && sc->use_sg ) {
        sglist = (struct scatterlist *)sc->buffer;
        
        /* We need to find the sg segment where this data should go.
         * We look at the offset and dlength to find it (amg)
         */
        do {
            if (relative_offset < sglist->length ) {
                found_segment = true;
            } else {
                relative_offset -= sglist->length;
                sglist += 1;
                segNum += 1;
            }
        } while (!found_segment);
        
        iovn = sc->use_sg - segNum;
        
        for(i = 0; i < iovn; i++) {
            if(i == 0) {
                session->RxIov[i].iov_base = sglist->address + relative_offset;
                session->RxIov[i].iov_len  = sglist->length  - relative_offset;
            } else {
                session->RxIov[i].iov_base = sglist->address;
                session->RxIov[i].iov_len  = sglist->length;
            }
            sglist += 1;
        }
    } 
    else if (sc) {
        session->RxIov[0].iov_base = sc->buffer + offset;
        session->RxIov[0].iov_len  = dlength;
        iovn = 1;
    }
    else {
        /* no task, just throw away the data and find the next PDU */
        session->RxIov[0].iov_base = sc->buffer + offset;
        session->RxIov[0].iov_len  = dlength;
        iovn = 1;
    }

    memset( &msg, 0, sizeof(struct msghdr) );
    msg.msg_iov = &session->RxIov[0];
    msg.msg_iovlen = iovn;
    
    rc = iscsi_recvmsg( session, &msg, dlength );
    if ( rc <= 0 ) {
        DEBUG_ERR1("iSCSI: RxData bad rx response, %d\n",rc);
        return;
    }
    if ( rc != dlength ) {
        DEBUG_ERR2("iSCSI: RxData not full 0x%x, 0x%x\n",dlength,rc);
    }
    
    task->rxdata += rc;
    
    ISCSI_TRACE( ISCSI_TRACE_RxData, sc->lun, sc->cmnd[0], sc->target,
              (unsigned int)task, dlength );

    if (pad) {
        iov.iov_base = session->RxBuf;
        iov.iov_len = pad;
        memset( &msg, 0, sizeof(struct msghdr) );
        msg.msg_iov = &iov;
        msg.msg_iovlen = 1;
        rc = iscsi_recvmsg( session, &msg, pad );
        if ( rc != pad ) {
            DEBUG_ERR2("iSCSI: failed to recv %d pad bytes, rc %d\n", pad, rc);
            if (session->rx_pid)
                kill_proc(session->rx_pid, SIGPIPE, 1);
        }
    }
}

static void iscsi_recv_mgmt_rsp(iscsi_session_t *session, struct IscsiScsiTaskMgtRspHdr *ststmrh )
{
    iscsi_task_t *task;

    updateSN(session, ntohl(ststmrh->expcmdsn), ntohl(ststmrh->maxcmdsn));

    task = find_task(session, ntohl(ststmrh->rtt));
    if (task) {
        printk("iSCSI: rtt %u, mgmt response found task %p, cmnd %p\n", 
               ntohl(ststmrh->rtt), task, task->scsi_cmnd);
        /* the eh_abort will wait for the aborted bit to be set
         * before returning.  We don't really care if the target 
         * actually aborted the command, we just want it to be done
         * so that we can free the task before returning from eh_abort.
         */
        set_bit(TASK_ABORTED, &task->flags);
    }

    if ( ststmrh->response ) {
        DEBUG_ERR3("iSCSI: itt %u, rtt %u, mgmt response rejected, 0x%x\n", 
                   ntohl(ststmrh->itt), ntohl(ststmrh->rtt), ststmrh->response);
    }
    else {
        DEBUG_FLOW2("iSCSI: itt %u, rtt %u, mgmt response 0\n", 
                    ntohl(ststmrh->itt), ntohl(ststmrh->rtt));
    }

    ISCSI_TRACE( ISCSI_TRACE_RxAbort, 0, 0, 0, (unsigned int)task, ststmrh->response );
}

static void iscsi_recv_async_event(iscsi_session_t *session, struct IscsiAsyncEvtHdr *staeh )
{
    DEBUG_ERR2("iSCSI: Async Event scsiEvtInd 0x%x, iscsiEvtInd 0x%x\n",
               staeh->scsi_event_ind, staeh->iscsi_event_ind);

    updateSN(session, ntohl(staeh->expcmdsn), ntohl(staeh->maxcmdsn));

    ISCSI_TRACE( ISCSI_TRACE_RxAsnEv, 0, staeh->scsi_event_ind,
                 staeh->iscsi_event_ind, ntoh24(staeh->dlength), 0);

    return;
}

static void iscsi_rx_thread(iscsi_session_t *session)
{
    int status, rc, xlen, pad;
    struct msghdr msg;
    struct iovec iov;
    struct IscsiHdr sth;
    unsigned char *rxbuf;
    iscsi_task_t *task;
    Scsi_Cmnd *sc = NULL;
    mm_segment_t fs;

    while (!signal_pending(current)) {

        /* check for anything to read on socket */
        iov.iov_base = &sth;
        iov.iov_len = sizeof(sth);
        memset( &msg, 0, sizeof(struct msghdr) );
        msg.msg_iov = &iov;
        msg.msg_iovlen = 1;
        DEBUG_INIT1("iSCSI: rx thread %s waiting to receive\n", session->TargetName);
        rc = iscsi_recvmsg( session, &msg, sizeof(sth) );
        if ( rc == sizeof(sth) ) {
            /* FIXME: read any additional header segments as well */

            /* received something */
            xlen = ntoh24( sth.dlength );

            /* If there are padding bytes, read them */
            pad = xlen % PAD_WORD_LEN;
            if (pad) {
                pad = PAD_WORD_LEN - pad;
                xlen += pad;
            }

            DEBUG_FLOW3("iSCSI: rx PDU, opcode %x, len %d, pad %d\n", sth.opcode, xlen, pad);
            if ( xlen && (sth.opcode != ISCSI_OP_SCSI_DATA_RSP) ) {
                if ( xlen > ISCSI_RXCTRL_SIZE ) {
                    DEBUG_ERR2("iSCSI: packet length too large, opcode %x, dlen %d\n", sth.opcode, xlen);
                    if (session->rx_pid)
                        kill_proc(session->rx_pid, SIGPIPE, 1);
                    break;
                }
                rxbuf = session->RxBuf;
                iov.iov_base = rxbuf;
                iov.iov_len = xlen;
                memset( &msg, 0, sizeof(struct msghdr) );
                msg.msg_iov = &iov;
                msg.msg_iovlen = 1;
                rc = iscsi_recvmsg( session, &msg, xlen );
                if ( rc != xlen ) {
                    DEBUG_ERR2("iSCSI: recvmsg %d failed, %d\n", xlen, rc);
                    if (session->rx_pid)
                        kill_proc(session->rx_pid, SIGPIPE, 1);
                    break;
                }
            } 
            else {
                rxbuf = NULL;
            }
            
            switch( sth.opcode ) {
                case ISCSI_OP_NOOP_IN:
                    iscsi_recv_nop( session, (struct IscsiNopInHdr *)&sth );
                    break;
                case ISCSI_OP_SCSI_RSP:
                    session->ExpStatSn = ntohl(((struct IscsiScsiRspHdr *)&sth)->statsn)+1;
                    iscsi_recv_cmd(session, (struct IscsiScsiRspHdr *)&sth, rxbuf );
                    break;
                case ISCSI_OP_SCSI_TASK_MGT_RSP:
                    session->ExpStatSn = ntohl(((struct IscsiScsiTaskMgtRspHdr *)&sth)->statsn)+1;
                    iscsi_recv_mgmt_rsp(session, (struct IscsiScsiTaskMgtRspHdr *)&sth );
                    break;
                case ISCSI_OP_RTT_RSP:
                    iscsi_recv_r2t(session, (struct IscsiRttHdr *)&sth );
                    break;
                case ISCSI_OP_SCSI_DATA_RSP:
                    iscsi_recv_data( session, (struct IscsiDataRspHdr *)&sth );
                    break;
                case ISCSI_OP_ASYNC_EVENT:
                    session->ExpStatSn = ntohl(((struct IscsiAsyncEvtHdr *)&sth)->statsn)+1;
                    iscsi_recv_async_event(session, (struct IscsiAsyncEvtHdr *)&sth );
                    break;
                default:
                    DEBUG_ERR1("iSCSI: unexpected RxCmd opcode, 0x%x\n",
                               sth.opcode);
                    goto ThreadExit;
            }
        } 
        else {
            if ( rc != -EAGAIN ) {
                if ( (rc == -ECONNRESET) || (rc == 0) ) {
                    DEBUG_ERR1( "iSCSI: Connection to %s reset.\n",
                                session->TargetName);
                } 
                else {
                    if ( rc == -ERESTARTSYS ) {
                        DEBUG_ERR1( "iSCSI: Connection to %s terminated.\n",
                                    session->TargetName);
                    } 
                    else {
                        DEBUG_ERR2("iSCSI: Unexpected read status %d on connection to %s\n",
                                   rc, session->TargetName);
                    }
                }
                goto ThreadExit;
            }
        }
    }
    
 ThreadExit:
    printk("iSCSI: rx thread %s going down\n", session->TargetName);
    session->ConnFlag &= ~CONN_ALIVE;
    mb();

    /* indicate that we're already going down, so that we don't get killed */
    session->rx_pid = 0;
    mb();

    fs = get_fs();   
    set_fs (get_ds()); 
    while ((rc = waitpid(session->tx_pid, &status, __WCLONE|WNOHANG)) == -ERESTARTSYS) {
        DEBUG_INIT1("iSCSI: restarting waitpid %u\n", session->tx_pid);
    }
    set_fs(fs); 
    
    if (rc == session->tx_pid) {
        session->tx_pid = 0;
    }
    else {
        /* if we can't reap it now, kill it, and try again later */
        DEBUG_INIT2("iSCSI: rx thread %s killing tx_pid %d\n", session->TargetName, session->tx_pid);
        if (session->tx_pid)
            kill_proc(session->tx_pid, SIGKILL, 1);
    }

    /* abort all tasks for this connection */
    task = session->task_head;
    while (task) {
        sc = task->scsi_cmnd;
        printk("iSCSI: Aborting task %p, command %p, soft error\n", task, sc);
        if ( sc && sc->result == 0 && test_bit(TASK_ABORTING, &task->flags) == 0) {
            sc->result = DID_SOFT_ERROR << 16;
            iscsi_task_done(task);
            sc->host_scribble = NULL;
        }
        task->scsi_cmnd = NULL;
        free_task( task );
        task = session->task_head;
    }
    
    /* The SCSI commands not yet issued are left in the queue.
     * The mid-layer will time them out eventually, but if we get
     * a new session up before then, we can continue as if nothing happened.
     */
    
    /* disconnect a target */
    atomic_dec(&NumTargets);
    
    /* wait for the TX thread to exit. We need a better way of doing
     * this. Perhaps forking in the daemon and using 2 separate ioctls
     * for connection rx and tx, rather than using
     * kernel_thread() and waitpid() from within the kernel module.
     * On the other hand, if we want to get away from ioctl()s that
     * block for long periods of time, we need to use kernel threads.
     */
    if (session->tx_pid) {
        DEBUG_INIT2("iSCSI: rx thread %s waiting for tx_pid %d to exit\n", session->TargetName, session->tx_pid);
        fs = get_fs();     /* save previous value */
        set_fs (get_ds()); /* use kernel limit */
        do {
            rc = waitpid(session->tx_pid, &status, __WCLONE);
        } while (rc == -ERESTARTSYS);
        set_fs(fs); /* restore before returning to user space */
        if (rc > 0) {
            DEBUG_INIT2("iSCSI: reaped tx_pid %d, status 0x%x\n", session->tx_pid, status);
            session->tx_pid = 0;
        }
        else {
            printk("iSCSI: failed to reap tx_pid %d, rc %d, errno %d\n", session->tx_pid, rc, errno);
        }
    }

    set_current_state(TASK_RUNNING);
    DEBUG_INIT1("iSCSI: rx thread for %s done\n",
                session->TargetName);
}


static int
iscsi_session( iscsi_session_ioctl_t *ioctld )
{
    struct file *ctrlfile;
    struct inode *ctrlinode;
    iscsi_session_t *session;
    iscsi_hba_t *hba;
    struct socket *sock;
    int hba_number;
    int channel_number;
    int rc;
    
    DEBUG_INIT1("iSCSI: Session to %s\n", ioctld->TargetName);

    ctrlfile = fget( ioctld->socket_fd );
    if ( ! ctrlfile ) {
        return -EINVAL;
    }
    
    ctrlinode = ctrlfile->f_dentry->d_inode;
    if ( !ctrlinode || !ctrlinode->i_sock ) {
        fput( ctrlfile );
        return -EINVAL;
    }
    
    sock = &ctrlinode->u.socket_i;
    if ( sock->file != ctrlfile ) {
        fput( ctrlfile );
        return -EINVAL;
    }

    sock->sk->allocation = GFP_ATOMIC;
    
    /* check to see if this session should be accepted */

    /* find the HBA that has the requested iSCSI bus */
    hba_number = ioctld->iscsi_bus / ISCSI_MAX_CHANNELS_PER_HBA;
    channel_number = ioctld->iscsi_bus % ISCSI_MAX_CHANNELS_PER_HBA;

    spin_lock(&hba_list_lock);
    hba = hba_list;
    while (hba && (hba_number-- > 0)) {
        hba = hba->next;
    }
    if (!hba) {
        spin_unlock(&hba_list_lock);
        printk("iSCSI: couldn't find HBA with iSCSI bus %d\n", ioctld->iscsi_bus);
        return -EINVAL;
    }
    spin_unlock(&hba_list_lock);

    /* check sessions on the HBA */
    spin_lock(&hba->session_lock);
    session = hba->session_list_head;
    while (session) {
        if ( session->address_length == ioctld->AddressLength &&
             memcmp(session->ip_address, ioctld->RemoteIpAddr, session->address_length) == 0 &&
             strcmp(session->TargetName, ioctld->TargetName) == 0) {
            if (session->channel == channel_number &&
                session->target_id == ioctld->TargetId) {
                /* same iSCSI bus and target id, it's a valid session restart */
                break;
            }
            else {
                printk("iSCSI: error - TargetName %s cannot move from bus %d id %d to bus %d id %d\n", ioctld->TargetName, 
                       session->iscsi_bus, session->target_id, ioctld->iscsi_bus, ioctld->TargetId);
                fput(ctrlfile);
                spin_unlock(&hba->session_lock);
                spin_unlock(&hba_list_lock);
                return -EINVAL;
            }
        }
        else if (session->channel == channel_number &&
                 session->target_id == ioctld->TargetId) {
            printk("iSCSI: error - TargetName %s cannot claim bus %d id %d, already in use by %s\n", 
                   ioctld->TargetName, ioctld->iscsi_bus, session->target_id, session->TargetName);
            fput(ctrlfile);
            spin_unlock(&hba->session_lock);
            spin_unlock(&hba_list_lock);
            return -EPERM;
        }
        session = session->next;
    }
    
    if ( session ) {
        printk("iSCSI: another session to %s entering full-feature phase\n", ioctld->TargetName);
    } 
    else {
        /*
         * must be a new session
         */
        printk("iSCSI: first session to %s entering full-feature phase\n", ioctld->TargetName);

        session = (iscsi_session_t *)kmalloc( sizeof(iscsi_session_t), GFP_KERNEL );
        if ( ! session ) {
            DEBUG_ERR0("iSCSI: Cannot allocate new connection entry\n");
            fput( ctrlfile );
            return -ENOMEM;
        }
        memset( session, 0, sizeof(iscsi_session_t) );

        session->hba = hba;

        spin_lock_init( &session->scsi_cmnd_lock);
        session->scsi_cmnd_head = session->scsi_cmnd_tail = NULL;

        spin_lock_init( &session->task_lock);
        session->task_head = session->task_tail = NULL;
        atomic_set(&session->num_tasks, 0);

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
        init_waitqueue_head(&session->tx_wait_q);
#else
        session->tx_wait_q = NULL;
#endif
        /* copy the IP address and port for the /proc entries */
        session->address_length = ioctld->AddressLength;
        memcpy(session->ip_address, ioctld->RemoteIpAddr, ioctld->AddressLength);
        session->Port = ioctld->Port;

        /* record target name, id */
        memset(session->TargetName, 0, sizeof(session->TargetName));
        strncpy(session->TargetName, ioctld->TargetName, sizeof(session->TargetName));
        session->TargetName[TARGET_NAME_MAXLEN] = '\0';
        session->target_id = ioctld->TargetId;
        session->iscsi_bus = ioctld->iscsi_bus;
        session->channel = ioctld->iscsi_bus % ISCSI_MAX_CHANNELS_PER_HBA;

        /* init ITT */
        session->itt = 0;

        /* attach the session to the HBA */
        if (hba->session_list_head) {
            session->next = NULL;
            session->prev = hba->session_list_tail;
            hba->session_list_tail->next = session;
            hba->session_list_tail = session;
        }
        else {
            hba->session_list_head = hba->session_list_tail = session;
            session->next = session->prev = NULL;
        }
    }
    spin_unlock(&hba->session_lock);

    atomic_inc(&NumTargets);
    DEBUG_INIT1("iSCSI: Target count: %d\n", atomic_read(&NumTargets));
    
    session->socket = sock;
    session->CmdSn = ioctld->CmdSn;
    session->ExpCmdSn = ioctld->ExpCmdSn;
    session->MaxCmdSn = ioctld->MaxCmdSn;
    session->ExpStatSn = ioctld->InitStatSn;
    session->initialR2T = ioctld->initialR2T;
    session->dataPDULength = ioctld->dataPDULength << 9; /* convert from 512 byte units */
    session->firstBurstSize= ioctld->firstBurstSize << 9; /* convert from 512 byte units */
    session->immediateData = 0; /* if we ever decide to do immediate data, copy this from the ioctld */
    session->ConnFlag = CONN_ALIVE;
    session->PingTimeout = ioctld->PingTimeout; 
    session->IdleTimeout = ioctld->IdleTimeout;
    session->ActiveTimeout = ioctld->ActiveTimeout;
    session->last_rx = jiffies;

#ifdef USE_DYNAMIC_DEVICE
    DEBUG_INIT1("iSCSI: iscsi_session %s waiting for timer thread\n", session->TargetName);
    while (! TimerActive ) {
        if ( signal_pending(current) ) {
                        break;
        }
        set_current_state(TASK_INTERRUPTIBLE);
        rc = schedule_timeout( 10 );
    }
#endif

    DEBUG_INIT1("iSCSI: iscsi_session %s about to start tx and rx threads\n", session->TargetName);
    
#ifdef ISCSI_CLONE
    session->tx_pid = kernel_thread( iscsi_tx_thread, (void *)session, 0 );
#else
    session->tx_pid = kernel_thread( iscsi_tx_thread, (void *)session,
                                  (CLONE_VM|CLONE_FS|CLONE_FILES|CLONE_SIGHAND) );
#endif

    /* become the rx thread */
    session->rx_pid = current->pid;
    iscsi_rx_thread(session);
    session->rx_pid = 0;
    mb();

    /* the rx thread is done, so we're finally returning out of the kernel */
    DEBUG_INIT1("iSCSI: session to %s leaving kernel\n",
                ioctld->TargetName);

    /* FIXME: do we really want to do this?  If we can keep the socket open
     * after the daemon dies, perhaps we could leave iSCSI disks mounted
     * until the system would normally unmount and sync devices prior to
     * shutdown/reboot, rather than have to do it ourselves in the scripts.
     */
    fput( ctrlfile );

    return 0;
}


/*
 *  FIXME: it'd probably be cleaner to move the timeout logic to the rx thread.
 *         The only danger is if the rx thread somehow blocks indefinately.
 *         Doing timeouts here makes sure the timeouts get checked, at the
 *         cost of having this code constantly loop.
 */
static int
iscsi_timer_thread( void ) { 
    int rc; 
    iscsi_session_t *session;
    iscsi_hba_t *hba;

    DEBUG_INIT0("iSCSI: Timer\n");

    TimerActive = 1;

    spin_lock(&hba_list_lock);
    while (hba_list == NULL) {
        spin_unlock(&hba_list_lock);
        if ( signal_pending(current) ) {
            return 0;
        }
        set_current_state(TASK_INTERRUPTIBLE);
        rc = schedule_timeout( 10 );
        spin_lock(&hba_list_lock);
    }

    while ( hba_list ) {
        /* FIXME: consider moving timeout processing to the rx and tx threads, since any timeout
         *        must either wake the tx thread to send a Nop/poll, or kill the session
         *        and all it's threads.  Either way, we need at least one of the threads awake, and so
         *        handling timeouts there lets us get rid of the Timer thread.
         */
        spin_unlock(&hba_list_lock);
        interruptible_sleep_on_timeout( &WaitQ, 10 );
        if ( signal_pending(current) ) {
            break;
        }

        spin_lock(&hba_list_lock);
        hba = hba_list;
        /* process timeouts for all sessions */
        spin_lock(&hba->session_lock);
        session = hba->session_list_head;
        while ( session ) {
            unsigned long timeout;

            if (session->rx_pid) {
                if (atomic_read(&session->num_tasks))
                    timeout = session->ActiveTimeout;
                else
                    timeout = session->IdleTimeout;

                if (timeout) {
                    /* since jiffies will wrap eventually, we use serial arithmetic */
                    if (session->PingTimeout &&
                        sna_lt(session->last_rx + (timeout * HZ) + (session->PingTimeout * HZ), jiffies)) {
                        /* should have received something by now, kill the connection */
                        printk("iSCSI: dropping session %s, %lu second timeout expired, rx %lu, ping %lu\n", 
                               session->TargetName, timeout + session->PingTimeout, session->last_rx, session->last_ping);
                        if ( session->tx_pid ) {
                            printk("iSCSI: killing tx thread %d\n", session->tx_pid);
                            kill_proc(session->tx_pid, SIGKILL, 1);
                        }
                        if ( session->rx_pid ) {
                            printk("iSCSI: killing rx thread %d\n", session->rx_pid);
                            kill_proc(session->rx_pid, SIGPIPE, 1);
                        }
                    }
                    else if (sna_lt(session->last_rx + (timeout * HZ), jiffies) &&
                             sna_lt(session->last_ping, session->last_rx)) {
                        /* send a ping to try to provoke some traffic */
                        DEBUG_FLOW3("iSCSI: sending ping to %s, rx %lu, ping %lu\n", 
                                    session->TargetName, session->last_rx, session->last_ping);
                        session->last_ping = jiffies - 1;
                        wake_tx_thread(TX_PING, session);
                    }
                }
            }
            session = session->next;
        }
        spin_unlock(&hba->session_lock);
    }
    spin_unlock(&hba_list_lock);
    
    TimerActive = 0;
    
    set_current_state(TASK_RUNNING);
    DEBUG_INIT0("iSCSI: Timer leaving kernel\n");
    
    return 0;
}


int
iscsi_detect( Scsi_Host_Template *sht )
{
    struct Scsi_Host *sh;
    iscsi_hba_t *hba;
    iscsi_task_t *task;
    int i; 

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,3,0)
    sht->proc_dir = &proc_dir_iscsi;
#else
    sht->proc_name = "iscsi";
#endif

    sh = scsi_register( sht, sizeof(iscsi_hba_t) );
    if ( ! sh ) {
        printk("iSCSI: Unable to register controller\n");
        return 0;
    }

    sh->max_id = ISCSI_MAX_TARGETS;
    sh->max_lun = ISCSI_MAX_LUN;
    sh->max_channel = ISCSI_MAX_CHANNELS_PER_HBA - 1; /* convert from count to index */

    hba = (iscsi_hba_t *)sh->hostdata;
    memset( hba, 0, sizeof(iscsi_hba_t) );

    hba->next = NULL;

    /* record our Linux host (HBA) number for later use */
    hba->hostno = sh->host_no;

    /* list of sessions on this HBA */
    spin_lock_init(&hba->session_lock);
    hba->session_list_head = NULL;
    hba->session_list_tail = NULL;

    /* pre-allocate command entries for this HBA 
     * FIXME: this doesn't scale.  We ought to be allocating these
     * per session, or better yet, per LUN detected.
     */
    spin_lock_init( &hba->task_lock);
    i = (ISCSI_TASKS_PER_HBA * sizeof(iscsi_task_t));
    task = kmalloc( i, GFP_KERNEL);
    if ( ! task ) {
        printk("iSCSI: Detect could not allocate memory\n");
        return 0;
    }
    memset(task, 0, i);
    hba->free_task_head = hba->task_mem = task;
    hba->free_task_tail = hba->task_max = task + ISCSI_TASKS_PER_HBA - 1;
    task->prev = NULL;
    while (task < hba->task_max) {
        task->next = (task + 1);
        task->next->prev = task;
        task++;
    }
    task->next = NULL;

    hba->active = 1;

    /* for now, there's just one iSCSI HBA */
    hba_list = hba;
    printk("iSCSI: detected HBA %p, host #%d\n", hba, hba->hostno);
    return 1;
}

int
iscsi_release( struct Scsi_Host *sh )
{
    iscsi_hba_t *hba;
    iscsi_session_t *session;

    hba = (iscsi_hba_t *)sh->hostdata;
    if ( ! hba ) {
        return FALSE;
    }

    printk("iSCSI: Release hba %p, host #%d\n", hba, hba->hostno);
    
    /* remove all sessions on this HBA */
    spin_lock(&hba->session_lock);
    session = hba->session_list_head;
    while (session) {
        /* FIXME: wait for all session tasks to complete, or abort them? */
        /* FIXME: abort all queued SCSI commands for this session (or just let them timeout in the mid-layer)?  */
        /* probably unneccsary, since the daemon will keep the module use count non-zero, so the module
         * can't possibly unload until after the daemon dies, and all the sessions are gone by then.
         */
        hba->session_list_head = session->next;
        kfree( session );
        session = hba->session_list_head;
    }
    spin_unlock(&hba->session_lock);

    /* free all the task memory */
    kfree( hba->task_mem );
    
    /* remove from the iSCSI HBA list */
    spin_lock(&hba_list_lock);
    if (hba == hba_list) {
        hba_list = hba_list->next;
    }
    else {
        iscsi_hba_t *prior = hba_list;

        while (prior && prior->next != hba)
            prior = prior->next;
        if (prior && prior->next == hba)
            prior->next = hba->next;
    }
    spin_unlock(&hba_list_lock);

    memset( hba, 0, sizeof(iscsi_hba_t) );
    scsi_unregister( sh );

    return FAILED;
}

#ifdef ISCSI_USE_STRATEGY
int
iscsi_strategy( struct Scsi_Host *host )
{
    Scsi_Device *sd;
    Scsi_Cmnd *sc;
    iscsi_hba_t *hba = (iscsi_hba_t *)host->hostdata;
    iscsi_session_t *session;
    iscsi_task_t *task;

    for( sd=host->host_queue; sd; sd=sd->next ) {
        printk("iSCSI: Strategy Device 0x%x, Id %d, Lun %d, Chan %d\n",(int)sd,sd->id,sd->lun,sd->channel);
        for( sc=sd->device_queue; sc; sc=sc->next ) {
            printk("iSCSI: Strategy Cmd 0x%x, state 0x%x, tgt 0x%x\n",(int)sc,sc->state,sc->target);
        }
    }

    spin_lock(&hba->session_lock);
    session = hba->session_list_head;
    while( session ) {
        sc = session->scsi_cmnd_head;
        while( sc ) {
            printk("iSCSI: scsi_cmnd_head 0x%x\n",(int)sc);
            sc = (Scsi_Cmnd *)sc->host_scribble;
        }
        task = session->task_head;
        while( task ) {
            printk("iSCSI: task_head 0x%x\n",(int)task->sc);
            task = task->next;
        }
        session = session->next;
    }
    spin_unlock(&hba->session_lock);

    return 0;
}
#endif


int
iscsi_eh_abort( Scsi_Cmnd *sc )
{
    struct Scsi_Host *host = NULL;
    iscsi_hba_t *hba = NULL;
    iscsi_task_t *task;
    iscsi_session_t *session;

    if ( ! sc ) {
        return FAILED;
    }
    host = sc->host;
    if (! host) {
        printk("iSCSI: no host for SCSI command %p\n", sc);
        return FAILED;
    }
    hba = (iscsi_hba_t *)host->hostdata;
    if (!hba) {
        printk("iSCSI: no iSCSI HBA associated with SCSI command %p\n", sc);
        return FAILED;
    }

    /* find the appropriate session for the command */
    session = find_session_for_cmnd(sc);
    if (!session) {
        printk("iSCSI: can't abort cmnd %p, command unknown\n", sc);
        return FAILED;
    }

    sc->result = DID_ABORT << 16;

    /* check if it's a command we haven't sent yet (still in the scsi_cmnd queue).
     * if we find the command, remove it from the queue.
     */
    spin_lock(&session->scsi_cmnd_lock);
    if (sc == session->scsi_cmnd_head) {
        /* it's the head, remove it */
        session->scsi_cmnd_head = (Scsi_Cmnd *)sc->host_scribble; /* next */
        if (sc == session->scsi_cmnd_tail)
            session->scsi_cmnd_tail = NULL;
        printk("iSCSI: aborted unsent command %p to (%u %u %u %u), Cmd 0x%x\n",
               sc, sc->host->host_no, sc->channel, sc->target, sc->lun, sc->cmnd[0]);
        spin_unlock(&session->scsi_cmnd_lock);
        return SUCCESS;
    }
    else if (session->scsi_cmnd_head) {
        Scsi_Cmnd *prior, *next;
        
        /* find the command prior to sc */
        prior = session->scsi_cmnd_head;
        next = (Scsi_Cmnd *)session->scsi_cmnd_head->host_scribble;
        while (next && next != sc) {
            prior = next;
            next = (Scsi_Cmnd *)prior->host_scribble; /* next command */
        }
        if (prior && next == sc) {
            /* remove the command */
            prior->host_scribble = next->host_scribble;
            if (session->scsi_cmnd_tail == sc)
                session->scsi_cmnd_tail = prior;
            sc->host_scribble = NULL;
            printk("iSCSI: aborted unsent command %p, channel %d tgt %d lun %d, Cmd 0x%x\n",
                   sc, sc->channel, sc->target, sc->lun, sc->cmnd[0]);
            spin_unlock(&session->scsi_cmnd_lock);
            return SUCCESS;
        }
    }
    spin_unlock(&session->scsi_cmnd_lock);

    /* check tasks (commands already sent to the target) */
    spin_lock(&session->task_lock);
    task = session->task_head;
    while (task && task->scsi_cmnd != sc)
        task = task->next;
    spin_unlock(&session->task_lock);

    if (task) {
        int i;

        /* mark the task as aborting, so that we don't complete or free it elsewhere */
        set_bit(TASK_ABORTING, &task->flags);
        wake_tx_thread(TX_ABORT, session);

        /* this is the error case, and doesn't have to be fast, so rather
         * than arrange to wake up when the abort response comes in, 
         * just poll for it once a second.
         */
        for (i = 0; i < 5; i++) {
            set_current_state(TASK_INTERRUPTIBLE);
            /* FIXME: do we need to release the io_request lock temporarily? 
             * if we don't, we can't complete any scsi commands, since the 
             * rx thread would block trying to get the io_request_lock.
             */
            schedule_timeout(HZ);
            if (test_bit(TASK_ABORTED, &task->flags)) {
                /* received abort response */
                printk("iSCSI: aborted command %p, task %p, (%u %u %u %u), Cmd 0x%x\n",
                       sc, task, sc->host->host_no, sc->channel, sc->target, sc->lun, sc->cmnd[0]);
                DEBUG_FLOW1("           Timeout %lu jiffies\n",(jiffies - task->jiffies));
                sc->host_scribble = NULL;
                task->scsi_cmnd = NULL;
                atomic_dec(&session->num_tasks);
                free_task(task);
                return SUCCESS;
            }
        }
        /* FIXME: there's a race here.  We need to find a safe way to free the task 
         * when the abort times out and fails.  The rx thread may find the task before
         * we free it, and this be accessing freed memory.  Add a refcount to the task?
         */
        sc->host_scribble = NULL;
        task->scsi_cmnd = NULL;
        free_task(task);
    }

    return FAILED;
}

int
iscsi_eh_host_reset( Scsi_Cmnd *sc )
{
    struct Scsi_Host *host = NULL;
    iscsi_hba_t *hba = NULL;
    iscsi_session_t *session;
    int i;

    if ( ! sc ) {
        return FAILED;
    }
    host = sc->host;
    if (! host) {
        printk("iSCSI: host reset, no host for SCSI command %p\n", sc);
        return FAILED;
    }
    hba = (iscsi_hba_t *)host->hostdata;
    if (!hba) {
        printk("iSCSI: host reset, no iSCSI HBA associated with SCSI command %p\n", sc);
        return FAILED;
    }

    /* check twice if the session exists but is not active */
    /* FIXME: who thought this was useful? */
    for (i = 0; i < 2; i++) {
        spin_lock(&hba->session_lock);
        session = hba->session_list_head;
        while (session && (session->channel != sc->channel || session->target_id != sc->target)) {
            session = session->next;
        }
        if (session) {
            if (session->ConnFlag & CONN_ALIVE) {
                spin_unlock(&hba->session_lock);
                printk("iSCSI: host reset %p succeeded\n", hba);
                /* FIXME: aren't we supposed to complete existing commands with DID_RESET */
                return SUCCESS;
            }
            else {
                /* wait a while and see if the daemon will create a new session */
                spin_unlock(&hba->session_lock);
                /* FIXME: release io_request_lock? */
                set_current_state(TASK_INTERRUPTIBLE);
                schedule_timeout( (30*HZ) );
            }
        }
        else {
            spin_unlock(&hba->session_lock);
            printk("iSCSI: host reset %p failed, no session for channel %u target %u\n", hba, sc->channel, sc->target);
            return FAILED;
        }
    }

    printk("iSCSI: host reset %p failed, no session for channel %d target %d\n", hba, sc->channel, sc->target);
    return FAILED;
}

int
iscsi_eh_bus_reset( Scsi_Cmnd *sc )
{
    struct Scsi_Host *host = NULL;
    iscsi_hba_t *hba = NULL;
    iscsi_session_t *session;
    int i;

    if ( ! sc ) {
        return FAILED;
    }
    host = sc->host;
    if (! host) {
        printk("iSCSI: bus reset, no host for SCSI command %p\n", sc);
        return FAILED;
    }
    hba = (iscsi_hba_t *)host->hostdata;
    if (!hba) {
        printk("iSCSI: bus reset, no iSCSI HBA associated with SCSI command %p\n", sc);
        return FAILED;
    }

    /* check twice if the session exists but is not active */
    /* FIXME: who thought this was useful? */
    for (i = 0; i < 2; i++) {
        spin_lock(&hba->session_lock);
        session = hba->session_list_head;
        while (session && (session->channel != sc->channel || session->target_id != sc->target)) {
            session = session->next;
        }
        if (session) {
            if (session->ConnFlag & CONN_ALIVE) {
                spin_unlock(&hba->session_lock);
                printk("iSCSI: bus reset %p succeeded\n", hba);
                /* FIXME: we're supposed to complete existing commands with DID_RESET */
                return SUCCESS;
            }
            else {
                /* wait a while and see if the daemon will create a new session */
                spin_unlock(&hba->session_lock);
                set_current_state(TASK_INTERRUPTIBLE);
                /* FIXME: release io_request_lock? */
                schedule_timeout( (4*HZ) );
            }
        }
        else {
            spin_unlock(&hba->session_lock);
            printk("iSCSI: bus reset %p failed, no session for channel %u target %u\n", hba, sc->channel, sc->target);
            return FAILED;
        }
    }

    printk("iSCSI: bus reset %p failed, no session for channel %d target %d\n", hba, sc->channel, sc->target);
    return FAILED;
}

int
iscsi_eh_device_reset( Scsi_Cmnd *sc )
{
    struct Scsi_Host *host = NULL;
    iscsi_hba_t *hba = NULL;
    iscsi_session_t *session;
    int i;

    if ( ! sc ) {
        return FAILED;
    }
    host = sc->host;
    if (! host) {
        printk("iSCSI: device reset, no host for SCSI command %p\n", sc);
        return FAILED;
    }
    hba = (iscsi_hba_t *)host->hostdata;
    if (!hba) {
        printk("iSCSI: device reset, no iSCSI HBA associated with SCSI command %p\n", sc);
        return FAILED;
    }

    /* check twice if the session exists but is not active */
    /* FIXME: who thought this was useful? */
    for (i = 0; i < 2; i++) {
        spin_lock(&hba->session_lock);
        session = hba->session_list_head;
        while (session && (session->channel != sc->channel || session->target_id != sc->target)) {
            session = session->next;
        }
        if (session) {
            /* FIXME: send target reset, and then abort all commands for the target */
            atomic_set(&session->num_tasks, 0);

            if (session->ConnFlag & CONN_ALIVE) {
                spin_unlock(&hba->session_lock);
                printk("iSCSI: device reset %p succeeded\n", hba);
                /* FIXME: we're supposed to complete existing commands with DID_RESET */
                return SUCCESS;
            }
            else {
                /* wait a while and see if the daemon will create a new session */
                spin_unlock(&hba->session_lock);
                set_current_state(TASK_INTERRUPTIBLE);
                /* FIXME: release io_request_lock? */
                schedule_timeout( (30*HZ) );
            }
        }
        else {
            spin_unlock(&hba->session_lock);
            printk("iSCSI: device reset %p failed, no session for channel %u target %u\n", hba, sc->channel, sc->target);
            return FAILED;
        }
    }

    printk("iSCSI: device reset %p failed, no session for channel %d target %d\n", hba, sc->channel, sc->target);
    return FAILED;
}

int
iscsi_queue( Scsi_Cmnd *sc, void (*done)(Scsi_Cmnd *) )
{
    iscsi_hba_t *hba;
    iscsi_session_t *session;

    if (sc->host == NULL) {
        printk("iSCSI: queuecommand but no Scsi_Host\n");
        sc->result = DID_NO_CONNECT << 16;
        return 1;
    }

    hba = (iscsi_hba_t *)sc->host->hostdata;
    if ( (!hba) || (!hba->active) ) {
        printk("iSCSI: queuecommand but no HBA\n");
        sc->result = DID_NO_CONNECT << 16;
        return 1;
    }

    DEBUG_FLOW6("iSCSI: queue (%u %u %u %u), Cmd 0x%x, %p\n",
                hba->hostno, sc->channel, sc->target, sc->lun, sc->cmnd[0], sc);

    sc->scsi_done = done;
    sc->result = 0;
    sc->host_scribble = NULL;
    ISCSI_TRACE( ISCSI_TRACE_Qd, sc->lun, sc->cmnd[0], sc->target, 0,
                 (unsigned int)sc );
    
    if (hba) {
        /* dispatch directly to the session's queue here, or fail.
         * this works since we keep iscsi_session_t structures around
         * even after a session drops, and re-use them if we get a new
         * session from the daemon for the same target.
         */
        spin_lock(&hba->session_lock);
        session = hba->session_list_head;
        while (session) {
            if (session->channel == sc->channel && session->target_id == sc->target) {
                /* add it to the session's command queue */
                spin_lock(&session->scsi_cmnd_lock);
                if (session->scsi_cmnd_head) {
                    /* append at the tail */
                    session->scsi_cmnd_tail->host_scribble = (unsigned char *)sc;
                    session->scsi_cmnd_tail = sc;
                }
                else {
                    /* make it the head */
                    session->scsi_cmnd_head = session->scsi_cmnd_tail = sc;
                }
                DEBUG_FLOW1("iSCSI: queued %p\n", sc);
                spin_unlock(&session->scsi_cmnd_lock);
                spin_unlock(&hba->session_lock);
                wake_tx_thread(TX_SCSI_COMMAND, session);
                return 0;
            }
            session = session->next;
        }
        spin_unlock(&hba->session_lock);
    }

    /* it's normal for there to be no session while the module is initializing. */
    if (test_bit(0, &init_module_complete))
        printk("iSCSI: queuecommand %p failed to find a session for HBA %p, channel %d id %d lun %d\n",
               sc, hba, sc->channel, sc->target, sc->lun);


    /* Complete with an error.  the docs say we shouldn't, but
     * returning 1 hangs the machine, and queuing it up for later
     * makes init_module take forever, since it goes thorugh the
     * whole timeout and reset cycle.
     */
    DEBUG_INIT1("iSCSI: completing %p with DID_NO_CONNECT\n", sc);
    sc->result = DID_NO_CONNECT << 16;
    done(sc);

    return 0;
}

int
iscsi_biosparam( Disk *disk, kdev_t dev, int geom[] )
{
    DEBUG_INIT1("iSCSI: BiosParam, capacity %d\n",disk->capacity);
    geom[0] = 64;                                   /* heads */
    geom[1] = 32;                                   /* sectors */
    geom[2] = disk->capacity / (64*32);     /* cylinders */
    return 1;
}

const char *
iscsi_info( struct Scsi_Host *sh )
{
    iscsi_hba_t *hba;
    static char buffer[256];
    char *bp;

    DEBUG_INIT0("iSCSI: Info\n");
    hba = (iscsi_hba_t *)sh->hostdata;
    if ( ! hba ) {
        return NULL;
    }

    bp = &buffer[0];
    memset( bp, 0, sizeof(buffer) );
#if defined(DRIVER_INTERNAL_VERSION) && (DRIVER_INTERNAL_VERSION > 0)        
    sprintf(bp, "iSCSI (%d.%d.%d.%d)", 
                DRIVER_MAJOR_VERSION, DRIVER_MINOR_VERSION, DRIVER_PATCH_VERSION, DRIVER_INTERNAL_VERSION);
#else
    sprintf(bp, "iSCSI (%d.%d.%d)", 
                DRIVER_MAJOR_VERSION, DRIVER_MINOR_VERSION, DRIVER_PATCH_VERSION);
#endif

    return bp;
}

static char *show_session_luns(iscsi_session_t *session, char *bp, int hostno)
{
    /* if we've already found LUNs, show them all */
    int lfound = 0;
    int l;

    /* FIXME: how much can we write to the buffer? */
    for (l=0; l<ISCSI_MAX_LUN; l++) {
        if ( test_bit(l, session->lun_bitmap)) {
            if (session->address_length == 4) {
                bp += sprintf(bp, "         %03x 000 %03x %03x           %u.%u.%u.%u %03x %s\n", 
                              hostno, session->target_id, l,
                              session->ip_address[0], session->ip_address[1], session->ip_address[2], session->ip_address[3],
                              session->target_id, session->TargetName);
            }
            else if (session->address_length == 16) {
                bp += sprintf(bp, "         %03x 000 %03x %03x           %02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x  %03x %s\n", 
                              hostno, session->target_id, l,
                              session->ip_address[0], session->ip_address[1], session->ip_address[2], session->ip_address[3],
                              session->ip_address[4], session->ip_address[5], session->ip_address[6], session->ip_address[7],
                              session->ip_address[8], session->ip_address[9], session->ip_address[10], session->ip_address[11],
                              session->ip_address[12], session->ip_address[13], session->ip_address[14], session->ip_address[15],
                              session->target_id, session->TargetName);
            }
            lfound += 1;
        }
    }
    /* if we haven't found any LUNs, use ??? to tell iscsilun to probe for them */
    if ( ! lfound ) {
        if (session->address_length == 4) {
            bp += sprintf(bp, "         %03x 000 %03x ???           %u.%u.%u.%u %03x %s\n", 
                          hostno, session->target_id,
                          session->ip_address[0], session->ip_address[1], session->ip_address[2], session->ip_address[3],
                          session->target_id, session->TargetName);
        }
        else if (session->address_length == 16) {
            bp += sprintf(bp, "         %03x 000 %03x ???           %02x:%02x:%02x:%02x%02x:%02x:%02x:%02x%02x:%02x:%02x:%02x%02x:%02x:%02x:%02x  %03x %s\n", 
                          hostno, session->target_id,
                          session->ip_address[0], session->ip_address[1], session->ip_address[2], session->ip_address[3],
                          session->ip_address[4], session->ip_address[5], session->ip_address[6], session->ip_address[7],
                          session->ip_address[8], session->ip_address[9], session->ip_address[10], session->ip_address[11],
                          session->ip_address[12], session->ip_address[13], session->ip_address[14], session->ip_address[15],
                          session->target_id, session->TargetName);
        }
    }

    return bp;
}

int iscsi_proc_info( char *buffer,
                     char **start,
                     off_t offset,
                     int length,
                     int hostno,
                     int func )
{
    char *bp = buffer;
    iscsi_hba_t *hba;
    iscsi_session_t *session;

    DEBUG_INIT4("iSCSI: ProcInfo Offset 0x%lx, Len 0x%x, HostNo %d, Func %d\n",
                offset,length,hostno,func);

    bp += sprintf(bp,"# iSCSI:\n");
    bp += sprintf(bp,"# Local: Hst Chn Tgt Lun     Remote:  IpAddr    Tgt Name\n");

    /* FIXME: find the HBA corresponding to hostno */
    spin_lock(&hba_list_lock);
    hba = hba_list;
    while (hba && hba->hostno != hostno)
        hba = hba->next;
    
    if (!hba) {
        printk("iSCSI: couldn't find iSCSI HBA #%d\n", hostno);
        spin_unlock(&hba_list_lock);
        return 0;
    }

    /* process every session on the HBA */
    spin_lock(&hba->session_lock);
    session = hba->session_list_head;
    while (session) {
        bp = show_session_luns(session, bp, hostno);
        session = session->next;
    }
    spin_unlock(&hba->session_lock);

    spin_unlock(&hba_list_lock);

    *start = buffer + offset;

    if ( (bp-buffer) < offset ) {
        return 0;
    }

    if ( (bp-buffer-offset) < length ) {
        return (bp-buffer-offset);
    }

    return length;
}

/*
 * We cannot include scsi_module.c because the daemon has not got a connection
 * up yet.
 */
static int
ControlOpen( struct inode *inode, struct file *file )
{
    MOD_INC_USE_COUNT;
    return 0;
}

static int
ControlClose( struct inode *inode, struct file *file )
{
    MOD_DEC_USE_COUNT;
    return 0;
}

static int
ControlIoctl( struct inode *inode,
              struct file *file,
              unsigned int cmd,
              unsigned long arg)
{
    int rc;
    iscsi_session_ioctl_t ioctld;

    if ( ! current->files ) {
        printk("iSCSI: driver does not match kernel\n");
        return -EFAULT;
    }

    /* FIXME: delete this when we get rid of the timer thread */
    if ( cmd == ISCSI_GO ) {
        if ( TimerActive ) {
            return -EBUSY;
        }
        rc = iscsi_timer_thread();
        return rc;
    }

    if ( cmd == ISCSI_SESSION ) {
        if ( copy_from_user(&ioctld,(void *)arg,sizeof(iscsi_session_ioctl_t)) ) {
            DEBUG_ERR0("iSCSI: Cannot copy user Ioctl args\n");
            return -EINVAL;
        }

        rc = iscsi_session( &ioctld );

        return rc;
    }

#ifdef DEBUG_TRACE
    if ( cmd == ISCSI_GETTRACE ) {
        if (copy_to_user((void *)arg, &TraceTable[0],
                         (sizeof(iscsiTraceEntry)*ISCSI_TRACE_COUNT)))
            return -EFAULT;
        return TraceIndex;
    }
#endif

    return -EINVAL;
}

Scsi_Host_Template iscsiDriverTemplate = ISCSI;

int
init_module( void )
{
#ifndef USE_DYNAMIC_DEVICE
    int timo;
#endif

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
    init_waitqueue_head(&WaitQ);
#endif

    DEBUG_INIT0("iSCSI init module\n");

#if defined(DRIVER_INTERNAL_VERSION) && (DRIVER_INTERNAL_VERSION > 0)
    /* show internal version number */
    printk("iSCSI version %d.%d.%d.%d (%s)\n",
           DRIVER_MAJOR_VERSION, DRIVER_MINOR_VERSION, DRIVER_PATCH_VERSION, DRIVER_INTERNAL_VERSION,
           ISCSI_DATE);
#else
    /* released version number (INTERNAL == 0) */
    printk("iSCSI version %d.%d.%d (%s)\n",
           DRIVER_MAJOR_VERSION, DRIVER_MINOR_VERSION, DRIVER_PATCH_VERSION,
           ISCSI_DATE);
#endif

#if defined(BUILD_STR)
    /* show a build string if we have one */
    if (NULL != BUILD_STR) {
        printk("iSCSI %s\n", (char *)BUILD_STR);
    }
#endif

    ControlMajor = register_chrdev( 0, ControlName, &ControlFops );
    if ( ControlMajor < 0 ) {
        printk("iSCSI failed to register the control device\n");
        return ControlMajor;
    }
    printk("iSCSI control device major number %d\n", ControlMajor);

#if ( LINUX_VERSION_CODE >= KERNEL_VERSION(2,3,0) )
    iscsiDriverTemplate.module = THIS_MODULE;
#else
    iscsiDriverTemplate.module = &__this_module;
#endif
    scsi_register_module( MODULE_SCSI_HA, &iscsiDriverTemplate );
    if ( ! iscsiDriverTemplate.present ) {
        DEBUG_ERR0("iSCSI scsi_register not present\n");
        scsi_unregister_module( MODULE_SCSI_HA, &iscsiDriverTemplate );
    }

    DEBUG_INIT0("iSCSI init module complete\n");
    set_bit(0, &init_module_complete);
    return 0;
}

void
cleanup_module( void )
{
    int rc;

    DEBUG_INIT0("iSCSI cleanup module\n");
    if ( ControlMajor > 0 ) {
        rc = unregister_chrdev( ControlMajor, ControlName );
        if ( rc ) {
            DEBUG_ERR0("iSCSI Error trying to unregister control device\n");
        } else {
            ControlMajor = 0;
        }
    }

    if ( iscsiDriverTemplate.present ) {
        DEBUG_INIT0("iSCSI SCSI present\n");
        scsi_unregister_module( MODULE_SCSI_HA, &iscsiDriverTemplate );
        iscsiDriverTemplate.present = 0;
    }

    DEBUG_INIT0("iSCSI cleanup module complete\n");
    return;
}


/*
 * Overrides for Emacs so that we follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 4 
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -4
 * c-argdecl-indent: 4
 * c-label-offset: -4
 * c-continued-statement-offset: 4
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
