/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  sap1sup.c
 *
 * Abstract: Drawbridge specific support functions
 *
 --*/

static char *ident_sap1 = "aacraid_ident sap1sup.c 1.0.7 2000/10/11 Adaptec, Inc.";

#include "osheaders.h"


#include "AacGenericTypes.h"

#include "aac_unix_defs.h"

#include "fsatypes.h"
#include "comstruc.h"
#include "fsact.h"
#include "protocol.h"

#define DEFINE_PCI_IDS
#include "sap1common.h"
#include "monkerapi.h"

#include "fsaport.h"
#include "fsaioctl.h"


#include "pcisup.h"
#include "sap1.h"

#include "port.h"

#include "nodetype.h"
#include "comsup.h"
#include "afacomm.h"
#include "adapter.h"

#define BugCheckFileId                   (FSAFS_BUG_CHECK_CYCLONESUP)

// #define SaBugCheck(A,B,C) { KeBugCheckEx(0x00000AFA, __LINE__, (ULONG)A, (ULONG)B,(ULONG)C ); }

#define SaBugCheck(A, B, C) {	cmn_err(CE_PANIC, "aacdisk : line %s, 0x%x, 0x%x, 0x%x ", __LINE__, A, B, C);  }

#define NUM_TICKS_PER_SECOND (1000 * 1000 * 10) /* time is in 100 nanoseconds */

int MiniPortRevision = Sa_MINIPORT_REVISION;


//
// The list of all the Sa adapter structures
//

PSa_ADAPTER_EXTENSION	SaAdapterList;

int
SaInitDevice(
	IN PPCI_MINIPORT_COMMON_EXTENSION CommonExtension,
	IN ULONG AdapterNumber,
	IN ULONG PciBus,
	IN ULONG PciSlot
);

BOOLEAN
SaSendSynchFib(
	PVOID Arg1,
	ULONG FibPhysicalAddress
	);

FSA_USER_VAR	SaUserVars[] = {
	{ "AfaPortPrinting", (PULONG)&AfaPortPrinting, NULL },
};


//
// Declare private use routines for this modual
//


/*++

Routine Description:

    The Isr routine for fsa Sa based adapter boards.

Arguments:


Return Value:

	TRUE - if the interrupt was handled by this isr
	FALSE - if the interrupt was not handled by this isr

--*/
u_int
SaPciIsr (IN PSa_ADAPTER_EXTENSION AdapterExtension)
{
		USHORT InterruptStatus, Mask;
		u_int OurInterrupt = INTR_UNCLAIMED;

	InterruptStatus = Sa_READ_USHORT( AdapterExtension, DoorbellReg_p);

	//
	// Read mask and invert because drawbridge is reversed.
	//
	// This allows us to only service interrupts that have been enabled.
	//

	Mask = ~(Sa_READ_USHORT( AdapterExtension, SaDbCSR.PRISETIRQMASK));

	// Check to see if this is our interrupt.  If it isn't just return FALSE.


	if (InterruptStatus & Mask) {

		OurInterrupt = INTR_CLAIMED;

		if (InterruptStatus & PrintfReady) {

			ULONG Length, Level;
			unsigned char *cp;

			cp = AdapterExtension->Common->PrintfBufferAddress;

			//
			// The size of the Printbuffer is set in port.c
			// There is no variable or define for it
			//
			if (Length > 255)
				Length = 255;

			if (cp[Length] != 0) {
				// cmn_err (CE_NOTE, "byte %d is 0x%x, should be 0", Length, cp[Length]);
				cp[Length] = 0;
			}

			if (Level == LOG_HIGH_ERROR)
				cmn_err (CE_WARN, "%s:%s", OsGetDeviceName(AdapterExtension), AdapterExtension->Common->PrintfBufferAddress);
			else
				cmn_err (CE_NOTE, "%s:%s", OsGetDeviceName(AdapterExtension), AdapterExtension->Common->PrintfBufferAddress);

			bzero (AdapterExtension->Common->PrintfBufferAddress, 256);

			Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p,PrintfReady); //clear PrintfReady

			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,PrintfDone);

		} else if (InterruptStatus & DOORBELL_1) {	// Adapter -> Host Normal Command Ready

			AdapterExtension->Common->AdapterFuncs.InterruptHost(AdapterExtension->Common->Adapter, HostNormCmdQue);
			Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p, DOORBELL_1);

		} else if (InterruptStatus & DOORBELL_2) {	// Adapter -> Host Normal Response Ready

				AdapterExtension->Common->AdapterFuncs.InterruptHost(AdapterExtension->Common->Adapter, HostNormRespQue);
				Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p,DOORBELL_2);

		} else if (InterruptStatus & DOORBELL_3) {	// Adapter -> Host Normal Command Not Full

			AdapterExtension->Common->AdapterFuncs.InterruptHost(AdapterExtension->Common->Adapter, AdapNormCmdNotFull);
			Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p, DOORBELL_3);

		} else if (InterruptStatus & DOORBELL_4) {	// Adapter -> Host Normal Response Not Full

			AdapterExtension->Common->AdapterFuncs.InterruptHost(AdapterExtension->Common->Adapter, AdapNormRespNotFull);
			Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p, DOORBELL_4);

		}

	}
	return(OurInterrupt);
}


/*++

Routine Description:

	This routine will enable the corresponding adapter event to cause an interrupt on 
	the host.

Arguments:

	AdapterExtension - Which adapter to enable.

	AdapterEvent - Which adapter event.

	AtDeviceIrq - Whether the system is in DEVICE irql

Return Value:

    Nothing.

--*/
VOID
SaEnableInterrupt (PVOID Arg1, ADAPTER_EVENT AdapterEvent, BOOLEAN AtDeviceIrq)
{
	PPCI_MINIPORT_COMMON_EXTENSION CommonExtension = (PPCI_MINIPORT_COMMON_EXTENSION) Arg1;
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) CommonExtension->MiniPort;

	switch (AdapterEvent) {

	  case HostNormCmdQue:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRICLEARIRQMASK, DOORBELL_1 );

		break;

	  case HostNormRespQue:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRICLEARIRQMASK, DOORBELL_2 );

		break;

      case AdapNormCmdNotFull:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRICLEARIRQMASK, DOORBELL_3 );

		break;

      case AdapNormRespNotFull:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRICLEARIRQMASK, DOORBELL_4 );

		break;

	}

}



/*++

Routine Description:

	This routine will disable the corresponding adapter event to cause an interrupt on 
	the host.

Arguments:

	AdapterExtension - Which adapter to enable.

	AdapterEvent - Which adapter event.

	AtDeviceIrq - Whether the system is in DEVICE irql

Return Value:

    Nothing.

--*/
VOID
SaDisableInterrupt (PVOID Arg1,	ADAPTER_EVENT AdapterEvent, BOOLEAN AtDeviceIrq)
{
	PPCI_MINIPORT_COMMON_EXTENSION CommonExtension = (PPCI_MINIPORT_COMMON_EXTENSION) Arg1;
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) CommonExtension->MiniPort;

	switch (AdapterEvent) {


	  case HostNormCmdQue:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRISETIRQMASK, DOORBELL_1 );

		break;

	  case HostNormRespQue:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRISETIRQMASK, DOORBELL_2 );

		break;

      case AdapNormCmdNotFull:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRISETIRQMASK, DOORBELL_3 );

		break;


      case AdapNormRespNotFull:

		Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRISETIRQMASK, DOORBELL_4 );

		break;

	}

}


SaDetachDevice (IN PPCI_MINIPORT_COMMON_EXTENSION CommonExtension)
{
	PSa_ADAPTER_EXTENSION AdapterExtension = CommonExtension->MiniPort;

	//
	// Free the register mapping.
	//

	OsDetachDevice(AdapterExtension);

	OsFreeMemory( AdapterExtension, sizeof(Sa_ADAPTER_EXTENSION) );

}


/*++

Routine Description:

	Scans the PCI bus looking for the Sa card. When found all resources for the
	device will be allocated and the interrupt vectors and csrs will be allocated and
	mapped.

 	The device_interface in the commregion will be allocated and linked to the comm region.

Arguments:


Return Value:

    TRUE - if the device was setup with not problems
    FALSE - if the device could not be mapped and init successfully

--*/
int
SaInitDevice (IN PPCI_MINIPORT_COMMON_EXTENSION CommonExtension,
			  IN ULONG AdapterNumber, IN ULONG PciBus,
			  IN ULONG PciSlot)
{
	AAC_STATUS Status;
	PSa_ADAPTER_EXTENSION AdapterExtension = NULL;
	FSA_NEW_ADAPTER NewAdapter;
	ULONG StartTime, EndTime, WaitTime;
	ULONG InitStatus;
	int instance;
	char *name;

    AfaPortPrint("In init device.\n");

	CommonExtension->AdapterNumber = AdapterNumber;

	CommonExtension->PciBusNumber = PciBus;
	CommonExtension->PciSlotNumber = PciSlot;

	AdapterExtension = OsAllocMemory( sizeof(Sa_ADAPTER_EXTENSION), OS_ALLOC_MEM_SLEEP );
	AdapterExtension->Common = CommonExtension;
	CommonExtension->MiniPort = AdapterExtension;

	instance = OsGetDeviceInstance(AdapterExtension);
	name     = OsGetDeviceName(AdapterExtension);

	//
	// Map in the registers from the adapter, register space 0 is config space,
	// register space 1 is the memery space.
	//

	if (OsMapDeviceRegisters(AdapterExtension)){
		cmn_err(CE_WARN, "%s%d SaInitDevice: failed OsMapDeviceRegisters", name, instance);
		return(FAILURE);
	}


	//
	// Check to see if the board failed any self tests.
	//

	if (Sa_READ_ULONG( AdapterExtension, Mailbox7) & SELF_TEST_FAILED) {

		cmn_err(CE_WARN, "%s%d: adapter self-test failed\n",
				name, instance);
		return(FAILURE);
	}

	//
	// Check to see if the board panic'd while booting.
	//

	if (Sa_READ_ULONG( AdapterExtension, Mailbox7) & KERNEL_PANIC) {

		cmn_err(CE_WARN, "%s%d: adapter kernel panic'd\n",
				name, instance);
		return(FAILURE);
	}


	StartTime = OsGetSeconds();
	WaitTime = 0;


	//
	//  Wait for the adapter to be up and running. Wait up until 3 minutes.
	//

	while (!(Sa_READ_ULONG( AdapterExtension, Mailbox7) & KERNEL_UP_AND_RUNNING)) {
	
		EndTime = OsGetSeconds();

		WaitTime = EndTime - StartTime;

		if ( WaitTime > (3 * 60) ) {

			InitStatus = Sa_READ_ULONG( AdapterExtension, Mailbox7) >> 16;

			cmn_err(CE_WARN, "%s%d: adapter kernel failed to start, init status = %d\n",
					name, instance, InitStatus);
			return(FAILURE);

		}
	}

	if (OsAttachInterrupt(AdapterExtension, SaISR)) {
		cmn_err(CE_WARN, "%s%d SaInitDevice: failed OsAttachIntterupt", name, instance);
		return(FAILURE);
	}

	if (OsAttachDMA(AdapterExtension)) {
		cmn_err(CE_WARN, "%s%d SaInitDevice: failed OsAttachDMA", name, instance);
		return(FAILURE);
	}


	//
	// Fill in the function dispatch table.
	//

	AdapterExtension->Common->AdapterFuncs.SizeOfFsaPortFuncs = sizeof(FSAPORT_FUNCS);
	AdapterExtension->Common->AdapterFuncs.AllocateAdapterCommArea = AfaPortAllocateAdapterCommArea;
	AdapterExtension->Common->AdapterFuncs.FreeAdapterCommArea = AfaPortFreeAdapterCommArea;
	AdapterExtension->Common->AdapterFuncs.BuildSgMap = AfaPortBuildSgMap;
	AdapterExtension->Common->AdapterFuncs.FreeDmaResources = AfaPortFreeDmaResources;
	AdapterExtension->Common->AdapterFuncs.AllocateAndMapFibSpace = AfaPortAllocateAndMapFibSpace;
	AdapterExtension->Common->AdapterFuncs.UnmapAndFreeFibSpace = AfaPortUnmapAndFreeFibSpace;
	AdapterExtension->Common->AdapterFuncs.InterruptAdapter = SaInterruptAdapter;
	AdapterExtension->Common->AdapterFuncs.EnableInterrupt = SaEnableInterrupt;
	AdapterExtension->Common->AdapterFuncs.DisableInterrupt = SaDisableInterrupt;
	AdapterExtension->Common->AdapterFuncs.NotifyAdapter = SaNotifyAdapter;
	AdapterExtension->Common->AdapterFuncs.ResetDevice = SaResetDevice;
	AdapterExtension->Common->AdapterFuncs.InterruptHost = NULL;

	AdapterExtension->Common->AdapterFuncs.SendSynchFib = SaSendSynchFib;

	NewAdapter.AdapterExtension = CommonExtension;
	NewAdapter.AdapterFuncs = &AdapterExtension->Common->AdapterFuncs;
	NewAdapter.AdapterInterruptsBelowDpc = FALSE;
	NewAdapter.AdapterUserVars = SaUserVars;
	NewAdapter.AdapterUserVarsSize = sizeof(SaUserVars) / sizeof(FSA_USER_VAR);

	NewAdapter.Dip = CommonExtension->OsDep.dip;

	
	if ( AfaCommInitNewAdapter( &NewAdapter ) == NULL) {
			cmn_err(CE_WARN, "SaInitDevice: AfaCommInitNewAdapter failed\n");
			return (FAILURE);
	};


	AdapterExtension->Common->Adapter = NewAdapter.Adapter;

	if (AdapterExtension->Common->Adapter == NULL) {

		AfaPortLogError(AdapterExtension->Common, FAILURE, NULL, 0);
		cmn_err(CE_WARN, "%s%d SaInitDevice: No Adapter pointer", name, instance);

		return (FAILURE); 
	}


    //
	// Start any kernel threads needed
	OsStartKernelThreads(AdapterExtension);

	//
	// Tell the adapter that all is configure, and it can start accepting requests
	//

	SaStartAdapter(AdapterExtension);



	//
	// Put this adapter into the list of Sa adapters
	//

	AdapterExtension->Next = SaAdapterList;
	SaAdapterList = AdapterExtension;

	AdapterExtension->Common->AdapterConfigured = TRUE;


#ifdef AACDISK
	//
	// Call the disk layer to initialize itself.
	//

	AfaDiskInitNewAdapter( AdapterExtension->Common->AdapterNumber, AdapterExtension->Common->Adapter );
#endif


init_done:

	AdapterExtension->Common->AdapterPrintfsToScreen = FALSE;

	OsAttachHBA(AdapterExtension);

	return (0);

init_error:

	return (FAILURE);
}



VOID
SaStartAdapter (PSa_ADAPTER_EXTENSION AdapterExtension)
{
	ULONG ReturnStatus;
	LARGE_INTEGER HostTime;
	ULONG ElapsedSeconds;
	PADAPTER_INIT_STRUCT InitStruct;

	//
	// Fill in the remaining pieces of the InitStruct.
	//

	InitStruct = AdapterExtension->Common->InitStruct;

	InitStruct->HostPhysMemPages = AfaPortGetMaxPhysicalPage(AdapterExtension->Common);

	ElapsedSeconds = OsGetSeconds();

	InitStruct->HostElapsedSeconds = ElapsedSeconds;

	//
	// Tell the adapter we are back and up and running so it will scan its command
	// queues and enable our interrupts
	//

	AdapterExtension->LocalMaskInterruptControl =
		(PrintfReady | DOORBELL_1 | DOORBELL_2 | DOORBELL_3 | DOORBELL_4);


	//
	// First clear out all interrupts.  Then enable the one's that we can handle.
	//

	Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRISETIRQMASK, (USHORT) 0xffff );
	Sa_WRITE_USHORT( AdapterExtension,  SaDbCSR.PRICLEARIRQMASK,
					(PrintfReady | DOORBELL_1 | DOORBELL_2 | DOORBELL_3 | DOORBELL_4) );

	SaSendSynchCommand(AdapterExtension, 
			   INIT_STRUCT_BASE_ADDRESS, 
			   (ULONG) AdapterExtension->Common->PhysicalInitStruct,
			   0,
			   0,
			   0,
			   &ReturnStatus);

}


VOID
SaResetDevice (PVOID Arg1){

}


/*++

Routine Description:

	The will cause the adapter to take a break point.

Arguments:

	None

Return Value:

    Nothing

--*/
VOID
SaInterruptAdapter (PVOID Arg1)
{
	PPCI_MINIPORT_COMMON_EXTENSION CommonExtension = (PPCI_MINIPORT_COMMON_EXTENSION) Arg1;
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) CommonExtension->MiniPort;

	ULONG ReturnStatus;

	SaSendSynchCommand(AdapterExtension, 
			   BREAKPOINT_REQUEST,
			   0,
			   0,
			   0,
			   0,
			   &ReturnStatus);

}


/*++

Routine Description:

    Will read the adapter CSRs to find the reason the adapter has
    interrupted us.

Arguments:

    AdapterEvent - Enumerated type the returns the reason why we were interrutped.

Return Value:

    Nothing

--*/
VOID
SaNotifyAdapter (PVOID Arg1, IN HOST_2_ADAP_EVENT AdapterEvent)
{
	PPCI_MINIPORT_COMMON_EXTENSION CommonExtension = (PPCI_MINIPORT_COMMON_EXTENSION) Arg1;
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) CommonExtension->MiniPort;
	ULONG ReturnStatus;

    switch (AdapterEvent) {
        case AdapNormCmdQue:

			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_1);
            break;

        case HostNormRespNotFull:

			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_4);
            break;

        case AdapNormRespQue:

			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_2);
            break;

        case HostNormCmdNotFull:

			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_3);
            break;

        case HostShutdown:

//			SaSendSynchCommand(AdapterExtension, HOST_CRASHING, 0, 0, 0, 0, &ReturnStatus);

            break;

		case FastIo:
			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_6);
			break;

		case AdapPrintfDone:
			Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s,DOORBELL_5);
			break;

        default:

			SaBugCheck(0,0,0);
            AfaPortPrint("Notify requested with an invalid request 0x%x.\n",AdapterEvent);
            break;
    }
}


/*++

Routine Description:

	This routine will send a synchronous comamnd to the adapter and wait for its
	completion.

Arguments:

	AdapterExtension - Pointer to adapter extension structure.
	Command - Which command to send
	Parameter1 - 4	- Parameters for command
	ReturnStatus - return status from adapter after completion of command


Return Value:

	AAC_STATUS

--*/
AAC_STATUS
SaSendSynchCommand(
		   PVOID Arg1,
		   ULONG Command,
		   ULONG Parameter1,
		   ULONG Parameter2,
		   ULONG Parameter3,
		   ULONG Parameter4,
		   PULONG	ReturnStatus
	)
{
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) Arg1;
	ULONG StartTime,EndTime,WaitTime;
	BOOLEAN CommandSucceeded;

	//
	// Write the Command into Mailbox 0
	//

	Sa_WRITE_ULONG( AdapterExtension, Mailbox0, Command);

	//
	// Write the parameters into Mailboxes 1 - 4
	//

	Sa_WRITE_ULONG( AdapterExtension, Mailbox1, Parameter1);
	Sa_WRITE_ULONG( AdapterExtension, Mailbox2, Parameter2);
	Sa_WRITE_ULONG( AdapterExtension, Mailbox3, Parameter3);
	Sa_WRITE_ULONG( AdapterExtension, Mailbox4, Parameter4);

	//
	// Clear the synch command doorbell to start on a clean slate.
	//
		
	Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p, DOORBELL_0);

	//
	// Signal that there is a new synch command
	//

	Sa_WRITE_USHORT( AdapterExtension, DoorbellReg_s, DOORBELL_0);

	CommandSucceeded = FALSE;

	StartTime = OsGetSeconds();
	WaitTime = 0;

	while (WaitTime < 30) { // wait up to 30 seconds

		drv_usecwait(5);				// delay 5 microseconds to let Mon960 get info.

		//
		// Mon110 will set doorbell0 bit when it has completed the command.
		//

		if( Sa_READ_USHORT( AdapterExtension, DoorbellReg_p) & DOORBELL_0 )  {

			CommandSucceeded = TRUE;
			break;
		}

		EndTime = OsGetSeconds();
		WaitTime = EndTime - StartTime;

	}

	if (CommandSucceeded != TRUE) {

		return (STATUS_IO_TIMEOUT);

	}

	//
	// Clear the synch command doorbell.
	//
		
	Sa_WRITE_USHORT( AdapterExtension, DoorbellClrReg_p, DOORBELL_0);

	//
	// Pull the synch status from Mailbox 0.
	//

	*ReturnStatus = Sa_READ_ULONG( AdapterExtension, Mailbox0);

	//
	// Return SUCCESS
	//

	return (STATUS_SUCCESS);

}


/*++

Routine Description:

	This routine will send a synchronous fib to the adapter and wait for its
	completion.

Arguments:

	AdapterExtension - Pointer to adapter extension structure.
	FibPhysicalAddress - Physical address of fib to send.


Return Value:

	BOOLEAN

--*/
BOOLEAN
SaSendSynchFib (PVOID Arg1, ULONG FibPhysicalAddress)
{
	PPCI_MINIPORT_COMMON_EXTENSION CommonExtension = (PPCI_MINIPORT_COMMON_EXTENSION) Arg1;
	PSa_ADAPTER_EXTENSION AdapterExtension = (PSa_ADAPTER_EXTENSION) CommonExtension->MiniPort;
	ULONG returnStatus;

	if (SaSendSynchCommand( AdapterExtension,
						    SEND_SYNCHRONOUS_FIB,
						    FibPhysicalAddress,
						    0,
						    0,
						    0,
						    &returnStatus ) != STATUS_SUCCESS ) {

		return (FALSE);
		
	}
	
	return (TRUE);
										
}

BOOLEAN
WriteFlash(
	PVOID AdapterExtension,
	ULONG *MappedBuffer)
{
	return (FALSE);
}

BOOLEAN
ReadFlash(
	PVOID AdapterExtension,
	ULONG *MappedBuffer)
{
	return (FALSE);
}

