/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  osfuncs.c
 *
 * Abstract: Holds all of the O/S specific interface functions.
 *	
 --*/

static char *ident_osfuncs = "aacraid_ident osfuncs.c 1.0.6 2000/10/09 Adaptec, Inc.";

#include "osheaders.h"

//static LKINFO_DECL(fsa_locks, "fsa_locks",0);

extern aac_options_t g_options;

OS_SOFTINTR g_idle_task = { 0, 0, 0, 0 };
wait_queue_t * g_wait_queue_ptr = NULL;
wait_queue_t g_wait;

void OsTimeoutHandler( 
	struct semaphore * sem );

int * OsIdleTask( void * data );

//-----------------------------------------------------------------------------
// Memory Allocation functions

/*----------------------------------------------------------------------------*/
void * OsAllocMemory(
	OS_SIZE_T Size,
	unsigned int Flags )
/*----------------------------------------------------------------------------*/
{
	void *mem_ptr;

	if( !( mem_ptr = kmalloc( Size, GFP_KERNEL ) ) )
		cmn_err( CE_WARN, "OsAllocMemory: FAILED\n" );
	return( mem_ptr );
}


/*----------------------------------------------------------------------------*/
void OsFreeMemory(
	void * Buffer,
	OS_SIZE_T Size )
/*----------------------------------------------------------------------------*/
{
	kfree( Buffer );
}


/*----------------------------------------------------------------------------*/
int OsRegisterInterrupt(
	unsigned int irq,		// interrupt number
	void ( *handler )( int, void*, struct pt_regs * ),	// handler function
	void *irq_data )		// argument to handler function
/*----------------------------------------------------------------------------*/
{
  return( request_irq (irq, handler, SA_INTERRUPT | SA_SHIRQ,	"aacraid", irq_data ) );
}


/*----------------------------------------------------------------------------*/
void OsUnregisterInterrupt(
	unsigned int irq,		// interrupt number
	void *irq_data)
/*----------------------------------------------------------------------------*/
{
	free_irq (
		irq,										// interrupt number
		irq_data );
}


/*----------------------------------------------------------------------------*/
unsigned long OsVirtToPhys( 
	void * virtual_address )
/*----------------------------------------------------------------------------*/
{
	return( virt_to_phys( virtual_address ) );
}


//-----------------------------------------------------------------------------
// MUTEX functions

/*----------------------------------------------------------------------------*/
OS_STATUS OsMutexInit( 
	OS_MUTEX *Mutex,
	OS_SPINLOCK_COOKIE Cookie )
/*----------------------------------------------------------------------------*/
{
	Mutex->lock_var = 0;
	//	bzero (&Mutex->wq, sizeof (Mutex->wq));
	init_waitqueue_head (&Mutex->wq);
	return ( 0 );
}


/*----------------------------------------------------------------------------*/
void OsMutexDestroy( 
	OS_MUTEX *Mutex )
/*----------------------------------------------------------------------------*/
{
}


/*----------------------------------------------------------------------------*/
void OsMutexAcquire( 
	OS_MUTEX *Mutex )
/*----------------------------------------------------------------------------*/
{
  //	wait_queue_t wait = { current, NULL };
	unsigned long time_stamp;

	DECLARE_WAITQUEUE (wait, current);

	time_stamp = jiffies;

	if( test_and_set_bit( 0, &( Mutex->lock_var ) ) != 0 )
	{
		if( in_interrupt() )
			panic( "OsMutexAcquire going to sleep at interrupt time\n" );
		current->state = TASK_INTERRUPTIBLE;
		add_wait_queue( &( Mutex->wq ), &wait );
		while( test_and_set_bit( 0, &( Mutex->lock_var ) ) != 0 )
			schedule();
		remove_wait_queue( &( Mutex->wq ), &wait );
	}

	if( ( jiffies - 1 ) > time_stamp )
		cmn_err( CE_WARN, "Mutex %ld locked out for %ld ticks", 
			Mutex, jiffies - time_stamp );
}


/*----------------------------------------------------------------------------*/
void OsMutexRelease( 
	OS_MUTEX *Mutex )
/*----------------------------------------------------------------------------*/
{
	if( test_and_clear_bit( 0, &( Mutex->lock_var ) ) == 0 )
		cmn_err( CE_WARN, "OsMutexRelease: mutex not locked" );
	wake_up_interruptible( &( Mutex->wq ) );
}

// see man hierarchy(D5)
#define FSA_LOCK 1

//-----------------------------------------------------------------------------
// Spinlock functions

/*----------------------------------------------------------------------------*/
OS_SPINLOCK * OsSpinLockAlloc( void ) 
/*----------------------------------------------------------------------------*/
{
	OS_SPINLOCK *SpinLock;
	int i;


	SpinLock = ( OS_SPINLOCK * )kmalloc( sizeof( OS_SPINLOCK ), GFP_KERNEL );

	if (SpinLock == NULL)
		cmn_err (CE_WARN, "WARNING: OsSpinLockAlloc Failed!!!");
	
	SpinLock->spin_lock = SPIN_LOCK_UNLOCKED;
	for( i = 0; i < NR_CPUS; i++ )
		SpinLock->cpu_lock_count[ i ] = 0;
	return( SpinLock );
}


/*----------------------------------------------------------------------------*/
OS_STATUS OsSpinLockInit( 
	OS_SPINLOCK *SpinLock,
	OS_SPINLOCK_COOKIE Cookie )
/*----------------------------------------------------------------------------*/
{
	return( 0 );
}


/*----------------------------------------------------------------------------*/
void OsSpinLockDestroy( 
	OS_SPINLOCK *SpinLock )
/*----------------------------------------------------------------------------*/
{
	kfree( SpinLock );
	SpinLock = NULL;
}


/*----------------------------------------------------------------------------*/
void OsSpinLockAcquire( 
	OS_SPINLOCK *SpinLock )
/*----------------------------------------------------------------------------*/
{
	unsigned cpu_id;

	if( SpinLock )
	{
		cpu_id = smp_processor_id();
		if( SpinLock->cpu_lock_count[ cpu_id ] ){
			cmn_err (CE_PANIC, "CPU %d trying to acquire lock again: lock count = %d\n",
				 cpu_id, SpinLock->cpu_lock_count[ cpu_id ]);
		}		  
		
		spin_lock_irqsave( &( SpinLock->spin_lock ), SpinLock->cpu_flags[ cpu_id ] );
		SpinLock->cpu_lock_count[ cpu_id ]++;
	   
	} else {
		cmn_err( CE_WARN, "OsSpinLockAcquire: lock does not exist" );
	}
}


/*----------------------------------------------------------------------------*/
void OsSpinLockRelease( 
	OS_SPINLOCK *SpinLock )
/*----------------------------------------------------------------------------*/
{
	unsigned cpu_id;

	if( SpinLock )
	{
		cpu_id = smp_processor_id();
		SpinLock->cpu_lock_count[ cpu_id ]--;
		spin_unlock_irqrestore( &( SpinLock->spin_lock ), SpinLock->cpu_flags[ cpu_id ] );
	}
	else
		cmn_err( CE_WARN, "OsSpinLockRelease: lock does not exist" );
}


/*----------------------------------------------------------------------------*/
int OsSpinLockOwned(
	OS_SPINLOCK *SpinLock )
/*----------------------------------------------------------------------------*/
{
#ifdef CONFIG_SMP
	return spin_is_locked(&(SpinLock->spin_lock));
#endif
		return( 0 );
}


//-----------------------------------------------------------------------------
// CvLock functions

/*----------------------------------------------------------------------------*/
OS_CVLOCK *OsCvLockAlloc( void ) 
{
	OS_CVLOCK *cv_lock;


#ifdef CVLOCK_USE_SPINLOCK
	cv_lock = OsSpinLockAlloc(); 
#else
	cv_lock = ( OS_CVLOCK * )kmalloc( sizeof( OS_CVLOCK ), GFP_KERNEL );
	cv_lock->wq = NULL;
	cv_lock->lock_var = 0;
#endif

	return( cv_lock );
}


/*----------------------------------------------------------------------------*/
OS_STATUS OsCvLockInit( 
	OS_CVLOCK *cv_lock,
	OS_SPINLOCK_COOKIE Cookie )
/*----------------------------------------------------------------------------*/
{
	return ( 0 );
}


/*----------------------------------------------------------------------------*/
void OsCvLockDestroy( 
	OS_CVLOCK *cv_lock )
/*----------------------------------------------------------------------------*/
{
	if( cv_lock )
		kfree( cv_lock );
	cv_lock = NULL;
}


/*----------------------------------------------------------------------------*/
void OsCvLockAcquire (OS_CVLOCK *cv_lock)
{
#ifdef CVLOCK_USE_SPINLOCK
		OsSpinLockAcquire( cv_lock );
#else
		OsMutexAcquire(	cv_lock );
#endif
}


/*----------------------------------------------------------------------------*/
void OsCvLockRelease( 
	OS_CVLOCK *cv_lock )
/*----------------------------------------------------------------------------*/
{
#ifdef CVLOCK_USE_SPINLOCK
		OsSpinLockRelease( cv_lock );
#else
		OsMutexRelease( cv_lock );
#endif
}


/*----------------------------------------------------------------------------*/
int OsCvLockOwned(
	OS_CVLOCK *cv_lock )
/*----------------------------------------------------------------------------*/
{
	return( 1 );
}


//-----------------------------------------------------------------------------
// Conditional variable functions

/*----------------------------------------------------------------------------*/
void OsCv_init ( 
	OS_CV_T *cv_ptr )
/*----------------------------------------------------------------------------*/
{
	cv_ptr->lock_var = 1;
	init_waitqueue_head (&cv_ptr->wq);
}


/*----------------------------------------------------------------------------*/
void OsCv_destroy( 
	OS_CV_T  *cv_ptr )
/*----------------------------------------------------------------------------*/
{
}


/*______________________________________________________________________________
 -
 -
 -----------------------------------------------------------------------------*/
OsCv_wait (OS_CV_T *cv_ptr, OS_CVLOCK *cv_lock_ptr)
{
	unsigned long flags;

	DECLARE_WAITQUEUE (wait, current);
	
	if( in_interrupt() )
		panic( "OsCv_wait going to sleep at interrupt time\n" );

	cv_ptr->type = TASK_UNINTERRUPTIBLE;
	current->state = TASK_UNINTERRUPTIBLE;

	add_wait_queue( &cv_ptr->wq, &wait );

	OsCvLockRelease( cv_lock_ptr );
	schedule();

	while( test_and_set_bit( 0, &( cv_ptr->lock_var ) ) != 0 )
	{
		if( in_interrupt() )
			panic( "OsCv_wait going to sleep at interrupt time\n" );
		schedule();
	}

	remove_wait_queue( &( cv_ptr->wq ), &wait );

	OsCvLockAcquire( cv_lock_ptr );
}


/*----------------------------------------------------------------------------*/
int OsCv_wait_sig( 
	OS_CV_T *cv_ptr,
	OS_CVLOCK *cv_lock_ptr ) 
/*----------------------------------------------------------------------------*/
{
	unsigned long flags;
	int signal_state = 1;

	DECLARE_WAITQUEUE (wait, current);
	
	if( in_interrupt() )
		panic( "OsCv_wait_sig going to sleep at interrupt time\n" );

	cv_ptr->type = TASK_INTERRUPTIBLE;
	current->state = TASK_INTERRUPTIBLE;

	add_wait_queue( &( cv_ptr->wq ), &wait );

	OsCvLockRelease( cv_lock_ptr );
	schedule();

	while( ( test_and_set_bit( 0, &( cv_ptr->lock_var ) ) != 0 ) && 
			( !signal_pending( current ) ) )
	{
		if( in_interrupt() )
			panic( "OsCv_wait_sig going to sleep at interrupt time\n" );
		schedule();
	}

	if( signal_pending( current ) )
		signal_state = 0;
	
	remove_wait_queue( &( cv_ptr->wq ), &wait );
	
	OsCvLockAcquire( cv_lock_ptr );
	return( signal_state );
}


/*----------------------------------------------------------------------------*/
void OsCv_signal( 
	OS_CV_T *cv_ptr )
/*----------------------------------------------------------------------------*/
{

	clear_bit( 0, &( cv_ptr->lock_var ) );
	if( cv_ptr->type == TASK_INTERRUPTIBLE )
			wake_up_interruptible( &( cv_ptr->wq ) );
	else{
			wake_up( &( cv_ptr->wq ) );
	}
}


// return time in seconds
/*----------------------------------------------------------------------------*/
unsigned long OsGetSeconds( void )
/*----------------------------------------------------------------------------*/
{
	return( jiffies/HZ );
}


//-----------------------------------------------------------------------------
// Deferred procedure call functions

// create a soft interrupt object
/*----------------------------------------------------------------------------*/
int OsSoftInterruptAdd( 
	OS_SOFTINTR **ptr,
	void * handler,
	void * data )
/*----------------------------------------------------------------------------*/
{
	OS_SOFTINTR *tmp_ptr;

	if( !( tmp_ptr = ( OS_SOFTINTR * )kmalloc( sizeof( OS_SOFTINTR ), GFP_KERNEL ) ) )
		return( -1 );
	tmp_ptr->routine = handler;
	tmp_ptr->data = data;
	tmp_ptr->sync = 0;
	INIT_LIST_HEAD(&tmp_ptr->list);

	*ptr = tmp_ptr; 

	return( 0 );
}

/*
	Use kernel_thread( ( int ( * )( void * ) )OsIdleTask, NULL, 0 ); to start
*/
/*----------------------------------------------------------------------------*/
int * OsIdleTask( void * data )
/*----------------------------------------------------------------------------*/
{
	DECLARE_WAITQUEUE (wait, current);

	while( 1 )
	{
		current->state = TASK_INTERRUPTIBLE;
		add_wait_queue( &g_wait_queue_ptr, &wait );
		schedule();
		remove_wait_queue( &g_wait_queue_ptr, &wait );
		wait.task =  current;
		wait.task_list.next = NULL;
	}
	return( NULL );
}


// dispatch a soft interrupt 
/*----------------------------------------------------------------------------*/
void OsSoftInterruptTrigger( 
	OS_SOFTINTR *soft_intr_ptr )
/*----------------------------------------------------------------------------*/
{
	// call the completion routine directly
	soft_intr_ptr->routine( soft_intr_ptr->data );
}


// delete a soft interrupt object
/*----------------------------------------------------------------------------*/
void OsSoftInterruptRemove( 
	OS_SOFTINTR *arg )
/*----------------------------------------------------------------------------*/
{
	if( arg )
		kfree( arg );
	arg = NULL;
}


/*----------------------------------------------------------------------------*/
void OsSleep( 
	unsigned time )		// in seconds
/*----------------------------------------------------------------------------*/
{
	struct semaphore sem;
	struct timer_list timer_var;
	
	init_MUTEX_LOCKED (&sem);

	//	if( in_interrupt() )
	//		panic( "OsSleep going to sleep at interrupt time\n" );

	init_timer( &timer_var );
	timer_var.function = ( void ( * )( unsigned long ) )OsTimeoutHandler;
	timer_var.data = ( unsigned long )&sem;
	timer_var.expires = jiffies + time * HZ;

	add_timer( &timer_var );
	down( &sem );

	del_timer( &timer_var );
}


/*----------------------------------------------------------------------------*/
void OsTimeoutHandler( 
	struct semaphore * sem )
/*----------------------------------------------------------------------------*/
{
	if( sem != NULL )
		up( sem );
}


/*----------------------------------------------------------------------------*/
void printk_err(
	int flag, 
	char *fmt, 
	...)
/*----------------------------------------------------------------------------*/
{
	char	buf[256];
	va_list	ap;

	va_start(ap, fmt);
	(void) vsprintf(buf, fmt, ap);
	va_end(ap);
	
	if( flag <= g_options.message_level )
		printk(KERN_ALERT "%s\n", buf);
}

/*  void aac_show_tasks (struct list_head *our_tasks){ */
 
/*  		cmn_err (CE_DEBUG, "Entering aac_show_tasks"); */

/*  		if (our_tasks->next == NULL || our_tasks->next == 0) */
/*  				cmn_err (CE_DEBUG, "list_head->next is NULL or 0"); */
/*  		else */
/*  				cmn_err (CE_DEBUG, "list_head->next: 0x%x", our_tasks->next); */

/*  		if (our_tasks->prev == NULL || our_tasks->prev == 0) */
/*  				cmn_err (CE_DEBUG, "list_head->prev is NULL or 0"); */
/*  		else */
/*  				cmn_err (CE_DEBUG, "list_head->prev: 0x%x", our_tasks->prev); */

/*  } */
