/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  dpcsup.c
 *
 * Abstract: All DPC processing routines for the cyclone board occur here.
 *
 *
 --*/

static char *ident_dpcsup = "aacraid_ident dpcsup.c 1.0.6 2000/10/09 Adaptec, Inc.";

#include "comprocs.h"


//
//  The Bug check file id for this module
//

#define BugCheckFileId                   (FSAFS_BUG_CHECK_DPCSUP)

#define Dbg                              (DEBUG_TRACE_DPCSUP)

u_int
CommonNotFullDpc(
	IN PCOMM_REGION CommRegion
    )
/*++

Routine Description:

    This DPC routine will be queued when the adapter interrupts us to let us know the queue is
    no longer full. The Isr will pass the queue that we will set the not full event.

Arguments:

    Dpc - Pointer to this routine.

    Dummy - is a pointer to the comm region which is global so we don't need it anyway

    Queue is a pointer to the queue structure we will operate on.

    MoreData2 are DPC parameters we don't need for this function. Maybe we can add some accounting
        stuff in here.

Return Value:
    Nothing.

--*/
{

#ifdef unix_queue_full
    KeSetEvent(&Queue->QueueFull, 0, FALSE);
#endif

}

int GatherFibTimes = 0;

// XXX - hack this in until I figure out which header file should contain it. <smb>
extern ULONG
FibGetMeterSize(
    PFIB pFib,
	ULONG MeterType,
	char SubCommand
	);


/*++

Routine Description:

    This DPC routine will be queued when the adapter interrupts us to let us know there
    is a response on our normal priority queue. We will pull off all QE there are and wake
    up all the waiters before exiting. We will take a spinlock out on the queue before operating
    on it.

Arguments:

    Dpc - Pointer to this routine.

    OurQueue is a pointer to the queue structure we will operate on.

    MoreData1&2 are DPC parameters we don't need for this function. Maybe we can add some accounting
        stuff in here.

Return Value:
    Nothing.

--*/
u_int
HostResponseNormalDpc (IN PCOMM_QUE OurQueue)
{
    PAFA_COMM_ADAPTER Adapter = OurQueue->Adapter;
    PQUEUE_ENTRY QueueEntry;
    PFIB Fib;
	PCOMM_FIB_CONTEXT FibContext;
    int Consumed = 0;
	KIRQL OldIrql;

	LARGE_INTEGER ResponseAllocSize;

#ifdef commdebug
    FsaCommPrint("entering the host normal reponse dpc routine.\n");
#endif

	OsSpinLockAcquire( OurQueue->QueueLock );	

    //
    // Keep pulling response QEs off the response queue and waking
    // up the waiters until there are no more QEs. We then return
    // back to the system. If no response was requesed we just
    // deallocate the Fib here and continue.
    //

 loop:
    while ( GetConsumerEntry( Adapter, OurQueue, &QueueEntry) ) {

		int IsFastResponse;

		IsFastResponse = (int) (QueueEntry->FibAddress & 0x01);
		Fib = (PFIB) (QueueEntry->FibAddress & ~0x01);

		FreeConsumerEntry(Adapter, OurQueue, HostNormRespQueue);

		FibContext = (PCOMM_FIB_CONTEXT)Fib->Header.SenderData;

		ASSERT(FibContext->Fib == Fib);

		//
		// Remove this FibContext from the Outstanding I/O queue.
		// But only if it has not already been timed out.
		//
		// If the fib has been timed out already, then just continue.
		// The caller has already been notified that the fib timed out.
		//

		if (!(FibContext->Flags & FIB_CONTEXT_FLAG_TIMED_OUT)) {

			RemoveEntryList( &FibContext->QueueEntry );
			Adapter->CommRegion->AdapNormCmdQue.NumOutstandingIos--;

		} else {

			FsaCommLogEvent(FibContext,
							FsaCommData.DeviceObject, 
							FSAFS_TIMED_OUT_FIB_COMPLETED,
							STATUS_UNSUCCESSFUL, 
							BugCheckFileId | __LINE__,
							FACILITY_FSAFS_ERROR_CODE,
							NULL,
							TRUE);			

			continue;

		}

		OsSpinLockRelease( OurQueue->QueueLock );

		if (IsFastResponse) {

			//
			// doctor the fib
			//

			*(FSASTATUS *)Fib->data = ST_OK;

			Fib->Header.XferState |= AdapterProcessed;

		}

		ASSERT((Fib->Header.XferState & (AdapterProcessed | HostOwned | SentFromHost)) == (AdapterProcessed | HostOwned | SentFromHost));

		FIB_COUNTER_INCREMENT(FsaCommData.FibRecved);

		ASSERT(FsaCommData.FibsSent >= FsaCommData.FibRecved);


		if (Fib->Header.Command == NuFileSystem) {

			FSASTATUS *pStatus = (FSASTATUS *)Fib->data;

			if (*pStatus & 0xffff0000) {

				ULONG Hint = *pStatus;

				*pStatus = ST_OK;

/*
				DbgPrint("Replacing hint in fid (drive = %d, f1 = 0x%x, f2 = 0x%x, hint = 0x%x, new_hint = 0x%x)\n", 
						 IrpContext->NonPaged->FileId.fid_driveno,
						 IrpContext->NonPaged->FileId.fid_f1,
						 IrpContext->NonPaged->FileId.fid_f2,
						 IrpContext->NonPaged->FileId.fid_hint,
						 Hint);
*/

			}

		}

		if (Fib->Header.XferState & (NoResponseExpected | Async) ) {

			ASSERT(FibContext->FibCallback);

        	if (Fib->Header.XferState & NoResponseExpected)
				FIB_COUNTER_INCREMENT(FsaCommData.NoResponseRecved);
			else 
				FIB_COUNTER_INCREMENT(FsaCommData.AsyncRecved);

			//
			// NOTE:  we can not touch the FibContext after this call, because it may have been
			// deallocated.
			//

			FibContext->FibCallback(FibContext->FibCallbackContext, FibContext, STATUS_SUCCESS);

		} else {

			OsCvLockAcquire( FibContext->FsaEventMutex);

			FibContext->FibComplete = 1;

			OsCv_signal( &FibContext->FsaEvent );

			OsCvLockRelease( FibContext->FsaEventMutex );

			FIB_COUNTER_INCREMENT(FsaCommData.NormalRecved);
			
		}


		Consumed++;

		OsSpinLockAcquire( OurQueue->QueueLock );
		
    }

	if (Consumed > FsaCommData.PeakFibsConsumed)
		FsaCommData.PeakFibsConsumed = Consumed;

	if (Consumed == 0) 
		FsaCommData.ZeroFibsConsumed++;

	if (FsaCommData.HardInterruptModeration) {

		//
		// Re-Enable the interrupt from the adapter, then recheck to see if anything has 
		// been put on the queue.  This removes the race condition that exists between the
		// last time we checked the queue, and when we re-enabled the interrupt.
		//
		// If there is something on the queue, then go handle it.
		//

		EnableInterrupt( Adapter, HostNormRespQue, FALSE );

		if (ConsumerEntryAvailable( Adapter, OurQueue ) ) {

			DisableInterrupt( Adapter, HostNormRespQue, FALSE );

			goto loop;

		}
	}

#ifdef commdebug
    FsaCommPrint("Exiting host normal reponse dpc routine after consuming %d QE(s).\n",Consumed);
#endif

	OsSpinLockRelease( OurQueue->QueueLock );

}

/*++

Routine Description:

    This DPC routine wiol be queued when the adapter interrupts us to let us know there
    is a response on our high priority queue. We will pull off all QE there are and wake
    up all the waiters before exiting. We will take a spinlock out on the queue before operating
    on it.

Arguments:

    Dpc - Pointer to this routine.

    OurQueue is a pointer to the queue structure we will operate on.

    MoreData1&2 are DPC parameters we don't need for this function. Maybe we can add some accounting
        stuff in here.

Return Value:
    Nothing.

--*/
u_int
HostResponseHighDpc (IN PCOMM_QUE OurQueue)
{}


/*++

Routine Description:

    This DPC routine will be queued when the adapter interrupts us to let us know there
    is a command on our high priority queue. We will pull off all QE there are and wake
    up all the waiters before exiting. We will take a spinlock out on the queue before operating
    on it.

Arguments:

    Dpc - Pointer to this routine.

    OurQueue is a pointer to the queue structure we will operate on.

    MoreData1&2 are DPC parameters we don't need for this function. Maybe we can add some accounting
        stuff in here.

Return Value:
    Nothing.

--*/
u_int
HostCommandHighDpc (IN PCOMM_QUE OurQueue)
{}


/*++

Routine Description:

    This DPC routine will be queued when the adapter interrupts us to let us know there
    is a command on our normal priority queue. We will pull off all QE there are and wake
    up all the waiters before exiting. We will take a spinlock out on the queue before operating
    on it.

Arguments:

    Dpc - Pointer to this routine.

    OurQueue is a pointer to the queue structure we will operate on.

    MoreData1&2 are DPC parameters we don't need for this function. Maybe we can add some accounting
        stuff in here.

Return Value:
    Nothing.

--*/
u_int
HostCommandNormDpc (IN PCOMM_QUE OurQueue)
{
    PAFA_COMM_ADAPTER Adapter = OurQueue->Adapter;
    PQUEUE_ENTRY QueueEntry;

	OsSpinLockAcquire( OurQueue->QueueLock );

    //
    // Keep pulling response QEs off the response queue and waking
    // up the waiters until there are no more QEs. We then return
    // back to the system.
    //

    while ( GetConsumerEntry( Adapter, OurQueue, &QueueEntry) ) {

		PFIB Fib;

		Fib = (PFIB)QueueEntry->FibAddress;


		if (Adapter->AifThreadStarted) {


//			cmn_err(CE_CONT, "^Received AIF, putting onto command queue\n");


	        InsertTailList(&OurQueue->CommandQueue, &Fib->Header.FibLinks);
 	        FreeConsumerEntry(Adapter, OurQueue, HostNormCmdQueue);
	        OsCv_signal(&OurQueue->CommandReady);



		} else {



			COMM_FIB_CONTEXT FibContext;

		

 	        FreeConsumerEntry(Adapter, OurQueue, HostNormCmdQueue);



			OsSpinLockRelease( OurQueue->QueueLock );



//			cmn_err(CE_CONT, "^Received AIF, thread not started\n");


			RtlZeroMemory( &FibContext, sizeof(COMM_FIB_CONTEXT) );

		    FibContext.NodeTypeCode = FSAFS_NTC_FIB_CONTEXT;
		    FibContext.NodeByteSize = sizeof( COMM_FIB_CONTEXT );
			FibContext.Fib = Fib;
			FibContext.FibData = Fib->data;
			FibContext.Adapter = Adapter;

			//
			// Set the status of this FIB
			//

			*(FSASTATUS *)Fib->data = ST_OK;
				
			CompleteAdapterFib( &FibContext, sizeof(FSASTATUS) );



			OsSpinLockAcquire( OurQueue->QueueLock );
		}		
    }

	OsSpinLockRelease( OurQueue->QueueLock );

}
