/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  commsup.c
 *
 * Abstract: Contain all routines that are required for FSA host/adapter
 *    commuication.
 *
 *
 --*/

static char *ident_commsup = "aacraid_ident commsup.c 1.0.7 2000/10/11 Adaptec, Inc.";

#include "comprocs.h"

#define BugCheckFileId                   (FSAFS_BUG_CHECK_COMMSUP)

int CommPrinting;

void
ThrottleExceptionHandler(
	IN PCOMM_REGION CommRegion,
	AAC_STATUS		Status
	);

void ThrottlePeriodEndDpcRtn(
    IN PKDPC Dpc,
    IN PVOID DeferredContext,
    IN PVOID SystemArgument1,
    IN PVOID SystemArgument2
    );


/*++

Routine Description:

	This routine will free all resources used by a given FibContextSegment.

Arguments:

	Adapter - The adapter that this COMM_FIB_CONTEXT will communicate with.
	ZoneSegment - The segment to release resources from.

Return Value:

	TRUE - All resources were properly freed.
	FALSE - An Error occured while freeing resources.

--*/
BOOLEAN
FsaFreeFibContextSegment (PAFA_COMM_ADAPTER Adapter,
						  PFIB_CONTEXT_ZONE_SEGMENT	ZoneSegment)
{
	PCOMM_FIB_CONTEXT FibContext;
	int i;
	
	// Account for the ZONE_SEGMENT_HEADER before the first actual FibContext.

	for (i = 0, FibContext = (PCOMM_FIB_CONTEXT)((PUCHAR)ZoneSegment->FibContextSegment + sizeof(ZONE_SEGMENT_HEADER));
		 i < ZoneSegment->ExtendSize; i++, FibContext++) {

		OsCvLockDestroy( FibContext->FsaEventMutex );
		OsCv_destroy( &FibContext->FsaEvent );

	}

	UnmapAndFreeFibSpace( Adapter, &ZoneSegment->MapFibContext );

	OsFreeMemory( ZoneSegment->FibContextSegment, ZoneSegment->FibContextSegmentSize );

	OsFreeMemory( ZoneSegment, sizeof( FIB_CONTEXT_ZONE_SEGMENT ) );

	return (TRUE);
}

BOOLEAN
FsaFreeFibContextZone(
	PAFA_COMM_ADAPTER Adapter
	)
/*++

Routine Description:

	This routine will walk through the FibContextSegmentList and free up all
	resources used by the FibContextZone.

Arguments:

	Adapter - The adapter that this COMM_FIB_CONTEXT will communicate with.

Return Value:

	TRUE - All resources were properly freed.
	FALSE - An Error occured while freeing resources.

--*/

{
	PFIB_CONTEXT_ZONE_SEGMENT ZoneSegment, NextZoneSegment;

	ZoneSegment = Adapter->FibContextSegmentList;

	while (ZoneSegment) {

		NextZoneSegment = ZoneSegment->Next;

		FsaFreeFibContextSegment( Adapter, ZoneSegment );

		ZoneSegment = NextZoneSegment;
	}

	return (TRUE);
}

	

BOOLEAN
FsaExtendFibContextZone (IN PAFA_COMM_ADAPTER Adapter)
{
    int ExtendSize;
    KIRQL SavedIrql;
	ULONG ZoneSegmentAllocSize, FibAllocSize;
	PVOID FibContextSegment;
	PCOMM_FIB_CONTEXT FibContext;
	PFIB Fib;
	PVOID FibPhysicalAddress;
	int i;
	PFIB_CONTEXT_ZONE_SEGMENT ZoneSegment;
	
	//
	// Allocate space to describe this zone segment.
	//

	cmn_err (CE_DEBUG, "Entered FsaExtendFibContextZone");
	ZoneSegment = OsAllocMemory( sizeof( FIB_CONTEXT_ZONE_SEGMENT ), OS_ALLOC_MEM_SLEEP );
	if (ZoneSegment == NULL) {
		return (FALSE);
	}

	ExtendSize = Adapter->FibContextZoneExtendSize;
	ZoneSegmentAllocSize = (ExtendSize * sizeof(COMM_FIB_CONTEXT)) + sizeof(ZONE_SEGMENT_HEADER);

	FibContextSegment = OsAllocMemory( ZoneSegmentAllocSize, OS_ALLOC_MEM_SLEEP );

	if (FibContextSegment == NULL) {
		OsFreeMemory(ZoneSegment);
		return (FALSE);
	}	

	RtlZeroMemory( FibContextSegment, ZoneSegmentAllocSize );

	ZoneSegment->FibContextSegment = FibContextSegment;
	ZoneSegment->FibContextSegmentSize = ZoneSegmentAllocSize;
	ZoneSegment->ExtendSize = ExtendSize;

	FibAllocSize = ExtendSize * sizeof(FIB);


	ZoneSegment->MapFibContext.Size = FibAllocSize;

	AllocateAndMapFibSpace( Adapter, &ZoneSegment->MapFibContext );

	Fib = ZoneSegment->MapFibContext.FibVirtualAddress;
	FibPhysicalAddress = ZoneSegment->MapFibContext.FibPhysicalAddress;

	RtlZeroMemory( Fib, FibAllocSize );

	// Account for the ZONE_SEGMENT_HEADER before the first actual FibContext.

	for (i = 0, FibContext = (PCOMM_FIB_CONTEXT)((PUCHAR)FibContextSegment + sizeof(ZONE_SEGMENT_HEADER));
		 i < ExtendSize; i++, FibContext++) {

		FibContext->Adapter = Adapter;

		FibContext->Fib = Fib;
		FibContext->FibData = (PVOID) FibContext->Fib->data;

		OsCv_init( &FibContext->FsaEvent);
		FibContext->FsaEventMutex = OsCvLockAlloc();
		OsCvLockInit( FibContext->FsaEventMutex, Adapter->SpinLockCookie );

		Fib->Header.XferState = 0xffffffff;
		Fib->Header.SenderSize = sizeof(FIB);

		FibContext->LogicalFibAddress.LowPart = (ULONG) FibPhysicalAddress;

		Fib = (PFIB)((PUCHAR)Fib + sizeof(FIB));
		FibPhysicalAddress = (PVOID)((PUCHAR)FibPhysicalAddress + sizeof(FIB));
	}

	//
	// If FibContextZone.TotalSegmentSize is non-zero, then a zone has already been
	// initialized, we just need to extend it.
	//

	if (Adapter->FibContextZone.TotalSegmentSize) {

		OsSpinLockAcquire( Adapter->FibContextZoneSpinLock );

		ExExtendZone( &Adapter->FibContextZone,
					  FibContextSegment,
					  ZoneSegmentAllocSize );

		OsSpinLockRelease( Adapter->FibContextZoneSpinLock );

	} else {

	    if (ExInitializeZone( &Adapter->FibContextZone,
							  sizeof(COMM_FIB_CONTEXT),
    	                      FibContextSegment,
	                          ZoneSegmentAllocSize ) != STATUS_SUCCESS)
			FsaBugCheck(0,0,0);

	}

	//
	// Add this segment to the adapter's list of segments
	//

	ZoneSegment->Next = Adapter->FibContextSegmentList;
	Adapter->FibContextSegmentList = ZoneSegment;

	return (TRUE);
}



/*++

Routine Description:

    This routine creates a new COMM_FIB_CONTEXT record

Arguments:

	Adapter - The adapter that this COMM_FIB_CONTEXT will communicate with.

Return Value:

    PCOMM_FIB_CONTEXT - returns a pointer to the newly allocate COMM_FIB_CONTEXT Record

--*/
PFIB_CONTEXT
AllocateFib (IN PVOID AdapterArg)
{
	PAFA_COMM_ADAPTER Adapter = (PAFA_COMM_ADAPTER) AdapterArg;
    KIRQL SavedIrql;
    PCOMM_FIB_CONTEXT FibContext;
	int FullZoneLoopCounter = 0;
        

        //
	// Acquire the zone spin lock, and check to see if the zone is full.
	// If it is, then release the spin lock and allocate more fibs for the 
	// zone.  The ExtendFibZone routine will re-acquire the spin lock to add
	// the new fibs onto the zone.
        //

    OsSpinLockAcquire( Adapter->FibContextZoneSpinLock );

    while (ExIsFullZone( &Adapter->FibContextZone )) {

		if (++FullZoneLoopCounter >  10)
			FsaBugCheck(0,0,0);

		OsSpinLockRelease( Adapter->FibContextZoneSpinLock );

                // bmb debug
                cmn_err (CE_DEBUG, "Extending FibContextZone");
		if (FsaExtendFibContextZone(Adapter) == FALSE) {
			return (NULL);
		}

                OsSpinLockAcquire( Adapter->FibContextZoneSpinLock );

	}

    //
	//  At this point we now know that the zone has at least one more
    //  IRP context record available.  So allocate from the zone and
    //  then release the mutex.
    //

    FibContext = (PCOMM_FIB_CONTEXT) ExAllocateFromZone( &Adapter->FibContextZone );

	OsSpinLockRelease( Adapter->FibContextZoneSpinLock );

    //
    //  Set the proper node type code and node byte size
    //

    FibContext->NodeTypeCode = FSAFS_NTC_FIB_CONTEXT;
    FibContext->NodeByteSize = sizeof( COMM_FIB_CONTEXT );

	// 
	// Null out fields that depend on being zero at the start of each I/O
	//

	FibContext->Fib->Header.XferState = 0;
	FibContext->FibCallback = NULL;
	FibContext->FibCallbackContext = NULL;


    //
    //  return and tell the caller
    //

    return ((PFIB_CONTEXT) FibContext);
}


/*++

Routine Description:

    This routine deallocates and removes the specified COMM_FIB_CONTEXT record
    from the Fsafs in memory data structures.  It should only be called
    by FsaCompleteRequest.

Arguments:

   	FibContext - Supplies the COMM_FIB_CONTEXT to remove

Return Value:

    None

--*/
VOID
FreeFib (IN PFIB_CONTEXT Context)
{
    KIRQL SavedIrql;
	PCOMM_FIB_CONTEXT FibContext = Context;

    ASSERT(FibContext->NodeTypeCode == FSAFS_NTC_FIB_CONTEXT);

    OsSpinLockAcquire( FibContext->Adapter->FibContextZoneSpinLock );

	if (FibContext->Flags & FIB_CONTEXT_FLAG_TIMED_OUT) {

		FsaCommData.TimedOutFibs++;

		FibContext->Next = FibContext->Adapter->FibContextTimedOutList;
		FibContext->Adapter->FibContextTimedOutList = FibContext;

	} else {

	    ASSERT(FibContext->Fib->Header.XferState == 0);

		if (FibContext->Fib->Header.XferState != 0) {
				cmn_err(CE_WARN, "FreeFib, XferState != 0, FibContext = 0x%x, XferState = 0x%x\n", 
					 FibContext, FibContext->Fib->Header.XferState);
		}

	    ExFreeToZone( &FibContext->Adapter->FibContextZone, FibContext );

	}	

	OsSpinLockRelease( FibContext->Adapter->FibContextZoneSpinLock );

    //
    //  return and tell the caller
    //

    return;
}


/*++

Routine Description:

    This routine deallocates and removes the specified COMM_FIB_CONTEXT record
    from the Fsafs in memory data structures.  It should only be called
    from the dpc routines to from dpc to free an FibContext from an async or
	no response io

Arguments:

    FibContext - Supplies the COMM_FIB_CONTEXT to remove

Return Value:

    None

--*/
VOID
FreeFibFromDpc (IN PFIB_CONTEXT Context)
{
    PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;

    ASSERT(FibContext->NodeTypeCode == FSAFS_NTC_FIB_CONTEXT);

    OsSpinLockAcquire(FibContext->Adapter->FibContextZoneSpinLock);

	if (FibContext->Flags & FIB_CONTEXT_FLAG_TIMED_OUT) {

		FsaCommData.TimedOutFibs++;

		FibContext->Next = FibContext->Adapter->FibContextTimedOutList;
		FibContext->Adapter->FibContextTimedOutList = FibContext;

	} else {

	    ASSERT(FibContext->Fib->Header.XferState == 0);

		if (FibContext->Fib->Header.XferState != 0) {
				cmn_err(CE_WARN, "FreeFibFromDpc, XferState != 0, FibContext = 0x%x, XferState = 0x%x\n", 
					 FibContext, FibContext->Fib->Header.XferState);
		}


	    ExFreeToZone( &FibContext->Adapter->FibContextZone, FibContext );

	}
		
	OsSpinLockRelease(FibContext->Adapter->FibContextZoneSpinLock);

    //
    //  return and tell the caller
    //

    return;
}


/*++

Routine Description:

    Will initialize a FIB of the requested size.
    
Arguments:

    Fib is a pointer to a location which will receive the address of the allocated
        FIB.

    Size is the size of the Fib to allocate.

Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
AAC_STATUS
InitializeFib (IN PFIB_CONTEXT Context)
{
    PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;
	PFIB Fib = FibContext->Fib;

    Fib->Header.StructType = TFib;
    Fib->Header.Size = sizeof(FIB);
//    if (Fib->Header.XferState & AllocatedFromPool)
//        Fib->Header.XferState = HostOwned | FibInitialized | FibEmpty | AllocatedFromPool;
//    else
        Fib->Header.XferState = HostOwned | FibInitialized | FibEmpty | FastResponseCapable;
    Fib->Header.SenderFibAddress = 0;
    Fib->Header.ReceiverFibAddress = 0;
    Fib->Header.SenderSize = sizeof(FIB);

    return(STATUS_SUCCESS);
}
    

/*++

Routine Description:

    Will allocate and initialize a FIB of the requested size and return a
    pointer to the structure. The size allocated may be larger than the size
    requested due to allocation performace optimizations.
    
Arguments:

    Fib is a pointer to a location which will receive the address of the allocated
        FIB.

    Size is the size of the Fib to allocate.

    JustInitialize is a boolean which indicates a Fib has been allocated most likly in an
        imbedded structure the FS always allocates. So just initiaize it and return.
    
Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
AAC_STATUS
AllocatePoolFib (OUT PFIB *Fib, IN USHORT Size)
{}
    

/*++

Routine Description:

    Will deallocate and return to the free pool the FIB pointed to by the
    caller. Upon return accessing locations pointed to by the FIB parameter
    could cause system access faults.

Arguments:

    Fib is a pointer to the FIB that caller wishes to deallocate.
    
Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
AAC_STATUS
DeallocateFib (PFIB_CONTEXT Context)
{
    PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;
	PFIB Fib = FibContext->Fib;

    if ( Fib->Header.StructType != TFib ) {
        FsaCommPrint("Error CompleteFib called with a non Fib structure.\n");
        return(STATUS_UNSUCCESSFUL);
    }


    Fib->Header.XferState = 0;        
        
    return(STATUS_SUCCESS);

}


AAC_STATUS
GetResponse(
    IN PCOMM_QUE ResponseQueue,
    OUT PFIB Fib
    )
/*++

Routine Description:

    Gets a QE off the requested response queue and gets the response FIB into
    host memory. The FIB may already be in host memory depending on the bus
    interface, or may require the host to DMA it over from the adapter. The routine
    will return the FIB to the caller.

Arguments:

    ResponseQueue - Is the queue the caller wishes to have the response gotten from.
    Fib - Is the Fib which was the response from the adapter

Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if there was no Fib to return to the caller.
    bkpfix - add in all the other possible errors ect

--*/
{
return(STATUS_UNSUCCESSFUL);
}

//
// Commuication primitives define and support the queuing method we use to
// support host to adapter commuication. All queue accesses happen through
// these routines and are the only routines which have a knowledge of the
// how these queues are implemented.
//


/*++

Routine Description:

    With a priority the routine returns a queue entry if the queue has free entries. If the queue
    is full(no free entries) than no entry is returned and the function returns FALSE otherwise TRUE is
    returned.

Arguments:

    Priority is an enumerated type which determines which priority level
        command queue the QE is going to be queued on.

    Entry is a pointer to the address of where to return the address of
        the queue entry from the requested command queue.

    Index is a pointer to the address of where to store the index of the new
        queue entry returned.

	DontInterrupt - We set this true if the queue state is such that we don't
		need to interrupt the adapter for this queue entry.

Return Value:

    TRUE - If a queue entry is returned
    FALSE - If there are no free queue entries on the requested command queue.

--*/
BOOLEAN
GetEntry (IN PAFA_COMM_ADAPTER Adapter, IN QUEUE_TYPES WhichQueue,
		  OUT PQUEUE_ENTRY *Entry, OUT PQUEUE_INDEX Index,
		  OUT ULONG *DontInterrupt)
{
    ULONG QueueOffset;
	BOOLEAN status;
	PCOMM_REGION CommRegion;

	CommRegion = Adapter->CommRegion;

    //
    // All of the queues wrap when they reach the end, so we check to see if they
    // have reached the end and if they have we just set the index back to zero.
    // This is a wrap. You could or off the high bits in all updates but this is
    // a bit faster I think.
    //

    if (WhichQueue == AdapHighCmdQueue) {
        *Index = *(CommRegion->AdapHighCmdQue.Headers.ProducerIndex);

		if (*Index - 2 == *(CommRegion->AdapHighCmdQue.Headers.ConsumerIndex))
			*DontInterrupt = TRUE; 

        if (*Index >= ADAP_HIGH_CMD_ENTRIES)
            *Index = 0;

        if (*Index + 1 == *(CommRegion->AdapHighCmdQue.Headers.ConsumerIndex)) { // Queue is full
			status = FALSE;
			cmn_err(CE_WARN, "Adapter High Command Queue full, %d outstanding",
					CommRegion->AdapHighCmdQue.NumOutstandingIos);
		} else {
	        QueueOffset = sizeof(QUEUE_ENTRY) * (*Index);
	        *Entry = QueueOffset + CommRegion->AdapHighCmdQue.BaseAddress;

			status = TRUE;
		}
    } else if (WhichQueue == AdapNormCmdQueue) {

        *Index = *(CommRegion->AdapNormCmdQue.Headers.ProducerIndex);

		if (*Index - 2 == *(CommRegion->AdapNormCmdQue.Headers.ConsumerIndex))
			*DontInterrupt = TRUE; 

		//
		// If we are at the end of the QUEUE then wrap back to 
		// the beginning.
        //

        if (*Index >= ADAP_NORM_CMD_ENTRIES) 
            *Index = 0; // Wrap to front of the Producer Queue.

		//
        // The IEEE spec says that it the producer is one behind the consumer then
        // the queue is full.
        //       

		ASSERT(*(CommRegion->AdapNormCmdQue.Headers.ConsumerIndex) != 0);

        if (*Index + 1 == *(CommRegion->AdapNormCmdQue.Headers.ConsumerIndex)) { // Queue is full
			cmn_err(CE_WARN, "Adapter Norm Command Queue full, %d outstanding",
					CommRegion->AdapNormCmdQue.NumOutstandingIos);
			status = FALSE;
		} else {        
	       	//
			// The success case just falls through and returns the a valid queue entry.
			//

#ifdef commdebug
	        FsaCommPrint("queue entry = %x.\n",CommRegion->AdapNormCmdQue.BaseAddress + *Index);
	        FsaCommPrint("GetEntry: Index = %d, QueueOffset = %x, Entry = %x, *Entry = %x.\n",
	                     *Index, QueueOffset, Entry, *Entry);
#endif
	        *Entry = CommRegion->AdapNormCmdQue.BaseAddress + *Index;

			status = TRUE;
		}
    } else if (WhichQueue == AdapHighRespQueue) {

        *Index = *(CommRegion->AdapHighRespQue.Headers.ProducerIndex);

		if (*Index - 2 == *(CommRegion->AdapHighRespQue.Headers.ConsumerIndex))
			*DontInterrupt = TRUE; 

        if (*Index >= ADAP_HIGH_RESP_ENTRIES)
            *Index = 0;

        if (*Index + 1 == *(CommRegion->AdapHighRespQue.Headers.ConsumerIndex)) { // Queue is full
			status = FALSE;
			cmn_err(CE_WARN, "Adapter High Resp Queue full, %d outstanding",
					CommRegion->AdapHighRespQue.NumOutstandingIos);
		} else {							
	        *Entry = CommRegion->AdapHighRespQue.BaseAddress + *Index;
    		status = TRUE;
		} 
    } else if (WhichQueue == AdapNormRespQueue) {

        *Index = *(CommRegion->AdapNormRespQue.Headers.ProducerIndex);

		if (*Index - 2 == *(CommRegion->AdapNormRespQue.Headers.ConsumerIndex))
			*DontInterrupt = TRUE; 

		//
		// If we are at the end of the QUEUE then wrap back to 
		// the beginning.
        //

        if (*Index >= ADAP_NORM_RESP_ENTRIES) 
            *Index = 0; // Wrap to front of the Producer Queue.

		//
        // The IEEE spec says that it the producer is one behind the consumer then
        // the queue is full.
        //       

        if (*Index + 1 == *(CommRegion->AdapNormRespQue.Headers.ConsumerIndex)) { // Queue is full
			status = FALSE; 
			cmn_err(CE_WARN, "Adapter Norm Resp Queue full, %d outstanding",
					CommRegion->AdapNormRespQue.NumOutstandingIos);
		} else {        
	       	//
			// The success case just falls through and returns the a valid queue entry.
			//

	        *Entry = CommRegion->AdapNormRespQue.BaseAddress + *Index;

#ifdef commdebug
	        FsaCommPrint("queue entry = %x.\n",CommRegion->AdapNormRespQue.BaseAddress + *Index);
	        FsaCommPrint("GetEntry: Index = %d, Entry = %x, *Entry = %x.\n",*Index, Entry, *Entry);
#endif
			status = TRUE;
		}     
    } else {
		cmn_err(CE_PANIC, "GetEntry: invalid queue %d", WhichQueue);
	}


	return (status);
}
   


#ifdef API_THROTTLE

void ThrottleCheck(
	IN PAFA_COMM_ADAPTER Adapter,
	IN PFIB Fib
	)
/*++

Routine Description:

    This routine implements data I/O throttling. Throttling occurs when
	a CLI FIB is detected. To ensure the CLI responds quickly (the user
	is waiting for the response), this mechanism restricts the queue
	depth of data IOs at the adapter for a period of time (called the
	Throttle Period, default 5 seconds).

    The mechanism uses a counted semaphore to place threads into a wait
	state should there be too many data I/Os outstanding.

	At the start of a throttle period (indicated by the first CLI FIB)
	a timer is started. When the timer expires, new requests can go to
	the adapter freely. Throttled requests gradually drain to the
	adapter as each outstanding throttle I/O completes.

    To avoid hurting regular I/O performance, we use a flag in the FIB
	header to mark FIBs involved in throttling. This means we only need
	take the extra spinlock in the response DPC routine for FIBs who
	were subject to throttling. If no throttling is occurring, the cost
	to the regular code paths is a handful of instructions.

Arguments:

	Adapter - Pointer to per-adapter context. This is used to locate the
			  throttle information for this adapter.
	
	Fib		- Pointer to the header for the fib being sent.

Return Value:

	None.

--*/
{
	PCOMM_REGION CommRegion = Adapter->CommRegion;
	AAC_STATUS	 Status;

	//
	// This routine is called under protection of the queue spinlock.
	// As such we are allowed to check and change the counts for the
	// throttle.
	// Check the FIB. If its not a data operation, send it on without
	// throttle check. If it is a data operation, check for throttle.
	//

	CommRegion->TotalFibs++;							// Keep statistics

	if ((Fib->Header.XferState & ApiFib) != 0) {

		CommRegion->ApiFibs++;							// Keep statistics

		//
		// Its an API fib. If the throttle is not already active,
		// make it so. This will prevent new data Fibs being sent
		// if they exceed the throttle check.
		//

		if (!CommRegion->ThrottleActive) {
			BOOLEAN		 InQue;

			CommRegion->ThrottleActive = TRUE;			// This causes new data I/Os to be throttled

			//
			// Schedule a timer for the throttle active period. When
			// it expires, we'll be called back at routine ThrottleDpcRoutine
			// above. This will signify the throttle active period ended
			// and any waiting threads will be signalled to restart.
			//

			FsaCommPrint("Throttle Period Start - CommRegion: %x\n", CommRegion);
			CommRegion->ThrottleTimerSets++;
			InQue = KeSetTimer( &CommRegion->ThrottleTimer,
								CommRegion->ThrottleTimeout,
								&CommRegion->ThrottleDpc);
			ASSERT(InQue == FALSE);
		}

		return;
	}

	//
	// Its a non-API fib, so subject to throttle checks.
	// The following are exempt from throttling:
	//		o FIBs marked as "throttle exempt" by upper layers.
	//		o I/Os issued from a raised IRQL. We can't suspend
	//		  a thread when at raised IRQL so throttling is exempt.
	//

	if (CommRegion->AdapNormCmdQue.SavedIrql != PASSIVE_LEVEL) {

		CommRegion->NonPassiveFibs++;
		FsaCommPrint("ThrottleCheck: Non-Passive level FIB bypasses throttle: %x\n", Fib);
		return;

	}

	if (CommRegion->ThrottleActive) {

		//
		// Throttle is active.
		// Check if the FIB is a read or write. If so, and its to the
		// file system information area, let it through without throttling.
		//

		if (Fib->Header.Command == ContainerCommand) {
			PBLOCKREAD BlockDisk = (PBLOCKREAD) &Fib->data;

			//
			// *** Note *** We are using read and write command formats
			// interchangably here. This is ok for this purpose as the
			// command is in the same place for both. Read and write command
			// formats are different at higher offsets though.
			//

			if ( ((BlockDisk->Command == VM_CtBlockRead) ||
				  (BlockDisk->Command == VM_CtBlockWrite)) &&
				  (BlockDisk->BlockNumber <= FILESYSTEM_INFO_MAX_BLKNO)) {

				CommRegion->FSInfoFibs++;							// Keep statistics
				return;

			}

		}

		//
		// Throttle the FIB.
		// Mark it as throttle active so that it can signal a waiter
		// when it completes.

		CommRegion->ThrottledFibs++;
		Fib->Header.Flags |= ThrottledFib;
		
		//
		// Release the spinlock so we can wait the thread if necessary.
		// Since we specify a timeout, check the caller is at passive level.
		//

		OsSpinLockRelease((CommRegion->AdapNormCmdQue.QueueLock), CommRegion->AdapNormCmdQue.SavedIrql);

		FsaCommPrint("ThrottleCheck - Thread Suspension - FIB: %x\n", Fib);

		Status = KeWaitForSingleObject(&CommRegion->ThrottleReleaseSema,
										Executive,							// Don't allow user APCs to wake us
										KernelMode,							// Wait in kernel mode
										FALSE,								// Not alertable
										&CommRegion->ThrottleWaitTimeout);	// Timeout after this time

		//
		// Check the signal status. If we've timed out, clear the throttle
		// flag on the FIB to avoid us signalling the semaphore on completion.
		// We never acquired the semaphore.
		//
		if (Status == STATUS_TIMEOUT) {

			CommRegion->ThrottleTimedoutFibs++;
			FsaCommPrint("ThrottledFib Timed Out - FIB: %x\n", Fib);
			Fib->Header.Flags &= ~ThrottledFib;						// Clear the throttledfib flag

		} else {

			ASSERT(Status == STATUS_SUCCESS);						// No other return is possible

		}

		//
		// We've been woken up and can now send the FIB to the adapter.
		// Acquire the spinlock again so we can get a queue entry. This
		// returns to GetQueueEntry.
		//

		FsaCommPrint("ThrottleCheck - Thread Resume - FIB: %x\n", Fib);
		KeAcquireSpinLock((CommRegion->AdapNormCmdQue.QueueLock), &(CommRegion->AdapNormCmdQue.SavedIrql));
		CommRegion->ThrottleOutstandingFibs++;		// There's another throttle controlled FIB going.
		return;

	}
}

#endif //#ifdef API_THROTTLE

int GetQueueEntryTimeouts = 0;


/*++

Routine Description:

    Gets the next free QE off the requested priorty adapter command queue and
    associates the Fib with the QE. The QE represented by index is ready to
     insert on the queue when this routine returns success.

Arguments:

    Index is the returned value which represents the QE which is ready to
        insert on the adapter's command queue.

    Priority is an enumerated type which determines which priority level
        command queue the QE is going to be queued on.

    Fib is a pointer to the FIB the caller wishes to have associated with the
        QE.

    Wait is a boolean which determines if the routine will wait if there are
        no free QEs on the requested priority command queue.

    FibContext is where the driver stores all system resources required to execute the
        command requested from the calling thread. This includes mapping resources for
        the FIB and the 'users' buffer.

	DontInterrupt - We set this true if the queue state is such that we don't
		need to interrupt the adapter for this queue entry.

Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
AAC_STATUS
GetQueueEntry (IN PAFA_COMM_ADAPTER Adapter, OUT PQUEUE_INDEX Index,
			   IN QUEUE_TYPES WhichQueue, IN PFIB Fib, IN BOOLEAN Wait,
			   IN PCOMM_FIB_CONTEXT FibContext, OUT ULONG *DontInterrupt)
{
    PQUEUE_ENTRY QueueEntry = NULL;
    BOOLEAN MapAddress = FALSE;
	int timeouts = 0;
	AAC_STATUS Status;
	PCOMM_REGION CommRegion;

	CommRegion = Adapter->CommRegion;

    //
    // Get the spinlock for the queue we are putting a command on
    //

    if (WhichQueue == AdapHighCmdQueue) 
        OsSpinLockAcquire(CommRegion->AdapHighCmdQue.QueueLock);
    else if (WhichQueue == AdapNormCmdQueue)
        OsSpinLockAcquire(CommRegion->AdapNormCmdQue.QueueLock);
    else if (WhichQueue == AdapHighRespQueue)
        OsSpinLockAcquire(CommRegion->AdapHighRespQue.QueueLock);
    else if (WhichQueue == AdapNormRespQueue)
        OsSpinLockAcquire(CommRegion->AdapNormRespQue.QueueLock);
    else {
        FsaCommPrint("Invalid queue priority passed to GetQueueEntry.\n");
        return(FSA_INVALID_QUEUE);
    }
    
    //
    // Get the pointers to a queue entry on the queue the caller wishes to queue
    // a command request on. If there are no entries then wait if that is what the
    // caller requested. 
    //

    if (WhichQueue == AdapHighCmdQueue) {
	  // if no entries wait for some if caller wants to
        while ( !GetEntry(Adapter, AdapHighCmdQueue, &QueueEntry, Index, DontInterrupt) ) { 
			cmn_err(CE_PANIC, "GetEntries failed (1)\n");
		}

        //
        // Setup queue entry with a command, status and Fib mapped
        //

        QueueEntry->Size = Fib->Header.Size;
        MapAddress = TRUE;
	
    } else if (WhichQueue == AdapNormCmdQueue) {
	  // if no entries wait for some if caller wants to
        while ( !GetEntry(Adapter, AdapNormCmdQueue, &QueueEntry, Index, DontInterrupt) ) { 
			cmn_err(CE_PANIC, "GetEntries failed (2)\n");
		}
 
        //
        // Setup queue entry with command, status and Fib mapped
        //

        QueueEntry->Size = Fib->Header.Size;
        MapAddress = TRUE;
        
     } else if (WhichQueue == AdapHighRespQueue) {

        while ( !GetEntry(Adapter, AdapHighRespQueue, &QueueEntry, Index, DontInterrupt) ) { // if no entries wait for some if caller wants to
		}

        //
        // Setup queue entry with command, status and Fib mapped
        //

        QueueEntry->Size = Fib->Header.Size;
        QueueEntry->FibAddress = Fib->Header.SenderFibAddress;     			// Restore adapters pointer to the FIB
		Fib->Header.ReceiverFibAddress = Fib->Header.SenderFibAddress;		// Let the adapter now where to find its data
        MapAddress = FALSE;
        
     } else if (WhichQueue == AdapNormRespQueue) {
        while ( !GetEntry(Adapter, AdapNormRespQueue, &QueueEntry, Index, DontInterrupt) ) { // if no entries wait for some if caller wants to
		}

		//
		// Setup queue entry with command, status, adapter's pointer to the Fib it sent
		//
	
        QueueEntry->Size = Fib->Header.Size;
        QueueEntry->FibAddress = Fib->Header.SenderFibAddress;     			// Restore adapters pointer to the FIB
		Fib->Header.ReceiverFibAddress = Fib->Header.SenderFibAddress;		// Let the adapter now where to find its data
        MapAddress = FALSE;
     }
                
    //
    // If MapFib is true than we need to map the Fib and put pointers in the queue entry.
    //

    if (MapAddress) {
		QueueEntry->FibAddress = (ULONG)(FibContext->LogicalFibAddress.LowPart);
    }
    
    //
    // Return
    //
#ifdef commdebug    
    FsaCommPrint("Queue Entry contents:.\n");
    FsaCommPrint("  Command =               %d.\n", QueueEntry->Command);
    FsaCommPrint("  Status  =               %x.\n", QueueEntry->Status);
    FsaCommPrint("  Rec Fib address low =   %x.\n", QueueEntry->FibAddressLow);        
    FsaCommPrint("  Fib size in bytes =     %d.\n", QueueEntry->Size);
#endif

    return(FSA_SUCCESS);
}


/*++

Routine Description:

    Gets the next free QE off the requested priorty adapter command queue and
      associates the Fib with the QE. The QE represented by index is ready to
    insert on the queue when this routine returns success.

Arguments:

    Index is the returned value which represents the QE which is ready to
        insert on the adapter's command queue.

    WhichQueue tells us which queue the caller wishes to have the entry put.
        
Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
AAC_STATUS
InsertQueueEntry(
                 IN PAFA_COMM_ADAPTER Adapter,
                 IN QUEUE_INDEX Index,
                 IN QUEUE_TYPES WhichQueue,
                 IN ULONG DontInterrupt
                 )
{
	PCOMM_REGION CommRegion;
    
	CommRegion = Adapter->CommRegion;

    //
    // We have already verified the queue in getentry, but we still have to make
    // sure we don't wrap here too.
    //

    if (WhichQueue == AdapHighCmdQueue) {

        *(CommRegion->AdapHighCmdQue.Headers.ProducerIndex) = Index + 1;
            
        OsSpinLockRelease(CommRegion->AdapHighCmdQue.QueueLock);

		if (!DontInterrupt)
	        NotifyAdapter(Adapter, AdapHighCmdQue);
        
    } else if (WhichQueue == AdapNormCmdQueue) {

#ifdef commdebug
        FsaCommPrint("InsertQueueEntry: Inerting with an index of %d.\n",Index);
#endif
        *(CommRegion->AdapNormCmdQue.Headers.ProducerIndex) = Index + 1;
	
        OsSpinLockRelease(CommRegion->AdapNormCmdQue.QueueLock);

		if (!DontInterrupt)
	        NotifyAdapter(Adapter, AdapNormCmdQue);

    } else if (WhichQueue == AdapHighRespQueue) {

        *(CommRegion->AdapHighRespQue.Headers.ProducerIndex) = Index + 1;

        OsSpinLockRelease(CommRegion->AdapHighRespQue.QueueLock);

		if (!DontInterrupt)
	        NotifyAdapter(Adapter, AdapHighRespQue);

    } else if (WhichQueue == AdapNormRespQueue) {

	    *(CommRegion->AdapNormRespQue.Headers.ProducerIndex) = Index + 1;
	    
	    OsSpinLockRelease(CommRegion->AdapNormRespQue.QueueLock);

	    if (!DontInterrupt)
		    NotifyAdapter(Adapter, AdapNormRespQue);

    } else {        
        FsaCommPrint("Invalid queue priority passed to InsertQueueEntry.\n");
        return(FSA_INVALID_QUEUE_PRIORITY);
    }

    return(FSA_SUCCESS);                
}

extern int GatherFibTimes;

BOOLEAN
SendSynchFib(
	PVOID			Arg,
	FIB_COMMAND  	Command,
	PVOID			Data,
	USHORT		 	Size,
	PVOID			Response,
	USHORT		 	*ResponseSize
	)
/*++

Routine Description:

	This routine will send a synchronous FIB to the adapter and wait for its
	completion.

Arguments:

	DeviceExtension - Pointer to adapter extension structure.


Return Value:

	BOOLEAN

--*/
{
	PAFA_COMM_ADAPTER Adapter = Arg;
	FIB *Fib;
	ULONG returnStatus;

	Fib = Adapter->SyncFib;

    Fib->Header.StructType = TFib;
    Fib->Header.Size = sizeof(FIB);
    Fib->Header.XferState = HostOwned | FibInitialized | FibEmpty;
    Fib->Header.ReceiverFibAddress = 0;
    Fib->Header.SenderSize = sizeof(FIB);
    Fib->Header.SenderFibAddress = (ULONG)Fib;
    Fib->Header.Command = Command;

	//
	// Copy the Data portion into the Fib.
	//

	RtlCopyMemory( Fib->data, Data, Size );


    Fib->Header.XferState |= (SentFromHost | NormalPriority);
    
	//
    // Set the size of the Fib we want to send to the adapter
	//

    Fib->Header.Size = sizeof(FIB_HEADER) + Size;

	if (!Adapter->AdapterFuncs->SendSynchFib( Adapter->AdapterExtension,
				  							  Adapter->SyncFibPhysicalAddress )) {

			return (FALSE);

	}

	//
	// Copy the response back to the caller's buffer.
	//

	RtlCopyMemory( Response, Fib->data, Fib->Header.Size - sizeof(FIB_HEADER) );

	*ResponseSize = Fib->Header.Size - sizeof(FIB_HEADER);

	//
	// Indicate success
	//

	return (TRUE);
}

//
// Define the highest level of host to adapter communication routines. These
// routines will support host to adapter FS commuication. These routines have
// no knowledge of the commuication method used. This level sends and receives
// FIBs. This level has no knowledge of how these FIBs get passed back and forth.
//



/*++

Routine Description:

    Sends the requested FIB to the adapter and optionally will wait for a
     response FIB. If the caller does not wish to wait for a response than
    an event to wait on must be supplied. This event will be set when a
    response FIB is received from the adapter.

Arguments:

    Fib is a pointer to the FIB the caller wishes to send to the adapter.
    
    Size - Size of the data portion of the Fib.
    
    Priority is an enumerated type which determines which priority level
        the caller wishes to send this command at. 

    Wait is a boolean which determines if the routine will wait for the
        completion Fib to be returned(TRUE), or return when the Fib has been
        successfully received by the adapter(FALSE).

    WaitOn is only vaild when Wait is FALSE. The Event will be set when the response
        FIB has been returned by the adapter.

    ReturnFib is an optional pointer to a FIB that if present the response FIB will
        copied to.     
        
Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

	--*/
AAC_STATUS
SendFib (IN FIB_COMMAND Command,
         IN PFIB_CONTEXT Context,
         IN ULONG Size, 
         IN COMM_PRIORITIES Priority,
         IN BOOLEAN Wait,
         IN PVOID WaitOn,
         IN BOOLEAN ResponseExpected,
         IN PFIB_CALLBACK FibCallback,
         IN PVOID FibCallbackContext)
{
		PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;
		QUEUE_INDEX Index;
		QUEUE_TYPES WhichQueue;
		LARGE_INTEGER Timeout;
		AAC_STATUS Status;
		PAFA_COMM_ADAPTER Adapter = FibContext->Adapter;
		ULONG DontInterrupt = FALSE;
		PFIB Fib = FibContext->Fib;
		IN PCOMM_QUE OurQueue;

		Timeout = FsaCommData.AdapterTimeout;

		if (!(Fib->Header.XferState & HostOwned)) {
				FsaCommPrint("SendFib was called with a xfer state not set to HostOwned!\n");
				FsaCommLogEvent(FibContext,
								FsaCommData.DeviceObject, 
								FSAFS_FIB_INVALID, 
								STATUS_UNSUCCESSFUL, 
								BugCheckFileId | __LINE__,
								FACILITY_FSAFS_ERROR_CODE,
								NULL,
								TRUE);			

				return(STATUS_UNSUCCESSFUL);

		}
    
		//
		// There are 5 cases with the wait and reponse requested flags. The only invalid cases
		// are if the caller requests to wait and  does not request a response and if the
		// caller does not want a response and the Fib is not allocated from pool. If a response
		// is not requesed the Fib will just be deallocaed by the DPC routine when the response
		// comes back from the adapter. No further processing will be done besides deleting the
		// Fib. We will have a debug mode where the adapter can notify the host it had a problem
		// and the host can log that fact.

		if (Wait && !ResponseExpected) {

				FsaCommLogEvent(FibContext,
                                                FsaCommData.DeviceObject, 
                                                FSAFS_FIB_INVALID, 
                                                STATUS_UNSUCCESSFUL, 
                                                BugCheckFileId | __LINE__,
                                                FACILITY_FSAFS_ERROR_CODE,
                                                NULL,
                                                TRUE);			

				return(STATUS_UNSUCCESSFUL);

		} else if (!Wait && ResponseExpected) {
				Fib->Header.XferState |= (Async | ResponseExpected);
				FIB_COUNTER_INCREMENT(FsaCommData.AsyncSent);
		} else if (!Wait && !ResponseExpected) {
				Fib->Header.XferState |= NoResponseExpected;
				FIB_COUNTER_INCREMENT(FsaCommData.NoResponseSent);
		} else if (Wait && ResponseExpected) {
                  Fib->Header.XferState |= ResponseExpected;
                  FIB_COUNTER_INCREMENT(FsaCommData.NormalSent);
		} 

		Fib->Header.SenderData = (ULONG)FibContext; // so we can complete the io in the dpc routine

		//
		// Set FIB state to indicate where it came from and if we want a response from the
		// adapter. Also load the command from the caller.
		//

		Fib->Header.SenderFibAddress = (ULONG)Fib;
		Fib->Header.Command = Command;
		Fib->Header.XferState |= SentFromHost;
		FibContext->Fib->Header.Flags = 0;				// Zero the flags field - its internal only...
    
		//
		// Set the size of the Fib we want to send to the adapter
		//

		Fib->Header.Size = sizeof(FIB_HEADER) + Size;
		if (Fib->Header.Size > Fib->Header.SenderSize) {
				return(STATUS_BUFFER_OVERFLOW);
		}                

		//
		// Get a queue entry connect the FIB to it and send an notify the adapter a command is ready.
		//
            
		if (Priority == FsaHigh) {
				Fib->Header.XferState |= HighPriority;
				WhichQueue = AdapHighCmdQueue;
				OurQueue = &Adapter->CommRegion->AdapHighCmdQue;
		} else {
				Fib->Header.XferState |= NormalPriority;
				WhichQueue = AdapNormCmdQueue;
				OurQueue = &Adapter->CommRegion->AdapNormCmdQue;
		}

		if (Wait) {
				OsCvLockAcquire( FibContext->FsaEventMutex );
		}

		if ( GetQueueEntry( Adapter, &Index, WhichQueue, Fib, TRUE, FibContext, &DontInterrupt) != FSA_SUCCESS )
				return(STATUS_UNSUCCESSFUL);

		// bmb debug

		cmn_err (CE_DEBUG,"SendFib: inserting a queue entry at index %d.\n",Index);
		cmn_err (CE_DEBUG,"Fib contents:.\n");
		cmn_err (CE_DEBUG,"  Command =               %d.\n", Fib->Header.Command);
		cmn_err (CE_DEBUG,"  XferState  =            %x.\n", Fib->Header.XferState );

		//
		// Fill in the Callback and CallbackContext if we are not going to wait.
		//

		if (!Wait) {

				FibContext->FibCallback = FibCallback;
				FibContext->FibCallbackContext = FibCallbackContext;

		}

		FIB_COUNTER_INCREMENT(FsaCommData.FibsSent);

		InsertTailList( &OurQueue->OutstandingIoQueue, &FibContext->QueueEntry );
		OurQueue->NumOutstandingIos++;

		FibContext->FibComplete = 0;



		if ( InsertQueueEntry( Adapter, Index, WhichQueue, (DontInterrupt & FsaCommData.EnableInterruptModeration)) != FSA_SUCCESS )
                        return(STATUS_UNSUCCESSFUL);

		//
		// If the caller wanted us to wait for response wait now. 
		// If Timeouts are enabled than set the timeout otherwise wait forever.
		//
    
		if (Wait) {
                        while (FibContext->FibComplete == 0) {
                                OsCv_wait( &FibContext->FsaEvent, FibContext->FsaEventMutex );
                        }	
                        
                        OsCvLockRelease( FibContext->FsaEventMutex );
					
                        if ( (FibContext->Flags & FIB_CONTEXT_FLAG_TIMED_OUT) ) {
                                return(STATUS_IO_TIMEOUT);
                        } else {
                                return(STATUS_SUCCESS);
                        }
		}

		//
		// If the user does not want a response than return success otherwise return pending
		// 

		ASSERT( FibCallback );

		if (ResponseExpected)
				return(STATUS_PENDING);
		else
				return(STATUS_SUCCESS);
}

BOOLEAN
GetConsumerEntry(
	IN PAFA_COMM_ADAPTER Adapter,
    PCOMM_QUE OurQueue,
    OUT PQUEUE_ENTRY *Entry
    )
/*++

Routine Description:

    Will return a pointer to the entry on the top of the queue requested that we are a consumer
    of, and return the address of the queue entry. It does not change the state of the queue. 

Arguments:

    OurQueue - is the queue the queue entry should be removed from.

    Entry - is a pointer where the address  of the queue entry should be returned.    
    
Return Value:

    TRUE if there was a queue entry on the response queue for the host to consume.
    FALSE if there were no queue entries to consume.
    
--*/

{
    QUEUE_INDEX Index;
	BOOLEAN status;

    if (*OurQueue->Headers.ProducerIndex == *OurQueue->Headers.ConsumerIndex) {
		status = FALSE;
	} else {

	    //
	    // The consumer index must be wrapped if we have reached the end of
	    // the queue. 
	    // Else we just use the entry pointed to by the header index
	    //
	    
	    if (*OurQueue->Headers.ConsumerIndex >= OurQueue->QueueEntries) 
			Index = 0;		
	    else
	        Index = *OurQueue->Headers.ConsumerIndex;
	    
	    *Entry = OurQueue->BaseAddress + Index;

#ifdef commdebug
	    FsaCommPrint("Got a QE at Index %d, QE Addrss of %x.\n",Index,*Entry);
#endif
		status = TRUE;
	}

    return(status);
}

BOOLEAN
ConsumerEntryAvailable(
	IN PAFA_COMM_ADAPTER Adapter,
    PCOMM_QUE OurQueue
	)
{
    return (*OurQueue->Headers.ProducerIndex != *OurQueue->Headers.ConsumerIndex);
}

VOID
FreeConsumerEntry(
	IN PAFA_COMM_ADAPTER Adapter,
    PCOMM_QUE OurQueue,
    QUEUE_TYPES WhichQueue
    )
/*++

Routine Description:

    Frees up the current top of the queue we are a consumer of. If the queue was full
    notify the producer that the queue is no longer full.

Arguments:

    OurQueue - is the queue we will free the current consumer entry on.

Return Value:

    TRUE if there was a queue entry on the response queue for the host to consume.
    FALSE if there were no queue entries to consume.
    
--*/

{
    BOOLEAN WasFull = FALSE;
    HOST_2_ADAP_EVENT Notify;

    if (*OurQueue->Headers.ProducerIndex+1 == *OurQueue->Headers.ConsumerIndex)
        WasFull = TRUE;
        
    if (*OurQueue->Headers.ConsumerIndex >= OurQueue->QueueEntries)
        *OurQueue->Headers.ConsumerIndex = 1;
    else
        *OurQueue->Headers.ConsumerIndex += 1;
        
    if (WasFull) {
        switch (WhichQueue) {

            case HostNormCmdQueue:
                Notify = HostNormCmdNotFull;
                break;
            case HostHighCmdQueue:
                Notify = HostHighCmdNotFull;
                break;

            case HostNormRespQueue:
                Notify = HostNormRespNotFull;
                break;

            case HostHighRespQueue:
                Notify = HostHighRespNotFull;
                break;

        }
        NotifyAdapter(Adapter, Notify);
    }

}        

AAC_STATUS
CompleteAdapterFib(
	IN PFIB_CONTEXT Context,
    IN USHORT Size
    )
/*++

Routine Description:

    Will do all necessary work to complete a FIB that was sent from the adapter.

Arguments:

    Fib is a pointer to the FIB that caller wishes to complete processing on. 

    Size - Size of the completion Packet(Opitional). If not present than the current
           largest size in the Fib will be used
    
	Adapter - Pointer to which adapter sent this FIB

Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
{
	PCOMM_FIB_CONTEXT FibContext = Context;
    PFIB Fib = FibContext->Fib;
	PAFA_COMM_ADAPTER Adapter = FibContext->Adapter;
    ULONG DontInterrupt = FALSE;

    if (Fib->Header.XferState == 0)
        return(STATUS_SUCCESS);

    //
    // If we plan to do anything check the structure type first.
    // 

    if ( Fib->Header.StructType != TFib ) {
        FsaCommPrint("Error CompleteFib called with a non Fib structure.\n");
        return(STATUS_UNSUCCESSFUL);
    }

    //
    // This block handles the case where the adapter had sent us a command and we
    // have finished processing the command. We call completeFib when we are done
    // processing the command and want to send a response back to the adapter. This
    // will send the completed cdb to the adapter.
    //

    if (Fib->Header.XferState & SentFromAdapter) {
        Fib->Header.XferState |= HostProcessed;
        if (Fib->Header.XferState & HighPriority) {
            QUEUE_INDEX Index;
            
            if (Size) {
                Size += sizeof(FIB_HEADER);
                if (Size > Fib->Header.SenderSize) 
                    return(STATUS_BUFFER_OVERFLOW);
                Fib->Header.Size = Size;
            }

            if (GetQueueEntry(Adapter, &Index, AdapHighRespQueue, Fib, TRUE, NULL, &DontInterrupt) != STATUS_SUCCESS) {
                FsaCommPrint("CompleteFib got an error geting a queue entry for a response.\n");
                return(FSA_FATAL);
            }
            if (InsertQueueEntry(Adapter, 
            					Index, 
            					AdapHighRespQueue, 
            					(DontInterrupt & (BOOLEAN)FsaCommData.EnableInterruptModeration)) != STATUS_SUCCESS) {
                FsaCommPrint("CompleteFib failed while inserting entry on the queue.\n");
            }
        } else if (Fib->Header.XferState & NormalPriority) {
            QUEUE_INDEX Index;

            if (Size) {
                Size += sizeof(FIB_HEADER);
                if (Size > Fib->Header.SenderSize) 
                    return(STATUS_BUFFER_OVERFLOW);
                Fib->Header.Size = Size;
            }
            
            if (GetQueueEntry(Adapter, &Index, AdapNormRespQueue, Fib, TRUE, NULL, &DontInterrupt) != STATUS_SUCCESS) {
                FsaCommPrint("CompleteFib got an error geting a queue entry for a response.\n");
                return(FSA_FATAL);
            }
            if (InsertQueueEntry(Adapter, 
            					Index, 
            					AdapNormRespQueue, 
            					(DontInterrupt & (BOOLEAN)FsaCommData.EnableInterruptModeration)) != STATUS_SUCCESS) {
                FsaCommPrint("CompleteFib failed while inserting entry on the queue.\n");
            }
		}
    } else {
        cmn_err(CE_WARN, "CompleteFib: Unknown xferstate detected.\n");
		FsaBugCheck(0,0,0);
    }   
    return(STATUS_SUCCESS);
}

AAC_STATUS
CompleteFib(
	IN PFIB_CONTEXT Context
    )
/*++

Routine Description:

    Will do all necessary work to complete a FIB. If the caller wishes to
    reuse the FIB after post processing has been completed Reinitialize
    should be called set to TRUE, otherwise the FIB will be returned to the
    free FIB pool. If Reinitialize is set to TRUE then the FIB header is
    reinitialzied and is ready for reuse on return from this routine.

Arguments:

    Fib is a pointer to the FIB that caller wishes to complete processing on. 

    Size - Size of the completion Packet(Opitional). If not present than the current
           largest size in the Fib will be used
    
    Reinitialize is a boolean which determines if the routine will ready the
        completed FIB for reuse(TRUE) or not(FALSE).

Return Value:

    NT_SUCCESS if a Fib was returned to the caller.
    NT_ERROR if event was an invalid event. 

--*/
{
    PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;
	PAFA_COMM_ADAPTER Adapter = FibContext->Adapter;
	PFIB Fib = FibContext->Fib;

    //
    // Check for a fib which has already been completed
    //

//	ASSERT(Fib->Header.XferState & AdapterProcessed);
    if (Fib->Header.XferState == 0)
        return(STATUS_SUCCESS);

    //
    // If we plan to do anything check the structure type first.
    // 

    if ( Fib->Header.StructType != TFib ) {
        FsaCommPrint("Error CompleteFib called with a non Fib structure.\n");
        return(STATUS_UNSUCCESSFUL);
    }

#if 0
//#if FSA_ADAPTER_METER
	//
	// Meter the completion
	//
	fsaMeterEnd(						// meter the end of an operation
		&(Adapter->FibMeter),			// .. the meter
		IrpContext->FibMeterType,		// .. type of operation
		&(IrpContext->FibStartTime),	// .. ptr to operation start timestamp
		FibGetMeterSize(Fib,			// .. number of bytes in operation
				IrpContext->FibMeterType, 
				IrpContext->FibSubCommand));
#endif // FSA_ADAPTER_METER
	
    //
    // This block completes a cdb which orginated on the host and we just need
    // to deallocate the cdb or reinit it. At this point the command is complete
    // that we had sent to the adapter and this cdb could be reused.
    //
	
    if ( (Fib->Header.XferState & SentFromHost) &&
         (Fib->Header.XferState & AdapterProcessed)) {
        
        ASSERT(FibContext->LogicalFibAddress.LowPart != 0);

        return( DeallocateFib(FibContext) ); 
	
    //
    // This handles the case when the host has aborted the I/O to the
    // adapter because the adapter is not responding
    //

    } else if (Fib->Header.XferState & SentFromHost) {

        ASSERT(FibContext->LogicalFibAddress.LowPart != 0);


        return( DeallocateFib(FibContext) ); 

    } else if (Fib->Header.XferState & HostOwned) {

        return(DeallocateFib(FibContext));

    } else {
        cmn_err(CE_WARN, "CompleteFib: Unknown xferstate detected.\n");
		FsaBugCheck(0,0,0);
    }   
    return(STATUS_SUCCESS);
}

VOID
HandleDriverAif(
    IN PAFA_COMM_ADAPTER Adapter,
	IN PCOMM_FIB_CONTEXT FibContext
    )
/*++

Routine Description:

	This routine handles a driver notify fib from the adapter and dispatches it to 
	the appropriate routine for handling.

Arguments:

	Adapter - Which adapter this fib is from
	FibContext - Pointer to FibContext from adapter.
    
Return Value:

    Nothing.
    
--*/
{
	PFIB Fib = FibContext->Fib;
	PAFA_CLASS_DRIVER ClassDriver;
	BOOLEAN Handled = FALSE;


	//
	// First loop through all of the class drivers to give them a chance to handle
	// the Fib.
	//

	ClassDriver = Adapter->ClassDriverList;

	while (ClassDriver) {

		if (ClassDriver->HandleAif) {

			if (ClassDriver->HandleAif( ClassDriver->ClassDriverExtension, FibContext ) ) {

				Handled = TRUE;
				break;

			}
		}

		ClassDriver = ClassDriver->Next;
	}

	if (!Handled) {

		//
		// Set the status of this FIB to be Invalid parameter.
		//

//		*(FSASTATUS *)Fib->data = ST_INVAL;
		*(FSASTATUS *)Fib->data = ST_OK;


		CompleteAdapterFib(FibContext, sizeof(FSASTATUS));

	}
}

int
NormCommandThread(
    IN PAFA_COMM_ADAPTER Adapter
    )
/*++

Routine Description:

    Waits on the commandready event in it's queue. When the event gets set it will
    pull FIBs off it's queue. It will continue to pull FIBs off till the queue is empty.
    When the queue is empty it will wait for more FIBs.

Arguments:

    Context is used. All data os global
    
Return Value:
    Nothing.
    
--*/
{
    PFIB Fib, NewFib;
	COMM_FIB_CONTEXT FibContext; // for error logging
    KIRQL SavedIrql;
	PCOMM_REGION CommRegion = Adapter->CommRegion;
	PLIST_ENTRY Entry;
	PGET_ADAPTER_FIB_CONTEXT AdapterFibContext;

	//
	// We can only have one thread per adapter for AIF's.
	//

	if (Adapter->AifThreadStarted) {
		return (EINVAL);
	}

// cmn_err(CE_DEBUG, "AIF thread started");

	//
	// Let the DPC know it has a place to send the AIF's to.
	//

	Adapter->AifThreadStarted = TRUE;

	RtlZeroMemory(&FibContext, sizeof(COMM_FIB_CONTEXT));

	OsSpinLockAcquire(CommRegion->HostNormCmdQue.QueueLock);

    while (TRUE) {

		//
		// NOTE : the QueueLock is held at the top of each loop.
		//

		ASSERT(OsSpinLockOwned(CommRegion->HostNormCmdQue.QueueLock));

		while (!IsListEmpty(&(CommRegion->HostNormCmdQue.CommandQueue))) {
			PLIST_ENTRY Entry;
			PAIFCOMMANDTOHOST AifCommandToHost;

			Entry = RemoveHeadList(&(CommRegion->HostNormCmdQue.CommandQueue));

			OsSpinLockRelease(CommRegion->HostNormCmdQue.QueueLock);

			Fib = CONTAINING_RECORD( Entry, FIB, Header.FibLinks );
						
			//
			// We will process the FIB here or pass it to a worker thread that is TBD. We Really
			// can't do anything at this point since we don't have anything defined for this thread to
			// do.
			//
			
			// cmn_err(CE_DEBUG, "Got Fib from the adapter with a NORMAL priority, command 0x%x.\n", Fib->Header.Command);

			RtlZeroMemory( &FibContext, sizeof(COMM_FIB_CONTEXT) );


		    FibContext.NodeTypeCode = FSAFS_NTC_FIB_CONTEXT;
		    FibContext.NodeByteSize = sizeof( COMM_FIB_CONTEXT );
			FibContext.Fib = Fib;
			FibContext.FibData = Fib->data;
			FibContext.Adapter = Adapter;

			
			//
			// We only handle AifRequest fibs from the adapter.
			//

			ASSERT(Fib->Header.Command == AifRequest);


			AifCommandToHost = (PAIFCOMMANDTOHOST) Fib->data;

			if (AifCommandToHost->command == AifCmdDriverNotify) {



				HandleDriverAif( Adapter, &FibContext );

			} else {
					AAC_UINT32 time_now, time_last;
					time_now = (AAC_UINT32)OsGetSeconds();
			

				OsCvLockAcquire(Adapter->AdapterFibMutex);

				Entry = Adapter->AdapterFibContextList.Flink;

				//
				// For each Context that is on the AdapterFibContextList, make a copy of the
				// fib, and then set the event to wake up the thread that is waiting for it.
				//

				while (Entry != &Adapter->AdapterFibContextList) {

					//
					// Extract the AdapterFibContext
					//

					AdapterFibContext = CONTAINING_RECORD( Entry, GET_ADAPTER_FIB_CONTEXT, NextContext );

					//
					// Check if the queue is getting backlogged
					//
					if ( AdapterFibContext->FibCount > 20 ) {
						time_last = (AAC_UINT32)(AdapterFibContext->FileObject);

						//
						// has it been > 2 minutes since the last read off the queue?
						//
						if ((time_now - time_last) > 120) {
							Entry = Entry->Flink;
							// cmn_err (CE_WARN, "aifd: Flushing orphaned AdapterFibContext: idle %d seconds, %d fibs",
							//		 time_now - time_last,
							//		 AdapterFibContext->FibCount);
							FsaCloseAdapterFibContext ( Adapter, AdapterFibContext );
							continue;
						}
					}
									
//  Warning: sleep possible while holding spinlock
					NewFib = OsAllocMemory(sizeof(FIB), OS_ALLOC_MEM_SLEEP);

					if (NewFib) {

						//
						// Make the copy of the FIB
						//

						RtlCopyMemory(NewFib, Fib, sizeof(FIB));

						//
						// Put the FIB onto the AdapterFibContext's FibList
						//

						InsertTailList(&AdapterFibContext->FibList, &NewFib->Header.FibLinks);
						AdapterFibContext->FibCount++;

						// 
						// Set the event to wake up the thread that will waiting.
						//

						OsCv_signal(&AdapterFibContext->UserEvent);

					} else {

						cmn_err (CE_WARN, "aifd: didn't allocate NewFib");

					}

					Entry = Entry->Flink;
				}

				//
				// Set the status of this FIB
				//

				*(FSASTATUS *)Fib->data = ST_OK;
				
				CompleteAdapterFib( &FibContext, sizeof(FSASTATUS) );

				OsCvLockRelease(Adapter->AdapterFibMutex);

			}

			OsSpinLockAcquire(CommRegion->HostNormCmdQue.QueueLock);

		}

		//
		// There are no more AIF's,  call cv_wait_sig to wait for more
		// to process.
		//

		// cmn_err(CE_DEBUG, "no more AIF's going to sleep\n");

		if (OsCv_wait_sig( &(CommRegion->HostNormCmdQue.CommandReady), 
						 CommRegion->HostNormCmdQue.QueueLock ) == 0) {

			OsSpinLockRelease(CommRegion->HostNormCmdQue.QueueLock);

			Adapter->AifThreadStarted = FALSE;

			// cmn_err(CE_DEBUG, "AifThread awoken by a signal\n");
			
			return (EINTR);
			
		}				 

		// cmn_err(CE_DEBUG, "Aif thread awake, going to look for more AIF's\n");

	}
}
    

PVOID
FsaGetFibData(
	IN PFIB_CONTEXT Context
	)
{
    PCOMM_FIB_CONTEXT FibContext = (PCOMM_FIB_CONTEXT) Context;

	return ((PVOID)FibContext->Fib->data);
}	    
                              

#ifdef API_THROTTLE

void ThrottlePeriodEndDpcRtn(
    IN PKDPC Dpc,
    IN PVOID DeferredContext,
    IN PVOID SystemArgument1,
    IN PVOID SystemArgument2
    )
/*++

Routine Description:

    This routine is called as a DPC when a throttle period expires. It
	restarts all threads suspended due to the throttling flow control.
	
	The throttling counted semaphore is signalled for all waiting threads
	and the indicator of throttling active is cleared.

Arguments:

    Dpc				- Pointer to Dpc structure. Not used.
	DefferedContext - Pointer to per-adapter context. This is used to locate the
					  throttle information for this adapter.
    SystemArgument1	- Not used
	SystemArgument2 - Not used
	
Return Value:

	None.

--*/
{
	PCOMM_REGION CommRegion;
	PAFA_COMM_ADAPTER Adapter = (PAFA_COMM_ADAPTER) DeferredContext;

	CommRegion = Adapter->CommRegion;

	//
	// Acquire the spinlock protecting the throttle status.
	//
	OsSpinLockAcquire(CommRegion->AdapNormCmdQue.QueueLock);

	FsaCommPrint("ThrottlePeriodEndDpc\n");

	//
	// Check that the timer has fired as many times as it was set !
	//

	CommRegion->ThrottleTimerFires++;
	ASSERT(CommRegion->ThrottleTimerFires == CommRegion->ThrottleTimerSets);

	//
	// The throttle period is now over. Restart all threads waiting
	// on the throttle being released.
	// Clear the throttle active indicator. This will allow new FIBs
	// to be sent to the adapter once we release the spinlock on exiting
	// the DPC. This means all restarted threads will be runnable
	// threads by then.
	//

	ASSERT(CommRegion->ThrottleActive == TRUE);		// The throttle had better be on !
	CommRegion->ThrottleActive = FALSE;				// This allows new data FIBs to go to the adapter on dpc exit

	OsSpinLockRelease(CommRegion->AdapNormCmdQue.QueueLock);
}

#endif // #ifdef API_THROTTLE

/*
 * Overrides for Emacs so that we almost follow Linus's tabbing style.
 * Emacs will notice this stuff at the end of the file and automatically
 * adjust the settings for this buffer only.  This must remain at the end
 * of the file.
 * ---------------------------------------------------------------------------
 * Local variables:
 * c-indent-level: 4
 * c-brace-imaginary-offset: 0
 * c-brace-offset: -4
 * c-argdecl-indent: 4
 * c-label-offset: -4
 * c-continued-statement-offset: 4
 * c-continued-brace-offset: 0
 * indent-tabs-mode: nil
 * tab-width: 8
 * End:
 */
