/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  comminit.c
 *
 * Abstract: This supports the initialization of the host adapter commuication interface.
 *    This is a platform dependent module for the pci cyclone board.
 *
 --*/

static char *ident_comminit = "aacraid_ident comminit.c 1.0.6 2000/10/09 Adaptec, Inc.";

#include "comprocs.h"

#define BugCheckFileId                   (FSAFS_BUG_CHECK_COMMINIT)

VOID
AfaCommBugcheckHandler(
		IN PVOID Buffer,
		IN ULONG Length
		);

VOID
ThrottlePeriodEndDpcRtn(
	IN PKDPC Dpc,
	IN PVOID DeferredContext,
	IN PVOID SystemArgument1,
	IN PVOID SystemArgument2);

FSA_COMM_DATA FsaCommData;

AAC_STATUS
HardInterruptModeration1Changed(
	IN PVOID AdapterContext,
	IN ULONG NewValue
	)
{
	PAFA_COMM_ADAPTER Adapter = AdapterContext;

	//
	// If we are using interrupt moderation, then disable the interrupt
	// until we need to use it.
	//
	if (FsaCommData.HardInterruptModeration1)
		DisableInterrupt( Adapter, AdapNormCmdNotFull, FALSE );
	else
		EnableInterrupt( Adapter, AdapNormCmdNotFull, FALSE );

	return (STATUS_SUCCESS);
}

AAC_STATUS
FsaFibTimeoutChanged(
	IN PVOID AdapterContext,
	IN ULONG NewValue
	)
{
	//
	// scale the new timeout from seconds to 100 nsec units
	//
//	FsaCommData.AdapterTimeout = RtlConvertLongToLargeInteger(-10*1000*1000*NewValue);

	return (STATUS_SUCCESS);
}

#ifdef GATHER_FIB_TIMES
extern int GatherFibTimes;
#endif

FSA_USER_VAR FsaCommUserVars[] = {
#ifdef FIB_CHECKSUMS
    { "do_fib_checksums", (PULONG)&FsaCommData.do_fib_checksums, NULL },
#endif
#ifdef GATHER_FIB_TIMES
	{ "GatherFibTimes", (PULONG)&GatherFibTimes, NULL },
#endif
	{ "EnableAdapterTimeouts", (PULONG)&FsaCommData.EnableAdapterTimeouts, NULL},
	{ "EnableInterruptModeration", (PULONG)&FsaCommData.EnableInterruptModeration, NULL },
	{ "FsaDataFibsSent", (PULONG) &FsaCommData.FibsSent, NULL },
	{ "FsaDataFibRecved", (PULONG) &FsaCommData.FibRecved, NULL },
	{ "HardInterruptModeration", (PULONG)&FsaCommData.HardInterruptModeration, NULL},
	{ "HardInterruptModeration1", (PULONG)&FsaCommData.HardInterruptModeration1, HardInterruptModeration1Changed},
	{ "EnableFibTimeoutBreak", (PULONG)&FsaCommData.EnableFibTimeoutBreak, NULL},
	{ "PeakFibsConsumed", (PULONG)&FsaCommData.PeakFibsConsumed, NULL },
	{ "ZeroFibsConsumed", (PULONG)&FsaCommData.ZeroFibsConsumed, NULL },
	{ "FibTimeoutSeconds", (PULONG) &FsaCommData.FibTimeoutSeconds, FsaFibTimeoutChanged },
};

#define NUM_COMM_USER_VARS	(sizeof(FsaCommUserVars) / sizeof(FSA_USER_VAR) )


AAC_STATUS
AacCommDriverEntry(
    )

/*++

Routine Description:

    This is the initialization routine for the FileArray Comm layer device driver.

Arguments:

    DriverObject - Pointer to driver object created by the system.

Return Value:

    AAC_STATUS - The function value is the final status from the initialization
        operation.

--*/

{
    AAC_STATUS Status;
	PVOID BugCheckBuffer;

	RtlZeroMemory( &FsaCommData, sizeof(FSA_COMM_DATA) );


    //
    // Load the global timeout value for the adapter timeout
    // Also init the global that enables or disables adapter timeouts
    //

//	FsaCommData.AdapterTimeout = RtlConvertLongToLargeInteger(-10*1000*1000*180);

	FsaCommData.FibTimeoutSeconds = 180;

	FsaCommData.EnableAdapterTimeouts = TRUE; 

//	FsaCommData.QueueFreeTimeout = RtlConvertLongToLargeInteger(QUEUE_ENTRY_FREE_TIMEOUT);

#ifdef unix_fib_timeout
	FsaCommData.FibTimeoutIncrement = (180 * 1000 * 1000 * 10) / KeQueryTimeIncrement();
#endif

	FsaCommData.EnableInterruptModeration = FALSE;

	//
	// Preload UserVars with all variables from the comm layer.  The class layers will
	// include theirs when they register.
	//

	FsaCommData.UserVars = OsAllocMemory(NUM_COMM_USER_VARS * sizeof(FSA_USER_VAR), OS_ALLOC_MEM_SLEEP );
	FsaCommData.NumUserVars = NUM_COMM_USER_VARS;

	RtlCopyMemory( FsaCommData.UserVars, &FsaCommUserVars, NUM_COMM_USER_VARS * sizeof(FSA_USER_VAR) );


#ifdef AACDISK
	//
	// Call the disk driver to initialize itself.
	//

	AacDiskDriverEntry();

#endif



	return (STATUS_SUCCESS);
}


VOID
DetachNTQueue(
	IN PAFA_COMM_ADAPTER Adapter,
    IN OUT PCOMM_QUE Queue,
    IN QUEUE_TYPES WhichQueue
    )
/*++

Routine Description:

	This routine will release all of the resources used by a given queue.

Arguments:

	Adapter - Which adapter the queue belongs to
	Queue - Pointer to the queue itself
	WhichQueue - Identifies which of the host queues this is.

Return Value:

	NONE.

--*/
{
    switch (WhichQueue) {

        case HostNormCmdQueue:

			Os_remove_softintr( Queue->ConsumerRoutine );
			OsSpinLockDestroy( Queue->QueueLock );
			OsCv_destroy( &Queue->CommandReady );
				
            break;

        case HostHighCmdQueue:

			Os_remove_softintr( Queue->ConsumerRoutine );
			OsSpinLockDestroy( Queue->QueueLock );
			OsCv_destroy( &Queue->CommandReady );
				
            break;

        case HostNormRespQueue:

			Os_remove_softintr( Queue->ConsumerRoutine );
			OsSpinLockDestroy( Queue->QueueLock );
            break;

        case HostHighRespQueue:

			Os_remove_softintr( Queue->ConsumerRoutine );
			OsSpinLockDestroy( Queue->QueueLock );
            break;

        case AdapNormCmdQueue:
        case AdapHighCmdQueue:
        case AdapNormRespQueue:
        case AdapHighRespQueue:
			OsCv_destroy( &Queue->QueueFull );
            break;
    }
}
    
VOID
InitializeNTQueue(
	IN PAFA_COMM_ADAPTER Adapter,
    IN OUT PCOMM_QUE Queue,
    IN QUEUE_TYPES WhichQueue
    )
/*++

Routine Description:

    Will initialize all entries in the queue that is NT specific.

Arguments:

Return Value:

    Nothing there is nothing to allocate so nothing should fail

--*/
{
    
	Queue->NumOutstandingIos = 0;

	//
	// Store a pointer to the adapter structure.
	//

	Queue->Adapter = Adapter;

	InitializeListHead( &Queue->OutstandingIoQueue );
	    
    switch (WhichQueue) {

        case HostNormCmdQueue:

			OsCv_init( &Queue->CommandReady);
			OsSpinLockInit( Queue->QueueLock, Adapter->SpinLockCookie);
			if (ddi_add_softintr( Adapter->Dip, DDI_SOFTINT_HIGH, &Queue->ConsumerRoutine, NULL,
								  NULL, (PUNIX_INTR_HANDLER)HostCommandNormDpc,
								  (caddr_t)Queue ) != DDI_SUCCESS) {

				cmn_err(CE_CONT, "OS_addr_intr failed\n");					
			}					

            InitializeListHead(&Queue->CommandQueue);

            break;

        case HostHighCmdQueue:

			OsCv_init( &Queue->CommandReady);
			OsSpinLockInit( Queue->QueueLock, Adapter->SpinLockCookie);
			if (ddi_add_softintr( Adapter->Dip, DDI_SOFTINT_HIGH, &Queue->ConsumerRoutine, NULL,
								  NULL, (PUNIX_INTR_HANDLER)HostCommandHighDpc,
								  (caddr_t) Queue ) != DDI_SUCCESS) {

				cmn_err(CE_CONT, "OS_addr_intr failed\n");					
			}					

            InitializeListHead(&Queue->CommandQueue);
            break;

        case HostNormRespQueue:

			OsSpinLockInit( Queue->QueueLock, Adapter->SpinLockCookie);
			if (ddi_add_softintr( Adapter->Dip, DDI_SOFTINT_HIGH, &Queue->ConsumerRoutine, NULL,
								  NULL, (PUNIX_INTR_HANDLER)HostResponseNormalDpc, 
								  (caddr_t) Queue ) != DDI_SUCCESS) {

				cmn_err(CE_CONT, "OS_addr_intr failed\n");					
			}					
            break;

        case HostHighRespQueue:


			OsSpinLockInit( Queue->QueueLock, Adapter->SpinLockCookie);
			if (ddi_add_softintr( Adapter->Dip, DDI_SOFTINT_HIGH, &Queue->ConsumerRoutine, NULL,
								  NULL, (PUNIX_INTR_HANDLER)HostResponseHighDpc, 
								  (caddr_t) Queue ) != DDI_SUCCESS) {

				cmn_err(CE_CONT, "OS_addr_intr failed\n");					
			}					
            break;

        case AdapNormCmdQueue:
        case AdapHighCmdQueue:
        case AdapNormRespQueue:
        case AdapHighRespQueue:

			OsCv_init( &Queue->QueueFull);
            break;
    }
}

BOOLEAN
StartFsaCommandThreads(PAFA_COMM_ADAPTER Adapter)
/*++

Routine Description:

    Create and start the command receiver threads.

Arguments:


Return Value:

    Nothing

--*/

{
    return(TRUE);
}



/*++

Routine Description:

	This routine gets called to detach all resources that have been allocated for 
	this adapter.

Arguments:

	Adapter - Pointer to the adapter structure to detach.

Return Value:

    TRUE - All resources have been properly released.
    FALSE - An error occured while trying to release resources.
--*/
BOOLEAN
AacCommDetachAdapter (IN PAFA_COMM_ADAPTER	Adapter)
{
	PAFA_CLASS_DRIVER ClassDriver;
	//
	// First remove this adapter from the list of adapters.
	//

	if (FsaCommData.AdapterList == Adapter) {
		
		FsaCommData.AdapterList = Adapter->NextAdapter;

	} else {

		PAFA_COMM_ADAPTER CurrentAdapter, NextAdapter;
	
		CurrentAdapter = FsaCommData.AdapterList;
		NextAdapter = CurrentAdapter->NextAdapter;

		while (NextAdapter) {
				
			if (NextAdapter == Adapter) {

				CurrentAdapter->NextAdapter = NextAdapter->NextAdapter;
				break;
			 
			}
			
			CurrentAdapter = NextAdapter;
			NextAdapter = CurrentAdapter->NextAdapter;
		}
	}			
		
	//
	// First send a shutdown to the adapter.
	//

	AfaCommShutdown( Adapter );

	//
	// Destroy the FibContextZone for this adapter.  This will free up all
	// of the fib space used by this adapter.
	//
	
	FsaFreeFibContextZone( Adapter );

	//
	// Destroy the mutex used for synch'ing adapter fibs.
	//

	OsCvLockDestroy( Adapter->AdapterFibMutex );

	//
	// Detach all of the host queues.
	//

    DetachNTQueue( Adapter, &Adapter->CommRegion->AdapHighRespQue, AdapHighRespQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->AdapNormRespQue, AdapNormRespQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->HostHighRespQue, HostHighRespQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->HostNormRespQue, HostNormRespQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->AdapHighCmdQue, AdapHighCmdQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->AdapNormCmdQue, AdapNormCmdQueue );
    DetachNTQueue( Adapter, &Adapter->CommRegion->HostHighCmdQue, HostHighCmdQueue );
	DetachNTQueue( Adapter, &Adapter->CommRegion->HostNormCmdQue, HostNormCmdQueue );

	//
	// Destroy the mutex used to protect the FibContextZone
	//

	OsSpinLockDestroy( Adapter->FibContextZoneSpinLock );

	//
	// Call the miniport to free the space allocated for the shared comm queues
	// between the host and the adapter.
	//

	FsaFreeAdapterCommArea( Adapter );

	//
	// Free the memory used by the comm region for this adapter
	//

	OsFreeMemory( Adapter->CommRegion, sizeof(COMM_REGION) );

	//
	// Free the memory used by the adapter structure.
	//
	ClassDriver = Adapter->ClassDriverList;
	Adapter->ClassDriverList = Adapter->ClassDriverList->Next;
	OsFreeMemory( ClassDriver, sizeof(AFA_CLASS_DRIVER) );
	
	OsFreeMemory( Adapter, sizeof(AFA_COMM_ADAPTER) );

	return (TRUE);
}

PVOID
AfaCommInitNewAdapter (IN PFSA_NEW_ADAPTER NewAdapter)
{
	PVOID BugCheckBuffer;
	PAFA_COMM_ADAPTER Adapter;
	MAPFIB_CONTEXT MapFibContext;
	LARGE_INTEGER Time;
	char ErrorBuffer[60];

	Adapter = (PAFA_COMM_ADAPTER) OsAllocMemory( sizeof(AFA_COMM_ADAPTER) , OS_ALLOC_MEM_SLEEP );

	if (Adapter == NULL)
		return (NULL);

	RtlZeroMemory(Adapter, sizeof(AFA_COMM_ADAPTER));


	//
	// Save the current adapter number and increment the total number.
	//

	Adapter->AdapterNumber = FsaCommData.TotalAdapters++;


	//
	// Fill in the pointer back to the device specific structures.
	// The device specific driver has also passed a pointer for us to 
	// fill in with the Adapter object that we have created.
	//

	Adapter->AdapterExtension = NewAdapter->AdapterExtension;
	Adapter->AdapterFuncs = NewAdapter->AdapterFuncs;
	Adapter->InterruptsBelowDpc = NewAdapter->AdapterInterruptsBelowDpc;
	Adapter->AdapterUserVars = NewAdapter->AdapterUserVars;
	Adapter->AdapterUserVarsSize = NewAdapter->AdapterUserVarsSize;

	Adapter->Dip = NewAdapter->Dip;

	//
	// Fill in Our address into the function dispatch table
	//

	NewAdapter->AdapterFuncs->InterruptHost = AfaCommInterruptHost;
	NewAdapter->AdapterFuncs->OpenAdapter = AfaCommOpenAdapter;
	NewAdapter->AdapterFuncs->CloseAdapter = AfaCommCloseAdapter;
	NewAdapter->AdapterFuncs->DeviceControl = AfaCommAdapterDeviceControl;

	//
	// Ok now init the communication subsystem
	//

	Adapter->CommRegion = (PCOMM_REGION) OsAllocMemory(sizeof(COMM_REGION), OS_ALLOC_MEM_SLEEP);
	if (Adapter->CommRegion == NULL) {
		cmn_err(CE_WARN, "Error could not allocate comm region.\n");
		return (NULL);
	}
	RtlZeroMemory(Adapter->CommRegion, sizeof(COMM_REGION));

	//
	// Get a pointer to the iblock_cookie
	//

	ddi_get_soft_iblock_cookie( Adapter->Dip, DDI_SOFTINT_HIGH, &Adapter->SpinLockCookie );

	if (!CommInit(Adapter)) {
		FsaCommPrint("Failed to init the commuication subsystem.\n");
		return(NULL);
	}


	//
	// Initialize the list of AdapterFibContext's.
	//

	InitializeListHead(&Adapter->AdapterFibContextList);

	//
	// Initialize the fast mutex used for synchronization of the adapter fibs
	//

	Adapter->AdapterFibMutex = OsCvLockAlloc();
	OsCvLockInit(Adapter->AdapterFibMutex, NULL);

    //
    // Allocate and start the FSA command threads. These threads will handle
    // command requests from the adapter. They will wait on an event then pull
    // all CDBs off the thread's queue. Each CDB will be given to a worker thread
    // upto a defined limit. When that limit is reached wait a event will be waited
    // on till a worker thread is finished.
    //

    if (!StartFsaCommandThreads(Adapter)) {
   	    FsaCommPrint("Fsainit could not initilize the command receiver threads.\n");
		return (NULL);
    }

#ifdef unix_crash_dump
	//
	// Allocate and map a fib for use by the synch path, which is used for crash
	// dumps.
	//
	// Allocate an entire page so that alignment is correct.
	//

	Adapter->SyncFib = OsAllocMemory( PAGE_SIZE, OS_ALLOC_MEM_SLEEP );
	MapFibContext.Fib = Adapter->SyncFib;
	MapFibContext.Size = sizeof(FIB);
	MapFib( Adapter, &MapFibContext );
	Adapter->SyncFibPhysicalAddress = MapFibContext.LogicalFibAddress.LowPart;
#endif

	Adapter->CommFuncs.SizeOfAfaCommFuncs = sizeof(AFACOMM_FUNCS);

	Adapter->CommFuncs.AllocateFib = AllocateFib;

	Adapter->CommFuncs.FreeFib = FreeFib;
	Adapter->CommFuncs.FreeFibFromDpc = FreeFibFromDpc;
	Adapter->CommFuncs.DeallocateFib = DeallocateFib;

	Adapter->CommFuncs.InitializeFib = InitializeFib;
	Adapter->CommFuncs.GetFibData = FsaGetFibData;
	Adapter->CommFuncs.SendFib = SendFib;
	Adapter->CommFuncs.CompleteFib = CompleteFib;
	Adapter->CommFuncs.CompleteAdapterFib = CompleteAdapterFib;

	Adapter->CommFuncs.SendSynchFib = SendSynchFib;

	Adapter->CommFuncs.FreeDmaResources = Adapter->AdapterFuncs->FreeDmaResources;
	Adapter->CommFuncs.BuildSgMap = Adapter->AdapterFuncs->BuildSgMap;

	//
	// Add this adapter in to our Adapter List.
	//

	Adapter->NextAdapter = FsaCommData.AdapterList;
	FsaCommData.AdapterList = Adapter;

	NewAdapter->Adapter = Adapter;

//	AfaDiskInitNewAdapter( Adapter->AdapterNumber, Adapter );

	return (Adapter);
}

AAC_STATUS
CommInitialize(
	PAFA_COMM_ADAPTER Adapter
	)
{
    //
    //  Now allocate and initialize the zone structures used as our pool
    //  of FIB context records.  The size of the zone is based on the
    //  system memory size.  We also initialize the mutex used to protect
    //  the zone.
    //
	Adapter->FibContextZoneSpinLock = OsSpinLockAlloc();
	OsSpinLockInit( Adapter->FibContextZoneSpinLock, Adapter->SpinLockCookie );

	Adapter->FibContextZoneExtendSize = 64;

	return (STATUS_SUCCESS);
}


    
/*++

Routine Description:

    Initializes the data structures that are required for the FSA commuication
    interface to operate.

Arguments:

    None - all global or allocated data.

Return Value:

    TRUE - if we were able to init the commuication interface.
    FALSE - If there were errors initing. This is a fatal error.
--*/
BOOLEAN
CommInit(PAFA_COMM_ADAPTER Adapter)
{
    
    ULONG SizeOfHeaders = (sizeof(QUEUE_INDEX) * NUMBER_OF_COMM_QUEUES) * 2;
    ULONG SizeOfQueues = sizeof(QUEUE_ENTRY) * TOTAL_QUEUE_ENTRIES;
    PQUEUE_INDEX Headers;
    PQUEUE_ENTRY Queues;
	ULONG TotalSize;
	PCOMM_REGION CommRegion = Adapter->CommRegion;

	 CommInitialize( Adapter );

	FsaCommPrint("CommInit: Queue entry size is 0x%x, Queue index size is 0x%x, Number of total entries is 0x%x, # queues = 0x%x.\n",
			  sizeof(QUEUE_ENTRY), sizeof(QUEUE_INDEX), TOTAL_QUEUE_ENTRIES, NUMBER_OF_COMM_QUEUES);
	//
	//
	// Allocate the physically contigous space for the commuication queue
	// headers. 
	//

	TotalSize = SizeOfHeaders + SizeOfQueues;

	if (!FsaAllocateAdapterCommArea(Adapter, (PVOID *)&Headers, TotalSize, QUEUE_ALIGNMENT))
		return (FALSE);

	Queues = (PQUEUE_ENTRY)((PUCHAR)Headers + SizeOfHeaders);

	if (ddi_add_softintr( Adapter->Dip, DDI_SOFTINT_HIGH, &CommRegion->QueueNotFullDpc, NULL,
						  NULL, (PUNIX_INTR_HANDLER)CommonNotFullDpc,
						  (caddr_t)CommRegion ) != DDI_SUCCESS) {

	  cmn_err(CE_CONT, "Os_addr_intr failed\n");					
	}					


    // Adapter to Host normal priority Command queue


    CommRegion->HostNormCmdQue.Headers.ProducerIndex = Headers++;
    CommRegion->HostNormCmdQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->HostNormCmdQue.Headers.ProducerIndex = HOST_NORM_CMD_ENTRIES;
    *CommRegion->HostNormCmdQue.Headers.ConsumerIndex = HOST_NORM_CMD_ENTRIES;

    CommRegion->HostNormCmdQue.SavedIrql = 0;
    CommRegion->HostNormCmdQue.BaseAddress = Queues;
    CommRegion->HostNormCmdQue.QueueEntries = HOST_NORM_CMD_ENTRIES;

	CommRegion->HostNormCmdQue.QueueLock = OsSpinLockAlloc();
	if (CommRegion->HostNormCmdQue.QueueLock == NULL) {
		return (FALSE);
	}
    InitializeNTQueue(Adapter, &CommRegion->HostNormCmdQue, HostNormCmdQueue);

    
    Queues += HOST_NORM_CMD_ENTRIES;

    // Adapter to Host high priority command queue
    
    CommRegion->HostHighCmdQue.Headers.ProducerIndex = Headers++;
    CommRegion->HostHighCmdQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->HostHighCmdQue.Headers.ProducerIndex = HOST_HIGH_CMD_ENTRIES;
    *CommRegion->HostHighCmdQue.Headers.ConsumerIndex = HOST_HIGH_CMD_ENTRIES;

    CommRegion->HostHighCmdQue.SavedIrql = 0;
    CommRegion->HostHighCmdQue.BaseAddress = Queues;
    CommRegion->HostHighCmdQue.QueueEntries = HOST_HIGH_CMD_ENTRIES;
//	CommRegion->HostHighCmdQue.QueueLock = (PKSPIN_LOCK) ExAllocatePool(NonPagedPool, sizeof(KSPIN_LOCK));
	CommRegion->HostHighCmdQue.QueueLock = OsSpinLockAlloc();
	if (CommRegion->HostHighCmdQue.QueueLock == NULL) {
		return (FALSE);
	}
    InitializeNTQueue(Adapter, &CommRegion->HostHighCmdQue, HostHighCmdQueue);
    
    Queues += HOST_HIGH_CMD_ENTRIES;

    // Host to adapter normal priority command queue
    
    CommRegion->AdapNormCmdQue.Headers.ProducerIndex = Headers++;
    CommRegion->AdapNormCmdQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->AdapNormCmdQue.Headers.ProducerIndex = ADAP_NORM_CMD_ENTRIES;
    *CommRegion->AdapNormCmdQue.Headers.ConsumerIndex = ADAP_NORM_CMD_ENTRIES;

    CommRegion->AdapNormCmdQue.SavedIrql = 0;    
    CommRegion->AdapNormCmdQue.BaseAddress = Queues;
    CommRegion->AdapNormCmdQue.QueueEntries = ADAP_NORM_CMD_ENTRIES;
    InitializeNTQueue(Adapter, &CommRegion->AdapNormCmdQue, AdapNormCmdQueue);
    
    Queues += ADAP_NORM_CMD_ENTRIES;

    // host to adapter high priority command queue
    
    CommRegion->AdapHighCmdQue.Headers.ProducerIndex = Headers++;
    CommRegion->AdapHighCmdQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->AdapHighCmdQue.Headers.ProducerIndex = ADAP_HIGH_CMD_ENTRIES;
    *CommRegion->AdapHighCmdQue.Headers.ConsumerIndex = ADAP_HIGH_CMD_ENTRIES;

    CommRegion->AdapHighCmdQue.SavedIrql = 0;    
    CommRegion->AdapHighCmdQue.BaseAddress = Queues;
    CommRegion->AdapHighCmdQue.QueueEntries = ADAP_HIGH_CMD_ENTRIES;
    InitializeNTQueue(Adapter, &CommRegion->AdapHighCmdQue, AdapHighCmdQueue);
    
    Queues += ADAP_HIGH_CMD_ENTRIES;

    // adapter to host normal priority response queue
    
    CommRegion->HostNormRespQue.Headers.ProducerIndex = Headers++;
    CommRegion->HostNormRespQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->HostNormRespQue.Headers.ProducerIndex = HOST_NORM_RESP_ENTRIES;
    *CommRegion->HostNormRespQue.Headers.ConsumerIndex = HOST_NORM_RESP_ENTRIES;

    CommRegion->HostNormRespQue.SavedIrql = 0;    
    CommRegion->HostNormRespQue.BaseAddress = Queues;
    CommRegion->HostNormRespQue.QueueEntries = HOST_NORM_RESP_ENTRIES;
//	CommRegion->HostNormRespQue.QueueLock = (PKSPIN_LOCK) ExAllocatePool(NonPagedPool, sizeof(KSPIN_LOCK));
	CommRegion->HostNormRespQue.QueueLock = OsSpinLockAlloc();
	if (CommRegion->HostNormRespQue.QueueLock == NULL) {
		return (FALSE);
	}
    InitializeNTQueue(Adapter, &CommRegion->HostNormRespQue, HostNormRespQueue);
    
    Queues += HOST_NORM_RESP_ENTRIES;

    // adapter to host high priority response queue
    
    CommRegion->HostHighRespQue.Headers.ProducerIndex = Headers++;
    CommRegion->HostHighRespQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->HostHighRespQue.Headers.ProducerIndex = HOST_HIGH_RESP_ENTRIES;
    *CommRegion->HostHighRespQue.Headers.ConsumerIndex = HOST_HIGH_RESP_ENTRIES;

    CommRegion->HostHighRespQue.SavedIrql = 0;    
    CommRegion->HostHighRespQue.BaseAddress = Queues;
    CommRegion->HostHighRespQue.QueueEntries = HOST_HIGH_RESP_ENTRIES;
//	CommRegion->HostHighRespQue.QueueLock = (PKSPIN_LOCK) ExAllocatePool(NonPagedPool, sizeof(KSPIN_LOCK));
	CommRegion->HostHighRespQue.QueueLock = OsSpinLockAlloc();
	if (CommRegion->HostHighRespQue.QueueLock == NULL) {
		return (FALSE);
	}
    InitializeNTQueue(Adapter, &CommRegion->HostHighRespQue, HostHighRespQueue);
    
    Queues += HOST_HIGH_RESP_ENTRIES;

    // host to adapter normal priority response queue
    
    CommRegion->AdapNormRespQue.Headers.ProducerIndex = Headers++;
    CommRegion->AdapNormRespQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->AdapNormRespQue.Headers.ProducerIndex = ADAP_NORM_RESP_ENTRIES;
    *CommRegion->AdapNormRespQue.Headers.ConsumerIndex = ADAP_NORM_RESP_ENTRIES;

    CommRegion->AdapNormRespQue.SavedIrql = 0;    
    CommRegion->AdapNormRespQue.BaseAddress = Queues;
    CommRegion->AdapNormRespQue.QueueEntries = ADAP_NORM_RESP_ENTRIES;
    InitializeNTQueue(Adapter, &CommRegion->AdapNormRespQue, AdapNormRespQueue);
    
    Queues += ADAP_NORM_RESP_ENTRIES;

    // host to adapter high priority response queue
    
    CommRegion->AdapHighRespQue.Headers.ProducerIndex = Headers++;
    CommRegion->AdapHighRespQue.Headers.ConsumerIndex = Headers++;
    *CommRegion->AdapHighRespQue.Headers.ProducerIndex = ADAP_HIGH_RESP_ENTRIES;
    *CommRegion->AdapHighRespQue.Headers.ConsumerIndex = ADAP_HIGH_RESP_ENTRIES;

    CommRegion->AdapHighRespQue.SavedIrql = 0;    
    CommRegion->AdapHighRespQue.BaseAddress = Queues;
    CommRegion->AdapHighRespQue.QueueEntries = ADAP_HIGH_RESP_ENTRIES;
    InitializeNTQueue(Adapter, &CommRegion->AdapHighRespQue, AdapHighRespQueue);

	CommRegion->AdapNormCmdQue.QueueLock = CommRegion->HostNormRespQue.QueueLock;
	CommRegion->AdapHighCmdQue.QueueLock = CommRegion->HostHighRespQue.QueueLock;
	CommRegion->AdapNormRespQue.QueueLock = CommRegion->HostNormCmdQue.QueueLock;
	CommRegion->AdapHighRespQue.QueueLock = CommRegion->HostHighCmdQue.QueueLock;

    return(TRUE);
}

AAC_STATUS
AfaCommShutdown(
	PAFA_COMM_ADAPTER Adapter
	)
/*++

Routine Description:

	This routine will send a shutdown request to each adapter.

Arguments:

	Adapter - which adapter to send the shutdown to.

Return Value:

    NT Status success.

--*/

{
	PFIB_CONTEXT FibContext;
	PCLOSECOMMAND CloseCommand;
	AAC_STATUS Status;

	FibContext = AllocateFib( Adapter );

	InitializeFib( FibContext );

	CloseCommand = (PCLOSECOMMAND) FsaGetFibData( FibContext );

	CloseCommand->Command = VM_CloseAll;
	CloseCommand->ContainerId = 0xffffffff;

	Status = SendFib( ContainerCommand, FibContext, sizeof(CLOSECOMMAND), FsaNormal, TRUE, NULL, TRUE, NULL, NULL );

	if (Status != STATUS_SUCCESS) {

		FreeFib( FibContext );

		goto ret;

	}

	CompleteFib( FibContext );

	FreeFib( FibContext );


	Status = STATUS_SUCCESS;

ret:

	return (Status);

}

VOID
AfaCommBugcheckHandler(
		IN PVOID Buffer,
		IN ULONG Length
		)
/*++

Routine Description:

	This routine will shutdown the adapter if there is a bugcheck and
	copy the shutdown data from the adapter response into the buffer
	so it will show up in the host dump file.
p
Arguments:

	Buffer - This buffer will be written to the host dump by nt for us.

	Length - The size of the buffer.

Return Value:

	N/A

--*/
{
	PAFA_COMM_ADAPTER Adapter = FsaCommData.AdapterList;

	while (Adapter) {

		NotifyAdapter(Adapter, HostShutdown);

		Adapter = Adapter->NextAdapter;

	}

}	

VOID
FsaCommLogEvent(
	PFIB_CONTEXT FibContext, 
	PDEVICE_OBJECT DeviceObject,
	AAC_STATUS FsaStatus,
	AAC_STATUS AacStatus,
	ULONG LocationCode,
	USHORT Category,
	PUCHAR String,
	BOOLEAN DumpFib
)
{
}

AfaCommProbeDisks(
	PAFA_COMM_ADAPTER	Adapter
	)
{
    PMNTINFO DiskInfo;
    PMNTINFORESPONSE DiskInfoResponse;
	AAC_STATUS Status;
	PCOMM_FIB_CONTEXT FibContext;
    
	FibContext = AllocateFib( Adapter );

	InitializeFib( FibContext );

	DiskInfo = (PMNTINFO) FibContext->Fib->data;
	DiskInfo->Command = VM_NameServe;
	DiskInfo->MntCount = 0;
	DiskInfo->MntType = FT_FILESYS;

    Status = SendFib(ContainerCommand,
		             FibContext,
	                 sizeof(MNTINFO),
	                 FsaNormal,
	                 TRUE,
	                 NULL,
	                 TRUE,
	                 NULL,
	                 NULL);

	DiskInfoResponse = (PMNTINFORESPONSE) FibContext->Fib->data;

	if (DiskInfoResponse->MntRespCount) {

		cmn_err(CE_CONT, "container found on adapter, size = 0x%x blocks\n", 
				DiskInfoResponse->MntTable[0].Capacity);
				
	} else {
	
		cmn_err(CE_CONT, "no containers found on adapter\n");
		
	}
					
	CompleteFib( FibContext );
	
	FreeFib( FibContext );				 
}


