/*++
 * Adaptec aacraid device driver for Linux.
 *
 * Copyright (c) 2000 Adaptec, Inc. (aacraid@adaptec.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Module Name:
 *  commctrl.c
 *
 * Abstract: Contains all routines for control of the AFA comm layer
 *
--*/

static char *ident_commctrl = "aacraid_ident commctrl.c 1.0.7 2000/10/11 Adaptec, Inc.";

#include "comprocs.h"
#include "osheaders.h"
#include "ostypes.h"





typedef BOOLEAN BOOL;
#define inline /* _inline */

#include <revision.h>
AAC_STATUS
FsaCtlCheckRevision(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
	)
/*++

Routine Description:

	This routine validates the revision of the caller with the current revision
	of the filesystem.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

	IrpContext - Supplies the IrpContext.

Return Value:

	AAC_STATUS

--*/

{
	RevCheck APIRevCheck;
	RevCheckResp APIRevCheckResp;
	RevComponent APICallingComponent;
	ULONG APIBuildNumber;

	if (COPYIN( (caddr_t) IoctlCmdPtr->arg, (caddr_t) &APIRevCheck, sizeof(RevCheck), IoctlCmdPtr->flag )) {
		return (EFAULT);
	}

	APICallingComponent = APIRevCheck.callingComponent;
	APIBuildNumber = APIRevCheck.callingRevision.buildNumber;

	APIRevCheckResp.possiblyCompatible = RevCheckCompatibility( RevMiniportDriver , APICallingComponent, APIBuildNumber );

	APIRevCheckResp.adapterSWRevision.external.ul = RevGetExternalRev();
	APIRevCheckResp.adapterSWRevision.buildNumber = RevGetBuildNumber();

	if (COPYOUT( (caddr_t) &APIRevCheckResp, (caddr_t) IoctlCmdPtr->arg, sizeof(RevCheckResp), IoctlCmdPtr->flag )) {
		return (EFAULT);
	}

	return (0);
}


int
AfaCommAdapterDeviceControl(
	IN PVOID AdapterArg,
	IN PAFA_IOCTL_CMD IoctlCmdPtr
	)
{
	PAFA_COMM_ADAPTER Adapter = (PAFA_COMM_ADAPTER) AdapterArg;
    int Status = ENOTTY;
//    PIO_STACK_LOCATION IrpSp;
	PAFA_CLASS_DRIVER ClassDriver;

	//
	// First loop through all of the class drivers to give them a chance to handle
	// the Device control first.
	//

	ClassDriver = Adapter->ClassDriverList;

	while (ClassDriver) {

		if (ClassDriver->DeviceControl) {

			if (ClassDriver->DeviceControl( ClassDriver->ClassDriverExtension, IoctlCmdPtr, &Status ) ) {

				return (Status);

			}
		}

		ClassDriver = ClassDriver->Next;
	}

    switch (IoctlCmdPtr->cmd) {


	  case FSACTL_SENDFIB:

		Status = AfaCommCtlSendFib( Adapter, IoctlCmdPtr );
		break;

	  case FSACTL_AIF_THREAD:

	  	Status = AfaCommCtlAifThread( Adapter, IoctlCmdPtr );	
		break;


	  case FSACTL_OPEN_GET_ADAPTER_FIB:

		Status = FsaCtlOpenGetAdapterFib( Adapter, IoctlCmdPtr );
		break;

	  case FSACTL_GET_NEXT_ADAPTER_FIB:

		Status = FsaCtlGetNextAdapterFib( Adapter, IoctlCmdPtr );
		break;

	  case FSACTL_CLOSE_GET_ADAPTER_FIB:

		Status = FsaCtlCloseGetAdapterFib( Adapter, IoctlCmdPtr );
		break;

	  case FSACTL_MINIPORT_REV_CHECK:
        
		Status = FsaCtlCheckRevision( Adapter , IoctlCmdPtr );
		break;


      default:
	  
	  	Status = ENOTTY;
	  	break;	

	}


	return (Status);
}

AAC_STATUS
AfaCommRegisterNewClassDriver(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_NEW_CLASS_DRIVER	NewClassDriver,
	OUT PAFA_NEW_CLASS_DRIVER_RESPONSE NewClassDriverResponse
	)
/*++

Routine Description:

	This routine registers a new class driver for the comm layer.

	It will return a pointer to the communication functions for the class driver
	to use.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	AAC_STATUS Status;
	PAFA_CLASS_DRIVER ClassDriver;


	ClassDriver = (PAFA_CLASS_DRIVER) OsAllocMemory( sizeof(AFA_CLASS_DRIVER), OS_ALLOC_MEM_SLEEP );

	if (ClassDriver == NULL) {

		Status = STATUS_INSUFFICIENT_RESOURCES;

		return Status;
	}

	//
	// If the class driver has sent in user Vars, then copy them into the global
	// area.
	//

	if (NewClassDriver->NumUserVars) {

		PFSA_USER_VAR	NewUserVars;

		NewUserVars = OsAllocMemory( (FsaCommData.NumUserVars +
								   NewClassDriver->NumUserVars) * sizeof(FSA_USER_VAR), OS_ALLOC_MEM_SLEEP );

		//
		// First copy the existing into the new area.
		//

		RtlCopyMemory( NewUserVars, FsaCommData.UserVars, FsaCommData.NumUserVars * sizeof(FSA_USER_VAR) );

		//
		// Next copy the new vars passed in from class driver.
		//

		RtlCopyMemory( (NewUserVars + FsaCommData.NumUserVars),
					   NewClassDriver->UserVars,
					   NewClassDriver->NumUserVars * sizeof(FSA_USER_VAR) );

		//
		// Free up the old user vars.
		//

		OsFreeMemory( FsaCommData.UserVars, FsaCommData.NumUserVars * sizeof(FSA_USER_VAR) );

		//
		// Point the global to the new area.
		//

		FsaCommData.UserVars = NewUserVars;

		//
		// Update the total count.
		//

		FsaCommData.NumUserVars += NewClassDriver->NumUserVars;

	}

	ClassDriver->OpenAdapter = NewClassDriver->OpenAdapter;
	ClassDriver->CloseAdapter = NewClassDriver->CloseAdapter;
	ClassDriver->DeviceControl = NewClassDriver->DeviceControl;
	ClassDriver->HandleAif = NewClassDriver->HandleAif;
	ClassDriver->ClassDriverExtension = NewClassDriver->ClassDriverExtension;

	ClassDriver->Next = Adapter->ClassDriverList;
	Adapter->ClassDriverList = ClassDriver;

	//
	// Now return the information needed by the class driver to communicate to us.
	//

	NewClassDriverResponse->CommFuncs = &Adapter->CommFuncs;
	NewClassDriverResponse->CommPortExtension = Adapter;
	NewClassDriverResponse->MiniPortExtension = Adapter->AdapterExtension;
	NewClassDriverResponse->SpinLockCookie = Adapter->SpinLockCookie;
	NewClassDriverResponse->Dip = Adapter->Dip;

	return (STATUS_SUCCESS);


}

int
AfaCommCtlSendFib(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
)
/*++

Routine Description:

	This routine sends a fib to the adapter on behalf of a user level
	program.

Arguments:

	Adapter - Supplies which adapter is being processed.

	IoctlCmdPtr - Pointer to the arguments to the IOCTL call

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_INSUFFICIENT_RESOURCES - If a memory allocation failed.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
    PFIB KFib;
//    PMDL DmaMdl = NULL;
	PCOMM_FIB_CONTEXT FibContext;
	PSGMAP_CONTEXT SgMapContext;
	SGMAP_CONTEXT _SgMapContext;
    QUEUE_TYPES WhichQueue;
    PVOID UsersAddress;
	AAC_STATUS Status;

	FibContext = AllocateFib( Adapter );

    KFib = FibContext->Fib;

	//
	// First copy in the header so that we can check the size field.
	//

	if (COPYIN( (caddr_t) IoctlCmdPtr->arg, (caddr_t) KFib, sizeof(FIB_HEADER), IoctlCmdPtr->flag )) {
		FreeFib( FibContext );
		Status = EFAULT;
		return (Status);
	}

	//
	//	Since we copy based on the fib header size, make sure that we
	//	will not overrun the buffer when we copy the memory. Return
	//	an error if we would.
	//

	if (KFib->Header.Size > sizeof(FIB) - sizeof(FIB_HEADER)) {
		FreeFib( FibContext );
		Status = EINVAL;
		return Status;

	}

	if (COPYIN( (caddr_t) IoctlCmdPtr->arg, (caddr_t) KFib, KFib->Header.Size + sizeof(FIB_HEADER), IoctlCmdPtr->flag )) {
		FreeFib( FibContext );
		Status = EFAULT;
		return (Status);
	}

    WhichQueue = AdapNormCmdQueue;


	if (KFib->Header.Command == TakeABreakPt) {

		InterruptAdapter(Adapter);

		//
		// Since we didn't really send a fib, zero out the state to allow 
		// cleanup code not to assert.
		//

		KFib->Header.XferState = 0;


	} else {
	
		if (SendFib(KFib->Header.Command, FibContext, KFib->Header.Size , FsaNormal,
					TRUE, NULL, TRUE, NULL, NULL) != FSA_SUCCESS) {
	        FsaCommPrint("User SendFib failed!.\n");


			FreeFib( FibContext );
			return (ENXIO);
		}

	    if (CompleteFib(FibContext) != FSA_SUCCESS) {
	        FsaCommPrint("User Complete FIB failed.\n");

			FreeFib( FibContext );
			return (ENXIO);
		}


    }


	//
	//	Make sure that the size returned by the adapter (which includes
	//	the header) is less than or equal to the size of a fib, so we
	//	don't corrupt application data. Then copy that size to the user
	//	buffer. (Don't try to add the header information again, since it
	//	was already included by the adapter.)
	//
    ASSERT(KFib->Header.Size <= sizeof(FIB));

	if (COPYOUT( (caddr_t) KFib, (caddr_t) IoctlCmdPtr->arg, KFib->Header.Size, IoctlCmdPtr->flag )) {
		FreeFib( FibContext );
		Status = EFAULT;
		return (Status);
	}

	FreeFib( FibContext );

    return (0);

}

int
AfaCommCtlAifThread(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
)
/*++

Routine Description:

	This routine will act as the AIF thread for this adapter.

Arguments:

	Adapter - Supplies which adapter is being processed.

	IoctlCmdPtr - Pointer to the arguments to the IOCTL call

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_INSUFFICIENT_RESOURCES - If a memory allocation failed.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	return (NormCommandThread(Adapter));
}



#ifdef GATHER_FIB_TIMES
AAC_STATUS
AfaCommGetFibTimes(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PIRP					Irp
	)
/*++

Routine Description:

	This routine returns the gathered fibtimes to the user.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_INSUFFICIENT_RESOURCES - If a memory allocation failed.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	PALL_FIB_TIMES AllFibTimes;
	PLARGE_INTEGER FreqPtr;
    PIO_STACK_LOCATION IrpSp;

    //
    //  Get a pointer to the current Irp stack location
    //

    IrpSp = IoGetCurrentIrpStackLocation( Irp );

	FreqPtr = (PLARGE_INTEGER)IrpSp->Parameters.FileSystemControl.Type3InputBuffer;

	*FreqPtr = Adapter->FibTimesFrequency;

	AllFibTimes = (PALL_FIB_TIMES)((PUCHAR)FreqPtr + sizeof(LARGE_INTEGER));

	RtlCopyMemory(AllFibTimes, Adapter->FibTimes, sizeof(ALL_FIB_TIMES));

	Irp->IoStatus.Information = 0;

	return (STATUS_SUCCESS);

}

AAC_STATUS
AfaCommZeroFibTimes(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PIRP					Irp
	)
/*++

Routine Description:

	This routine zero's the FibTimes structure within the adapter structure.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_INSUFFICIENT_RESOURCES - If a memory allocation failed.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	PFIB_TIMES FibTimesPtr;
	int i;
    PIO_STACK_LOCATION IrpSp;

    //
    //  Get a pointer to the current Irp stack location
    //

    IrpSp = IoGetCurrentIrpStackLocation( Irp );

	//
	// Initialize the Fib timing data structures
	//
	RtlZeroMemory(Adapter->FibTimes, sizeof(ALL_FIB_TIMES));

	for (i = 0; i < MAX_FSACOMMAND_NUM; i++) {

		FibTimesPtr = &Adapter->FibTimes->FileSys[i];

		FibTimesPtr->Minimum.LowPart = 0xffffffff;
		FibTimesPtr->Minimum.HighPart = 0x7fffffff;
		FibTimesPtr->AdapterMinimum.LowPart = 0xffffffff;
		FibTimesPtr->AdapterMinimum.HighPart = 0x7fffffff;
	}
	for (i = 0; i < MAX_RW_FIB_TIMES; i++) {

		FibTimesPtr = &Adapter->FibTimes->Read[i];

		FibTimesPtr->Minimum.LowPart = 0xffffffff;
		FibTimesPtr->Minimum.HighPart = 0x7fffffff;
		FibTimesPtr->AdapterMinimum.LowPart = 0xffffffff;
		FibTimesPtr->AdapterMinimum.HighPart = 0x7fffffff;
	}
	for (i = 0; i < MAX_RW_FIB_TIMES; i++) {

		FibTimesPtr = &Adapter->FibTimes->Write[i];

		FibTimesPtr->Minimum.LowPart = 0xffffffff;
		FibTimesPtr->Minimum.HighPart = 0x7fffffff;
		FibTimesPtr->AdapterMinimum.LowPart = 0xffffffff;
		FibTimesPtr->AdapterMinimum.HighPart = 0x7fffffff;
	}

	FibTimesPtr = &Adapter->FibTimes->Other;

	FibTimesPtr->Minimum.LowPart = 0xffffffff;
	FibTimesPtr->Minimum.HighPart = 0x7fffffff;
	FibTimesPtr->AdapterMinimum.LowPart = 0xffffffff;
	FibTimesPtr->AdapterMinimum.HighPart = 0x7fffffff;

	Irp->IoStatus.Information = 0;

	return (STATUS_SUCCESS);

}
#endif	// GATHER_FIB_TIMES

#ifndef unix_aif
int
FsaCtlOpenGetAdapterFib(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
	)
/*++

Routine Description:

    This routine will get the next Fib, if available, from the AdapterFibContext
	passed in from the user.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_INSUFFICIENT_RESOURCES - If a memory allocation failed.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	PGET_ADAPTER_FIB_CONTEXT AdapterFibContext;
//	HANDLE Event;
//    PKEVENT eventObject = (PKEVENT) NULL;
	int Status;

	//
	// The context must be allocated from NonPagedPool because we need to use MmIsAddressValid.
	//

	AdapterFibContext = OsAllocMemory(sizeof(GET_ADAPTER_FIB_CONTEXT), OS_ALLOC_MEM_SLEEP);

	if (AdapterFibContext == NULL) {

		Status = ENOMEM;

	} else {

		AdapterFibContext->NodeTypeCode = FSAFS_NTC_GET_ADAPTER_FIB_CONTEXT;
		AdapterFibContext->NodeByteSize = sizeof(GET_ADAPTER_FIB_CONTEXT);


		//
		// Initialize the conditional variable use to wait for the next AIF.
		//

		OsCv_init( &AdapterFibContext->UserEvent);

		//
		// Set WaitingForFib to FALSE to indicate we are not in a WaitForSingleObject
		//

		AdapterFibContext->WaitingForFib = FALSE;

		//
		// Initialize the FibList and set the count of fibs on the list to 0.
		//

		AdapterFibContext->FibCount = 0;
		InitializeListHead(&AdapterFibContext->FibList);

		//
		// Overload FileObject with a time stamp.
		//
		AdapterFibContext->FileObject = (void *)OsGetSeconds();

		//
		// Now add this context onto the adapter's AdapterFibContext list.
		//

		OsCvLockAcquire(Adapter->AdapterFibMutex);

		InsertTailList(&Adapter->AdapterFibContextList, &AdapterFibContext->NextContext);

		OsCvLockRelease(Adapter->AdapterFibMutex);

		if (COPYOUT( &AdapterFibContext, (caddr_t) IoctlCmdPtr->arg, sizeof(PGET_ADAPTER_FIB_CONTEXT), 
						 IoctlCmdPtr->flag )) {

			Status = EFAULT;
			
		} else {
		
			Status = 0;

		}	

	}

	return (Status);
}

int
FsaCtlGetNextAdapterFib(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
	)
/*++

Routine Description:

    This routine will get the next Fib, if available, from the AdapterFibContext
	passed in from the user.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_NO_MORE_ENTRIES - There are no more Fibs for this AdapterFibContext.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	GET_ADAPTER_FIB_IOCTL AdapterFibIoctl;
	PGET_ADAPTER_FIB_CONTEXT AdapterFibContext, aifcp;
	PFIB Fib;
	int Status;
	PLIST_ENTRY Entry;
	int found;

	if (COPYIN( (caddr_t) IoctlCmdPtr->arg, (caddr_t) &AdapterFibIoctl,
					sizeof(GET_ADAPTER_FIB_IOCTL), IoctlCmdPtr->flag )) {
		return (EFAULT);
	}

	//
	// Extract the AdapterFibContext from the Input parameters.
	//

	AdapterFibContext = (PGET_ADAPTER_FIB_CONTEXT) AdapterFibIoctl.AdapterFibContext;

	//
	// Verify that the HANDLE passed in was a valid AdapterFibContext
	//
	// Search the list of AdapterFibContext addresses on the adapter to be sure
	// this is a valid address

	found = 0;
	Entry = Adapter->AdapterFibContextList.Flink;

	while ( Entry != &Adapter->AdapterFibContextList ) {
			aifcp = CONTAINING_RECORD ( Entry, GET_ADAPTER_FIB_CONTEXT, NextContext );
			if ( AdapterFibContext == aifcp ) {   // We found a winner
					found = 1;
					break;
			}
			Entry = Entry->Flink;
	}

	if ( found == 0 ) {
			return ( EINVAL );;
	}

	if ( (AdapterFibContext->NodeTypeCode != FSAFS_NTC_GET_ADAPTER_FIB_CONTEXT) ||
		 (AdapterFibContext->NodeByteSize != sizeof(GET_ADAPTER_FIB_CONTEXT)) ) {

		return ( EINVAL );

	}

	Status = STATUS_SUCCESS;

	OsCvLockAcquire(Adapter->AdapterFibMutex);

	//
	// If there are no fibs to send back, then either wait or return EAGAIN
	//
return_fib:

	if (!IsListEmpty(&AdapterFibContext->FibList)) {

		PLIST_ENTRY Entry;

		//
		// Pull the next fib from the FibList
		//
		Entry = RemoveHeadList(&AdapterFibContext->FibList);

		Fib = CONTAINING_RECORD( Entry, FIB, Header.FibLinks );

		AdapterFibContext->FibCount--;

		if (COPYOUT( Fib, AdapterFibIoctl.AifFib, sizeof(FIB), IoctlCmdPtr->flag )) {

			OsCvLockRelease( Adapter->AdapterFibMutex );
			OsFreeMemory( Fib, sizeof(Fib) );
			return (EFAULT);

		}	

		//
		// Free the space occupied by this copy of the fib.
		//

		OsFreeMemory(Fib, sizeof(FIB));

        Status = 0;

		//
		// Overload FileObject with a time stamp
		// 
		AdapterFibContext->FileObject = ( void * )OsGetSeconds();

	} else {

		if (AdapterFibIoctl.Wait) {
			
			if (OsCv_wait_sig( &AdapterFibContext->UserEvent, Adapter->AdapterFibMutex ) == 0) {

				Status = EINTR;

			} else {
			
				goto return_fib;
				
			}
		} else {
					
			Status = EAGAIN;

		}	

	}
	OsCvLockRelease( Adapter->AdapterFibMutex );

	return (Status);
}

int
FsaCtlCloseGetAdapterFib(
	IN PAFA_COMM_ADAPTER	Adapter,
	IN PAFA_IOCTL_CMD		IoctlCmdPtr
	)
/*++

Routine Description:

    This routine will close down the AdapterFibContext passed in from the user.

Arguments:

	Adapter - Supplies which adapter is being processed.

    Irp - Supplies the Irp being processed.

Return Value:

	STATUS_INVALID_PARAMETER - If the AdapterFibContext was not a valid pointer.

	STATUS_SUCCESS		 - Everything OK.

--*/
{
	PGET_ADAPTER_FIB_CONTEXT AdapterFibContext, aifcp;
	AAC_STATUS Status;

	PLIST_ENTRY Entry;
	int found;

	//
	// Extract the AdapterFibContext from the Input parameters
	//

	AdapterFibContext = (PGET_ADAPTER_FIB_CONTEXT) IoctlCmdPtr->arg;

	if (AdapterFibContext == 0) {
		cmn_err(CE_WARN, "FsaCtlCloseGetAdapterFib: AdapterFibContext is NULL");
		return(EINVAL);
	}

	//
	// Verify that the HANDLE passed in was a valid AdapterFibContext
	//
	// Search the list of AdapterFibContext addresses on the adapter to be sure
	// this is a valid address

	found = 0;
	Entry = Adapter->AdapterFibContextList.Flink;

	while ( Entry != &Adapter->AdapterFibContextList ) {
			aifcp = CONTAINING_RECORD ( Entry, GET_ADAPTER_FIB_CONTEXT, NextContext );
			if ( AdapterFibContext == aifcp ) {   // We found a winner
					found = 1;
					break;
			}
			Entry = Entry->Flink;
	}

	if ( found == 0 ) {
		return ( 0 ); // Already Gone
	}

	if ( (AdapterFibContext->NodeTypeCode != FSAFS_NTC_GET_ADAPTER_FIB_CONTEXT) ||
		 (AdapterFibContext->NodeByteSize != sizeof(GET_ADAPTER_FIB_CONTEXT)) ) {

		return (EINVAL);

	}

	OsCvLockAcquire(Adapter->AdapterFibMutex);

	Status = FsaCloseAdapterFibContext(Adapter, AdapterFibContext);

	OsCvLockRelease(Adapter->AdapterFibMutex);

	return (Status);
}

int
FsaCloseAdapterFibContext(
	IN PAFA_COMM_ADAPTER			Adapter,
	IN PGET_ADAPTER_FIB_CONTEXT		AdapterFibContext
	)
{
	int Status;
	PFIB Fib;

	//
	// First free any FIBs that have not been consumed yet.
	//

	while (!IsListEmpty(&AdapterFibContext->FibList)) {

		PLIST_ENTRY Entry;

		//
		// Pull the next fib from the FibList
		//

		Entry = RemoveHeadList(&AdapterFibContext->FibList);

		Fib = CONTAINING_RECORD( Entry, FIB, Header.FibLinks );

		AdapterFibContext->FibCount--;

		//
		// Free the space occupied by this copy of the fib.
		//

		OsFreeMemory(Fib, sizeof(FIB));
	}

	//
	// Remove the Context from the AdapterFibContext List
	//

	RemoveEntryList(&AdapterFibContext->NextContext);

	OsCv_destroy( &AdapterFibContext->UserEvent );

	//
	// Invalidate context
	//

	AdapterFibContext->NodeTypeCode = 0;

	//
	// Free the space occupied by the Context
	//

	OsFreeMemory(AdapterFibContext, sizeof(GET_ADAPTER_FIB_CONTEXT));

	Status = STATUS_SUCCESS;

    return Status;
}
#endif 

AAC_STATUS
AfaCommOpenAdapter(
	IN PVOID Arg
	)
/*++

Routine Description:

	The routine will get called by the miniport each time a user issues a CreateFile on the DeviceObject
	for the adapter.

	The main purpose of this routine is to set up any data structures that may be needed
	to handle any requests made on this DeviceObject.

Arguments:

	Adapter - Pointer to which adapter miniport was opened.


Return Value:

	STATUS_SUCCESS

--*/

{
	PAFA_COMM_ADAPTER	Adapter = (PAFA_COMM_ADAPTER) Arg;
	AAC_STATUS Status = STATUS_SUCCESS;
	PAFA_CLASS_DRIVER ClassDriver;

	ClassDriver = Adapter->ClassDriverList;

	while (ClassDriver) {

		if (ClassDriver->OpenAdapter) {

			Status = ClassDriver->OpenAdapter( ClassDriver->ClassDriverExtension );

			if (Status != STATUS_SUCCESS)
				break;
		}

		ClassDriver = ClassDriver->Next;
	}

	return ( Status );
}

AAC_STATUS
AfaCommCloseAdapter(
	IN PVOID Arg
	)
/*++

Routine Description:

	This routine will get called by the miniport each time a user issues a CloseHandle on the DeviceObject
	for the adapter.

	The main purpose of this routine is to cleanup any data structures that have been set up
	while this FileObject has been opened.

	This routine loops through all of the AdapterFibContext structures to determine if any need
	to be deleted for this FileObject.

Arguments:

	Adapter - Pointer to adapter miniport

	Irp - Pointer to Irp that caused this close

Return Value:

	Status value returned from File system driver AdapterClose

--*/
{
	PAFA_COMM_ADAPTER	Adapter = (PAFA_COMM_ADAPTER) Arg;
	PLIST_ENTRY Entry, NextEntry;
	PGET_ADAPTER_FIB_CONTEXT AdapterFibContext;
	AAC_STATUS Status = STATUS_SUCCESS;
	PAFA_CLASS_DRIVER ClassDriver;

	OsCvLockAcquire(Adapter->AdapterFibMutex);

	Entry = Adapter->AdapterFibContextList.Flink;

	//
	// Loop through all of the AdapterFibContext, looking for any that
	// were created with the FileObject that is being closed.
	//
	while (Entry != &Adapter->AdapterFibContextList) {

		//
		// Extract the AdapterFibContext
		//
		AdapterFibContext = CONTAINING_RECORD( Entry, GET_ADAPTER_FIB_CONTEXT, NextContext );

		// 
		// Save the next entry because CloseAdapterFibContext will delete the AdapterFibContext
		//
		NextEntry = Entry->Flink;

		Entry = NextEntry;

	}

#ifdef unix_config_file
	//
	// If this FileObject had the adapter open for configuration, then release it.
	//
	if ( Adapter->AdapterConfigFileObject == IrpSp->FileObject ) {

		Adapter->AdapterConfigFileObject = NULL;

	}
#endif

	OsCvLockRelease(Adapter->AdapterFibMutex);

	ClassDriver = Adapter->ClassDriverList;

	while (ClassDriver) {

		if (ClassDriver->CloseAdapter) {

			Status = ClassDriver->CloseAdapter( ClassDriver->ClassDriverExtension );

			if (Status != STATUS_SUCCESS)
				break;
		}

		ClassDriver = ClassDriver->Next;
	}

	return ( Status );

}

