/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999-2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
 this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation 
 and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
 may be used to endorse or promote products derived from this software 
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
****************************************************************************/
/*
*                                                                     		*
* This software is supplied under the terms of the license included   		*
* above.  All use of this driver must be in accordance with the terms 		*
* of that license.                                                    		*
*                                                                     		*
Module Name:                                                                            **
    eeprom.c                                                                            **
                                                                                        **
Abstract:                                                                               **
    This module contains routines to read and write to a serial EEPROM                  **
                                                                                        **
    This driver runs on the following hardware:                                         **
        82557 and 82558 based PCI 10/100Mb ethernet adapters                            **
                                                                                        **
*****************************************************************************************/

#include "eeprom.h"

//****************************************************************************************
//
//            I/O based Read EEPROM Routines
//
//****************************************************************************************


//-----------------------------------------------------------------------------
// Procedure: ValidateEEprom
//
// Description: Determine whether the eeprom is accurate
//
// Arguments:
//          Ptr to this card's adapter data structure 
// 
// Returns: TRUE or FALSE whether the eeprom is clean
//-----------------------------------------------------------------------------

boolean_t ValidateEEprom(bd_config_t *adapter)
{
    __u16 idx, checksum = 0;
    int   status = B_TRUE;
    
    // eeprom size is initialized to zero
    if (!adapter->bddp->EEpromSize)
        adapter->bddp->EEpromSize = GetEEpromSize (adapter);

    for (idx = 0; idx < adapter->bddp->EEpromSize; idx++)
        checksum += ReadEEprom (adapter, idx);

    if (checksum != le16_to_cpu(EEPROM_CHECKSUM))
        status = B_FALSE;

    return status;
}


//----------------------------------------------------------------------------------------
// Procedure:   GetEEpromSize
//
// Description: This routine determines the size of the EEPROM.  This value should be
//              checked for validity - ie. is it too big or too small.  The size returned
//              is then passed to the read/write functions.
//
// Warning:
//      Do not call this function before InitializeEEpromStructure is called 
//      (in MiniportInitialize).  This will result in dereferencing a NULL pointer.
//
// Returns:
//      Size of the eeprom, or zero if an error occured
//----------------------------------------------------------------------------------------

__u16 GetEEpromSize(bd_config_t *adapter)
{
    __u16 x, size = 1;        // must be one to accumulate a product

    // if we've already stored this data, read from memory
    if (adapter->EepromData.Size) {
        size = adapter->EepromData.Size;

    } else { // otherwise, read from the eeprom
        // Set EEPROM semaphore.
        if (adapter->bddp->rev_id >= D102_REV_ID) {
            WAIT_FOR_CONDITION (adapter, EEPROM_SEMAPHORE_TIMEOUT,
                                EepromSetSemaphore (adapter));
        }

        // enable the eeprom by setting EECS.
        x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
        x &= ~(EEDI | EEDO | EESK);
        x |= EECS;
        writew(x ,&CSR_EEPROM_CONTROL_FIELD(adapter));

        // write the read opcode
        ShiftOutBits (adapter, EEPROM_READ_OPCODE, 3);

        // experiment to discover the size of the eeprom.  request register zero
        // and wait for the eeprom to tell us it has accepted the entire address.
        x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
        do {
            size *= 2;          // each bit of address doubles eeprom size
            x |= EEDO;          // set bit to detect "dummy zero"
            x &= ~EEDI;         // address consists of all zeros

            writew(x ,&CSR_EEPROM_CONTROL_FIELD(adapter));
            udelay (STALL_TIME);
            RaiseClock (adapter, &x);
            LowerClock (adapter, &x);

            // check for "dummy zero"
            x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
            if (size > EEPROM_MAX_WORD_SIZE) {
                size = 0;
                break;
            }
        } while (x & EEDO);

        // read in the value requested
        (void) ShiftInBits (adapter);
        EEpromCleanup (adapter);

        // Clear EEPROM Semaphore.
        if (adapter->bddp->rev_id >= D102_REV_ID) {
            EepromResetSemaphore (adapter);
        }

    }

    return size;
}


//----------------------------------------------------------------------------------------
// Procedure:   EEpromAddressSize
//
// Description: determines the number of bits in an address for the eeprom acceptable
//              values are 64, 128, and 256
//
// Arguments:
//      size of the eeprom
//
// Returns:
//      bits in an address for that size eeprom
//----------------------------------------------------------------------------------------

__u16 EEpromAddressSize(__u16 size)
{
    switch (size)
    {
        case 64:    return 6;
        case 128:   return 7;
        case 256:   return 8;
    }

    return 0;  //fix compiler warning or error!
}


//----------------------------------------------------------------------------------------
// Procedure:   ReadEEprom
//
// Description: This routine serially reads one word out of the EEPROM.
//
// Arguments:
//      adapter - our adapter context
//      reg - EEPROM word to read.
//
// Warning:
//      Do not call this function before InitializeEEpromStructure is called 
//      (in MiniportInitialize).  This will result in dereferencing a NULL pointer.
//
// Returns:
//      Contents of EEPROM word (reg).
//----------------------------------------------------------------------------------------

__u16 ReadEEprom(bd_config_t *adapter, __u16 reg)
{
    __u16      x, data, bits;

    // Set EEPROM semaphore.
    if (adapter->bddp->rev_id >= D102_REV_ID) {
        WAIT_FOR_CONDITION (adapter, EEPROM_SEMAPHORE_TIMEOUT, EepromSetSemaphore (adapter));
    }
    
    // eeprom size is initialized to zero
    if (!adapter->bddp->EEpromSize)
        adapter->bddp->EEpromSize = GetEEpromSize (adapter);
    
    bits = EEpromAddressSize (adapter->bddp->EEpromSize);
    
    // select EEPROM, reset bits, set EECS
    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
    
    x &= ~(EEDI | EEDO | EESK);
    x |= EECS;
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));
    
    // write the read opcode and register number in that order
    // The opcode is 3bits in length, reg is 'bits' bits long
    ShiftOutBits (adapter, EEPROM_READ_OPCODE, 3);
    ShiftOutBits (adapter, reg, bits);
    
    // Now read the data (16 bits) in from the selected EEPROM word
    data = ShiftInBits (adapter);
    
    EEpromCleanup (adapter);
    
    // Clear EEPROM Semaphore.
    if (adapter->bddp->rev_id >= D102_REV_ID) {
        EepromResetSemaphore (adapter);
    }
    
    return data;
}


//----------------------------------------------------------------------------------------
// Procedure:   ShiftOutBits
//
// Description: This routine shifts data bits out to the EEPROM.
//
// Arguments:
//      data - data to send to the EEPROM.
//      count - number of data bits to shift out.
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void ShiftOutBits(bd_config_t *adapter, __u16 data, __u16 count)
{
    __u16 x, mask;

    mask = 1 << (count - 1);
    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
    x &= ~(EEDO | EEDI);

    do {
        x &= ~EEDI;
        if (data & mask)
            x |= EEDI;

        writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));
        udelay (STALL_TIME);
        RaiseClock (adapter, &x);
        LowerClock (adapter, &x);
        mask = mask >> 1;
    } while (mask);

    x &= ~EEDI;
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));
}


//----------------------------------------------------------------------------------------
// Procedure:   RaiseClock
//
// Description: This routine raises the EEPROM's clock input (EESK)
//
// Arguments:
//      x - Ptr to the EEPROM control register's current value
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void RaiseClock(bd_config_t *adapter, __u16 * x)
{
    *x = *x | EESK;
    writew(*x, &CSR_EEPROM_CONTROL_FIELD(adapter));
    udelay (STALL_TIME);
}


//----------------------------------------------------------------------------------------
// Procedure:   LowerClock
//
// Description: This routine lower's the EEPROM's clock input (EESK)
//
// Arguments:
//      x - Ptr to the EEPROM control register's current value
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void LowerClock(bd_config_t *adapter, __u16 *x)
{
    *x = *x & ~EESK;
    writew(*x, &CSR_EEPROM_CONTROL_FIELD(adapter));
    udelay (STALL_TIME);
}


//----------------------------------------------------------------------------------------
// Procedure:   ShiftInBits
//
// Description: This routine shifts data bits in from the EEPROM.
//
// Arguments:
//
// Returns:
//      The contents of that particular EEPROM word
//----------------------------------------------------------------------------------------

__u16 ShiftInBits(bd_config_t *adapter)
{
    __u16 x, d, i;

    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
    x &= ~(EEDO | EEDI);
    d = 0;

    for (i = 0; i < 16; i++) {
        d <<= 1;
        RaiseClock (adapter, &x);

        x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));

        x &= ~EEDI;
        if (x & EEDO)
            d |= 1;

        LowerClock (adapter, &x);
    }

    return d;
}


//----------------------------------------------------------------------------------------
// Procedure:   EEpromCleanup
//
// Description: This routine returns the EEPROM to an idle state
//
// Arguments:
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void EEpromCleanup(bd_config_t *adapter)
{
    __u16 x;
    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));

    x &= ~(EECS | EEDI);
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));

    RaiseClock (adapter, &x);
    LowerClock (adapter, &x);
}

//----------------------------------------------------------------------------------------
// Procedure:   MemReadEEpromNodeAddress
//
// Description: This routine returns the ethernet address from the eeprom
//
// Arguments: pointer to the CSR mapped memory i/o space, pointer to buffer to receive
//            the address, and the size of the buffer.
//
// Returns: true/false  succeed/fail
//----------------------------------------------------------------------------------------

boolean_t ReadEEpromNodeAddress(bd_config_t *adapter, __u8 *buffer, unsigned int size)
{
    __u16 value;
    __u8  data [ETHERNET_ADDRESS_LENGTH];
    __u16 idx;

    if (size < ETHERNET_ADDRESS_LENGTH)
        return FALSE;

    for (idx = 0; idx < ETHERNET_ADDRESS_LENGTH / 2; idx++) {
        value = ReadEEprom (adapter, (__u16) (EEPROM_NODE_ADDRESS_BYTE_0 + idx));
        data [idx*2]     = (__u8) value;
        data [idx*2 + 1] = (__u8) (value >> 8);
    }
    memcpy(buffer, data, 6);

    return TRUE;
}

//****************************************************************************************
//
//            EEPROM Write Routines
//
//****************************************************************************************

//----------------------------------------------------------------------------------------
// Procedure:   UpdateChecksum
//
// Description: Calculates the checksum and writes it to the EEProm.  This routine
//              assumes that the checksum word is the last word in a 64 word EEPROM.  It
//              calculates the checksum accroding to the formula: Checksum = 0xBABA -
//              (sum of first 63 words).
//
// Arguments:
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void UpdateChecksum(bd_config_t *adapter)
{
    __u16 idx, xsumIndex, checksum = 0;

    // eeprom size is initialized to zero
    if (!adapter->bddp->EEpromSize)
        adapter->bddp->EEpromSize = GetEEpromSize (adapter);

    xsumIndex = adapter->bddp->EEpromSize - 1;
    for (idx = 0; idx < xsumIndex; idx++)
        checksum += ReadEEprom (adapter, idx);

    checksum = EEPROM_CHECKSUM - checksum;
    WriteEEprom (adapter, idx, checksum);
}

/******************************************************************************
 * Procedure:   QuickUpdateChecksum
 *
 * Description:
 *    Calculates the checksum and writes it to the EEProm.  This routine
 *    assumes that the checksum word is the last WORD in a EEPROM memory.
 *    It calculates the checksum accroding to the formula:
 *          checksum = current checksum - new WORD value + old WORD value
 *
 * Arguments:
 *      adapter - pointer to adapter
 *      reg - updated EEPROM memory WORD location
 *      oldData - old value at the specified memory location
 *      newData - new value
 *
 * Returns: none
 *****************************************************************************/
void QuickUpdateChecksum(bd_config_t *adapter, __u16 reg, __u16 oldData, __u16 newData)
{
    __u16 xsumIndex, curData, curXsum;

    // eeprom size is initialized to zero
    if (!adapter->bddp->EEpromSize)
        adapter->bddp->EEpromSize = GetEEpromSize (adapter);

    xsumIndex = adapter->bddp->EEpromSize - 1;
    curData = ReadEEprom(adapter, reg);
    curXsum = ReadEEprom(adapter, xsumIndex);
    if (curData == newData) {
        curXsum -= (newData - oldData);
        WriteEEprom (adapter, xsumIndex, curXsum);

    } else if (curData != oldData) {     /* unexpected data */
        UpdateChecksum(adapter);      /* calculate xsum - use all values */
    }
}

//----------------------------------------------------------------------------------------
// Procedure:   WriteEEprom
//
// Description: This routine writes a word to a specific EEPROM location.
//
// Arguments:
//      reg - The EEPROM word that we are going to write to.
//      data - The data (word) that we are going to write to the EEPROM.
//
// Warning:
//      Do not call this function before InitializeEEpromStructure is called 
//      (in MiniportInitialize).  This will result in dereferencing a NULL pointer.
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void WriteEEprom(bd_config_t *adapter, __u16 reg, __u16 data)
{
    __u16 x;
    __u16 bits;

    // Set EEPROM semaphore.
    if (adapter->bddp->rev_id >= D102_REV_ID) {
        WAIT_FOR_CONDITION (adapter, EEPROM_SEMAPHORE_TIMEOUT, EepromSetSemaphore (adapter));
    }

    // eeprom size is initialized to zero
    if (!adapter->bddp->EEpromSize)
        adapter->bddp->EEpromSize = GetEEpromSize (adapter);

    bits = EEpromAddressSize (adapter->bddp->EEpromSize);

    /* select EEPROM, mask off ASIC and reset bits, set EECS */
    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
    x &= ~(EEDI | EEDO | EESK);
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));
    udelay(1);
    x |= EECS;
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));

    ShiftOutBits (adapter, EEPROM_EWEN_OPCODE, 5);
    ShiftOutBits (adapter, reg, (__u16) (bits-2));
    if (WaitEEpromCmdDone (adapter) == FALSE) {
        return;
    }
    
    /* write the new word to the EEPROM & send the write opcode the EEPORM */
    ShiftOutBits (adapter, EEPROM_WRITE_OPCODE, 3);

    /* select which word in the EEPROM that we are writing to */
    ShiftOutBits (adapter, reg, bits);

    /* write the data to the selected EEPROM word */
    ShiftOutBits (adapter, data, 16);
    if (WaitEEpromCmdDone (adapter) == FALSE) {
        return;
    }
    
    ShiftOutBits (adapter, EEPROM_EWDS_OPCODE, 5);
    ShiftOutBits (adapter, reg, (__u16) (bits-2));
    if (WaitEEpromCmdDone (adapter) == FALSE) {
        return;
    }

    EEpromCleanup (adapter);

    // Clear EEPROM Semaphore.
    if (adapter->bddp->rev_id >= D102_REV_ID) {
        EepromResetSemaphore (adapter);
    }

    // update the eeprom buffer so subsequent reads will be accurate
    if (adapter->EepromData.Initialized) {
        adapter->EepromData.Data[reg] = data;
    }
}

/******************************************************************************
 * Procedure:   WriteBlockEEprom
 *
 * Description: This routine writes a block of words to the specified EEPROM
 *       locations.
 *
 * Arguments:
 *      adapter - pointer to adapter
 *    startReg - starting EEPROM word that we are going to write to.
 *    dataLength - number of words that we are going to write
 *    pData - Data (words) that we are going to write to the EEPROM.
 *
 * Returns: none
 *****************************************************************************/
void WriteBlockEEprom(bd_config_t *adapter, __u16 startReg, __u16 dataLength, __u16 *ptrData)
{
    __u16 index, reg;

    for (index = 0, reg = startReg; index < dataLength; index++, reg++) {
         WriteEEprom(adapter, reg, ptrData[index]);
    }
}

//----------------------------------------------------------------------------------------
// Procedure:   WriteEEpromAndChecksum
//
// Description: This routine writes a word to a specific EEPROM location. Then
//            calls updates the checksum value.
//
// Arguments:
//     adapter - pointer to adapter
//    reg - The EEPROM word that we are going to write to.
//    data - The data (word) that we are going to write to the EEPROM.
//
// Returns: (none)
//----------------------------------------------------------------------------------------
void WriteEEpromAndChecksum(bd_config_t *adapter, __u16 reg, __u16 data)
{
    __u16 oldData = ReadEEprom(adapter, reg);
    
    WriteEEprom (adapter, reg, data);
    QuickUpdateChecksum (adapter, reg, oldData, data);
}

//----------------------------------------------------------------------------------------
// Procedure:   WaitEEpromCmdDone
//
// Description: This routine waits for the the EEPROM to finish its command.  Specifically,
//              it waits for EEDO (data out) to go high.
//
// Arguments:
//
// Returns:
//      TRUE - If the command finished
//      FALSE - If the command never finished (EEDO stayed low)
//----------------------------------------------------------------------------------------

__u16 WaitEEpromCmdDone(bd_config_t *adapter)
{
    __u16 x, i;

    StandBy (adapter);
    for (i=0; i < 200; i++) {
        x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));
        if (x & EEDO)
            return TRUE;
        udelay (STALL_TIME);
    }
    
    return FALSE;
}


//----------------------------------------------------------------------------------------
// Procedure:   StandBy
//
// Description: This routine lowers the EEPROM chip select (EECS) for a few microseconds.
//
// Arguments:
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void StandBy(bd_config_t *adapter)
{
    __u16 x;

    x = readw(&CSR_EEPROM_CONTROL_FIELD(adapter));

    x &= ~(EECS | EESK);
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));

    udelay(4);
    x |= EECS;
    writew(x, &CSR_EEPROM_CONTROL_FIELD(adapter));
    udelay(1);    
}

//----------------------------------------------------------------------------------------
// Procedure:   EnableDebugMode
//
// Description:
//
// Arguments:
//
// Returns: (none)
//----------------------------------------------------------------------------------------

void EnableDebugMode(bd_config_t *adapter)
{

}
