/*****************************************************************************
 *****************************************************************************

 Copyright (c) 1999 - 2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, 
     this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation 
     and/or other materials provided with the distribution.

  3. Neither the name of Intel Corporation nor the names of its contributors 
     may be used to endorse or promote products derived from this software 
     without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 *****************************************************************************/

/* Workfile: phy.c  */
/* Revision: 31  */
/* Date: 8/03/01 10:39a  */

#include "e1000.h"

#define GOOD_MII_IF 0

u16 e1000_read_phy_register (IN struct adapter * Adapter, IN u32 RegAddress, IN u32 PhyAddress);

void e1000_write_phy_register (IN struct adapter * Adapter, IN u32 RegAddress, IN u32 PhyAddress, IN u16 Data);

static void MIIShiftOutPhyData (IN struct adapter * Adapter, IN u32 Data, IN u16 Count);

static void RaiseMdcClock (IN struct adapter * Adapter, IN OUT u32 * CtrlRegValue);

static void LowerMdcClock (IN struct adapter * Adapter, IN OUT u32 * CtrlRegValue);

static u16 MIIShiftInPhyData (IN struct adapter * Adapter);

void e1000_phy_hardware_reset (IN struct adapter * Adapter);

unsigned char e1000_phy_reset (IN struct adapter * Adapter);

unsigned char e1000_phy_setup (IN struct adapter * Adapter, IN u32 DeviceControlReg);

static unsigned char e1000_phy_setupAutoNegAdvertisement (IN struct adapter * Adapter);

static void PhyForceSpeedAndDuplex (IN struct adapter * Adapter);

void e1000_configure_mac_to_phySettings (IN struct adapter * Adapter, IN u16 MiiRegisterData);

void e1000_configure_collision_distance (IN struct adapter * Adapter);

void e1000_display_mii_contents (IN struct adapter * Adapter, IN u8 PhyAddress);

u32 e1000_auto_detect_gigabit_phy (IN struct adapter * Adapter);

void Pxne1000_phy_resetDsp (IN struct adapter * Adapter);

void PxnIntegratedPhyLoopback (IN struct adapter * Adapter, IN u16 Speed);

void PxnPhyEnableReceiver (IN struct adapter * Adapter);

void PxnPhyDisableReceiver (IN struct adapter * Adapter);

unsigned char e1000_wait_for_auto_neg (IN struct adapter * Adapter);

u16
e1000_read_phy_register (IN struct adapter * Adapter, IN u32 RegAddress, IN u32 PhyAddress)
{
	u32 i;
	u32 Data;
	u32 Command = 0;

	ASSERT (RegAddress <= MAX_PHY_REG_ADDRESS);

	if (Adapter->MacType > MAC_LIVENGOOD) {

		Command = ((RegAddress << MDI_REGADD_SHIFT) | (PhyAddress << MDI_PHYADD_SHIFT) | (E1000_MDI_READ));

		E1000_WRITE_REG (Mdic, Command);

		for (i = 0; i < 10; i++) {
			udelay (10);

			Data = E1000_READ_REG (Mdic);

			if (Data & E1000_MDI_READY)
				break;
		}
	} else {

		MIIShiftOutPhyData (Adapter, PHY_PREAMBLE, PHY_PREAMBLE_SIZE);

		Command = ((RegAddress) | (PhyAddress << 5) | (PHY_OP_READ << 10) | (PHY_SOF << 12));

		MIIShiftOutPhyData (Adapter, Command, 14);

		Data = (u32) MIIShiftInPhyData (Adapter);
	}

	ASSERT (!(Data & E1000_MDI_ERR));

	return ((u16) Data);
}

void
e1000_write_phy_register (IN struct adapter * Adapter, IN u32 RegAddress, IN u32 PhyAddress, IN u16 Data)
{
	u32 i;
	u32 Command = 0;
	u32 MdicRegValue;

	ASSERT (RegAddress <= MAX_PHY_REG_ADDRESS);

	if (Adapter->MacType > MAC_LIVENGOOD) {

		Command = (((u32) Data) | (RegAddress << MDI_REGADD_SHIFT) | (PhyAddress << MDI_PHYADD_SHIFT) | (E1000_MDI_WRITE));

		E1000_WRITE_REG (Mdic, Command);

		for (i = 0; i < 10; i++) {
			udelay (10);

			MdicRegValue = E1000_READ_REG (Mdic);

			if (MdicRegValue & E1000_MDI_READY)
				break;
		}

	} else {

		MIIShiftOutPhyData (Adapter, PHY_PREAMBLE, PHY_PREAMBLE_SIZE);

		Command = ((PHY_TURNAROUND) | (RegAddress << 2) | (PhyAddress << 7) | (PHY_OP_WRITE << 12) | (PHY_SOF << 14));
		Command <<= 16;
		Command |= ((u32) Data);

		MIIShiftOutPhyData (Adapter, Command, 32);
	}

	return;
}

static u16
MIIShiftInPhyData (IN struct adapter * Adapter)
{
	u32 CtrlRegValue;
	u16 Data = 0;
	u8 i;

	CtrlRegValue = E1000_READ_REG (Ctrl);

	CtrlRegValue &= ~E1000_CTRL_MDIO_DIR;
	CtrlRegValue &= ~E1000_CTRL_MDIO;

	E1000_WRITE_REG (Ctrl, CtrlRegValue);

	RaiseMdcClock (Adapter, &CtrlRegValue);
	LowerMdcClock (Adapter, &CtrlRegValue);

	for (Data = 0, i = 0; i < 16; i++) {
		Data = Data << 1;
		RaiseMdcClock (Adapter, &CtrlRegValue);

		CtrlRegValue = E1000_READ_REG (Ctrl);

		if (CtrlRegValue & E1000_CTRL_MDIO)
			Data |= 1;

		LowerMdcClock (Adapter, &CtrlRegValue);
	}

	RaiseMdcClock (Adapter, &CtrlRegValue);
	LowerMdcClock (Adapter, &CtrlRegValue);

	CtrlRegValue &= ~E1000_CTRL_MDIO;

	return (Data);
}

static void
MIIShiftOutPhyData (IN struct adapter * Adapter, IN u32 Data, IN u16 Count)
{
	u32 CtrlRegValue;
	u32 Mask;

	if (Count > 32)
		ASSERT (0);

	Mask = 0x01;
	Mask <<= (Count - 1);

	CtrlRegValue = E1000_READ_REG (Ctrl);

	CtrlRegValue |= (E1000_CTRL_MDIO_DIR | E1000_CTRL_MDC_DIR);

	while (Mask) {

		if (Data & Mask)
			CtrlRegValue |= E1000_CTRL_MDIO;
		else
			CtrlRegValue &= ~E1000_CTRL_MDIO;

		E1000_WRITE_REG (Ctrl, CtrlRegValue);

		udelay (2);

		RaiseMdcClock (Adapter, &CtrlRegValue);
		LowerMdcClock (Adapter, &CtrlRegValue);

		Mask = Mask >> 1;
	}

	CtrlRegValue &= ~E1000_CTRL_MDIO;
}

static void
RaiseMdcClock (IN struct adapter * Adapter, IN OUT u32 * CtrlRegValue)
{

	E1000_WRITE_REG (Ctrl, (*CtrlRegValue | E1000_CTRL_MDC));

	udelay (2);
}

static void
LowerMdcClock (IN struct adapter * Adapter, IN OUT u32 * CtrlRegValue)
{

	E1000_WRITE_REG (Ctrl, (*CtrlRegValue & ~E1000_CTRL_MDC));

	udelay (2);
}

void
e1000_phy_hardware_reset (IN struct adapter * Adapter)
{
	u32 ExtCtrlRegValue, CtrlRegValue;

	DEBUGFUNC ("e1000_phy_hardware_reset")

	    DEBUGOUT ("Resetting Phy...\n");

	if (Adapter->MacType > MAC_LIVENGOOD) {

		CtrlRegValue = E1000_READ_REG (Ctrl);

		CtrlRegValue |= E1000_CTRL_PHY_RST;

		E1000_WRITE_REG (Ctrl, CtrlRegValue);

		mdelay (20);

		CtrlRegValue &= ~E1000_CTRL_PHY_RST;

		E1000_WRITE_REG (Ctrl, CtrlRegValue);

		mdelay (20);
	} else {

		ExtCtrlRegValue = E1000_READ_REG (Exct);

		ExtCtrlRegValue |= E1000_CTRL_PHY_RESET_DIR4;

		E1000_WRITE_REG (Exct, ExtCtrlRegValue);

		mdelay (20);

		ExtCtrlRegValue = E1000_READ_REG (Exct);

		ExtCtrlRegValue &= ~E1000_CTRL_PHY_RESET4;

		E1000_WRITE_REG (Exct, ExtCtrlRegValue);

		mdelay (20);

		ExtCtrlRegValue = E1000_READ_REG (Exct);

		ExtCtrlRegValue |= E1000_CTRL_PHY_RESET4;

		E1000_WRITE_REG (Exct, ExtCtrlRegValue);

		mdelay (20);
	}

	return;
}

unsigned char
e1000_phy_reset (IN struct adapter * Adapter)
{
	u16 RegData;
	u16 i;

	DEBUGFUNC ("e1000_phy_reset")

	    RegData = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress);

	RegData |= MII_CR_RESET;

	e1000_write_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress, RegData);

	i = 0;
	while ((RegData & MII_CR_RESET) && i++ < 500) {
		RegData = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress);
		udelay (1);
	}

	if (i >= 500) {
		DEBUGOUT ("Timeout waiting for PHY to reset.\n");
		return FALSE;
	}

	return TRUE;
}

unsigned char
e1000_phy_setup (IN struct adapter * Adapter, u32 DeviceControlReg)
{
	u32 DeviceCtrlReg = 0;
	u16 MiiCtrlReg, MiiStatusReg;
	u16 MiiAutoNegAdvertiseReg, Mii1000TCtrlReg;
	u16 i, Data;
	u16 AutoNegHwSetting;
	u16 AutoNegFCSetting;
	unsigned char RestartAutoNeg = FALSE;
	unsigned char ForceAutoNegRestart = FALSE;

	DEBUGFUNC ("e1000_phy_setup")

	    ASSERT (Adapter->MacType >= MAC_LIVENGOOD);

	if (Adapter->MacType > MAC_WAINWRIGHT) {
		DeviceControlReg |= (E1000_CTRL_ASDE | E1000_CTRL_SLU);
		E1000_WRITE_REG (Ctrl, DeviceControlReg);
	} else {
		DeviceControlReg |= (E1000_CTRL_FRCSPD | E1000_CTRL_FRCDPX | E1000_CTRL_SLU);
		E1000_WRITE_REG (Ctrl, DeviceControlReg);

		if (Adapter->MacType == MAC_LIVENGOOD)
			e1000_phy_hardware_reset (Adapter);
	}

	Adapter->PhyAddress = e1000_auto_detect_gigabit_phy (Adapter);

	if (Adapter->PhyAddress > MAX_PHY_REG_ADDRESS) {

		DEBUGOUT ("e1000_phy_setup failure, did not detect valid phy.\n");
		return (FALSE);
	}

	DEBUGOUT1 ("Phy ID = %x \n", Adapter->PhyId);

	MiiCtrlReg = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress);

	DEBUGOUT1 ("MII Ctrl Reg contents = %x\n", MiiCtrlReg);

	if (!(MiiCtrlReg & MII_CR_AUTO_NEG_EN))
		ForceAutoNegRestart = TRUE;

	MiiCtrlReg &= ~(MII_CR_ISOLATE);

	e1000_write_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress, MiiCtrlReg);

	Data = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress);

	Data |= PXN_PSCR_ASSERT_CRS_ON_TX;

	DEBUGOUT1 ("Paxson PSCR: %x \n", Data);

	e1000_write_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress, Data);

	Data = e1000_read_phy_register (Adapter, PXN_EXT_PHY_SPEC_CTRL_REG, Adapter->PhyAddress);

	Data |= PXN_EPSCR_TX_CLK_25;

	e1000_write_phy_register (Adapter, PXN_EXT_PHY_SPEC_CTRL_REG, Adapter->PhyAddress, Data);

	MiiAutoNegAdvertiseReg = e1000_read_phy_register (Adapter, PHY_AUTONEG_ADVERTISEMENT, Adapter->PhyAddress);

	AutoNegHwSetting = (MiiAutoNegAdvertiseReg >> 5) & 0xF;

	Mii1000TCtrlReg = e1000_read_phy_register (Adapter, PHY_1000T_CTRL_REG, Adapter->PhyAddress);

	AutoNegHwSetting |= ((Mii1000TCtrlReg & 0x0300) >> 4);

	AutoNegFCSetting = ((MiiAutoNegAdvertiseReg & 0x0C00) >> 10);

	Adapter->AutoNegAdvertised &= AUTONEG_ADVERTISE_SPEED_DEFAULT;

	if (Adapter->AutoNegAdvertised == 0)
		Adapter->AutoNegAdvertised = AUTONEG_ADVERTISE_SPEED_DEFAULT;

	if (!ForceAutoNegRestart && Adapter->AutoNeg && (Adapter->AutoNegAdvertised == AutoNegHwSetting) && (Adapter->FlowControl == AutoNegFCSetting)) {
		DEBUGOUT ("No overrides - Reading MII Status Reg..\n");

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		DEBUGOUT1 ("MII Status Reg contents = %x\n", MiiStatusReg);

		if (MiiStatusReg & MII_SR_LINK_STATUS) {
			Data = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_STAT_REG, Adapter->PhyAddress);
			DEBUGOUT1 ("Paxson Phy Specific Status Reg contents = %x\n", Data);

			if (Adapter->MacType > MAC_WAINWRIGHT)
				e1000_configure_collision_distance (Adapter);
			else
				e1000_configure_mac_to_phySettings (Adapter, Data);

			e1000_config_flow_control_afterLinkUp (Adapter);

			return (TRUE);
		}
	}

	if (Adapter->AutoNeg) {
		DEBUGOUT ("Livengood - Reconfiguring auto-neg advertisement params\n");
		RestartAutoNeg = e1000_phy_setupAutoNegAdvertisement (Adapter);
	} else {
		DEBUGOUT ("Livengood - Forcing speed and duplex\n");
		PhyForceSpeedAndDuplex (Adapter);
	}

	if (RestartAutoNeg) {
		DEBUGOUT ("Restarting Auto-Neg\n");

		MiiCtrlReg = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress);

		MiiCtrlReg |= (MII_CR_AUTO_NEG_EN | MII_CR_RESTART_AUTO_NEG);

		e1000_write_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress, MiiCtrlReg);

		if (Adapter->WaitAutoNegComplete)
			e1000_wait_for_auto_neg (Adapter);

	}

	MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

	MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

	DEBUGOUT1 ("Checking for link status - MII Status Reg contents = %x\n", MiiStatusReg);

	for (i = 0; i < 10; i++) {
		if (MiiStatusReg & MII_SR_LINK_STATUS) {
			break;
		}
		udelay (10);
		DEBUGOUT (". ");

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);
	}

	if (MiiStatusReg & MII_SR_LINK_STATUS) {

		Data = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_STAT_REG, Adapter->PhyAddress);

		DEBUGOUT1 ("Paxson Phy Specific Status Reg contents = %x\n", Data);

		if (Adapter->MacType > MAC_WAINWRIGHT)
			e1000_configure_collision_distance (Adapter);
		else
			e1000_configure_mac_to_phySettings (Adapter, Data);

		e1000_config_flow_control_afterLinkUp (Adapter);

		DEBUGOUT ("Valid link established!!!\n");
	} else {
		DEBUGOUT ("Unable to establish link!!!\n");
	}

	return (TRUE);
}

unsigned char
e1000_phy_setupAutoNegAdvertisement (IN struct adapter * Adapter)
{
	u16 MiiAutoNegAdvertiseReg, Mii1000TCtrlReg;

	DEBUGFUNC ("e1000_phy_setupAutoNegAdvertisement")

	    MiiAutoNegAdvertiseReg = e1000_read_phy_register (Adapter, PHY_AUTONEG_ADVERTISEMENT, Adapter->PhyAddress);

	Mii1000TCtrlReg = e1000_read_phy_register (Adapter, PHY_1000T_CTRL_REG, Adapter->PhyAddress);

	MiiAutoNegAdvertiseReg &= ~REG4_SPEED_MASK;
	Mii1000TCtrlReg &= ~REG9_SPEED_MASK;

	DEBUGOUT1 ("AutoNegAdvertised %x\n", Adapter->AutoNegAdvertised);

	if (Adapter->AutoNegAdvertised & ADVERTISE_10_HALF) {
		DEBUGOUT ("Advertise 10mb Half duplex\n");
		MiiAutoNegAdvertiseReg |= NWAY_AR_10T_HD_CAPS;
	}

	if (Adapter->AutoNegAdvertised & ADVERTISE_10_FULL) {
		DEBUGOUT ("Advertise 10mb Full duplex\n");
		MiiAutoNegAdvertiseReg |= NWAY_AR_10T_FD_CAPS;
	}

	if (Adapter->AutoNegAdvertised & ADVERTISE_100_HALF) {
		DEBUGOUT ("Advertise 100mb Half duplex\n");
		MiiAutoNegAdvertiseReg |= NWAY_AR_100TX_HD_CAPS;
	}

	if (Adapter->AutoNegAdvertised & ADVERTISE_100_FULL) {
		DEBUGOUT ("Advertise 100mb Full duplex\n");
		MiiAutoNegAdvertiseReg |= NWAY_AR_100TX_FD_CAPS;
	}

	if (Adapter->AutoNegAdvertised & ADVERTISE_1000_HALF) {
		DEBUGOUT ("Advertise 1000mb Half duplex requested, request denied!\n");
	}

	if (Adapter->AutoNegAdvertised & ADVERTISE_1000_FULL) {
		DEBUGOUT ("Advertise 1000mb Full duplex\n");
		Mii1000TCtrlReg |= CR_1000T_FD_CAPS;
	}

	switch (Adapter->FlowControl) {
	case FLOW_CONTROL_NONE:

		MiiAutoNegAdvertiseReg &= ~(NWAY_AR_ASM_DIR | NWAY_AR_PAUSE);

		break;

	case FLOW_CONTROL_RECEIVE_PAUSE:

		MiiAutoNegAdvertiseReg |= (NWAY_AR_ASM_DIR | NWAY_AR_PAUSE);

		break;

	case FLOW_CONTROL_TRANSMIT_PAUSE:

		MiiAutoNegAdvertiseReg |= NWAY_AR_ASM_DIR;
		MiiAutoNegAdvertiseReg &= ~NWAY_AR_PAUSE;

		break;

	case FLOW_CONTROL_FULL:

		MiiAutoNegAdvertiseReg |= (NWAY_AR_ASM_DIR | NWAY_AR_PAUSE);

		break;

	default:

		DEBUGOUT ("Flow control param set incorrectly\n");
		ASSERT (0);
		break;
	}

	e1000_write_phy_register (Adapter, PHY_AUTONEG_ADVERTISEMENT, Adapter->PhyAddress, MiiAutoNegAdvertiseReg);

	DEBUGOUT1 ("Auto-Neg Advertising %x\n", MiiAutoNegAdvertiseReg);

	e1000_write_phy_register (Adapter, PHY_1000T_CTRL_REG, Adapter->PhyAddress, Mii1000TCtrlReg);
	return (TRUE);
}

static void
PhyForceSpeedAndDuplex (IN struct adapter * Adapter)
{
	u16 MiiCtrlReg;
	u16 MiiStatusReg;
	u16 PhyData;
	u16 i;
	u32 TctlReg;
	u32 DeviceCtrlReg;
	u32 Shift32;

	DEBUGFUNC ("PhyForceSpeedAndDuplex")

	    Adapter->FlowControl = FLOW_CONTROL_NONE;

	DEBUGOUT1 ("Adapter->FlowControl = %d\n", Adapter->FlowControl);

	DeviceCtrlReg = E1000_READ_REG (Ctrl);

	DeviceCtrlReg |= (E1000_CTRL_FRCSPD | E1000_CTRL_FRCDPX);
	DeviceCtrlReg &= ~(DEVICE_SPEED_MASK);

	DeviceCtrlReg &= ~E1000_CTRL_ASDE;

	MiiCtrlReg = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress);

	MiiCtrlReg &= ~MII_CR_AUTO_NEG_EN;

	if (Adapter->ForcedSpeedDuplex == FULL_100 || Adapter->ForcedSpeedDuplex == FULL_10) {

		DeviceCtrlReg |= E1000_CTRL_FD;
		MiiCtrlReg |= MII_CR_FULL_DUPLEX;

		DEBUGOUT ("Full Duplex\n");
	} else {

		DeviceCtrlReg &= ~E1000_CTRL_FD;
		MiiCtrlReg &= ~MII_CR_FULL_DUPLEX;

		DEBUGOUT ("Half Duplex\n");
	}

	if (Adapter->ForcedSpeedDuplex == FULL_100 || Adapter->ForcedSpeedDuplex == HALF_100) {

		DeviceCtrlReg |= E1000_CTRL_SPD_100;
		MiiCtrlReg |= MII_CR_SPEED_100;
		MiiCtrlReg &= ~(MII_CR_SPEED_1000 | MII_CR_SPEED_10);

		DEBUGOUT ("Forcing 100mb ");
	} else {

		DeviceCtrlReg &= ~(E1000_CTRL_SPD_1000 | E1000_CTRL_SPD_100);
		MiiCtrlReg |= MII_CR_SPEED_10;
		MiiCtrlReg &= ~(MII_CR_SPEED_1000 | MII_CR_SPEED_100);

		DEBUGOUT ("Forcing 10mb ");
	}

	TctlReg = E1000_READ_REG (Tctl);
	DEBUGOUT1 ("TctlReg = %x\n", TctlReg);

	if (!(MiiCtrlReg & MII_CR_FULL_DUPLEX)) {

		TctlReg &= ~E1000_TCTL_COLD;
		Shift32 = E1000_HDX_COLLISION_DISTANCE;
		Shift32 <<= E1000_COLD_SHIFT;
		TctlReg |= Shift32;
	} else {

		TctlReg &= ~E1000_TCTL_COLD;
		Shift32 = E1000_FDX_COLLISION_DISTANCE;
		Shift32 <<= E1000_COLD_SHIFT;
		TctlReg |= Shift32;
	}

	E1000_WRITE_REG (Tctl, TctlReg);

	E1000_WRITE_REG (Ctrl, DeviceCtrlReg);

	PhyData = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress);

	PhyData &= ~PXN_PSCR_AUTO_X_MODE;

	e1000_write_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress, PhyData);

	DEBUGOUT1 ("Paxson PSCR: %x \n", PhyData);

	MiiCtrlReg |= MII_CR_RESET;

	e1000_write_phy_register (Adapter, PHY_MII_CTRL_REG, Adapter->PhyAddress, MiiCtrlReg);

	if (Adapter->WaitAutoNegComplete) {

		DEBUGOUT ("Waiting for forced speed/duplex link.\n");
		MiiStatusReg = 0;

#define PHY_WAIT_FOR_FORCED_TIME    20

		for (i = 20; i > 0; i--) {

			MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

			MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

			if (MiiStatusReg & MII_SR_LINK_STATUS) {
				break;
			}
			mdelay (100);
		}

		if (i == 0) {

			Pxne1000_phy_resetDsp (Adapter);
		}

		for (i = 20; i > 0; i--) {
			if (MiiStatusReg & MII_SR_LINK_STATUS) {
				break;
			}

			mdelay (100);

			MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

			MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		}
	}

	PhyData = e1000_read_phy_register (Adapter, PXN_EXT_PHY_SPEC_CTRL_REG, Adapter->PhyAddress);

	PhyData |= PXN_EPSCR_TX_CLK_25;

	e1000_write_phy_register (Adapter, PXN_EXT_PHY_SPEC_CTRL_REG, Adapter->PhyAddress, PhyData);

	PhyData = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress);

	PhyData |= PXN_PSCR_ASSERT_CRS_ON_TX;

	e1000_write_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, Adapter->PhyAddress, PhyData);
	DEBUGOUT1 ("After force, Paxson Phy Specific Ctrl Reg = %4x\r\n", PhyData);

	return;
}

void
e1000_configure_mac_to_phySettings (IN struct adapter * Adapter, IN u16 MiiRegisterData)
{
	u32 DeviceCtrlReg, TctlReg;
	u32 Shift32;

	DEBUGFUNC ("e1000_configure_mac_to_phySettings")

	    TctlReg = E1000_READ_REG (Tctl);
	DEBUGOUT1 ("TctlReg = %x\n", TctlReg);

	DeviceCtrlReg = E1000_READ_REG (Ctrl);

	DeviceCtrlReg |= (E1000_CTRL_FRCSPD | E1000_CTRL_FRCDPX);
	DeviceCtrlReg &= ~(DEVICE_SPEED_MASK);

	DEBUGOUT1 ("MII Register Data = %x\r\n", MiiRegisterData);

	DeviceCtrlReg &= ~E1000_CTRL_ILOS;

	if (MiiRegisterData & PXN_PSSR_DPLX) {
		DeviceCtrlReg |= E1000_CTRL_FD;

		TctlReg &= ~E1000_TCTL_COLD;
		Shift32 = E1000_FDX_COLLISION_DISTANCE;
		Shift32 <<= E1000_COLD_SHIFT;
		TctlReg |= Shift32;
	} else {
		DeviceCtrlReg &= ~E1000_CTRL_FD;

		if ((MiiRegisterData & PXN_PSSR_SPEED) == PXN_PSSR_1000MBS) {
			TctlReg &= ~E1000_TCTL_COLD;
			Shift32 = E1000_GB_HDX_COLLISION_DISTANCE;
			Shift32 <<= E1000_COLD_SHIFT;
			TctlReg |= Shift32;

			TctlReg |= E1000_TCTL_PBE;

		} else {
			TctlReg &= ~E1000_TCTL_COLD;
			Shift32 = E1000_HDX_COLLISION_DISTANCE;
			Shift32 <<= E1000_COLD_SHIFT;
			TctlReg |= Shift32;
		}
	}

	if ((MiiRegisterData & PXN_PSSR_SPEED) == PXN_PSSR_1000MBS)
		DeviceCtrlReg |= E1000_CTRL_SPD_1000;
	else if ((MiiRegisterData & PXN_PSSR_SPEED) == PXN_PSSR_100MBS)
		DeviceCtrlReg |= E1000_CTRL_SPD_100;
	else
		DeviceCtrlReg &= ~(E1000_CTRL_SPD_1000 | E1000_CTRL_SPD_100);

	E1000_WRITE_REG (Tctl, TctlReg);

	E1000_WRITE_REG (Ctrl, DeviceCtrlReg);

	return;
}

void
e1000_configure_collision_distance (IN struct adapter * Adapter)
{
	u32 TctlReg;
	u16 Speed;
	u16 Duplex;
	u32 Shift32;

	DEBUGFUNC ("e1000_configure_collision_distance")

	    e1000_get_speed_and_duplex (Adapter, &Speed, &Duplex);

	TctlReg = E1000_READ_REG (Tctl);
	DEBUGOUT1 ("TctlReg = %x\n", TctlReg);

	TctlReg &= ~E1000_TCTL_COLD;

	if (Duplex == FULL_DUPLEX) {

		Shift32 = E1000_FDX_COLLISION_DISTANCE;
		Shift32 <<= E1000_COLD_SHIFT;
		TctlReg |= Shift32;
	} else {

		if (Speed == SPEED_1000) {
			Shift32 = E1000_GB_HDX_COLLISION_DISTANCE;
			Shift32 <<= E1000_COLD_SHIFT;
			TctlReg |= Shift32;

			TctlReg |= E1000_TCTL_PBE;

		} else {
			Shift32 = E1000_HDX_COLLISION_DISTANCE;
			Shift32 <<= E1000_COLD_SHIFT;
			TctlReg |= Shift32;
		}
	}

	E1000_WRITE_REG (Tctl, TctlReg);

	return;
}

void
e1000_display_mii_contents (IN struct adapter * Adapter, IN u8 PhyAddress)
{
	u16 Data, PhyIDHi, PhyIDLo;
	u32 PhyID;

	DEBUGFUNC ("e1000_display_mii_contents")

	    DEBUGOUT1 ("Adapter Base Address = %x\n", Adapter->HardwareVirtualAddress);

	Data = e1000_read_phy_register (Adapter, PHY_MII_CTRL_REG, PhyAddress);

	DEBUGOUT1 ("MII Ctrl Reg contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, PhyAddress);

	Data = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, PhyAddress);

	DEBUGOUT1 ("MII Status Reg contents = %x\n", Data);

	PhyIDHi = e1000_read_phy_register (Adapter, PHY_PHY_ID_REG1, PhyAddress);

	udelay (2);

	PhyIDLo = e1000_read_phy_register (Adapter, PHY_PHY_ID_REG2, PhyAddress);

	PhyID = (PhyIDLo | (PhyIDHi << 16)) & PHY_REVISION_MASK;

	DEBUGOUT1 ("Phy ID = %x \n", PhyID);

	Data = e1000_read_phy_register (Adapter, PHY_AUTONEG_ADVERTISEMENT, PhyAddress);

	DEBUGOUT1 ("Reg 4 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_AUTONEG_LP_BPA, PhyAddress);

	DEBUGOUT1 ("Reg 5 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_AUTONEG_EXPANSION_REG, PhyAddress);

	DEBUGOUT1 ("Reg 6 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_AUTONEG_NEXT_PAGE_TX, PhyAddress);

	DEBUGOUT1 ("Reg 7 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_AUTONEG_LP_RX_NEXT_PAGE, PhyAddress);

	DEBUGOUT1 ("Reg 8 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_1000T_CTRL_REG, PhyAddress);

	DEBUGOUT1 ("Reg 9 contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_1000T_STATUS_REG, PhyAddress);

	DEBUGOUT1 ("Reg A contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PHY_IEEE_EXT_STATUS_REG, PhyAddress);

	DEBUGOUT1 ("Reg F contents = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_CTRL_REG, PhyAddress);

	DEBUGOUT1 ("Paxson Specific Control Reg (0x10) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_STAT_REG, PhyAddress);

	DEBUGOUT1 ("Paxson Specific Status Reg (0x11) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_INT_ENABLE_REG, PhyAddress);

	DEBUGOUT1 ("Paxson Interrupt Enable Reg (0x12) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_INT_STATUS_REG, PhyAddress);

	DEBUGOUT1 ("Paxson Interrupt Status Reg (0x13) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_EXT_PHY_SPEC_CTRL_REG, PhyAddress);

	DEBUGOUT1 ("Paxson Ext. Phy Specific Control (0x14) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_RX_ERROR_COUNTER, PhyAddress);

	DEBUGOUT1 ("Paxson Receive Error Counter (0x15) = %x\n", Data);

	Data = e1000_read_phy_register (Adapter, PXN_LED_CTRL_REG, PhyAddress);

	DEBUGOUT1 ("Paxson LED control reg (0x18) = %x\n", Data);
}

u32
e1000_auto_detect_gigabit_phy (IN struct adapter * Adapter)
{
	u32 PhyAddress = 1;
	u32 PhyIDHi;
	u16 PhyIDLo;
	unsigned char GotOne = FALSE;

	DEBUGFUNC ("e1000_auto_detect_gigabit_phy")

	    while ((!GotOne) && (PhyAddress <= MAX_PHY_REG_ADDRESS)) {

		PhyIDHi = e1000_read_phy_register (Adapter, PHY_PHY_ID_REG1, PhyAddress);

		udelay (2);

		PhyIDLo = e1000_read_phy_register (Adapter, PHY_PHY_ID_REG2, PhyAddress);

		Adapter->PhyId = (PhyIDLo | (PhyIDHi << 16)) & PHY_REVISION_MASK;

		if (Adapter->PhyId == PAXSON_PHY_88E1000 || Adapter->PhyId == PAXSON_PHY_88E1000S || Adapter->PhyId == PAXSON_PHY_INTEGRATED) {
			DEBUGOUT2 ("PhyId 0x%x detected at address 0x%x\n", Adapter->PhyId, PhyAddress);

			GotOne = TRUE;
		} else {
			PhyAddress++;
		}

	}

	if (PhyAddress > MAX_PHY_REG_ADDRESS) {
		DEBUGOUT ("Could not auto-detect Phy!\n");
	}

	return (PhyAddress);
}

void
Pxne1000_phy_resetDsp (IN struct adapter * Adapter)
{
	e1000_write_phy_register (Adapter, 29, Adapter->PhyAddress, 0x1d);
	e1000_write_phy_register (Adapter, 30, Adapter->PhyAddress, 0xc1);
	e1000_write_phy_register (Adapter, 30, Adapter->PhyAddress, 0x00);
}

unsigned char
e1000_wait_for_auto_neg (IN struct adapter * Adapter)
{
	unsigned char AutoNegComplete = FALSE;
	u16 i;
	u16 MiiStatusReg;

	DEBUGFUNC ("e1000_wait_for_auto_neg");

	DEBUGOUT ("Waiting for Auto-Neg to complete.\n");
	MiiStatusReg = 0;

	for (i = PHY_AUTO_NEG_TIME; i > 0; i--) {

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		if (MiiStatusReg & MII_SR_AUTONEG_COMPLETE) {
			AutoNegComplete = TRUE;
			break;
		}

		mdelay (100);
	}

	return (AutoNegComplete);
}
