/*****************************************************************************
 *****************************************************************************

 Copyright (c) 1999-2001 Intel Corporation.

 All rights reserved.

 Redistribution and use in source and binary forms, with or without
 modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice,
     this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation
     and/or other materials provided with the distribution.

  3. Neither the name of Intel Corporation nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 *****************************************************************************
 *****************************************************************************/

/*********************************************************************/
/*! @file   e1000_main.c
 *  @author Intel Corporation
 *  @date   1999-2000
 *  @brief  Linux PRO/1000 Ethernet Driver main source file
 *
 *  The Linux specific code for e1000,
 *  a driver for the PRO/1000 line of gigabit ethernet adapters
 *********************************************************************/

#define __E1000_MAIN__
#define _IANS_MAIN_MODULE_C_
#include "e1000.h"

/* Zero Copy Tx and Highmem do not mesh well */
/* only allow it for platforms where we can do without pci_map/pci_unmap */
#if defined(MAX_SKB_FRAGS) && (!defined(CONFIG_HIGHMEM) || defined(CONFIG_X86))
#define ZEROCOPY 1
#else
#define ZEROCOPY 0
#endif

#if ZEROCOPY && defined(CONFIG_HIGHMEM)
static inline uint64_t pci_map_single_high (struct pci_dev *hwdev, struct page *page, int offset, size_t size, int dir) {
	uint64_t phys;
	phys = ((page - mem_map) * (uint64_t) PAGE_SIZE) + offset;
	return phys;
}
#else
static inline uint64_t pci_map_single_high (struct pci_dev *hwdev, struct page *page, int offset, size_t size, int dir) {
	dma_addr_t phys;
	phys = pci_map_single (hwdev, (void *) page_address (page) + offset, size, dir);
	return (uint64_t) phys;
}
#endif

/*********************************************************************/
/*! @defgroup globals e1000 Global Variables
 *********************************************************************/

/*********************************************************************/
/*! @ingroup globals
 *  @brief Driver name string
 *********************************************************************/

char e1000_driver_name[] = "e1000";

/*********************************************************************/
/*! @ingroup globals
 *  @brief Driver ID string, displayed when loading
 *********************************************************************/

char e1000_driver_string[] = "Intel(R) PRO/1000 Network Driver";

/*********************************************************************/
/*! @ingroup globals
 *  @brief Driver version
 *********************************************************************/

char e1000_driver_version[] = "3.1.22";

/*********************************************************************/
/*! @ingroup globals
 *  @brief Copyright string, displayed when loading
 *********************************************************************/

char e1000_copyright[] = "Copyright (c) 1999-2001 Intel Corporation.";

/*********************************************************************/
/*! @ingroup globals
 *  @brief Linked list of board private structures for all NICs found
 *********************************************************************/

struct adapter * e1000_adapter_list = NULL;

/*********************************************************************/
/*! @ingroup globals
 *  @sa e1000_strings
 *  @brief PCI Device ID Table
 *
 *  for selecting devices to load on
 *  private driver_data field (last one) stores an index
 *  into e1000_strings
 *  Wildcard entries (PCI_ANY_ID) should come last
 *  Last entry must be all 0s
 *
 *  { Vendor ID, Device ID, SubVendor ID, SubDevice ID,
 *    Class, Class Mask, String Index }
 *********************************************************************/

static struct pci_device_id e1000_pci_table[] = {
	/* Intel(R) PRO/1000 Network Connection */
	{0x8086, 0x1000, 0x8086, 0x1000, 0, 0, 0},
	{0x8086, 0x1001, 0x8086, 0x1003, 0, 0, 0},
	{0x8086, 0x1004, 0x8086, 0x1004, 0, 0, 0},
	/* Compaq Gigabit Ethernet Server Adapter */
	{0x8086, 0x1000, 0x0E11, PCI_ANY_ID, 0, 0, 1},
	{0x8086, 0x1001, 0x0E11, PCI_ANY_ID, 0, 0, 1},
	{0x8086, 0x1004, 0x0E11, PCI_ANY_ID, 0, 0, 1},
	/* IBM Mobile, Desktop & Server Adapters */
	{0x8086, 0x1000, 0x1014, PCI_ANY_ID, 0, 0, 2},
	{0x8086, 0x1001, 0x1014, PCI_ANY_ID, 0, 0, 2},
	{0x8086, 0x1004, 0x1014, PCI_ANY_ID, 0, 0, 2},
	/* Generic */
	{0x8086, 0x1000, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0x8086, 0x1001, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0x8086, 0x1004, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0x8086, 0x1008, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0x8086, 0x100D, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	/* required last entry */
	{0,}
};

MODULE_DEVICE_TABLE (pci, e1000_pci_table);

/*********************************************************************/
/*! @ingroup globals
 *  @sa e1000_pci_table
 *  @brief Table of branding strings for all supported NICs.
 *********************************************************************/

static char *e1000_strings[] = {
	"Intel(R) PRO/1000 Network Connection",
	"Compaq Gigabit Ethernet Server Adapter",
	"IBM Mobile, Desktop & Server Adapters"
};

/*********************************************************************/
/*! @ingroup globals
 *  @brief PCI driver information (Linux 2.4 driver API)
 *********************************************************************/

static struct pci_driver e1000_driver = {
	name:e1000_driver_name,
	id_table:e1000_pci_table,
	probe:e1000_probe,
	remove:e1000_remove,
	/* Power Managment Hooks */
	suspend:NULL,
	resume:NULL
};

/*********************************************************************/
/*! @defgroup MODULE_PARM e1000 Module Parameters
 *  @bug HotPlug will mess things up
 *
 *  Module Parameters are always initialized to -1, so that the driver
 *  can tell the difference between no user specified value or the
 *  user asking for the default value.
 *  The true default values are loaded in when CheckOptions is called.
 *********************************************************************/

/*********************************************************************/
/*! This is the only thing that needs to be changed to adjust the
 *  maximum number of ports that the driver can manage.
 *********************************************************************/

#define E1000_MAX_NIC 8

/*********************************************************************/
/*! This is a GCC extension to ANSI C.
 *  See the item "Labeled Elements in Initializers" in the section
 *  "Extensions to the C Language Family" of the GCC documentation.
 *********************************************************************/

#define E1000_OPTION_INIT { [0 ... E1000_MAX_NIC] = OPTION_UNSET }

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Transmit Descriptor Count
 *
 *  Valid Range: 80-256
 *
 *  Default Value: 256
 *********************************************************************/

static int TxDescriptors[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Receive Descriptor Count
 *
 *  Valid Range: 80-256
 *
 *  Default Value: 256
 *********************************************************************/

static int RxDescriptors[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief User Specified Speed Override
 *
 *  Valid Range: 0, 10, 100, 1000
 *  - 0    - auto-negotiate at all supported speeds
 *  - 10   - only link at 10 Mbps
 *  - 100  - only link at 100 Mbps
 *  - 1000 - only link at 1000 Mbps
 *
 *  Default Value: 0
 *********************************************************************/

static int Speed[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief User Specified Duplex Override
 *
 *  Valid Range: 0-2
 *  - 0 - auto-negotiate for duplex
 *  - 1 - only link at half duplex
 *  - 2 - only link at full duplex
 *
 *  Default Value: 0
 *********************************************************************/

static int Duplex[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Auto-negotiation Advertisement Override
 *  @warning this is not easy to use correctly
 *
 *  Valid Range: 0x00-0x0F, 0x20-0x2F
 *
 *  The AutoNeg value is a bit mask describing which speed and duplex
 *  combinations should be advertised during auto-negotiation.
 *  The supported speed and duplex modes are listed below
 *
 * @code
 *
 * Bit           7     6     5      4      3     2     1      0
 * Speed (Mbps)  N/A   N/A   1000   N/A    100   100   10     10
 * Duplex                    Full          Full  Half  Full   Half
 *
 * @endcode
 *
 *  Default Value: 0x2F
 *********************************************************************/

static int AutoNeg[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief User Specified Flow Control Override
 *
 *  Valid Range: 0-3
 *  - 0 - No Flow Control
 *  - 1 - Rx only, respond to PAUSE frames but do not generate them
 *  - 2 - Tx only, generate PAUSE frames but ignore them on receive
 *  - 3 - Full Flow Control Support
 *
 *  Default Value: 3
 *********************************************************************/

static int FlowControl[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief WaitForLink
 *
 *  Valid Range: 0, 1
 *
 *  Default Value: 1
 *********************************************************************/

static int WaitForLink[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Jumbo Frame Support Enable/Disable
 *
 *  Valid Range: 0, 1
 *  - 0 - disables all jumbo frame support
 *  - 1 - enables jumbo frames up to 16000 bytes on 82543 based NICs
 *
 *  Default Value: 1
 *********************************************************************/

static int Jumbo[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Receive Checksum Offload Enable/Disable
 *
 *  Valid Range: 0, 1
 *  - 0 - disables all checksum offload
 *  - 1 - enables receive IP/TCP/UDP checksum offload
 *        on 82543 based NICs
 *
 *  Default Value: 1
 *********************************************************************/

static int XsumRX[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Transmit Interrupt Delay in units of 1.024 microseconds
 *
 *  Valid Range: 0-65535
 *
 *  Default Value: 64
 *********************************************************************/

static int TxIntDelay[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

/*********************************************************************/
/*! @ingroup MODULE_PARM
 *  @brief Receive Interrupt Delay in units of 1.024 microseconds
 *
 *  Valid Range: 0-65535
 *
 *  Default Value: 64
 *********************************************************************/

static int RxIntDelay[E1000_MAX_NIC + 1] = E1000_OPTION_INIT;

#ifdef MODULE

MODULE_AUTHOR ("Intel Corporation, <linux.nics@intel.com>");
MODULE_DESCRIPTION ("Intel(R) PRO/1000 Network Driver");

MODULE_PARM (TxDescriptors, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (RxDescriptors, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (Speed, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (Duplex, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (AutoNeg, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (WaitForLink, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (Jumbo, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (XsumRX, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (FlowControl, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (TxIntDelay, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");
MODULE_PARM (RxIntDelay, "1-" __MODULE_STRING (E1000_MAX_NIC) "i");

EXPORT_SYMBOL (e1000_init_module);
EXPORT_SYMBOL (e1000_exit_module);
EXPORT_SYMBOL (e1000_probe);
EXPORT_SYMBOL (e1000_remove);
EXPORT_SYMBOL (e1000_open);
EXPORT_SYMBOL (e1000_close);
EXPORT_SYMBOL (e1000_xmit_frame);
EXPORT_SYMBOL (e1000_intr);
EXPORT_SYMBOL (e1000_set_multi);
EXPORT_SYMBOL (e1000_change_mtu);
EXPORT_SYMBOL (e1000_set_mac);
EXPORT_SYMBOL (e1000_get_stats);
EXPORT_SYMBOL (e1000_watchdog);
EXPORT_SYMBOL (e1000_ioctl);
EXPORT_SYMBOL (e1000_adapter_list);

#endif

/*********************************************************************
 * Local Function Prototypes
 *********************************************************************/

static void CheckOptions (struct adapter * Adapter);
static void CheckLinkOptionsFiber (struct adapter * Adapter);
static void CheckLinkOptionsCopper (struct adapter * Adapter);
static int SoftwareInit (struct adapter * Adapter);
static int HardwareInit (struct adapter * Adapter);
static void ReadNodeAddress (struct adapter * Adapter, u8 * NodeAddress);
static int SetupTransmitStructures (struct adapter * Adapter);
static int SetupReceiveStructures (struct adapter * Adapter);
static void SetupRctl (struct adapter * Adapter);
static void ResetRU (struct adapter * Adapter);
static void ResetTU (struct adapter * Adapter);
static void FreeTransmitStructures (struct adapter * Adapter);
static void FreeReceiveStructures (struct adapter * Adapter);
static void UpdateStatsCounters (struct adapter * Adapter);
static inline void DisableInterrupts (struct adapter * Adapter);
static inline void EnableInterrupts (struct adapter * Adapter);
static void ProcessTransmitInterrupts (struct adapter * Adapter);
static void ProcessReceiveInterrupts (struct adapter * Adapter);
static inline void RxChecksum (struct adapter * Adapter, PE1000_RECEIVE_DESCRIPTOR RxDescriptor, struct sk_buff *skb);
static void ReceiveBufferFill (uintptr_t data);
static void CleanTxRing (struct adapter * Adapter);
static void CleanRxRing (struct adapter * Adapter);

void e1000_hibernate_adapter (struct net_device *netdev);
void e1000_wakeup_adapter (struct net_device *netdev);

/*********************************************************************/
/*! @defgroup entry_point Linux network driver entry points
 *********************************************************************/

/*********************************************************************/
/*! @brief  Driver Registration Routine
 *  @ingroup entry_point
 *  @return 0 on success, negative on failure
 *
 *  e1000_init_module is the first routine called when the driver is
 *  loaded.
 *********************************************************************/

int
e1000_init_module (void)
{
	E1000_DBG ("e1000_init_module\n");

	/* Print the driver ID string and copyright notice */

	printk ("%s - version %s\n%s\n", e1000_driver_string, e1000_driver_version, e1000_copyright);

	/* register the driver with the PCI subsystem */

	return pci_module_init (&e1000_driver);
}

/*********************************************************************
 * this macro creates a special symbol in the object file that
 * identifies the driver initialization routine
 *********************************************************************/

module_init (e1000_init_module);

/*********************************************************************/
/*! @brief  Driver Exit Cleanup Routine
 *  @ingroup entry_point
 *
 *  e1000_exit_module is called just before the driver is removed
 *  from memory.
 *********************************************************************/

void
e1000_exit_module (void)
{
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *de;
#endif

	E1000_DBG ("e1000_exit_module\n");

	pci_unregister_driver (&e1000_driver);

#ifdef CONFIG_PROC_FS
	/* if there is no e1000_proc_dir (proc creation failure on load)
	 * then we're done */

	if (e1000_proc_dir == NULL)
		return;

	/* If ADAPTERS_PROC_DIR (/proc/net/PRO_LAN_Adapters) is empty
	 * it can be removed now (might still be in use by e100) */

	for (de = e1000_proc_dir->subdir; de; de = de->next) {

		/* ignore . and .. */

		if (*(de->name) == '.')
			continue;
		break;
	}
	if (de)
		return;
	remove_proc_entry (ADAPTERS_PROC_DIR, proc_net);
#endif

	return;
}

/*********************************************************************
 * this macro creates a special symbol in the object file that
 * identifies the driver cleanup routine
 *********************************************************************/

module_exit (e1000_exit_module);

/*********************************************************************/
/*! @brief  Device Initialization Routine
 *  @ingroup entry_point
 *  @param  pdev PCI device information struct
 *  @param  ent entry in e1000_pci_table
 *  @return 0 on success, negative on failure
 *
 *  e1000_probe initializes an adapter identified by a pci_dev
 *  structure.  The OS initialization is handled here, and
 *  SoftwareInit and HardwareInit are called to handle the driver
 *  specific software structures and hardware initialization
 *  respectively.
 *********************************************************************/

int
e1000_probe (struct pci_dev *pdev, const struct pci_device_id *ent)
{
	struct net_device *netdev = NULL;
	struct adapter * Adapter;
	static int cards_found = 0;

#ifdef CONFIG_PROC_FS
	int len;
#endif

	E1000_DBG ("e1000_probe\n");

	/* Make sure the PCI device has the proper resources available */

	if (pci_enable_device (pdev) != 0) {
		E1000_ERR ("pci_enable_device failed\n");
		return -ENODEV;
	}

	/* Make sure we are enabled as a bus mastering device */

	pci_set_master (pdev);

	/* Check to see if our PCI addressing needs are supported */
	if (pci_set_dma_mask (pdev, E1000_DMA_MASK) < 0) {
		E1000_ERR ("PCI DMA not supported by the system\n");
		return -ENODEV;
	}

	/* Register a new network interface + allocate
	 * private data structure (struct adapter) */

	netdev = init_etherdev (netdev, sizeof (struct adapter));
	if (netdev == NULL) {
		E1000_ERR ("Unable to allocate net_device struct\n");
		return -ENOMEM;
	}

	/* Calling init_etherdev with sizeof(struct adapter) allocates
	 * a single buffer of size net_device + struct adapter +
	 * alignment. If this is not done then the struct adapter needs
	 * to be allocated and freed separately. */

	Adapter = (struct adapter *) netdev->priv;
	memset (Adapter, 0, sizeof (struct adapter));
	Adapter->netdev = netdev;
	Adapter->pdev = pdev;

	/* link the struct adapter into the list */

	if (e1000_adapter_list != NULL)
		e1000_adapter_list->prev = Adapter;
	Adapter->next = e1000_adapter_list;
	e1000_adapter_list = Adapter;

	/* reserve the MMIO region as ours */

	if (!request_mem_region (pci_resource_start (pdev, BAR_0), pci_resource_len (pdev, BAR_0), e1000_driver_name)) {
		E1000_ERR ("request_mem_region failed\n");
		e1000_remove (pdev);
		return -ENODEV;
	}

	/* map the MMIO region into the kernel virtual address space */

	Adapter->HardwareVirtualAddress = ioremap (pci_resource_start (pdev, BAR_0), pci_resource_len (pdev, BAR_0));

	if (Adapter->HardwareVirtualAddress == NULL) {
		E1000_ERR ("ioremap failed\n");
		release_mem_region (pci_resource_start (pdev, BAR_0), pci_resource_len (pdev, BAR_0));
		e1000_remove (pdev);
		return -ENOMEM;
	}

	/* don't actually register the interrupt handler until e1000_open */

	netdev->irq = pdev->irq;

	/* Set the MMIO base address for the NIC */

#ifdef IANS
	netdev->base_addr = pci_resource_start (pdev, BAR_0);
#endif
	netdev->mem_start = pci_resource_start (pdev, BAR_0);
	netdev->mem_end = netdev->mem_start + pci_resource_len (pdev, BAR_0);

	/* set up function pointers to driver entry points */

	netdev->open = &e1000_open;
	netdev->stop = &e1000_close;
	netdev->hard_start_xmit = &e1000_xmit_frame;
	netdev->get_stats = &e1000_get_stats;
	netdev->set_multicast_list = &e1000_set_multi;
	netdev->set_mac_address = &e1000_set_mac;
	netdev->change_mtu = &e1000_change_mtu;
	netdev->do_ioctl = &e1000_ioctl;

	/* set up the struct adapter */

	Adapter->bd_number = cards_found;
	Adapter->id_string = e1000_strings[ent->driver_data];
	printk ("\n%s\n", Adapter->id_string);

	/* Order is important here.  SoftwareInit also identifies the
	 * hardware, so that CheckOptions can treat command line parameters
	 * differently depending on the hardware. */

	SoftwareInit (Adapter);
	CheckOptions (Adapter);

#if ZEROCOPY
	if (Adapter->MacType >= MAC_LIVENGOOD) {
		netdev->features = NETIF_F_SG | NETIF_F_IP_CSUM | NETIF_F_HIGHDMA;
	} else {
		netdev->features = NETIF_F_SG | NETIF_F_HIGHDMA;
	}
#endif

#ifdef CONFIG_PROC_FS
	/* set up the proc fs entry */

	len = strlen (ADAPTERS_PROC_DIR);

	for (e1000_proc_dir = proc_net->subdir; e1000_proc_dir; e1000_proc_dir = e1000_proc_dir->next) {
		if ((e1000_proc_dir->namelen == len) && (memcmp (e1000_proc_dir->name, ADAPTERS_PROC_DIR, len) == 0))
			break;
	}

	if (e1000_proc_dir == NULL)
		e1000_proc_dir = create_proc_entry (ADAPTERS_PROC_DIR, S_IFDIR, proc_net);

	if (e1000_proc_dir != NULL)
		if (e1000_create_proc_dev (Adapter) < 0) {
			e1000_remove_proc_dev (Adapter->netdev);
		}
#endif

#ifdef IANS
	Adapter->iANSdata = kmalloc (sizeof (iANSsupport_t), GFP_KERNEL);
	if (Adapter->iANSdata == NULL) {
		e1000_remove (pdev);
		return -ENOMEM;
	}
	memset (Adapter->iANSdata, 0, sizeof (iANSsupport_t));
	bd_ans_drv_InitANS (Adapter, Adapter->iANSdata);
#endif

	/* finally, we get around to setting up the hardware */

	if (HardwareInit (Adapter) < 0) {
		e1000_remove (pdev);
		return -ENODEV;
	}
	cards_found++;

	/* reset stats */

	e1000_clear_hw_stats_counters (Adapter);
	UpdateStatsCounters (Adapter);
	Adapter->GetLinkStatus = TRUE;
	e1000_check_for_link (Adapter);

	/* print the link status */

	if (Adapter->LinkIsActive == TRUE)
		printk ("%s:  Mem:0x%p  IRQ:%d  Speed:%d Mbps  Duplex:%s\n",
			netdev->name, (void *) netdev->mem_start, netdev->irq, Adapter->LineSpeed, Adapter->FullDuplex == FULL_DUPLEX ? "Full" : "Half");
	else
		printk ("%s:  Mem:0x%p  IRQ:%d  Speed:N/A  Duplex:N/A\n", netdev->name, (void *) netdev->mem_start, netdev->irq);

	return 0;
}

/*********************************************************************/
/*! @brief  Device Removal Routine
 *  @ingroup entry_point
 *  @param  pdev PCI device information struct
 *
 *  e1000_remove is called by the PCI subsystem to alert the driver
 *  that it should release a PCI device.  The could be caused by a
 *  Hot-Plug event, or because the driver is going to be removed from
 *  memory.
 *
 *  This routine is also called to clean up from a failure in
 *  e1000_probe.  The Adapter struct and netdev will always exist,
 *  all other pointers must be checked for NULL before freeing.
 *********************************************************************/

void
e1000_remove (struct pci_dev *pdev)
{
	struct net_device *netdev;
	struct adapter * Adapter;

	/* find the Adapter struct that matches this PCI device */

	for (Adapter = e1000_adapter_list; Adapter != NULL; Adapter = Adapter->next) {
		if (Adapter->pdev == pdev)
			break;
	}
	if (Adapter == NULL)
		return;

	netdev = Adapter->netdev;

	/* this must be called before freeing anything,
	 * otherwise there is a case where the open entry point can be
	 * running at the same time as remove. Calling unregister_netdev on an
	 * open interface results in a call to dev_close, which locks
	 * properly against the other netdev entry points, so this takes
	 * care of the hotplug issue of removing an active interface as well. */

	unregister_netdev (netdev);

	e1000_phy_hardware_reset (Adapter);

#ifdef CONFIG_PROC_FS
	/* remove the proc nodes */

	e1000_remove_proc_dev (Adapter->netdev);
#endif

	/* remove from the adapter list */

	if (e1000_adapter_list == Adapter)
		e1000_adapter_list = Adapter->next;
	if (Adapter->next != NULL)
		Adapter->next->prev = Adapter->prev;
	if (Adapter->prev != NULL)
		Adapter->prev->next = Adapter->next;

	/* free system resources */

#ifdef IANS
	if (Adapter->iANSdata != NULL)
		kfree (Adapter->iANSdata);
#endif

	if (Adapter->HardwareVirtualAddress != NULL) {
		iounmap (Adapter->HardwareVirtualAddress);
		release_mem_region (pci_resource_start (pdev, BAR_0), pci_resource_len (pdev, BAR_0));
	}

	/* free the net_device _and_ struct adapter memory */

	kfree (netdev);

	return;
}

/*********************************************************************/
/*! @brief  Range Checking for Command Line Parameters
 *  @param  Adapter board private structure
 *
 *  This routine checks all command line paramters for valid user
 *  input.  If an invalid value is given, or if no user specified
 *  value exists, a default value is used.  The final value is stored
 *  in a variable in the Adapter structure.
 *********************************************************************/

static void
CheckOptions (struct adapter * Adapter)
{
	int board = Adapter->bd_number;

	if (board >= E1000_MAX_NIC) {
		printk ("Warning: no configuration for board #%i\n", board);
		printk ("Using defaults for all values\n");
		board = E1000_MAX_NIC;
	}

	E1000_DBG ("CheckOptions\n");

	/* Transmit Descriptor Count */

	if (TxDescriptors[board] == OPTION_UNSET) {
		Adapter->NumTxDescriptors = DEFAULT_TXD;
	} else if ((TxDescriptors[board] > MAX_TXD) || (TxDescriptors[board] < MIN_TXD)) {
		printk ("Invalid TxDescriptor specified (%i), using default %i\n", TxDescriptors[board], DEFAULT_TXD);
		Adapter->NumTxDescriptors = DEFAULT_TXD;
	} else {
		printk ("Using specified value of %i TxDescriptors\n", TxDescriptors[board]);
		Adapter->NumTxDescriptors = TxDescriptors[board];
	}

	/* NumTxDescriptors must be a multiple of 8 */

	Adapter->NumTxDescriptors = E1000_ROUNDUP2 (Adapter->NumTxDescriptors, REQ_TX_DESCRIPTOR_MULTIPLE);

	/* Receive Descriptor Count */

	if (RxDescriptors[board] == OPTION_UNSET) {
		Adapter->NumRxDescriptors = DEFAULT_RXD;
	} else if ((RxDescriptors[board] > MAX_RXD) || (RxDescriptors[board] < MIN_RXD)) {
		printk ("Invalid RxDescriptor specified (%i), using default %i\n", RxDescriptors[board], DEFAULT_RXD);
		Adapter->NumRxDescriptors = DEFAULT_RXD;
	} else {
		printk ("Using specified value of %i RxDescriptors\n", RxDescriptors[board]);
		Adapter->NumRxDescriptors = RxDescriptors[board];
	}

	/* NumRxDescriptors must be a multiple of 8 */

	Adapter->NumRxDescriptors = E1000_ROUNDUP2 (Adapter->NumRxDescriptors, REQ_RX_DESCRIPTOR_MULTIPLE);
	/* Jumbo Frame Enable */

	if (Jumbo[board] == OPTION_UNSET) {
		Adapter->JumboFrames = JUMBO_DEFAULT;
	} else if ((Jumbo[board] != OPTION_ENABLED) && (Jumbo[board] != OPTION_DISABLED)) {
		printk ("Invalid Jumbo specified (%i), using default of %i\n", Jumbo[board], JUMBO_DEFAULT);
		Adapter->JumboFrames = JUMBO_DEFAULT;
	} else {
		printk ("Jumbo Frames %s\n", Jumbo[board] == OPTION_ENABLED ? "Enabled" : "Disabled");
		Adapter->JumboFrames = Jumbo[board];
	}

	/* Receive Checksum Offload Enable */

	if (XsumRX[board] == OPTION_UNSET) {
		Adapter->RxChecksum = XSUMRX_DEFAULT;
	} else if ((XsumRX[board] != OPTION_ENABLED) && (XsumRX[board] != OPTION_DISABLED)) {
		printk ("Invalid XsumRX specified (%i), using default of %i\n", XsumRX[board], XSUMRX_DEFAULT);
		Adapter->RxChecksum = XSUMRX_DEFAULT;
	} else {
		printk ("Receive Checksum Offload %s\n", XsumRX[board] == OPTION_ENABLED ? "Enabled" : "Disabled");
		Adapter->RxChecksum = XsumRX[board];
	}

	/* Wait for link at driver load */

	if (WaitForLink[board] == OPTION_UNSET) {
		Adapter->WaitAutoNegComplete = WAITFORLINK_DEFAULT;
	} else if ((WaitForLink[board] != OPTION_ENABLED) && (WaitForLink[board] != OPTION_DISABLED)) {
		printk ("Invalid WaitForLink specified (%i), using default of %i\n", WaitForLink[board], WAITFORLINK_DEFAULT);
		Adapter->WaitAutoNegComplete = WAITFORLINK_DEFAULT;
	} else {
		printk ("WaitForLink %s\n", WaitForLink[board] == OPTION_ENABLED ? "Enabled" : "Disabled");
		Adapter->WaitAutoNegComplete = WaitForLink[board];
	}

	/* Flow Control */

	if (FlowControl[board] == OPTION_UNSET) {
		Adapter->FlowControl = FLOW_CONTROL_DEFAULT;
	} else if ((FlowControl[board] > FLOW_CONTROL_FULL) || (FlowControl[board] < FLOW_CONTROL_NONE)) {
		printk ("Invalid FlowControl specified (%i), using default %i\n", FlowControl[board], FLOW_CONTROL_DEFAULT);
		Adapter->FlowControl = FLOW_CONTROL_DEFAULT;
	} else {
		Adapter->FlowControl = FlowControl[board];
		switch (Adapter->FlowControl) {
		case FLOW_CONTROL_NONE:
			printk ("Flow Control Disabled\n");
			break;
		case FLOW_CONTROL_RECEIVE_PAUSE:
			printk ("Flow Control Receive Only\n");
			break;
		case FLOW_CONTROL_TRANSMIT_PAUSE:
			printk ("Flow Control Transmit Only\n");
			break;
		case FLOW_CONTROL_FULL:
			printk ("Flow Control Enabled\n");
		}
	}

	/* Transmit Interrupt Delay */

	if (TxIntDelay[board] == OPTION_UNSET) {
		Adapter->TxIntDelay = DEFAULT_TIDV;
	} else if ((TxIntDelay[board] > MAX_TIDV) || (TxIntDelay[board] < MIN_TIDV)) {
		printk ("Invalid TxIntDelay specified (%i), using default %i\n", TxIntDelay[board], DEFAULT_TIDV);
		Adapter->TxIntDelay = DEFAULT_TIDV;
	} else {
		printk ("Using specified TxIntDelay of %i\n", TxIntDelay[board]);
		Adapter->TxIntDelay = TxIntDelay[board];
	}

	/* Receive Interrupt Delay */

	if (RxIntDelay[board] == OPTION_UNSET) {
		Adapter->RxIntDelay = DEFAULT_RIDV;
	} else if ((RxIntDelay[board] > MAX_RIDV) || (RxIntDelay[board] < MIN_RIDV)) {
		printk ("Invalid RxIntDelay specified (%i), using default %i\n", RxIntDelay[board], DEFAULT_RIDV);
		Adapter->RxIntDelay = DEFAULT_RIDV;
	} else {
		printk ("Using specified RxIntDelay of %i\n", RxIntDelay[board]);
		Adapter->RxIntDelay = RxIntDelay[board];
	}

	/* Speed, Duplex, and AutoNeg */

	switch (Adapter->MediaType) {

	case MEDIA_TYPE_FIBER:
		CheckLinkOptionsFiber (Adapter);
		break;

	case MEDIA_TYPE_COPPER:
		CheckLinkOptionsCopper (Adapter);
		break;
	}

	return;
}

/*********************************************************************/
/*! @brief  Range Checking for Command Link Options, Fiber Version
 *  @param  Adapter board private structure
 *
 *  Handles speed and duplex options on fiber based adapters
 *********************************************************************/

static void
CheckLinkOptionsFiber (struct adapter * Adapter)
{
	int board = Adapter->bd_number > E1000_MAX_NIC ? E1000_MAX_NIC : Adapter->bd_number;

	E1000_DBG ("CheckSpeedDuplexFiber\n");

	/* Speed, Duplex, and AutoNeg are not valid on fiber NICs */

	if ((Speed[board] != OPTION_UNSET)) {
		printk ("Warning: Speed not valid for fiber adapters\n");
		printk ("Speed Parameter Ignored\n");
	}
	if ((Duplex[board] != OPTION_UNSET)) {
		printk ("Warning: Duplex not valid for fiber adapters\n");
		printk ("Duplex Parameter Ignored\n");
	}
	if ((AutoNeg[board] != OPTION_UNSET)) {
		printk ("Warning: AutoNeg not valid for fiber adapters\n");
		printk ("AutoNeg Parameter Ignored\n");
	}

	return;
}

/*********************************************************************/
/*! @brief  Range Checking for Command Link Options, Copper Version
 *  @param  Adapter board private structure
 *
 *  Handles speed and duplex options on copper based adapters
 *********************************************************************/

static void
CheckLinkOptionsCopper (struct adapter * Adapter)
{
	int board = Adapter->bd_number > E1000_MAX_NIC ? E1000_MAX_NIC : Adapter->bd_number;
	int speed, duplex;
	unsigned char all_default = TRUE;

	E1000_DBG ("CheckSpeedDuplexCopper\n");

	/* User Specified Auto-negotiation Settings */

	if (AutoNeg[board] == OPTION_UNSET) {

		Adapter->AutoNegAdvertised = AUTONEG_ADV_DEFAULT;

	} else if ((Speed[board] != 0 && Speed[board] != OPTION_UNSET) || (Duplex[board] != 0 && Duplex[board] != OPTION_UNSET)) {

		printk ("Warning: AutoNeg specified along with Speed or Duplex\n");
		printk ("AutoNeg Parameter Ignored\n");

		Adapter->AutoNegAdvertised = AUTONEG_ADV_DEFAULT;

	} else {

		if (AutoNeg[board] & ~AUTONEG_ADV_MASK) {

			printk ("Invalid AutoNeg Specified (0x%X), Parameter Ignored\n", AutoNeg[board]);

			Adapter->AutoNegAdvertised = AUTONEG_ADV_DEFAULT;

		} else {

			Adapter->AutoNegAdvertised = AutoNeg[board];
		}

		printk ("AutoNeg Advertising ");
		if (Adapter->AutoNegAdvertised & ADVERTISE_1000_FULL) {
			printk ("1000/FD");
			if (Adapter->AutoNegAdvertised & (ADVERTISE_1000_FULL - 1))
				printk (", ");
		}
		if (Adapter->AutoNegAdvertised & ADVERTISE_1000_HALF) {
			printk ("1000/HD");
			if (Adapter->AutoNegAdvertised & (ADVERTISE_1000_HALF - 1))
				printk (", ");
		}
		if (Adapter->AutoNegAdvertised & ADVERTISE_100_FULL) {
			printk ("100/FD");
			if (Adapter->AutoNegAdvertised & (ADVERTISE_100_FULL - 1))
				printk (", ");
		}
		if (Adapter->AutoNegAdvertised & ADVERTISE_100_HALF) {
			printk ("100/HD");
			if (Adapter->AutoNegAdvertised & (ADVERTISE_100_HALF - 1))
				printk (", ");
		}
		if (Adapter->AutoNegAdvertised & ADVERTISE_10_FULL) {
			printk ("10/FD");
			if (Adapter->AutoNegAdvertised & (ADVERTISE_10_FULL - 1))
				printk (", ");
		}
		if (Adapter->AutoNegAdvertised & ADVERTISE_10_HALF)
			printk ("10/HD");
		printk ("\n");
	}

	/* Forced Speed and Duplex */

	switch (Speed[board]) {
	default:
		printk ("Invalid Speed Specified (%i), Parameter Ignored\n", Speed[board]);
		all_default = FALSE;
	case OPTION_UNSET:
		speed = 0;
		break;
	case 0:
	case 10:
	case 100:
	case 1000:
		speed = Speed[board];
		break;
	}

	switch (Duplex[board]) {
	default:
		printk ("Invalid Duplex Specified (%i), Parameter Ignored\n", Duplex[board]);
		all_default = FALSE;
	case OPTION_UNSET:
		duplex = 0;
		break;
	case 0:
	case 1:
	case 2:
		duplex = Duplex[board];
		break;
	}

	switch (speed + duplex) {
	case 0:
		if (all_default == FALSE)
			printk ("Speed and Duplex Auto-negotiation Enabled\n");
		Adapter->AutoNeg = 1;
		break;
	case 1:
		printk ("Warning: Half Duplex specified without Speed\n");
		printk ("Using Auto-negotiation at Half Duplex only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_10_HALF | ADVERTISE_100_HALF;
		break;
	case 2:
		printk ("Warning: Full Duplex specified without Speed\n");
		printk ("Using Auto-negotiation at Full Duplex only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_10_FULL | ADVERTISE_100_FULL | ADVERTISE_1000_FULL;
		break;
	case 10:
		printk ("Warning: 10 Mbps Speed specified without Duplex\n");
		printk ("Using Auto-negotiation at 10 Mbps only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_10_HALF | ADVERTISE_10_FULL;
		break;
	case 11:
		printk ("Forcing to 10 Mbps Half Duplex\n");
		Adapter->AutoNeg = 0;
		Adapter->ForcedSpeedDuplex = HALF_10;
		Adapter->AutoNegAdvertised = 0;
		break;
	case 12:
		printk ("Forcing to 10 Mbps Full Duplex\n");
		Adapter->AutoNeg = 0;
		Adapter->ForcedSpeedDuplex = FULL_10;
		Adapter->AutoNegAdvertised = 0;
		break;
	case 100:
		printk ("Warning: 100 Mbps Speed specified without Duplex\n");
		printk ("Using Auto-negotiation at 100 Mbps only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_100_HALF | ADVERTISE_100_FULL;
		break;
	case 101:
		printk ("Forcing to 100 Mbps Half Duplex\n");
		Adapter->AutoNeg = 0;
		Adapter->ForcedSpeedDuplex = HALF_100;
		Adapter->AutoNegAdvertised = 0;
		break;
	case 102:
		printk ("Forcing to 100 Mbps Full Duplex\n");
		Adapter->AutoNeg = 0;
		Adapter->ForcedSpeedDuplex = FULL_100;
		Adapter->AutoNegAdvertised = 0;
		break;
	case 1000:
		printk ("Warning: 1000 Mbps Speed specified without Duplex\n");
		printk ("Using Auto-negotiation at 1000 Mbps Full Duplex only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_1000_FULL;
		break;
	case 1001:
		printk ("Warning: Half Duplex is not supported at 1000 Mbps\n");
		printk ("Using Auto-negotiation at 1000 Mbps Full Duplex only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_1000_FULL;
		break;
	case 1002:
		printk ("Using Auto-negotiation at 1000 Mbps Full Duplex only\n");
		Adapter->AutoNeg = 1;
		Adapter->AutoNegAdvertised = ADVERTISE_1000_FULL;
		break;
	default:
		panic ("something is wrong in CheckLinkOptionsCopper");
	}

	return;
}

/*********************************************************************/
/*! @brief  Initialize general software structures (struct adapter)
 *  @param  Adapter board private structure to initialize
 *
 *  SoftwareInit initializes the Adapter private data structure.
 *  Fields are initialized based on PCI device information and
 *  OS network device settings (MTU size).
 *********************************************************************/

static int
SoftwareInit (struct adapter * Adapter)
{
	struct net_device *netdev = Adapter->netdev;
	struct pci_dev *pdev = Adapter->pdev;
	u32 DeviceStatusReg;

	E1000_DBG ("SoftwareInit\n");

	/* PCI config space info */

	pci_read_config_word (pdev, PCI_VENDOR_ID, &Adapter->VendorId);
	pci_read_config_word (pdev, PCI_DEVICE_ID, &Adapter->DeviceId);
	pci_read_config_byte (pdev, PCI_REVISION_ID, &Adapter->RevId);
	pci_read_config_word (pdev, PCI_SUBSYSTEM_VENDOR_ID, &Adapter->SubVendorId);
	pci_read_config_word (pdev, PCI_SUBSYSTEM_ID, &Adapter->SubSystemId);
	pci_read_config_word (pdev, PCI_COMMAND, &Adapter->PciCommandWord);

	/* PCI bus type, speed, and width */

	DeviceStatusReg = E1000_READ_REG (Status);
	Adapter->BusType = (DeviceStatusReg & E1000_STATUS_PCIX_MODE) ? E1000_BUS_TYPE_PCIX : E1000_BUS_TYPE_PCI;
	if (Adapter->BusType == E1000_BUS_TYPE_PCI) {
		Adapter->BusSpeed = (DeviceStatusReg & E1000_STATUS_PCI66) ? E1000_BUS_SPEED_PCI_66MHZ : E1000_BUS_SPEED_PCI_33MHZ;
	} else {
		switch (DeviceStatusReg & E1000_STATUS_PCIX_SPEED) {
		case E1000_STATUS_PCIX_SPEED_66:
			Adapter->BusSpeed = E1000_BUS_SPEED_PCIX_50_66MHZ;
			break;
		case E1000_STATUS_PCIX_SPEED_100:
			Adapter->BusSpeed = E1000_BUS_SPEED_PCIX_66_100MHZ;
			break;
		case E1000_STATUS_PCIX_SPEED_133:
			Adapter->BusSpeed = E1000_BUS_SPEED_PCIX_100_133MHZ;
			break;
		default:
			Adapter->BusSpeed = E1000_BUS_SPEED_PCIX_RESERVED;
			break;
		}
	}
	Adapter->BusWidth = (DeviceStatusReg & E1000_STATUS_BUS64) ? E1000_BUS_WIDTH_64_BIT : E1000_BUS_WIDTH_32_BIT;

	/* Initial Receive Buffer Length */

	if ((netdev->mtu + ENET_HEADER_SIZE + CRC_LENGTH) < E1000_RXBUFFER_2048)
		Adapter->RxBufferLen = E1000_RXBUFFER_2048;
	else if ((netdev->mtu + ENET_HEADER_SIZE + CRC_LENGTH) < E1000_RXBUFFER_4096)
		Adapter->RxBufferLen = E1000_RXBUFFER_4096;
	else if ((netdev->mtu + ENET_HEADER_SIZE + CRC_LENGTH) < E1000_RXBUFFER_8192)
		Adapter->RxBufferLen = E1000_RXBUFFER_8192;
	else
		Adapter->RxBufferLen = E1000_RXBUFFER_16384;

	Adapter->MaxFrameSize = netdev->mtu + ENET_HEADER_SIZE + CRC_LENGTH;

	/* MAC and Phy settings */

	switch (Adapter->DeviceId) {
	case PCI_DEVICE_ID_82542:
		switch (Adapter->RevId) {
		case WISEMAN_2_0_REV_ID:
			Adapter->MacType = MAC_WISEMAN_2_0;
			break;
		case WISEMAN_2_1_REV_ID:
			Adapter->MacType = MAC_WISEMAN_2_1;
			break;
		default:
			Adapter->MacType = MAC_WISEMAN_2_0;
			E1000_ERR ("Could not identify 82542 revision\n");
		}
		break;
	case PCI_DEVICE_ID_82543GC_FIBER:
	case PCI_DEVICE_ID_82543GC_COPPER:
		Adapter->MacType = MAC_LIVENGOOD;
		break;
	case PCI_DEVICE_ID_82544EI_COPPER:
	case PCI_DEVICE_ID_82544GC_CREB:
		Adapter->MacType = MAC_CORDOVA;
		break;
	default:
		E1000_ERR ("Could not identify hardware\n");
		return -ENODEV;
	}

	Adapter->FlowControlHighWatermark = FC_DEFAULT_HI_THRESH;
	Adapter->FlowControlLowWatermark = FC_DEFAULT_LO_THRESH;
	Adapter->FlowControlPauseTime = FC_DEFAULT_TX_TIMER;
	Adapter->FlowControlSendXon = TRUE;

	/* Identify the Hardware - this is done by the gigabit shared code
	 * in e1000_initialize_hardware, but it would help to identify the NIC
	 * before bringing the hardware online for use in CheckOptions. */

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		DeviceStatusReg = E1000_READ_REG (Status);
		if (DeviceStatusReg & E1000_STATUS_TBIMODE) {
			Adapter->MediaType = MEDIA_TYPE_FIBER;
		} else {
			Adapter->MediaType = MEDIA_TYPE_COPPER;
		}
	} else {
		Adapter->MediaType = MEDIA_TYPE_FIBER;
	}

	if ((E1000_REPORT_TX_EARLY == 0) || (E1000_REPORT_TX_EARLY == 1)) {
		Adapter->ReportTxEarly = E1000_REPORT_TX_EARLY;
	} else {
		if (Adapter->MacType < MAC_LIVENGOOD) {

			Adapter->ReportTxEarly = 0;
		} else {
			Adapter->ReportTxEarly = 1;
		}
	}

	Adapter->TbiCompatibilityEnable = TRUE;

	atomic_set (&Adapter->tx_timeout, 0);

	spin_lock_init (&Adapter->stats_lock);
	spin_lock_init (&Adapter->rx_fill_lock);

	return 0;
}

/*********************************************************************/
/*! @brief  prepare the hardware
 *  @param  Adapter board private struct containing configuration
 *  @return 0 on success, negative on failure
 *
 *  Initialize the hardware to a configuration as specified by the
 *  Adapter structure.  The controler is reset, the EEPROM is
 *  verified, the MAC address is set, then the shared initilization
 *  routines are called.
 *********************************************************************/

static int
HardwareInit (struct adapter * Adapter)
{
	struct net_device *netdev = Adapter->netdev;

	E1000_DBG ("HardwareInit\n");

	/* Issue a global reset */

	Adapter->e1000_adapter_stopped = FALSE;
	e1000_adapter_stop (Adapter);
	Adapter->e1000_adapter_stopped = FALSE;

	/* make sure the EEPROM is good */

	if (!e1000_validate_eeprom_checksum (Adapter)) {
		E1000_ERR ("The EEPROM Checksum Is Not Valid\n");
		return -1;
	}

	/* copy the MAC address out of the EEPROM */

	ReadNodeAddress (Adapter, Adapter->PermNetAddress);
	memcpy (netdev->dev_addr, Adapter->PermNetAddress, netdev->addr_len);
	memcpy (Adapter->CurrentNetAddress, netdev->dev_addr, netdev->addr_len);

	e1000_read_part_number (Adapter, &(Adapter->PartNumber));

	if (!e1000_initialize_hardware (Adapter)) {
		E1000_ERR ("Hardware Initialization Failed\n");
		return -1;
	}

	e1000_check_for_link (Adapter);

	if (E1000_READ_REG (Status) & E1000_STATUS_LU)
		Adapter->LinkIsActive = TRUE;
	else
		Adapter->LinkIsActive = FALSE;

	if (Adapter->LinkIsActive) {
		e1000_get_speed_and_duplex (Adapter, &Adapter->LineSpeed, &Adapter->FullDuplex);
	} else {
		Adapter->LineSpeed = 0;
		Adapter->FullDuplex = 0;
	}

	return 0;
}

/*********************************************************************/
/*! @brief  Loads a physical address into a descriptor
 *  @param  desc_addr pointer to the address structure
 *  @param  addr physical address (dma_addr_t)
 *
 *  Loads a physical address into a 64-bit Little-Endian structure
 *  that the PRO/1000 can understand, regardless of host CPU
 *  endianness or pointer length
 *********************************************************************/

static inline void
e1000_load_addr (E1000_64_BIT_PHYSICAL_ADDRESS * desc_addr, uint64_t addr)
{
	desc_addr->Lo32 = cpu_to_le32 ((u32) (addr & 0x00000000FFFFFFFF));
	desc_addr->Hi32 = cpu_to_le32 (addr >> 32);
	return;
}

/*********************************************************************/
/*! @brief  Reads the MAC address from the EEPROM
 *  @param  Adapter board private structure
 *  @param  NodeAddress pointer to an array of bytes
 *********************************************************************/

static void
ReadNodeAddress (struct adapter * Adapter, u8 * NodeAddress)
{
	u16 EepromWordValue;
	int i;

	E1000_DBG ("ReadNodeAddress\n");

	for (i = 0; i < NODE_ADDRESS_SIZE; i += 2) {
		EepromWordValue = e1000_read_eeprom_word (Adapter, EEPROM_NODE_ADDRESS_BYTE_0 + (i / 2));
		NodeAddress[i] = (u8) (EepromWordValue & 0x00FF);
		NodeAddress[i + 1] = (u8) (EepromWordValue >> 8);
	}

	return;
}

/*********************************************************************/
/*! @brief  Called when a network interface is made active
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *  @return 0 on success, negative value on failure
 *
 *  The open entry point is called when a network interface is made
 *  active by the system (IFF_UP).  At this point all resources needed
 *  for transmit and receive operations are allocated, the interrupt
 *  handler is registered with the OS, the watchdog timer is started,
 *  and the stack is notified that the interface is ready.
 *********************************************************************/

int
e1000_open (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;

	E1000_DBG ("e1000_open\n");

	/* prevent multiple opens when dealing with iANS */

	if (test_and_set_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
		return -EBUSY;
	}

	/* e1000_close issues a global reset (e1000_adapter_stop)
	 * so HardwareInit must be called again or the hardware
	 * will resume in it's default state */

	if (HardwareInit (Adapter) < 0) {
		clear_bit (E1000_BOARD_OPEN, &Adapter->flags);
		return -EBUSY;
	}
#ifdef IANS
	/* restore VLAN settings */
	if ((IANS_BD_TAGGING_MODE) (ANS_PRIVATE_DATA_FIELD (Adapter)->tag_mode) != IANS_BD_TAGGING_NONE)
		bd_ans_hw_EnableVLAN (Adapter);
#endif

	/* hook the interrupt */

	if (request_irq (netdev->irq, &e1000_intr, SA_SHIRQ, e1000_driver_name, netdev) != 0) {
		clear_bit (E1000_BOARD_OPEN, &Adapter->flags);
		return -EBUSY;
	}

	Adapter->e1000_adapter_stopped = FALSE;

	/* allocate transmit descriptors */

	if (SetupTransmitStructures (Adapter) != 0) {
		e1000_adapter_stop (Adapter);
		free_irq (netdev->irq, netdev);
		clear_bit (E1000_BOARD_OPEN, &Adapter->flags);
		return -ENOMEM;
	}
	ResetTU (Adapter);

	/* allocate receive descriptors and buffers */

	if (SetupReceiveStructures (Adapter) != 0) {
		e1000_adapter_stop (Adapter);
		FreeTransmitStructures (Adapter);
		free_irq (netdev->irq, netdev);
		clear_bit (E1000_BOARD_OPEN, &Adapter->flags);
		return -ENOMEM;
	}
	SetupRctl (Adapter);
	ResetRU (Adapter);

	/* fill Rx ring with sk_buffs */

	tasklet_init (&Adapter->rx_fill_tasklet, ReceiveBufferFill, (unsigned long) Adapter);

	tasklet_schedule (&Adapter->rx_fill_tasklet);

	/* Set the watchdog timer for 2 seconds */

	init_timer (&Adapter->timer_id);
	Adapter->timer_id.function = &e1000_watchdog;
	Adapter->timer_id.data = (unsigned long) netdev;
	mod_timer (&Adapter->timer_id, (jiffies + 2 * HZ));

	/* stats accumulated while down are dropped
	 * this does not clear the running total */

	e1000_clear_hw_stats_counters (Adapter);

	Adapter->IntMask = IMS_ENABLE_MASK;
	EnableInterrupts (Adapter);
	netif_start_queue (netdev);

#ifdef MODULE

	/* Incrementing the module use count prevents a driver from being
	 * unloaded while an active network interface is using it. */

	MOD_INC_USE_COUNT;

#endif

	return 0;
}

/*********************************************************************/
/*! @brief  Disables a network interface
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *  @return 0, this is not allowed to fail
 *
 *  The close entry point is called when an interface is de-activated
 *  by the OS.  The hardware is still under the drivers control, but
 *  needs to be disabled.  A global MAC reset is issued to stop the
 *  hardware, and all transmit and receive resources are freed.
 *********************************************************************/

int
e1000_close (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;

	E1000_DBG ("e1000_close\n");

	if (!test_bit (E1000_BOARD_OPEN, &Adapter->flags))
		return 0;

	/* Issue a global reset */

	e1000_adapter_stop (Adapter);

	/* free OS resources */

	netif_stop_queue (netdev);
	free_irq (netdev->irq, netdev);
	del_timer (&Adapter->timer_id);

	/* Make sure the tasklet won't be left after ifconfig down */
	/*
	   Assumption: tasklet is ALREADY enabled, ie, t->count == 0.
	   Otherwise, tasklet is still left in the tasklet list, and,
	   tasklet_kill will not be able to return (hang).
	 */
	tasklet_kill (&Adapter->rx_fill_tasklet);

	/* free software resources */

	FreeTransmitStructures (Adapter);
	FreeReceiveStructures (Adapter);

#ifdef MODULE

	/* decrement the module usage count
	 * so that the driver can be unloaded */

	MOD_DEC_USE_COUNT;

#endif

	clear_bit (E1000_BOARD_OPEN, &Adapter->flags);
	return 0;
}

/*********************************************************************/
/*! @brief  allocate Tx resources (Descriptors)
 *  @param  Adapter board private structure
 *  @return 0 on success, negative on failure
 *
 *  SetupTransmitStructures allocates all software transmit resources
 *  and enabled the Tx unit of the MAC.
 *********************************************************************/

static int
SetupTransmitStructures (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	int size;

	E1000_DBG ("SetupTransmitStructures\n");

	size = sizeof (struct e1000_buffer) * Adapter->NumTxDescriptors;
	Adapter->tx_skb = kmalloc (size, GFP_KERNEL);
	if (Adapter->tx_skb == NULL) {
		return -ENOMEM;
	}
	memset (Adapter->tx_skb, 0, size);

	/* round up to nearest 4K */

	size = E1000_ROUNDUP2 (Adapter->NumTxDescriptors * sizeof (E1000_TRANSMIT_DESCRIPTOR), 4096);

	Adapter->TxDescriptors = pci_alloc_consistent (pdev, size, &Adapter->TxDescDMA);
	if (Adapter->TxDescriptors == NULL) {
		kfree (Adapter->tx_skb);
		return -ENOMEM;
	}
	memset (Adapter->TxDescriptors, 0, size);

	atomic_set (&Adapter->NumTxDescriptorsAvail, Adapter->NumTxDescriptors);
	Adapter->NextAvailTxDescriptor = 0;
	Adapter->OldestUsedTxDescriptor = 0;

	return 0;
}

/*********************************************************************/
/*! @brief  Configure 8254x Transmit Unit after Reset
 *  @param  Adapter board private structure
 *
 *  Configure the Tx unit of the MAC after a reset.
 *********************************************************************/

static void
ResetTU (struct adapter * Adapter)
{
	u32 reg_tctl, reg_tipg;

	/* Setup the Base and Length of the Rx Descriptor Ring */
	/* TxDescDMA can be either a 32 or 64 bit value */

#if (BITS_PER_LONG == 32)
	E1000_WRITE_REG (Tdbal, Adapter->TxDescDMA);
	E1000_WRITE_REG (Tdbah, 0);
#elif ( BITS_PER_LONG == 64)
	E1000_WRITE_REG (Tdbal, (u32) (Adapter->TxDescDMA & 0x00000000FFFFFFFF));
	E1000_WRITE_REG (Tdbah, (u32) (Adapter->TxDescDMA >> 32));
#else
#error "Unsupported System - does not use 32 or 64 bit pointers!"
#endif

	E1000_WRITE_REG (Tdl, Adapter->NumTxDescriptors * sizeof (E1000_TRANSMIT_DESCRIPTOR));

	/* Setup the HW Tx Head and Tail descriptor pointers */

	E1000_WRITE_REG (Tdh, 0);
	E1000_WRITE_REG (Tdt, 0);

	/* Zero out the 82542 Tx Queue State registers - we don't use them */

	if (Adapter->MacType < MAC_LIVENGOOD) {
		E1000_WRITE_REG (Tqsal, 0);
		E1000_WRITE_REG (Tqsah, 0);
	}

	/* Set the default values for the Tx Inter Packet Gap timer */

	switch (Adapter->MacType) {
	case MAC_LIVENGOOD:
	case MAC_CORDOVA:
		if (Adapter->MediaType == MEDIA_TYPE_FIBER)
			reg_tipg = DEFAULT_LVGD_TIPG_IPGT_FIBER;
		else
			reg_tipg = DEFAULT_LVGD_TIPG_IPGT_COPPER;
		reg_tipg |= DEFAULT_LVGD_TIPG_IPGR1 << E1000_TIPG_IPGR1_SHIFT;
		reg_tipg |= DEFAULT_LVGD_TIPG_IPGR2 << E1000_TIPG_IPGR2_SHIFT;
		break;
	case MAC_WISEMAN_2_0:
	case MAC_WISEMAN_2_1:
	default:
		reg_tipg = DEFAULT_WSMN_TIPG_IPGT;
		reg_tipg |= DEFAULT_WSMN_TIPG_IPGR1 << E1000_TIPG_IPGR1_SHIFT;
		reg_tipg |= DEFAULT_WSMN_TIPG_IPGR2 << E1000_TIPG_IPGR2_SHIFT;
		break;
	}
	E1000_WRITE_REG (Tipg, reg_tipg);

	/* Set the Tx Interrupt Delay register */

	E1000_WRITE_REG (Tidv, Adapter->TxIntDelay);

	/* Program the Transmit Control Register */

	reg_tctl = E1000_TCTL_PSP | E1000_TCTL_EN | (E1000_COLLISION_THRESHOLD << E1000_CT_SHIFT);
	if (Adapter->FullDuplex == 1) {
		reg_tctl |= E1000_FDX_COLLISION_DISTANCE << E1000_COLD_SHIFT;
	} else {
		reg_tctl |= E1000_HDX_COLLISION_DISTANCE << E1000_COLD_SHIFT;
	}
	E1000_WRITE_REG (Tctl, reg_tctl);

	/* Setup Transmit Descriptor Settings for this adapter */
	Adapter->TxdCmd = E1000_TXD_CMD_IFCS;

	if (Adapter->TxIntDelay > 0)
		Adapter->TxdCmd |= E1000_TXD_CMD_IDE;
	if (Adapter->ReportTxEarly == 1)
		Adapter->TxdCmd |= E1000_TXD_CMD_RS;
	else
		Adapter->TxdCmd |= E1000_TXD_CMD_RPS;

	return;
}

/*********************************************************************/
/*! @brief allocate Rx resources (Descriptors, receive SKBs)
 *  @param  Adapter board private structure
 *  @return 0 on success, negative on failure
 *
 *  SetupReceiveStructures allocates all software receive resources
 *  and network buffers, and enables the Rx unit of the MAC.
 *********************************************************************/

static int
SetupReceiveStructures (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	int size;

	E1000_DBG ("SetupReceiveStructures\n");

	size = sizeof (struct e1000_buffer) * Adapter->NumRxDescriptors;
	Adapter->rx_skb = kmalloc (size, GFP_KERNEL);
	if (Adapter->rx_skb == NULL) {
		return -ENOMEM;
	}
	memset (Adapter->rx_skb, 0, size);

	/* Round up to nearest 4K */

	size = E1000_ROUNDUP2 (Adapter->NumRxDescriptors * sizeof (E1000_RECEIVE_DESCRIPTOR), 4096);

	Adapter->RxDescriptors = pci_alloc_consistent (pdev, size, &Adapter->RxDescDMA);
	if (Adapter->RxDescriptors == NULL) {
		kfree (Adapter->rx_skb);
		return -ENOMEM;
	}
	memset (Adapter->RxDescriptors, 0, size);

	Adapter->NextRxDescriptorToCheck = 0;
	atomic_set (&Adapter->NumRxDescriptorsEmpty, Adapter->NumRxDescriptors);

	Adapter->NextRxDescriptorToFill = 0;

	return 0;
}

/*********************************************************************/
/*!
 *********************************************************************/
static void
SetupRctl (struct adapter * Adapter)
{
	u32 reg_rctl;

	/* Setup the Receive Control Register */
	reg_rctl = E1000_RCTL_EN | E1000_RCTL_BAM | E1000_RCTL_LBM_NO | E1000_RCTL_RDMTS0_HALF | (Adapter->MulticastFilterType << E1000_RCTL_MO_SHIFT);

	if (Adapter->TbiCompatibilityOn == TRUE)
		reg_rctl |= E1000_RCTL_SBP;

	switch (Adapter->RxBufferLen) {
	case E1000_RXBUFFER_2048:
	default:
		reg_rctl |= E1000_RCTL_SZ_2048;
		break;
	case E1000_RXBUFFER_4096:
		reg_rctl |= E1000_RCTL_SZ_4096 | E1000_RCTL_BSEX | E1000_RCTL_LPE;
		break;
	case E1000_RXBUFFER_8192:
		reg_rctl |= E1000_RCTL_SZ_8192 | E1000_RCTL_BSEX | E1000_RCTL_LPE;
		break;
	case E1000_RXBUFFER_16384:
		reg_rctl |= E1000_RCTL_SZ_16384 | E1000_RCTL_BSEX | E1000_RCTL_LPE;
		break;
	}

	E1000_WRITE_REG (Rctl, reg_rctl);
}

/*********************************************************************/
/*! @brief  Configure 8254x Receive Unit after Reset
 *  @param  Adapter board private structure
 *
 *  Configure the Rx unit of the MAC after a reset.
 *********************************************************************/

static void
ResetRU (struct adapter * Adapter)
{
	u32 reg_rctl;
	u32 reg_rxcsum;

	/* make sure receives are disabled while setting up the descriptor ring */
	reg_rctl = E1000_READ_REG (Rctl);
	E1000_WRITE_REG (Rctl, reg_rctl & ~E1000_RCTL_EN);

	/* set the Receive Delay Timer Register */
	E1000_WRITE_REG (Rdtr0, Adapter->RxIntDelay | E1000_RDT0_FPDB);

	/* Setup the Base and Length of the Rx Descriptor Ring */
	/* RxDescDMA can be either a 32 or 64 bit value */

#if (BITS_PER_LONG == 32)
	E1000_WRITE_REG (Rdbal0, Adapter->RxDescDMA);
	E1000_WRITE_REG (Rdbah0, 0);
#elif ( BITS_PER_LONG == 64)
	E1000_WRITE_REG (Rdbal0, (u32) (Adapter->RxDescDMA & 0x00000000FFFFFFFF));
	E1000_WRITE_REG (Rdbah0, (u32) (Adapter->RxDescDMA >> 32));
#else
#error "Unsupported System - does not use 32 or 64 bit pointers!"
#endif

	E1000_WRITE_REG (Rdlen0, Adapter->NumRxDescriptors * sizeof (E1000_RECEIVE_DESCRIPTOR));

	/* Setup the HW Rx Head and Tail Descriptor Pointers */
	E1000_WRITE_REG (Rdh0, 0);
	E1000_WRITE_REG (Rdt0, 0);

	/* Zero out the registers associated with the 82542 second receive
	 * descriptor ring - we don't use it */
	if (Adapter->MacType < MAC_LIVENGOOD) {
		E1000_WRITE_REG (Rdbal1, 0);
		E1000_WRITE_REG (Rdbah1, 0);
		E1000_WRITE_REG (Rdlen1, 0);
		E1000_WRITE_REG (Rdh1, 0);
		E1000_WRITE_REG (Rdt1, 0);
	}

	/* Enable 82543 Receive Checksum Offload for TCP and UDP */
	if ((Adapter->MacType >= MAC_LIVENGOOD) && (Adapter->RxChecksum == 1)) {
		reg_rxcsum = E1000_READ_REG (Rxcsum);
		reg_rxcsum |= E1000_RXCSUM_TUOFL;
		E1000_WRITE_REG (Rxcsum, reg_rxcsum);
	}

	/* Enable Receives */
	E1000_WRITE_REG (Rctl, reg_rctl);

	return;
}

/*********************************************************************/
/*! @brief  Free Tx Resources
 *  @param  Adapter board private structure
 *
 *  Free all transmit software resources
 *********************************************************************/

static void
FreeTransmitStructures (struct adapter * Adapter)
{
	int size;
	struct pci_dev *pdev = Adapter->pdev;

	E1000_DBG ("FreeTransmitStructures\n");

	CleanTxRing (Adapter);

	kfree (Adapter->tx_skb);
	Adapter->tx_skb = NULL;

	size = E1000_ROUNDUP2 (Adapter->NumTxDescriptors * sizeof (E1000_TRANSMIT_DESCRIPTOR), 4096);

	pci_free_consistent (pdev, size, Adapter->TxDescriptors, Adapter->TxDescDMA);

	Adapter->TxDescriptors = NULL;

	return;
}

/*********************************************************************/
/*! @brief  Free Tx Buffers
 *  @param  Adapter board private structure
 *********************************************************************/

static void
CleanTxRing (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	unsigned long size;
	int i;

	/* Free all the Tx ring sk_buffs */

	for (i = 0; i < Adapter->NumTxDescriptors; i++) {
		if (Adapter->tx_skb[i].skb != NULL) {

			pci_unmap_single (pdev, Adapter->tx_skb[i].dma, Adapter->tx_skb[i].length, PCI_DMA_TODEVICE);

			dev_kfree_skb (Adapter->tx_skb[i].skb);

			Adapter->tx_skb[i].skb = NULL;
		}
	}

	size = sizeof (struct e1000_buffer) * Adapter->NumTxDescriptors;
	memset (Adapter->tx_skb, 0, size);

	/* round up to nearest 4K */

	size = E1000_ROUNDUP2 (Adapter->NumTxDescriptors * sizeof (E1000_TRANSMIT_DESCRIPTOR), 4096);

	/* Zero out the descriptor ring */

	memset (Adapter->TxDescriptors, 0, size);

	atomic_set (&Adapter->NumTxDescriptorsAvail, Adapter->NumTxDescriptors);
	Adapter->NextAvailTxDescriptor = 0;
	Adapter->OldestUsedTxDescriptor = 0;

	return;
}

/*********************************************************************/
/*! @brief  Free Rx Resources
 *  @param  Adapter board private structure
 *
 *  Free all receive software resources
 *********************************************************************/

static void
FreeReceiveStructures (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	int size;

	E1000_DBG ("FreeReceiveStructures\n");

	tasklet_disable (&Adapter->rx_fill_tasklet);

	CleanRxRing (Adapter);

	kfree (Adapter->rx_skb);
	Adapter->rx_skb = NULL;

	size = E1000_ROUNDUP2 (Adapter->NumRxDescriptors * sizeof (E1000_RECEIVE_DESCRIPTOR), 4096);

	pci_free_consistent (pdev, size, Adapter->RxDescriptors, Adapter->RxDescDMA);

	Adapter->RxDescriptors = NULL;

	return;
}

/*********************************************************************/
/*! @brief  Free Rx Buffers
 *  @param  Adapter board private structure
 *********************************************************************/

static void
CleanRxRing (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	unsigned long size;
	int i;

	/* Free all the Rx ring sk_buffs */

	for (i = 0; i < Adapter->NumRxDescriptors; i++) {
		if (Adapter->rx_skb[i].skb != NULL) {

			pci_unmap_single (pdev, Adapter->rx_skb[i].dma, Adapter->rx_skb[i].length, PCI_DMA_FROMDEVICE);

			dev_kfree_skb (Adapter->rx_skb[i].skb);

			Adapter->rx_skb[i].skb = NULL;
		}
	}

	size = sizeof (struct e1000_buffer) * Adapter->NumRxDescriptors;
	memset (Adapter->rx_skb, 0, size);

	/* round up to nearest 4K */

	size = E1000_ROUNDUP2 (Adapter->NumRxDescriptors * sizeof (E1000_RECEIVE_DESCRIPTOR), 4096);

	/* Zero out the descriptor ring */

	memset (Adapter->RxDescriptors, 0, size);

	atomic_set (&Adapter->NumRxDescriptorsEmpty, Adapter->NumRxDescriptors);
	Adapter->NextRxDescriptorToCheck = 0;
	Adapter->NextRxDescriptorToFill = 0;

	return;
}

/*********************************************************************/
/*! @brief  Multicast and Promiscuous mode set
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *
 *  The set_multi entry point is called whenever the multicast address
 *  list or the network interface flags are updated.  This routine is
 *  resposible for configuring the hardware for proper multicast,
 *  promiscuous mode, and all-multi behavior.
 *********************************************************************/

void
e1000_set_multi (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;
	struct pci_dev *pdev = Adapter->pdev;
	u32 reg_rctl;
	unsigned char mta[MAX_NUM_MULTICAST_ADDRESSES * ETH_LENGTH_OF_ADDRESS];
	u16 PciCommandWord;
	struct dev_mc_list *mc_ptr;
	int i;

	E1000_DBG ("e1000_set_multi\n");

	reg_rctl = E1000_READ_REG (Rctl);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		if (Adapter->PciCommandWord & PCI_COMMAND_INVALIDATE) {
			PciCommandWord = Adapter->PciCommandWord & ~PCI_COMMAND_INVALIDATE;
			pci_write_config_word (pdev, PCI_COMMAND, PciCommandWord);
		}
		reg_rctl |= E1000_RCTL_RST;
		E1000_WRITE_REG (Rctl, reg_rctl);
		mdelay (5);
		if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
			tasklet_disable (&Adapter->rx_fill_tasklet);
			CleanRxRing (Adapter);
		}
	}

	/* Check for Promiscuous and All Multicast modes */

	if (netdev->flags & IFF_PROMISC) {
		reg_rctl |= (E1000_RCTL_UPE | E1000_RCTL_MPE);
		Adapter->NosEnabledMulticastPromiscuous = TRUE;
	} else if (netdev->flags & IFF_ALLMULTI) {
		reg_rctl |= E1000_RCTL_MPE;
		reg_rctl &= ~E1000_RCTL_UPE;
		Adapter->NosEnabledMulticastPromiscuous = TRUE;
	} else {
		reg_rctl &= ~(E1000_RCTL_UPE | E1000_RCTL_MPE);
		Adapter->NosEnabledMulticastPromiscuous = FALSE;
	}

	if (netdev->mc_count > MAX_NUM_MULTICAST_ADDRESSES) {
		reg_rctl |= E1000_RCTL_MPE;
		E1000_WRITE_REG (Rctl, reg_rctl);
	} else {
		E1000_WRITE_REG (Rctl, reg_rctl);
		for (i = 0, mc_ptr = netdev->mc_list; mc_ptr; i++, mc_ptr = mc_ptr->next)
			memcpy (&mta[i * ETH_LENGTH_OF_ADDRESS], mc_ptr->dmi_addr, ETH_LENGTH_OF_ADDRESS);
		e1000_multicast_address_list_update (Adapter, mta, netdev->mc_count, 0);
	}

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		reg_rctl = E1000_READ_REG (Rctl);
		reg_rctl &= ~E1000_RCTL_RST;
		E1000_WRITE_REG (Rctl, reg_rctl);
		mdelay (5);
		if (Adapter->PciCommandWord & PCI_COMMAND_INVALIDATE) {
			pci_write_config_word (pdev, PCI_COMMAND, Adapter->PciCommandWord);
		}
		if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
			ResetRU (Adapter);
			tasklet_enable (&Adapter->rx_fill_tasklet);
		}
	}

	return;
}

#ifdef IANS
/* flush Tx queue without link */
static void
TxFlush (struct adapter * Adapter)
{
	u32 ctrl, txcw, icr;

	Adapter->IntMask = 0;
	DisableInterrupts (Adapter);
	synchronize_irq ();

	if (Adapter->MacType < MAC_LIVENGOOD) {
		/* Transmit Unit Reset */
		E1000_WRITE_REG (Tctl, E1000_TCTL_RST);
		E1000_WRITE_REG (Tctl, 0);
		CleanTxRing (Adapter);
		ResetTU (Adapter);
	} else {
		/* turn off autoneg, set link up, and invert loss of signal */
		txcw = E1000_READ_REG (Txcw);
		ctrl = E1000_READ_REG (Ctrl);
		E1000_WRITE_REG (Txcw, txcw & ~E1000_TXCW_ANE);
		E1000_WRITE_REG (Ctrl, ctrl | E1000_CTRL_SLU | E1000_CTRL_ILOS);
		/* delay to flush queue, then clean up */
		mdelay (20);
		ProcessTransmitInterrupts (Adapter);
		E1000_WRITE_REG (Ctrl, ctrl);
		E1000_WRITE_REG (Txcw, txcw);
		/* clear the link status change interrupts this caused */
		icr = E1000_READ_REG (Icr);
	}

	Adapter->IntMask = IMS_ENABLE_MASK;
	EnableInterrupts (Adapter);
	return;
}
#endif

/*********************************************************************/
/*! @brief  Timer Call-back
 *  @ingroup entry_point
 *  @param  data pointer to netdev cast into an unsigned long
 *********************************************************************/

void
e1000_watchdog (unsigned long data)
{
	struct net_device *netdev = (struct net_device *) data;
	struct adapter * Adapter = netdev->priv;
	int flags;

	e1000_check_for_link (Adapter);

	if (E1000_READ_REG (Status) & E1000_STATUS_LU) {
		if (Adapter->LinkIsActive == FALSE) {

			e1000_get_speed_and_duplex (Adapter, &Adapter->LineSpeed, &Adapter->FullDuplex);
			printk (KERN_ERR "e1000: %s NIC Link is Up %d Mbps %s\n",
				netdev->name, Adapter->LineSpeed, Adapter->FullDuplex == FULL_DUPLEX ? "Full Duplex" : "Half Duplex");

			Adapter->LinkIsActive = TRUE;
		}
	} else {
		if (Adapter->LinkIsActive == TRUE) {
			Adapter->LineSpeed = 0;
			Adapter->FullDuplex = 0;
			printk (KERN_ERR "e1000: %s NIC Link is Down\n", netdev->name);
			Adapter->LinkIsActive = FALSE;
		}
	}

	UpdateStatsCounters (Adapter);

	if (atomic_read (&Adapter->tx_timeout) > 1)
		atomic_dec (&Adapter->tx_timeout);

	if ((Adapter->LinkIsActive == TRUE) && (atomic_read (&Adapter->tx_timeout) == 1)) {

		if (E1000_READ_REG (Status) & E1000_STATUS_TXOFF) {
			atomic_set (&Adapter->tx_timeout, 3);
		} else {

			e1000_hibernate_adapter (netdev);

#ifdef IANS
			if (Adapter->iANSdata->reporting_mode == IANS_STATUS_REPORTING_ON) {
				Adapter->LinkIsActive = FALSE;
				bd_ans_os_Watchdog (netdev, Adapter);
				Adapter->LinkIsActive = TRUE;
			}
#endif
			atomic_set (&Adapter->tx_timeout, 0);
			e1000_wakeup_adapter (netdev);
		}
	}
#ifdef IANS
	if (Adapter->iANSdata->reporting_mode == IANS_STATUS_REPORTING_ON)
		bd_ans_os_Watchdog (netdev, Adapter);

	if ((Adapter->iANSdata->iANS_status == IANS_COMMUNICATION_UP) && (Adapter->LinkIsActive == FALSE)) {

		/* don't sit on SKBs while link is down */

		if (atomic_read (&Adapter->NumTxDescriptorsAvail) < Adapter->NumTxDescriptors) {

			spin_lock_irqsave (&netdev->xmit_lock, flags);
			TxFlush (Adapter);
			spin_unlock_irqrestore (&netdev->xmit_lock, flags);
		}

		spin_lock_irqsave (&netdev->queue_lock, flags);
		qdisc_reset (netdev->qdisc);
		spin_unlock_irqrestore (&netdev->queue_lock, flags);
	}
#endif

	/* Reset the timer */
	mod_timer (&Adapter->timer_id, jiffies + 2 * HZ);

	return;
}

/*********************************************************************/
/*********************************************************************/
static inline void
TxChecksumSetup (struct adapter * Adapter, struct sk_buff *skb, u32 * txd_upper, u32 * txd_lower)
{

	PE1000_TCPIP_CONTEXT_TRANSMIT_DESCRIPTOR TXD;
	int i;

	if (skb->protocol != __constant_htons (ETH_P_IP)) {
		*txd_upper = 0;
		*txd_lower = Adapter->TxdCmd;
		return;
	}

	switch (skb->nh.iph->protocol) {
	case IPPROTO_TCP:
		/* Offload TCP checksum */
		*txd_upper = E1000_TXD_POPTS_TXSM << 8;
		*txd_lower = Adapter->TxdCmd | E1000_TXD_CMD_DEXT | E1000_TXD_DTYP_D;
		if (Adapter->ActiveChecksumContext == OFFLOAD_TCP_IP)
			return;
		else
			Adapter->ActiveChecksumContext = OFFLOAD_TCP_IP;
		break;
	case IPPROTO_UDP:
		/* Offload UDP checksum */
		*txd_upper = E1000_TXD_POPTS_TXSM << 8;
		*txd_lower = Adapter->TxdCmd | E1000_TXD_CMD_DEXT | E1000_TXD_DTYP_D;
		if (Adapter->ActiveChecksumContext == OFFLOAD_UDP_IP)
			return;
		else
			Adapter->ActiveChecksumContext = OFFLOAD_UDP_IP;
		break;
	default:
		/* no checksum to offload */
		*txd_upper = 0;
		*txd_lower = Adapter->TxdCmd;
		return;
	}

	/* If we reach this point, the checksum offload context
	 * needs to be reset */

	i = Adapter->NextAvailTxDescriptor;
	TXD = (PE1000_TCPIP_CONTEXT_TRANSMIT_DESCRIPTOR) & Adapter->TxDescriptors[i];

	TXD->LowerXsumSetup.IpFields.Ipcss = skb->nh.raw - skb->data;
	TXD->LowerXsumSetup.IpFields.Ipcso = ((skb->nh.raw + offsetof (struct iphdr, check)) - skb->data);
	TXD->LowerXsumSetup.IpFields.Ipcse = cpu_to_le16 (skb->h.raw - skb->data - 1);

	TXD->UpperXsumSetup.TcpFields.Tucss = (skb->h.raw - skb->data);
	TXD->UpperXsumSetup.TcpFields.Tucso = ((skb->h.raw + skb->csum) - skb->data);
	TXD->UpperXsumSetup.TcpFields.Tucse = 0;

	TXD->TcpSegSetup.DwordData = 0;
	TXD->CmdAndLength = cpu_to_le32 (E1000_TXD_CMD_DEXT) | Adapter->TxdCmd;

	i = (i + 1) % Adapter->NumTxDescriptors;
	atomic_dec (&Adapter->NumTxDescriptorsAvail);
	Adapter->NextAvailTxDescriptor = i;
	E1000_WRITE_REG (Tdt, Adapter->NextAvailTxDescriptor);
	return;
}

/*********************************************************************/
/*! @brief  Transmit entry point
 *  @ingroup entry_point
 *  @param  skb buffer with frame data to transmit
 *  @param  netdev network interface device structure
 *  @return 0 on success, negative on error
 *
 *  e1000_xmit_frame is called by the stack to initiate a transmit.
 *  The out of resource condition is checked after each successful Tx
 *  so that the stack can be notified, preventing the driver from
 *  ever needing to drop a frame.  The atomic operations on
 *  NumTxDescriptorsAvail are used to syncronize with the transmit
 *  interrupt processing code without the need for a spinlock.
 *********************************************************************/
#define txd_use_count(x) (((x) >> 12) + ((x) & 0x0fff ? 1 : 0))

int
e1000_xmit_frame (struct sk_buff *skb, struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;
	struct pci_dev *pdev = Adapter->pdev;
	PE1000_TRANSMIT_DESCRIPTOR CurrentTxDescriptor;
	int i, len, offset, txd_needed;
	u32 txd_upper, txd_lower;
#if ZEROCOPY
	int f;
	skb_frag_t *frag;
#endif

	E1000_DBG ("e1000_xmit_frame\n");

	if (Adapter->LinkIsActive == FALSE) {
		netif_stop_queue (netdev);
		return 1;
	}
#if ZEROCOPY
	txd_needed = txd_use_count (skb->len - skb->data_len);
	for (f = 0; f < skb_shinfo (skb)->nr_frags; f++) {
		frag = &skb_shinfo (skb)->frags[f];
		txd_needed += txd_use_count (frag->size);
	}
#else
	txd_needed = txd_use_count (skb->len);
#endif

	/* make sure there are enough Tx descriptors available in the ring */
	if (atomic_read (&Adapter->NumTxDescriptorsAvail) <= (txd_needed + 1)) {
		Adapter->net_stats.tx_dropped++;
#ifdef IANS
		if (Adapter->iANSdata->iANS_status == IANS_COMMUNICATION_UP)
			if (e1000_ans_notify)
				e1000_ans_notify (netdev, IANS_IND_XMIT_QUEUE_FULL);
#endif
		netif_stop_queue (netdev);

		return 1;
	}

	if (skb->ip_summed == CHECKSUM_HW) {
		TxChecksumSetup (Adapter, skb, &txd_upper, &txd_lower);
	} else {
		txd_upper = 0;
		txd_lower = Adapter->TxdCmd;
	}

	i = Adapter->NextAvailTxDescriptor;
	CurrentTxDescriptor = &Adapter->TxDescriptors[i];

#ifdef IANS
	if (Adapter->iANSdata->iANS_status == IANS_COMMUNICATION_UP) {
		CurrentTxDescriptor->Lower.DwordData = cpu_to_le32 (txd_lower);
		CurrentTxDescriptor->Upper.DwordData = cpu_to_le32 (txd_upper);
		if (bd_ans_os_Transmit (Adapter, CurrentTxDescriptor, &skb) == BD_ANS_FAILURE) {
			return 1;
		}
		txd_lower = le32_to_cpu (CurrentTxDescriptor->Lower.DwordData);
		txd_upper = le32_to_cpu (CurrentTxDescriptor->Upper.DwordData);
	}
#endif

#if ZEROCOPY
	len = skb->len - skb->data_len;
#else
	len = skb->len;
#endif
	offset = 0;

	while (len > 4096) {
		Adapter->tx_skb[i].length = 4096;
		Adapter->tx_skb[i].dma = pci_map_single (pdev, skb->data + offset, 4096, PCI_DMA_TODEVICE);

		e1000_load_addr (&(CurrentTxDescriptor->BufferAddress), Adapter->tx_skb[i].dma);

		CurrentTxDescriptor->Lower.DwordData = cpu_to_le32 (txd_lower | 4096);
		CurrentTxDescriptor->Upper.DwordData = cpu_to_le32 (txd_upper);

		len -= 4096;
		offset += 4096;
		i = (i + 1) % Adapter->NumTxDescriptors;
		atomic_dec (&Adapter->NumTxDescriptorsAvail);
		CurrentTxDescriptor = &Adapter->TxDescriptors[i];
	}
	Adapter->tx_skb[i].length = len;
	Adapter->tx_skb[i].dma = pci_map_single (pdev, skb->data + offset, len, PCI_DMA_TODEVICE);

	e1000_load_addr (&(CurrentTxDescriptor->BufferAddress), Adapter->tx_skb[i].dma);

	CurrentTxDescriptor->Lower.DwordData = cpu_to_le32 (txd_lower | len);
	CurrentTxDescriptor->Upper.DwordData = cpu_to_le32 (txd_upper);

#if ZEROCOPY
	if (!skb_shinfo (skb)->nr_frags) {
#endif
		CurrentTxDescriptor->Lower.DwordData |= cpu_to_le32 (E1000_TXD_CMD_EOP);
		Adapter->tx_skb[i].skb = skb;
#if ZEROCOPY
	} else {
		for (f = 0; f < skb_shinfo (skb)->nr_frags; f++) {
			frag = &skb_shinfo (skb)->frags[f];
			i = (i + 1) % Adapter->NumTxDescriptors;
			atomic_dec (&Adapter->NumTxDescriptorsAvail);
			CurrentTxDescriptor = &Adapter->TxDescriptors[i];

			len = frag->size;
			offset = 0;

			while (len > 4096) {
				Adapter->tx_skb[i].length = 4096;
				Adapter->tx_skb[i].dma = pci_map_single_high (pdev, frag->page, frag->page_offset + offset, 4096, PCI_DMA_TODEVICE);

				e1000_load_addr (&(CurrentTxDescriptor->BufferAddress), Adapter->tx_skb[i].dma);

				CurrentTxDescriptor->Lower.DwordData = cpu_to_le32 (txd_lower | 4096);
				CurrentTxDescriptor->Upper.DwordData = cpu_to_le32 (txd_upper);

				len -= 4096;
				offset += 4096;
				i = (i + 1) % Adapter->NumTxDescriptors;
				atomic_dec (&Adapter->NumTxDescriptorsAvail);
				CurrentTxDescriptor = &Adapter->TxDescriptors[i];
			}
			Adapter->tx_skb[i].dma = pci_map_single_high (pdev, frag->page, frag->page_offset + offset, len, PCI_DMA_TODEVICE);
			Adapter->tx_skb[i].length = len;
			e1000_load_addr (&(CurrentTxDescriptor->BufferAddress), Adapter->tx_skb[i].dma);
			CurrentTxDescriptor->Lower.DwordData = (txd_lower | frag->size);
			CurrentTxDescriptor->Upper.DwordData = (txd_upper);
		}
		/* EOP and SKB pointer go with the last fragment */
		CurrentTxDescriptor->Lower.DwordData |= (E1000_TXD_CMD_EOP);
		Adapter->tx_skb[i].skb = skb;
	}
#endif
	i = (i + 1) % Adapter->NumTxDescriptors;
	atomic_dec (&Adapter->NumTxDescriptorsAvail);

	/* Move the HW Tx Tail Pointer */
	Adapter->NextAvailTxDescriptor = i;

	E1000_WRITE_REG (Tdt, Adapter->NextAvailTxDescriptor);

	if (atomic_read (&Adapter->tx_timeout) == 0)
		atomic_set (&Adapter->tx_timeout, 3);

	netdev->trans_start = jiffies;

	return 0;
}

/*********************************************************************/
/*! @brief  Get System Network Statistics
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *  @return net_device_stats, system network stats structure
 *
 *  Returns the address of the device statistics structure.
 *  The statistics are actually updated from the timer callback.
 *********************************************************************/

struct net_device_stats *
e1000_get_stats (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;
	E1000_DBG ("e1000_get_stats\n");

	return &Adapter->net_stats;
}

/*********************************************************************/
/*! @brief  Change the Maximum Transfer Unit
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *  @param  new_mtu new value for maximum frame size
 *  @return 0 on success, negative on failure
 *********************************************************************/

int
e1000_change_mtu (struct net_device *netdev, int new_mtu)
{
	struct adapter * Adapter = netdev->priv;
	u32 old_mtu = Adapter->RxBufferLen;

	E1000_DBG ("e1000_change_mtu\n");
	if ((new_mtu < MINIMUM_ETHERNET_PACKET_SIZE - ENET_HEADER_SIZE) || (new_mtu > MAX_JUMBO_FRAME_SIZE - ENET_HEADER_SIZE)) {
		E1000_ERR ("Invalid MTU setting\n");
		return -EINVAL;
	}

	if (new_mtu <= MAXIMUM_ETHERNET_PACKET_SIZE - ENET_HEADER_SIZE) {
		/* 2k buffers */
		Adapter->LongPacket = FALSE;
		Adapter->RxBufferLen = E1000_RXBUFFER_2048;

	} else if (Adapter->MacType < MAC_LIVENGOOD) {
		E1000_ERR ("Jumbo Frames not supported on 82542\n");
		return -EINVAL;
	} else if (Adapter->JumboFrames != 1) {
		E1000_ERR ("Jumbo Frames Disabled\n");
		return -EINVAL;

	} else if (new_mtu <= E1000_RXBUFFER_4096 - ENET_HEADER_SIZE - CRC_LENGTH) {
		/* 4k buffers */
		Adapter->LongPacket = TRUE;
		Adapter->RxBufferLen = E1000_RXBUFFER_4096;

	} else if (new_mtu <= E1000_RXBUFFER_8192 - ENET_HEADER_SIZE - CRC_LENGTH) {
		/* 8k buffers */
		Adapter->LongPacket = TRUE;
		Adapter->RxBufferLen = E1000_RXBUFFER_8192;

	} else {
		/* 16k buffers */
		Adapter->LongPacket = TRUE;
		Adapter->RxBufferLen = E1000_RXBUFFER_16384;
	}

	if (old_mtu != Adapter->RxBufferLen && test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {

		/* stop */
		tasklet_disable (&Adapter->rx_fill_tasklet);
		netif_stop_queue (netdev);
		Adapter->e1000_adapter_stopped = FALSE;
		e1000_adapter_stop (Adapter);

		/* clean out old buffers */
		CleanRxRing (Adapter);
		CleanTxRing (Adapter);

		/* reset hardware */
		Adapter->e1000_adapter_stopped = FALSE;
		HardwareInit (Adapter);

		/* go */
		SetupRctl (Adapter);
		ResetRU (Adapter);
		ResetTU (Adapter);
#ifdef IANS
		/* restore VLAN settings */
		if ((IANS_BD_TAGGING_MODE) (ANS_PRIVATE_DATA_FIELD (Adapter)->tag_mode)
		    != IANS_BD_TAGGING_NONE)
			bd_ans_hw_EnableVLAN (Adapter);
#endif
		tasklet_enable (&Adapter->rx_fill_tasklet);
		tasklet_schedule (&Adapter->rx_fill_tasklet);
		EnableInterrupts (Adapter);
		netif_start_queue (netdev);
	}

	netdev->mtu = new_mtu;
	Adapter->MaxFrameSize = new_mtu + ENET_HEADER_SIZE + CRC_LENGTH;

	return 0;
}

/*********************************************************************/
/*! @brief  Change the Ethernet Address of the NIC
 *  @ingroup entry_point
 *  @param  netdev network interface device structure
 *  @param  p pointer to an address structure
 *  @return 0 on success, negative on failure
 *********************************************************************/

int
e1000_set_mac (struct net_device *netdev, void *p)
{
	struct adapter * Adapter = netdev->priv;
	struct pci_dev *pdev = Adapter->pdev;
	struct sockaddr *addr = (struct sockaddr *) p;
	u32 HwLowAddress, HwHighAddress;
	u32 PciCommandWord;
	u32 reg_rctl;

	E1000_DBG ("e1000_set_mac\n");

	reg_rctl = E1000_READ_REG (Rctl);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		if (Adapter->PciCommandWord & PCI_COMMAND_INVALIDATE) {
			PciCommandWord = Adapter->PciCommandWord & ~PCI_COMMAND_INVALIDATE;
			pci_write_config_word (pdev, PCI_COMMAND, PciCommandWord);
		}
		E1000_WRITE_REG (Rctl, reg_rctl | E1000_RCTL_RST);
		mdelay (5);
		if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
			tasklet_disable (&Adapter->rx_fill_tasklet);
			CleanRxRing (Adapter);
		}
	}

	memcpy (netdev->dev_addr, addr->sa_data, netdev->addr_len);
	memcpy (Adapter->CurrentNetAddress, addr->sa_data, netdev->addr_len);

	HwLowAddress = Adapter->CurrentNetAddress[0] | (Adapter->CurrentNetAddress[1] << 8) | (Adapter->CurrentNetAddress[2] << 16) | (Adapter->CurrentNetAddress[3] << 24);

	HwHighAddress = Adapter->CurrentNetAddress[4] | (Adapter->CurrentNetAddress[5] << 8) | E1000_RAH_AV;

	E1000_WRITE_REG (Rar[0].Low, HwLowAddress);
	E1000_WRITE_REG (Rar[0].High, HwHighAddress);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		E1000_WRITE_REG (Rctl, reg_rctl);
		mdelay (5);
		if (Adapter->PciCommandWord & PCI_COMMAND_INVALIDATE) {
			pci_write_config_word (pdev, PCI_COMMAND, Adapter->PciCommandWord);
		}
		if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
			ResetRU (Adapter);
			tasklet_enable (&Adapter->rx_fill_tasklet);
		}
	}

	return 0;
}

/*********************************************************************/
/*! @brief  Update the board statistics counters
 *  @param  Adapter board private structure
 *********************************************************************/

static void
UpdateStatsCounters (struct adapter * Adapter)
{
	unsigned long flags;

	spin_lock_irqsave (&Adapter->stats_lock, flags);

	Adapter->Crcerrs += E1000_READ_REG (Crcerrs);
	Adapter->Symerrs += E1000_READ_REG (Symerrs);
	Adapter->Mpc += E1000_READ_REG (Mpc);
	Adapter->Scc += E1000_READ_REG (Scc);
	Adapter->Ecol += E1000_READ_REG (Ecol);
	Adapter->Mcc += E1000_READ_REG (Mcc);
	Adapter->Latecol += E1000_READ_REG (Latecol);
	Adapter->Colc += E1000_READ_REG (Colc);
	Adapter->Dc += E1000_READ_REG (Dc);
	Adapter->Sec += E1000_READ_REG (Sec);
	Adapter->Rlec += E1000_READ_REG (Rlec);
	Adapter->Xonrxc += E1000_READ_REG (Xonrxc);
	Adapter->Xontxc += E1000_READ_REG (Xontxc);
	Adapter->Xoffrxc += E1000_READ_REG (Xoffrxc);
	Adapter->Xofftxc += E1000_READ_REG (Xofftxc);
	Adapter->Fcruc += E1000_READ_REG (Fcruc);
	Adapter->Prc64 += E1000_READ_REG (Prc64);
	Adapter->Prc127 += E1000_READ_REG (Prc127);
	Adapter->Prc255 += E1000_READ_REG (Prc255);
	Adapter->Prc511 += E1000_READ_REG (Prc511);
	Adapter->Prc1023 += E1000_READ_REG (Prc1023);
	Adapter->Prc1522 += E1000_READ_REG (Prc1522);
	Adapter->Gprc += E1000_READ_REG (Gprc);
	Adapter->Bprc += E1000_READ_REG (Bprc);
	Adapter->Mprc += E1000_READ_REG (Mprc);
	Adapter->Gptc += E1000_READ_REG (Gptc);

	/* for the 64-bit byte counters the low dword must be read first */
	/* both registers clear on the read of the high dword */

/*
    temp64 = E1000_READ_REG(Gorl);
    temp64 += ((uint64_t)E1000_READ_REG(Gorh)) << 32;
    Adapter->Gor += temp64;
*/
	Adapter->Gorcl += E1000_READ_REG (Gorl);
	Adapter->Gorch += E1000_READ_REG (Gorh);
/*
    temp64 = E1000_READ_REG(Gotl);
    temp64 += ((uint64_t)E1000_READ_REG(Goth)) << 32;
    Adapter->Got += temp64;
*/
	Adapter->Gotcl += E1000_READ_REG (Gotl);
	Adapter->Gotch += E1000_READ_REG (Goth);

	Adapter->Rnbc += E1000_READ_REG (Rnbc);
	Adapter->Ruc += E1000_READ_REG (Ruc);
	Adapter->Rfc += E1000_READ_REG (Rfc);
	Adapter->Roc += E1000_READ_REG (Roc);
	Adapter->Rjc += E1000_READ_REG (Rjc);

/*
    temp64 = E1000_READ_REG(Torl);
    temp64 += ((uint64_t)E1000_READ_REG(Torh)) << 32;
    Adapter->Tor += temp64;
*/
	Adapter->Torcl += E1000_READ_REG (Torl);
	Adapter->Torch += E1000_READ_REG (Torh);
/*
    temp64 = E1000_READ_REG(Totl);
    temp64 += ((uint64_t)E1000_READ_REG(Toth)) << 32;
    Adapter->Tot += temp64;
*/
	Adapter->Totcl += E1000_READ_REG (Totl);
	Adapter->Totch += E1000_READ_REG (Toth);

	Adapter->Tpr += E1000_READ_REG (Tpr);
	Adapter->Tpt += E1000_READ_REG (Tpt);
	Adapter->Ptc64 += E1000_READ_REG (Ptc64);
	Adapter->Ptc127 += E1000_READ_REG (Ptc127);
	Adapter->Ptc255 += E1000_READ_REG (Ptc255);
	Adapter->Ptc511 += E1000_READ_REG (Ptc511);
	Adapter->Ptc1023 += E1000_READ_REG (Ptc1023);
	Adapter->Ptc1522 += E1000_READ_REG (Ptc1522);
	Adapter->Mptc += E1000_READ_REG (Mptc);
	Adapter->Bptc += E1000_READ_REG (Bptc);

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		Adapter->Algnerrc += E1000_READ_REG (Algnerrc);
		Adapter->Rxerrc += E1000_READ_REG (Rxerrc);
		Adapter->Tuc += E1000_READ_REG (Tuc);
		Adapter->Tncrs += E1000_READ_REG (Tncrs);
		Adapter->Cexterr += E1000_READ_REG (Cexterr);
		Adapter->Rutec += E1000_READ_REG (Rutec);
		Adapter->Tsctc += E1000_READ_REG (Tsctc);
		Adapter->Tsctfc += E1000_READ_REG (Tsctfc);
	}

	/* Fill out the OS statistics structure */

	Adapter->net_stats.rx_packets = Adapter->Gprc;
	Adapter->net_stats.tx_packets = Adapter->Gptc;
	Adapter->net_stats.rx_bytes = Adapter->Gorcl;
	Adapter->net_stats.tx_bytes = Adapter->Gotcl;
	Adapter->net_stats.multicast = Adapter->Mprc;
	Adapter->net_stats.collisions = Adapter->Colc;

	/* Rx Errors */

	Adapter->net_stats.rx_errors = Adapter->Rxerrc + Adapter->Crcerrs + Adapter->Algnerrc + Adapter->Rlec + Adapter->Rnbc + Adapter->Mpc + Adapter->Cexterr;
	Adapter->net_stats.rx_dropped = Adapter->Rnbc;
	Adapter->net_stats.rx_length_errors = Adapter->Rlec;
	Adapter->net_stats.rx_crc_errors = Adapter->Crcerrs;
	Adapter->net_stats.rx_frame_errors = Adapter->Algnerrc;
	Adapter->net_stats.rx_fifo_errors = Adapter->Mpc;
	Adapter->net_stats.rx_missed_errors = Adapter->Mpc;

	/* Tx Errors */

	Adapter->net_stats.tx_errors = Adapter->Ecol + Adapter->Tuc + Adapter->Latecol;
	Adapter->net_stats.tx_aborted_errors = Adapter->Ecol;
	Adapter->net_stats.tx_fifo_errors = Adapter->Tuc;
	Adapter->net_stats.tx_window_errors = Adapter->Latecol;

	/* Tx Dropped needs to be maintained elsewhere */

	spin_unlock_irqrestore (&Adapter->stats_lock, flags);
	return;
}

/*********************************************************************/
/*! @brief  Mask off interrupt generation on the NIC
 *  @param  Adapter board private structure
 *********************************************************************/

static inline void
DisableInterrupts (struct adapter * Adapter)
{
	E1000_DBG ("DisableInterrupts\n");

	/* Mask off all interrupts */

	E1000_WRITE_REG (Imc, ~0);
	return;
}

/*********************************************************************/
/*! @brief  Enable default interrupt generation settings
 *  @param  Adapter board private structure
 *********************************************************************/

static inline void
EnableInterrupts (struct adapter * Adapter)
{
	E1000_DBG ("EnableInterrupts\n");

	E1000_WRITE_REG (Ims, Adapter->IntMask);
	return;
}

/*********************************************************************/
/*! @brief  Interrupt Handler
 *  @ingroup entry_point
 *  @param  irq interrupt number
 *  @param  data pointer to a network interface device structure
 *  @param  pt_regs CPU registers structure
 *********************************************************************/

void
e1000_intr (int irq, void *data, struct pt_regs *regs)
{
	struct net_device *netdev = (struct net_device *) data;
	struct adapter * Adapter = netdev->priv;
	u32 IcrContents;
	uint ProcessCount = E1000_MAX_INTR;
	u32 CtrlRegValue, TxcwRegValue, RxcwRegValue;

	E1000_DBG ("e1000_intr\n");

	DisableInterrupts (Adapter);

	while (ProcessCount > 0 && (IcrContents = E1000_READ_REG (Icr)) != 0) {

		if (IcrContents & (E1000_ICR_RXSEQ | E1000_ICR_LSC)) {
			Adapter->GetLinkStatus = Adapter->LinkStatusChanged = TRUE;

			/* run the watchdog ASAP */
			mod_timer (&Adapter->timer_id, jiffies);
		}

		if (Adapter->LinkStatusChanged && Adapter->MediaType == MEDIA_TYPE_FIBER) {

			CtrlRegValue = E1000_READ_REG (Ctrl);
			TxcwRegValue = E1000_READ_REG (Txcw);
			RxcwRegValue = E1000_READ_REG (Rxcw);

			if ((CtrlRegValue & E1000_CTRL_SWDPIN1) || ((RxcwRegValue & E1000_RXCW_C) && !(TxcwRegValue & E1000_TXCW_ANE))) {

				E1000_WRITE_REG (Txcw, Adapter->TxcwRegValue);
				E1000_WRITE_REG (Ctrl, CtrlRegValue & ~E1000_CTRL_SLU);
				Adapter->AutoNegFailed = 0;
			}
			Adapter->LinkStatusChanged = FALSE;
		}
		if (Adapter->LinkStatusChanged && Adapter->MediaType == MEDIA_TYPE_COPPER) {

			e1000_check_for_link (Adapter);
			Adapter->LinkStatusChanged = FALSE;
		}

		ProcessReceiveInterrupts (Adapter);
		ProcessTransmitInterrupts (Adapter);
		ProcessCount--;
	}

	EnableInterrupts (Adapter);

	return;
}

/*********************************************************************/
/*! @brief  Reclaim resources after transmit completes
 *  @param  Adapter board private structure
 *********************************************************************/

static void
ProcessTransmitInterrupts (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	int i;

	PE1000_TRANSMIT_DESCRIPTOR TransmitDescriptor;
	struct net_device *netdev = Adapter->netdev;

	E1000_DBG ("ProcessTransmitInterrupts\n");

	i = Adapter->OldestUsedTxDescriptor;
	TransmitDescriptor = &Adapter->TxDescriptors[i];

	while (TransmitDescriptor->Upper.DwordData & cpu_to_le32 (E1000_TXD_STAT_DD)) {

		if (Adapter->tx_skb[i].dma != 0) {
			pci_unmap_single (pdev, Adapter->tx_skb[i].dma, Adapter->tx_skb[i].length, PCI_DMA_TODEVICE);
			Adapter->tx_skb[i].dma = 0;
		}

		if (Adapter->tx_skb[i].skb != NULL) {
			dev_kfree_skb_irq (Adapter->tx_skb[i].skb);
			Adapter->tx_skb[i].skb = NULL;
		}

		atomic_inc (&Adapter->NumTxDescriptorsAvail);
		i = (i + 1) % Adapter->NumTxDescriptors;

		TransmitDescriptor->Upper.DwordData = 0;
		TransmitDescriptor = &Adapter->TxDescriptors[i];

		atomic_set (&Adapter->tx_timeout, 0);
	}

	Adapter->OldestUsedTxDescriptor = i;

	if (netif_queue_stopped (netdev) && (atomic_read (&Adapter->NumTxDescriptorsAvail) > (Adapter->NumTxDescriptors * 3 / 4))) {

#ifdef IANS
		if (Adapter->iANSdata->iANS_status == IANS_COMMUNICATION_UP)
			if (e1000_ans_notify)
				e1000_ans_notify (netdev, IANS_IND_XMIT_QUEUE_READY);
#endif
		netif_wake_queue (netdev);
	}

	return;
}

/*********************************************************************/
/*! @brief  Send received data up the network stack,
 *  @param  Adapter board private structure
 *********************************************************************/

static void
ProcessReceiveInterrupts (struct adapter * Adapter)
{
	struct net_device *netdev = Adapter->netdev;
	struct pci_dev *pdev = Adapter->pdev;
	PE1000_RECEIVE_DESCRIPTOR CurrentDescriptor;
	int i;
	u32 Length;
	struct sk_buff *skb;
	u8 LastByte;
	unsigned long flags;

	E1000_DBG ("ProcessReceiveInterrupts\n");

	i = Adapter->NextRxDescriptorToCheck;
	CurrentDescriptor = &Adapter->RxDescriptors[i];

	while (CurrentDescriptor->ReceiveStatus & E1000_RXD_STAT_DD) {
		pci_unmap_single (pdev, Adapter->rx_skb[i].dma, Adapter->rx_skb[i].length, PCI_DMA_FROMDEVICE);

		skb = Adapter->rx_skb[i].skb;
		Length = le16_to_cpu (CurrentDescriptor->Length);

		if (!(CurrentDescriptor->ReceiveStatus & E1000_RXD_STAT_EOP)) {

			/* All receives must fit into a single buffer */

			E1000_DBG ("Receive packet consumed multiple buffers\n");

			dev_kfree_skb_irq (skb);
			memset (CurrentDescriptor, 0, 16);
			mb ();
			Adapter->rx_skb[i].skb = NULL;

			atomic_inc (&Adapter->NumRxDescriptorsEmpty);

			i = (i + 1) % Adapter->NumRxDescriptors;

			CurrentDescriptor = &Adapter->RxDescriptors[i];
			continue;
		}

		if (CurrentDescriptor->Errors & E1000_RXD_ERR_FRAME_ERR_MASK) {

			LastByte = *(skb->data + Length - 1);

			if (TBI_ACCEPT (CurrentDescriptor->Errors, LastByte, Length)) {
				spin_lock_irqsave (&Adapter->stats_lock, flags);
				e1000_adjust_tbi_accepted_stats (Adapter, Length, skb->data);
				spin_unlock_irqrestore (&Adapter->stats_lock, flags);
			} else {

				E1000_DBG ("Receive Errors Reported by Hardware\n");

				dev_kfree_skb_irq (skb);
				memset (CurrentDescriptor, 0, 16);
				mb ();
				Adapter->rx_skb[i].skb = NULL;

				atomic_inc (&Adapter->NumRxDescriptorsEmpty);
				i = (i + 1) % Adapter->NumRxDescriptors;

				CurrentDescriptor = &Adapter->RxDescriptors[i];
				continue;
			}
		}

		/* Good Receive */
		skb_put (skb, Length - CRC_LENGTH);

		/* Adjust socket buffer accounting to only cover the ethernet frame
		 * Not what the stack intends, but there exist TCP problems that
		 * break NFS for network interfaces that need 2k receive buffers */
		skb->truesize = skb->len;

		/* Receive Checksum Offload */
		RxChecksum (Adapter, CurrentDescriptor, skb);

#ifdef IANS
		if (Adapter->iANSdata->iANS_status == IANS_COMMUNICATION_UP) {
			if (bd_ans_os_Receive (Adapter, CurrentDescriptor, skb) == BD_ANS_FAILURE)
				dev_kfree_skb_irq (skb);
			else
				netif_rx (skb);
		} else {
			skb->protocol = eth_type_trans (skb, netdev);
			netif_rx (skb);
		}
#else
		skb->protocol = eth_type_trans (skb, netdev);
		netif_rx (skb);
#endif
		memset (CurrentDescriptor, 0, 16);
		mb ();
		Adapter->rx_skb[i].skb = NULL;

		atomic_inc (&Adapter->NumRxDescriptorsEmpty);

		i = (i + 1) % Adapter->NumRxDescriptors;

		CurrentDescriptor = &Adapter->RxDescriptors[i];
	}

	/* if the Rx ring is less than 3/4 full, allocate more sk_buffs */

	if (atomic_read (&Adapter->NumRxDescriptorsEmpty) > (Adapter->NumRxDescriptors / 4)) {
		tasklet_schedule (&Adapter->rx_fill_tasklet);
	}
	Adapter->NextRxDescriptorToCheck = i;

	return;
}

/*********************************************************************/
/*! @brief  Replace used receive buffers
 *  @param  data address of board private structure
 *********************************************************************/
static void
ReceiveBufferFill (uintptr_t data)
{
	struct adapter * Adapter = (struct adapter *) data;
	struct net_device *netdev = Adapter->netdev;
	struct pci_dev *pdev = Adapter->pdev;
	PE1000_RECEIVE_DESCRIPTOR CurrentDescriptor;
	struct sk_buff *skb;
	int i;

	E1000_DBG ("ReceiveBufferFill\n");

	/* kernel 2.4.7 seems to be broken with respect to tasklet locking */
	if (!spin_trylock (&Adapter->rx_fill_lock))
		return;

	if (!test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {
		spin_unlock (&Adapter->rx_fill_lock);
		return;
	}

	i = Adapter->NextRxDescriptorToFill;

	while (Adapter->rx_skb[i].skb == NULL) {
		CurrentDescriptor = &Adapter->RxDescriptors[i];

#ifdef IANS
		skb = alloc_skb (Adapter->RxBufferLen + E1000_ROUNDUP2 (BD_ANS_INFO_SIZE, 16) + 2, GFP_ATOMIC);
		if (skb == NULL) {
			/* keep re-trying when failures occur */
			/* does this need some delay ? */
			tasklet_schedule (&Adapter->rx_fill_tasklet);
			break;
		}
		/* Make buffer alignment 2 beyond a 16 byte boundary
		 * this will result in a 16 byte aligned IP header after
		 * the 14 byte MAC header is removed */
		skb_reserve (skb, E1000_ROUNDUP2 (BD_ANS_INFO_SIZE, 16) + 2);
#else
		skb = alloc_skb (Adapter->RxBufferLen + 2, GFP_ATOMIC);
		if (skb == NULL) {
			/* keep re-trying when failures occur */
			tasklet_schedule (&Adapter->rx_fill_tasklet);
			break;
		}
		/* Make buffer alignment 2 beyond a 16 byte boundary
		 * this will result in a 16 byte aligned IP header after
		 * the 14 byte MAC header is removed */
		skb_reserve (skb, 2);
#endif
		skb->dev = netdev;

		Adapter->rx_skb[i].skb = skb;
		Adapter->rx_skb[i].length = Adapter->RxBufferLen;
		Adapter->rx_skb[i].dma = pci_map_single (pdev, skb->data, Adapter->RxBufferLen, PCI_DMA_FROMDEVICE);

		e1000_load_addr (&(CurrentDescriptor->BufferAddress), Adapter->rx_skb[i].dma);

		/* move tail */
		E1000_WRITE_REG (Rdt0, i);

		atomic_dec (&Adapter->NumRxDescriptorsEmpty);

		i = (i + 1) % Adapter->NumRxDescriptors;
	}

	Adapter->NextRxDescriptorToFill = i;

	return;
}

static int netdev_ethtool_ioctl(struct net_device *dev, void *useraddr)
{
	struct adapter *ap = dev->priv;
	u32 ethcmd;
		
	if (copy_from_user(&ethcmd, useraddr, sizeof(ethcmd)))
		return -EFAULT;

        switch (ethcmd) {
        case ETHTOOL_GDRVINFO: {
		struct ethtool_drvinfo info = {ETHTOOL_GDRVINFO};
		strcpy(info.driver,  e1000_driver_name);
		strcpy(info.version, e1000_driver_version);
		strcpy(info.bus_info, ap->pdev->slot_name);
		if (copy_to_user(useraddr, &info, sizeof(info)))
			return -EFAULT;
		return 0;
	}

        }
	
	return -EOPNOTSUPP;
}



/*********************************************************************/
/*! @ingroup entry_point
 *  @param  netdev
 *  @param  ifreq
 *  @param  cmd
 *  @return
 *********************************************************************/

int
e1000_ioctl (struct net_device *netdev, struct ifreq *ifr, int cmd)
{
#ifdef IANS
	IANS_BD_PARAM_HEADER *header;
#endif

	E1000_DBG ("e1000_do_ioctl\n");

	switch (cmd) {

#ifdef IANS
	case IANS_BASE_SIOC:
		header = (IANS_BD_PARAM_HEADER *) ifr->ifr_data;
		if ((header->Opcode != IANS_OP_EXT_GET_STATUS) && (!capable (CAP_NET_ADMIN)))
			return -EPERM;
		return bd_ans_os_Ioctl (netdev, ifr, cmd);
		break;
#endif

#ifdef IDIAG
	case IDIAG_PRO_BASE_SIOC:
		if (!capable (CAP_NET_ADMIN))
			return -EPERM;

#ifdef DIAG_DEBUG
		printk ("Entering diagnostics\n");
#endif
		e1000_diag_ioctl (netdev, ifr);
		break;
#endif
	case SIOCETHTOOL:
		return netdev_ethtool_ioctl(netdev, (void *) ifr->ifr_data);

	default:
		return -EOPNOTSUPP;
	}

	return 0;
}

/*********************************************************************/
/*! @brief  Receive Checksum Offload for 82543
 *  @param  Adapter board private structure
 *  @param  RxDescriptor receive descriptor
 *  @param  sk_buff socket buffer with received data
 *********************************************************************/

static inline void
RxChecksum (struct adapter * Adapter, PE1000_RECEIVE_DESCRIPTOR RxDescriptor, struct sk_buff *skb)
{
	/* 82543 or newer only */
	if ((Adapter->MacType < MAC_LIVENGOOD) ||
	    /* Ignore Checksum bit is set */
	    (RxDescriptor->ReceiveStatus & E1000_RXD_STAT_IXSM) ||
	    /* TCP Checksum has not been calculated */
	    (!(RxDescriptor->ReceiveStatus & E1000_RXD_STAT_TCPCS))) {

		skb->ip_summed = CHECKSUM_NONE;
		return;
	}

	/* At this point we know the hardware did the TCP checksum */
	/* now look at the TCP checksum error bit */
	if (RxDescriptor->Errors & E1000_RXD_ERR_TCPE) {
		/* let the stack verify checksum errors */
		skb->ip_summed = CHECKSUM_NONE;
		Adapter->XsumRXError++;
	} else {
		/* TCP checksum is good */
		skb->ip_summed = CHECKSUM_UNNECESSARY;
		Adapter->XsumRXGood++;
	}

	return;
}

void
e1000_hibernate_adapter (struct net_device *netdev)
{
	u32 icr;
	struct adapter * Adapter = netdev->priv;

	DisableInterrupts (Adapter);
	netif_stop_queue (netdev);
	Adapter->e1000_adapter_stopped = FALSE;
	e1000_adapter_stop (Adapter);

	if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {

		/* Disable tasklet only when interface is opened. */
		tasklet_disable (&Adapter->rx_fill_tasklet);

		/* clean out old buffers */
		CleanRxRing (Adapter);
		CleanTxRing (Adapter);

		/* Delete watchdog timer */
		del_timer (&Adapter->timer_id);

		/* Unhook irq */
		DisableInterrupts (Adapter);
		icr = E1000_READ_REG (Icr);
		free_irq (netdev->irq, netdev);
	}
}

void
e1000_wakeup_adapter (struct net_device *netdev)
{
	u32 icr;
	struct adapter * Adapter = netdev->priv;
	Adapter->e1000_adapter_stopped = FALSE;
	e1000_adapter_stop (Adapter);
	Adapter->e1000_adapter_stopped = FALSE;

	if (!e1000_initialize_hardware (Adapter))
		printk ("Hardware Init Failed at wakeup\n");

	if (test_bit (E1000_BOARD_OPEN, &Adapter->flags)) {

		/* Setup Rctl */
		SetupRctl (Adapter);
		ResetRU (Adapter);
		ReceiveBufferFill ((uintptr_t) Adapter);
		e1000_set_multi (netdev);
		ResetTU (Adapter);

#ifdef IANS
		if ((IANS_BD_TAGGING_MODE) (ANS_PRIVATE_DATA_FIELD (Adapter)->tag_mode)
		    != IANS_BD_TAGGING_NONE)
			bd_ans_hw_EnableVLAN (Adapter);
#endif

		/* Set the watchdog timer for 2 seconds */
		init_timer (&Adapter->timer_id);
		Adapter->timer_id.function = &e1000_watchdog;
		Adapter->timer_id.data = (unsigned long) netdev;
		mod_timer (&Adapter->timer_id, (jiffies + 2 * HZ));

		tasklet_enable (&Adapter->rx_fill_tasklet);

		/* Hook irq */
		DisableInterrupts (Adapter);
		icr = E1000_READ_REG (Icr);
		if (request_irq (netdev->irq, &e1000_intr, SA_SHIRQ, e1000_driver_name, netdev) != 0)
			printk (KERN_ERR "e1000: Unable to hook irq.\n");

		EnableInterrupts (Adapter);
		netif_start_queue (netdev);
	}
}

#ifdef IDIAG
int
e1000_xmit_lbtest_frame (struct sk_buff *skb, struct adapter * Adapter)
{
	/*struct adapter * Adapter = netdev->priv; */
	struct pci_dev *pdev = Adapter->pdev;
	PE1000_TRANSMIT_DESCRIPTOR CurrentTxDescriptor;
	int i;
#ifdef DIAG_DEBUG
	printk ("%x \n", *(skb->data + 3));
#endif
	i = Adapter->NextAvailTxDescriptor;
	CurrentTxDescriptor = &Adapter->TxDescriptors[i];

	Adapter->tx_skb[i].skb = skb;
	Adapter->tx_skb[i].length = skb->len;
	Adapter->tx_skb[i].dma = pci_map_single (pdev, skb->data, skb->len, PCI_DMA_TODEVICE);

	e1000_load_addr (&(CurrentTxDescriptor->BufferAddress), Adapter->tx_skb[i].dma);

	CurrentTxDescriptor->Lower.DwordData = cpu_to_le32 (skb->len);

	/* zero out the status field in the descriptor */

	CurrentTxDescriptor->Upper.DwordData = 0;

	CurrentTxDescriptor->Lower.DwordData |= E1000_TXD_CMD_EOP;
	CurrentTxDescriptor->Lower.DwordData |= E1000_TXD_CMD_IFCS;
	CurrentTxDescriptor->Lower.DwordData |= E1000_TXD_CMD_IDE;

	if (Adapter->ReportTxEarly == 1)
		CurrentTxDescriptor->Lower.DwordData |= E1000_TXD_CMD_RS;
	else
		CurrentTxDescriptor->Lower.DwordData |= E1000_TXD_CMD_RPS;

	/* Move the HW Tx Tail Pointer */

	Adapter->NextAvailTxDescriptor++;
	Adapter->NextAvailTxDescriptor %= Adapter->NumTxDescriptors;

#ifdef DIAG_DEBUG
	printk ("%x \n", E1000_READ_REG (Tdh));
#endif

	E1000_WRITE_REG (Tdt, Adapter->NextAvailTxDescriptor);
	mdelay (10);

#ifdef DIAG_DEBUG
	printk ("%x \n", E1000_READ_REG (Tdh));
#endif

	atomic_dec (&Adapter->NumTxDescriptorsAvail);

	if (atomic_read (&Adapter->NumTxDescriptorsAvail) <= 1) {

		/* this driver never actually drops transmits,
		 * so use tx_dropped count to indicate the number of times
		 * netif_stop_queue is called due to no available descriptors */

		Adapter->net_stats.tx_dropped++;
		return (FALSE);
	}

	return (TRUE);
}

int
e1000_rcv_lbtest_frame (struct adapter * Adapter)
{
	struct pci_dev *pdev = Adapter->pdev;
	PE1000_RECEIVE_DESCRIPTOR CurrentDescriptor;
	int i, j = 0, rcved_pkt = 0;
	u32 Length;
	struct sk_buff *skb;

	mdelay (100);
	i = Adapter->NextRxDescriptorToCheck;
	CurrentDescriptor = &Adapter->RxDescriptors[i];

	while (CurrentDescriptor->ReceiveStatus & E1000_RXD_STAT_DD) {
		Length = le16_to_cpu (CurrentDescriptor->Length) - CRC_LENGTH;
		skb = Adapter->rx_skb[i].skb;

		/* Snoop the packet for pattern */

		if (*(skb->data + 3) == 0xFF) {
			if (*(skb->data + 600) == 0xBA) {
#ifdef DIAG_DEBUG
				printk ("Received packet ");
#endif
				rcved_pkt = TRUE;
			}
		}
		pci_unmap_single (pdev, Adapter->rx_skb[i].dma, Adapter->rx_skb[i].length, PCI_DMA_FROMDEVICE);

		dev_kfree_skb_irq (skb);
		Adapter->rx_skb[i].skb = NULL;

		CurrentDescriptor->ReceiveStatus = 0;
		atomic_inc (&Adapter->NumRxDescriptorsEmpty);

		i++;
		i %= Adapter->NumRxDescriptors;
		CurrentDescriptor = &Adapter->RxDescriptors[i];

		if (rcved_pkt)
			break;

		/* waited enough */
		if (j++ > 100)
			return FALSE;

		mdelay (5);

	}

	Adapter->NextRxDescriptorToCheck = i;

	if (rcved_pkt)
		return TRUE;
	else
		return FALSE;
}

void
e1000_selective_wakeup_adapter (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;
	HardwareInit (Adapter);
	SetupRctl (Adapter);
	ResetRU (Adapter);
	ReceiveBufferFill ((uintptr_t) Adapter);
	ResetTU (Adapter);
}

void
e1000_selective_hibernate_adapter (struct net_device *netdev)
{
	struct adapter * Adapter = netdev->priv;
	/* clean out old buffers */
	CleanRxRing (Adapter);
	CleanTxRing (Adapter);
}
#endif
/* e1000_main.c */
