/*****************************************************************************
 *****************************************************************************

 Copyright (c) 1999 - 2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, 
     this list of conditions and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice,
     this list of conditions and the following disclaimer in the documentation 
     and/or other materials provided with the distribution.

  3. Neither the name of Intel Corporation nor the names of its contributors 
     may be used to endorse or promote products derived from this software 
     without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
 *****************************************************************************/

/* Workfile: fxhw.c  */
/* Revision: 42  */
/* Date: 8/03/01 10:39a  */

#include "e1000.h"

#define IN
#define OUT

void e1000_adapter_stop (struct adapter * Adapter);

unsigned char e1000_initialize_hardware (struct adapter * Adapter);

void e1000_init_rx_addresses (struct adapter * Adapter);

void e1000_multicast_address_list_update (struct adapter * Adapter, unsigned char * MulticastAddressList, u32 MulticastAddressCount, u32 Padding);

u32 e1000_hash_multicast_address (struct adapter * Adapter, unsigned char * MulticastAddress);

void e1000_mta_set (struct adapter * Adapter, u32 HashValue);

void e1000_rar_set (struct adapter * Adapter, unsigned char * MulticastAddress, u32 RarIndex);

void e1000_write_vfta (struct adapter * Adapter, u32 Offset, u32 Value);

void e1000_clear_vfta (struct adapter * Adapter);

unsigned char e1000_setup_flow_control_andLink (struct adapter * Adapter);

unsigned char e1000_setup_pcs_link (struct adapter * Adapter, u32 DeviceControlReg);

void e1000_config_flow_control_afterLinkUp (struct adapter * Adapter);

void e1000_force_mac_flow_controlSetting (struct adapter * Adapter);

void e1000_check_for_link (struct adapter * Adapter);

void e1000_clear_hw_stats_counters (struct adapter * Adapter);

void e1000_get_speed_and_duplex (struct adapter * Adapter, u16 * Speed, u16 * Duplex);

u16 e1000_read_eeprom_word (struct adapter * Adapter, u16 Reg);

static void ShiftOutBits (struct adapter * Adapter, u16 Data, u16 Count);

static void RaiseClock (struct adapter * Adapter, u32 * EecdRegValue);

static void LowerClock (struct adapter * Adapter, u32 * EecdRegValue);

static unsigned short ShiftInBits (struct adapter * Adapter);

static void EepromCleanup (struct adapter * Adapter);

unsigned char e1000_validate_eeprom_checksum (struct adapter * Adapter);

void e1000_update_eeprom_checksum (struct adapter * Adapter);

unsigned char e1000_write_eeprom_word (struct adapter * Adapter, unsigned short reg, unsigned short data);

static u16 WaitEepromCommandDone (struct adapter * Adapter);

static void StandBy (struct adapter * Adapter);

unsigned char e1000_read_part_number (struct adapter * Adapter, u32 * PartNumber);

void e1000_id_led_on (struct adapter * Adapter);

void e1000_id_led_off (struct adapter * Adapter);

void e1000_set_id_led_forPCIX (struct adapter * Adapter);

unsigned char e1000_is_low_profile (struct adapter * Adapter);

void
e1000_adapter_stop (struct adapter * Adapter)
{

	u32 IcrContents;

	u16 PciCommandWord;

	DEBUGFUNC ("e1000_adapter_stop")

	    if (Adapter->e1000_adapter_stopped) {
		DEBUGOUT ("Exiting because the adapter is already stopped!!!\n");
		return;
	}

	Adapter->e1000_adapter_stopped = TRUE;

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		if (Adapter->PciCommandWord & CMD_MEM_WRT_INVALIDATE) {
			DEBUGOUT ("Disabling MWI on rev 2.0 Wiseman silicon\n");

			PciCommandWord = Adapter->PciCommandWord & ~CMD_MEM_WRT_INVALIDATE;

			WritePciConfigWord (PCI_COMMAND_REGISTER, &PciCommandWord);
		}
	}

	DEBUGOUT ("Masking off all interrupts\n");
	E1000_WRITE_REG (Imc, 0xffffffff);

	E1000_WRITE_REG (Rctl, 0);
	E1000_WRITE_REG (Tctl, 0);

	Adapter->TbiCompatibilityOn = FALSE;

	mdelay (10);

	DEBUGOUT ("Issuing a global reset to MAC\n");
	E1000_WRITE_REG (Ctrl, E1000_CTRL_RST);

	mdelay (10);

	DEBUGOUT ("Masking off all interrupts\n");
	E1000_WRITE_REG (Imc, 0xffffffff);

	IcrContents = E1000_READ_REG (Icr);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		if (Adapter->PciCommandWord & CMD_MEM_WRT_INVALIDATE) {
			WritePciConfigWord (PCI_COMMAND_REGISTER, &Adapter->PciCommandWord);
		}
	}

}

unsigned char
e1000_initialize_hardware (struct adapter * Adapter)
{
	unsigned int i;
	u16 PciCommandWord;
	unsigned char Status;
	u32 RegisterValue;

	DEBUGFUNC ("e1000_initialize_hardware");

	if (Adapter->MacType != MAC_LIVENGOOD) {

		Adapter->TbiCompatibilityEnable = FALSE;
	}

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		RegisterValue = E1000_READ_REG (Status);
		if (RegisterValue & E1000_STATUS_TBIMODE) {
			Adapter->MediaType = MEDIA_TYPE_FIBER;

			Adapter->TbiCompatibilityEnable = FALSE;
		} else {
			Adapter->MediaType = MEDIA_TYPE_COPPER;
		}
	} else {

		Adapter->MediaType = MEDIA_TYPE_FIBER;
	}

	DEBUGOUT ("Initializing the IEEE VLAN\n");
	E1000_WRITE_REG (Vet, 0);

	e1000_clear_vfta (Adapter);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {

		if (Adapter->PciCommandWord & CMD_MEM_WRT_INVALIDATE) {
			DEBUGOUT ("Disabling MWI on rev 2.0 Wiseman silicon\n");

			PciCommandWord = Adapter->PciCommandWord & ~CMD_MEM_WRT_INVALIDATE;

			WritePciConfigWord (PCI_COMMAND_REGISTER, &PciCommandWord);
		}

		E1000_WRITE_REG (Rctl, E1000_RCTL_RST);

		mdelay (5);
	}

	e1000_init_rx_addresses (Adapter);

	if (Adapter->MacType == MAC_WISEMAN_2_0) {
		E1000_WRITE_REG (Rctl, 0);

		mdelay (1);

		if (Adapter->PciCommandWord & CMD_MEM_WRT_INVALIDATE) {
			WritePciConfigWord (PCI_COMMAND_REGISTER, &Adapter->PciCommandWord);
		}

	}

	DEBUGOUT ("Zeroing the MTA\n");
	for (i = 0; i < E1000_MC_TBL_SIZE; i++) {
		E1000_WRITE_REG (Mta[i], 0);
	}

	Status = e1000_setup_flow_control_andLink (Adapter);

	e1000_clear_hw_stats_counters (Adapter);

	return (Status);
}

void
e1000_init_rx_addresses (struct adapter * Adapter)
{
	unsigned int i;
	u32 HwLowAddress;
	u32 HwHighAddress;

	DEBUGFUNC ("e1000_init_rx_addresses")

	    DEBUGOUT ("Programming IA into RAR[0]\n");
	HwLowAddress = (Adapter->CurrentNetAddress[0] | (Adapter->CurrentNetAddress[1] << 8) | (Adapter->CurrentNetAddress[2] << 16) | (Adapter->CurrentNetAddress[3] << 24));

	HwHighAddress = (Adapter->CurrentNetAddress[4] | (Adapter->CurrentNetAddress[5] << 8) | E1000_RAH_AV);

	E1000_WRITE_REG (Rar[0].Low, HwLowAddress);
	E1000_WRITE_REG (Rar[0].High, HwHighAddress);

	DEBUGOUT ("Clearing RAR[1-15]\n");
	for (i = 1; i < E1000_RAR_ENTRIES; i++) {
		E1000_WRITE_REG (Rar[i].Low, 0);
		E1000_WRITE_REG (Rar[i].High, 0);
	}

	return;
}

void
e1000_multicast_address_list_update (struct adapter * Adapter, unsigned char * MulticastAddressList, u32 MulticastAddressCount, u32 Padding)
{

	u32 HashValue;
	u32 i;
	u32 RarUsedCount = 1;
	u32 RctlValue;

	DEBUGFUNC ("e1000_multicast_address_list_update");

	Adapter->NumberOfMcAddresses = MulticastAddressCount;

	DEBUGOUT (" Clearing RAR[1-15]\n");
	for (i = RarUsedCount; i < E1000_RAR_ENTRIES; i++) {
		E1000_WRITE_REG (Rar[i].Low, 0);
		E1000_WRITE_REG (Rar[i].High, 0);
	}

	DEBUGOUT (" Clearing MTA\n");
	for (i = 0; i < E1000_NUM_MTA_REGISTERS; i++) {
		E1000_WRITE_REG (Mta[i], 0);
	}

	for (i = 0; i < MulticastAddressCount; i++) {
		DEBUGOUT (" Adding the multicast addresses:\n");
		DEBUGOUT7 (" MC Addr #%d =%.2X %.2X %.2X %.2X %.2X %.2X\n", i,
			   MulticastAddressList[i *
						(ETH_LENGTH_OF_ADDRESS + Padding)],
			   MulticastAddressList[i *
						(ETH_LENGTH_OF_ADDRESS + Padding) +
						1],
			   MulticastAddressList[i *
						(ETH_LENGTH_OF_ADDRESS + Padding) +
						2],
			   MulticastAddressList[i *
						(ETH_LENGTH_OF_ADDRESS + Padding) +
						3],
			   MulticastAddressList[i * (ETH_LENGTH_OF_ADDRESS + Padding) + 4], MulticastAddressList[i * (ETH_LENGTH_OF_ADDRESS + Padding) + 5]);

		HashValue = e1000_hash_multicast_address (Adapter, MulticastAddressList + (i * (ETH_LENGTH_OF_ADDRESS + Padding)));

		DEBUGOUT1 (" Hash value = 0x%03X\n", HashValue);

		if (RarUsedCount < E1000_RAR_ENTRIES) {
			e1000_rar_set (Adapter, MulticastAddressList + (i * (ETH_LENGTH_OF_ADDRESS + Padding)), RarUsedCount);
			RarUsedCount++;
		} else {
			e1000_mta_set (Adapter, HashValue);
		}

	}

	DEBUGOUT ("MC Update Complete\n");
}

u32
e1000_hash_multicast_address (struct adapter * Adapter, unsigned char * MulticastAddress)
{
	u32 HashValue = 0;

	switch (Adapter->MulticastFilterType) {

	case 0:

		HashValue = ((MulticastAddress[4] >> 4) | (((u16) MulticastAddress[5]) << 4));

		break;

	case 1:
		HashValue = ((MulticastAddress[4] >> 3) | (((u16) MulticastAddress[5]) << 5));

		break;

	case 2:
		HashValue = ((MulticastAddress[4] >> 2) | (((u16) MulticastAddress[5]) << 6));

		break;

	case 3:
		HashValue = ((MulticastAddress[4]) | (((u16) MulticastAddress[5]) << 8));

		break;
	}

	HashValue &= 0xFFF;
	return (HashValue);

}

void
e1000_mta_set (struct adapter * Adapter, u32 HashValue)
{
	u32 HashBit, HashReg;
	u32 MtaRegisterValue;
	u32 Temp;

	HashReg = (HashValue >> 5) & 0x7F;
	HashBit = HashValue & 0x1F;

	MtaRegisterValue = E1000_READ_REG (Mta[(HashReg)]);

	MtaRegisterValue |= (1 << HashBit);

	if ((Adapter->MacType == MAC_CORDOVA) && ((HashReg & 0x0E) == 1)) {
		Temp = E1000_READ_REG (Mta[HashReg - 1]);
		E1000_WRITE_REG (Mta[HashReg], HashValue);
		E1000_WRITE_REG (Mta[HashReg - 1], Temp);
	} else {
		E1000_WRITE_REG (Mta[HashReg], MtaRegisterValue);
	}

}

void
e1000_rar_set (struct adapter * Adapter, unsigned char * MulticastAddress, u32 RarIndex)
{
	u32 RarLow, RarHigh;

	RarLow = ((u32) MulticastAddress[0] | ((u32) MulticastAddress[1] << 8) | ((u32) MulticastAddress[2] << 16) | ((u32) MulticastAddress[3] << 24));

	RarHigh = ((u32) MulticastAddress[4] | ((u32) MulticastAddress[5] << 8) | E1000_RAH_AV);

	E1000_WRITE_REG (Rar[RarIndex].Low, RarLow);
	E1000_WRITE_REG (Rar[RarIndex].High, RarHigh);
}

void
e1000_write_vfta (struct adapter * Adapter, u32 Offset, u32 Value)
{
	u32 Temp;

	if ((Adapter->MacType == MAC_CORDOVA) && ((Offset & 0x0E) == 1)) {
		Temp = E1000_READ_REG (Vfta[Offset - 1]);
		E1000_WRITE_REG (Vfta[Offset], Value);
		E1000_WRITE_REG (Vfta[Offset - 1], Temp);
	} else {
		E1000_WRITE_REG (Vfta[Offset], Value);
	}
}

void
e1000_clear_vfta (struct adapter * Adapter)
{
	u32 Offset;

	for (Offset = 0; Offset < E1000_VLAN_FILTER_TBL_SIZE; Offset++)
		E1000_WRITE_REG (Vfta[Offset], 0);

}

unsigned char
e1000_setup_flow_control_andLink (struct adapter * Adapter)
{
	u32 TempEepromWord;
	u32 DeviceControlReg;
	u32 ExtDevControlReg;
	unsigned char Status = TRUE;

	DEBUGFUNC ("e1000_setup_flow_control_andLink")

	    TempEepromWord = e1000_read_eeprom_word (Adapter, EEPROM_INIT_CONTROL1_REG);

	DeviceControlReg = (((TempEepromWord & EEPROM_WORD0A_SWDPIO) << SWDPIO_SHIFT) | ((TempEepromWord & EEPROM_WORD0A_ILOS) << ILOS_SHIFT));

	if (Adapter->DmaFairness)
		DeviceControlReg |= E1000_CTRL_PRIOR;

	TempEepromWord = e1000_read_eeprom_word (Adapter, EEPROM_INIT_CONTROL2_REG);

	if (Adapter->FlowControl > FLOW_CONTROL_FULL) {
		if ((TempEepromWord & EEPROM_WORD0F_PAUSE_MASK) == 0)
			Adapter->FlowControl = FLOW_CONTROL_NONE;
		else if ((TempEepromWord & EEPROM_WORD0F_PAUSE_MASK) == EEPROM_WORD0F_ASM_DIR)
			Adapter->FlowControl = FLOW_CONTROL_TRANSMIT_PAUSE;
		else
			Adapter->FlowControl = FLOW_CONTROL_FULL;
	}

	Adapter->OriginalFlowControl = Adapter->FlowControl;

	if (Adapter->MacType == MAC_WISEMAN_2_0)
		Adapter->FlowControl &= (~FLOW_CONTROL_TRANSMIT_PAUSE);

	if ((Adapter->MacType < MAC_LIVENGOOD)
	    && (Adapter->ReportTxEarly == 1))
		Adapter->FlowControl &= (~FLOW_CONTROL_RECEIVE_PAUSE);

	DEBUGOUT1 ("After fix-ups FlowControl is now = %x\n", Adapter->FlowControl);

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		ExtDevControlReg = ((TempEepromWord & EEPROM_WORD0F_SWPDIO_EXT)
				    << SWDPIO__EXT_SHIFT);
		E1000_WRITE_REG (Exct, ExtDevControlReg);
	}

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		if (Adapter->MediaType == MEDIA_TYPE_FIBER) {
			Status = e1000_setup_pcs_link (Adapter, DeviceControlReg);

		} else {

			Status = e1000_phy_setup (Adapter, DeviceControlReg);
		}
	} else {
		Status = e1000_setup_pcs_link (Adapter, DeviceControlReg);
	}

	DEBUGOUT ("Initializing the Flow Control address, type and timer regs\n");

	E1000_WRITE_REG (Fcal, FLOW_CONTROL_ADDRESS_LOW);
	E1000_WRITE_REG (Fcah, FLOW_CONTROL_ADDRESS_HIGH);
	E1000_WRITE_REG (Fct, FLOW_CONTROL_TYPE);
	E1000_WRITE_REG (Fcttv, Adapter->FlowControlPauseTime);

	if (!(Adapter->FlowControl & FLOW_CONTROL_TRANSMIT_PAUSE)) {
		E1000_WRITE_REG (Fcrtl, 0);
		E1000_WRITE_REG (Fcrth, 0);
	} else {

		if (Adapter->FlowControlSendXon) {
			E1000_WRITE_REG (Fcrtl, (Adapter->FlowControlLowWatermark | E1000_FCRTL_XONE));
			E1000_WRITE_REG (Fcrth, Adapter->FlowControlHighWatermark);
		} else {
			E1000_WRITE_REG (Fcrtl, Adapter->FlowControlLowWatermark);
			E1000_WRITE_REG (Fcrth, Adapter->FlowControlHighWatermark);
		}
	}

	return (Status);
}

unsigned char
e1000_setup_pcs_link (struct adapter * Adapter, u32 DeviceControlReg)
{
	u32 i;
	u32 StatusContents;
	u32 TctlReg;
	u32 TransmitConfigWord;
	u32 Shift32;

	DEBUGFUNC ("e1000_setup_pcs_link")

	    TctlReg = E1000_READ_REG (Tctl);
	Shift32 = E1000_FDX_COLLISION_DISTANCE;
	Shift32 <<= E1000_COLD_SHIFT;
	TctlReg |= Shift32;
	E1000_WRITE_REG (Tctl, TctlReg);

	switch (Adapter->FlowControl) {
	case FLOW_CONTROL_NONE:

		TransmitConfigWord = (E1000_TXCW_ANE | E1000_TXCW_FD);

		break;

	case FLOW_CONTROL_RECEIVE_PAUSE:

		TransmitConfigWord = (E1000_TXCW_ANE | E1000_TXCW_FD | E1000_TXCW_PAUSE_MASK);

		break;

	case FLOW_CONTROL_TRANSMIT_PAUSE:

		TransmitConfigWord = (E1000_TXCW_ANE | E1000_TXCW_FD | E1000_TXCW_ASM_DIR);

		break;

	case FLOW_CONTROL_FULL:

		TransmitConfigWord = (E1000_TXCW_ANE | E1000_TXCW_FD | E1000_TXCW_PAUSE_MASK);

		break;

	default:

		DEBUGOUT ("Flow control param set incorrectly\n");
		ASSERT (0);
		break;
	}

	DEBUGOUT ("Auto-negotiation enabled\n");

	E1000_WRITE_REG (Txcw, TransmitConfigWord);
	E1000_WRITE_REG (Ctrl, DeviceControlReg);

	Adapter->TxcwRegValue = TransmitConfigWord;
	mdelay (1);

	if (!(E1000_READ_REG (Ctrl) & E1000_CTRL_SWDPIN1)) {

		DEBUGOUT ("Looking for Link\n");
		for (i = 0; i < (LINK_UP_TIMEOUT / 10); i++) {
			mdelay (10);

			StatusContents = E1000_READ_REG (Status);
			if (StatusContents & E1000_STATUS_LU)
				break;
		}

		if (i == (LINK_UP_TIMEOUT / 10)) {

			DEBUGOUT ("Never got a valid link from auto-neg!!!\n");

			Adapter->AutoNegFailed = 1;
			e1000_check_for_link (Adapter);
			Adapter->AutoNegFailed = 0;
		} else {
			Adapter->AutoNegFailed = 0;
			DEBUGOUT ("Valid Link Found\n");
		}
	} else {
		DEBUGOUT ("No Signal Detected\n");
	}

	return (TRUE);
}

void
e1000_config_flow_control_afterLinkUp (struct adapter * Adapter)
{
	u16 MiiStatusReg, MiiNWayAdvertiseReg, MiiNWayBasePgAbleReg;
	u16 Speed, Duplex;

	DEBUGFUNC ("e1000_config_flow_control_afterLinkUp")

	    if (((Adapter->MediaType == MEDIA_TYPE_FIBER)
		 && (Adapter->AutoNegFailed))
		|| ((Adapter->MediaType == MEDIA_TYPE_COPPER)
		    && (!Adapter->AutoNeg))) {
		e1000_force_mac_flow_controlSetting (Adapter);
	}

	if ((Adapter->MediaType == MEDIA_TYPE_COPPER) && Adapter->AutoNeg) {

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		MiiStatusReg = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		if (MiiStatusReg & MII_SR_AUTONEG_COMPLETE) {

			MiiNWayAdvertiseReg = e1000_read_phy_register (Adapter, PHY_AUTONEG_ADVERTISEMENT, Adapter->PhyAddress);

			MiiNWayBasePgAbleReg = e1000_read_phy_register (Adapter, PHY_AUTONEG_LP_BPA, Adapter->PhyAddress);

			if ((MiiNWayAdvertiseReg & NWAY_AR_PAUSE) && (MiiNWayBasePgAbleReg & NWAY_LPAR_PAUSE)) {

				if (Adapter->OriginalFlowControl == FLOW_CONTROL_FULL) {
					Adapter->FlowControl = FLOW_CONTROL_FULL;
					DEBUGOUT ("Flow Control = FULL.\r\n");
				} else {
					Adapter->FlowControl = FLOW_CONTROL_RECEIVE_PAUSE;
					DEBUGOUT ("Flow Control = RX PAUSE frames only.\r\n");
				}
			}

			else if (!(MiiNWayAdvertiseReg & NWAY_AR_PAUSE) &&
				 (MiiNWayAdvertiseReg & NWAY_AR_ASM_DIR) && (MiiNWayBasePgAbleReg & NWAY_LPAR_PAUSE) && (MiiNWayBasePgAbleReg & NWAY_LPAR_ASM_DIR)) {
				Adapter->FlowControl = FLOW_CONTROL_TRANSMIT_PAUSE;
				DEBUGOUT ("Flow Control = TX PAUSE frames only.\r\n");
			}

			else if ((MiiNWayAdvertiseReg & NWAY_AR_PAUSE) &&
				 (MiiNWayAdvertiseReg & NWAY_AR_ASM_DIR) && !(MiiNWayBasePgAbleReg & NWAY_LPAR_PAUSE) && (MiiNWayBasePgAbleReg & NWAY_LPAR_ASM_DIR)) {
				Adapter->FlowControl = FLOW_CONTROL_RECEIVE_PAUSE;
				DEBUGOUT ("Flow Control = RX PAUSE frames only.\r\n");
			}

			else if (Adapter->OriginalFlowControl == FLOW_CONTROL_NONE || Adapter->OriginalFlowControl == FLOW_CONTROL_TRANSMIT_PAUSE) {
				Adapter->FlowControl = FLOW_CONTROL_NONE;
				DEBUGOUT ("Flow Control = NONE.\r\n");
			} else {
				Adapter->FlowControl = FLOW_CONTROL_RECEIVE_PAUSE;
				DEBUGOUT ("Flow Control = RX PAUSE frames only.\r\n");
			}

			e1000_get_speed_and_duplex (Adapter, &Speed, &Duplex);

			if (Duplex == HALF_DUPLEX)
				Adapter->FlowControl = FLOW_CONTROL_NONE;

			e1000_force_mac_flow_controlSetting (Adapter);
		} else {
			DEBUGOUT ("Copper PHY and Auto Neg has not completed.\r\n");
		}
	}
}

void
e1000_force_mac_flow_controlSetting (struct adapter * Adapter)
{
	u32 CtrlRegValue;

	DEBUGFUNC ("e1000_force_mac_flow_controlSetting")

	    CtrlRegValue = E1000_READ_REG (Ctrl);

	switch (Adapter->FlowControl) {
	case FLOW_CONTROL_NONE:

		CtrlRegValue &= (~(E1000_CTRL_TFCE | E1000_CTRL_RFCE));
		break;

	case FLOW_CONTROL_RECEIVE_PAUSE:

		CtrlRegValue &= (~E1000_CTRL_TFCE);
		CtrlRegValue |= E1000_CTRL_RFCE;
		break;

	case FLOW_CONTROL_TRANSMIT_PAUSE:

		CtrlRegValue &= (~E1000_CTRL_RFCE);
		CtrlRegValue |= E1000_CTRL_TFCE;
		break;

	case FLOW_CONTROL_FULL:

		CtrlRegValue |= (E1000_CTRL_TFCE | E1000_CTRL_RFCE);
		break;

	default:

		DEBUGOUT ("Flow control param set incorrectly\n");
		ASSERT (0);

		break;
	}

	if (Adapter->MacType == MAC_WISEMAN_2_0)
		CtrlRegValue &= (~E1000_CTRL_TFCE);

	E1000_WRITE_REG (Ctrl, CtrlRegValue);
}

void
e1000_check_for_link (struct adapter * Adapter)
{
	u32 RxcwRegValue;
	u32 CtrlRegValue;
	u32 StatusRegValue;
	u32 RctlRegValue;
	u16 PhyData;
	u16 LinkPartnerCapability;

	DEBUGFUNC ("e1000_check_for_link")

	    CtrlRegValue = E1000_READ_REG (Ctrl);

	StatusRegValue = E1000_READ_REG (Status);

	RxcwRegValue = E1000_READ_REG (Rxcw);

	if (Adapter->MediaType == MEDIA_TYPE_COPPER && Adapter->GetLinkStatus) {

		PhyData = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		PhyData = e1000_read_phy_register (Adapter, PHY_MII_STATUS_REG, Adapter->PhyAddress);

		if (PhyData & MII_SR_LINK_STATUS) {
			Adapter->GetLinkStatus = FALSE;
		} else {
			DEBUGOUT ("**** CFL - No link detected. ****\r\n");
			return;
		}

		if (!Adapter->AutoNeg) {
			return;
		}

		switch (Adapter->PhyId) {
		case PAXSON_PHY_88E1000:
		case PAXSON_PHY_88E1000S:
		case PAXSON_PHY_INTEGRATED:

			if (Adapter->MacType > MAC_WAINWRIGHT) {
				DEBUGOUT ("CFL - Auto-Neg complete.  Configuring Collision Distance.");
				e1000_configure_collision_distance (Adapter);
			} else {

				PhyData = e1000_read_phy_register (Adapter, PXN_PHY_SPEC_STAT_REG, Adapter->PhyAddress);

				DEBUGOUT1 ("CFL - Auto-Neg complete.  PhyData = %x\r\n", PhyData);
				e1000_configure_mac_to_phySettings (Adapter, PhyData);
			}

			e1000_config_flow_control_afterLinkUp (Adapter);
			break;

		default:
			DEBUGOUT ("CFL - Invalid PHY detected.\r\n");

		}

		if (Adapter->TbiCompatibilityEnable) {
			LinkPartnerCapability = e1000_read_phy_register (Adapter, PHY_AUTONEG_LP_BPA, Adapter->PhyAddress);
			if (LinkPartnerCapability & (NWAY_LPAR_10T_HD_CAPS | NWAY_LPAR_10T_FD_CAPS | NWAY_LPAR_100TX_HD_CAPS | NWAY_LPAR_100TX_FD_CAPS | NWAY_LPAR_100T4_CAPS)) {

				if (Adapter->TbiCompatibilityOn) {

					RctlRegValue = E1000_READ_REG (Rctl);
					RctlRegValue &= ~E1000_RCTL_SBP;
					E1000_WRITE_REG (Rctl, RctlRegValue);
					Adapter->TbiCompatibilityOn = FALSE;
				}
			} else {

				if (!Adapter->TbiCompatibilityOn) {
					Adapter->TbiCompatibilityOn = TRUE;
					RctlRegValue = E1000_READ_REG (Rctl);
					RctlRegValue |= E1000_RCTL_SBP;
					E1000_WRITE_REG (Rctl, RctlRegValue);
				}
			}
		}
	}

	else if ((Adapter->MediaType == MEDIA_TYPE_FIBER) && (!(StatusRegValue & E1000_STATUS_LU)) && (!(CtrlRegValue & E1000_CTRL_SWDPIN1)) && (!(RxcwRegValue & E1000_RXCW_C))) {
		if (Adapter->AutoNegFailed == 0) {
			Adapter->AutoNegFailed = 1;
			return;
		}

		DEBUGOUT ("NOT RXing /C/, disable AutoNeg and force link.\r\n");

		E1000_WRITE_REG (Txcw, (Adapter->TxcwRegValue & ~E1000_TXCW_ANE));

		CtrlRegValue = E1000_READ_REG (Ctrl);
		CtrlRegValue |= (E1000_CTRL_SLU | E1000_CTRL_FD);
		E1000_WRITE_REG (Ctrl, CtrlRegValue);

		e1000_config_flow_control_afterLinkUp (Adapter);

	} else if ((Adapter->MediaType == MEDIA_TYPE_FIBER) && (CtrlRegValue & E1000_CTRL_SLU) && (RxcwRegValue & E1000_RXCW_C)) {

		DEBUGOUT ("RXing /C/, enable AutoNeg and stop forcing link.\r\n");

		E1000_WRITE_REG (Txcw, Adapter->TxcwRegValue);

		E1000_WRITE_REG (Ctrl, (CtrlRegValue & ~E1000_CTRL_SLU));
	}

	return;
}

void
e1000_clear_hw_stats_counters (struct adapter * Adapter)
{
	volatile u32 RegisterContents;

	DEBUGFUNC ("e1000_clear_hw_stats_counters")

	    if (Adapter->e1000_adapter_stopped) {
		DEBUGOUT ("Exiting because the adapter is stopped!!!\n");
		return;
	}

	RegisterContents = E1000_READ_REG (Crcerrs);
	RegisterContents = E1000_READ_REG (Symerrs);
	RegisterContents = E1000_READ_REG (Mpc);
	RegisterContents = E1000_READ_REG (Scc);
	RegisterContents = E1000_READ_REG (Ecol);
	RegisterContents = E1000_READ_REG (Mcc);
	RegisterContents = E1000_READ_REG (Latecol);
	RegisterContents = E1000_READ_REG (Colc);
	RegisterContents = E1000_READ_REG (Dc);
	RegisterContents = E1000_READ_REG (Sec);
	RegisterContents = E1000_READ_REG (Rlec);
	RegisterContents = E1000_READ_REG (Xonrxc);
	RegisterContents = E1000_READ_REG (Xontxc);
	RegisterContents = E1000_READ_REG (Xoffrxc);
	RegisterContents = E1000_READ_REG (Xofftxc);
	RegisterContents = E1000_READ_REG (Fcruc);
	RegisterContents = E1000_READ_REG (Prc64);
	RegisterContents = E1000_READ_REG (Prc127);
	RegisterContents = E1000_READ_REG (Prc255);
	RegisterContents = E1000_READ_REG (Prc511);
	RegisterContents = E1000_READ_REG (Prc1023);
	RegisterContents = E1000_READ_REG (Prc1522);
	RegisterContents = E1000_READ_REG (Gprc);
	RegisterContents = E1000_READ_REG (Bprc);
	RegisterContents = E1000_READ_REG (Mprc);
	RegisterContents = E1000_READ_REG (Gptc);
	RegisterContents = E1000_READ_REG (Gorl);
	RegisterContents = E1000_READ_REG (Gorh);
	RegisterContents = E1000_READ_REG (Gotl);
	RegisterContents = E1000_READ_REG (Goth);
	RegisterContents = E1000_READ_REG (Rnbc);
	RegisterContents = E1000_READ_REG (Ruc);
	RegisterContents = E1000_READ_REG (Rfc);
	RegisterContents = E1000_READ_REG (Roc);
	RegisterContents = E1000_READ_REG (Rjc);
	RegisterContents = E1000_READ_REG (Torl);
	RegisterContents = E1000_READ_REG (Torh);
	RegisterContents = E1000_READ_REG (Totl);
	RegisterContents = E1000_READ_REG (Toth);
	RegisterContents = E1000_READ_REG (Tpr);
	RegisterContents = E1000_READ_REG (Tpt);
	RegisterContents = E1000_READ_REG (Ptc64);
	RegisterContents = E1000_READ_REG (Ptc127);
	RegisterContents = E1000_READ_REG (Ptc255);
	RegisterContents = E1000_READ_REG (Ptc511);
	RegisterContents = E1000_READ_REG (Ptc1023);
	RegisterContents = E1000_READ_REG (Ptc1522);
	RegisterContents = E1000_READ_REG (Mptc);
	RegisterContents = E1000_READ_REG (Bptc);

	if (Adapter->MacType < MAC_LIVENGOOD)
		return;

	RegisterContents = E1000_READ_REG (Algnerrc);
	RegisterContents = E1000_READ_REG (Rxerrc);
	RegisterContents = E1000_READ_REG (Tuc);
	RegisterContents = E1000_READ_REG (Tncrs);
	RegisterContents = E1000_READ_REG (Cexterr);
	RegisterContents = E1000_READ_REG (Rutec);

	RegisterContents = E1000_READ_REG (Tsctc);
	RegisterContents = E1000_READ_REG (Tsctfc);

}

void
e1000_get_speed_and_duplex (struct adapter * Adapter, u16 * Speed, u16 * Duplex)
{
	u32 DeviceStatusReg;

	DEBUGFUNC ("e1000_get_speed_and_duplex")

	    if (Adapter->e1000_adapter_stopped) {
		*Speed = 0;
		*Duplex = 0;
		return;
	}

	if (Adapter->MacType >= MAC_LIVENGOOD) {
		DEBUGOUT ("Livengood MAC\n");
		DeviceStatusReg = E1000_READ_REG (Status);
		if (DeviceStatusReg & E1000_STATUS_SPEED_1000) {
			*Speed = SPEED_1000;
			DEBUGOUT ("   1000 Mbs\n");
		} else if (DeviceStatusReg & E1000_STATUS_SPEED_100) {
			*Speed = SPEED_100;
			DEBUGOUT ("   100 Mbs\n");
		} else {
			*Speed = SPEED_10;
			DEBUGOUT ("   10 Mbs\n");
		}

		if (DeviceStatusReg & E1000_STATUS_FD) {
			*Duplex = FULL_DUPLEX;
			DEBUGOUT ("   Full Duplex\r\n");
		} else {
			*Duplex = HALF_DUPLEX;
			DEBUGOUT ("   Half Duplex\r\n");
		}
	} else {
		DEBUGOUT ("Wiseman MAC - 1000 Mbs, Full Duplex\r\n");
		*Speed = SPEED_1000;
		*Duplex = FULL_DUPLEX;
	}

	return;
}

u16
e1000_read_eeprom_word (struct adapter * Adapter, u16 Reg)
{
	u16 Data;

	ASSERT (Reg < EEPROM_WORD_SIZE);

	E1000_WRITE_REG (Eecd, E1000_EECS);

	ShiftOutBits (Adapter, EEPROM_READ_OPCODE, 3);
	ShiftOutBits (Adapter, Reg, 6);

	Data = ShiftInBits (Adapter);

	EepromCleanup (Adapter);
	return (Data);
}

static void ShiftOutBits (struct adapter * Adapter, u16 Data, u16 Count)
{
	u32 EecdRegValue;
	u32 Mask;

	Mask = 0x01 << (Count - 1);

	EecdRegValue = E1000_READ_REG (Eecd);

	EecdRegValue &= ~(E1000_EEDO | E1000_EEDI);

	do {

		EecdRegValue &= ~E1000_EEDI;

		if (Data & Mask)
			EecdRegValue |= E1000_EEDI;

		E1000_WRITE_REG (Eecd, EecdRegValue);

		udelay (50);

		RaiseClock (Adapter, &EecdRegValue);
		LowerClock (Adapter, &EecdRegValue);

		Mask = Mask >> 1;

	} while (Mask);

	EecdRegValue &= ~E1000_EEDI;

	E1000_WRITE_REG (Eecd, EecdRegValue);
}

static void
RaiseClock (struct adapter * Adapter, u32 * EecdRegValue)
{

	*EecdRegValue = *EecdRegValue | E1000_EESK;

	E1000_WRITE_REG (Eecd, *EecdRegValue);

	udelay (50);
}

static void
LowerClock (struct adapter * Adapter, u32 * EecdRegValue)
{

	*EecdRegValue = *EecdRegValue & ~E1000_EESK;

	E1000_WRITE_REG (Eecd, *EecdRegValue);

	udelay (50);
}

static u16
ShiftInBits (struct adapter * Adapter)
{
	u32 EecdRegValue;
	unsigned int i;
	u16 Data;

	EecdRegValue = E1000_READ_REG (Eecd);

	EecdRegValue &= ~(E1000_EEDO | E1000_EEDI);
	Data = 0;

	for (i = 0; i < 16; i++) {
		Data = Data << 1;
		RaiseClock (Adapter, &EecdRegValue);

		EecdRegValue = E1000_READ_REG (Eecd);

		EecdRegValue &= ~(E1000_EEDI);
		if (EecdRegValue & E1000_EEDO)
			Data |= 1;

		LowerClock (Adapter, &EecdRegValue);
	}

	return (Data);
}

static void
EepromCleanup (struct adapter * Adapter)
{
	u32 EecdRegValue;

	EecdRegValue = E1000_READ_REG (Eecd);

	EecdRegValue &= ~(E1000_EECS | E1000_EEDI);

	E1000_WRITE_REG (Eecd, EecdRegValue);

	RaiseClock (Adapter, &EecdRegValue);
	LowerClock (Adapter, &EecdRegValue);
}

unsigned char
e1000_validate_eeprom_checksum (struct adapter * Adapter)
{
	u16 Checksum = 0;
	u16 Iteration;

	for (Iteration = 0; Iteration < (EEPROM_CHECKSUM_REG + 1); Iteration++)
		Checksum += e1000_read_eeprom_word (Adapter, Iteration);

	if (Checksum == (u16) EEPROM_SUM)
		return (TRUE);
	else
		return (FALSE);
}

void
e1000_update_eeprom_checksum (struct adapter * Adapter)
{
	u16 Checksum = 0;
	u16 Iteration;

	for (Iteration = 0; Iteration < EEPROM_CHECKSUM_REG; Iteration++)
		Checksum += e1000_read_eeprom_word (Adapter, Iteration);

	Checksum = (u16) EEPROM_SUM - Checksum;

	e1000_write_eeprom_word (Adapter, EEPROM_CHECKSUM_REG, Checksum);
}

unsigned char
e1000_write_eeprom_word (struct adapter * Adapter, u16 Reg, u16 Data)
{

	E1000_WRITE_REG (Eecd, E1000_EECS);

	StandBy (Adapter);

	ShiftOutBits (Adapter, EEPROM_EWEN_OPCODE, 5);
	ShiftOutBits (Adapter, Reg, 4);

	StandBy (Adapter);

	ShiftOutBits (Adapter, EEPROM_ERASE_OPCODE, 3);
	ShiftOutBits (Adapter, Reg, 6);

	if (!WaitEepromCommandDone (Adapter)) {
		return (FALSE);
	}

	ShiftOutBits (Adapter, EEPROM_EWDS_OPCODE, 5);
	ShiftOutBits (Adapter, Reg, 4);

	StandBy (Adapter);

	ShiftOutBits (Adapter, EEPROM_EWEN_OPCODE, 5);
	ShiftOutBits (Adapter, Reg, 4);

	StandBy (Adapter);

	ShiftOutBits (Adapter, EEPROM_WRITE_OPCODE, 3);
	ShiftOutBits (Adapter, Reg, 6);
	ShiftOutBits (Adapter, Data, 16);

	if (!WaitEepromCommandDone (Adapter)) {
		return (FALSE);
	}

	ShiftOutBits (Adapter, EEPROM_EWDS_OPCODE, 5);
	ShiftOutBits (Adapter, Reg, 4);

	EepromCleanup (Adapter);

	return (TRUE);
}

static u16
WaitEepromCommandDone (struct adapter * Adapter)
{
	u32 EecdRegValue;
	unsigned int i;

	StandBy (Adapter);

	for (i = 0; i < 200; i++) {
		EecdRegValue = E1000_READ_REG (Eecd);

		if (EecdRegValue & E1000_EEDO)
			return (TRUE);

		udelay (5);
	}

	return (FALSE);
}

static void
StandBy (struct adapter * Adapter)
{
	u32 EecdRegValue;

	EecdRegValue = E1000_READ_REG (Eecd);

	EecdRegValue &= ~(E1000_EECS | E1000_EESK);

	E1000_WRITE_REG (Eecd, EecdRegValue);

	udelay (5);

	EecdRegValue |= E1000_EECS;

	E1000_WRITE_REG (Eecd, EecdRegValue);
}

unsigned char
e1000_read_part_number (struct adapter * Adapter, u32 * PartNumber)
{
	u16 EepromWordValue;

	DEBUGFUNC ("e1000_read_part_number")

	    if (Adapter->e1000_adapter_stopped) {
		*PartNumber = 0;
		return (FALSE);
	}

	EepromWordValue = e1000_read_eeprom_word (Adapter, (u16) (EEPROM_PBA_BYTE_1));

	DEBUGOUT ("Read first part number word\n");

	*PartNumber = (u32) EepromWordValue;
	*PartNumber = *PartNumber << 16;

	EepromWordValue = e1000_read_eeprom_word (Adapter, (u16) (EEPROM_PBA_BYTE_1 + 1));

	DEBUGOUT ("Read second part number word\n");

	*PartNumber |= EepromWordValue;

	return (TRUE);

}

void
e1000_id_led_on (struct adapter * Adapter)
{
	u32 CtrlRegValue;

	if (Adapter->e1000_adapter_stopped) {
		return;
	}

	CtrlRegValue = E1000_READ_REG (Ctrl);

	CtrlRegValue |= E1000_CTRL_SWDPIO0;

	if (e1000_is_low_profile (Adapter)) {
		CtrlRegValue &= ~E1000_CTRL_SWDPIN0;
	} else {
		CtrlRegValue |= E1000_CTRL_SWDPIN0;
	}

	E1000_WRITE_REG (Ctrl, CtrlRegValue);

}

void
e1000_id_led_off (struct adapter * Adapter)
{
	u32 CtrlRegValue;

	if (Adapter->e1000_adapter_stopped) {
		return;
	}

	CtrlRegValue = E1000_READ_REG (Ctrl);

	CtrlRegValue |= E1000_CTRL_SWDPIO0;

	if (e1000_is_low_profile (Adapter)) {
		CtrlRegValue |= E1000_CTRL_SWDPIN0;
	} else {
		CtrlRegValue &= ~E1000_CTRL_SWDPIN0;
	}

	E1000_WRITE_REG (Ctrl, CtrlRegValue);
}

void
e1000_set_id_led_forPCIX (struct adapter * Adapter)
{
	u32 PciStatus;

	PciStatus = E1000_READ_REG (Status);

	if (PciStatus & E1000_STATUS_PCIX_MODE) {
		e1000_id_led_on (Adapter);
	} else {
		e1000_id_led_off (Adapter);
	}
}

unsigned char
e1000_is_low_profile (struct adapter * Adapter)
{
	u16 LedLogicWord;
	unsigned char ReturnValue = FALSE;

	if (Adapter->MacType >= MAC_CORDOVA) {

		LedLogicWord = e1000_read_eeprom_word (Adapter, E1000_EEPROM_LED_LOGIC);

		if (LedLogicWord & E1000_EEPROM_SWDPIN0)
			ReturnValue = TRUE;
		else
			ReturnValue = FALSE;
	}

	return ReturnValue;
}

void
e1000_adjust_tbi_accepted_stats (struct adapter * Adapter, u32 FrameLength, unsigned char * MacAddress)
{
	u32 CarryBit;

	FrameLength--;

	Adapter->Crcerrs--;

	Adapter->Gprc++;

	CarryBit = 0x80000000 & Adapter->Gorcl;
	Adapter->Gorcl += FrameLength;

	if (CarryBit && ((Adapter->Gorcl & 0x80000000) == 0)) {
		Adapter->Gorch++;
	}

	if ((MacAddress[0] == (unsigned char) 0xff) && (MacAddress[1] == (unsigned char) 0xff)) {

		Adapter->Bprc++;
	} else if (*MacAddress & 0x01) {

		Adapter->Mprc++;
	}
	if (FrameLength == Adapter->MaxFrameSize) {

		Adapter->Roc--;
	}

	if (FrameLength == 64) {
		Adapter->Prc64++;
		Adapter->Prc127--;
	} else if (FrameLength == 127) {
		Adapter->Prc127++;
		Adapter->Prc255--;
	} else if (FrameLength == 255) {
		Adapter->Prc255++;
		Adapter->Prc511--;
	} else if (FrameLength == 511) {
		Adapter->Prc511++;
		Adapter->Prc1023--;
	} else if (FrameLength == 1023) {
		Adapter->Prc1023++;
		Adapter->Prc1522--;
	} else if (FrameLength == 1522) {
		Adapter->Prc1522++;
	}
}
