/*
 * Compaq Hot Plug Controller Driver
 *
 * Copyright (c) 1995,2001 Compaq Computer Corporation
 * Copyright (c) 2001 Greg Kroah-Hartman (greg@kroah.com)
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <linuxhotplug@compaq.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/smp_lock.h>
#include <asm/uaccess.h>
#include "hotplug_pci.h"
#include "hotplug_pci_ioctl.h"


/*
This is a mechanism to notify user level processes asynchronously of events
that happened on a driver level.  The mechanism transmits messages by
dispatching SIGIO and then expects the application to read the message in a
messages queue.
 */

static struct fasync_struct *fasync;	/* = NULL */

static struct semaphore notify_sem;	/* mutex for process loop (only up if something to process) */
static struct semaphore exit_sem;	/* guard to make sure thread has exited before calling it quits */

static struct semaphore queue_sem;	/* locks the queue to prevent new messages from being written in parallel */
static struct semaphore data_sem;	/* protects msg_buffer to prevent new messages from being read or written in parallel */

static struct pci_hp_message msg_buffer[HPCD_BUFFER_LENGTH];
static int msg_start_index;
static int msg_serial_number;
static int hpcd_finished;

static int pci_visit_bridge(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent);
static int raw_pci_visit_bridge(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent);
static void * get_subsequent_smbios_entry(void *smbios_table, void *curr);


static void init_notifier(void)
{
        init_MUTEX(&queue_sem);
	init_MUTEX(&data_sem);
	init_MUTEX_LOCKED(&notify_sem);
	init_MUTEX_LOCKED(&exit_sem);
	memset(&msg_buffer, 0, sizeof(struct pci_hp_message) * HPCD_BUFFER_LENGTH);
	msg_start_index = 0;
	msg_serial_number = 0;
	hpcd_finished = 0;
}


int pci_hp_register_user_notifier(int fd, struct file* filp)
{
	int ret;

	down (&queue_sem);
	ret = fasync_helper(fd, filp, 1, &fasync);
	up (&queue_sem);
	return ret;
}


int pci_hp_unregister_user_notifier(int fd, struct file* filp)
{
	int ret;

	down (&queue_sem);
	ret = fasync_helper(fd, filp, 0, &fasync);
	up (&queue_sem);
	return ret;
}


void pci_hp_notify_id(int id)
{
	DBG(__FUNCTION__": setting up message (%d, %d)\n", msg_serial_number, id);

	down (&queue_sem);
	down (&data_sem);

	msg_buffer[msg_start_index].eb_serial_number = msg_serial_number;
	msg_buffer[msg_start_index].message_id= id;

	msg_serial_number++;
	msg_start_index = (msg_start_index + 1) % HPCD_BUFFER_LENGTH;
	
	up (&data_sem);
	up (&notify_sem);
	up (&queue_sem);
}


static int hpcd_start_thread(void* data)
{
	lock_kernel();
	// release insmod memory map, use the kernel's mm
	// init is session and group leader
	// disconnect filesystem usage
	daemonize();
	// New name
	strcpy(current->comm, "phpd_notify");
	unlock_kernel();

	hpcd_finished = 0;
	while (1) {
		down(&notify_sem);
		DBG("hpcd_notifier_thread woken finished = %d\n", hpcd_finished);
		if (hpcd_finished)
			break;

		if (fasync) {
			down (&queue_sem);
			kill_fasync(&fasync, SIGIO, POLL_IN);
			up (&queue_sem);
		}
	}
	DBG("hpcd_notifier_thread signals exit\n");
	up(&exit_sem);
	return 0;
}


int pci_hp_start_notify(void)
{
	int pid;

	init_notifier();
	pid = kernel_thread(hpcd_start_thread, 0, 0);
	if (pid < 0) {
		err ("Can't start up our notifier thread\n");
		return -1;
	}
	DBG("Our notifier thread pid = %d\n", pid);
	return 0;
}


void pci_hp_stop_notify(void)
{
	DBG("hpcd_notifier thread finish command given\n");
	down (&queue_sem);
	hpcd_finished = 1;
	up (&notify_sem);
	up (&queue_sem);
	
	DBG("wait for hpcd_notifier to exit\n");
	down (&exit_sem);
	DBG("wait for hpcd_notifier exited\n");
}


ssize_t pci_hp_read_helper(struct file* filp, char* buf, size_t count, loff_t *ppos)
{
	int curr = (int) filp->private_data;

	DBG("reading queue element %d\n", curr);

	down (&data_sem);
	if (buf) {
		DBG("reading message (%d, %d)\n", msg_buffer[curr].eb_serial_number, msg_buffer[curr].message_id); 
		copy_to_user(buf, &msg_buffer[curr], sizeof(struct pci_hp_message));
	}
	filp->private_data = (void*) ((curr + 1) % HPCD_BUFFER_LENGTH);
	up (&data_sem);

	return sizeof(struct pci_hp_message);
}


void pci_hp_open_helper(struct file* filp)
{
	DBG("initializing reader at %d\n", msg_start_index);
	filp->private_data = (void*) (msg_start_index);
}


/*
This is code that scans the pci buses. The idea is to either rely on linux
mechanism (linux_*) or a raw pci scan (PCIBIOS routines) to accomplish that
goal. In either case, every bus and every function is presented to a custom
function that can act upon it.
 */

int pci_hp_linux_pci_visit_dev(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent)
{
	struct pci_dev* dev = wrapped_dev ? wrapped_dev->dev : NULL;

	if (!dev) return 0;

	if (fn->pre_visit_linux_pci_dev)
		if ((*(fn->pre_visit_linux_pci_dev)) (wrapped_dev, wrapped_parent))
			return -1;
	switch (dev->class >> 8) {
		case PCI_CLASS_BRIDGE_PCI:
			if (pci_visit_bridge(fn, wrapped_dev, wrapped_parent))
				return -1;
			break;
		default:
			DBG("scanning device %02x, %02x\n", PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));
			if (fn->visit_linux_pci_dev)
				if ((*(fn->visit_linux_pci_dev)) (wrapped_dev, wrapped_parent))
					return -1;
	}
	if (fn->post_visit_linux_pci_dev)
		if ((*(fn->post_visit_linux_pci_dev)) (wrapped_dev, wrapped_parent))
			return -1;
	return 0;
}


static int pci_visit_bus(struct visit_linux_pci * fn, struct wrapped_pci_bus *wrapped_bus, struct wrapped_pci_dev *wrapped_parent)
{
	struct list_head *ln;
	struct pci_dev *dev;
	struct wrapped_pci_dev wrapped_dev;

	DBG("scanning bus %02x\n", wrapped_bus->bus->number);

	if (fn->pre_visit_linux_pci_bus)
		if ((*(fn->pre_visit_linux_pci_bus)) (wrapped_bus, wrapped_parent))
			return -1;


	ln = wrapped_bus->bus->devices.next; 
	while (ln != &wrapped_bus->bus->devices) {
		dev = pci_dev_b(ln);
		ln = ln->next;

		memset(&wrapped_dev, 0, sizeof(struct wrapped_pci_dev));
		wrapped_dev.dev = dev;

		if (pci_hp_linux_pci_visit_dev(fn, &wrapped_dev, wrapped_bus))
			return -1;
	}

	if (fn->post_visit_linux_pci_bus)
		if ((*(fn->post_visit_linux_pci_bus)) (wrapped_bus, wrapped_parent))
			return -1;

	return 0;
}


static int pci_visit_bridge(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent)
{
	struct pci_bus *bus = wrapped_dev->dev->subordinate;
	struct wrapped_pci_bus wrapped_bus;

	memset(&wrapped_bus, 0, sizeof(struct wrapped_pci_bus));
	wrapped_bus.bus = bus;

	DBG("scanning bridge %02x, %02x\n", wrapped_dev->dev->devfn >> 3, wrapped_dev->dev->devfn & 0x7);

	if (fn->visit_linux_pci_dev)
		if ((*(fn->visit_linux_pci_dev)) (wrapped_dev, wrapped_parent))
			return -1;

	if (pci_visit_bus(fn, &wrapped_bus, wrapped_dev))
		return -1;
	return 0;
}


static int raw_pci_visit_function(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_bus)
{
    int func = 0;
    int is_multi = 0;
    unsigned char hdr_type;
    unsigned int class;

    for (func = 0; func < 8; func++, wrapped_dev->dev->devfn++) {
	if (func && !is_multi)	/* not a multi-function device */
	    continue;
	if (pci_read_config_byte(wrapped_dev->dev, PCI_HEADER_TYPE, &hdr_type))
	    continue;
	if (pci_read_config_dword(wrapped_dev->dev, PCI_CLASS_REVISION, &class))
	    continue;
	class >>= 8;
	wrapped_dev->dev->hdr_type = hdr_type & 0x7f;
	wrapped_dev->dev->class = class;

	switch (wrapped_dev->dev->class >> 8) {
	case PCI_CLASS_BRIDGE_PCI:
	    if (raw_pci_visit_bridge(fn, wrapped_dev, wrapped_bus))
		return -1;
	    break;
	default:
	    if (fn->visit_linux_pci_dev)
		if ((*(fn->visit_linux_pci_dev)) (wrapped_dev, wrapped_bus))
		    return -1;
	}

	if (!func) {
	    is_multi = hdr_type & 0x80;
	}
    }
    return 0;
}


static void raw_pci_set_bus(struct pci_bus *child, struct pci_bus *parent, struct pci_dev *dev)
{
	memset(child, 0, sizeof(struct wrapped_pci_dev));

	child->self = dev;
	dev->subordinate = child;
	child->parent = parent;
	child->ops = parent->ops;
	child->sysdata = parent->sysdata;

	child->number = child->secondary = 0xff;
	child->primary = parent->secondary;
	child->subordinate = 0xff;
}


static int raw_pci_visit_bus(struct visit_linux_pci * fn, struct wrapped_pci_bus *wrapped_bus, struct wrapped_pci_dev *wrapped_parent)
{
	unsigned int devfn;
	struct pci_bus* bus= wrapped_bus->bus;
	struct pci_dev dev0;
	struct wrapped_pci_dev wrapped_dev0;

	memset(&dev0, 0, sizeof(struct pci_dev));
	memset(&wrapped_dev0, 0, sizeof(struct wrapped_pci_dev));
	wrapped_dev0.dev=&dev0;

	DBG("scanning bus %02x\n", bus->number);

	if (fn->pre_visit_linux_pci_bus)
		if ((*(fn->pre_visit_linux_pci_bus)) (wrapped_bus, wrapped_parent))
		return -1;

	/* Create a device template */
	dev0.bus = bus;
	dev0.sysdata = bus->sysdata;

	/* Go find them, Rover! */
	for (devfn = 0; devfn < 0x100; devfn += 8) {
		dev0.devfn = devfn;
		if (raw_pci_visit_function(fn, &wrapped_dev0, wrapped_bus))
			return -1;
	}

	if (fn->post_visit_linux_pci_bus)
		if ((*(fn->post_visit_linux_pci_bus)) (wrapped_bus, wrapped_parent))
			return -1;

	return 0;
}


static int raw_pci_visit_bridge(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent)
{
	unsigned int buses;
	unsigned short cr;
	struct pci_dev* dev = wrapped_dev->dev;
	struct pci_bus* bus= wrapped_parent->bus;
	struct pci_bus child;
	struct wrapped_pci_bus wrapped_child;

	memset(&child, 0, sizeof(struct pci_bus));
	memset(&wrapped_child, 0, sizeof(struct wrapped_pci_bus));
	wrapped_child.bus =&child;

	if (fn->visit_linux_pci_dev)
		if ((*(fn->visit_linux_pci_dev)) (wrapped_dev, wrapped_parent))
			return -1;

	raw_pci_set_bus(&child, bus, dev);

	/*
	 * Clear all status bits and turn off memory,
	 * I/O and master enables.
	*/
	//TODO: If we ever use this function, we need to understand that the
	//      following 3 commands make this visitation have side-effects.
	pci_read_config_word(dev, PCI_COMMAND, &cr);
	pci_write_config_word(dev, PCI_COMMAND, 0x0000);
	pci_write_config_word(dev, PCI_STATUS, 0xffff);

	/*
	 * Read the existing primary/secondary/subordinate bus
	 * number configuration to determine if the PCI bridge
	 * has already been configured by the system.  If so,
	 * do not modify the configuration, merely note it.
	 */
	pci_read_config_dword(dev, PCI_PRIMARY_BUS, &buses);
	if ((buses & 0xFFFFFF) != 0) {
		child.primary = buses & 0xFF;
		child.secondary = (buses >> 8) & 0xFF;
		child.subordinate = (buses >> 16) & 0xFF;
		child.number = child.secondary;

	}
	if (raw_pci_visit_bus (fn, &wrapped_child, wrapped_dev))
		return -1;

	pci_write_config_word(dev, PCI_COMMAND, cr);
	return 0;
}


/* Subsequently some routines that read SMBIOS entries */

/**
 * pci_hp_get_SMBIOS_entry
 *
 * @type:SMBIOS structure type to be returned
 * @previous: %NULL or pointer to previously returned structure
 *
 * Gets the first entry of the specified type if previous == NULL
 * Otherwise, returns the next entry of the given type.
 * Uses global SMBIOS Table pointer
 * Uses get_subsequent_smbios_entry
 *
 * returns a pointer to an SMBIOS structure or %NULL if none found
 */
void * pci_hp_get_SMBIOS_entry (void *smbios_table, u8 type, void * previous)
{
	if (!smbios_table)
		return NULL;

	if (!previous) {		  
		previous = pci_hp_smbios_start;
	} else {
		previous = get_subsequent_smbios_entry(smbios_table, previous);
	}

	while (previous) {
	       	if (readb(previous + SMBIOS_GENERIC_TYPE) != type) {
			previous = get_subsequent_smbios_entry(smbios_table, previous);
		} else {
			break;
		}
	}

	return previous;
}


/*
 * get_subsequent_smbios_entry
 *
 * Gets the first entry if previous == NULL
 * Otherwise, returns the next entry
 * Uses global SMBIOS Table pointer
 *
 * @curr: %NULL or pointer to previously returned structure
 *
 * returns a pointer to an SMBIOS structure or NULL if none found
 */
static void * get_subsequent_smbios_entry(void *smbios_table, void *curr)
{
	u8 bail = 0;
	u8 previous_byte = 1;
	void *p_temp;
	void *p_max;

	if (!smbios_table || !curr)
		return(NULL);

	// set p_max to the end of the table
	p_max = pci_hp_smbios_start + readw(smbios_table + ST_LENGTH);

	p_temp = curr;
	p_temp += readb(curr + SMBIOS_GENERIC_LENGTH);

	while ((p_temp < p_max) && !bail) {
		// Look for the double NULL terminator
		// The first condition is the previous byte and the second is the curr
		if (!previous_byte && !(readb(p_temp))) {
			bail = 1;
		}

		previous_byte = readb(p_temp);
		p_temp++;
	}

	if (p_temp < p_max) {
		return p_temp;
	} else {
		return NULL;
	}
}

