/*
 * Compaq Hot Plug Controller Driver
 *
 * Copyright (c) 1995,2001 Compaq Computer Corporation
 * Copyright (c) 2001 Greg Kroah-Hartman (greg@kroah.com)
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <linuxhotplug@compaq.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/slab.h>
#include "hotplug_pci.h"


/*
 * pci_hp_sort_and_combine
 *
 * Sorts all of the nodes in the list in ascending order by
 * their base addresses.  Also does garbage collection by
 * combining adjacent nodes.
 *
 * returns 0 if success
 */
u32 pci_hp_sort_and_combine(struct pci_resource **head)
{
	struct pci_resource *node1;
	struct pci_resource *node2;
	int out_of_order = 1;

	DBG(__FUNCTION__": head = %p, *head = %p\n", head, *head);

	if (!(*head))
		return(1);

	DBG("*head->next = %p\n",(*head)->next);

	if (!(*head)->next)
		return(0);	/* only one item on the list, already sorted! */

	DBG("*head->base = 0x%x\n",(*head)->base);
	DBG("*head->next->base = 0x%x\n",(*head)->next->base);
	while (out_of_order) {
		out_of_order = 0;

		// Special case for swapping list head
		if (((*head)->next) &&
		    ((*head)->base > (*head)->next->base)) {
			node1 = *head;
			(*head) = (*head)->next;
			node1->next = (*head)->next;
			(*head)->next = node1;
			out_of_order++;
		}

		node1 = (*head);

		while (node1->next && node1->next->next) {
			if (node1->next->base > node1->next->next->base) {
				out_of_order++;
				node2 = node1->next;
				node1->next = node1->next->next;
				node1 = node1->next;
				node2->next = node1->next;
				node1->next = node2;
			} else
				node1 = node1->next;
		}
	}  // End of out_of_order loop

	node1 = *head;

	while (node1 && node1->next) {
		if ((node1->base + node1->length) == node1->next->base) {
			// Combine
			DBG("8..\n");
			node1->length += node1->next->length;
			node2 = node1->next;
			node1->next = node1->next->next;
			kfree(node2);
		} else
			node1 = node1->next;
	}

	return(0);
}


/*
 * sort_by_size
 *
 * Sorts nodes on the list by their length.
 * Smallest first.
 *
 */
static u32 sort_by_size(struct pci_resource **head)
{
	struct pci_resource *current_res;
	struct pci_resource *next_res;
	int out_of_order = 1;

	if (!(*head))
		return(1);

	if (!((*head)->next))
		return(0);

	while (out_of_order) {
		out_of_order = 0;

		// Special case for swapping list head
		if (((*head)->next) &&
		    ((*head)->length > (*head)->next->length)) {
			out_of_order++;
			current_res = *head;
			*head = (*head)->next;
			current_res->next = (*head)->next;
			(*head)->next = current_res;
		}

		current_res = *head;

		while (current_res->next && current_res->next->next) {
			if (current_res->next->length > current_res->next->next->length) {
				out_of_order++;
				next_res = current_res->next;
				current_res->next = current_res->next->next;
				current_res = current_res->next;
				next_res->next = current_res->next;
				current_res->next = next_res;
			} else
				current_res = current_res->next;
		}
	}  // End of out_of_order loop

	return(0);
}


/*
 * sort_by_max_size
 *
 * Sorts nodes on the list by their length.
 * Largest first.
 *
 */
static u32 sort_by_max_size(struct pci_resource **head)
{
	struct pci_resource *current_res;
	struct pci_resource *next_res;
	int out_of_order = 1;

	if (!(*head))
		return(1);

	if (!((*head)->next))
		return(0);

	while (out_of_order) {
		out_of_order = 0;

		// Special case for swapping list head
		if (((*head)->next) &&
		    ((*head)->length < (*head)->next->length)) {
			out_of_order++;
			current_res = *head;
			*head = (*head)->next;
			current_res->next = (*head)->next;
			(*head)->next = current_res;
		}

		current_res = *head;

		while (current_res->next && current_res->next->next) {
			if (current_res->next->length < current_res->next->next->length) {
				out_of_order++;
				next_res = current_res->next;
				current_res->next = current_res->next->next;
				current_res = current_res->next;
				next_res->next = current_res->next;
				current_res->next = next_res;
			} else
				current_res = current_res->next;
		}
	}  // End of out_of_order loop

	return(0);
}


/*
 * pci_hp_get_max_resource
 *
 * Gets the largest node that is at least "size" big from the
 * list pointed to by head.  It aligns the node on top and bottom
 * to "size" alignment before returning it.
 */
struct pci_resource *pci_hp_get_max_resource (struct pci_resource **head, u32 size)
{
	struct pci_resource *max;
	struct pci_resource *temp;
	struct pci_resource *split_node;
	u32 temp_dword;

	if (!(*head))
		return(NULL);

	if (pci_hp_sort_and_combine(head))
		return(NULL);

	if (sort_by_max_size(head))
		return(NULL);

	for (max = *head;max; max = max->next) {

		// If not big enough we could probably just bail, 
		// instead we'll continue to the next.
		if (max->length < size)
			continue;

		if (max->base & (size - 1)) {
			// this one isn't base aligned properly
			// so we'll make a new entry and split it up
			temp_dword = (max->base | (size-1)) + 1;

			// Short circuit if adjusted size is too small
			if ((max->length - (temp_dword - max->base)) < size)
				continue;

			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);

			split_node->base = max->base;
			split_node->length = temp_dword - max->base;
			max->base = temp_dword;
			max->length -= split_node->length;

			// Put it next in the list
			split_node->next = max->next;
			max->next = split_node;
		}

		if ((max->base + max->length) & (size - 1)) {
			// this one isn't end aligned properly at the top
			// so we'll make a new entry and split it up
			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);
			temp_dword = ((max->base + max->length) & ~(size - 1));
			split_node->base = temp_dword;
			split_node->length = max->length + max->base
					     - split_node->base;
			max->length -= split_node->length;

			// Put it in the list
			split_node->next = max->next;
			max->next = split_node;
		}

		// Make sure it didn't shrink too much when we aligned it
		if (max->length < size)
			continue;

		// Now take it out of the list
		temp = (struct pci_resource*) *head;
		if (temp == max) {
			*head = max->next;
		} else {
			while (temp && temp->next != max) {
				temp = temp->next;
			}

			temp->next = max->next;
		}

		max->next = NULL;
		return(max);
	}

	// If we get here, we couldn't find one
	return(NULL);
}


/*
 * pci_hp_get_resource
 *
 * this function sorts the resource list by size and then
 * returns the first node of "size" length.  If it finds a node
 * larger than "size" it will split it up.
 *
 * size must be a power of two.
 */
struct pci_resource *pci_hp_get_resource (struct pci_resource **head, u32 size)
{
	struct pci_resource *prevnode;
	struct pci_resource *node;
	struct pci_resource *split_node;
	u32 temp_dword;

	if (!(*head))
		return(NULL);

	if ( pci_hp_sort_and_combine(head) )
		return(NULL);

	if ( sort_by_size(head) )
		return(NULL);

	for (node = *head; node; node = node->next) {
		DBG(__FUNCTION__": req_size =%x node=%p, base=%x, length=%x\n",
		    size, node, node->base, node->length);
		if (node->length < size)
			continue;

		if (node->base & (size - 1)) {
			DBG(__FUNCTION__": not aligned\n");
			// this one isn't base aligned properly
			// so we'll make a new entry and split it up
			temp_dword = (node->base | (size-1)) + 1;

			// Short circuit if adjusted size is too small
			if ((node->length - (temp_dword - node->base)) < size)
				continue;

			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);

			split_node->base = node->base;
			split_node->length = temp_dword - node->base;
			node->base = temp_dword;
			node->length -= split_node->length;

			// Put it in the list
			split_node->next = node->next;
			node->next = split_node;
		} // End of non-aligned base

		// Don't need to check if too small since we already did
		if (node->length > size) {
			DBG(__FUNCTION__": too big\n");
			// this one is longer than we need
			// so we'll make a new entry and split it up
			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);

			split_node->base = node->base + size;
			split_node->length = node->length - size;
			node->length = size;

			// Put it in the list
			split_node->next = node->next;
			node->next = split_node;
		}  // End of too big on top end

		DBG(__FUNCTION__": got one!!!\n");
		// If we got here, then it is the right size
		// Now take it out of the list
		if (*head == node) {
			*head = node->next;
		} else {
			prevnode = *head;
			while (prevnode->next != node)
				prevnode = prevnode->next;

			prevnode->next = node->next;
		}
		node->next = NULL;
		// Stop looping
		break;
	}
	return(node);
}


/*
 * pci_hp_get_io_resource
 *
 * this function sorts the resource list by size and then
 * returns the first node of "size" length that is not in the
 * ISA aliasing window.  If it finds a node larger than "size"
 * it will split it up.
 *
 * size must be a power of two.
 */
struct pci_resource *pci_hp_get_io_resource (struct pci_resource **head, u32 size)
{
	struct pci_resource *prevnode;
	struct pci_resource *node;
	struct pci_resource *split_node;
	u32 temp_dword;

	if (!(*head))
		return(NULL);

	if ( pci_hp_sort_and_combine(head) )
		return(NULL);

	if ( sort_by_size(head) )
		return(NULL);

	for (node = *head; node; node = node->next) {
		if (node->length < size)
			continue;

		if (node->base & (size - 1)) {
			// this one isn't base aligned properly
			// so we'll make a new entry and split it up
			temp_dword = (node->base | (size-1)) + 1;

			// Short circuit if adjusted size is too small
			if ((node->length - (temp_dword - node->base)) < size)
				continue;

			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);

			split_node->base = node->base;
			split_node->length = temp_dword - node->base;
			node->base = temp_dword;
			node->length -= split_node->length;

			// Put it in the list
			split_node->next = node->next;
			node->next = split_node;
		} // End of non-aligned base

		// Don't need to check if too small since we already did
		if (node->length > size) {
			// this one is longer than we need
			// so we'll make a new entry and split it up
			split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

			if (!split_node)
				return(NULL);

			split_node->base = node->base + size;
			split_node->length = node->length - size;
			node->length = size;

			// Put it in the list
			split_node->next = node->next;
			node->next = split_node;
		}  // End of too big on top end

		// For IO make sure it's not in the ISA aliasing space
		if (node->base & 0x300L)
			continue;

		// If we got here, then it is the right size
		// Now take it out of the list
		if (*head == node) {
			*head = node->next;
		} else {
			prevnode = *head;
			while (prevnode->next != node)
				prevnode = prevnode->next;

			prevnode->next = node->next;
		}
		node->next = NULL;
		// Stop looping
		break;
	}

	return(node);
}


/*
 * pci_hp_do_bridge_resource_split
 *
 *	Returns zero or one node of resources that aren't in use
 *
 */
struct pci_resource *pci_hp_do_bridge_resource_split (struct pci_resource **head, u32 alignment)
{
	struct pci_resource *prevnode = NULL;
	struct pci_resource *node;
	u32 rc;
	u32 temp_dword;

	if (!(*head))
		return(NULL);

	rc = pci_hp_sort_and_combine(head);

	if (rc)
		return(NULL);

	node = *head;

	while (node->next) {
		prevnode = node;
		node = node->next;
		kfree(prevnode);
	}

	if (node->length < alignment) {
		kfree(node);
		return(NULL);
	}

	if (node->base & (alignment - 1)) {
		// Short circuit if adjusted size is too small
		temp_dword = (node->base | (alignment-1)) + 1;
		if ((node->length - (temp_dword - node->base)) < alignment) {
			kfree(node);
			return(NULL);
		}

		node->length -= (temp_dword - node->base);
		node->base = temp_dword;
	}

	if (node->length & (alignment - 1)) {
		// There's stuff in use after this node
		kfree(node);
		return(NULL);
	}

	return(node);
}


/*
 * pci_hp_do_pre_bridge_resource_split
 *
 *	Returns zero or one node of resources that aren't in use
 *
 */
struct pci_resource *pci_hp_do_pre_bridge_resource_split (struct pci_resource **head, struct pci_resource **orig_head, u32 alignment)
{
	struct pci_resource *prevnode = NULL;
	struct pci_resource *node;
	struct pci_resource *split_node;
	u32 rc;
	u32 temp_dword;
	DBG("do_pre_bridge_resource_split\n");

	if (!(*head) || !(*orig_head))
		return(NULL);

	rc = pci_hp_sort_and_combine(head);

	if (rc)
		return(NULL);

	if ((*head)->base != (*orig_head)->base)
		return(NULL);

	if ((*head)->length == (*orig_head)->length)
		return(NULL);


	// If we got here, there the bridge requires some of the resource, but
	// we may be able to split some off of the front

	node = *head;

	if (node->length & (alignment -1)) {
		// this one isn't an aligned length, so we'll make a new entry
		// and split it up.
		split_node = (struct pci_resource*) kmalloc(sizeof(struct pci_resource), GFP_KERNEL);

		if (!split_node)
			return(NULL);

		temp_dword = (node->length | (alignment-1)) + 1 - alignment;

		split_node->base = node->base;
		split_node->length = temp_dword;

		node->length -= temp_dword;
		node->base += split_node->length;

		// Put it in the list
		*head = split_node;
		split_node->next = node;
	}

	if (node->length < alignment) {
		return(NULL);
	}

	// Now unlink it
	if (*head == node) {
		*head = node->next;
		node->next = NULL;
	} else {
		prevnode = *head;
		while (prevnode->next != node)
			prevnode = prevnode->next;

		prevnode->next = node->next;
		node->next = NULL;
	}

	return(node);
}

