/*
 * Compaq Hot Plug Controller Driver
 *
 * Copyright (c) 1995,2001 Compaq Computer Corporation
 * Copyright (c) 2001 Greg Kroah-Hartman (greg@kroah.com)
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <linuxhotplug@compaq.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/pci.h>
#include "hotplug_pci.h"


/* A few routines that create proc entries for the hot plug controller */

static int read_ctrl (char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	struct controller *ctrl = (struct controller *)data;
	char * out = buf;
	int index;
	struct pci_resource *res;

	if (offset > 0)	return 0; /* no partial requests */
	len  = 0;
	*eof = 1;

	out += sprintf(out, "hot plug ctrl Info Page\n");
	out += sprintf(out, "bus = %d, device = %d, function = %d\n",ctrl->bus,
		       ctrl->device, ctrl->function);
	out += sprintf(out, "Free resources: memory\n");
	index = 11;
	res = ctrl->mem_head;
	while (res && index--) {
		out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
		res = res->next;
	}
	out += sprintf(out, "Free resources: prefetchable memory\n");
	index = 11;
	res = ctrl->p_mem_head;
	while (res && index--) {
		out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
		res = res->next;
	}
	out += sprintf(out, "Free resources: IO\n");
	index = 11;
	res = ctrl->io_head;
	while (res && index--) {
		out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
		res = res->next;
	}
	out += sprintf(out, "Free resources: bus numbers\n");
	index = 11;
	res = ctrl->bus_head;
	while (res && index--) {
		out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
		res = res->next;
	}

	*start = buf;
	len = out-buf;

	return len;
}


static int read_dev (char *buf, char **start, off_t offset, int len, int *eof, void *data)
{
	struct controller *ctrl = (struct controller *)data;
	char * out = buf;
	int index;
	struct pci_resource *res;
	struct pci_func *new_slot;
	struct slot_info *slot;

	if (offset > 0)	return 0; /* no partial requests */
	len  = 0;
	*eof = 1;

	out += sprintf(out, "hot plug ctrl Info Page\n");
	out += sprintf(out, "bus = %d, device = %d, function = %d\n",ctrl->bus,
		       ctrl->device, ctrl->function);

	slot=ctrl->slot;

	while (slot) {
		new_slot = pci_hp_slot_find(slot->bus, slot->device, 0);
		out += sprintf(out, "assigned resources: memory\n");
		index = 11;
		res = new_slot->mem_head;
		while (res && index--) {
			out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
			res = res->next;
		}
		out += sprintf(out, "assigned resources: prefetchable memory\n");
		index = 11;
		res = new_slot->p_mem_head;
		while (res && index--) {
			out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
			res = res->next;
		}
		out += sprintf(out, "assigned resources: IO\n");
		index = 11;
		res = new_slot->io_head;
		while (res && index--) {
			out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
			res = res->next;
		}
		out += sprintf(out, "assigned resources: bus numbers\n");
		index = 11;
		res = new_slot->bus_head;
		while (res && index--) {
			out += sprintf(out, "start = %8.8x, length = %8.8x\n", res->base, res->length);
			res = res->next;
		}
		slot=slot->next;
	}

	*start = buf;
	len = out-buf;

	return len;
}


/*
This creates a proc hierarchy similar to /proc/bus/pci. However, the pci entries
are deeply nested (if buses are present) and are slot oriented.
/bus/slot/dev/bus/dev/bus/dev/ etc.
Each directory has a .info structure that contains the relevant information.
*/


static struct irq_routing_table *pci_irq_routing_table; 
static int pci_irq_routing_table_len; 

static struct proc_dir_entry* pci_proc_root;

struct proc_creation {
	char name[16];
	struct proc_dir_entry* parent;
	struct proc_dir_entry* this;
	void* kernel_data;	/* kernel specific data (pci_dev, etc.) */
	int bus_devfn;		/* bus & device & function */
	int user_data;		/* If slot data is present, slot, otherwise zero. */
	struct proc_creation* next;
};

static struct proc_creation* slot_stack[MAX_PCI_SLOTS];
static struct proc_creation* nonslot_stack;
static int curr_slot;


static int push_proc_creation(struct proc_creation** stack, char* name, struct proc_dir_entry* parent) 
{
	struct proc_creation* new_creation;
	 
	new_creation = (struct proc_creation*) kmalloc(sizeof(struct proc_creation), GFP_KERNEL);
	if (new_creation == NULL) {
		err(__FUNCTION__": out of memory\n");
		return -ENOMEM;
	}
	memset(new_creation, 0, sizeof(struct proc_creation));

	strcpy(new_creation->name, name);
	new_creation->parent = parent;
	new_creation->next = *stack;

	*stack = new_creation;
	DBG("Adding %s\n", new_creation->name);
	return 0;
}


static struct proc_creation* pop_proc_creation(struct proc_creation** stack)
{
	struct proc_creation* old_creation = *stack;   
	if (!old_creation) return 0;
	*stack = (*stack)->next;
	DBG("Deleting %s, stack is now %p\n", old_creation->name, *stack);
	return old_creation;
}


static int proc_display_pci_dev(char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len=0;
	struct proc_creation* stack = (struct proc_creation*)data;
	char* new_page = page;

	struct pci_dev* dev = stack ? (struct pci_dev*) (stack->kernel_data) : NULL;
	int slot = stack->user_data;
	struct slot_info* slot_info = NULL;
	struct controller* ctrl= NULL;

	ctrl = pci_hp_get_controller_from_slot(slot);
	DBG("slot %d -> ctrl = %p\n", slot, ctrl);

	if (ctrl && dev) {
		DBG("bus, dev, fn = %d, %d, %d\n", dev->bus->number, PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));
		slot_info = find_slot_info(ctrl, PCI_SLOT(dev->devfn));
	}

	DBG("dev = %p\n", dev);

	new_page += sprintf(new_page, "Type                   %d\n", 2);
	if (dev) {
		int is_driver_loaded = 0;
		char name[32];

		strcpy(name, "-");
		if (dev->driver) {
			is_driver_loaded = 1;
			strcpy(name, "-");
		} else {
			struct pci_driver* drv = pci_dev_driver(dev);
			is_driver_loaded = (drv != NULL);
		}


		new_page += sprintf(new_page, "IsBridge               %x\n", dev->hdr_type);
		new_page += sprintf(new_page, "IsDriverLoaded         %d\n", is_driver_loaded);
		new_page += sprintf(new_page, "SupportedSpeedMode     %d\n", slot_info ? is_slot66mhz(slot_info) : 0);
		new_page += sprintf(new_page, "DeviceName             (%s)\n", dev->name);
		new_page += sprintf(new_page, "DriverName             (%s)\n", name);
		new_page += sprintf(new_page, "Vendor                 %x\n", dev->vendor);
		new_page += sprintf(new_page, "Device                 %x\n", dev->device);
		new_page += sprintf(new_page, "SubVendor              %x\n", dev->subsystem_vendor);
		new_page += sprintf(new_page, "SubDevice              %x\n", dev->subsystem_device);

		new_page += sprintf(new_page, "Bus                    ");
		if (dev->bus) new_page += sprintf(new_page, "%02d", dev->bus->number);
		new_page += sprintf(new_page, "\n");

		new_page += sprintf(new_page, "Device                 %02d\n", PCI_SLOT(dev->devfn));
		new_page += sprintf(new_page, "Function               %02d\n", PCI_FUNC(dev->devfn));
		new_page += sprintf(new_page, "Class                  %x\n", dev->class);
		new_page += sprintf(new_page, "CurrentSpeedMode       %d\n", ctrl ? ctrl->speed : 0);
	}

	len = new_page - page;

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;
	return len;
}


static int proc_display_slot (char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len=0;
	struct proc_creation* stack = (struct proc_creation*)data;
	char* new_page = page;
	int slot = stack->user_data;
	u8 bus = (u8) ((stack->bus_devfn) >> 8);
	u8 devfn = (u8) ((stack->bus_devfn) & 0xff);
	struct slot_info* slot_info = NULL;
	struct controller* ctrl = NULL;
	struct pci_dev* dev = pci_find_slot(bus, devfn);

	ctrl = pci_hp_get_controller_from_slot(slot);
	DBG("slot %d -> ctrl = %p\n", slot, ctrl);

	if (ctrl) {
		DBG("bus, dev, fn = %d, %d, %d\n", bus, PCI_SLOT(devfn), PCI_FUNC(devfn));
		slot_info = find_slot_info(ctrl, PCI_SLOT(devfn));
	}

	new_page += sprintf(new_page, "Type                   %d\n", 0);
	new_page += sprintf(new_page, "Number                 %d\n", slot);
	new_page += sprintf(new_page, "HotPluggable           %d\n", ctrl ? 1 : 0);
	new_page += sprintf(new_page, "Power                  %d\n", ctrl ? pci_hp_get_slot_enabled(slot_info, ctrl): -1);
	new_page += sprintf(new_page, "AttentionStatus        %d\n", ctrl ? pci_hp_get_attention_status(slot_info, ctrl) : -1);
	new_page += sprintf(new_page, "LatchStatus            %d\n", ctrl ? pci_hp_get_latch_status(slot_info, ctrl) : -1);
	new_page += sprintf(new_page, "AdapterPresence        %d\n", ctrl ? pci_hp_get_presence_status(slot_info, ctrl): (dev != 0));
	new_page += sprintf(new_page, "Bit32/64               %d\n", slot ? is_slot64bit(slot_info) : 0);

	len = new_page - page;

	if (len <= off+count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}


static int proc_display_pci_bus (char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len=0;
	struct proc_creation* stack = (struct proc_creation*)data;
	struct pci_bus* bus = (struct pci_bus*) (stack->kernel_data);
	int bus_nr = stack->user_data;
	char* new_page = page;

	if (bus) {
		new_page += sprintf(new_page, "Type                   %d\n", 1);
		new_page += sprintf(new_page, "Number                 %d\n", bus->number);
		new_page += sprintf(new_page, "CurrentSpeed           %d\n", 0);
		new_page += sprintf(new_page, "SupportedSpeed         %d\n", 0);

	} else {
		new_page += sprintf(new_page, "Type                   %d\n", 1);
		new_page += sprintf(new_page, "Number                 %d\n", bus_nr);
		new_page += sprintf(new_page, "CurrentSpeed           %d\n", 0);
		new_page += sprintf(new_page, "SupportedSpeed         %d\n", 0);
	}

	len = new_page - page;

	if (len <= off+count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}


static struct proc_dir_entry* proc_create_slot_helper(struct proc_creation** p_stack, int slot, u8 bus, u8 devfn, struct proc_dir_entry* parent, int no_dupes) {

	char name[16];
	struct proc_dir_entry* proc_slot = NULL;
	struct proc_dir_entry* proc_info;
	struct proc_creation* stack = *p_stack;

	sprintf(name, "%d", slot);

	if (no_dupes) {
		struct proc_creation* curr_stack = stack;
		while (curr_stack) {
			DBG("Checking creation_stack element %s, %p, %p\n", curr_stack->name, curr_stack->parent, curr_stack->kernel_data);
			if ((parent == curr_stack->parent) && (strcmp(name, curr_stack->name) == 0)) {
				DBG("Found creation_stack element with match in parent and name\n");
				return curr_stack->this;
			}
			curr_stack= curr_stack->next;
		}
	}

	/* First, create a new directory */
	if (push_proc_creation(p_stack, name, parent))
		goto exit;
	stack = *p_stack;
	proc_slot = proc_mkdir(name, parent);
	stack->this = proc_slot;

	/* Now, create the .info structure underneath it */
	if (push_proc_creation(p_stack, ".info", proc_slot))
		goto exit;
	stack = *p_stack;
	proc_info = create_proc_entry(".info", S_IFREG | S_IRUGO, proc_slot);
	stack->this = proc_info;
	stack->bus_devfn= (bus << 8) | devfn;
	stack->user_data= slot;
	proc_info->data = stack;
	proc_info->read_proc = &proc_display_slot;

exit:
	return proc_slot;
}


static struct proc_dir_entry* proc_create_pci_dev_helper(struct proc_creation** p_stack, int slot, struct pci_dev* dev, struct proc_dir_entry* parent, int no_dupes) {

	char name[16];
	struct proc_dir_entry* proc_dev = NULL;
	struct proc_dir_entry* proc_info;
	struct proc_creation* stack = *p_stack;

	if (no_dupes) {
		struct proc_creation* curr_stack = stack;
		while (curr_stack) {
			DBG("Checking creation_stack element %s, %p, %p\n", curr_stack->name, curr_stack->parent, curr_stack->kernel_data);
			if ((parent == curr_stack->parent) && (strcmp(name, curr_stack->name) == 0)) {
				DBG("Found creation_stack element with match in parent and name\n");
				return curr_stack->this;
			}
			curr_stack= curr_stack->next;
		}
	}

	/* First, create a new directory */
	sprintf(name, "%02x.%x", PCI_SLOT(dev->devfn), PCI_FUNC(dev->devfn));   
	if (push_proc_creation(p_stack, name, parent))
		goto exit;
	stack = *p_stack;
	proc_dev = proc_mkdir(name, parent);
	stack->this = proc_dev;

	/* Now, create the .info structure underneath it */
	if (push_proc_creation(p_stack, ".info", proc_dev))
		goto exit;
	stack = *p_stack;
	proc_info = create_proc_entry(".info", S_IFREG | S_IRUGO, proc_dev);
	stack->this = proc_info;
	stack->kernel_data = dev;
	stack->user_data= slot;
	proc_info->data = stack;
	proc_info->read_proc = &proc_display_pci_dev;

exit:
	return proc_dev;
}


static struct proc_dir_entry* proc_create_pci_bus_helper(struct proc_creation** p_stack, struct pci_bus* bus, int bus_nr, struct proc_dir_entry* parent, int no_dupes) {

	char name[16];
	struct proc_dir_entry* proc_bus = NULL;
	struct proc_dir_entry* proc_info;
	struct proc_creation* stack = *p_stack;

	sprintf(name, "%d", bus_nr);

	if (no_dupes) {
		struct proc_creation* curr_stack = stack;
		while (curr_stack) {
			DBG("Checking creation_stack element %s, %p, %p\n", curr_stack->name, curr_stack->parent, curr_stack->kernel_data);
			if ((parent == curr_stack->parent) && (strcmp(name, curr_stack->name) == 0)) {
				DBG("Found creation_stack element with match in parent and name\n");
				return curr_stack->this;
			}
			curr_stack= curr_stack->next;
		}
	}
	/* First, create a new directory */
	if (!bus) {
		DBG("bus = NULL!\n");
	} else {
		sprintf(name, "%d", bus->number);
	}
	if (push_proc_creation(p_stack, name, parent))
		goto exit;
	stack = *p_stack;
	proc_bus = proc_mkdir(name, parent);
	stack->this = proc_bus;

	/* Now, create the .info structure underneath it */
	if (push_proc_creation(p_stack, ".info", proc_bus))
		goto exit;
	stack = *p_stack;
	proc_info = create_proc_entry(".info", S_IFREG | S_IRUGO, proc_bus);
	stack->this = proc_info;
	stack->kernel_data = (bus == NULL) ? pci_hp_find_bus(bus_nr) : bus;
	stack->user_data = bus_nr;
	proc_info->data = stack;
	proc_info->read_proc = &proc_display_pci_bus;

exit:
	return proc_bus;
}


static int proc_visit_pci_dev (struct wrapped_pci_dev* wrapped_dev, struct wrapped_pci_bus* wrapped_bus) 
{
	struct pci_bus* bus = wrapped_bus->bus;
	struct pci_dev* dev = wrapped_dev->dev;

	struct proc_dir_entry* proc_parent = ((bus == NULL) || (wrapped_bus->data == NULL)) ? pci_proc_root : (struct proc_dir_entry*)(wrapped_bus->data);
	struct proc_dir_entry* proc_dev = proc_create_pci_dev_helper(&(slot_stack[curr_slot]), curr_slot, dev, proc_parent, 0);

	DBG("visit_pci_dev dev=%p bus=%p dev->bus=%p\n", dev, bus, dev->bus);
	wrapped_dev->data = proc_dev; 
	return 0;
}


static int proc_visit_pci_bus (struct wrapped_pci_bus* wrapped_bus, struct wrapped_pci_dev* wrapped_dev) 
{
	struct pci_bus* bus = wrapped_bus->bus;

	struct proc_dir_entry* proc_parent = ((bus == NULL) || (wrapped_dev->data == NULL)) ? pci_proc_root : (struct proc_dir_entry*) wrapped_dev->data;
	struct proc_dir_entry* proc_bus = proc_create_pci_bus_helper(&(slot_stack[curr_slot]), bus, bus->number, proc_parent, 0);

	wrapped_bus->data = proc_bus; 
	return 0;
}


static struct visit_linux_pci visit_functions = {
	pre_visit_linux_pci_bus:	proc_visit_pci_bus,
	visit_linux_pci_dev:		proc_visit_pci_dev,
};


static int proc_create_slot(u8 bus_nr, u8 devfn, u8 slot, struct pci_dev* dev) 
{
	struct proc_dir_entry* proc_bus; 
	struct proc_dir_entry* proc_slot; 
	struct wrapped_pci_dev wrapped_dev;
	struct wrapped_pci_bus wrapped_bus;
	struct pci_bus* bus = NULL;

	curr_slot = slot; //curr_slot is global; important for visit functions

	if (dev) {
		bus = dev->bus;
		devfn = dev->devfn;
	}

	//The bus proc entry is created. It is important to use the existing one, if we encountered this already
	proc_bus = proc_create_pci_bus_helper(&nonslot_stack, bus, bus_nr, pci_proc_root, 1); //has to weed out dupes
	DBG("proc_bus = %p\n", proc_bus);

	proc_slot = proc_create_slot_helper(&nonslot_stack, slot, bus_nr, devfn , proc_bus, 1);	//Again duplication is possible, needs weeding out

	if (!dev)
		return 0;

	memset(&wrapped_dev, 0, sizeof(struct wrapped_pci_dev));
	memset(&wrapped_bus, 0, sizeof(struct wrapped_pci_bus));

	wrapped_dev.dev = dev;
	wrapped_bus.bus = bus;
	wrapped_bus.data = proc_slot;

	pci_hp_linux_pci_visit_dev(&visit_functions, &wrapped_dev, &wrapped_bus);
	return 0;
}


static int proc_destroy_slot(u8 slot)
{
	struct proc_creation* stack_element;

	curr_slot=slot;

	while ((stack_element = pop_proc_creation(&(slot_stack[slot])))) {
		remove_proc_entry(stack_element->name, stack_element->parent);
	}
	return 0;
}


static int proc_create_slot_for_multi_func(int bus, int devfn, int slot, struct pci_dev* dev) 
{
	int j=0;
	int rc = proc_create_slot(bus, devfn, slot, dev); //First, the passed inpci_dev 
	DBG("initial proc create slot bus, dev, fn, dev (%x, %x, %x, %p)\n", bus, PCI_SLOT(devfn), PCI_FUNC(devfn), dev);
	//Does it have "sibling" functions ?
	for (j=0; j<8 ; j++) {
		struct pci_dev* temp = pci_find_slot(bus, (devfn & 0xf8) | j);
		DBG("pci_find_slot(%x, %x)\n", bus, (devfn & 0xf8)|j);
		if (temp && (temp != dev)) {
			DBG("Found MultiFunction bus, dev, fn (%x, %x, %x, %p)\n", bus, PCI_SLOT(devfn), j, temp);
			rc |= proc_create_slot(bus, (devfn & 0xf8) | j , slot, temp); //Now create for newly found pci_dev 
		}
	}
	return rc;
}


int pci_hp_proc_create_pci (void)
{
	int i;
	int rc;

	pci_irq_routing_table = pcibios_get_irq_routing_table();
	pci_irq_routing_table_len = (pci_irq_routing_table->size - sizeof(struct irq_routing_table)) / sizeof(struct irq_info);

	DBG(__FUNCTION__"\n");

	// Make sure I got at least one entry
	if (pci_irq_routing_table_len == 0) {
		if (pci_irq_routing_table != NULL)
			kfree(pci_irq_routing_table);
		return -1;
	}

	rc = push_proc_creation(&nonslot_stack, "cpqphpd", &proc_root);
	if (rc)
		return rc;
	pci_proc_root = proc_mkdir("cpqphpd", &proc_root);

	for (i= 0; i< pci_irq_routing_table_len; i++) {
		u8 bus   = pci_irq_routing_table->slots[i].bus;
		u8 devfn = pci_irq_routing_table->slots[i].devfn;
		u8 slot  = pci_irq_routing_table->slots[i].slot;
		DBG("bus, dev, fn, slot %d,%d,%d,%d\n", bus, PCI_SLOT(devfn), PCI_FUNC(devfn), slot);
		if (slot > 0) {
			struct pci_dev* dev = pci_find_slot(bus, devfn);
			proc_create_slot_for_multi_func(bus, devfn, slot, dev);
		}
	}

	return 0;
}


int pci_hp_proc_destroy_pci (void)
{
	u8 i;
	struct proc_creation* stack_element;

	for (i=0; i < MAX_PCI_SLOTS; i++) {
		proc_destroy_slot(i);
	}
	while ((stack_element = pop_proc_creation(&nonslot_stack))) {
		remove_proc_entry(stack_element->name, stack_element->parent);
	}

	if (pci_irq_routing_table != NULL)
		kfree(pci_irq_routing_table);
	return 0;
}


int pci_hp_proc_create_dev (struct pci_dev* dev) 
{
	int i;

	if (PCI_FUNC(dev->devfn) > 0)
		return 1;

	for (i= 0; i< pci_irq_routing_table_len; i++) {
		u8 bus   = pci_irq_routing_table->slots[i].bus;
		u8 devfn = pci_irq_routing_table->slots[i].devfn;
		u8 slot  = pci_irq_routing_table->slots[i].slot;
		DBG("bus, dev, fn, slot %d,%d,%d,%d\n", bus, PCI_SLOT(devfn), PCI_FUNC(devfn), slot);
		if ((bus == dev->bus->number) && (devfn == dev->devfn) && (slot > 0)) {
			proc_destroy_slot(slot); //First delete every occurence
			proc_create_slot_for_multi_func(bus, devfn, slot, dev);
		}
	}
	return 0;
}


int pci_hp_proc_destroy_dev (struct pci_dev* dev) 
{
	int i;

	if (PCI_FUNC(dev->devfn) > 0)
		return 1;

	for (i= 0; i< pci_irq_routing_table_len; i++) {
		u8 bus   = pci_irq_routing_table->slots[i].bus;
		u8 devfn = pci_irq_routing_table->slots[i].devfn;
		u8 slot  = pci_irq_routing_table->slots[i].slot;
		DBG("bus, dev, fn, slot %d,%d,%d,%d\n", bus, PCI_SLOT(devfn), PCI_FUNC(devfn), slot);
		if ((bus == dev->bus->number) && (devfn == dev->devfn) && (slot > 0)) {
			return proc_destroy_slot(slot);
		}
	}
	return 0;
}


int pci_hp_proc_create_ctrl (struct controller *ctrl)
{
	strcpy(ctrl->proc_name, "hpca");
	ctrl->proc_name[3] = 'a' + ctrl->bus;

	ctrl->proc_entry = create_proc_entry(ctrl->proc_name, S_IFREG | S_IRUGO, pci_hp_proc_root);
	ctrl->proc_entry->data = ctrl;
	ctrl->proc_entry->read_proc = &read_ctrl;

	strcpy(ctrl->proc_name2, "slot_a");
	ctrl->proc_name2[5] = 'a' + ctrl->bus;
	ctrl->proc_entry2 = create_proc_entry(ctrl->proc_name2, S_IFREG | S_IRUGO, pci_hp_proc_root);
	ctrl->proc_entry2->data = ctrl;
	ctrl->proc_entry2->read_proc = &read_dev;

	return 0;
}

