/*
 * Compaq Hot Plug Controller Driver
 *
 * Copyright (c) 1995,2001 Compaq Computer Corporation
 * Copyright (c) 2001 Greg Kroah-Hartman (greg@kroah.com)
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <linuxhotplug@compaq.com>
 *
 */
#ifndef _CPQPHPD
#define _CPQPHPD

#include <asm/io.h>		/* for read? and write? functions */


/*#if !defined(CONFIG_HOTPLUG_PCI_COMPAQ_MODULE)*/
	#define MY_NAME	"hotplug_pci.o"
/*#else
	#define MY_NAME	THIS_MODULE->name
#endif*/

#define DBG(fmt, arg...) do { if (pci_hp_debug) printk(KERN_DEBUG "%s: " fmt , MY_NAME , ## arg); } while (0)
#define err(format, arg...) printk(KERN_ERR "%s: " format , MY_NAME , ## arg)
#define info(format, arg...) printk(KERN_INFO "%s: " format , MY_NAME , ## arg)
#define warn(format, arg...) printk(KERN_WARNING "%s: " format , MY_NAME , ## arg)



struct smbios_system_slot {
	u8 type;
	u8 length;
	u16 handle;
	u8 name_string_num;
	u8 slot_type;
	u8 slot_width;
	u8 slot_current_usage;
	u8 slot_length;
	u16 slot_number;
	u8 properties1;
	u8 properties2;
} __attribute__ ((packed));

/* offsets to the smbios generic type based on the above structure layout */
enum smbios_system_slot_offsets {
	SMBIOS_SLOT_GENERIC_TYPE =	offsetof(struct smbios_system_slot, type),
	SMBIOS_SLOT_GENERIC_LENGTH =	offsetof(struct smbios_system_slot, length),
	SMBIOS_SLOT_GENERIC_HANDLE =	offsetof(struct smbios_system_slot, handle),
	SMBIOS_SLOT_NAME_STRING_NUM =	offsetof(struct smbios_system_slot, name_string_num),
	SMBIOS_SLOT_TYPE =		offsetof(struct smbios_system_slot, slot_type),
	SMBIOS_SLOT_WIDTH =		offsetof(struct smbios_system_slot, slot_width),
	SMBIOS_SLOT_CURRENT_USAGE =	offsetof(struct smbios_system_slot, slot_current_usage),
	SMBIOS_SLOT_LENGTH =		offsetof(struct smbios_system_slot, slot_length),
	SMBIOS_SLOT_NUMBER =		offsetof(struct smbios_system_slot, slot_number),
	SMBIOS_SLOT_PROPERTIES1 =	offsetof(struct smbios_system_slot, properties1),
	SMBIOS_SLOT_PROPERTIES2 =	offsetof(struct smbios_system_slot, properties2),
};

struct smbios_generic {
	u8 type;
	u8 length;
	u16 handle;
} __attribute__ ((packed));

/* offsets to the smbios generic type based on the above structure layout */
enum smbios_generic_offsets {
	SMBIOS_GENERIC_TYPE =	offsetof(struct smbios_generic, type),
	SMBIOS_GENERIC_LENGTH =	offsetof(struct smbios_generic, length),
	SMBIOS_GENERIC_HANDLE =	offsetof(struct smbios_generic, handle),
};

struct smbios_entry_point {
	char anchor[4];
	u8 ep_checksum;
	u8 ep_length;
	u8 major_version;
	u8 minor_version;
	u16 max_size_entry;
	u8 ep_rev;
	u8 reserved[5];
	char int_anchor[5];
	u8 int_checksum;
	u16 st_length;
	u32 st_address;
	u16 number_of_entrys;
	u8 bcd_rev;
} __attribute__ ((packed));

/* offsets to the smbios entry point based on the above structure layout */
enum smbios_entry_point_offsets {
	ANCHOR =		offsetof(struct smbios_entry_point, anchor[0]),
	EP_CHECKSUM =		offsetof(struct smbios_entry_point, ep_checksum),
	EP_LENGTH =		offsetof(struct smbios_entry_point, ep_length),
	MAJOR_VERSION =		offsetof(struct smbios_entry_point, major_version),
	MINOR_VERSION =		offsetof(struct smbios_entry_point, minor_version),
	MAX_SIZE_ENTRY =	offsetof(struct smbios_entry_point, max_size_entry),
	EP_REV =		offsetof(struct smbios_entry_point, ep_rev),
	INT_ANCHOR =		offsetof(struct smbios_entry_point, int_anchor[0]),
	INT_CHECKSUM =		offsetof(struct smbios_entry_point, int_checksum),
	ST_LENGTH =		offsetof(struct smbios_entry_point, st_length),
	ST_ADDRESS =		offsetof(struct smbios_entry_point, st_address),
	NUMBER_OF_ENTRYS =	offsetof(struct smbios_entry_point, number_of_entrys),
	BCD_REV =		offsetof(struct smbios_entry_point, bcd_rev),
};

struct pci_func {
	struct pci_func *next;
	u8 bus;
	u8 device;
	u8 function;
	u8 is_a_board;
	u16 status;
	u8 configured;
	u8 switch_save;
	u8 presence_save;
	u32 base_length[0x06];
	u8 base_type[0x06];
	u16 reserved2;
	u32 config_space[0x20];
	struct pci_resource *mem_head;
	struct pci_resource *p_mem_head;
	struct pci_resource *io_head;
	struct pci_resource *bus_head;
	struct timer_list *p_task_event;
	struct pci_dev* pci_dev;
};

struct slot_info {
	struct slot_info *next;
	u8 bus;
	u8 device;
	u8 number;
	u8 is_a_board;
	u8 configured;
	u8 state;
	u8 switch_save;
	u8 presence_save;
	u32 capabilities;
	u16 reserved2;
	struct timer_list task_event;
	u8 hp_slot;
	struct controller *ctrl;
	void *p_sm_slot;
};

struct ctrl_reg {			/* offset */
	u8	slot_RST;		/* 0x00 */
	u8	slot_enable;		/* 0x01 */
	u16	misc;			/* 0x02 */
	u32	led_control;		/* 0x04 */
	u32	int_input_clear;	/* 0x08 */
	u32	int_mask;		/* 0x0a */
	u8	reserved0;		/* 0x10 */
	u8	reserved1;		/* 0x11 */
	u8	reserved2;		/* 0x12 */
	u8	gen_output_AB;		/* 0x13 */
	u32	non_int_input;		/* 0x14 */
	u32	reserved3;		/* 0x18 */
	u32	reserved4;		/* 0x1a */
	u32	reserved5;		/* 0x20 */
	u8	reserved6;		/* 0x24 */
	u8	reserved7;		/* 0x25 */
	u16	reserved8;		/* 0x26 */
	u8	slot_mask;		/* 0x28 */
	u8	reserved9;		/* 0x29 */
	u8	reserved10;		/* 0x2a */
	u8	reserved11;		/* 0x2b */
	u8	slot_SERR;		/* 0x2c */
	u8	slot_power;		/* 0x2d */
} __attribute__ ((packed));

/* offsets to the controller registers based on the above structure layout */
enum ctrl_offsets {
	SLOT_RST = 		offsetof(struct ctrl_reg, slot_RST),
	SLOT_ENABLE =		offsetof(struct ctrl_reg, slot_enable),
	MISC =			offsetof(struct ctrl_reg, misc),
	LED_CONTROL =		offsetof(struct ctrl_reg, led_control),
	INT_INPUT_CLEAR =	offsetof(struct ctrl_reg, int_input_clear),
	INT_MASK = 		offsetof(struct ctrl_reg, int_mask),
	CTRL_RESERVED0 = 	offsetof(struct ctrl_reg, reserved0),
	CTRL_RESERVED1 =	offsetof(struct ctrl_reg, reserved1),
	CTRL_RESERVED2 =	offsetof(struct ctrl_reg, reserved1),
	GEN_OUTPUT_AB = 	offsetof(struct ctrl_reg, gen_output_AB),
	NON_INT_INPUT = 	offsetof(struct ctrl_reg, non_int_input),
	CTRL_RESERVED3 =	offsetof(struct ctrl_reg, reserved3),
	CTRL_RESERVED4 =	offsetof(struct ctrl_reg, reserved4),
	CTRL_RESERVED5 =	offsetof(struct ctrl_reg, reserved5),
	CTRL_RESERVED6 =	offsetof(struct ctrl_reg, reserved6),
	CTRL_RESERVED7 =	offsetof(struct ctrl_reg, reserved7),
	CTRL_RESERVED8 =	offsetof(struct ctrl_reg, reserved8),
	SLOT_MASK = 		offsetof(struct ctrl_reg, slot_mask),
	CTRL_RESERVED9 = 	offsetof(struct ctrl_reg, reserved9),
	CTRL_RESERVED10 =	offsetof(struct ctrl_reg, reserved10),
	CTRL_RESERVED11 =	offsetof(struct ctrl_reg, reserved11),
	SLOT_SERR =		offsetof(struct ctrl_reg, slot_SERR),
	SLOT_POWER =		offsetof(struct ctrl_reg, slot_power),
};

struct hrt {
	char sig0;
	char sig1;
	char sig2;
	char sig3;
	u16 unused_IRQ;
	u16 PCIIRQ;
	u8 number_of_entries;
	u8 revision;
	u16 reserved1;
	u32 reserved2;
} __attribute__ ((packed));

/* offsets to the hotplug resource table registers based on the above structure layout */
enum hrt_offsets {
	SIG0 =			offsetof(struct hrt, sig0),
	SIG1 =			offsetof(struct hrt, sig1),
	SIG2 =			offsetof(struct hrt, sig2),
	SIG3 =			offsetof(struct hrt, sig3),
	UNUSED_IRQ =		offsetof(struct hrt, unused_IRQ),
	PCIIRQ =		offsetof(struct hrt, PCIIRQ),
	NUMBER_OF_ENTRIES =	offsetof(struct hrt, number_of_entries),
	REVISION =		offsetof(struct hrt, revision),
	HRT_RESERVED1 =		offsetof(struct hrt, reserved1),
	HRT_RESERVED2 =		offsetof(struct hrt, reserved2),
};

struct pci_resource {
	struct pci_resource * next;
	u32 base;
	u32 length;
};

struct event_info {
	struct event_info *next;
	struct pci_func *func;
	struct controller *ctrl;
	u32 event_type;
	u8 hp_slot;
	u32 Time;
};

struct controller {
	struct controller *next;
	u32 ctrl_int_comp;
	struct semaphore crit_sect;	/* critical section semaphore */
	void *hpc_reg;			/* cookie for our pci controller location */
	struct pci_resource *mem_head;
	struct pci_resource *p_mem_head;
	struct pci_resource *io_head;
	struct pci_resource *bus_head;
	struct pci_dev* pci_dev;
	struct proc_dir_entry* proc_entry;
	struct proc_dir_entry* proc_entry2;
	struct event_info event_queue[10];
	struct slot_info *slot;
	u8 next_event;
	u8 interrupt;
	u8 bus;
	u8 device;
	u8 function;
	u8 rev;
	u8 slot_device_offset;
	u8 first_slot;
	u8 add_support;
	u8 speed;
	u8 speed_capability;		/* 0 = 33MHz, 1 = 66MHz */
	u8 push_button;			/* 0 = no pushbutton, 1 = pushbutton present */
	u8 slot_switch_type;		/* 0 = no switch, 1 = switch present */
	u8 defeature_PHP;		/* 0 = PHP not supported, 1 = PHP supported */
	u8 alternate_base_address;	/* 0 = not supported, 1 = supported */
	u8 pci_config_space;		/* Index/data access to working registers 0 = not supported, 1 = supported */
	u8 pcix_speed_capability;	/* PCI-X */
	u8 pcix_support;		/* PCI-X */
	u8 push_flag;
	u16 vendor_id;
	char proc_name[8];
	char proc_name2[8];
	struct tq_struct int_task_event;
	wait_queue_head_t queue;	/* sleep & wake process */
};

#define CTRL_SPEED_33MHz	0
#define CTRL_SPEED_66MHz	1

struct irq_mapping {
	u8 barber_pole;
	u8 valid_INT;
	u8 interrupt[4];
};

struct resource_lists {
	struct pci_resource *mem_head;
	struct pci_resource *p_mem_head;
	struct pci_resource *io_head;
	struct pci_resource *bus_head;
	struct irq_mapping *irqs;
};

struct slot_rt {
	u8 dev_func;
	u8 primary_bus;
	u8 secondary_bus;
	u8 max_bus;
	u16 io_base;
	u16 io_length;
	u16 mem_base;
	u16 mem_length;
	u16 pre_mem_base;
	u16 pre_mem_length;
};


struct ev_hrt_header {
	u8 Version;
	u8 num_of_ctrl;
	u8 next;
};

struct ev_hrt_ctrl {
	u8 bus;
	u8 device;
	u8 function;
	u8 mem_avail;
	u8 p_mem_avail;
	u8 io_avail;
	u8 bus_avail;
	u8 next;
};

#pragma pack(1)
struct irq_routing_options {
		  u16 size;
		  struct irq_info *table;
		  u16 segment;
};

struct irq_info {
        u8 bus, devfn;                  /* bus, device and function */
        struct {
                u8 link;                /* IRQ line ID, chipset dependent, 0=not routed */
                u16 bitmap;             /* Available IRQs */
        }irq[4];
        u8 slot;                        /* slot number, 0=onboard */
        u8 rfu;
}; 

struct irq_routing_table {
        u32 signature;                  /* PIRQ_SIGNATURE should be here */
        u16 version;                    /* PIRQ_VERSION */
        u16 size;                       /* Table size in bytes */
        u8 rtr_bus, rtr_devfn;          /* Where the interrupt router lies */
        u16 exclusive_irqs;             /* IRQs devoted exclusively to PCI usage */
        u16 rtr_vendor, rtr_device;     /* Vendor and device ID of interrupt router */
        u32 miniport_data;              /* Crap */
        u8 rfu[11];
        u8 checksum;                    /* Modulo 256 checksum must give zero */
        struct irq_info slots[0];
} ;
#pragma pack()


#define ROM_PHY_ADDR			0x0F0000
#define ROM_PHY_LEN			0x00ffff

#define PCI_HPC_ID			0xA0F7
#define PCI_SUB_HPC_ID			0xA2F7
#define PCI_SUB_HPC_ID2			0xA2F8
#define PCI_SUB_HPC_ID3			0xA2F9
#define PCI_SUB_HPC_ID_INTC		0xA2FA

#define INT_BUTTON_IGNORE		0
#define INT_PRESENCE_ON			1
#define INT_PRESENCE_OFF		2
#define INT_SWITCH_CLOSE		3
#define INT_SWITCH_OPEN			4
#define INT_POWER_FAULT			5
#define INT_POWER_FAULT_CLEAR		6
#define INT_BUTTON_PRESS		7
#define INT_BUTTON_RELEASE		8
#define INT_BUTTON_CANCEL		9

#define STATIC_STATE			0
#define BLINKINGON_STATE		1
#define BLINKINGOFF_STATE		2
#define POWERON_STATE			3
#define POWEROFF_STATE			4

#define PCISLOT_INTERLOCK_CLOSED	0x00000001
#define PCISLOT_ADAPTER_PRESENT		0x00000002
#define PCISLOT_POWERED			0x00000004
#define PCISLOT_66_MHZ_OPERATION	0x00000008
#define PCISLOT_64_BIT_OPERATION	0x00000010
#define PCISLOT_REPLACE_SUPPORTED	0x00000020
#define PCISLOT_ADD_SUPPORTED		0x00000040
#define PCISLOT_INTERLOCK_SUPPORTED	0x00000080
#define PCISLOT_66_MHZ_SUPPORTED	0x00000100
#define PCISLOT_64_BIT_SUPPORTED	0x00000200


/* pci_bios command codes */
#define	READ_CONFIG_BYTE		0x08
#define	READ_CONFIG_WORD		0x09
#define	READ_CONFIG_DWORD		0x0A
#define	WRITE_CONFIG_BYTE		0x0B
#define	WRITE_CONFIG_WORD		0x0C
#define	WRITE_CONFIG_DWORD		0x0D


// If using a hardware mechanism for accessing PCI config space, we need
//   to know that we are on a supported platform.
#define SYS_ID_REG			0x0C80

//
// PCI port Addresses
//
// Used for H/W mechanism #1
#define PCI_CFG_ADDRESS			0x0CF8
#define PCI_CFG_DATA			0x0CFC

// Used for H/W mechanism #2
#define PCI_CSE_REGISTER		0x0CF8	// Config Space Enable Register
#define PCI_FWD_REGISTER		0x0CFA	// Forward Register (choose bus #)
#define PCI_CFG_IO_BASE			0xC000

#define PCI_MECH2_FUNC0_ENABLE		0x10
#define PCI_MECH2_FUNC1_ENABLE		0x12

#define PCI_TO_PCI_BRIDGE_CLASS		0x00060400


#define INTERLOCK_OPEN			0x00000002
#define ADD_NOT_SUPPORTED		0x00000003
#define CARD_FUNCTIONING		0x00000005
#define ADAPTER_NOT_SAME		0x00000006
#define NO_ADAPTER_PRESENT		0x00000009
#define NOT_ENOUGH_RESOURCES		0x0000000B
#define DEVICE_TYPE_NOT_SUPPORTED	0x0000000C
#define POWER_FAILURE			0x0000000E

#define REMOVE_NOT_SUPPORTED		0x00000003


/*
 * error Messages
 */
#define msg_initialization_err	"Initialization failure, error=%d\n"
#define msg_power_fault		"slot power fault detected\n"
#define msg_HPC_rev_error	"Unsupported revision of the PCI hot plug controller found.\n"
#define msg_HPC_non_compaq_or_intel	"The PCI hot plug controller is not supported by this driver.\n"
#define msg_HPC_not_supported	"this system is not supported by this version of cpqphpd. Upgrade to a newer version of cpqphpd\n"
#define msg_unable_to_save	"unable to store PCI hot plug add resource information. This system must be rebooted before adding any PCI devices.\n"
#define msg_button_on		"PCI slot #%d - powering on due to button press.\n"
#define msg_button_off		"PCI slot #%d - powering off due to button press.\n"
#define msg_button_cancel	"PCI slot #%d - action canceled due to button press.\n"
#define msg_button_ignore	"PCI slot #%d - button press ignored.  (action in progress...)\n"



struct wrapped_pci_dev {
	struct pci_dev* dev;
	void* data;
};

struct wrapped_pci_bus {
	struct pci_bus* bus;
	void* data;
};

typedef int (*fn_visit_linux_pci_bus) (struct wrapped_pci_bus *, struct wrapped_pci_dev *);
typedef int (*fn_visit_linux_pci_dev) (struct wrapped_pci_dev *, struct wrapped_pci_bus *);

struct visit_linux_pci {
	fn_visit_linux_pci_bus pre_visit_linux_pci_bus;
	fn_visit_linux_pci_bus post_visit_linux_pci_bus;

	fn_visit_linux_pci_dev pre_visit_linux_pci_dev;
	fn_visit_linux_pci_dev visit_linux_pci_dev;
	fn_visit_linux_pci_dev post_visit_linux_pci_dev;
};


/* Global proc functions */
extern int pci_hp_proc_create_pci	(void);
extern int pci_hp_proc_destroy_pci	(void);
extern int pci_hp_proc_destroy_dev	(struct pci_dev* dev);
extern int pci_hp_proc_create_dev	(struct pci_dev* dev);
extern int pci_hp_proc_create_ctrl	(struct controller *ctrl);

/* Global support functions */
extern int	pci_hp_linux_pci_visit_dev	(struct visit_linux_pci * fn, struct wrapped_pci_dev *wrapped_dev, struct wrapped_pci_bus *wrapped_parent);
extern int	pci_hp_register_user_notifier	(int fd, struct file* filp);
extern int	pci_hp_unregister_user_notifier	(int fd, struct file* filp);
extern void	pci_hp_notify_id 		(int id);
extern int	pci_hp_start_notify		(void);
extern void	pci_hp_stop_notify		(void);
extern ssize_t	pci_hp_read_helper		(struct file* filp, char* buf, size_t count, loff_t *ppos);
extern void	pci_hp_open_helper		(struct file* filep);

extern u32	pci_hp_valid_replace		(struct pci_func * func);


/* Global controller functions */
extern void	pci_hp_pushbutton_thread	(unsigned long event_pointer);
extern void	pci_hp_ctrl_intr		(int IRQ, struct controller *ctrl_input, struct pt_regs *regs);
extern int	pci_hp_find_available_resources (struct controller *ctrl, void *pci_hp_rom_start);
extern int	pci_hp_ctrl_slot_setup		(struct controller *ctrl, void *smbios_table);
extern int	pci_hp_ctrl_slot_cleanup	(struct controller *ctrl);
extern int	pci_hp_get_slot_enabled		(struct slot_info *slot, struct controller *ctrl);
extern int	pci_hp_get_attention_status	(struct slot_info *slot, struct controller *ctrl);
extern int	pci_hp_get_latch_status		(struct slot_info *slot, struct controller *ctrl);
extern u8	pci_hp_get_presence_status	(struct slot_info *slot, struct controller *ctrl);
extern struct controller *	pci_hp_get_controller_from_slot	(int slot);
extern struct pci_func *	pci_hp_slot_create		(unsigned char busnumber);
extern struct pci_func *	pci_hp_slot_find		(unsigned char bus,unsigned char device,unsigned char index);
extern void *	pci_hp_get_SMBIOS_entry		(void *smbios_table, u8 type, void *previous);
extern int	pci_hp_event_start_thread	(void);
extern void	pci_hp_event_stop_thread	(void);

/* Global resource functions */
extern struct pci_resource *pci_hp_get_max_resource	(struct pci_resource **head,u32 size);
extern struct pci_resource *pci_hp_get_resource		(struct pci_resource **head,u32 size);
extern struct pci_resource *pci_hp_get_io_resource	(struct pci_resource **head,u32 size);
extern struct pci_resource *pci_hp_do_bridge_resource_split	(struct pci_resource **head,u32 alignment);
extern struct pci_resource *pci_hp_do_pre_bridge_resource_split	(struct pci_resource **head, struct pci_resource **orig_head,u32 alignment);
extern u32 pci_hp_sort_and_combine	(struct pci_resource **head);


/* Global pci functions */
extern struct pci_bus *	pci_hp_find_bus		(int bus_nr);
extern int	pci_hp_set_irq			(u8 bus_num, u8 dev_num, u8 int_pin, u8 irq_num);
extern long	pci_hp_get_bus_dev		(u8 *bus_num, u8 *dev_num, u8 slot);
extern int	pci_hp_save_config		(int busnumber, int is_hot_plug);
extern int	pci_hp_save_base_addr_length	(struct pci_func * func);
extern int	pci_hp_save_used_resources	(struct pci_func * func);
extern int	pci_hp_configure_board		(struct pci_func * func);
extern int	pci_hp_save_slot_config		(struct pci_func * new_slot);
extern void	pci_hp_destroy_board_resources	(struct pci_func * func);
extern u32	pci_hp_return_board_resources	(struct pci_func * func, struct resource_lists * resources);
extern void	pci_hp_destroy_resource_list	(struct resource_lists * resources);
extern int	pci_hp_configure_device		(struct controller* ctrl, struct pci_func* func);
extern int	pci_hp_unconfigure_device	(struct pci_func* func);


/* Global variables */
extern int pci_hp_debug;
extern struct controller *pci_hp_ctrl_list;
extern void *pci_hp_rom_start;
extern void *pci_hp_smbios_start;
extern struct pci_dev* pci_hp_pci_dev;
extern struct pci_func *pci_hp_slot_list[256];
extern struct proc_dir_entry* pci_hp_proc_root;

/* these can be gotten rid of, but for debugging they are purty */
extern u8 pci_hp_nic_irq;
extern u8 pci_hp_disk_irq;



/* inline functions */

static inline u8 is_slot64bit (struct slot_info *slot)
{
	if (!slot || !slot->p_sm_slot)
		return 0;

	if (readb(slot->p_sm_slot + SMBIOS_SLOT_WIDTH) == 0x06)
		return 1;

	return 0;
}


static inline u8 is_slot66mhz (struct slot_info *slot)
{
	if (!slot || !slot->p_sm_slot)
		return 0;

	if (readb(slot->p_sm_slot + SMBIOS_SLOT_TYPE) == 0x0E)
		return 1;

	return 0;
}


/*
 * return_resource
 *
 * Puts node back in the resource list pointed to by head
 *
 */
static inline void return_resource (struct pci_resource **head, struct pci_resource *node)
{
	if (!node || !head)
		return;
	node->next = *head;
	*head = node;
}


/*
 * find_slot_info
 */
static inline struct slot_info *find_slot_info (struct controller * ctrl, u8 device)
{
	struct slot_info *slot;

	if (!ctrl)
		return NULL;

	slot = ctrl->slot;

	while (slot && (slot->device != device)) {
		slot = slot->next;
	}

	return slot;
}


static inline void set_SOGO (struct controller *ctrl)
{
	u16 misc;
	
	misc = readw(ctrl->hpc_reg + MISC);
	misc = (misc | 0x0001) & 0xFFFB;
	writew(misc, ctrl->hpc_reg + MISC);
}


static inline void amber_LED_on (struct controller *ctrl, u8 slot)
{
	u32 led_control;
	
	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control |= (0x01010000L << slot);
	writel(led_control, ctrl->hpc_reg + LED_CONTROL);
}


static inline void amber_LED_off (struct controller *ctrl, u8 slot)
{
	u32 led_control;
	
	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control &= ~(0x01010000L << slot);
	writel(led_control, ctrl->hpc_reg + LED_CONTROL);
}


static inline int read_amber_LED (struct controller *ctrl, u8 slot)
{
	u32 led_control;

	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control &= (0x01010000L << slot);
	
	return led_control ? 1 : 0;
}


static inline void green_LED_on (struct controller *ctrl, u8 slot)
{
	u32 led_control;
	
	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control |= 0x0101L << slot;
	writel(led_control, ctrl->hpc_reg + LED_CONTROL);
}

static inline void green_LED_off (struct controller *ctrl, u8 slot)
{
	u32 led_control;
	
	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control &= ~(0x0101L << slot);
	writel(led_control, ctrl->hpc_reg + LED_CONTROL);
}


static inline void green_LED_blink (struct controller *ctrl, u8 slot)
{
	u32 led_control;
	
	led_control = readl(ctrl->hpc_reg + LED_CONTROL);
	led_control |= (0x0001L << slot);
	writel(led_control, ctrl->hpc_reg + LED_CONTROL);
}


static inline void slot_disable (struct controller *ctrl, u8 slot)
{
	u8 slot_enable;

	slot_enable = readb(ctrl->hpc_reg + SLOT_ENABLE);
	slot_enable &= ~(0x01 << slot);
	writeb(slot_enable, ctrl->hpc_reg + SLOT_ENABLE);
}


static inline void slot_enable (struct controller *ctrl, u8 slot)
{
	u8 slot_enable;

	slot_enable = readb(ctrl->hpc_reg + SLOT_ENABLE);
	slot_enable |= (0x01 << slot);
	writeb(slot_enable, ctrl->hpc_reg + SLOT_ENABLE);
}


static inline u8 is_slot_enabled (struct controller *ctrl, u8 slot)
{
	u8 slot_enable;

	slot_enable = readb(ctrl->hpc_reg + SLOT_ENABLE);
	slot_enable &= (0x01 << slot);
	return slot_enable ? 1 : 0;
}


static inline u8 read_slot_enable (struct controller *ctrl)
{
	return readb(ctrl->hpc_reg + SLOT_ENABLE);
}


static inline u8 get_controller_speed (struct controller *ctrl)
{
	u16 misc;
	
	misc = readw(ctrl->hpc_reg + MISC);
	return (misc & 0x0800) ? 1 : 0;
}


static inline void enable_slot_power (struct controller *ctrl, u8 slot)
{
	u8 slot_power;

	slot_power = readb(ctrl->hpc_reg + SLOT_POWER);
	slot_power |= (0x01 << slot);
	writeb(slot_power, ctrl->hpc_reg + SLOT_POWER);
}

static inline void disable_slot_power (struct controller *ctrl, u8 slot)
{
	u8 slot_power;

	slot_power = readb(ctrl->hpc_reg + SLOT_POWER);
	slot_power &= ~(0x01 << slot);
	writeb(slot_power, ctrl->hpc_reg + SLOT_POWER);
}

extern int pci_access_config(u8 command, u8 bus, u8 device, u8 function, u8 offset, void *value);
extern int pci_get_slot(u8 bus_num, u8 dev_num, u8 *slot);



/* Maximum number of PCI Slots the system can have */
#define MAX_PCI_SLOTS		64

/* these functions live in arch/i386/kernel/pci-pc.c */
extern int pcibios_set_irq_routing(struct pci_dev *dev, int pin, int irq);
extern struct irq_routing_table *pcibios_get_irq_routing_table(void);

#endif

