/*
 * Compaq Hot Plug Controller Driver
 *
 * Copyright (c) 1995,2001 Compaq Computer Corporation
 * Copyright (c) 2001 Greg Kroah-Hartman (greg@kroah.com)
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, GOOD TITLE or
 * NON INFRINGEMENT.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send feedback to <linuxhotplug@compaq.com>
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <linux/miscdevice.h>
#include <linux/slab.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include "hotplug_pci.h"
#include "hotplug_pci_ioctl.h"


/* Global variables */
int pci_hp_debug;
struct pci_dev *pci_hp_pci_dev;
struct controller *pci_hp_ctrl_list;	/* = NULL */
struct pci_func *pci_hp_slot_list[256];
void *pci_hp_rom_start;
void *pci_hp_smbios_start;
struct proc_dir_entry* pci_hp_proc_root;

/* local variables */
static void *smbios_table;
static u8 power_mode;
static int debug;

#define DRIVER_VERSION	"0.9.6"
#define DRIVER_AUTHOR	"Dan Zink <dan.zink@compaq.com>, Greg Kroah-Hartman <greg@kroah.com>"
#define DRIVER_DESC	"Compaq Hot Plug PCI Controller Driver"

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_PARM(power_mode, "b");
MODULE_PARM_DESC(power_mode, "Power mode enabled or not");

MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Debugging mode enabled or not");

#define CPQHPC_MODULE_MINOR 208

static int one_time_init	(void);


/**
 * detect_SMBIOS_pointer - find the system Management BIOS Table in the specified region of memory.
 *
 * @begin: begin pointer for region to be scanned.
 * @end: end pointer for region to be scanned.
 *
 * Returns pointer to the head of the SMBIOS tables (or NULL)
 *
 */
static void * detect_SMBIOS_pointer(void *begin, void *end)
{
	void *fp;
	void *endp;
	u8 temp1, temp2, temp3, temp4;
	int status = 0;

	endp = (end - sizeof(u32) + 1);

	for (fp = begin; fp <= endp; fp += 16) {
		temp1 = readb(fp);
		temp2 = readb(fp+1);
		temp3 = readb(fp+2);
		temp4 = readb(fp+3);
		if (temp1 == '_' &&
		    temp2 == 'S' &&
		    temp3 == 'M' &&
		    temp4 == '_') {
			status = 1;
			break;
		}
	}
	
	if (!status)
		fp = NULL;

	DBG("Discovered SMBIOS Entry point at %p\n", fp);

	return fp;
}

/**
 * init_SERR - Initializes the per slot SERR generation.
 *
 * For unexpected switch opens
 *
 */
static int init_SERR(struct controller * ctrl)
{
	u32 tempdword;
	u32 number_of_slots;
	u8 physical_slot;

	if (!ctrl)
		return 1;

	tempdword = ctrl->first_slot;

	number_of_slots = readb(ctrl->hpc_reg + SLOT_MASK) & 0x0F;
	// Loop through slots
	while (number_of_slots) {
		physical_slot = tempdword;
		writeb(0, ctrl->hpc_reg + SLOT_SERR);
		tempdword++;
		number_of_slots--;
	}

	return 0;
}


/* nice debugging output */
static int pci_print_IRQ_route (void)
{
	struct irq_routing_table *routing_table;
	int len;
	int loop;

	u8 tbus, tdevice, tslot;

	routing_table = pcibios_get_irq_routing_table();
	if (routing_table == NULL) {
		err("No BIOS Routing Table??? Not good\n");
		return -ENOMEM;
	}

	len = (routing_table->size - sizeof(struct irq_routing_table)) / sizeof(struct irq_info);
	// Make sure I got at least one entry
	if (len == 0) {
		kfree(routing_table);
		return -1;
	}

	DBG("bus dev func slot\n");

	for (loop = 0; loop < len; ++loop) {
		tbus = routing_table->slots[loop].bus;
		tdevice = routing_table->slots[loop].devfn;
		tslot = routing_table->slots[loop].slot;
		DBG("%d %d %d %d\n", tbus, tdevice >> 3, tdevice & 0x7, tslot);

	}
	kfree(routing_table);
	return 0;
}


static int cpqhpc_probe(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	u8 num_of_slots = 0;
	u8 hp_slot = 0;
	u8 device;
	u8 rev;
	u16 temp_word;
	u16 vendor_id;
	u16 subsystem_vid;
	u16 subsystem_deviceid;
	u32 rc;
	struct controller *ctrl;
	struct pci_func *func;

	pci_hp_pci_dev = pdev;

	// Need to read VID early b/c it's used to differentiate CPQ and INTC discovery
	rc = pci_read_config_word(pdev, PCI_VENDOR_ID, &vendor_id);
	if (rc || ((vendor_id != PCI_VENDOR_ID_COMPAQ) && (vendor_id != PCI_VENDOR_ID_INTEL))) {
		err(msg_HPC_non_compaq_or_intel);
		return -ENODEV;
	}
	DBG("Vendor ID: %x\n", vendor_id);

	rc = pci_read_config_byte(pdev, PCI_REVISION_ID, &rev);
	DBG("revision: %d\n", rev);
	if (rc || ((vendor_id == PCI_VENDOR_ID_COMPAQ) && (!rev))) {
		err(msg_HPC_rev_error);
		return -ENODEV;
	}

	/* Check for the proper subsytem ID's
	 * Intel uses a different SSID programming model than Compaq.  
	 * For Intel, each SSID bit identifies a PHP capability.
	 * Also Intel HPC's may have RID=0.
	 */
	if ((rev > 2) || (vendor_id == PCI_VENDOR_ID_INTEL)) {
		// TODO: This code can be made to support non-Compaq or Intel subsystem IDs
		rc = pci_read_config_word(pdev, PCI_SUBSYSTEM_VENDOR_ID, &subsystem_vid);
		if (rc) {
			err(__FUNCTION__" : pci_read_config_word failed\n");
			return rc;
		}
		DBG("Subsystem Vendor ID: %x\n", subsystem_vid);
		if ((subsystem_vid != PCI_VENDOR_ID_COMPAQ) && (subsystem_vid != PCI_VENDOR_ID_INTEL)) {
			err(msg_HPC_non_compaq_or_intel);
			return -ENODEV;
		}

		ctrl = (struct controller *) kmalloc(sizeof(struct controller), GFP_KERNEL);
		if (!ctrl) {
			err(__FUNCTION__" : out of memory\n");
			return -ENOMEM;
		}
		memset(ctrl, 0, sizeof(struct controller));

		rc = pci_read_config_word(pdev, PCI_SUBSYSTEM_ID, &subsystem_deviceid);
		if (rc) {
			err(__FUNCTION__" : pci_read_config_word failed\n");
			goto err_free_ctrl;
		}

		info("Hot Plug Subsystem Device ID: %x\n", subsystem_deviceid);

		/* Set Vendor ID, so it can be accessed later from other functions */
		ctrl->vendor_id = vendor_id;

		switch (subsystem_vid) {
			case PCI_VENDOR_ID_COMPAQ:
				switch (subsystem_deviceid) {
					case PCI_SUB_HPC_ID:
						/* Original 6500/7000 implementation */
						ctrl->slot_switch_type = 1;		// Switch is present
						ctrl->speed_capability = CTRL_SPEED_33MHz;
						ctrl->push_button = 0;			// No pushbutton
						ctrl->pci_config_space = 1;		// Index/data access to working registers 0 = not supported, 1 = supported
						ctrl->defeature_PHP = 1;		// PHP is supported
						ctrl->pcix_support = 0;			// PCI-X not supported
						ctrl->pcix_speed_capability = 0;	// N/A since PCI-X not supported
						break;
					case PCI_SUB_HPC_ID2:
						/* First Pushbutton implementation */
						ctrl->push_flag = 1;
						ctrl->slot_switch_type = 1;		// Switch is present
						ctrl->speed_capability = CTRL_SPEED_33MHz;
						ctrl->push_button = 1;			// Pushbutton is present
						ctrl->pci_config_space = 1;		// Index/data access to working registers 0 = not supported, 1 = supported
						ctrl->defeature_PHP = 1;		// PHP is supported
						ctrl->pcix_support = 0;			// PCI-X not supported
						ctrl->pcix_speed_capability = 0;	// N/A since PCI-X not supported
						break;
					case PCI_SUB_HPC_ID_INTC:
						/* Third party (6500/7000) */
						ctrl->slot_switch_type = 1;		// Switch is present
						ctrl->speed_capability = CTRL_SPEED_33MHz;
						ctrl->push_button = 0;			// No pushbutton
						ctrl->pci_config_space = 1;		// Index/data access to working registers 0 = not supported, 1 = supported
						ctrl->defeature_PHP = 1;		// PHP is supported
						ctrl->pcix_support = 0;			// PCI-X not supported
						ctrl->pcix_speed_capability = 0;	// N/A since PCI-X not supported
						break;
					case PCI_SUB_HPC_ID3:
						/* First 66 Mhz implementation */
						ctrl->push_flag = 1;
						ctrl->slot_switch_type = 1;		// Switch is present
						ctrl->speed_capability = CTRL_SPEED_66MHz;
						ctrl->push_button = 1;			// Pushbutton is present
						ctrl->pci_config_space = 1;		// Index/data access to working registers 0 = not supported, 1 = supported
						ctrl->defeature_PHP = 1;		// PHP is supported
						ctrl->pcix_support = 0;			// PCI-X not supported
						ctrl->pcix_speed_capability = 0;	// N/A since PCI-X not supported
						break;
					default:
						// TODO: Add SSIDs for CPQ systems that support PCI-X
						err(msg_HPC_not_supported);
						rc = -ENODEV;
						goto err_free_ctrl;
				}
				break;

			case PCI_VENDOR_ID_INTEL:
				/* Check for speed capability (0=33, 1=66) */
				if (subsystem_deviceid & 0x0001) {
					ctrl->speed_capability = CTRL_SPEED_66MHz;
				} else {
					ctrl->speed_capability = CTRL_SPEED_33MHz;
				}

				/* Check for push button */
				if (subsystem_deviceid & 0x0002) {
					/* no push button */
					ctrl->push_button = 0;
				} else {
					/* push button supported */
					ctrl->push_button = 1;
				}

				/* Check for slot switch type (0=mechanical, 1=not mechanical) */
				if (subsystem_deviceid & 0x0004) {
					/* no switch */
					ctrl->slot_switch_type = 0;
				} else {
					/* switch */
					ctrl->slot_switch_type = 1;
				}

				/* PHP Status (0=De-feature PHP, 1=Normal operation) */
				if (subsystem_deviceid & 0x0008) {
					ctrl->defeature_PHP = 1;	// PHP supported
				} else {
					ctrl->defeature_PHP = 0;	// PHP not supported
				}

				/* Alternate Base Address Register Interface (0=not supported, 1=supported) */
				if (subsystem_deviceid & 0x0010) {
					ctrl->alternate_base_address = 1;	// supported
				} else {
					ctrl->alternate_base_address = 0;	// not supported
				}

				/* PCI Config Space Index (0=not supported, 1=supported) */
				if (subsystem_deviceid & 0x0020) {
					ctrl->pci_config_space = 1;		// supported
				} else {
					ctrl->pci_config_space = 0;		// not supported
				}

				/* PCI-X support */
				if (subsystem_deviceid & 0x0080) {
					/* PCI-X capable */
					ctrl->pcix_support = 1;
					/* Frequency of operation in PCI-X mode */
					if (subsystem_deviceid & 0x0040) {
						/* 133MHz PCI-X if bit 7 is 1 */
						ctrl->pcix_speed_capability = 1;
					} else {
						/* 100MHz PCI-X if bit 7 is 1 and bit 0 is 0, */
						/* 66MHz PCI-X if bit 7 is 1 and bit 0 is 1 */
						ctrl->pcix_speed_capability = 0;
					}
				} else {
					/* Conventional PCI */
					ctrl->pcix_support = 0;
					ctrl->pcix_speed_capability = 0;
				}
				break;

			default:
				err(msg_HPC_not_supported);
				rc = -ENODEV;
				goto err_free_ctrl;
		}

	} else {
		err(msg_HPC_not_supported);
		return -ENODEV;
	}

	// Tell the user that we found one.
	info("Initializing the PCI hot plug controller residing on PCI bus %d\n", pdev->bus->number);

	DBG ("Hotplug controller capabilities:\n");
	DBG ("    speed_capability       %s\n", ctrl->speed_capability == CTRL_SPEED_33MHz ? "33MHz" : "66Mhz");
	DBG ("    slot_switch_type       %s\n", ctrl->slot_switch_type == 0 ? "no switch" : "switch present");
	DBG ("    defeature_PHP          %s\n", ctrl->defeature_PHP == 0 ? "PHP not supported" : "PHP supported");
	DBG ("    alternate_base_address %s\n", ctrl->alternate_base_address == 0 ? "not supported" : "supported");
	DBG ("    pci_config_space       %s\n", ctrl->pci_config_space == 0 ? "not supported" : "supported");
	DBG ("    pcix_speed_capability  %s\n", ctrl->pcix_speed_capability == 0 ? "not supported" : "supported");
	DBG ("    pcix_support           %s\n", ctrl->pcix_support == 0 ? "not supported" : "supported");

	ctrl->pci_dev = pdev;
	ctrl->bus = pdev->bus->number;
	ctrl->device = PCI_SLOT(pdev->devfn);
	ctrl->function = PCI_FUNC(pdev->devfn);
	ctrl->rev = rev;
	DBG("bus device function rev: %d %d %d %d\n", ctrl->bus, ctrl->device, ctrl->function, ctrl->rev);

	init_MUTEX(&ctrl->crit_sect);
	init_waitqueue_head(&ctrl->queue);

	/* initialize our threads if they haven't already been started up */
	rc = one_time_init();
	if (rc) {
		goto err_free_ctrl;
	}
	
	DBG("pdev = %p\n", pdev);
	DBG("pci resource start %lx\n", pci_resource_start(pdev, 0));
	DBG("pci resource len %lx\n", pci_resource_len(pdev, 0));

	if (!request_mem_region(pci_resource_start(pdev, 0),
				pci_resource_len(pdev, 0), MY_NAME)) {
		err("cannot reserve MMIO region\n");
		rc = -ENOMEM;
		goto err_free_ctrl;
	}

	ctrl->hpc_reg = ioremap(pci_resource_start(pdev, 0), pci_resource_len(pdev, 0));
	if (!ctrl->hpc_reg) {
		err("cannot remap MMIO region %lx @ %lx\n", pci_resource_len(pdev, 0), pci_resource_start(pdev, 0));
		rc = -ENODEV;
		goto err_free_mem_region;
	}

	// Check for 66Mhz operation
	// TODO: Add PCI-X support
	ctrl->speed = get_controller_speed(ctrl);


	//**************************************************
	//
	//              Save configuration headers for this and
	//              subordinate PCI buses
	//
	//**************************************************

	// find the physical slot number of the first hot plug slot

	// Get slot won't work for devices behind bridges, but
	// in this case it will always be called for the "base"
	// bus/dev/func of a slot.
	// CS: this is leveraging the PCIIRQ routing code from the kernel (pci-pc.c: get_irq_routing_table)
	DBG("pci_get_slot: \n");
	rc = pci_get_slot(pdev->bus->number, (readb(ctrl->hpc_reg + SLOT_MASK) >> 4), &(ctrl->first_slot));
	DBG("pci_get_slot: %d %d =%d\n", pdev->bus->number, (readb(ctrl->hpc_reg + SLOT_MASK) >> 4), ctrl->first_slot);
	if (rc) {
		err(msg_initialization_err, rc);
		goto err_iounmap;
	}

	// Store PCI Config Space for all devices on this bus
	rc = pci_hp_save_config(ctrl->bus, readb(ctrl->hpc_reg + SLOT_MASK));
	if (rc) {
		err(__FUNCTION__": unable to save PCI configuration data, error %d\n", rc);
		goto err_iounmap;
	}

	/*
	 * Get IO, memory, and IRQ resources for new devices
	 */
	rc = pci_hp_find_available_resources(ctrl, pci_hp_rom_start);
	ctrl->add_support = !rc;
	if (rc) {
		DBG("pci_hp_find_available_resources = 0x%x\n", rc);
		err("unable to locate PCI configuration resources for hot plug add.\n");
		goto err_iounmap;
	}

	/* Setup the slot information structures */
	rc = pci_hp_ctrl_slot_setup(ctrl, smbios_table);
	if (rc) {
		err(msg_initialization_err, 6);
		err(__FUNCTION__": unable to save PCI configuration data, error %d\n", rc);
		goto err_iounmap;
	}
	
	/*
	 * Finish setting up the hot plug ctrl device
	 */
	ctrl->slot_device_offset = readb(ctrl->hpc_reg + SLOT_MASK) >> 4;
	DBG("NumSlots %d \n", readb(ctrl->hpc_reg + SLOT_MASK) & 0x0F);

	ctrl->next_event = 0;

	/* Mask all general input interrupts */
	writel(0xFFFFFFFFL, ctrl->hpc_reg + INT_MASK);

	/* set up the interrupt */
	ctrl->interrupt = pdev->irq;
	DBG("HPC interrupt = %d \n", ctrl->interrupt);
	if (request_irq(ctrl->interrupt,
			(void (*)(int, void *, struct pt_regs *)) &pci_hp_ctrl_intr,
			SA_SHIRQ, MY_NAME, ctrl)) {
		err("Can't get irq %d for the hotplug pci controller\n", ctrl->interrupt);
		rc = -ENODEV;
		goto err_iounmap;
	}

	/* Enable Shift Out interrupt and clear it, also enable SERR on power fault */
	temp_word = readw(ctrl->hpc_reg + MISC);
	temp_word |= 0x4006;
	writew(temp_word, ctrl->hpc_reg + MISC);

	// Changed 05/05/97 to clear all interrupts at start
	writel(0xFFFFFFFFL, ctrl->hpc_reg + INT_INPUT_CLEAR);

	ctrl->ctrl_int_comp = readl(ctrl->hpc_reg + INT_INPUT_CLEAR);

	writel(0x0L, ctrl->hpc_reg + INT_MASK);

	if (!pci_hp_ctrl_list) {
		pci_hp_ctrl_list = ctrl;
		ctrl->next = NULL;
	} else {
		ctrl->next = pci_hp_ctrl_list;
		pci_hp_ctrl_list = ctrl;
	}

	// turn off empty slots here unless command line option "ON" set
	// Wait for exclusive access to hardware
	down(&ctrl->crit_sect);

	num_of_slots = readb(ctrl->hpc_reg + SLOT_MASK) & 0x0F;

	// find first device number for the ctrl
	device = readb(ctrl->hpc_reg + SLOT_MASK) >> 4;

	while (num_of_slots) {
		DBG("num_of_slots: %d\n", num_of_slots);
		func = pci_hp_slot_find(ctrl->bus, device, 0);
		if (!func)
			break;

		hp_slot = func->device - ctrl->slot_device_offset;
		DBG("hp_slot: %d\n", hp_slot);

		// We have to save the presence info for these slots
		temp_word = ctrl->ctrl_int_comp >> 16;
		func->presence_save = (temp_word >> hp_slot) & 0x01;
		func->presence_save |= (temp_word >> (hp_slot + 7)) & 0x02;

		if (ctrl->ctrl_int_comp & (0x1L << hp_slot)) {
			func->switch_save = 0;
		} else {
			func->switch_save = 0x10;
		}

		if (!power_mode) {
			if (!func->is_a_board) {
				green_LED_off (ctrl, hp_slot);
				slot_disable (ctrl, hp_slot);
			}
		}

		device++;
		num_of_slots--;
	}

	if (!power_mode) {
		set_SOGO(ctrl);

		// Wait for SOBS to be unset
		interruptible_sleep_on(&ctrl->queue);
	}

	rc = init_SERR(ctrl);
	if (rc) {
		err("init_SERR failed\n");
		up(&ctrl->crit_sect);
		goto err_free_irq;
	}

	// Done with exclusive hardware access
	up(&ctrl->crit_sect);

	rc = pci_hp_proc_create_ctrl (ctrl);
	if (rc) {
		err("pci_hp_proc_create_ctrl failed\n");
		goto err_free_irq;
	}

	return 0;

err_free_irq:
	free_irq(ctrl->interrupt, ctrl);
err_iounmap:
	iounmap(ctrl->hpc_reg);
err_free_mem_region:
	release_mem_region(pci_resource_start(pdev, 0), pci_resource_len(pdev, 0));
err_free_ctrl:
	kfree(ctrl);
	return rc;
}


static int one_time_init(void)
{
	int loop;
	int retval = 0;
	static int initialized = 0;

	if (initialized)
		return 0;

	DBG("create pci_dev\n");
	retval = pci_hp_proc_create_pci();
	if (retval) {
		DBG("pci_hp_proc_create_pci() failed, retval = %d\n", retval);
		goto error;
	}

	power_mode = 0;

	retval = pci_print_IRQ_route();
	if (retval)
		goto error;

	DBG("Initialize + Start the notification mechanism \n");

	retval = pci_hp_start_notify();
	if (retval)
		goto error;

	retval = pci_hp_event_start_thread();
	if (retval)
		goto error;

	DBG("Initialize slot lists\n");
	for (loop = 0; loop < 256; loop++) {
		pci_hp_slot_list[loop] = NULL;
	}

	// FIXME: We also need to hook the NMI handler eventually.
	// this also needs to be worked with Christoph
	// register_NMI_handler();

	// Map rom address
	pci_hp_rom_start = ioremap(ROM_PHY_ADDR, ROM_PHY_LEN);
	if (!pci_hp_rom_start) {
		err ("Could not ioremap memory region for ROM\n");
		retval = -EIO;;
		goto error;
	}
	
	/* Map smbios table entry point structure */
	smbios_table = detect_SMBIOS_pointer(pci_hp_rom_start, pci_hp_rom_start + ROM_PHY_LEN);
	if (!smbios_table) {
		err ("Could not find the SMBIOS pointer in memory\n");
		retval = -EIO;;
		goto error;
	}

	pci_hp_smbios_start = ioremap(readl(smbios_table + ST_ADDRESS), readw(smbios_table + ST_LENGTH));
	if (!pci_hp_smbios_start) {
		err ("Could not ioremap memory region taken from SMBIOS values\n");
		retval = -EIO;;
		goto error;
	}

	pci_hp_proc_root = create_proc_entry("hpc", S_IFDIR, 0);
	if (!pci_hp_proc_root) {
		retval = -ENOMEM;
		goto error;
	}

	initialized = 1;

	return retval;

error:
	if (pci_hp_rom_start)
		iounmap(pci_hp_rom_start);
	if (pci_hp_smbios_start)
		iounmap(pci_hp_smbios_start);
	
	return retval;
}


static void unload_cpqphpd(void)
{
	struct pci_func *next;
	struct pci_func *TempSlot;
	int loop;
	u32 rc;
	struct controller *ctrl;
	struct controller *tctrl;
	struct pci_resource *res;
	struct pci_resource *tres;

	ctrl = pci_hp_ctrl_list;

	while (ctrl) {

		if (ctrl->proc_entry) {
			remove_proc_entry(ctrl->proc_name, pci_hp_proc_root);
		}
		if (ctrl->proc_entry2) {
			remove_proc_entry(ctrl->proc_name2, pci_hp_proc_root);
		}

		if (ctrl->hpc_reg) {
			u16 misc;
			rc = read_slot_enable (ctrl);
			
			writeb(0, ctrl->hpc_reg + SLOT_SERR);
			writel(0xFFFFFFC0L | ~rc, ctrl->hpc_reg + INT_MASK);
			
			misc = readw(ctrl->hpc_reg + MISC);
			misc &= 0xFFFD;
			writew(misc, ctrl->hpc_reg + MISC);
		}

		pci_hp_ctrl_slot_cleanup(ctrl);

		res = ctrl->io_head;
		while (res) {
			tres = res;
			res = res->next;
			kfree(tres);
		}

		res = ctrl->mem_head;
		while (res) {
			tres = res;
			res = res->next;
			kfree(tres);
		}

		res = ctrl->p_mem_head;
		while (res) {
			tres = res;
			res = res->next;
			kfree(tres);
		}

		res = ctrl->bus_head;
		while (res) {
			tres = res;
			res = res->next;
			kfree(tres);
		}

		tctrl = ctrl;
		ctrl = ctrl->next;
		kfree(tctrl);
	}

	for (loop = 0; loop < 256; loop++) {
		next = pci_hp_slot_list[loop];
		while (next != NULL) {
			res = next->io_head;
			while (res) {
				tres = res;
				res = res->next;
				kfree(tres);
			}

			res = next->mem_head;
			while (res) {
				tres = res;
				res = res->next;
				kfree(tres);
			}

			res = next->p_mem_head;
			while (res) {
				tres = res;
				res = res->next;
				kfree(tres);
			}

			res = next->bus_head;
			while (res) {
				tres = res;
				res = res->next;
				kfree(tres);
			}

			TempSlot = next;
			next = next->next;
			kfree(TempSlot);
		}
	}

	remove_proc_entry("hpc", 0);

	// Stop the notification mechanism
	pci_hp_stop_notify();
	pci_hp_event_stop_thread();

	//unmap the rom address
	if (pci_hp_rom_start)
		iounmap(pci_hp_rom_start);
	if (pci_hp_smbios_start)
		iounmap(pci_hp_smbios_start);
}


static int hpcd_open(struct inode *inode, struct file *filp)
{
	pci_hp_open_helper(filp);
	return 0;
}


static int hpcd_close(struct inode *inode, struct file *filp)
{
	return 0;
}


static int hpcd_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
	struct ioctl_data stuff;
	int rc = -ENOTTY;

	DBG("hpcd_ioctl (cmd =%d)\n", cmd);

	switch (cmd) {
		case GUI_BACK_DOOR:
			if (copy_from_user(&stuff, (void *) arg, sizeof(struct ioctl_data)))
				return -EFAULT;

			rc = pci_hp_gui_back_door(&stuff);
			
			if (copy_to_user((void *)arg, &stuff, sizeof(struct ioctl_data)))
				return -EFAULT;
			break;

		default:
			break;
	}
	return rc;
}


static int hpcd_fasync(int fd, struct file *filp, int mode)
{
	DBG("Entered hpcd_fasync\n");
	if (!mode) {
		pci_hp_unregister_user_notifier(fd, filp);
	} else {
		pci_hp_register_user_notifier(fd, filp);
	}
	DBG("Leaving hpcd_fasync\n");
	return 0;
}


static ssize_t hpcd_read(struct file *filp, char *buf, size_t count, loff_t * ppos)
{
	return pci_hp_read_helper(filp, buf, count, ppos);
}


static struct pci_device_id hpcd_pci_tbl[] __devinitdata = {
//	{PCI_VENDOR_ID_COMPAQ, PCI_HPC_ID,
//		PCI_ANY_ID, PCI_ANY_ID,},
	{
	/* handle any PCI Hotplug controller */
	class:          ((PCI_CLASS_SYSTEM_PCI_HOTPLUG << 8) | 0x00),
	class_mask:     ~0,
	
	/* no matter who makes it */
	vendor:         PCI_ANY_ID,
	device:         PCI_ANY_ID,
	subvendor:      PCI_ANY_ID,
	subdevice:      PCI_ANY_ID,
	
	}, { /* end: all zeroes */ }
};

MODULE_DEVICE_TABLE(pci, hpcd_pci_tbl);



static struct pci_driver cpqhpc_driver = {
	name:		"pci_hotplug",
	id_table:	hpcd_pci_tbl,
	probe:		cpqhpc_probe,
	/* remove:	cpqhpc_remove_one, */
};


static struct file_operations hpcd_fops = {
	owner:		THIS_MODULE,
	read:		&hpcd_read,
	ioctl:		&hpcd_ioctl,
	open:		&hpcd_open,
	release:	&hpcd_close,
	fasync:		&hpcd_fasync,
};


static struct miscdevice cpqphpc_miscdev = {
	minor:	CPQHPC_MODULE_MINOR,
	name:	"pci_hotplug",
	fops:	&hpcd_fops,
};


static int __init cpqhpc_init(void)
{
	int result;

	pci_hp_debug = debug;

	result = misc_register(&cpqphpc_miscdev);
	if (result < 0) {
		warn("can't get minor %d\n", CPQHPC_MODULE_MINOR);
		return result;
	}

	DBG("register pci driver\n");
	result = pci_module_init(&cpqhpc_driver);
	DBG("pci_module_init = %d\n", result);
	if (result)
		misc_deregister(&cpqphpc_miscdev);
	else
		info (DRIVER_DESC " version: " DRIVER_VERSION "\n");
	return result;
}


static void __exit cpqhpc_cleanup(void)
{
	DBG("destroy pci_dev\n");
	pci_hp_proc_destroy_pci();        
	DBG("unload_cpqphpd()\n");
	unload_cpqphpd();
	DBG("pci_unregister_driver\n");
	pci_unregister_driver(&cpqhpc_driver);
	misc_deregister(&cpqphpc_miscdev);
	DBG("misc_deregister\n");
	misc_deregister(&cpqphpc_miscdev);
}

module_init(cpqhpc_init);
module_exit(cpqhpc_cleanup);


