
/*
 * Broadcom Cryptonet Driver software is distributed as is, without any warranty
 * of any kind, either express or implied as further specified in the GNU Public
 * License. This software may be used and distributed according to the terms of
 * the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */

/******************************************************************************
 *
 * Copyright 2000
 * Broadcom Corporation
 * 16215 Alton Parkway
 * PO Box 57013
 * Irvine CA 92619-7013
 *
 *****************************************************************************/

/* 
 * Broadcom Corporation uBSec SDK 
 */

/*
 * ubsrsa.c: RSA parameter setup functions.
 *
 *
 * Revision History:
 *
 * May 2000 SOR Created
 * 07/26/2000 SOR Virtual/Physical Memory manipulation modifications
 * 09/xx/2000 DRE BCM5820 Support
 * 07/16/2001 RJT Added support for BCM5821
 */

#include "ubsincl.h"

#ifdef UBSEC_PKEY_SUPPORT


/* 
 * Function: RSA_SetupPublicParams()
 * Set up KeyContext Buffer, MCR Input Buffer and MCR Output Buffer 
 * for RSA Public operation with parameters provided by pRSAParams 
 * The KeyContext Buffer includes exponent_length, modulus_length,
 * N, and g
 */

long 
RSA_SetupPublicParams(MasterCommand_pt pMCR, ubsec_RSA_Params_pt pRSAParams)
     /* pRSAParams points to a structure which contains all of the info
	needed for RSA operations. In addition to regular numerical 
	parameters, memory buffer locations are passed as follows:

	  pRSAParams->ModN.KeyValue            virtual pointer 
	  pRSAParams->ExpE.KeyValue            virtual pointer 
	  pRSAParams->InputKeyInfo.KeyValue    DMA memory handle 
	  pRSAParams->OutputKeyInfo.KeyValue   DMA memory handle  

	Note that pointers and handles must be treated differently. 
	The virtual and physical pointers associated with the handle 
	must be extracted by using the following macros:

	  ()
	  virt_to_bus()
	
	Results from virt_to_bus() may be written to CryptoNet
	control structures in (DMA) memory. 
	Results from () may be used (if necessary) as
	regular old pointers.
     */
{
#ifdef UBSEC_RSA_SUPPORT
  volatile DataBufChainList_pt  FragPtr;
  unsigned short                DataLength;
  volatile Packet_t 		*pPacket;
  Pub_RSA_CtxCmdBuf_t		*pRSACtx;
  volatile int                  PacketIndex;
  long Status=UBSEC_STATUS_SUCCESS;
  int ModNLen,NormBits;
  unsigned long PhysAddr;

  PacketIndex = pMCR->NumberOfPackets;
  pRSACtx = &pMCR->KeyContextList[PacketIndex]->CtxCmdBuf.Pub_RSA_CtxCmdBuf;
  pPacket = &(pMCR->PacketArray[PacketIndex]); /* Set up the current packet */
  memset(pRSACtx,0,sizeof(*pRSACtx));

  pRSACtx->modulus_length  = (unsigned short)CPU_TO_CTRL_SHORT(pRSAParams->ModN.KeyLength) ;
  pRSAParams->ExpE.KeyLength=ROUNDUP_TO_32_BIT(pRSAParams->ExpE.KeyLength);
  pRSACtx->exponent_length = (unsigned short) CPU_TO_CTRL_SHORT(pRSAParams->ExpE.KeyLength);

  if (pRSAParams->ModN.KeyLength <=512)
    ModNLen=512;
  else
    if (pRSAParams->ModN.KeyLength <=768)
      ModNLen=768;
    else
    if (pRSAParams->ModN.KeyLength  <= 1024)
      ModNLen=1024;
    else
#ifdef UBSEC_582x_CLASS_DEVICE
      if (pRSAParams->ModN.KeyLength <=1536)
	ModNLen=1536;
      else
	if (pRSAParams->ModN.KeyLength <= 2048)
	  ModNLen=2048;
	else
#endif
      return(UBSEC_STATUS_INVALID_PARAMETER);

#ifndef UBSEC_HW_NORMALIZE
  NormBits = ubsec_NormalizeDataTo(&pRSAParams->ModN,ModNLen);
  pRSAParams->OutputKeyInfo.KeyLength=ModNLen;
  if (NormBits) {
    /*
     * Normalize message to align with modulus.
     */
    ubsec_ShiftData(&pRSAParams->InputKeyInfo, NormBits);
    pRSAParams->InputKeyInfo.KeyLength=
      ROUNDUP_TO_32_BIT(pRSAParams->InputKeyInfo.KeyLength+NormBits);
    pMCR->KeyContextList[PacketIndex]->ResultKey=pRSAParams->OutputKeyInfo;
  }
  pMCR->KeyContextList[PacketIndex]->NormBits=NormBits;
#else
  NormBits=ModNLen-pRSAParams->ModN.KeyLength;
  pRSAParams->OutputKeyInfo.KeyLength=ModNLen;
  pRSAParams->InputKeyInfo.KeyLength=ModNLen;
  pMCR->KeyContextList[PacketIndex]->ResultKey=pRSAParams->OutputKeyInfo;
  pMCR->KeyContextList[PacketIndex]->NormBits=NormBits;
#endif

  ModNLen/=8;

#ifndef UBSEC_HW_NORMALIZE
  /* N setup */
  RTL_Memcpy( &pRSACtx->Ng[0],(pRSAParams->ModN.KeyValue),ModNLen);
  /* g setup */
  RTL_Memcpy( &pRSACtx->Ng[ModNLen/4],(pRSAParams->ExpE.KeyValue)
	      ,pRSAParams->ExpE.KeyLength/8);
#else
  /* N setup */
  RTL_Memcpy( &pRSACtx->Ng[0],(pRSAParams->ModN.KeyValue),ROUNDUP_TO_32_BIT(pRSAParams->ModN.KeyLength)/8);
  /* g setup */
  RTL_Memcpy( &pRSACtx->Ng[ModNLen/4],(pRSAParams->ExpE.KeyValue)
	      ,pRSAParams->ExpE.KeyLength/8);
#endif
  ModNLen+=pRSAParams->ExpE.KeyLength/8;
  /* Now set the extra length. */
  pMCR->KeyContextList[PacketIndex]->cmd_structure_length+=(ModNLen);

#ifdef UBSDBG
  /* Print out the context information if required */
  {
  int WordLen,i;
  WordLen=(pMCR->KeyContextList[PacketIndex]->cmd_structure_length-RSA_STATIC_PUBLIC_CONTEXT_SIZE)/4;
  Dbg_Print(DBG_RSAKEY,(   "ubsec:  ---- RSA Public Modulus Length [%d] Normbits [%d]\n[",
			   pRSACtx->modulus_length,NormBits)); 
  Dbg_Print(DBG_RSAKEY,(   "ubsec:  ---- RSA Public G Length [%d] Context Len %d Value-\n[",
			   CTRL_TO_CPU_SHORT(pRSACtx->exponent_length),
			   pMCR->KeyContextList[PacketIndex]->cmd_structure_length)); 
  for ( i=0 ; i < WordLen ; i++) {
    Dbg_Print(DBG_RSAKEY,( "%08x ",SYS_TO_BE_LONG(pRSACtx->Ng[i])));
  }
  Dbg_Print(DBG_RSAKEY,( "]\n"));
  }
#endif

  /* Input Buffer setup for RSA Public */
  FragPtr=(DataBufChainList_pt)&pPacket->InputHead;
  PhysAddr=virt_to_bus( pRSAParams->InputKeyInfo.KeyValue);
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pRSAParams->InputKeyInfo.KeyLength+7)/8;
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
 /* Copy (endian-adjusted) fragment length into MCR structure */
  pPacket->PacketLength = FragPtr->DataLength;
  Dbg_Print(DBG_RSAKEY,( "RSA Public Key, InputKeyInfo:  FragI <%d,%08x %08x>\n",
            DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
  FragPtr->pNext = 0;

  /* Output Buffer setup for RSA Public */
  FragPtr=(DataBufChainList_pt)&pPacket->OutputHead;
  PhysAddr=virt_to_bus(pRSAParams->OutputKeyInfo.KeyValue);
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pRSAParams->OutputKeyInfo.KeyLength+7)/8;
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
  Dbg_Print(DBG_RSAKEY,( "RSA Public Key, OutputKeyInfo:  FragO <%d,%08x %08x>\n",
            DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
  FragPtr->pNext = 0;

  pRSAParams->OutputKeyInfo.KeyLength-=NormBits;
  return(UBSEC_STATUS_SUCCESS);
#else
    return(UBSEC_STATUS_NO_DEVICE);
#endif
}

/* 
 * Function: RSA_SetupPrivateParams()
 * Set up KeyContext Buffer, MCR Input Buffer and MCR Output Buffer 
 * for RSA Private operation with parameters provided by pRSAParams 
 * The KeyContext Buffer includes q_length,p_length, dq_length,                
 * dp_length, pinv_length, p, q, dp, dq, and  pinv 
 */
long 
RSA_SetupPrivateParams(MasterCommand_pt pMCR, ubsec_RSA_Params_pt pRSAParams)
{
#ifdef UBSEC_RSA_SUPPORT
  volatile DataBufChainList_pt  FragPtr;
  unsigned short                DataLength;
  volatile Packet_t 		*pPacket;
  volatile Pri_RSA_CtxCmdBuf_t		*pRSACtx;
  int                  PacketIndex;
  int Offset;
  int ParamLen,NormBits;
  long Status=UBSEC_STATUS_SUCCESS;
  unsigned long PhysAddr;

  PacketIndex = pMCR->NumberOfPackets;
  pRSACtx = &pMCR->KeyContextList[PacketIndex]->CtxCmdBuf.Pri_RSA_CtxCmdBuf;
  pPacket = &(pMCR->PacketArray[PacketIndex]); /* Set up the current packet */
  memset(pRSACtx,0,sizeof(*pRSACtx));
  pRSACtx->q_length    = (unsigned short)CPU_TO_CTRL_SHORT(pRSAParams->PrimeQ.KeyLength) ;
  pRSACtx->p_length    = (unsigned short)CPU_TO_CTRL_SHORT(pRSAParams->PrimeP.KeyLength) ;

  /*
   * All parameters need to be aligned on the 
   * same length so we use the length of the 
   * largest.
   */

  /* Both P & Q Must be normalized and aligned on the largest 
     length. */
  if ((pRSAParams->PrimeQ.KeyLength > pRSAParams->PrimeP.KeyLength)) 
    ParamLen=pRSAParams->PrimeQ.KeyLength;
  else
    ParamLen=pRSAParams->PrimeP.KeyLength;
    
  if (ParamLen <=256)
    ParamLen=256;
  else
    if (ParamLen <= 384)
      ParamLen=384;
    else
      if (ParamLen <= 512)
	ParamLen=512;
      else
#ifdef UBSEC_582x_CLASS_DEVICE
	if (ParamLen <= 768)
	  ParamLen=768;
	else
	  if (ParamLen <= 1024)
	    ParamLen=1024;
#else
	  return(UBSEC_STATUS_INVALID_PARAMETER);
#endif

#ifndef UBSEC_HW_NORMALIZE
  NormBits = ubsec_NormalizeDataTo(&pRSAParams->PrimeP,ParamLen);
  if (NormBits)
    ubsec_ShiftData(&pRSAParams->PrimeEdp, NormBits);
  NormBits = ubsec_NormalizeDataTo(&pRSAParams->PrimeQ,ParamLen);
  if (NormBits) {
    ubsec_ShiftData(&pRSAParams->PrimeEdq, NormBits); 
    ubsec_ShiftData(&pRSAParams->Pinv, NormBits); 
  }

  /* Return the number of bits the result will have to be shifted by. */
  NormBits=((ParamLen*2)
	    -(pRSAParams->PrimeQ.KeyLength+pRSAParams->PrimeP.KeyLength)); 

  if (NormBits) {
    pMCR->KeyContextList[PacketIndex]->ResultKey=pRSAParams->OutputKeyInfo;
    pMCR->KeyContextList[PacketIndex]->ResultKey.KeyLength=
      pRSAParams->PrimeQ.KeyLength+pRSAParams->PrimeP.KeyLength; 
  }
  pMCR->KeyContextList[PacketIndex]->NormBits=NormBits;
#else
  NormBits=((ParamLen*2)
	    -(pRSAParams->PrimeQ.KeyLength+pRSAParams->PrimeP.KeyLength)); 

  pMCR->KeyContextList[PacketIndex]->NormBits=NormBits;
  pMCR->KeyContextList[PacketIndex]->ResultKey=pRSAParams->OutputKeyInfo;
  pMCR->KeyContextList[PacketIndex]->ResultKey.KeyLength=ParamLen*2;
#endif

  pRSAParams->InputKeyInfo.KeyLength = 2*ParamLen; /* Pad it out.*/
  pRSAParams->OutputKeyInfo.KeyLength=2*ParamLen;

  ParamLen/=8;

  /* p setup */
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pRSACtx->CtxParams[0],(pRSAParams->PrimeP.KeyValue),
	      ParamLen);
#else
  RTL_Memcpy( &pRSACtx->CtxParams[0],(pRSAParams->PrimeP.KeyValue),
	      ROUNDUP_TO_32_BIT(pRSAParams->PrimeP.KeyLength)/8);
#endif
  Offset=ParamLen;

  /* q setup */
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeQ.KeyValue),
	      ParamLen);
#else
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeQ.KeyValue),
	      ROUNDUP_TO_32_BIT(pRSAParams->PrimeQ.KeyLength)/8);
#endif
  Offset+=ParamLen;

  /* dp setup */
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeEdp.KeyValue),
	      ParamLen);
#else
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeEdp.KeyValue),
	      ROUNDUP_TO_32_BIT(pRSAParams->PrimeEdp.KeyLength)/8);
#endif
  Offset+=ParamLen;

  /* dq setup */
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeEdq.KeyValue),
	      ParamLen);
#else
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->PrimeEdq.KeyValue),
	      ROUNDUP_TO_32_BIT(pRSAParams->PrimeEdq.KeyLength)/8);
#endif

  Offset+=ParamLen;

  /* pinv setup */
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->Pinv.KeyValue),
	      ParamLen);
#else
  RTL_Memcpy( &pRSACtx->CtxParams[Offset/4],(pRSAParams->Pinv.KeyValue),
	      ROUNDUP_TO_32_BIT(pRSAParams->Pinv.KeyLength)/8);
#endif
  Offset+=ParamLen;

  pMCR->KeyContextList[PacketIndex]->cmd_structure_length+=(Offset);

#ifdef UBSDBG
  /* Print out the context information if required */
  {
  int WordLen,i;
  WordLen=(pMCR->KeyContextList[PacketIndex]->cmd_structure_length-RSA_STATIC_PRIVATE_CONTEXT_SIZE)/4;
  Dbg_Print(DBG_RSAKEY,(   "ubsec:  ---- RSA Private P-Q Length [%d] P-D Length [%d] \n[",pRSACtx->q_length,pRSACtx->p_length)); 
  Dbg_Print(DBG_RSAKEY,(   "ubsec:  ---- ParamLen %d Context Len %d Value -\n[",
			   CTRL_TO_CPU_SHORT(ParamLen),pMCR->KeyContextList[PacketIndex]->cmd_structure_length )); 

  for ( i=0 ; i < WordLen ; i++) {
    Dbg_Print(DBG_RSAKEY,( "%08x ",SYS_TO_BE_LONG(pRSACtx->CtxParams[i])));
  }
  Dbg_Print(DBG_RSAKEY,( "]\n"));
  }
#endif

  /* Input Buffer setup for RSA Private */
  FragPtr=(DataBufChainList_pt)&pPacket->InputHead;
  PhysAddr=virt_to_bus(pRSAParams->InputKeyInfo.KeyValue);
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pRSAParams->InputKeyInfo.KeyLength+7)/8;
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
 /* Copy (endian-adjusted) fragment length into MCR structure */
  pPacket->PacketLength = FragPtr->DataLength;
  Dbg_Print(DBG_RSAKEY,( "RSA Private Key, InputKeyInfo:  FragI <%d,%08x %08x>\n",
            DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
  FragPtr->pNext = 0;

  /* Output Buffer setup for RSA Public */
  FragPtr=(DataBufChainList_pt)&pPacket->OutputHead;
  PhysAddr=virt_to_bus(pRSAParams->OutputKeyInfo.KeyValue);
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pRSAParams->OutputKeyInfo.KeyLength+7)/8;
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
  Dbg_Print(DBG_RSAKEY,( "RSA Private Key, OutputKeyInfo:  FragO <%d,%08x %08x>\n",
            DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
  FragPtr->pNext = 0;

  pRSAParams->OutputKeyInfo.KeyLength-=NormBits;

  return(UBSEC_STATUS_SUCCESS);
#else
    return(UBSEC_STATUS_NO_DEVICE);
#endif
}

#endif
