
/*
 * Broadcom Cryptonet Driver software is distributed as is, without any warranty
 * of any kind, either express or implied as further specified in the GNU Public
 * License. This software may be used and distributed according to the terms of
 * the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */

/******************************************************************************
 *
 * Copyright 2000
 * Broadcom Corporation
 * 16215 Alton Parkway
 * PO Box 57013
 * Irvine CA 92619-7013
 *
 *****************************************************************************/

/* 
 * Broadcom Corporation uBSec SDK 
 */

/*
 * ubsmath.c: Math function acceleration functions
 *
 * Revision History:
 *
 * 03/17/2000 SOR Created
 * 07/06/2000 DPA Fixes for SMP operation
 * 07/26/00 SOR Virtual/Physical Memory manipulation modifications
 * 04/20/2001 RJT Added support for CPU-DMA memory synchronization
 * 07/16/2001 RJT Added support for BCM5821
 */

#include "ubsincl.h"

/*
 * ubsec_MathCommand: Process a list of Math commands.
 *
 * Immediate Status is returned. Completion status is returned
 * on a per command callback
 */
long 
ubsec_MathCommand(ubsec_DeviceContext_t Context,
	      ubsec_MathCommandInfo_pt pCommand,
	      int *NumCommands)
{
  DeviceInfo_pt 		pDevice=(DeviceInfo_pt)Context;
  volatile MasterCommand_t  	*pMCR;
  volatile Packet_t         	*pPacket;
  volatile KeyContext_t  	*pContext;
  volatile int             	PacketIndex;
  int 				CommandIndex=0;
  int 				CommandCount=*NumCommands;
  long 		Status;
  unsigned long 		SaveConfig;
  ubsec_MathCommandParams_pt    pParams;
  volatile Math_CtxCmdBuf_t     *pMathContext;
  int offset;
  volatile DataBufChainList_t *FragPtr, *NextFragPtr;
  unsigned int DataLength;
  int NormalizeLen,NrmBits=0;
  unsigned long PhysAddr;
  CallBackInfo_t *pCompletionContext;

  if (!UBSEC_IS_KEY_DEVICE(pDevice)) {
    Dbg_Print(DBG_FATAL,( "ubsec: Math Command for a crypto device\n " ));
    return(UBSEC_STATUS_NO_DEVICE );
  }

  Dbg_Print(DBG_MATH,( "ubsec:  Math command %d ",*NumCommands ));
  /*
   * Check some parameters
   */    
  if(pDevice==NULL_DEVICE_INFO) {
    Dbg_Print(DBG_FATAL,( "NO DEV\n " ));
    return(UBSEC_STATUS_NO_DEVICE );
  }
  Dbg_Print(DBG_MATH,( "\n"));

  if (OS_EnterCriticalSection(pDevice,SaveConfig)) {
    return(UBSEC_STATUS_DEVICE_BUSY);
  }

  /* Get the next MCR to load */
 Get_New_MCR:
  *NumCommands=CommandIndex; /* Update number completed */

  if ((pMCR=GetFreeMCR(pDevice,UBSEC_KEY_LIST,&Status))== NULL_MASTER_COMMAND) 
    goto Error_Return;

  /* Add packets to this MCR. */

  Dbg_Print(DBG_MATH,( "ubsec: mcr_index %d MCR <%0x,%0x>\n",pMCR->Index,pMCR,pMCR->MCRPhysicalAddress));
  /* Initialize the packet information */
  PacketIndex = pMCR->NumberOfPackets; 
  pPacket = &(pMCR->PacketArray[PacketIndex]); /* Set up the current packet. */
  pContext = pMCR->KeyContextList[PacketIndex]; 
  pMathContext=&pContext->CtxCmdBuf.Math_CtxCmdBuf;
  Status=UBSEC_STATUS_SUCCESS; /* Wishful thinking? */


  Dbg_Print(DBG_MATH,( "ubsec: PacketIndex %d \n",pMCR->NumberOfPackets));

  /* Process all the commands in the command list. */
  for (; CommandIndex < CommandCount ; CommandIndex++) { /* Add all the packets to the MCR*/
    if( PacketIndex >= MCR_MAXIMUM_PACKETS ) {
      Dbg_Print(DBG_MATH,( "ubsec:  overran mcr buffer. %d\n",PacketIndex,CommandIndex ));
      /* 
       * We have filled this MCR. 
       * Advance next free. Wrap around if necessary
       */
      pDevice->NextFreeMCR[UBSEC_KEY_LIST]=pMCR->pNextMCR;
      Dbg_Print(DBG_MATH,( "ubsec:  PushMCR ..." ));
      PushMCR(pDevice); /* Get it going (pipeline) */
      goto Get_New_MCR; /* Try to add to the next MCR */
    }

    pCompletionContext=&pMCR->CompletionArray[PacketIndex];

    /* First set up the command type and parameters. */
    Dbg_Print(DBG_MATH,( "ubsec: Math Command packet_Index %d, Context Buf <%0x,%0x>\n",PacketIndex,pContext,pContext->PhysicalAddress ));
    pPacket->PacketContextBuffer=pContext->PhysicalAddress;
    
    switch (pCommand->Command) {
    case UBSEC_MATH_MODADD :
      pContext->operation_type	= OPERATION_MOD_ADD;
      break;
    case UBSEC_MATH_MODSUB :
      pContext->operation_type	= OPERATION_MOD_SUB;
      break;
    case UBSEC_MATH_MODMUL :
      pContext->operation_type	= OPERATION_MOD_MULT;
      break;
    case UBSEC_MATH_MODEXP :
      pContext->operation_type	= OPERATION_MOD_EXPON;
      break;
    case UBSEC_MATH_MODREM :
      pContext->operation_type	= OPERATION_MOD_REDUCT;
      break;
    case UBSEC_MATH_MODINV :
      pContext->operation_type	= OPERATION_MOD_INVERSE;
      break;

    default:
      Status=(UBSEC_STATUS_INVALID_PARAMETER);
      goto Error_Return;
    }

    pParams=&pCommand->Parameters;

    /* Clear the context. */
    memset(pMathContext,0,sizeof(Math_CtxCmdBuf_t));

    pContext->cmd_structure_length= MATH_STATIC_CONTEXT_SIZE;

    /* The modulus needs to be aligned on a 512/768 or 1024 bit boundary. */
    /*
     * Save amount to normalize/renormalize.
     */
    if (pParams->ModN.KeyLength <=512)
      NormalizeLen=512;
    else
      if (pParams->ModN.KeyLength <=768)
	NormalizeLen=768;
      else
	if (pParams->ModN.KeyLength <=1024)
	  NormalizeLen=1024;
	else
#ifdef UBSEC_582x_CLASS_DEVICE
	  if (pParams->ModN.KeyLength <=1536)
	    NormalizeLen=1536;
	  else
	    NormalizeLen=2048;
#else
          return(UBSEC_STATUS_INVALID_PARAMETER);
#endif
    

#ifndef UBSEC_HW_NORMALIZE
    if ((NrmBits = ubsec_NormalizeDataTo(&pParams->ModN,NormalizeLen))) {
      Dbg_Print(DBG_FATAL,("ubsec: MATH NrmBits %d\n",NrmBits));
      pMCR->KeyContextList[PacketIndex]->ResultKey=pParams->Result;
      ubsec_ShiftData(&pParams->ParamA, NrmBits);
    }

    pMCR->KeyContextList[PacketIndex]->NormBits=NrmBits;
#else
    NrmBits=0;
#endif

    /*
     * Output value may need to be rounded up to represent an integral
     * number of 32 bit words, same total length as modulus N.
     */

    /* N Copy the modulo value modulo */
    pMathContext->modulus_length = (unsigned short)CPU_TO_CTRL_SHORT(pParams->ModN.KeyLength);
    offset=NormalizeLen/8;

#ifndef UBSEC_HW_NORMALIZE
    RTL_Memcpy(&pMathContext->NE[0],(pParams->ModN.KeyValue),offset);
#else
    RTL_Memcpy(&pMathContext->NE[0],(pParams->ModN.KeyValue),
	       ROUNDUP_TO_32_BIT(pParams->ModN.KeyLength)/8);
#endif

    /* Now if required copy the exponent value. */
    if (pCommand->Command==UBSEC_MATH_MODINV) {
#ifndef UBSEC_HW_NORMALIZE
      ubsec_ShiftData(&pParams->ExpE, NrmBits);
#endif
      pMathContext->exponent_length=(unsigned short)CPU_TO_CTRL_SHORT(pParams->ExpE.KeyLength);
#ifndef UBSEC_HW_NORMALIZE
      RTL_Memcpy( &pMathContext->NE[(offset/4)],
		  (pParams->ExpE.KeyValue),offset);
#else
      RTL_Memcpy( &pMathContext->NE[(offset/4)],
		  (pParams->ExpE.KeyValue),
		  ROUNDUP_TO_32_BIT(pParams->ExpE.KeyLength)/8);
#endif
      offset+=(pParams->ExpE.KeyLength+31)/8; /* Round up to DWORD Bytes */
    }
    else   { /* Message length needs to be present in context for modrem/exp */
      if (pCommand->Command==UBSEC_MATH_MODREM)
	pMathContext->exponent_length=
	  (unsigned short)CPU_TO_CTRL_SHORT(pParams->ParamA.KeyLength);
      else
	if (pCommand->Command==UBSEC_MATH_MODEXP)
  	  pMathContext->exponent_length= 
  	    (unsigned short)CPU_TO_CTRL_SHORT(pParams->ParamB.KeyLength); 
    } 
    
    /* Now set the total context length. */
    pContext->cmd_structure_length+=(offset);

#ifdef UBSDBG
  /* Print out the context information if required */
  {
    int WordLen,i;
    WordLen=(pContext->cmd_structure_length-MATH_STATIC_CONTEXT_SIZE)/4;
    Dbg_Print(DBG_MATH,(   "ubsec:  ---- DH Math ModN Length Len [%d] ExpE Len [%d]\n",
			   CTRL_TO_CPU_SHORT(pMathContext->modulus_length),
			   CTRL_TO_CPU_SHORT(pMathContext->exponent_length))); 
    Dbg_Print(DBG_MATH,(   "Context Len %d Context Value=[",
			    (pContext->cmd_structure_length))); 
    for ( i=0 ; i < WordLen ; i++) {
      Dbg_Print(DBG_MATH,( "%08x ",SYS_TO_BE_LONG(pMathContext->NE[i])));
    }
    Dbg_Print(DBG_MATH,( "]\n"));
    }
#endif

    /* Now do the Input parameter A. All operations have one */
    FragPtr=(DataBufChainList_pt)&pPacket->InputHead;
    PhysAddr=virt_to_bus(pParams->ParamA.KeyValue); 
    FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);

    DataLength=(unsigned short)NormalizeLen/8;

    FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );

    Dbg_Print(DBG_MATH,( "Input Param 1: <%d,%08x (%08x)>\n",
			 DataLength, 
			 CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
    
    /* For some operations we need a second input parameter */
    if ((pCommand->Command!=UBSEC_MATH_MODINV) && (pCommand->Command!=UBSEC_MATH_MODREM)) {
#ifndef UBSEC_HW_NORMALIZE
			if (pCommand->Command!=UBSEC_MATH_MODEXP)
				ubsec_ShiftData(&pParams->ParamB, NrmBits);
#endif
      /* get the next fragment pointer */
      NextFragPtr=&pMCR->InputFragmentList[PacketIndex*(UBSEC_MAX_FRAGMENTS)];
      FragPtr->pNext =NextFragPtr->PhysicalAddress;
      FragPtr=NextFragPtr;
      /* The second input data buffer has 2nd math parameter (B or E) */
      PhysAddr=virt_to_bus(pParams->ParamB.KeyValue); 
      FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
#if 0
      DataLength=ROUNDUP_TO_32_BIT(pParams->ParamB.KeyLength+NrmBits);
#else
      if (pCommand->Command==UBSEC_MATH_MODEXP)
	DataLength = (CTRL_TO_CPU_SHORT(pMathContext->exponent_length) + 7)/8;
      else
	DataLength = (unsigned short)NormalizeLen/8;
#endif
      FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
      Dbg_Print(DBG_MATH,( "Input Param 2: <%d,%08x (%08x)>\n",
			 DataLength, 
			 CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));

#ifndef STATIC_F_LIST
      Dbg_Print(DBG_FRAG_SYNC,( "ubsec: ubsec_MathCommand Sync IFrag Descriptor to Device (0x%08X,%d,%d)\n", pMCR->InputFragmentListHandle,
	      PacketIndex*(UBSEC_MAX_FRAGMENTS)*sizeof(DataBufChainList_t),
	      sizeof(DataBufChainList_t)));
      OS_SyncToDevice(pMCR->InputFragmentListHandle,
	    PacketIndex*(UBSEC_MAX_FRAGMENTS)*sizeof(DataBufChainList_t),
	    sizeof(DataBufChainList_t));
#endif

    } /* end of two-input-fragments code */

    FragPtr->pNext=0; /* Terminate the input fragment descriptor list. */

    /* Now do the Output data buffer. All operations have exactly one */

    FragPtr=(DataBufChainList_pt)&pPacket->OutputHead;
    PhysAddr=virt_to_bus(pParams->Result.KeyValue); 
    FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);

    DataLength=(unsigned short)NormalizeLen/8;

    FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
    FragPtr->pNext=0; /* Terminate. */

    Dbg_Print(DBG_MATH,( "Result : <%d,%08x (%08x)>\n",
			 DataLength, 
			 CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));

#ifdef UBSDBG
      /* Sanity check debug info for conditions that will hang the chip. */
    if ( (int) (CTRL_TO_CPU_LONG( FragPtr->DataAddress )) & 0x03) {
      Dbg_Print(DBG_FATAL,("ubsec:MATH #########INVALID OUTPUT ADDRESS %08x\n", FragPtr->DataAddress));
      Status=UBSEC_STATUS_INVALID_PARAMETER;
      goto Error_Return;
    }
    if ((DataLength) & 0x03) {
      Dbg_Print(DBG_FATAL,("ubsec:MATH #########INVALID OUTPUT LENGTH %08x\n", DataLength)); 
      Status=UBSEC_STATUS_INVALID_PARAMETER;
      goto Error_Return;
    }
#endif 

#ifndef UBSEC_HW_NORMALIZE
    /* 
     *
     * If we have normalized the data then we need to
     * shift back the results. We will do this by calling
     * an intermediate callback function to do the job.
     */
    if (pContext->NormBits) {
      /* First save the user parameters */
      pContext->UserCallback = pCommand->CompletionCallback;
      pContext->UserContext = pCommand->CommandContext;
      /* Now setup the intermediate routine */
      pCompletionContext->CommandContext = (unsigned long)pContext;
      pCompletionContext->CompletionCallback = KeyUnShiftResult;
    }
    else {
      pCompletionContext->CompletionCallback = pCommand->CompletionCallback;
      pCompletionContext->CommandContext = pCommand->CommandContext;
    }
#else
    pCompletionContext->CompletionCallback = pCommand->CompletionCallback;
    pCompletionContext->CommandContext = pCommand->CommandContext;
#endif

#if (UBS_CPU_ATTRIBUTE != UBS_CRYPTONET_ATTRIBUTE)
    pContext->cmd_structure_length= CPU_TO_CTRL_SHORT(pContext->cmd_structure_length);
#endif

    /* For key (math) MCRs, contexts are accessed by an array of handles. */
    /* This means that memory for each context was separately allocated.  */
    /* Therefore we must sync each context separately as it is built.     */
    Dbg_Print(DBG_CNTXT_SYNC,( "ubsec: ubsec_MathCommand Sync Context to Device (0x%08X,%d,%d)\n", pMCR->ContextListHandle[PacketIndex],
		       0,
		       CTRL_TO_CPU_SHORT(pContext->cmd_structure_length)));
    OS_SyncToDevice(pMCR->ContextListHandle[PacketIndex],
		    0,
		    CTRL_TO_CPU_SHORT(pContext->cmd_structure_length));

    /* Now inc the number of packets and prepare for the next command. */
    pMCR->NumberOfPackets++;
    pCommand++;
    PacketIndex++;
    pPacket++;
    pContext++;

  } /* For NumCommands-- */

  /*
   * If we are here then the MCR is built.
   * Push it to the device. 
   */
  *NumCommands=CommandIndex; /* Update number completed */
  PushMCR(pDevice);

#ifdef BLOCK 
  /* Wait for all outstanding  to complete */
    while ((Status=WaitForCompletion(pDevice,(unsigned long)1000000,UBSEC_KEY_LIST))
	   == UBSEC_STATUS_SUCCESS);
    if (Status!=UBSEC_STATUS_TIMEOUT) /* We are nested, return success */
      Status=UBSEC_STATUS_SUCCESS;
 Error_Return:
#else

 Error_Return:  /* Label to make sure that IRQs are enabled. */
#ifdef COMPLETE_ON_COMMAND_THREAD
    ubsec_PollDevice(pDevice);  /* Try to complete some & cut down on ints */
#endif

#endif
    OS_LeaveCriticalSection(pDevice,SaveConfig);
    return(Status);
}
