
/*
 * Broadcom Cryptonet Driver software is distributed as is, without any warranty
 * of any kind, either express or implied as further specified in the GNU Public
 * License. This software may be used and distributed according to the terms of
 * the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */

/******************************************************************************
 *
 * Copyright 2000
 * Broadcom Corporation
 * 16215 Alton Parkway
 * PO Box 57013
 * Irvine CA 92619-7013
 *
 *****************************************************************************/

/* 
 * Broadcom Corporation uBSec SDK 
 */

/*
 * ubsdh.c: Diffie Hellman parameter setup functions.
 *
 * Revision History:
 *
 * May 2000 SOR Created
 * Sep 2000 SOR 5820 upgrade
 * 04/20/2001 RJT Added support for CPU-DMA memory synchronization
 * 04/24/2001 DPA Allow for unnormalize of D-H random number (x) output for BCM5805
 * 07/16/2001 RJT Added support for BCM5821
 */

#include "ubsincl.h"

#ifdef UBSEC_PKEY_SUPPORT

/*
 * DH_SetupPublicParams:
 */
long 
DH_SetupPublicParams(volatile MasterCommand_pt 	pMCR,
		     ubsec_DH_Params_pt pDHParams)

     /* pDHParams points to a structure which contains all of the info
	needed for Diffie-Hellman operations. In addition to regular
	numerical parameters, memory buffer locations are passed as follows:

	  pDHParams->Y.KeyValue          DMA memory handle 
	  pDHParams->N.KeyValue          virtual pointer 
	  pDHParams->G.KeyValue          virtual pointer 
	  pDHParams->X.KeyValue          DMA memory handle  
	  pDHParams->UserX.KeyValue      DMA memory handle (optional)  

	Note that pointers and handles must be treated differently. 
	Pointers are virtual addresses of actual target data, while
	handles are not; they are an abstraction of the (DMA) memory.
	The virtual and physical pointers associated with the handle 
	must be extracted by using the following macros:

	  ()
	  virt_to_bus()
	
	Results from virt_to_bus() may be written to CryptoNet
	control structures in (DMA) memory. 
	Results from () may be used (if necessary) as
	regular old pointers.
     */

{
#ifdef UBSEC_DH_SUPPORT
  volatile DataBufChainList_t  *FragPtr, *NextFragPtr;
  unsigned short 		DataLength;
  volatile Packet_t *pPacket;
  DH_Send_CtxCmdBuf_t *pDHSendCtx;
  volatile int             	PacketIndex;
  long Status=UBSEC_STATUS_SUCCESS;
  int NgLen;
  int NormalizeLen,NrmBits;
  unsigned char *pNg;
  unsigned long PhysAddr;

  PacketIndex = pMCR->NumberOfPackets; 
  pDHSendCtx = &pMCR->KeyContextList[PacketIndex]->CtxCmdBuf.DH_Send_CtxCmdBuf; 
  memset(pDHSendCtx,0,sizeof(*pDHSendCtx));
  pPacket = &(pMCR->PacketArray[PacketIndex]); /* Set up the current packet. */

  pDHSendCtx->rng_enable= CPU_TO_CTRL_SHORT(pDHParams->RNGEnable); 

  /* The modulus needs to be aligned on a 512/768 or 1024 bit boundary.
   (2048 for 5820) */

  /*
   * Save amount to normalize/renormalize.
   */
  if (pDHParams->N.KeyLength <=512)
    NormalizeLen=512;
  else
    if (pDHParams->N.KeyLength <=768)
      NormalizeLen=768;
    else
      if (pDHParams->N.KeyLength <=1024)
	NormalizeLen=1024;
      else
#ifdef UBSEC_582x_CLASS_DEVICE
	if (pDHParams->N.KeyLength <=1536)
	  NormalizeLen=1536;
	else
	  if (pDHParams->N.KeyLength <= 2048)
	    NormalizeLen=2048;
	  else
#endif
        return(UBSEC_STATUS_INVALID_PARAMETER);

#ifndef UBSEC_HW_NORMALIZE
  if ((NrmBits = ubsec_NormalizeDataTo(&pDHParams->N,NormalizeLen))) {
      ubsec_ShiftData(&pDHParams->G, NrmBits);
      pMCR->KeyContextList[PacketIndex]->ResultKey=pDHParams->Y;
  }

  pMCR->KeyContextList[PacketIndex]->NormBits=NrmBits;
#else
  NrmBits=0;
#endif

  /*
   * Output Y value may need to be rounded up to represent an integral
   * number of 32 bit words, same total length as modulus N.
   */

#ifndef UBSEC_HW_NORMALIZE
  pDHParams->Y.KeyLength = NormalizeLen;    
#else
#if 0
  pDHParams->Y.KeyLength = ROUNDUP_TO_32_BIT(pDHParams->Y.KeyLength);
#else
  pDHParams->Y.KeyLength = NormalizeLen;    
#endif
#endif

  /* Now setup some of the parameters that need to be aligned. */
  pDHParams->X.KeyLength = ROUNDUP_TO_32_BIT(pDHParams->X.KeyLength);
#ifndef UBSEC_HW_NORMALIZE
  /* now save output X info, in case needed for unshifting result */
  pMCR->KeyContextList[PacketIndex]->ResultRNG=pDHParams->X;
#endif

	/* N Copy the modulo value */
  pDHSendCtx->modulus_length = (unsigned short)CPU_TO_CTRL_SHORT(pDHParams->N.KeyLength);
  pNg=(unsigned char *)&pDHSendCtx->Ng[0]; /* For convenience */
  NgLen=NormalizeLen/8;
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy(&pNg[0],pDHParams->N.KeyValue,NgLen);
#else
  RTL_Memcpy(&pNg[0],pDHParams->N.KeyValue,ROUNDUP_TO_32_BIT(pDHParams->N.KeyLength)/8);
#endif

  /*
   * G  Copy the input value. This is the public key for private
   * operation or the Baseg value for public key generation.
   * It also needs to be aligned on the same  length
   * as N
   */
  pNg+=NgLen; /* Starting G location. */
  pDHSendCtx->generator_length=(unsigned short)CPU_TO_CTRL_SHORT(pDHParams->G.KeyLength);
#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy(&pNg[0],pDHParams->G.KeyValue,NgLen);
#else
  RTL_Memcpy(&pNg[0],pDHParams->G.KeyValue,ROUNDUP_TO_32_BIT(pDHParams->G.KeyLength)/8);
#endif
  
  /* "extra" length is always 2 x Ng length */
  pMCR->KeyContextList[PacketIndex]->cmd_structure_length+=(NgLen*2);

  /*
   * Input Buffer setup for DH Send (Public): 
   * If the private key x is provided by software, save
   * it in the first input data buffer, otherwise, the 
   * input data buffer will not be used by the chip
   */
  FragPtr=(DataBufChainList_pt)&pPacket->InputHead;
  if (! pDHParams->RNGEnable ) { /* Random number manually provided */
    pDHParams->UserX.KeyLength = ROUNDUP_TO_32_BIT(pDHParams->UserX.KeyLength);
    PhysAddr=virt_to_bus(pDHParams->UserX.KeyValue);
    FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
    DataLength=(unsigned short)(pDHParams->UserX.KeyLength+7)/8; /* map the length from bits to bytes */
    FragPtr->DataLength = CPU_TO_CTRL_SHORT(DataLength);
   pDHSendCtx->private_key_length = (unsigned short)CPU_TO_CTRL_SHORT(pDHParams->UserX.KeyLength);
    Dbg_Print(DBG_DHKEY,( "Public (Send) NormBits %d Input Key, UserX: <%d,%08x (%08x)>\n",NrmBits,DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr));
  }
  else { /* CryptoNet chip to internally generate random number X */
    pDHSendCtx->private_key_length=CPU_TO_CTRL_SHORT(pDHParams->RandomKeyLen);
    FragPtr->DataLength = 0;
    FragPtr->DataAddress = 0;
    DataLength=0;
  } /* Manual/CryptoNet random number generation if-else */
  /* The CryptoNet chip ignores the pPacket->PacketLength field for this */
  /* operation. We'll zero that field out for consistency's sake.        */
  pPacket->PacketLength = 0;
  FragPtr->pNext = 0; /* Terminate the (empty or single) input fragment list */
#ifdef UBSDBG
  /* Print out the context information if required */
  {
    int WordLen,i;
    WordLen=(pMCR->KeyContextList[PacketIndex]->cmd_structure_length-DH_STATIC_SEND_CONTEXT_SIZE)/4;
    Dbg_Print(DBG_DHKEY,(   "ubsec:  ---- DH_Public - RNG-Enable [%d] Private Klen [%d] Generator Len [%d]\n",
			    CTRL_TO_CPU_SHORT(pDHSendCtx->rng_enable),
			    CTRL_TO_CPU_SHORT(pDHSendCtx->private_key_length),
			    CTRL_TO_CPU_SHORT(pDHSendCtx->generator_length))); 

    Dbg_Print(DBG_DHKEY,(   "ubsec:  ---- Modulus Length [%d] ",
			    CTRL_TO_CPU_SHORT(pDHSendCtx->modulus_length))); 
    Dbg_Print(DBG_DHKEY,(   "Context Len %d Context Value=[",
			    (pMCR->KeyContextList[PacketIndex]->cmd_structure_length))); 
    for ( i=0 ; i < WordLen ; i++) {
      Dbg_Print(DBG_DHKEY,( "%08x ",SYS_TO_BE_LONG(pDHSendCtx->Ng[i])));
    }
    Dbg_Print(DBG_DHKEY,( "]\n"));

  }
#endif

  /* Output Buffer setup for DH Send (Public):
   * The output buffer has Public Key Y, followed by
   * Private Key X
   */
  FragPtr=(DataBufChainList_pt)&pPacket->OutputHead;
        /* The first output data buffer has Public Key Y */
  PhysAddr=virt_to_bus(pDHParams->Y.KeyValue); 
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pDHParams->Y.KeyLength+7)/8; /* map the length from bits to bytes */
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );

#ifdef UBSDBG
      /* Sanity check debug info for conditions that will hang the chip. */
  if ( (int) (CTRL_TO_CPU_LONG( FragPtr->DataAddress )) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT ADDRESS %08x\n", FragPtr->DataAddress));
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
  if ((DataLength) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT LENGTH %08x\n", DataLength)); 
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
#endif 
        /* get the next fragment pointer */
  NextFragPtr=&pMCR->OutputFragmentList[PacketIndex*(UBSEC_MAX_FRAGMENTS)];
  FragPtr->pNext =NextFragPtr->PhysicalAddress; 
  Dbg_Print(DBG_DHKEY,( "Public (Send) NormBits %d Output Key  Y: <%d,%08x, (%08x,Next-%08x)>\n",
      NrmBits,DataLength,CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr,FragPtr->pNext));
  FragPtr=NextFragPtr;
        /* The second output data buffer has Private Key X */
  PhysAddr=virt_to_bus(pDHParams->X.KeyValue); 
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pDHParams->X.KeyLength+7)/8; /* map the length from bits to bytes */
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );

  Dbg_Print(DBG_DHKEY,( "Public (Send) Output Key X: <%d, %08x, (%08x,Next-%08x)>\n",
      DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr,FragPtr->pNext));

#ifdef UBSDBG
      /* Sanity check debug info for conditions that will hang the chip. */
  if ( (int) (CTRL_TO_CPU_LONG( FragPtr->DataAddress )) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT_B ADDRESS %08x\n", FragPtr->DataAddress));
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
  if ((DataLength) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT_B LENGTH %08x\n", DataLength)); 
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
#endif 
  
  FragPtr->pNext = 0;

#ifndef STATIC_F_LIST
  /* Fragment lists are external to MCR structures, must sync separately */
  /* Always 2 output frags, need to sync one entry in OutputFragmentList */
  Dbg_Print(DBG_FRAG_SYNC,( "ubsec: DH_SetupPublicParams Sync OFrag Descriptor to Device (0x%08X,%d,%d)\n", pMCR->OutputFragmentListHandle,
	      PacketIndex*(UBSEC_MAX_FRAGMENTS)*sizeof(DataBufChainList_t),
	      sizeof(DataBufChainList_t)));
  OS_SyncToDevice(pMCR->OutputFragmentListHandle,
	    PacketIndex*(UBSEC_MAX_FRAGMENTS)*sizeof(DataBufChainList_t),
	    sizeof(DataBufChainList_t));
#endif /* STATIC_F_LIST not defined */

#ifdef UBSDBG
 Error_Return:
#endif 

  return(Status);
#else
    return(UBSEC_STATUS_NO_DEVICE);
#endif


}



/*
 * DH_SetupSharedParams:
 */
long 
DH_SetupSharedParams(volatile MasterCommand_pt 	pMCR,
		     ubsec_DH_Params_pt pDHParams)
{
#ifdef UBSEC_DH_SUPPORT
  volatile DataBufChainList_t  *FragPtr, *NextFragPtr;
  unsigned short 		DataLength;
  long Status=UBSEC_STATUS_SUCCESS;
  volatile Packet_t *pPacket;
  DH_REC_CtxCmdBuf_t *pDHRecCtx;
  volatile int             	PacketIndex;
  int NgLen;
  int NormalizeLen,NrmBits=0;
  unsigned long PhysAddr;

  PacketIndex = pMCR->NumberOfPackets; 
  pDHRecCtx = &pMCR->KeyContextList[PacketIndex]->CtxCmdBuf.DH_REC_CtxCmdBuf; 
  memset(pDHRecCtx,0,sizeof(*pDHRecCtx));
  pPacket = &(pMCR->PacketArray[PacketIndex]); /* Set up the current packet. */

  if (pDHParams->N.KeyLength <=512)
    NormalizeLen=512;
  else
    if (pDHParams->N.KeyLength <=768)
      NormalizeLen=768;
    else
      if (pDHParams->N.KeyLength <=1024)
	NormalizeLen=1024;
      else
#ifdef UBSEC_582x_CLASS_DEVICE
	if (pDHParams->N.KeyLength <=1536)
	  NormalizeLen=1536;
	else
	  NormalizeLen=2048;
#else
        return(UBSEC_STATUS_INVALID_PARAMETER);
#endif

    /*
     * Output K value may need to be rounded up to represent an integral
     * number of 32 bit words, same total length as modulus N.
     */
#ifndef UBSEC_HW_NORMALIZE
  pDHParams->K.KeyLength = NormalizeLen;    
  if ((NrmBits = ubsec_NormalizeDataTo(&pDHParams->N,NormalizeLen))) {
    ubsec_ShiftData(&pDHParams->Y, NrmBits);
    pMCR->KeyContextList[PacketIndex]->ResultKey=pDHParams->K;
  }
  pMCR->KeyContextList[PacketIndex]->NormBits=NrmBits;
#else
#if 1
  pDHParams->K.KeyLength = NormalizeLen;    
#else
  pDHParams->K.KeyLength = ROUNDUP_TO_32_BIT(pDHParams->K.KeyLength);
#endif
  NrmBits=0;
#endif


  pDHParams->Y.KeyLength = NormalizeLen;    

  /* Now setup some of the parameters that need to be aligned. */
  pDHParams->X.KeyLength = ROUNDUP_TO_32_BIT(pDHParams->X.KeyLength);

  NgLen=NormalizeLen/8;

  /* N Copy the modulo value modulo */
  pDHRecCtx->modulus_length = (unsigned short)CPU_TO_CTRL_SHORT(pDHParams->N.KeyLength);

#ifndef UBSEC_HW_NORMALIZE
  RTL_Memcpy( &pDHRecCtx->N[0],pDHParams->N.KeyValue,NgLen);
#else
  RTL_Memcpy( &pDHRecCtx->N[0],pDHParams->N.KeyValue,ROUNDUP_TO_32_BIT(pDHParams->N.KeyLength)/8);
#endif
  pMCR->KeyContextList[PacketIndex]->cmd_structure_length+=NgLen;

  /* Set the private key value */
  pDHRecCtx->modulus_length = (unsigned short)CPU_TO_CTRL_SHORT(pDHParams->N.KeyLength);
  pDHRecCtx->private_key_length=(unsigned short)CPU_TO_CTRL_SHORT(pDHParams->X.KeyLength);

#ifdef UBSDBG
  /* Print out the context information if required */
  {
  int WordLen,i;
  WordLen=(NgLen/4);
  Dbg_Print(DBG_DHKEY,(   "ubsec:  ---- DH Shared Mod Length [%d] Pkey Len [%d] Context Len %d, Value -\n[",
			  CTRL_TO_CPU_SHORT(pDHRecCtx->modulus_length),
			  CTRL_TO_CPU_SHORT(pDHRecCtx->private_key_length),		
			  pMCR->KeyContextList[PacketIndex]->cmd_structure_length)); 
  for ( i=0 ; i < WordLen ; i++) {
    Dbg_Print(DBG_DHKEY,( "%08x ",SYS_TO_BE_LONG(pDHRecCtx->N[i])));
  }
  Dbg_Print(DBG_DHKEY,( "]\n"));
  }
#endif

  /* Input Buffer setup for DH Receive (Shared): */
  FragPtr=(DataBufChainList_pt)&pPacket->InputHead;
     /* The first fragment has Y */
  PhysAddr=virt_to_bus(pDHParams->Y.KeyValue);
  FragPtr->DataAddress = (unsigned long) CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pDHParams->Y.KeyLength+7)/8; /* map the length from bits to bytes */
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
  /* The CryptoNet chip ignores the pPacket->PacketLength field for this */
  /* operation. We'll zero that field out for consistency's sake.        */
  pPacket->PacketLength = 0;
        /* get the next fragment pointer */
  NextFragPtr=&pMCR->InputFragmentList[PacketIndex*(UBSEC_MAX_FRAGMENTS)];
  FragPtr->pNext =NextFragPtr->PhysicalAddress; 
  Dbg_Print(DBG_DHKEY,( "DH Shared  Y:  FragI <%d,%08x %08x,Next-%08x>\n",
		      DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr,FragPtr->pNext));

  FragPtr=NextFragPtr;
        /* The second Input data buffer has Private Key x */
  PhysAddr=virt_to_bus(pDHParams->X.KeyValue);
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pDHParams->X.KeyLength+7)/8; /* map the length from bits to bytes */
  FragPtr->DataLength = CPU_TO_CTRL_SHORT( DataLength );
  Dbg_Print(DBG_DHKEY,( "Shared Private Key X: <%d, %08x, (%08x,Next-%08x)>\n",
      DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress ), FragPtr,FragPtr->pNext));
  FragPtr->pNext = 0;

        /* Output Buffer setup for DH Received (Shared):
	 * The output buffer contains shared secret K 
	 */
  FragPtr=(DataBufChainList_pt)&pPacket->OutputHead;
  PhysAddr=virt_to_bus(pDHParams->K.KeyValue); 
  FragPtr->DataAddress = CPU_TO_CTRL_LONG(PhysAddr);
  DataLength=(unsigned short)(pDHParams->K.KeyLength+7)/8; /* map the length from bits to bytes */
  FragPtr->DataLength = (unsigned short)CPU_TO_CTRL_SHORT( DataLength );

#ifdef UBSDBG
      /* Sanity check debug info for conditions that will hang the chip. */
  if ( (int) (CTRL_TO_CPU_LONG( FragPtr->DataAddress )) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT ADDRESS %08x\n", FragPtr->DataAddress));
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
  if ((DataLength) & 0x03) {
    Dbg_Print(DBG_FATAL,("ubsec:#########INVALID OUTPUT LENGTH %08x\n", DataLength)); 
    Status=UBSEC_STATUS_INVALID_PARAMETER;
    goto Error_Return;
  }
#endif 
  Dbg_Print(DBG_DHKEY,( "Receive Buffer K:FragO <%d, %08x>\n",
		      DataLength, CTRL_TO_CPU_LONG( FragPtr->DataAddress )));
  FragPtr->pNext = 0;


#ifdef UBSDBG
 Error_Return:
#endif 


  return(Status);
#else
    return(UBSEC_STATUS_NO_DEVICE);
#endif

}


#endif
