
/*
 *  Broadcom Cryptonet Driver software is distributed as is, without any warranty
 *  of any kind, either express or implied as further specified in the GNU Public
 *  License. This software may be used and distributed according to the terms of
 *  the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */
/******************************************************************************
 *
 *  Copyright 2000
 *  Broadcom Corporation
 *  16215 Alton Parkway
 *  PO Box 57013
 *  Irvine CA 92619-7013
 *
 *****************************************************************************/

/*******************************************************************************
 *
 * File: Linux/sslarc4.c
 * 
 * Description: 
 *              Character driver interface to SSL ARC4 routines in the 
 *              ubsec driver.
 *
 * Revision History:
 *                   When       Who   What
 *                   10/11/00   DNA   Created
 * March 2001 PW Release for Linux 2.4 UP and SMP kernel
 *
 ******************************************************************************/

#include "cdevincl.h"

/* This is useful only for diagnostics. */
#undef STATIC_ALLOC_OF_CRYPTO_BUFFERS

#ifdef POLL
#undef GOTOSLEEP
#endif

#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS

unsigned char *  pKernSourceBuffer = NULL;
unsigned char *  pKernDestBuffer   = NULL;
unsigned char *  pKernStateBuffer  = NULL;
static int       PhysSourceBuf;
static int       PhysDestBuf;
static int       PhysStateBuf;

#endif /* ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS */


/**************************************************************************
 *
 * Function:  init_arc4if
 * 
 * Called from: init_module() in Linux/dispatch.c
 *
 * Description: 
 *              Buffer sent to SRL is not malloc'd here after all, to allow 
 *              for re-entrant code.
 *
 * Return Values:
 *                == 0   =>   Success
 *                != 0   =>   Failure
 *
 *************************************************************************/

int 
init_arc4if(void) {
  
  int error = 0;
  
#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  
  pKernSourceBuffer = pKernDestBuffer = (char *)kmalloc((MAX_FILE_SIZE),GFP_KERNEL|GFP_ATOMIC);
  if(pKernSourceBuffer == NULL) {
    printk("Cryptonet: no memory for source buffer %d\n", MAX_FILE_SIZE);
    error = -ENOMEM;
    return(error);
  }
  memset(pKernSourceBuffer,0, MAX_FILE_SIZE);
  
  pKernStateBuffer = (char *)kmalloc((260),GFP_KERNEL|GFP_ATOMIC);
  if(pKernStateBuffer == NULL) {
    printk("Cryptonet: no memory for source buffer %d\n", 260);
    error = -ENOMEM;
    return(error);
  }
  memset(pKernStateBuffer,0, 260);
  
#ifdef DEBUG
  printk("Allocate Source: %x %x\n", pKernSourceBuffer, vtophys(pKernSourceBuffer));
  printk("Allocate State: %x %x\n", pKernStateBuffer, vtophys(pKernStateBuffer));
#endif
  
  PhysSourceBuf = PhysDestBuf = virt_to_bus(pKernSourceBuffer);
  PhysStateBuf  = virt_to_bus(pKernStateBuffer);

#ifdef DEBUG
  printk("Cryptonet: Memory Alloc Source %x %x Dest %x %x for source buffer\n",
	 pKernSourceBuffer, PhysSourceBuf, KernDestBuffer, PhysDestBuf);
  printk("Cryptonet: Memory Alloc State %x %x state buffer\n", pKernStateBuffer, PhysStateBuf);
#endif
  
#endif /* ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS */
  
  return(error);
}


/**************************************************************************
 *
 * Function: shutdown_arc4if
 *
 * Called from: cleaup_module() in Linux/dispatch.c
 *
 * Description:
 *
 *************************************************************************/

void 
shutdown_arc4if(void) {
  
#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  
  if(pKernSourceBuffer != NULL)
    kfree(pKernSourceBuffer);
  if(pKernStateBuffer != NULL)
    kfree(pKernStateBuffer);
  
#endif /* ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS */
  
  return;
}

/**************************************************************************
 *
 * Function:  ubsec_arc4
 * 
 * Called from: ubsec_ioctl() in Linux/dispatch.c
 *
 * Purpose:
 *          Wrapper function between user call and call to SRL.
 *
 * Description: 
 *              Prepare all the data from the user's call for a call to 
 *              the SRL, set up the timer, and finally place this thread
 *              on the wait queue and go to sleep. When ubsec_arc4 is 
 *              called, the relevant data for an SSL ARC4 op reside in
 *              user space. The struct ubecom_arc4_io_pt argument, pIOInfo, 
 *              points to data on the stack and in kernal space. However, 
 *              pIOInfo is a struct that contains pointers which still 
 *              point to memory in user space. ubecom_ssl copies the 
 *              contents of pIOInfo as well as the input data it points
 *              to into memory allocated in kernal space with LinuxAllocateMemory.
 *
 * Return Values: 
 *                == 0   =>   Success
 *                != 0   =>   Failure
 *
 *************************************************************************/

int
ubsec_sslarc4(
	      ubsec_DeviceContext_t pContext,
	      ubsec_arc4_io_pt *    ppIOInfo
	      ) {
  
  ubsec_arc4_io_t               IOInfo;
  ubsec_arc4_io_pt              pIOInfo           = &IOInfo;
  ubsec_SSLCommandInfo_t        SslCommand;
  ubsec_SSLCommandInfo_pt       pSslCommand       = &SslCommand;
  ubsec_ARC4Params_pt           pArc4Params       = &(pSslCommand->Parameters.ARC4Params);
  volatile CommandContext_t  CommandContext;
  CommandContext_t          *pCommandContext   = &CommandContext;
  int                           NumCommands       = 1;
  unsigned long		        delayTotalUs      = 0;
  ubsec_FragmentInfo_pt         pSourceFragments  = NULL;
  unsigned char *               pUserSourceBuffer = pIOInfo->SourceBuffer;
  unsigned int                  SourceBufferBytes = pIOInfo->SourceBufferBytes;
  int                           error             = 0;
  int                           i                 = 0;

#ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  
  unsigned char *               pKernSourceBuffer = NULL;
  unsigned char *               pKernDestBuffer   = NULL;
  unsigned char *               pKernStateBuffer  = NULL;
  int                           PhysSourceBuf;
  int                           PhysDestBuf;
  int                           PhysStateBuf;
  
#endif /* ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS */

  copy_from_user(pIOInfo, *ppIOInfo, sizeof(ubsec_arc4_io_t));
  pUserSourceBuffer = pIOInfo->SourceBuffer;
  SourceBufferBytes = pIOInfo->SourceBufferBytes;

  /* 
   *  First, allocate memory for the source buffer (which doubles as 
   *  the destination buffer) and the state output buffer. Translate
   *  both addresses to physical addresses and clear this memory.
   */
  
#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  
  if(SourceBufferBytes > MAX_FILE_SIZE) {
    printk("Cryptonet: input file too large <%d,%d>\n", SourceBufferBytes, MAX_FILE_SIZE);
    error = EINVAL;
    return(error);
  }
  
#else
  
  pKernSourceBuffer = pKernDestBuffer = (char *)kmalloc((SourceBufferBytes),GFP_KERNEL|GFP_ATOMIC);
  if(pKernSourceBuffer == NULL) {
    printk("Cryptonet: no memory for source buffer %d\n", SourceBufferBytes);
    error = -ENOMEM;
    goto ReturnErrorLabel;
  }
  memset(pKernSourceBuffer,0, SourceBufferBytes);
  
  pKernStateBuffer = (char *)kmalloc((260),GFP_KERNEL|GFP_ATOMIC);
  if(pKernStateBuffer == NULL) {
    printk("Cryptonet: no memory for source buffer %d\n", 260);
    error = -ENOMEM;
    goto ReturnErrorLabel;
  }
  memset(pKernStateBuffer,0, 260);
  
#ifdef DEBUG
  printk("Allocate Source: %x %x\n", pKernSourceBuffer, vtophys(pKernSourceBuffer));
  printk("Allocate State: %x %x\n", pKernStateBuffer, vtophys(pKernStateBuffer));
#endif /* ifdef DEBUG */
  
  PhysSourceBuf = PhysDestBuf = virt_to_bus(pKernSourceBuffer);
  PhysStateBuf  = virt_to_bus(pKernStateBuffer);

#ifdef DEBUG
  printk("Cryptonet: Memory Alloc Source %x %x Dest %x %x for source buffer\n",
	 pKernSourceBuffer, PhysSourceBuf, pKernDestBuffer, PhysDestBuf);
  printk("Cryptonet: Memory Alloc State %x %x state buffer\n", pKernStateBuffer, PhysStateBuf);
#endif
  
#endif /* ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS */
  
  /*
   * Next fill data buffers and data structures with appropriate data
   * for this ARC4 op.
   */

  pArc4Params->KeyStateFlag = 0;
  pArc4Params->KeyStateFlag = UBSEC_ARC4_STATE_WRITEBACK; /* Always write back */
  if(pIOInfo->KeyStateFlag == ARC4_KEY) 
    pArc4Params->KeyStateFlag |= UBSEC_ARC4_STATE_STATEKEY;

#if DEBUG
  printk("keystateflag io = %d\n", pIOInfo->KeyStateFlag);
  printk("keystateflag pa = %d\n", pArc4Params->KeyStateFlag);
#endif
  
  /* 
   *  Format of pIOInfo->KeyState buffer is same as context command structure
   *  with state, index_i and index_j. Format of pARC4Params->KeyState is
   *  key/state only, no index_i nor index_j.
   */
  
  pArc4Params->KeyState = (ubsec_ARC4_State_pt)kmalloc((256),GFP_KERNEL|GFP_ATOMIC);
  if(pArc4Params->KeyState == NULL) {
    printk("Cryptonet: no memory for keystate buffer %d\n", 256);
    error = -ENOMEM;
    goto ReturnErrorLabel;
  }
  memset(pArc4Params->KeyState,0, 256);
  copy_from_user(pArc4Params->KeyState, &(pIOInfo->IndicesKeyState[4]), 256);
  copy_from_user(&pArc4Params->index_i, &(pIOInfo->IndicesKeyState[2]), 1);
  copy_from_user(&pArc4Params->index_j, &(pIOInfo->IndicesKeyState[0]), 1);

  /* Assemble Source Fragments */
  
  pSourceFragments = kmalloc((((sizeof(ubsec_FragmentInfo_t) * UBSEC_MAX_FRAGMENTS))),GFP_KERNEL|GFP_ATOMIC);
  if(pSourceFragments == NULL) {
    printk("Cryptonet: no memory for fragment buffer\n");
    error = -ENOMEM;
    return(error);
  }
  
  pSslCommand->SourceFragments = pSourceFragments;
  pSslCommand->NumSource       = SetupFragmentList(pSslCommand->SourceFragments,
						   pKernSourceBuffer,
						   SourceBufferBytes);
  
  copy_from_user(pKernSourceBuffer, pUserSourceBuffer, SourceBufferBytes);
  
  /* Assemble Destination Fragments */

  pSslCommand->NumDestination            = pSslCommand->NumSource;
  pSslCommand->DestinationFragments      = pSslCommand->SourceFragments;
  pArc4Params->state_out.FragmentAddress = PhysStateBuf;
  pArc4Params->state_out.FragmentLength  = 260;
  
  pCommandContext->CallBackStatus = 0;
  pSslCommand->CommandContext     = pCommandContext;
  pSslCommand->CompletionCallback = CmdCompleteCallback;
  pSslCommand->Command            = UBSEC_ARC4;
  
  start_time(&(pCommandContext->tv_start));
  
#ifdef DEBUG
  printk("Linux:ubsec_sslarc4 just before SRL call...\n");
#endif

#ifndef LINUX2dot2
  init_waitqueue_head(&pCommandContext->WaitQ);
#else
   pCommandContext->WaitQ         = 0;
#endif 

  pIOInfo->result_status = ubsec_SSLCommand(pContext, pSslCommand, &NumCommands);
  
  switch(pIOInfo->result_status) {
  case UBSEC_STATUS_SUCCESS:
    break;
    
  case UBSEC_STATUS_TIMEOUT:
    printk("ubsec error: ubsec_SslCommand() timeout\n");
    ubsec_ResetDevice(pContext);
    error = ETIMEDOUT;
    goto ReturnErrorLabel;
    break;
    
  case UBSEC_STATUS_INVALID_PARAMETER:
    printk("ubsec error:  ubsec_SslCommand() invalid parameter\n");
    error = EINVAL;
    goto ReturnErrorLabel;
    break;
    
  case UBSEC_STATUS_NO_RESOURCE:
    printk("ubsec error: ubsec_SslCommand() no resource. Number done: %d\n", NumCommands);
    error = ENOBUFS;
    goto ReturnErrorLabel;
    break;
    
  default:
    error = ENOMSG;
    goto ReturnErrorLabel;
    break;
  }
  
#ifdef GOTOSLEEP
  
  if(!(CommandContext.CallBackStatus))  /* Just in case completed on same thread. */
    Gotosleep(&CommandContext.WaitQ);
    if (!pCommandContext->CallBackStatus) {
         pCommandContext->Status=UBSEC_STATUS_TIMEOUT;
    }
#else
  
  for(delayTotalUs = 1; !(CommandContext.CallBackStatus); delayTotalUs++) {
    
#ifdef POLL
    
    /* We need to poll the device if we are operating in POLL mode. */
    ubsec_PollDevice(pContext);
    
#endif
    
    if(delayTotalUs >= 3000000) {
      error = ETIMEDOUT;
      goto ReturnErrorLabel;
    }
    udelay(1);
  }
  
#endif

#if DEBUG
  printk("Linux: Dest Buffer (post-SRL)\n");
  for(i = 0; i < 32; i++) {
    printk("%02X", pKernDestBuffer[i]);
  }
  printk("\n");
#endif
#if DEBUG
  printk("Linux: State Buffer (post-SRL)\n");
  for(i = 0; i < 260; i++) {
    printk("%02X", pKernStateBuffer[i]);
  }
  printk("\n");
#endif
  
  pIOInfo->time_us = CommandContext.tv_start.tv_sec * 1000000 + CommandContext.tv_start.tv_usec;
  if(pIOInfo->result_status == UBSEC_STATUS_SUCCESS) {
    copy_to_user(pIOInfo->DestBuffer, pKernDestBuffer, SourceBufferBytes);
    copy_to_user(pIOInfo->StateOutBuffer, pKernStateBuffer, 260);
    copy_to_user(*ppIOInfo, pIOInfo, sizeof(ubsec_arc4_io_t));
  }
  
 ReturnErrorLabel:
  
#ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  
  if(pKernSourceBuffer != NULL)
    kfree(pKernSourceBuffer);
  if(pKernStateBuffer != NULL)
    kfree(pKernStateBuffer);
  if(pArc4Params->KeyState != NULL)
    kfree(pArc4Params->KeyState);
  
#endif /* ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS */
  
  if(pSourceFragments != NULL)
    kfree(pSourceFragments);
  
  return(error);
}

