
/*
 *  Broadcom Cryptonet Driver software is distributed as is, without any warranty
 *  of any kind, either express or implied as further specified in the GNU Public
 *  License. This software may be used and distributed according to the terms of
 *  the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */
/******************************************************************************
 *
 *  Copyright 2000
 *  Broadcom Corporation
 *  16215 Alton Parkway
 *  PO Box 57013
 *  Irvine CA 92619-7013
 *
 *****************************************************************************/
/*
 * math.c: Character driver interface to MATH routines in the ubsec driver
 *
 * SOR
 * JJT
 * March 2001 PW Release for Linux 2.4 UP and SMP kernel
 */

#include "cdevincl.h"

/* Maximum key size in bytes */
#define MAX_KEY_BYTE_SIZE 256

/* Convert bit length to byte length */
#define BITSTOBYTES(bitsize) ((bitsize+7)/8)
/* Intermediate key copy location offsets */

/* Offset of Key information within kernel buffer. */
#define MATH_MODN_OFFSET 	0
#define MATH_EXPE_OFFSET 	1
#define MATH_PARAMA_OFFSET 	2
#define MATH_PARAMB_OFFSET 	3
#define MATH_RESULT_OFFSET 	4

#define MAX_NUM_MATH_PARAMS 5

#define POLL
#ifdef POLL
#undef GOTOSLEEP
#endif

#define ROUNDUP_BIT(n,m) ((((n)+(m)-1)/(m))*(m))
#define ROUNDUP_TO_32_BIT(n) ROUNDUP_BIT(n,32)

static int ubsec_mathsetup(unsigned long command, 
	ubsec_MathCommandParams_pt pIOparams, ubsec_MathCommandParams_pt pSRLparams, 	
	unsigned char *MathLoc);

void ubsec_pkey(ubsec_io_t *at);

/**************************************************************************
 *
 *  Function:  init_pkeyif
 *   
 *************************************************************************/
int init_mathif(void)
{
 return 0; /* success */
}



/**************************************************************************
 *
 *  Function:  cleanup_module
 *
 *************************************************************************/
void shutdown_mathif(void)
{
  return;
}

/*
 * Math function  setup function. Builds MathCommandInfo for call to SRL.
 * Returns number of bits to normalize.
 */
static int ubsec_mathsetup(unsigned long command, 
			   ubsec_MathCommandParams_pt pSRLparams, 	
			   ubsec_MathCommandParams_pt pIOparams, 
			   unsigned char *MathLoc)
{
  /*
   * Do a brute force copy of the math command. This will set the
   * lengths etc but we still need to set the pointers.
   */
  *pSRLparams=*pIOparams;

  /* 
   * Setup Math parameter locations and align them. Start with modulus N.
   */
  copy_from_user( &MathLoc[MAX_KEY_BYTE_SIZE*MATH_MODN_OFFSET],pIOparams->ModN.KeyValue,
	 BITSTOBYTES(pIOparams->ModN.KeyLength));
  /*  Modulus is a virtual address. */
  pSRLparams->ModN.KeyValue=(void *)&MathLoc[MAX_KEY_BYTE_SIZE*MATH_MODN_OFFSET];

  /*
   * Optionally do the exponent.
   */
  if (command==UBSEC_MATH_MODINV) {
    copy_from_user( &MathLoc[MAX_KEY_BYTE_SIZE*MATH_EXPE_OFFSET],pIOparams->ExpE.KeyValue,
	   BITSTOBYTES(pIOparams->ExpE.KeyLength));
    pSRLparams->ExpE.KeyValue=(void *)&MathLoc[MAX_KEY_BYTE_SIZE*MATH_EXPE_OFFSET];
    }

  /* Always copy in paramA */
  copy_from_user( &MathLoc[MAX_KEY_BYTE_SIZE*MATH_PARAMA_OFFSET],pIOparams->ParamA.KeyValue,
	 BITSTOBYTES(pIOparams->ParamA.KeyLength));
  pSRLparams->ParamA.KeyValue=(void *) (&MathLoc[MAX_KEY_BYTE_SIZE*MATH_PARAMA_OFFSET]);

  /* Optionally copy in paramB */
  if ((command!=UBSEC_MATH_MODINV) && (command!=UBSEC_MATH_MODREM)) {
    copy_from_user( &MathLoc[MAX_KEY_BYTE_SIZE*MATH_PARAMB_OFFSET],pIOparams->ParamB.KeyValue,
	 BITSTOBYTES(pIOparams->ParamB.KeyLength));
    pSRLparams->ParamB.KeyValue=(void *) (&MathLoc[MAX_KEY_BYTE_SIZE*MATH_PARAMB_OFFSET]);
  }
   
  pSRLparams->Result.KeyValue=(void *) (&MathLoc[MAX_KEY_BYTE_SIZE*MATH_RESULT_OFFSET]);
  return 0;
}

/*
 *
 */
int
ubsec_math(ubsec_DeviceContext_t pContext,
	   ubsec_math_io_pt *pIOInfo)
{
  ubsec_MathCommandInfo_pt	kcmd;
  ubsec_MathCommandParams_pt	pMathparams=NULL, 
				pIOparams = NULL;
  unsigned long			delay_total_us;
  unsigned char			*MathLoc;
  unsigned int			num_commands=1;
  ubsec_math_io_t		MathCommand;
  ubsec_math_io_pt		pMathIOInfo=&MathCommand;
  int				error = 0;
  CommandContext_pt		pCommandContext;
  char				*pmath_buf=NULL;

  pmath_buf = (char *) kmalloc((4096),GFP_KERNEL|GFP_ATOMIC);
  if( pmath_buf == NULL ) {
    printk("Cryptonet: no memory for math buffer\n");
    return -ENOMEM;
  }
  memset(pmath_buf,0,4096);

  copy_from_user( pMathIOInfo,*pIOInfo, sizeof(*pMathIOInfo));
  pIOparams=&pMathIOInfo->Math;

#if 0
#ifdef GOTOSLEEP
  WaitQ= (struct wait_queue *)pmath_buf;
  kcmd=&WaitQ[1];
#else
  kcmd = (ubsec_MathCommandInfo_pt)pmath_buf;
#endif
  tv_start =(struct timeval *)(&kcmd[1]);
  pStatus = (long *)(&tv_start[1]);
  pCallbackStatus = (int *)(&pStatus[1]);

	/* Set up starting key location. */
  MathLoc=(unsigned char *)&pCallbackStatus[1];
#endif

  pCommandContext = (CommandContext_pt)pmath_buf;
  kcmd = (ubsec_MathCommandInfo_pt)&pCommandContext[1];
  pMathparams=&kcmd->Parameters;
  MathLoc=(unsigned char *)&kcmd[1];

#ifndef LINUX2dot2
  init_waitqueue_head(&pCommandContext->WaitQ);
#else
   pCommandContext->WaitQ         = 0; 
#endif
    
	/*
	 * Now we need to format the command for the SRL.
	 * This depends on the type of the key command.
	 */
  switch (kcmd->Command=pMathIOInfo->command) {
  case UBSEC_MATH_MODADD :
  case UBSEC_MATH_MODSUB :
  case UBSEC_MATH_MODMUL :
  case UBSEC_MATH_MODEXP :
  case UBSEC_MATH_MODREM :
  case UBSEC_MATH_MODINV :
    ubsec_mathsetup(pMathIOInfo->command, &kcmd->Parameters,
			       &pMathIOInfo->Math, MathLoc);
    break;

  default:
    printk("ubsec CDEV: Invalid Math Command %lx\n",kcmd->Command);
    return EINVAL;
  }

  kcmd->CompletionCallback = CmdCompleteCallback;
  kcmd->CommandContext=(unsigned long)pmath_buf;

	/*
	 *  Let the system do anything it may want/need to do before we begin
	 *  timing.
	 */
  do_gettimeofday(&pCommandContext->tv_start);
  
  pCommandContext->CallBackStatus=0; /* inc'd on callback */
  switch (pMathIOInfo->result_status=ubsec_MathCommand(pContext,kcmd,&num_commands) ) {
  case UBSEC_STATUS_SUCCESS:
    break;
  case UBSEC_STATUS_TIMEOUT:
    printk("ubsec  ubsec_Command() Timeout\n");
    ubsec_ResetDevice(pContext);
    error = -ETIMEDOUT;
    goto Return;
    break;

  case UBSEC_STATUS_INVALID_PARAMETER:
    printk("Cryptonet:  ubsec_Command() Invalid parameter\n");
    error = -EINVAL;
    goto Return;
    break;

  case UBSEC_STATUS_NO_RESOURCE:
    printk("ubsec  ubsec_Command() No math resource. Num Done %d\n",num_commands);
    error = -ENOBUFS;
  default:
    error = -EIO;
    goto Return;
    break;
  }

#ifndef GOTOSLEEP      /* We need to poll the device if we are operating in POLL mode. */
  for (delay_total_us=1  ; !pCommandContext->CallBackStatus ; delay_total_us++) {
#ifdef POLL
    ubsec_PollDevice(pContext);
#endif
    if (delay_total_us >= 30000000) {
      printk("Cryptonet: Command timeout\n");
      pCommandContext->Status=UBSEC_STATUS_TIMEOUT;
      break;
    }
    udelay(1);
  }
#else
  if (!pCommandContext->CallBackStatus) { /* Just in case completed on same thread. */
    Gotosleep(&pCommandContext->WaitQ);
    if (!pCommandContext->CallBackStatus) {
              pCommandContext->Status=UBSEC_STATUS_TIMEOUT;
	      error = ETIMEDOUT;
    }                          
	goto Return;
    }
  }
#endif

  pMathIOInfo->result_status = pCommandContext->Status;
  pMathIOInfo->time_us = pCommandContext->tv_start.tv_sec * 1000000 + 
	pCommandContext->tv_start.tv_usec;

  /*
   * Status gets set above to timeout or in callback to success indicator.
   */
  if (pMathIOInfo->result_status == UBSEC_STATUS_SUCCESS) {
  	/* 
  	 * Now we need to copyout those parameters that were changed
  	 */
    pIOparams->Result.KeyLength = pMathparams->ModN.KeyLength;
    copyout(&MathLoc[MAX_KEY_BYTE_SIZE*MATH_RESULT_OFFSET],
	    pIOparams->Result.KeyValue,
	    BITSTOBYTES(pIOparams->Result.KeyLength));

  } else {
		error = -ENOMSG;
	}

 Return:
	/*
	 * Copyback the result
	 */
	copyout(pMathIOInfo, *pIOInfo, sizeof(*pMathIOInfo));
  	if (pmath_buf) kfree(pmath_buf);
	return error;
}

