
/*
 *  Broadcom Cryptonet Driver software is distributed as is, without any warranty
 *  of any kind, either express or implied as further specified in the GNU Public
 *  License. This software may be used and distributed according to the terms of
 *  the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */
/******************************************************************************
 *
 *  Copyright 2000
 *  Broadcom Corporation
 *  16215 Alton Parkway
 *  PO Box 57013
 *  Irvine CA 92619-7013
 *
 *****************************************************************************/
/*
 * device.c:  Modules to interface with the ubsec device.
 */
/*
 * Revision History:
 *
 * 10/xx/99 SOR Created.
 * 11/16/1999 DWP added PCI memory access enable for fixing linuxppc operation
 * March 2001 PW Release for Linux 2.4 UP and SMP kernel  
 * May   2001 PW added selftest for bcmdiag
 * June  2001 SRM added per device testing and forced device failure. 
 * July  2001 RJT Added support for BCM5821
 */

#include "cdevincl.h"

//#define SET_PCI_MAX_LATENCY 0x80
//#define SET_PCI_CLS
#define SET_PCI_RETRY_COUNT 0xff
#define SET_PCI_TRDY_COUNT 0xff

/* Devices */
struct DeviceTable_s {
  short VendorID;
  short DeviceID;
};

/* Define supported device list. */
#ifdef UBSEC_582x_CLASS_DEVICE
#define NUM_SUPPORTED_DEVICES 2
#else
#define NUM_SUPPORTED_DEVICES 3
#endif

static struct DeviceTable_s DeviceList[NUM_SUPPORTED_DEVICES]={
#ifndef UBSEC_582x_CLASS_DEVICE
  {BROADCOM_VENDOR_ID,BROADCOM_DEVICE_ID_5801},
  {BROADCOM_VENDOR_ID,BROADCOM_DEVICE_ID_5802},
  {BROADCOM_VENDOR_ID,BROADCOM_DEVICE_ID_5805},
#else
  {BROADCOM_VENDOR_ID,BROADCOM_DEVICE_ID_5820},
  {BROADCOM_VENDOR_ID,BROADCOM_DEVICE_ID_5821}
#endif
};

DeviceInfo_t DeviceInfoList[MAX_SUPPORTED_DEVICES];

void (*ubsec_callback)(void);
static void SelfTestCryptoCallback(unsigned long PacketContext,ubsec_Status_t Result);

/*
 *
 * interrupt_proc
 *
 * This function receives the completion signal from the device.  At completion
 * time, it marks the time and schedules the bottom half.
 *
 */
void
interrupt_proc( int irq, void* pvoid, struct pt_regs *regs )
{
}

/*
 * Initialize and return all devices.
 * Return the number of initialized devices.
 */

#ifndef LINUX2dot2
extern struct pci_dev *globalpDev;
#endif

int 
InitDevices(int NumberOfCryptoMCRs,int NumberOfKeyMCRs)
{
  short value;
  ubsec_Status_t Status;
  int i;
  int NumDevices=0;
  DeviceInfo_pt pDevice=&DeviceInfoList[0];
  struct pci_dev* pDev=NULL;

  if( !pcibios_present() ) {
    printk("cryptonet:  No PCI bus detected!\n" );
    return( 0 ); /* XXX make an error message */
  }

  /* Find and initialize all the devices we can */
  for (i=0; i < NUM_SUPPORTED_DEVICES ; ) {
    /* Find the next device. */

#if 0
    printk("Checking for %x-%x\n",DeviceList[i].VendorID,DeviceList[i].DeviceID);
#endif

    if ((pDev = pci_find_device(DeviceList[i].VendorID,DeviceList[i].DeviceID, pDev))==NULL) {
      i++;
      continue;
    }
#ifndef LINUX2dot2
 globalpDev = pDev;
#endif

    /* set this device as a bus-mastering device */
    pci_set_master(pDev);

    /* below paragraph added 16-nov-1999 dwp to make work on linuxppc */
    pci_read_config_word( pDev, PCI_COMMAND, &value ); /* SOR added */
    if( !(value & PCI_COMMAND_MEMORY) )    {
      value |= PCI_COMMAND_MEMORY;
      pci_write_config_word( pDev, PCI_COMMAND, value );
      pci_read_config_word( pDev, PCI_COMMAND, &value );
      if( !(value & PCI_COMMAND_MEMORY ))   {
	printk( "cryptonet:  memory access enable failed :(\n" );
      }
    }
    if( !(value & PCI_COMMAND_MASTER) ) {
      value |= PCI_COMMAND_MASTER;
      pci_write_config_word( pDev, PCI_COMMAND, value );
      pci_read_config_word( pDev, PCI_COMMAND, &value );
      if( !(value & PCI_COMMAND_MASTER ))  {
	printk( "Cryptonet  bus master enable failed\n" );
      }
    }	
#ifdef SET_PCI_MAX_LATENCY
    pci_write_config_byte(pDev,PCI_LATENCY_TIMER,SET_PCI_MAX_LATENCY);
#endif
#ifdef SET_PCI_CLS // Set the cache line size
    pci_write_config_byte(pDev,PCI_CACHE_LINE_SIZE,1);
#endif
#ifdef SET_PCI_RETRY_COUNT
    pci_write_config_byte(pDev,0x41,SET_PCI_RETRY_COUNT);
#endif
#ifdef SET_PCI_TRDY_COUNT
    pci_write_config_byte(pDev,0x40,SET_PCI_TRDY_COUNT);
#endif

    /* let the outside world know what we found */
    printk( "    Bus %i, Slot %d, device %04X, ",pDev->bus->number, PCI_SLOT(pDev->devfn),pDev->device );
    printk( "IRQ %d ", pDev->irq );
    sema_init (&pDevice->Semaphore_SRL, 1);
    Status=ubsec_InitDevice(pDev->device,
#ifndef LINUX2dot2
			    pci_resource_start(pDev,0),
#else
			    pDev->base_address[0], 
#endif
			    pDev->irq,
			    NumberOfCryptoMCRs,
			    NumberOfKeyMCRs, &(pDevice->Context),(OS_DeviceInfo_t) pDevice);

    if (Status != UBSEC_STATUS_SUCCESS) {
      printk("InitDevice Failed (%d)\n",Status);
      continue;
    } else {
      printk("\n");
    }

    pDevice->pDev=pDev;

    Status=Selftest(pDevice->Context,pDev);

    pDevice->DeviceStatus = Status;

#if 0
    if (Status != UBSEC_STATUS_SUCCESS) {
      printk("Cryptonet Device failed %lx\n",Status);
    } else {
#endif

      NumDevices++;
      if (NumDevices == MAX_SUPPORTED_DEVICES)
        break;
      pDevice++;

#if 0
    }
#endif

  }

  return(NumDevices);
}

/*
 * dump_pci_config: Dump the contents of the pci configuration associated
 * with the device.
 */
void
dump_pci_config(struct pci_dev *pDev)
{
  int i;
  int j;
  unsigned char uc;
  int val;

  printk("Cryptonet  --------------- PCI CONFIG REGISTERS ---------------");

  for( i = 0; i < 64; i += 4 ) {
    if( !(i%16) ) {
      printk( "\nCryptonet  " );
    }
    printk( "--%02d:", i/4);
    for( j = i+3; j >= i; j-- ) {
      pci_read_config_byte( pDev, j, &uc );
      printk( "%02X", uc );
    }
  }
  printk( "\nCryptonet  " );

  printk("\nCryptonet  ----------------------------------------------------\n");
}

/*
 * Use an Intermediate ISR callback because of
 * differences in parameters. For ISR and callback
 * instead of saving the SRL functions we just call
 * them directly.
 */
void
inter_callback(int irq,void *param,struct pt_regs *regs)
{
  DeviceInfo_pt pDevice=(DeviceInfo_pt)param;
  if (pDevice->Context==0)
    return;
  ubsec_ISR(pDevice->Context);
}

void
LinuxAllocateIRQ(int irq,void *context,void (*callback)())
{
  request_irq(irq,inter_callback, SA_INTERRUPT | SA_SHIRQ, UBSEC_DEVICE_NAME,(void *)context); 
}

void
LinuxFreeIRQ(int irq, void *context)
{
  free_irq(irq,context);
}

/*
 * Redirection of callback to make it fit
 * 
 */
void
callback_dpc(void *param)
{
  DeviceInfo_pt pDevice = (DeviceInfo_pt) param;
  if (pDevice->Context==0)
    return;
  ubsec_ISRCallback(pDevice->Context);
}

/*
 * Schedule it for a local function
 */
int
LinuxScheduleCallback(void (*callback)(void), void *Context, DeviceInfo_pt pDevice)
{
  pDevice->completion_handler_task.routine = callback_dpc;
  pDevice->completion_handler_task.data =pDevice;

    /* queue up the handler in the bottom half */
  queue_task( &pDevice->completion_handler_task, &tq_immediate );
  mark_bh( IMMEDIATE_BH );
  return 0;

}

/*
 * Self test callback Status:
 */
static void
SelfTestCryptoCallback(unsigned long PacketContext,ubsec_Status_t Result)
{
  struct pci_dev *pDev=(struct pci_dev *) PacketContext; /* This is our handle to the pci device */

  printk("Cryptonet Device crytpo selftest <%i-%i,%x> ",
    pDev->bus->number, PCI_SLOT(pDev->devfn),pDev->device);
  if (Result==UBSEC_STATUS_SUCCESS) 
    printk("passed!\n");
  else
    printk("failed %x\n",Result);
}

/*
 * LinuxInitCriticalSection:
 */
void
LinuxInitCriticalSection(volatile DeviceInfo_pt pDevice)
{
  sema_init(&pDevice->Semaphore_SRL, 1);
}

/*
 * LinuxEnterCriticalSection:
 */
unsigned long
LinuxEnterCriticalSection(DeviceInfo_pt pDevice)
{
  down_interruptible(&pDevice->Semaphore_SRL);
  return 0;
}

/*
 * LinuxLeaveCriticalSection: Called by SRL
 */
void
LinuxLeaveCriticalSection(DeviceInfo_pt pDevice)
{
  up (&pDevice->Semaphore_SRL);
}

/*
 * LinuxTestCriticalSection:
 */
unsigned long
LinuxTestCriticalSection(DeviceInfo_pt pDevice)
{
  if (down_trylock (&pDevice->Semaphore_SRL))
    return -1;
  return 0;
}

void
LinuxWaitus( int wait_us )
{
  __udelay(wait_us);
}

int
TestDevice(int SelectedDevice)
{
  
  int status = 1;
  volatile static int flag = 0;

#ifdef DEBUG_FAILOVER
  printk("Entering TestDevice, SelectedDevice = %d, DeviceStatus = %d\n", 
	 SelectedDevice, DeviceInfoList[SelectedDevice].DeviceStatus);
#endif

  if(flag == 0xdeadbeef) {
    
#ifdef DEBUG_FAILOVER
    printk("Returning from TestDevice, because of flag.\n");
#endif

    return 1;

  } else {

    flag = 0xdeadbeef;
    /* should we clear the semaphore here ????? */
    if(DeviceInfoList[SelectedDevice].DeviceStatus != -1) {
      status = Selftest(DeviceInfoList[SelectedDevice].Context, DeviceInfoList[SelectedDevice].pDev);
      SetDeviceStatus(&DeviceInfoList[SelectedDevice], status);
    } else {
#ifdef DEBUG_FAILOVER
      printk("TestDevice is resetting device %d whose status was %d and will be %d.\n", 
	      SelectedDevice, DeviceInfoList[SelectedDevice].DeviceStatus, status);
#endif
      /* keep device failing but next time it'll pass and status will be cleared */
      ubsec_ResetDevice(DeviceInfoList[SelectedDevice].Context);
      SetDeviceStatus(&DeviceInfoList[SelectedDevice], status);
    }
    
    flag = 0;
  }
  
  return DeviceInfoList[SelectedDevice].DeviceStatus;
}

int
TestDevices(PInt pm)
{
  int Retval=0;
  int i, timestogo;
  int seldev = *pm;

#ifdef DEBUG_FAILOVER
  printk("Entering TestDevices, seldev = %d\n", seldev);
#endif

  if (seldev > NumDevices) {
    printk("Selected Device is not installed.\n");
    return (-1);
  }
  if (seldev > 0) {
      seldev--;
      timestogo = 1;
  } else
      timestogo = NumDevices;
  
  /* test all the devices or a given device. */
  for (i=seldev; i < seldev+timestogo ; i++) {
    if (Retval=TestDevice(i))
      break;
  }
  return(Retval);
}

int
FailDevices(PInt pm)
{
  int i;
  int seldev = *pm;

#ifdef DEBUG_FAILOVER
  if(seldev) { 
    printk("Entering FailDevice, will fail device %d whose status is %d.\n", 
	   seldev - 1, DeviceInfoList[seldev-1].DeviceStatus);
  } else {
    printk("Entering FailDevice, will fail all devices\n", 
	   seldev - 1, DeviceInfoList[seldev-1].DeviceStatus);
  }
#endif

  if (seldev > NumDevices) {
    printk("Cryptonet: Selected device is not installed.\n");
    return (-1);
  }
  if (seldev > 0) {
     DeviceInfoList[seldev-1].DeviceStatus = -1;
     ubsec_DisableInterrupt(DeviceInfoList[seldev-1].Context);
  } else {
    /* Fail all the devices. */
    for (i=0; i < NumDevices ; i++) {
      DeviceInfoList[i].DeviceStatus = -1;
      ubsec_DisableInterrupt(DeviceInfoList[i].Context);
    }
  }
  
  return(0);
}

int
DumpDeviceInfo(PInt pm)
{
  int i, timestogo;
  int seldev = *pm;

  if (seldev > NumDevices) {
    printk("Selected device is not installed.\n");
    return (-1);
  }
  if (seldev > 0) {
      seldev--;
      timestogo = 1;
  } else
      timestogo = NumDevices;
  /* Dump Info for all the devices or a given device. */
  for (i=seldev; i < seldev+timestogo ; i++) {
    dump_pci_config(DeviceInfoList[i].pDev);
    ubsec_DumpDeviceInfo(DeviceInfoList[i].Context);
  }
  return(0);
}

/*
 * Get hardware version:
 * Return the ID (lower byte) and revision ID
 */
int
GetHardwareVersion(PInt pm)
{
  unsigned char uc;
  unsigned short retval;
  DeviceInfo_pt pDevice;
  int DNum = *pm;

  if (DNum > NumDevices) {
    printk("Selected device is not installed.\n");
    return (-1);
  }
  if (DNum > 0)
    DNum--;

  pDevice=&DeviceInfoList[DNum];

  pci_read_config_byte( pDevice->pDev, PCI_REVISION_ID, &uc );
  retval=(((pDevice->pDev->device)&0xff)<<8); /* Only need the lower byte */
  retval+=uc;
  return(retval);
}
