
/*
 *  Broadcom Cryptonet Driver software is distributed as is, without any warranty
 *  of any kind, either express or implied as further specified in the GNU Public
 *  License. This software may be used and distributed according to the terms of
 *  the GNU Public License.
 *
 * Cryptonet is a registered trademark of Broadcom Corporation.
 */
/******************************************************************************
 *
 *  Copyright 2000
 *  Broadcom Corporation
 *  16215 Alton Parkway
 *  PO Box 57013
 *  Irvine CA 92619-7013
 *
 *****************************************************************************/
/* 
 * Broadcom Corporation uBSec SDK 
 */
/*
 * Crypto operations for character driver interface to the ubsec driver
 *
 * Revision History:
 *
 * May 2000 SOR/JTT Created.
 * March 2001 PW Release for Linux 2.4 UP and SMP kernel 
 */

#include "cdevincl.h"

#define AUTH_BUF_SIZE 4096

#ifdef POLL
#undef GOTOSLEEP
#endif

/* These are useful only for diagnostics. */
#undef SETFRAGMENTS
#undef STATIC_ALLOC_OF_CRYPTO_BUFFERS

#ifdef SETFRAGMENTS
ubsec_FragmentInfo_t SourceFragments[MAX_COMMANDS][MAX_FRAGMENTS];
ubsec_FragmentInfo_t DestinationFragments[MAX_COMMANDS][MAX_FRAGMENTS];
#endif

int SetupFragmentList(ubsec_FragmentInfo_pt Frags, unsigned char *packet,int packet_len);


#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
unsigned char *kern_source_buf = NULL;
unsigned char *kern_dest_buf   = NULL;
unsigned char *kern_auth_buf   = NULL;
static int PhysSourceBuf;
static int PhysDestBuf;
static int PhysAuthBuf;
#endif

#define ALLOC_PAGE_SIZE ((unsigned long)PAGE_SIZE)
#define ALLOC_PAGE_MASK (ALLOC_PAGE_SIZE-1)
#define BYTES_TO_PAGE_BOUNDARY_PHYS(v) \
 ((unsigned long)(ALLOC_PAGE_SIZE-(((unsigned long)v)&(ALLOC_PAGE_MASK))))
#define BYTES_TO_PAGE_BOUNDARY(p) (BYTES_TO_PAGE_BOUNDARY_PHYS(virt_to_bus(p)))

/**************************************************************************
 *
 *  Function:  init_cryptoif
 *   
 *************************************************************************/
int init_cryptoif(void)
{
#ifdef DEBUG
  printk("ubsec library version %d.%x%c\n", UBSEC_VERSION_MAJOR,
	 UBSEC_VERSION_MINOR, UBSEC_VERSION_REV);
#endif

#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  kern_source_buf = 
    (char *) kmalloc((MAX_FILE_SIZE ),GFP_KERNEL|GFP_ATOMIC);

  if( kern_source_buf == NULL ) {
    printk("Cryptonet: no memory for source buffer\n");
    return -ENOMEM;
  }

#ifdef DEBUG
  printk("Allocate %x %x\n",kern_source_buf,vtophys(kern_source_buf));
#endif

  kern_dest_buf=kern_source_buf;
  kern_auth_buf = (char *) kmalloc((AUTH_BUF_SIZE),GFP_KERNEL|GFP_ATOMIC);
  if( kern_auth_buf == NULL ) {
    kfree(kern_source_buf);
    printk("Cryptonet: no memory for auth buffer\n");
    return -ENOMEM;
  }

  PhysSourceBuf=virt_to_bus(kern_source_buf);
  PhysDestBuf=virt_to_bus(kern_dest_buf);
  PhysAuthBuf=virt_to_bus(kern_auth_buf);

#ifdef DEBUG
  printk("Cryptonet: Memory Alloc source %x %x Dest %x %x for source buffer\n",
	 kern_source_buf,PhysSourceBuf,kern_dest_buf,PhysDestBuf);
#endif

#endif /* STATIC_ALLOC */

  return 0; /* success */
}



/**************************************************************************
 *
 *  Function:  cleanup_module
 *
 *************************************************************************/
void shutdown_cryptoif(void)
{
#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  if (kern_source_buf)
    kfree(kern_source_buf);
  if (kern_auth_buf)
    kfree(kern_auth_buf);
#endif /* STATIC_ALLOC */
}

int
do_encrypt(ubsec_DeviceContext_t pContext,ubsec_io_pt *pat)
{
  ubsec_io_t		at_buf;
  ubsec_io_pt		at = &at_buf;
#ifndef SETFRAGMENTS
  ubsec_FragmentInfo_t *SourceFragments=NULL;
#endif
  unsigned int num_packets;
  unsigned int total_packets;
  unsigned int packets_done;
  ubsec_HMAC_State_t HMAC_State;
  ubsec_CipherCommandInfo_pt acmd;
  volatile CommandContext_t CommandContext;
  unsigned int i;
#ifdef SETFRAGMENTS
  unsigned int  j;
#endif
  unsigned int src_pos;
  unsigned int dest_pos;
  unsigned char *user_source_buf;
  unsigned char *user_dest_buf;

  unsigned int source_buf_size;
  unsigned int dest_buf_size;
  unsigned int in_packet_size;
  unsigned int out_packet_size;
  int MacSize=0;
  unsigned long delay_total_us;
  ubsec_CipherCommandInfo_t *ubsec_commands = NULL;

#ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  unsigned char *kern_source_buf = NULL;
  unsigned char *kern_dest_buf   = NULL;
  unsigned char *kern_auth_buf   = NULL;
  int PhysSourceBuf;
  int PhysDestBuf;
  int PhysAuthBuf=0;
  int Status=0;
#endif

	/*
	 * Copy control packet into kernel space.
	 */
  copy_from_user( at,*pat , sizeof(*at));

  user_source_buf = at->source_buf;
  user_dest_buf   = at->dest_buf;
  source_buf_size   = at->source_buf_size;
  dest_buf_size     = at->dest_buf_size;
  in_packet_size    = source_buf_size / at->num_packets;
  out_packet_size   = dest_buf_size / at->num_packets;

#ifdef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  if( source_buf_size > MAX_FILE_SIZE ) {
    printk("Cryptonet: input file too large <%d,%d>\n",source_buf_size,MAX_FILE_SIZE);
    Status=EINVAL;
    goto Error_Ret;
  }
#endif

#ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  kern_source_buf = 
    (char *) kmalloc((dest_buf_size),GFP_KERNEL|GFP_ATOMIC);

  if( kern_source_buf == NULL ) {
    printk("Cryptonet: no memory for source buffer %d\n",dest_buf_size);
    Status=-ENOMEM;
    goto Error_Ret;
  }

#ifdef DEBUG
  printk("Allocate %x %x\n",kern_source_buf,vtophys(kern_source_buf));
#endif

  kern_dest_buf=kern_source_buf;

  PhysSourceBuf=virt_to_bus(kern_source_buf);
  PhysDestBuf=virt_to_bus(kern_dest_buf);

#ifdef DEBUG
  printk("Cryptonet: Memory Alloc source %x %x Dest %x %x for source buffer\n",
	 kern_source_buf,PhysSourceBuf,kern_dest_buf,PhysDestBuf);
#endif

#endif /* STATIC_ALLOC */

  packets_done = 0;   /* incremented every time callback is called */
  num_packets = at->num_packets;
  if( num_packets > MAX_COMMANDS ) {
    printk("Cryptonet: too many packets/commands\n");
    Status=EINVAL;
    goto Error_Ret;
  }

#ifdef SETFRAGMENTS
  if (at->num_fragments > UBSEC_MAX_FRAGMENTS)
  {
    printk("Cryptonet: too many fragments\n");
    Status=EINVAL;
    goto Error_Ret;
  }
#endif

  /*
   * File size is a little different when using MAC
   */
  if(UBSEC_USING_MAC( at->flags ) ) {
    /*
     * We need to initialize the inner and outer hash keys.
     */
    MacSize= (at->flags & UBSEC_MAC_MD5) ? 16 : 20;
    if ( at->flags & UBSEC_DECODE ) {
      kern_auth_buf = (char *) kmalloc((AUTH_BUF_SIZE),GFP_KERNEL|GFP_ATOMIC);
      if( kern_auth_buf == NULL ) {
	printk("Cryptonet: no memory for auth buffer\n");
	Status=-ENOMEM;
	goto Error_Ret;
      }
      in_packet_size    = (source_buf_size - (at->num_packets*MacSize)) / at->num_packets;
      PhysAuthBuf=virt_to_bus(kern_auth_buf);
      memset(kern_auth_buf,0,AUTH_BUF_SIZE);
    }
    else
      PhysAuthBuf=virt_to_bus(&kern_dest_buf[at->num_packets*in_packet_size]); /* Add to end */
    out_packet_size   = in_packet_size;
   }

  /* Allocate memory for fragment information. */
  SourceFragments=(char *) kmalloc(((sizeof(*SourceFragments)*MAX_COMMANDS*UBSEC_MAX_FRAGMENTS)),GFP_KERNEL|GFP_ATOMIC);
  if( SourceFragments == NULL ) {
    printk("Cryptonet: no memory for fragment buffer\n");
    Status=-ENOMEM;
    goto Error_Ret;
  }

/*
  memset(ubsec_commands,0, sizeof(ubsec_commands));
*/
  memset(&CommandContext,0, sizeof(CommandContext_t));
  src_pos = 0;
  dest_pos = 0;

  ubsec_commands=(char *) kmalloc((sizeof(ubsec_CipherCommandInfo_t)*num_packets),GFP_KERNEL|GFP_ATOMIC);

  if( ubsec_commands == NULL ) {
    printk("Cryptonet: no memory for source buffer\n");
    return -ENOMEM;
  }


  for(i = 0; i < num_packets; i++) {
    acmd = &ubsec_commands[i];
    acmd->InitialVector = (ubsec_IV_pt) at->initial_vector;
    acmd->CryptKey = (ubsec_CryptKey_pt) at->crypt_key;
    acmd->Command=at->flags;
    if (UBSEC_USING_MAC(at->flags)) 
      ubsec_InitHMACState(&HMAC_State,UBSEC_USING_MAC(acmd->Command),at->mac_key);
    acmd->HMACState=&HMAC_State;
    acmd->CryptHeaderSkip = at->crypt_header_skip;

#ifdef SETFRAGMENTS
    acmd->NumSource = at->num_fragments;
    acmd->NumDestination = at->num_fragments;
    for( j = 0; j < at->num_fragments; j++ ) {
      SourceFragments[i][j].FragmentAddress = PhysSourceBuf+src_pos;
      SourceFragments[i][j].FragmentLength = in_packet_size / at->num_fragments;
      if( j+1 == at->num_fragments )
	SourceFragments[i][j].FragmentLength += in_packet_size % at->num_fragments;
      src_pos += SourceFragments[i][j].FragmentLength;
    }
    acmd->SourceFragments=&SourceFragments[i][0];

    /*
     * Keep destination fragments separate since there
     * are more restrictions on them
     */
    if ((at->num_fragments == 1) || (!(at->num_fragments % 4)))
      acmd->NumDestination = at->num_fragments;
    else
      acmd->NumDestination=4;  /* Default. */

    for( j = 0; j < acmd->NumDestination; j++ ) {
      DestinationFragments[i][j].FragmentAddress =  PhysDestBuf+dest_pos;
      DestinationFragments[i][j].FragmentLength = out_packet_size / acmd->NumDestination; /*at->num_fragments; */
      if( j+1 == acmd->NumDestination)
	DestinationFragments[i][j].FragmentLength += out_packet_size % acmd->NumDestination;
      dest_pos += DestinationFragments[i][j].FragmentLength;
    }
    acmd->DestinationFragments=&DestinationFragments[i][0];

#else
    /* Since we do not have a large enough contiguous buffer, we override
       the fragment num setting and set the fragment accordingly. */
    acmd->SourceFragments=&SourceFragments[i*MAX_FRAGMENTS];
    acmd->NumSource=SetupFragmentList(acmd->SourceFragments,&kern_source_buf[src_pos],in_packet_size);
    acmd->NumDestination=acmd->NumSource;
    acmd->DestinationFragments=acmd->SourceFragments;
    src_pos += in_packet_size;
    dest_pos+= in_packet_size;
#endif
    /*
     * If we are doing authentication then we need to allocate
     * another fragment for the output
     */
    if(UBSEC_USING_MAC( acmd->Command ) ) {
      acmd->AuthenticationInfo.FragmentAddress = PhysAuthBuf+(MacSize*i);
    }
    if (i==(num_packets-1)) {
	/*
	 * We only set the callback for the last one since that
	 * they are completed in batch mode
	 */
      acmd->CompletionCallback = CmdCompleteCallback;
      acmd->CommandContext = &CommandContext;
    }
    else {
      acmd->CompletionCallback = NULL;
      acmd->CommandContext=0;
    }
  }

  if(UBSEC_USING_MAC( at->flags )) {
    if (at->flags & UBSEC_DECODE) {
      src_pos+=(MacSize*num_packets);
      }
    else {
      dest_pos+=(MacSize*num_packets);
    }
  }
  if( src_pos != source_buf_size ) {
    printk("Cryptonet: invalid source buffer size -- "
	   "given size %u -- total used/needed %u\n", 
	   source_buf_size, src_pos);
    Status=EINVAL;
    goto Error_Ret;
  }
  if(dest_pos != dest_buf_size ) {
    printk("Cryptonet: invalid dest buffer size -- "
	   "given size %u -- total used/needed %u\n", 
	   dest_buf_size, dest_pos);
    Status=EINVAL;
    goto Error_Ret;
  }

  memset(kern_source_buf,0,dest_buf_size);
  copy_from_user(kern_source_buf, user_source_buf, source_buf_size);

  /*
   *  Let the system do anything it may want/need to do before we begin
   *  timing.
   */
  start_time(&CommandContext.tv_start);
  total_packets=num_packets;  

#ifndef LINUX2dot2
  init_waitqueue_head(&CommandContext.WaitQ);
#else
   CommandContext.WaitQ         = 0; 
#endif    

  switch ( ubsec_CipherCommand(pContext,ubsec_commands,&num_packets) ) {
  case UBSEC_STATUS_SUCCESS:
    break;
  case UBSEC_STATUS_TIMEOUT:
    printk("ubsec  ubsec_Command() Timeout\n");
    ubsec_ResetDevice(pContext);
    Status=ETIMEDOUT;
    goto Error_Ret;
    break;
  case UBSEC_STATUS_INVALID_PARAMETER:
    printk("Cryptonet:  ubsec_Command() Invalid parameter\n");
    Status=EINVAL;
    goto Error_Ret;
    break;
  case UBSEC_STATUS_NO_RESOURCE:
    printk("ubsec  ubsec_Command() No crypto resource. Num Done %d\n",num_packets);
  default:
    Status=ENOMSG;
    goto Error_Ret;
    break;
  }

#ifdef GOTOSLEEP
  if (!(CommandContext.CallBackStatus)) { /* Just in case completed on same thread. */
     Gotosleep(&CommandContext.WaitQ);
     if (!CommandContext.CallBackStatus) { /* interrupt never happened? */
        CommandContext.Status=UBSEC_STATUS_TIMEOUT;
	goto Error_Ret;
     }
  }
#else
  for (delay_total_us=1  ; !(CommandContext.CallBackStatus); delay_total_us++) {
#ifdef POLL /* We need to poll the device if we are operating in POLL mode. */
    ubsec_PollDevice(pContext);
#endif
    if (delay_total_us >= 3000000) {
    Status=ETIMEDOUT;
    goto Error_Ret;
    }
    udelay(1);
  }
#endif

  if (UBSEC_USING_MAC( at->flags ) && (at->flags & UBSEC_DECODE)) {
    /* Check the Auth */
    if (memcmp(kern_auth_buf,&kern_source_buf[at->num_packets*in_packet_size],MacSize)) { /*at->num_packets*MacSize)) */
#ifdef PRINT_AUTH_ERROR_INFO
      printk("Cryptonet:  failed -- Authentication error\n");
      for (i=0; i < MacSize ; i++)
	printk("<%x,%x>",kern_auth_buf[i],kern_source_buf[(at->num_packets*in_packet_size)+i]);
        printk("\n");
#endif
	Status=-1;
	goto Error_Ret;
    }
  }

  copy_to_user(user_dest_buf, kern_dest_buf, dest_buf_size);
  at->time_us = CommandContext.tv_start.tv_sec * 1000000 + CommandContext.tv_start.tv_usec;

 Error_Ret:

#ifndef STATIC_ALLOC_OF_CRYPTO_BUFFERS
  if (kern_source_buf)
    kfree(kern_source_buf);
  if (kern_auth_buf)
    kfree(kern_auth_buf);
#endif /* STATIC_ALLOC */

#ifndef SETFRAGMENTS
    if (SourceFragments)
     kfree(SourceFragments);
#endif

     kfree( ubsec_commands);
  /*
   * Copy back time to user space.
   */
#if 0
  copyout((unsigned char *)&time_us, (unsigned char *)&(*pat)->time_us, sizeof(time_us));
#else
  copyout(at, *pat, sizeof(*at));
#endif
  return 0;

  return(Status);
}



/*
 * Setup Fragment list: Initializes a fragment list for a logically
 * contigous but maybe physically fragmented buffer
 *
 * Return the number of fragments allocated.
 */
int
SetupFragmentList(ubsec_FragmentInfo_pt Frags, unsigned char *packet,int packet_len)
{
  int NumFrags=1;
  int i;

  /* Initial case. */
  Frags[0].FragmentLength=packet_len;
  Frags[0].FragmentAddress=virt_to_bus(packet);

  if (packet_len > BYTES_TO_PAGE_BOUNDARY(packet)) {
    /* First case is special since we are dealing with offsets. Readjust length */
    Frags[0].FragmentLength=BYTES_TO_PAGE_BOUNDARY(packet);
    packet_len -= Frags[0].FragmentLength;
    packet += Frags[0].FragmentLength;

#ifdef DEBUG_FRAGS
    printk("Frag %d <%x,%x>",0,Frags[0].FragmentLength,Frags[0].FragmentAddress);
#endif
      /* From now on the Frags should be equal and of length ALLOC_PAGE_SIZE until the last. */
    for  (i=1;packet_len > 0;i++) {
      Frags[i].FragmentAddress=virt_to_bus(packet);
#ifdef DEBUG_FRAGS
      if (packet_len > ALLOC_PAGE_SIZE) {
	Frags[i].FragmentLength = ALLOC_PAGE_SIZE;
      } else {
	Frags[i].FragmentLength = packet_len;
      }
#else
      if (packet_len > BYTES_TO_PAGE_BOUNDARY(packet)) {
	Frags[i].FragmentLength = BYTES_TO_PAGE_BOUNDARY(packet);
      } else {
	Frags[i].FragmentLength = packet_len;
      }
#endif
#ifdef DEBUG_FRAGS
    printk("Frag %d <%x,%x>",i,Frags[i].FragmentLength,Frags[i].FragmentAddress);
#endif
	NumFrags++;
        packet+=Frags[i].FragmentLength;
        packet_len-= Frags[i].FragmentLength;
      }
  }
#ifdef DEBUG_FRAGS
printk("Returning %d fragments\n",NumFrags);
#endif
return(NumFrags);
}




