/* $Id: maps.c,v 1.2 2001/07/02 21:13:42 hch Exp $
 * map.c - generic mapping routines for binary emulation
 *
 * Copyright (C) 1994  Mike Jagdis (jaggy@purplet.demon.co.uk)
 */

#include <linux/kernel.h>
#include <linux/module.h>

#include <linux/sched.h>
#include <linux/net.h>
#include <linux/socket.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/personality.h>

#include <abi/socket.h>
#include <abi/map.h>


struct map_segment abi_sockopt_map[] =  {
	{ 0x0001, 0x0001, (char *)SO_DEBUG },
	{ 0x0002, 0x0002, (char *)__SO_ACCEPTCON },
	{ 0x0004, 0x0004, (char *)SO_REUSEADDR },
	{ 0x0008, 0x0008, (char *)SO_KEEPALIVE },
	{ 0x0010, 0x0010, (char *)SO_DONTROUTE },
	{ 0x0020, 0x0020, (char *)SO_BROADCAST },
	{ 0x0040, 0x0040, (char *)SO_USELOOPBACK },
	{ 0x0080, 0x0080, (char *)SO_LINGER },
	{ 0x0100, 0x0100, (char *)SO_OOBINLINE },
	{ 0x0200, 0x0200, (char *)SO_ORDREL },
	{ 0x0400, 0x0400, (char *)SO_IMASOCKET },
	{ 0x1001, 0x1001, (char *)SO_SNDBUF },
	{ 0x1002, 0x1002, (char *)SO_RCVBUF },
	{ 0x1003, 0x1003, (char *)SO_SNDLOWAT },
	{ 0x1004, 0x1004, (char *)SO_RCVLOWAT },
	{ 0x1005, 0x1005, (char *)SO_SNDTIMEO },
	{ 0x1006, 0x1006, (char *)SO_RCVTIMEO },
	{ 0x1007, 0x1007, (char *)SO_ERROR },
	{ 0x1008, 0x1008, (char *)SO_TYPE },
	{ 0x1009, 0x1009, (char *)SO_PROTOTYPE },
	{ -1 }
};


struct map_segment abi_af_map[] =  {
	/*
	 * The first three entries (AF_UNSPEC, AF_UNIX and AF_INET)
	 * are identity mapped. All others aren't available under
	 * Linux, nor are Linux's AF_AX25 and AF_IPX available from
	 * SCO as far as I know.
	 */
	{ 0, 2, NULL },
	{ -1 }
};

long
map_bitvec(u_long vec, long map[])
{
	u_long			newvec = 0, m = 1;
	int			i;

	for (i = 1; i <= 32; i++) {
		if ((vec & m) && map[i] != -1)
			newvec |= (1 << map[i]);
		m <<= 1;
	}

	return newvec;
}

u_long
map_sigvec_from_kernel(sigset_t vec, u_long map[])
{
	u_long			newvec = 0;
	int			i;

	for (i = 1; i <= 32; i++) {
		if (sigismember(&vec, i) && map[i] != -1)
			newvec |= (1 << map[i]);
	}

	return newvec;
}

sigset_t
map_sigvec_to_kernel(u_long vec, u_long map[])
{
	sigset_t		newvec;
	u_long			m = 1;
	int			i;

	sigemptyset(&newvec);
	for (i = 1; i <= 32; i++) {
		if ((vec & m) && map[i] != -1)
			sigaddset(&newvec, map[i]);
		m <<= 1;
	}

	return newvec;
}

int
map_value(struct map_segment *m, int val, int def)
{
	struct map_segment	*seg;

	/*
	 * If no mapping exists in this personality just return the
	 * number we were given.
	 */
	if (!m)
		return val;

	/*
	 * Search the map looking for a mapping for the given number.
	 */
	for (seg = m; seg->start != -1; seg++) {
		if (seg->start <= val && val <= seg->end) {
			/*
			 * If the start and end are the same then this
			 * segment has one entry and the map is the value
			 * it maps to. Otherwise if we have a vector we
			 * pick out the relevant value, if we don't have
			 * a vector we give identity mapping.
			 */
			if (seg->start == seg->end)
				return (int)seg->map;
			else
				return (seg->map ? seg->map[val-seg->start] : val);
		}
	}

	/* Number isn't mapped. Returned the requested default. */
	return def;
}
