/* $Id: lcall7.c,v 1.9 2001/07/16 15:13:54 hch Exp $
 * lcall7.c - syscall dispatcher for personalities using lcall7 entries
 *
 * Copyright (C) 1993  Linus Torvalds
 * Modified by Eric Youngdale to include all ibcs syscalls.
 * Re-written by Drew Sullivan to handle lots more of the syscalls correctly.
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/stddef.h>
#include <linux/unistd.h>
#include <linux/ptrace.h>
#include <linux/fcntl.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/sys.h>
#include <linux/personality.h>
#include <asm/uaccess.h>

#include <abi/abi.h>
#include <abi/trace.h>
#include <abi/sysent.h>

#include "errmap.h"

#define last(x)	((sizeof(x)/sizeof(*x))-1)

typedef int (*sysfun_p)(void);
typedef int (*sysfun_pregs)(struct pt_regs *);
typedef int (*sysfun_p1int)(int);
typedef int (*sysfun_p2int)(int, int);
typedef int (*sysfun_p3int)(int, int, int);
typedef int (*sysfun_p4int)(int, int, int, int);
typedef int (*sysfun_p5int)(int, int, int, int, int);
typedef int (*sysfun_p6int)(int, int, int, int, int, int);
typedef int (*sysfun_p7int)(int, int, int, int, int, int, int);

extern sysfun_p sys_call_table[];


MODULE_AUTHOR("Mike Jagdis <jaggy@purplet.demon.co.uk>");
MODULE_DESCRIPTION("Support for non-Linux programs");

static void	fail(long, abi_func_t *);
static void	plist(char *, char *, int *);


static const char * const sig_names[] = {
	"SIGHUP",	"SIGINT",	"SIGQUIT",	"SIGILL",
	"SIGTRAP",	"SIGABRT/SIGIOT","SIGUNUSED",	"SIGFPE",
	"SIGKILL",	"SIGUSR1",	"SIGSEGV",	"SIGUSR2",
	"SIGPIPE",	"SIGALRM",	"SIGTERM",	"SIGSTKFLT",
	"SIGCHLD",	"SIGCONT",	"SIGSTOP",	"SIGTSTP",
	"SIGTTIN",	"SIGTTOU",	"SIGIO/SIGPOLL/SIGURG",
	"SIGXCPU",	"SIGXFSZ",	"SIGVTALRM",	"SIGPROF",
	"SIGWINCH",	"SIGLOST",	"SIGPWR",	"SIG 31",
	"SIG 32"
};


void
abi_dispatch(struct pt_regs *regp, abi_func_t *ap, int of)
{
	void			*kfunc = ap->kfunc;
	short			nargs = ap->nargs;
	int			args[8], rvalue, i;

	/*
	 * If the number of arguments is negative this is an unfudged
	 * system call function and we need to look up the real function
	 * address in the kernel's sys_call_table.
	 * Note that we never modify the callmap itself but do the lookup
	 * for each call. This allows modules that provide syscalls to
	 * be loaded and unloaded without any messy locking.
	 *
	 * XXX remove this cruft ASAP  --hch
	 */
	if (nargs < 0) {
		int		sysno = (int)kfunc;

		/*
		 * Watch for a magic zero.
		 * This exists because we can't use -0 to represent
		 * a system call that takes no arguments.
		 */
		nargs = (nargs == -ZERO) ? 0 : -nargs;
		kfunc = sys_call_table[sysno];
	}

	if (nargs <= ARRAY_SIZE(args)) {
		for (i = 0; i < nargs; i++)
			get_user(args[i], ((u_long *)regp->esp)+(i+of));
	}

	if (abi_traced(ABI_TRACE_API)) {
		if (nargs == Spl) {
			for (i = 0; i < strlen(ap->args); i++)
				get_user(args[i], ((u_long *)regp->esp) + (i + of));
		}
		plist(ap->name, ap->args, args);
	}

	rvalue = -ENOSYS;
	if (kfunc) {
		switch (nargs) {
		case Fast:
			((sysfun_pregs)kfunc)(regp);
			if (abi_traced(ABI_TRACE_API|ABI_TRACE_SIGNAL) &&
			    signal_pending(current)) {
				u_long signr;

				signr = current->pending.signal.sig[0] &
					~current->blocked.sig[0];

				__asm__("bsf %1,%0\n\t"
					:"=r" (signr)
					:"0" (signr));
	
				__abi_trace("SIGNAL %lu <%s>\n",
					signr + 1, sig_names[signr]);
			}
			return;
		case Spl:
			rvalue = ((sysfun_pregs)kfunc)(regp);
			break;
		case 0:
			rvalue = ((sysfun_p)kfunc)();
			break;
		case 1:
			rvalue = ((sysfun_p1int)kfunc)(args[0]);
			break;
		case 2:
			rvalue = ((sysfun_p2int)kfunc)(args[0], args[1]);
			break;
		case 3:
			rvalue = ((sysfun_p3int)kfunc)(args[0], args[1], args[2]);
			break;
		case 4:
			rvalue = ((sysfun_p4int)kfunc)(args[0], args[1], args[2], args[3]);
			break;
		case 5:
			rvalue = ((sysfun_p5int)kfunc)(args[0], args[1], args[2],
					     args[3], args[4]);
			break;
		case 6:
			rvalue = ((sysfun_p6int)kfunc)(args[0], args[1], args[2],
					     args[3], args[4], args[5]);
			break;
		case 7:
			rvalue = ((sysfun_p7int)kfunc)(args[0], args[1], args[2],
					     args[3], args[4], args[5], 
						   args[6]);
			break;
		default:
			fail(regp->eax, ap);
		}
	} else
		fail(regp->eax, ap);
	
	if (rvalue >= 0 || rvalue < -ENOIOCTLCMD) {
		regp->eflags &= ~1; /* Clear carry flag */
		regp->eax = rvalue;

		abi_trace(ABI_TRACE_API, "%s returns %ld {%ld}\n",
				ap->name, regp->eax, regp->edx);
	} else {
		regp->eflags |= 1; /* Set carry flag */
		regp->eax = iABI_errors(-rvalue);

		if (abi_traced(ABI_TRACE_API)) {
#ifdef CONFIG_ABI_VERBOSE_ERRORS
			__abi_trace("%s error return linux=%d -> ibcs=%ld <%s>\n",
				ap->name, rvalue, regp->eax,
				-rvalue < ARRAY_SIZE(errmsg) ?
					errmsg[-rvalue] : "unknown");
#else
			__abi_trace("%s error return linux=%d -> ibcs=%ld\n",
					ap->name, rvalue, regp->eax);
#endif
		}
	}

	if (abi_traced(ABI_TRACE_API|ABI_TRACE_SIGNAL) && signal_pending(current)) {
		u_long signr;
		
		signr = current->pending.signal.sig[0] &
			~current->blocked.sig[0];

		__asm__("bsf %1,%0\n\t"
			:"=r" (signr)
			:"0" (signr));

		__abi_trace("SIGNAL %lu <%s>, queued 0x%08lx\n",
			signr + 1, sig_names[signr],
			current->pending.signal.sig[0]);
	}
}

int
abi_syscall(struct pt_regs *regp)
{
	get_user(regp->eax, ((u_long *)regp->esp) + 1);

	++regp->esp;
	current->exec_domain->handler(-1,regp);
	--regp->esp;

	return 0;
}

/*
 * plist is used by the trace code to show the arg list
 */
static void
plist(char *name, char *args, int *list)
{
	int	error;
	char	*tmp, *p, arg_buf[512];

	if (!abi_traced(ABI_TRACE_API))
		return;

	arg_buf[0] = '\0';
	p = arg_buf;
	while (*args) {
		switch(*args++) {
		case 'd':
			sprintf(p, "%d", *list++);
			break;
		case 'o':
			sprintf(p, "0%o", *list++);
			break;
		case 'p':
			sprintf(p, "0x%p", (void *)(*list++));
			break;
		case '?': 
		case 'x':
			sprintf(p, "0x%x", *list++);
			break;
		case 's': 
			tmp = getname((char *)(*list++));
			error = PTR_ERR(tmp);
			if (!IS_ERR(tmp)) {
				/* we are debugging, we don't need to see it all */
				tmp[80] = '\0';
				sprintf(p, "\"%s\"", tmp);
				putname(tmp);
			}
			break;
		default:
			sprintf(p, "?%c%c?", '%', args[-1]);
			break;
		}
		while (*p)
			++p;
		if (*args) {
			*p++ = ',';
			*p++ = ' ';
			*p = '\0';
		}
	}
	__abi_trace("%s(%s)\n", name, arg_buf);
}

static void
fail(long eax, abi_func_t *ap)
{
	if (!abi_traced(ABI_TRACE_API))
		return;

	printk(KERN_ERR "[%s:%d] Unsupported ABI function 0x%lx (%s)\n",
			current->comm, current->pid, eax, ap->name);
}
